﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/uart/uart.h>

#include <nn/gpio/gpio.h>

#include <nnt/gtest/gtest.h>

#include <nn/dd.h>

#include "../Common/testUart_Helper.h"
#include "testUart_Bcm4356_details.h"

namespace
{

// Bluetooth Reset ピンの GPIO パッド定義
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    const auto BluetoothResetGpioPad = nn::gpio::GpioPadName_BtRst;
#else
    const auto BluetoothResetGpioPad = nn::gpio::GpioPadName_BluetoothPowerEnable;
#endif

// 内部送信バッファ
NN_ALIGNAS(nn::os::MemoryPageSize) char g_SendBuffer[nnt::uart::bcm4356::details::UartBufferSize];

// 内部受信バッファ
NN_ALIGNAS(nn::os::MemoryPageSize) char g_ReceiveBuffer[nnt::uart::bcm4356::details::UartBufferSize];

// 受信完了イベント
nn::os::SystemEventType g_ReceiveEndEvent;

/*
 * BCM4356 の Bluetooth 有効/無効を切り替えます。
 */
void SetBluetoothEnabled(bool isEnabled) NN_NOEXCEPT
{
    nn::gpio::GpioPadSession gpioBtPowerEn;
    nn::gpio::OpenSession(&gpioBtPowerEn, BluetoothResetGpioPad);
    nn::gpio::SetDirection(&gpioBtPowerEn, nn::gpio::Direction_Output);

    nn::gpio::SetValue(&gpioBtPowerEn,
                       isEnabled ? nn::gpio::GpioValue_High : nn::gpio::GpioValue_Low);

    // Bluetooth の ON/OFF 切り替え時は 10msec 以上待つ必要がある。
    // (BCM4356 Advance Data Sheet p.182)
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));

    nn::gpio::CloseSession(&gpioBtPowerEn);
}

/*
 * HCI_Reset コマンドを確認します。
 */
void TestHciReset(nn::uart::PortSession* pSession) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSession);

    /*
     * UART HCI コマンドパケットフォーマット
     *
     * +------+--------+--------------+--------+
     * | Type | OpCode | Param length | Params |
     * | 8bit | 16bit  |     8bit     |  ...   |
     * +------+--------+--------------+--------+
     *
     * [Type]
     *   0x01 : HCI Command Packet
     *   0x02 : HCI ACL (Asynchronous Connection-Less) Data Packet
     *   0x03 : HCI SCO (Synchronous Connection-Oriented) Data Packet
     *   0x04 : HCI Event Packet
     *
     * [OpCode]
     *   (Little endian)
     *   +------+-------+
     *   | OGF  |  OCF  |
     *   | 6bit | 10bit |
     *   +------+-------+
     *
     *   OGF = OpCode Group Field
     *   OCF = OpCode Command Field
     */
    const char Command[] =
        {
            nnt::uart::bcm4356::details::HciPacketType_Command,
            0x03, 0x0C, 0x00  // OGF=0x03, OCF=0x0003, No param
        };

    /*
     * UART HCI イベントパケットフォーマット
     *
     * +--------+-------+--------+-------------+-----+
     * |  Type  | Event | Params |    Event    |     |
     * | (0x04) | Code  | Length | Parameter x | ... |
     * |  8bit  | 8bit  |  8bit  |    xxbit    |     |
     * +--------+-------+--------+-------------+-----+
     *
     * [Event Code]
     *   0x0E : Command complete event
     */
    const char ExpectedResult[] =
        {
            nnt::uart::bcm4356::details::HciPacketType_Event,
            0x0E,        // Command complete event
            0x04,        // 後続のデータ長
            0x01,        // 次に受け付けるコマンドの数
            0x03, 0x0C,  // コマンド 0x0C03 に対する応答であることを表す
            0x00         // 成功
        };

    NN_LOG("## Test [HCI_Reset]\n");

    size_t receivedBytes;
    char receiveBuffer[sizeof(ExpectedResult)];
    nnt::uart::bcm4356::details::TransferCommand(
            &receivedBytes,
            receiveBuffer,
            sizeof(receiveBuffer),
            pSession,
            Command,
            sizeof(Command),
            true);

    // 受信したデータ全体とサイズを確認
    ASSERT_EQ(receivedBytes, sizeof(ExpectedResult));
    ASSERT_TRUE(nnt::uart::CompareTestData(ExpectedResult,
                                receiveBuffer,
                                sizeof(ExpectedResult)));
}

/*
 * HCI_Read_Local_Version_Information コマンドを確認します。
 *
 * (memo)
 *  Jetson-TK2 上の BCM4356 が返す値。
 * ----------------
 *  0x04, 0x0E, 0x0C, 0x01, 0x01, 0x10,
 *  0x00,         // success
 *  0x06,         // HCI Version (0x06 = 4.0)
 *  0x00, 0x10,   // HCI Revision
 *  0x06,         // LMP/PAL Version
 *  0x0F, 0x00,   // Manufacturer Name (0x000F = Broadcom)
 *  0x0C, 0x61    // LMP/PAL Subversion
 * ----------------
 */
void TestHciReadLocalVersionInfo(nn::uart::PortSession* pSession) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSession);

    const char Command[] =
        {
            nnt::uart::bcm4356::details::HciPacketType_Command,
            0x01, 0x10, 0x00  // OGF=0x04, OCF=0x0001, No param
        };

    // HCI_Read_Local_Version_Information の応答全体は 15 バイト
    const size_t ExpectedResultSize = 15;

    // Jetson-TK2 上の BCM4356 が返す値の先頭 7 バイト
    // (先頭 7 バイトまでは HCI コマンドの仕様上、固定値のはず)
    const char ExpectedResultHead[] =
        {
            nnt::uart::bcm4356::details::HciPacketType_Event,
            0x0E,        // Command complete event
            0x0C,        // 後続のデータ長
            0x01,        // 次に受け付けるコマンドの数
            0x01, 0x10,  // コマンド 0x1001 に対する応答であることを表す
            0x00,        // success
        };

    NN_LOG("## Test [HCI_Read_Local_Version_Information]\n");

    size_t receivedBytes;
    char receiveBuffer[ExpectedResultSize];
    nnt::uart::bcm4356::details::TransferCommand(
            &receivedBytes,
            receiveBuffer,
            sizeof(receiveBuffer),
            pSession,
            Command,
            sizeof(Command),
            true);

    // 受信したデータ全体のサイズと先頭部分を比較
    // (8 バイト目以降は変わらない保証がなさそうなので、比較せず受信確認のみ)
    ASSERT_EQ(receivedBytes, ExpectedResultSize);
    ASSERT_TRUE(nnt::uart::CompareTestData(ExpectedResultHead,
                                receiveBuffer,
                                sizeof(ExpectedResultHead)));
}

/*
 * 対向デバイスのバッファサイズ情報を取得します。
 */
void TestHciReadBufferSize(nn::uart::PortSession* pSession,
                           nnt::uart::bcm4356::details::HciBufferInfo* pBufferInfo) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSession);
    NN_ABORT_UNLESS_NOT_NULL(pBufferInfo);

    NN_LOG("## Test [HCI_Read_Buffer_Size]\n");

    const char Command[] =
        {
            nnt::uart::bcm4356::details::HciPacketType_Command,
            0x05, 0x10, 0x00  // OGF=0x04, OCF=0x0005, No param
        };

    const size_t ExpectedResultSize = 14;

    const char ExpectedResultHead[] =
        {
            nnt::uart::bcm4356::details::HciPacketType_Event,
            0x0E,        // Command complete event
            0x0B,        // 後続のデータ長
            0x01,        // 次に受け付けるコマンドの数
            0x05, 0x10,  // コマンド 0x1005 に対する応答であることを表す
            0x00         // success
        };

    size_t receivedBytes;
    char receiveBuffer[ExpectedResultSize];
    nnt::uart::bcm4356::details::TransferCommand(
            &receivedBytes,
            receiveBuffer,
            sizeof(receiveBuffer),
            pSession,
            Command,
            sizeof(Command),
            true);
    if (::testing::Test::HasFatalFailure())
    {
        return;
    }

    // 受信したデータ全体のサイズと先頭部分を比較
    ASSERT_EQ(receivedBytes, ExpectedResultSize);
    ASSERT_TRUE(nnt::uart::CompareTestData(ExpectedResultHead,
                                receiveBuffer,
                                sizeof(ExpectedResultHead)));

    pBufferInfo->maxAclPacketSize = (receiveBuffer[8] << 8) | receiveBuffer[7];
    pBufferInfo->maxScoPacketSize = receiveBuffer[9];
    pBufferInfo->aclBufferCount   = (receiveBuffer[11] << 8) | receiveBuffer[10];
    pBufferInfo->scoBufferCount   = (receiveBuffer[13] << 8) | receiveBuffer[12];
    NN_LOG("[HCI Buffer Info]\n");
    NN_LOG("  ACL Data Packet Length: %d\n", pBufferInfo->maxAclPacketSize);
    NN_LOG("  SCO Data Packet Length: %d\n", pBufferInfo->maxScoPacketSize);
    NN_LOG("  ACL Buffer Count:       %d\n", pBufferInfo->aclBufferCount);
    NN_LOG("  SCO Buffer Count:       %d\n", pBufferInfo->scoBufferCount);
}

/*
 * BCM4356 へのコマンド送信テストを行います。
 */
void TestCommandForBcm4356(nn::uart::PortSession* pSession) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSession);

    // 受信完了イベントの登録
    ASSERT_TRUE(nn::uart::BindPortEvent(&g_ReceiveEndEvent,
                                        pSession,
                                        nn::uart::PortEventType_ReceiveEnd,
                                        0));

    typedef void (*TestCommandFunctionType)(nn::uart::PortSession*);

    // テスト関数の一覧
    const TestCommandFunctionType TestFunctions[] =
    {
        TestHciReset,
        TestHciReadLocalVersionInfo
    };

    for (const auto TestFunction : TestFunctions)
    {
        NN_ABORT_UNLESS_NOT_NULL(TestFunction);

        TestFunction(pSession);

        // 失敗していたらテストを中断
        if (::testing::Test::HasFatalFailure())
        {
            break;
        }
    }

    // 受信完了イベントの削除
    ASSERT_TRUE(nn::uart::UnbindPortEvent(&g_ReceiveEndEvent, pSession));
    nn::os::DestroySystemEvent(&g_ReceiveEndEvent);
}

/*
 * HCI UART のボーレート設定を変更します。
 */
void UpdateBaudRate(nn::uart::PortSession* pSession, uint32_t baudRateHz) NN_NOEXCEPT
{
    // HCI_Update_UART_Baud_Rate
    const char Command[] =
        {
            nnt::uart::bcm4356::details::HciPacketType_Command,
            0x18, 0xFC, 0x06,                              // OGF=0x3F, OCF=0x0018, ParamLength=6
            0x00, 0x00,                                    // Always 0
            static_cast<char>( baudRateHz        & 0xFF),  // Baud rate (Little endian)
            static_cast<char>((baudRateHz >>  8) & 0xFF),
            static_cast<char>((baudRateHz >> 16) & 0xFF),
            static_cast<char>((baudRateHz >> 24) & 0xFF),
        };
    const char ExpectedResult[] =
        {
            nnt::uart::bcm4356::details::HciPacketType_Event,
            0x0E,        // Command complete event
            0x04,        // 後続のデータ長
            0x01,        // 次に受け付けるコマンドの数
            0x18, 0xFC,  // コマンド 0xFC18 に対する応答であることを表す
            0x00         // 成功
        };

    NN_LOG("## Test [HCI_Update_UART_Baud_Rate]\n");

    size_t receivedBytes;
    char receiveBuffer[sizeof(ExpectedResult)];
    nnt::uart::bcm4356::details::TransferCommand(
            &receivedBytes,
            receiveBuffer,
            sizeof(receiveBuffer),
            pSession,
            Command,
            sizeof(Command),
            true);

    // 受信したデータ全体とサイズを確認
    ASSERT_EQ(receivedBytes, sizeof(ExpectedResult));
    ASSERT_TRUE(nnt::uart::CompareTestData(ExpectedResult,
                                receiveBuffer,
                                sizeof(ExpectedResult)));
}

/*
 * HCI UART の設定を行い、バッファ情報を取得します。
 */
void ConfigureHciUart(nnt::uart::bcm4356::details::HciBufferInfo* pOutHciBufferInfo,
                      nn::uart::PortName portName,
                      nn::uart::BaudRate baudRate,
                      nn::uart::FlowControlMode flowControlMode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutHciBufferInfo);

    NN_LOG("Using 115200 baud for setting\n");

    nn::uart::PortConfig  portConfig(
        nn::uart::BaudRate_115200,  // 設定時は 115200 baud を使用
        g_SendBuffer,
        sizeof(g_SendBuffer),
        g_ReceiveBuffer,
        sizeof(g_ReceiveBuffer)
        );
    portConfig.SetFlowControlMode(flowControlMode);

    nn::uart::PortSession portSession;
    ASSERT_TRUE(nn::uart::OpenPort(&portSession, portName, portConfig));

    // 受信完了イベントの登録
    ASSERT_TRUE(nn::uart::BindPortEvent(&g_ReceiveEndEvent,
                                        &portSession,
                                        nn::uart::PortEventType_ReceiveEnd,
                                        0));

    SetBluetoothEnabled(true);

    // バッファ情報を取得
    TestHciReadBufferSize(&portSession, pOutHciBufferInfo);
    if (::testing::Test::HasFatalFailure())
    {
        return;
    }

    // HCI UART のボーレートを変更
    UpdateBaudRate(&portSession, static_cast<int>(baudRate));
    if (::testing::Test::HasFatalFailure())
    {
        return;
    }

    // 受信完了イベントの削除
    ASSERT_TRUE(nn::uart::UnbindPortEvent(&g_ReceiveEndEvent, &portSession));
    nn::os::DestroySystemEvent(&g_ReceiveEndEvent);

    nn::uart::ClosePort(&portSession);
}

}

namespace nnt {
namespace uart {
namespace bcm4356 {

void TEST_BasicCommand(nn::uart::BaudRate baudRate,
                       nn::uart::FlowControlMode flowControlMode) NN_NOEXCEPT
{
    nn::uart::Initialize();
    nn::gpio::Initialize();

    SetBluetoothEnabled(true);

    // 使用するポートの設定
    const nn::uart::PortName PortName = nn::uart::PortName_Bluetooth;

    // ReceiveEnd イベントを使用するので、対応可否を確認
    ASSERT_TRUE(nn::uart::IsSupportedPortEvent(PortName,
                                               nn::uart::PortEventType_ReceiveEnd));

    nn::uart::PortConfig  portConfig(
        baudRate,
        g_SendBuffer,
        sizeof(g_SendBuffer),
        g_ReceiveBuffer,
        sizeof(g_ReceiveBuffer)
        );
    portConfig.SetFlowControlMode(flowControlMode);

    nn::uart::PortSession portSession;
    ASSERT_TRUE(nn::uart::OpenPort(&portSession, PortName, portConfig));

    TestCommandForBcm4356(&portSession);

    nn::uart::ClosePort(&portSession);

    SetBluetoothEnabled(false);

    nn::gpio::Finalize();
    nn::uart::Finalize();
}

void TEST_Loopback(nn::uart::BaudRate baudRate,
                   nn::uart::FlowControlMode flowControlMode) NN_NOEXCEPT
{
    nn::uart::Initialize();
    nn::gpio::Initialize();

    // 使用するポートの設定
    const nn::uart::PortName PortName = nn::uart::PortName_Bluetooth;

    // HCI UART の初期化とバッファ情報の取得
    details::HciBufferInfo hciBufferInfo;
    ConfigureHciUart(&hciBufferInfo, PortName, baudRate, flowControlMode);

    NN_LOG("Using %d baud for testing\n", static_cast<int>(baudRate));

    nn::uart::PortConfig  portConfig(
        baudRate,
        g_SendBuffer,
        sizeof(g_SendBuffer),
        g_ReceiveBuffer,
        sizeof(g_ReceiveBuffer)
        );
    portConfig.SetFlowControlMode(flowControlMode);

    nn::uart::PortSession portSession;
    ASSERT_TRUE(nn::uart::OpenPort(&portSession, PortName, portConfig));

    // ループバックテストを実行
    details::TEST_LoopbackBody(&portSession, hciBufferInfo, baudRate);

    SetBluetoothEnabled(false);

    nn::uart::ClosePort(&portSession);

    nn::gpio::Finalize();
    nn::uart::Finalize();
}

namespace details {

void TransferCommand(size_t* pReceiveBytes,
                     char* pOutBuffer,
                     size_t outBufferSize,
                     nn::uart::PortSession* pSession,
                     const char* pCommand,
                     size_t commandSize,
                     bool isDumpData) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pReceiveBytes);
    NN_ABORT_UNLESS_NOT_NULL(pOutBuffer);
    NN_ABORT_UNLESS_NOT_NULL(pSession);
    NN_ABORT_UNLESS_NOT_NULL(pCommand);

    nn::os::ClearSystemEvent(&g_ReceiveEndEvent);

    // コマンドを送信
    size_t doneBytes;
    nn::uart::Send(&doneBytes,
                   pSession,
                   pCommand,
                   commandSize);
    ASSERT_EQ(commandSize, doneBytes);

    if (isDumpData)
    {
        NN_LOG("Sent data (%d bytes):\n  ", doneBytes);
        nnt::uart::DumpBinary(pCommand, doneBytes);
    }

    // 受信完了待ち
    nn::os::WaitSystemEvent(&g_ReceiveEndEvent);

    // コマンドの応答を受信
    size_t readableLength = nn::uart::GetReadableLength(pSession);
    auto result = nn::uart::Receive(pReceiveBytes, pOutBuffer, readableLength, pSession);
    ASSERT_TRUE(result.IsSuccess());

    if (isDumpData)
    {
        NN_LOG("Received data (%d bytes):\n  ", *pReceiveBytes);
        nnt::uart::DumpBinary(pOutBuffer, *pReceiveBytes);
    }
}

}  // details

}}}
