﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnc/nn_Macro.h>
#include <nnc/nn_Result.h>
#include <nnc/uart/uart.h>
#include <nnc/uart/uart_PortApiDev.h>
#include <nnc/os/os_SystemEvent.h>

#define NNT_UART_RETURN_FALSE_UNLESS(condition)                         \
                                do                                      \
                                {                                       \
                                    if (!(condition)) { return false; } \
                                } while (NNC_STATIC_CONDITION(false))


#define NNT_UART_MEMORY_PAGE_SIZE (4 * 1024) // nn::os::MemoryPageSize の代用

// テスト対象の開発用ポート数 (testUart_Common.h 内の定義の代用)
#if   defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
    #define NNT_UART_DEV_PORT_COUNT_MAX  4
#elif defined(NN_BUILD_CONFIG_HARDWARE_NX)
    #define NNT_UART_DEV_PORT_COUNT_MAX  4
#else
    #define NNT_UART_DEV_PORT_COUNT_MAX  1
#endif

// 内部送信バッファ
char NNC_ALIGNAS(NNT_UART_MEMORY_PAGE_SIZE) s_SendBuffer[NNT_UART_MEMORY_PAGE_SIZE * 1];
// 内部受信バッファ
char NNC_ALIGNAS(NNT_UART_MEMORY_PAGE_SIZE) s_ReceiveBuffer[NNT_UART_MEMORY_PAGE_SIZE * 3];

// 最初に見つかった利用可能なポート番号を返します。
static int GetFirstAvailablePortIndex(void)
{
    for (int i = 0; i < NNT_UART_DEV_PORT_COUNT_MAX; ++i)
    {
        if (nnuartHasPortForDev(i))
        {
            return i;
        }
    }

    return -1;
}

bool nntuartPortSendReceive(void)
{
    bool        resultBool;
    nnResult    resultNn;
    size_t      resultSize;

    nnuartInitialize();
    NNT_UART_RETURN_FALSE_UNLESS(nnuartIsInitialized());

    int portIndex = GetFirstAvailablePortIndex();
    NNT_UART_RETURN_FALSE_UNLESS(portIndex >= 0);

    const nnuartBaudRate          BaudRate = nnuartBaudRate_57600;
    const nnuartFlowControlMode   FlowControlMode = nnuartFlowControlMode_None;

    nnuartPortConfigType    config;
    nnuartPortSession       session;
    nnosSystemEventType     emptyEvent;

    const char StringToSend[256] = {'A'};
          char bufferToReceive[256] = {0x00};

    // 今回使うポートや通信設定が有効であることをテスト
    NNT_UART_RETURN_FALSE_UNLESS(nnuartHasPortForDev(portIndex));
    NNT_UART_RETURN_FALSE_UNLESS(nnuartIsSupportedBaudRateForDev(portIndex, BaudRate));
    NNT_UART_RETURN_FALSE_UNLESS(nnuartIsSupportedFlowControlModeForDev(portIndex, FlowControlMode));

    nnuartInitializePortConfig(&config, BaudRate,
        s_SendBuffer, sizeof(s_SendBuffer), s_ReceiveBuffer, sizeof(s_ReceiveBuffer) );
    nnuartSetPortConfigFlowControlMode(&config, FlowControlMode);

    // オープンできることのテスト
    resultBool = nnuartOpenPortForDev(&session, portIndex, &config);
    NNT_UART_RETURN_FALSE_UNLESS(resultBool);

    // ポートイベントが登録できることをテスト
    NNT_UART_RETURN_FALSE_UNLESS(nnuartIsSupportedPortEventForDev(portIndex, nnuartPortEventType_SendBufferEmpty));
    NNT_UART_RETURN_FALSE_UNLESS(nnuartBindPortEvent(&emptyEvent, &session, nnuartPortEventType_SendBufferEmpty, 0));

    // 送信バッファは空なことをテスト
    resultSize = nnuartGetWritableLength(&session);
    NNT_UART_RETURN_FALSE_UNLESS(resultSize == sizeof(s_SendBuffer));

    // 指定したバッファがすべて送信されることのテスト
    nnuartSend(&resultSize, &session, StringToSend, sizeof(StringToSend));
    NNT_UART_RETURN_FALSE_UNLESS(resultSize == sizeof(StringToSend));

    // 送信が完全に終了することをテスト
    const int64_t TimeoutNs = 5LL * 1000LL * 1000LL * 1000LL;  // タイムアウト: 5 秒
    NNT_UART_RETURN_FALSE_UNLESS(nnosTimedWaitSystemEvent(&emptyEvent, TimeoutNs));

    // 受信バッファは空なことをテスト
    resultSize = nnuartGetReadableLength(&session);
    NNT_UART_RETURN_FALSE_UNLESS(resultSize == 0);

    // 受信が成功することのテスト
    resultSize = 100; // 0 がセットされることをテストするため、非 0 にしておく
    resultNn = nnuartReceive(&resultSize, bufferToReceive, sizeof(bufferToReceive), &session);
    NNT_UART_RETURN_FALSE_UNLESS(nnResultIsSuccess(resultNn));

    // しかし受信するものがないので受信済データの量は 0 であることをテスト
    NNT_UART_RETURN_FALSE_UNLESS(resultSize == 0);

    // ポートイベントの登録を解除できることをテスト
    NNT_UART_RETURN_FALSE_UNLESS(nnuartUnbindPortEvent(&emptyEvent, &session));
    nnosDestroySystemEvent(&emptyEvent);

    nnuartClosePort(&session);

    nnuartFinalize();
    NNT_UART_RETURN_FALSE_UNLESS(!nnuartIsInitialized());

    return true;
}

bool nntuartPortApi(void)
{
    const nnuartPortName Name = nnuartPortName_Bluetooth;

    nnuartInitialize();
    NNT_UART_RETURN_FALSE_UNLESS(nnuartIsInitialized());

    // ポートが存在する場合、通信設定が有効であることをテスト
    if (nnuartHasPort(Name))
    {
        bool        resultBool;

        const nnuartBaudRate          BaudRate = nnuartBaudRate_57600;
        const nnuartFlowControlMode   FlowControlMode = nnuartFlowControlMode_None;

        nnuartPortConfigType    config;
        nnuartPortSession       session;

        NNT_UART_RETURN_FALSE_UNLESS(nnuartIsSupportedBaudRate(Name, BaudRate));
        NNT_UART_RETURN_FALSE_UNLESS(nnuartIsSupportedFlowControlMode(Name, FlowControlMode));

        nnuartInitializePortConfig(&config, BaudRate,
            s_SendBuffer, sizeof(s_SendBuffer), s_ReceiveBuffer, sizeof(s_ReceiveBuffer) );
        nnuartSetPortConfigFlowControlMode(&config, FlowControlMode);

        // オープンできることのテスト
        resultBool = nnuartOpenPort(&session, Name, &config);
        NNT_UART_RETURN_FALSE_UNLESS(resultBool);

        // ポートイベント関連のテスト
        const nnuartPortEventType PortEventList[] =
        {
            nnuartPortEventType_SendBufferEmpty,
            nnuartPortEventType_SendBufferReady,
            nnuartPortEventType_ReceiveBufferReady,
            nnuartPortEventType_ReceiveEnd
        };
        const size_t PortEventNum = sizeof(PortEventList) / sizeof(PortEventList[0]);
        for (size_t i = 0; i < PortEventNum; ++i)
        {
            const nnuartPortEventType portEvent = PortEventList[i];

            // ポートイベントの対応確認
            if (!nnuartIsSupportedPortEvent(Name, portEvent))
            {
                continue;
            }

            // ポートイベントの登録/解除
            const int Threshold = 1;  // 事前条件を満たす適当な閾値
            nnosSystemEventType systemEvent;
            NNT_UART_RETURN_FALSE_UNLESS(nnuartBindPortEvent(&systemEvent, &session, portEvent, Threshold));
            NNT_UART_RETURN_FALSE_UNLESS(nnuartUnbindPortEvent(&systemEvent, &session));
            nnosDestroySystemEvent(&systemEvent);
        }

        nnuartClosePort(&session);
    }

    nnuartFinalize();
    NNT_UART_RETURN_FALSE_UNLESS(!nnuartIsInitialized());

    return true;
}

bool nntuartLibInit(void)
{
    // Initialize によって初期化状態になることをテスト
    nnuartInitialize();
    NNT_UART_RETURN_FALSE_UNLESS(nnuartIsInitialized());

    // Finalize によって未初期化状態になることをテスト
    nnuartFinalize();
    NNT_UART_RETURN_FALSE_UNLESS(!nnuartIsInitialized());

    return true;
}
