﻿using System;
using System.Text;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using ZarfCreator.Parser;
using ZarfCreator.ZarfDefinitionData;
using System.Linq;
using System.Collections.Specialized;

namespace ZarfCreatorTest.ParserTest
{
    /// <summary>
    /// ZarfDefinitionParseTest の概要の説明
    /// </summary>
    [TestClass]
    public class ZarfDefinitionParseTest
    {
        private ZarfDefinitionParser parser;

        private List<object> relatedZarfInfo;
        private List<RelatedZarfInfo> parsedRelatedZarfInfo;

        private List<object> commandList;
        private List<InstructionCommandInfo> parsedCommandList;

        private Dictionary<string, object> instructionInfo;
        private InstructionInfo parsedInstructionInfo;

        #region 共通データの準備
        public ZarfDefinitionParseTest()
        {
            this.parser = new ZarfDefinitionParser("");

            this.SetRelatedZarf();
            this.SetInstructions();
        }

        private void SetRelatedZarf()
        {
            var relatedZarf1 = new Dictionary<string, object>()
            {
                {"Target", "TargetZarf"}
            };

            var relatedZarf2 = new Dictionary<string, object>()
            {
                {"Target", "Target ZarfDefinition"},
                {"Versions", new List<object>()
                    {
                        "0.3.0"
                    }
                }
            };

            this.relatedZarfInfo = new List<object>() { relatedZarf1, relatedZarf2 };
            this.parsedRelatedZarfInfo = new RelatedZarfParser().Parse(this.relatedZarfInfo).ToList();
        }

        private void SetInstructions()
        {
            var command1 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Parser" },
                { "CmdSpecifier", "unzip" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-deploy-bin}\\${File1}"
                    }
                }
            };

            var command2 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Parser" },
                { "CmdSpecifier", "replacementCopy" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-temp}\\NintendoSDK",
                        "{release-installdir}"
                    }
                }
            };

            var command3 = new Dictionary<string, object>()
            {
                { "Handler", "External-Exe" },
                { "CmdSpecifier", "msiexec" },
                { "CmdArgs", new List<object>()
                    {
                        "/i",
                        @"{fulldir-deploy-bin}\\Setup.msi",
                        "/qn",
                        "/promptrestart"
                    }
                },
                { "SuccessReturnCodes", new List<object>() { "0" }}
            };

            this.commandList = new List<object>() { command1, command2, command3 };
            this.parsedCommandList = new InstructionCommandParser().Parse(this.commandList).ToList();


            this.instructionInfo = new Dictionary<string, object>()
            {
                { "Install", this.commandList },
                { "Uninstall", this.commandList },
                { "InstallDirectory", "NintendoSDK" }
            };

            this.parsedInstructionInfo = new InstructionParser().Parse(this.instructionInfo);
        }
        #endregion

        [TestMethod]
        public void ParseSuccessTest1()
        {
            var data = new Dictionary<string, object>()
            {
                { "Name", "test" },
                { "Type", "Public" },
                { "Family", "NintendoSDK" },
                { "Department", "Nintendo" },
                { "Description", new Dictionary<string, object> { { "En", "Test" } } },
                { "VersionFile", "0.3.0" },
                { "Instructions", this.instructionInfo },
            };

            var result = this.parser.Parse(data);
            Assert.AreEqual(result.Name, "test");
            Assert.AreEqual(result.Type, "Public");
            Assert.AreEqual(result.Family, "NintendoSDK");
            Assert.AreEqual(result.Department, "Nintendo");
            Assert.AreEqual(result.Description, new TextInfo { English = "Test" });
            Assert.AreEqual(result.VersionFile, "0.3.0");
            Assert.AreEqual(result.Instruction, this.parsedInstructionInfo);
            CollectionAssert.AreEquivalent(result.Platforms, new StringCollection() { "NintendoSDK" });
            Assert.IsNull(result.Eula);

            var data2 = new Dictionary<string, object>()
            {
                { "Name", "hoge" },
                { "Type", "Private" },
                { "Description", new Dictionary<string, object>
                    {
                        { "Ja", "テストです。" },
                        { "En", "Test" }
                    }
                },
                { "VersionFile", @"D:\home\NintendoSDK\Version.h" },
                { "Family", "NintendoSDK Hogehoge Package" },
                { "Department", "PSEG" },
                { "SetupFiles", new List<object>()
                    {
                        "hoge.bat"
                    }
                },
                { "Dependencies", this.relatedZarfInfo },
                { "Compatibilities", this.relatedZarfInfo },
                { "Instructions", this.instructionInfo },
                { "Platforms", new List<object>()
                    {
                        "PC", "WiiU", "3DS"
                    }
                },
                { "RuntimeDependencies", this.relatedZarfInfo },
                { "Eula", new Dictionary<string, object>
                    {
                        { "Ja", "Eula です。" },
                        { "En", "Eula." }
                    }
                }
            };

            result = this.parser.Parse(data2);
            Assert.AreEqual(result.Name, "hoge");
            Assert.AreEqual(result.Type, "Private");
            Assert.AreEqual(result.Description, new TextInfo { Japanese = "テストです。", English = "Test" });
            Assert.AreEqual(result.VersionFile, @"D:\home\NintendoSDK\Version.h");
            Assert.AreEqual(result.Family, "NintendoSDK Hogehoge Package");
            Assert.AreEqual(result.Department, "PSEG");
            CollectionAssert.AreEquivalent(result.SetupFiles.ToList(), new List<string>() { "hoge.bat" });
            CollectionAssert.AreEquivalent(result.Platforms, new StringCollection() { "PC", "WiiU", "3DS" });

            CollectionAssert.AreEquivalent(result.Dependencies.ToList(), this.parsedRelatedZarfInfo);
            CollectionAssert.AreEquivalent(result.Compatibilities.ToList(), this.parsedRelatedZarfInfo);
            CollectionAssert.AreEquivalent(result.RuntimeDependencies.ToList(), this.parsedRelatedZarfInfo);
            Assert.AreEqual(result.Instruction, this.parsedInstructionInfo);
            Assert.AreEqual(result.Eula, new TextInfo { Japanese = "Eula です。", English = "Eula." });
        }

        [TestMethod]
        public void ParseContentsLocaleTest()
        {
            // デフォルトのContentsLocaleを設定
            var parser = new ZarfDefinitionParser("", "ja");

            var data = new Dictionary<string, object>()
            {
                { "Name", "test" },
                { "Type", "Public" },
                { "Family", "NintendoSDK" },
                { "Department", "Nintendo" },
                { "Description", new Dictionary<string, object> { { "En", "Test" } } },
                { "VersionFile", "0.3.0" },
                { "Instructions", this.instructionInfo },
            };

            var result = parser.Parse(data);
            // 明示的に指定しない場合はデフォルト値になることを期待。
            Assert.AreEqual("ja", result.ContentsLocale);

            var data2 = new Dictionary<string, object>()
            {
                { "Name", "test" },
                { "Type", "Public" },
                { "Family", "NintendoSDK" },
                { "Department", "Nintendo" },
                { "Description", new Dictionary<string, object> { { "En", "Test" } } },
                { "VersionFile", "0.3.0" },
                { "Instructions", this.instructionInfo },

                { "ContentsLocale", "All" },
            };

            result = parser.Parse(data2);
            // "All" にした場合は、ContentsLocale は nullになることを期待。
            Assert.IsNull(result.ContentsLocale);
        }

        [TestMethod]
        public void ParseFailByNotMeetRequirementsTest()
        {
            Console.Error.WriteLine("---");
            var data = new Dictionary<string, object>() { };

            TestUtil.ExpectException<FormatException>(() => this.parser.Parse(data), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data2 = new Dictionary<string, object>()
            {
                { "Type", "Public" },
                { "Family", "NintendoSDK" },
                { "Department", "Nintendo" },
                { "Description", new Dictionary<string, object> { { "En", "Test" } } },
                { "VersionFile", "0.3.0" },
                { "Compatibilities", this.relatedZarfInfo },
                { "Instructions", this.instructionInfo },
            };

            // Name が不足
            TestUtil.ExpectException<FormatException>(() => this.parser.Parse(data2), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data3 = new Dictionary<string, object>()
            {
                { "Name", "hoge" },
                { "Family", "test" },
                { "Department", "NOA" },
                { "Description", new Dictionary<string, object> { { "Ja", "テスト" } } },
                { "VersionFile", "hoge.bat" },
                { "Compatibilities", this.relatedZarfInfo },
                { "Instructions", this.instructionInfo },
                { "RuntimeDependencies", this.relatedZarfInfo }
            };

            // Type が不足
            TestUtil.ExpectException<FormatException>(() => this.parser.Parse(data3), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data4 = new Dictionary<string, object>()
            {
                { "Dependencies", this.relatedZarfInfo },
                { "Compatibilities", this.relatedZarfInfo },
                { "Name", "test" },
                { "Type", "Public" },
                { "Department", "Nintendo" },
                { "Description", new Dictionary<string, object> { { "En", "Test" } } },
                { "VersionFile", "0.3.0" },
                { "Instructions", this.instructionInfo },
            };

            // Family が不足
            TestUtil.ExpectException<FormatException>(() => this.parser.Parse(data4), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data5 = new Dictionary<string, object>()
            {
                { "Name", "test" },
                { "Type", "Public" },
                { "Family", "NintendoSDK" },
                { "Description", new Dictionary<string, object> { { "En", "Test" } } },
                { "SetupFiles", new List<object>()
                    {
                        "hoge.bat"
                    }
                },
                { "VersionFile", "0.3.0" },
                { "Instructions", this.instructionInfo },
            };

            // Department が不足
            TestUtil.ExpectException<FormatException>(() => this.parser.Parse(data5), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data6 = new Dictionary<string, object>()
            {
                { "Name", "test" },
                { "Type", "Public" },
                { "Family", "NintendoSDK" },
                { "Department", "Nintendo" },
                { "VersionFile", "0.3.0" },
                { "Instructions", this.instructionInfo },
                { "Platforms", new List<object>()
                    {
                        "PC", "WiiU", "3DS"
                    }
                },
                { "RuntimeDependencies", this.relatedZarfInfo }
            };

            // Description が不足
            TestUtil.ExpectException<FormatException>(() => this.parser.Parse(data6), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data7 = new Dictionary<string, object>()
            {
                { "Name", "test" },
                { "Type", "Public" },
                { "Family", "NintendoSDK" },
                { "Compatibilities", this.relatedZarfInfo },
                { "Department", "Nintendo" },
                { "Description", new Dictionary<string, object> { { "En", "Test" } } },
                { "Instructions", this.instructionInfo },
                { "RuntimeDependencies", this.relatedZarfInfo }
            };

            // VersionFile が不足
            TestUtil.ExpectException<FormatException>(() => this.parser.Parse(data7), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data8 = new Dictionary<string, object>()
            {
                { "Name", "test" },
                { "Type", "Public" },
                { "Family", "NintendoSDK" },
                { "Department", "Nintendo" },
                { "Description", new Dictionary<string, object> { { "En", "Test" } } },
                { "VersionFile", "0.3.0" },
                { "Platforms", new List<object>()
                    {
                        "PC", "WiiU", "3DS"
                    }
                },
            };

            // Instructions が不足
            TestUtil.ExpectException<FormatException>(() => this.parser.Parse(data8), "Required items have not been completed.");
        }

        /// <summary>
        /// typo している場合
        /// </summary>
        [TestMethod]
        public void ParseFailByTypographicalErrorTest()
        {
            Console.Error.WriteLine("---");
            var data1 = new Dictionary<string, object>()
            {
                { "Name", "Test Zarf" },
                { "Type", "Public" },
                { "Family", "NintendoSDK" },
                { "Department", "Nintendo" },
                { "Description", new Dictionary<string, object> { { "En", "Test" } } },
                { "VersionFile", "0.3.0" },
                { "Compatibility", this.relatedZarfInfo },
                { "Instructions", this.instructionInfo },
            };

            TestUtil.ExpectException<FormatException>(() => this.parser.Parse(data1), "Unknown key has specified.");

            Console.Error.WriteLine("---");
            var data2 = new Dictionary<string, object>()
            {
                { "Name", "Test Zarf" },
                { "Type", "Public" },
                { "Family", "NintendoSDK" },
                { "Department", "Nintendo" },
                { "Description", new Dictionary<string, object> { { "En", "Test" } } },
                { "VersionFileHoge", "0.3.0" },
                { "Compatibilities", this.relatedZarfInfo },
                { "Instructions", this.instructionInfo },
            };

            TestUtil.ExpectException<FormatException>(() => this.parser.Parse(data2), "Unknown key has specified.");

        }
    }
}
