﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using VsSolutionLibrary;
using System.Collections.Generic;

namespace VsSolutionLibraryTest
{
    /// <summary>
    /// .sln ファイルをパースするテストです。
    /// </summary>
    [TestClass]
    public class TypicalParseTest
    {
        private const string SolutionText = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""Project0"", ""Project0\Project0.vcxproj"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Project(""{FAE04EC0-301F-11D3-BF4B-00C04F79EFBC}"") = ""Project 1"", ""全角\Project 1.csproj"", ""{22222222-2222-2222-2222-222222222222}""
    ProjectSection(ProjectDependencies) = postProject
        {11111111-1111-1111-1111-111111111111} = {11111111-1111-1111-1111-111111111111}
    EndProjectSection
EndProject
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Siglo 0 = Debug|Siglo 0
        Debug|Siglo 1 = Debug|Siglo 1
        Release|Siglo 0 = Release|Siglo 0
        Release|Siglo 1 = Release|Siglo 1
    EndGlobalSection
    GlobalSection(ProjectConfigurationPlatforms) = postSolution
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 0.ActiveCfg = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 0.Build.0 = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 1.ActiveCfg = Debug|Siglo 1
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 1.Build.0 = Debug|Siglo 1
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 0.ActiveCfg = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 1.ActiveCfg = Release|Siglo 1
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 1.Build.0 = Release|Siglo 1
        {22222222-2222-2222-2222-222222222222}.Debug|Siglo 0.ActiveCfg = Debug|Siglo 0
        {22222222-2222-2222-2222-222222222222}.Debug|Siglo 0.Build.0 = Debug|Siglo 0
        {22222222-2222-2222-2222-222222222222}.Debug|Siglo 1.ActiveCfg = Debug|Siglo 1
        {22222222-2222-2222-2222-222222222222}.Debug|Siglo 1.Build.0 = Debug|Siglo 1
        {22222222-2222-2222-2222-222222222222}.Release|Siglo 0.ActiveCfg = Release|Siglo 0
        {22222222-2222-2222-2222-222222222222}.Release|Siglo 0.Build.0 = Release|Siglo 0
        {22222222-2222-2222-2222-222222222222}.Release|Siglo 1.ActiveCfg = Release|Siglo 1
        {22222222-2222-2222-2222-222222222222}.Release|Siglo 1.Build.0 = Release|Siglo 1
    EndGlobalSection
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";

        [TestMethod]
        public void TestProjects()
        {
            VsSolutionLibrary.VsSolution solution = new VsSolutionLibrary.VsSolution(SolutionText);

            List<Guid> expected = new List<Guid>();
            expected.Add(new Guid("{11111111-1111-1111-1111-111111111111}"));
            expected.Add(new Guid("{22222222-2222-2222-2222-222222222222}"));

            Assert.IsTrue(Enumerable.SequenceEqual<Guid>(expected, solution.Projects));
        }

        [TestMethod]
        public void TestVersion()
        {
            VsSolutionLibrary.VsSolution solution = new VsSolutionLibrary.VsSolution(SolutionText);

            Assert.AreEqual(VsSolutionVersion.VisualStudio2012, solution.Version);
        }

        [TestMethod]
        public void TestConfigurations()
        {
            VsSolutionLibrary.VsSolution solution = new VsSolutionLibrary.VsSolution(SolutionText);

            List<VsConfigurationPair> expected = new List<VsConfigurationPair>();
            expected.Add(new VsConfigurationPair("Debug", "Siglo 0"));
            expected.Add(new VsConfigurationPair("Debug", "Siglo 1"));
            expected.Add(new VsConfigurationPair("Release", "Siglo 0"));
            expected.Add(new VsConfigurationPair("Release", "Siglo 1"));

            Assert.IsTrue(Enumerable.SequenceEqual<VsConfigurationPair>(expected, solution.Configurations));
        }

        [TestMethod]
        public void TestSolutionProperties()
        {
            VsSolutionLibrary.VsSolution solution = new VsSolutionLibrary.VsSolution(SolutionText);

            Assert.AreEqual(VsSolutionVersion.VisualStudio2012, solution.Version);
        }

        [TestMethod]
        public void TestToString()
        {
            VsSolutionLibrary.VsSolution solution = new VsSolutionLibrary.VsSolution(SolutionText);

            Assert.IsTrue(TestUtility.CompareSolutionString(solution.ToString(), SolutionText));
        }

        [TestMethod]
        public void TestGetVsProjectProperty()
        {
            VsSolutionLibrary.VsSolution solution = new VsSolutionLibrary.VsSolution(SolutionText);

            {
                Guid project = new Guid("{11111111-1111-1111-1111-111111111111}");

                VsSolutionProjectProperty expected = new VsSolutionProjectProperty(VsProjectType.VisualC, "Project0", "Project0\\Project0.vcxproj", null);
                Assert.AreEqual(expected, solution.GetProjectProperty(project));
            }

            {
                Guid project = new Guid("{22222222-2222-2222-2222-222222222222}");
                Guid[] dependencies = new Guid[] { new Guid("{11111111-1111-1111-1111-111111111111}") };

                VsSolutionProjectProperty expected = new VsSolutionProjectProperty(VsProjectType.CSharp, "Project 1", "全角\\Project 1.csproj", dependencies);
                Assert.AreEqual(expected, solution.GetProjectProperty(project));
            }
        }

        [TestMethod]
        public void TestGetProjectConfiguration()
        {
            VsSolutionLibrary.VsSolution solution = new VsSolutionLibrary.VsSolution(SolutionText);

            foreach (Guid projectGuid in new Guid[] { new Guid("{11111111-1111-1111-1111-111111111111}"), new Guid("{22222222-2222-2222-2222-222222222222}") })
            {
                foreach (string configuration in new string[] { "Debug", "Release" })
                {
                    foreach (string platform in new string[] { "Siglo 0", "Siglo 1" })
                    {
                        VsConfigurationPair solutionConfiguration = new VsConfigurationPair(configuration, platform);
                        VsProjectConfiguration expected;

                        if (projectGuid == new Guid("{11111111-1111-1111-1111-111111111111}") && configuration == "Release" && platform == "Siglo 0")
                        {
                            expected = new VsProjectConfiguration(new VsConfigurationPair("Debug", platform), false);
                        }
                        else
                        {
                            expected = new VsProjectConfiguration(new VsConfigurationPair(configuration, platform), true);
                        }

                        Assert.AreEqual(expected, solution.GetProjectConfiguration(solutionConfiguration, projectGuid));
                    }
                }
            }
        }
    }
}
