﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using VsSolutionLibrary;

namespace VsSolutionLibraryTest
{
    /// <summary>
    /// VsSolutionLibrary の異常系テストです。
    /// </summary>
    [TestClass]
    public class AbnormalSequenceTest
    {
        /// <summary>
        /// VsSolution のコンストラクタ引数に null を指定して例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentNullException))]
        public void TestConstructor_ArgumentNull()
        {
            VsSolution solution = new VsSolution(null);
        }

        /// <summary>
        /// VsSolution のコンストラクタに無効なフォーマットの文字列を指定し、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentException))]
        public void TestConstructor_WrongFormat()
        {
            VsSolution solution = new VsSolution("Wrong format Text");
        }

        /// <summary>
        /// AddVsProject() の引数に null を指定して例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentNullException))]
        public void TestAddProject_ArgumentNull()
        {
            VsSolution solution = new VsSolution();
            solution.AddProject(new Guid("{11111111-1111-1111-1111-111111111111}"), null);
        }

        /// <summary>
        /// 同じ GUID の VsProject を2つ追加して、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentException))]
        public void TestAddProject_Duplication()
        {
            VsSolution solution = new VsSolution();
            VsSolutionProjectProperty property = new VsSolutionProjectProperty(VsProjectType.VisualC, "Name", "Path", null);
            solution.AddProject(new Guid("{11111111-1111-1111-1111-111111111111}"), property);
            solution.AddProject(new Guid("{11111111-1111-1111-1111-111111111111}"), property);
        }

        /// <summary>
        /// 存在しない GUID の VsProject を削除するテストです。例外は発生しません。
        /// </summary>
        [TestMethod]
        public void TestRemoveProject_KeyNotFound()
        {
            VsSolution solution = new VsSolution();
            VsSolutionProjectProperty property = new VsSolutionProjectProperty(VsProjectType.VisualC, "Name", "Path", null);
            solution.AddProject(new Guid("{11111111-1111-1111-1111-111111111111}"), property);
            solution.RemoveProject(new Guid("{22222222-2222-2222-2222-222222222222}"));
        }

        /// <summary>
        /// 存在しない GUID の VsProjectProperty を取得して、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentException))]
        public void TestGetProjectProperty_KeyNotFound()
        {
            VsSolution solution = new VsSolution();
            VsSolutionProjectProperty property = new VsSolutionProjectProperty(VsProjectType.VisualC, "Name", "Path", null);
            solution.AddProject(new Guid("{11111111-1111-1111-1111-111111111111}"), property);
            solution.GetProjectProperty(new Guid("{22222222-2222-2222-2222-222222222222}"));
        }

        /// <summary>
        /// 存在しない GUID の VsProjectProperty を設定して、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentException))]
        public void TestSetProjectProperty_KeyNotFound()
        {
            VsSolution solution = new VsSolution();
            VsSolutionProjectProperty property = new VsSolutionProjectProperty(VsProjectType.VisualC, "Name", "Path", null);
            solution.SetProjectProperty(new Guid("{11111111-1111-1111-1111-111111111111}"), property);
        }

        /// <summary>
        /// AddSolutionConfiguration() の引数に null を指定して、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentNullException))]
        public void TestAddSolutionConfiguration_ArgumentNull()
        {
            VsSolution solution = new VsSolution();
            solution.AddSolutionConfiguration(null);
        }

        /// <summary>
        /// 同じ内容のソリューション構成を2つ追加して、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentException))]
        public void TestAddSolutionConfiguration_Duplication()
        {
            VsSolution solution = new VsSolution();
            VsConfigurationPair solutionConfiguration = new VsConfigurationPair("Debug", "Siglo 0");
            solution.AddSolutionConfiguration(solutionConfiguration);
            solution.AddSolutionConfiguration(solutionConfiguration);
        }

        /// <summary>
        /// 存在しないソリューション構成を削除するテストです。例外は発生しません。
        /// </summary>
        [TestMethod]
        public void TestRemoveSolutionConfiguration_KeyNotFound()
        {
            VsSolution solution = new VsSolution();
            VsConfigurationPair solutionConfiguration = new VsConfigurationPair("Debug", "Siglo 0");
            solution.RemoveSolutionConfiguration(solutionConfiguration);
        }

        /// <summary>
        /// GetProjectConfiguration() の引数に null を渡して、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentNullException))]
        public void TestGetProjectConfiguration_ArgumentNull()
        {
            VsSolution solution = new VsSolution();
            solution.GetProjectConfiguration(null, new Guid("{11111111-1111-1111-1111-111111111111}"));
        }

        /// <summary>
        /// 存在しないソリューション構成・プロジェクトのペアに対してプロジェクト構成を取得し、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentException))]
        public void TestGetProjectConfiguration_KeyNotFound()
        {
            VsSolution solution = new VsSolution();

            VsConfigurationPair solutionConfiguration = new VsConfigurationPair("Debug", "Siglo 0");
            solution.AddSolutionConfiguration(solutionConfiguration);

            solution.GetProjectConfiguration(solutionConfiguration, new Guid("{11111111-1111-1111-1111-111111111111}"));
        }

        /// <summary>
        /// SetProjectConfiguration() の引数に null を渡して、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentNullException))]
        public void TestSetProjectConfiguration_ArgumentNull1()
        {
            VsSolution solution = new VsSolution();

            VsProjectConfiguration projectConfiguration = new VsProjectConfiguration(new VsConfigurationPair("Debug", "Siglo 0"), true);

            solution.SetProjectConfiguration(null, new Guid("{11111111-1111-1111-1111-111111111111}"), projectConfiguration);
        }

        /// <summary>
        /// SetProjectConfiguration() の引数に null を渡して、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentNullException))]
        public void TestSetProjectConfiguration_ArgumentNull2()
        {
            VsSolution solution = new VsSolution();

            VsConfigurationPair solutionConfiguration = new VsConfigurationPair("Debug", "Siglo 0");
            solution.AddSolutionConfiguration(solutionConfiguration);

            solution.SetProjectConfiguration(solutionConfiguration, new Guid("{11111111-1111-1111-1111-111111111111}"), null);
        }

        /// <summary>
        /// 存在しないソリューション構成・プロジェクトのペアに対してプロジェクト構成を設定し、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentException))]
        public void TestSetProjectConfiguration_KeyNotFound()
        {
            VsSolution solution = new VsSolution();

            VsConfigurationPair solutionConfiguration = new VsConfigurationPair("Debug", "Siglo 0");
            solution.AddSolutionConfiguration(solutionConfiguration);

            VsProjectConfiguration projectConfiguration = new VsProjectConfiguration(new VsConfigurationPair("Debug", "Siglo 0"), true);

            solution.SetProjectConfiguration(solutionConfiguration, new Guid("{11111111-1111-1111-1111-111111111111}"), projectConfiguration);
        }

        /// <summary>
        /// AddNestRelation() の引数に null を渡して、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentNullException))]
        public void TestAddNestRelation_ArgumentNull()
        {
            VsSolution solution = new VsSolution();
            solution.AddNestRelation(null);
        }

        /// <summary>
        /// RemoveNestRelation() の引数に null を渡して、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentNullException))]
        public void TestRemoveNestRelation_ArgumentNull()
        {
            VsSolution solution = new VsSolution();
            solution.RemoveNestRelation(null);
        }

        /// <summary>
        /// RemoveNestRelation() の引数に存在しない VsNestRelation を指定して、例外を発生させるテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(ArgumentException))]
        public void TestRemoveNestRelation_RelationNotExist()
        {
            VsSolution solution = new VsSolution();

            Guid parent = new Guid("{11111111-1111-1111-1111-111111111111}");
            Guid child = new Guid("{22222222-2222-2222-2222-222222222222}");
            VsNestRelation relation = new VsNestRelation(parent, child);

            VsSolutionProjectProperty parentProperty = new VsSolutionProjectProperty(VsProjectType.VisualC, "Name0", "Path0", null);
            VsSolutionProjectProperty childProperty = new VsSolutionProjectProperty(VsProjectType.SolutionFolder, "Name1", "Path1", null);

            solution.AddProject(parent, parentProperty);
            solution.AddProject(child, childProperty);

            solution.AddNestRelation(relation);

            VsNestRelation falseRelation = new VsNestRelation(child, parent);
            solution.RemoveNestRelation(falseRelation);
        }
    }
}
