﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO.Ports;
using System.Diagnostics;

namespace UartTestHost
{
    /// <summary>
    /// テスト対象となるボーレート設定値を集めた列挙体です。
    /// </summary>
    /// <remarks>
    /// ハンドシェイクにおいてターゲットと共有するテストパラメータ内で使用します。
    /// そのため、定義値とデータサイズは密にターゲット側テストプログラムの同名型と一致している必要があります。
    /// </remarks>
    public enum TestBaudRateType : byte
    {
        Baud_57600 = 0x01,
        Baud_115200 = 0x02
    }

    /// <summary>
    /// テスト対象となるフロー制御モード設定値を集めた列挙体です。
    /// </summary>
    /// <remarks>
    /// ハンドシェイクにおいてターゲットと共有するテストパラメータ内で使用します。
    /// そのため、定義値とデータサイズは密にターゲット側テストプログラムの同名型と一致している必要があります。
    /// </remarks>
    public enum TestFlowControlMode : byte
    {
        FlowControlMode_None = 0x01,
        FlowControlMode_Hardware = 0x02
    }

    public class SerialTypesUtil
    {
        /// <summary>
        /// TestBaudRateType 型のボーレートタイプを int に変換します。
        /// </summary>
        /// <param name="baudRateType">ボーレートタイプ</param>
        /// <returns>ボーレート</returns>
        public static int ToIntBaudRate(TestBaudRateType baudRateType)
        {
            switch (baudRateType)
            {
                case TestBaudRateType.Baud_57600:
                    {
                        return 57600;
                    }
                case TestBaudRateType.Baud_115200:
                    {
                        return 115200;
                    }
                default:
                    {
                        Trace.Fail("Invalid value as TestBaudRateType.");
                        return 0;
                    }
            }
        }

        /// <summary>
        /// TestFlowControlMode 型のフロー制御モード値を System.IO.Ports.Handshake 型に変換します。
        /// </summary>
        /// <param name="flowControlMode">フロー制御モード値</param>
        /// <returns>System.IO.Ports.Handshake 型のフロー制御モード</returns>
        public static Handshake ToHandshake(TestFlowControlMode flowControlMode)
        {
            switch (flowControlMode)
            {
                case TestFlowControlMode.FlowControlMode_None:
                    {
                        return Handshake.None;
                    }
                case TestFlowControlMode.FlowControlMode_Hardware:
                    {
                        return Handshake.RequestToSend;
                    }
                default:
                    {
                        Trace.Fail("Invalid value as TestFlowControlMode.");
                        return Handshake.None;
                    }
            }
        }
    }
}
