﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunnerTest.TestStatementInfoTest
{
    using System.Collections.Generic;
    using System.Linq;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using TestRunner;
    using TestRunner.Executer;

    /// <summary>
    /// [TestRunner.TestStatementInfo]::ProduceTestStatementInfos系用のテスト関数を定義します。
    /// </summary>
    [TestClass]
    public class ProduceTestStatementInfosTest
    {
        /// <summary>
        /// 不変条件: ステートメント ID は連番
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosInvariantTest001()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>());

            var statements = statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>(),
                new Dictionary<string, object>(),
                new Dictionary<string, object>()
            });

            var statementIds = statements.Select(x => x.GetListContext(new ListContext()).StatementId)
                                         .ToArray();

            Assert.AreEqual(3, statementIds.Count());
            Assert.AreEqual(1u, statementIds[0]);
            Assert.AreEqual(1u, statementIds[1] - statementIds[0]);
            Assert.AreEqual(1u, statementIds[2] - statementIds[1]);
        }

        /// <summary>
        /// 不変条件: プラットフォーム ID は指定順と一致
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosInvariantTest002()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Platforms", new List<object>() { "Foo", "Bar", "Baz" } }
            });

            var platforms = new[] { "Bar", "Baz", "Foo" };

            var statements = statement.ProduceTestStatementInfosByPlatforms(platforms);

            var platformIds = statements.Select(x => x.GetListContext(new ListContext()).PlatformId)
                                        .ToArray();

            Assert.AreEqual(3, platformIds.Count());
            Assert.AreEqual(1u, platformIds[0]);
            Assert.AreEqual(1u, platformIds[1] - platformIds[0]);
            Assert.AreEqual(1u, platformIds[2] - platformIds[1]);

            var platformNames = statements.Select(x => x.GetListContext(new ListContext()).PlatformName)
                                          .ToArray();

            Assert.AreEqual(platforms[0], platformNames[0]);
            Assert.AreEqual(platforms[1], platformNames[1]);
            Assert.AreEqual(platforms[2], platformNames[2]);
        }

        /// <summary>
        /// 不変条件: ビルドタイプ ID は指定順と一致
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosInvariantTest003()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "BuildTypes", new List<object>() { "Foo", "Bar", "Baz" } }
            });

            var buildTypes = new[] { "Bar", "Baz", "Foo" };

            var statements = statement.ProduceTestStatementInfosByBuildTypes(buildTypes);

            var buildTypeIds = statements.Select(x => x.GetListContext(new ListContext()).BuildTypeId)
                                         .ToArray();

            Assert.AreEqual(3, buildTypeIds.Count());
            Assert.AreEqual(1u, buildTypeIds[0]);
            Assert.AreEqual(1u, buildTypeIds[1] - buildTypeIds[0]);
            Assert.AreEqual(1u, buildTypeIds[2] - buildTypeIds[1]);

            var buildTypeNames = statements.Select(x => x.GetListContext(new ListContext()).BuildTypeName)
                                           .ToArray();

            Assert.AreEqual(buildTypes[0], buildTypeNames[0]);
            Assert.AreEqual(buildTypes[1], buildTypeNames[1]);
            Assert.AreEqual(buildTypes[2], buildTypeNames[2]);
        }

        /// <summary>
        /// 境界値分析: ステートメント中の Platforms 要素が指定されたプラットフォームを含まない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest001()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Platforms", new List<object>() { "Foo", "Bar", "Baz" } }
            });

            Assert.AreEqual(0, statement.ProduceTestStatementInfosByPlatforms(new List<string>()).Count);

            Assert.AreEqual(0, statement.ProduceTestStatementInfosByPlatforms(new List<string>() { "Qux" }).Count);

            Assert.AreEqual(0, statement.ProduceTestStatementInfosByPlatforms(new List<string>() { "Win32" }).Count);

            var listContexts = statement.ProduceTestStatementInfosByPlatforms(new List<string>() { "Bar" })
                                        .Select(x => x.GetListContext(new ListContext()))
                                        .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual("Bar", listContexts[0].PlatformName);

            statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Platforms", new List<object>() { "Foo", "Win64_VS2015", "Bar" } }
            });

            listContexts = statement.ProduceTestStatementInfosByPlatforms(new List<string>())
                                    .Select(x => x.GetListContext(new ListContext()))
                                    .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual(string.Empty, listContexts[0].PlatformName);

            Assert.AreEqual(0, statement.ProduceTestStatementInfosByPlatforms(new List<string>() { "Qux" }).Count);

            listContexts = statement.ProduceTestStatementInfosByPlatforms(new List<string>() { "Win64_VS2015" })
                                    .Select(x => x.GetListContext(new ListContext()))
                                    .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual("Win64_VS2015", listContexts[0].PlatformName);

            listContexts = statement.ProduceTestStatementInfosByPlatforms(new List<string>() { "Bar" })
                                        .Select(x => x.GetListContext(new ListContext()))
                                        .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual("Bar", listContexts[0].PlatformName);

            statement = new TestStatementInfo(new Dictionary<string, object>());

            listContexts = statement.ProduceTestStatementInfosByPlatforms(new List<string>())
                                    .Select(x => x.GetListContext(new ListContext()))
                                    .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual(string.Empty, listContexts[0].PlatformName);

            Assert.AreEqual(0, statement.ProduceTestStatementInfosByPlatforms(new List<string>() { "Qux" }).Count);

            listContexts = statement.ProduceTestStatementInfosByPlatforms(new List<string>() { "Win32" })
                                    .Select(x => x.GetListContext(new ListContext()))
                                    .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual("Win32", listContexts[0].PlatformName);
        }

        /// <summary>
        /// 境界値分析: ステートメントが Constants 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest002()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Constants", new Dictionary<string, object>() { { "FOO", "BAR" } } }
            });

            var nodeInfo = statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First();

            Assert.AreEqual("BAR", nodeInfo.VariableManager.ExpandVariables("${FOO}"));

            nodeInfo = statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Constants", new Dictionary<string, object>() { { "BAZ", "QUX" } } },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First();

            Assert.AreEqual("${FOO}", nodeInfo.VariableManager.ExpandVariables("${FOO}"));
            Assert.AreEqual("QUX", nodeInfo.VariableManager.ExpandVariables("${BAZ}"));
        }

        /// <summary>
        /// 境界値分析: ステートメントが Platforms 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest003()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Platforms", new List<object>() { "FOO", "BAR" } }
            });

            var statements = statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().ProduceTestStatementInfosByPlatforms(new[] { "FOO", "BAR" });

            Assert.AreEqual(2, statements.Count);

            statements = statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Platforms", new List<object>() { "BAZ", "QUX", "QUUX" } },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().ProduceTestStatementInfosByPlatforms(new[] { "FOO", "BAZ", "QUX", "QUUX" });

            Assert.AreEqual(3, statements.Count);
        }

        /// <summary>
        /// 境界値分析: ステートメントが Modules 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest004()
        {
            var primaryModules = new List<object>() { "FOO", "BAR" };

            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Modules", primaryModules }
            });

            var primaryStatement = statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First();

            Assert.AreEqual(primaryModules.Count, primaryStatement.Modules.Count);

            foreach (string module in primaryModules)
            {
                Assert.IsTrue(primaryStatement.Modules.Contains(module));
            }

            var secondaryModules = new List<object>() { "BAZ", "QUX", "QUUX" };

            var secondaryStatement = statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Modules", secondaryModules },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First();

            Assert.AreEqual(secondaryModules.Count, secondaryStatement.Modules.Count);

            foreach (string module in secondaryModules)
            {
                Assert.IsTrue(secondaryStatement.Modules.Contains(module));
            }
        }

        /// <summary>
        /// 境界値分析: ステートメントが Categories 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest005()
        {
            var categories = new List<object>() { "FOO", "BAR" };

            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Categories", categories }
            });

            var statementCategories = statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().Categories;

            Assert.AreEqual(categories.Count, statementCategories.Count);

            foreach (string category in categories)
            {
                Assert.IsTrue(statementCategories.Contains(category));
            }

            categories = new List<object>() { "BAZ", "QUX", "QUUX" };

            statementCategories = statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Categories", categories },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().Categories;

            Assert.AreEqual(categories.Count, statementCategories.Count);

            foreach (string category in categories)
            {
                Assert.IsTrue(statementCategories.Contains(category));
            }
        }

        /// <summary>
        /// 境界値分析: ステートメントが Parallelizable 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest006()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Parallelizable", true }
            });

            Assert.IsTrue(statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().Parallelizable);

            Assert.IsFalse(statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Parallelizable", false },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().Parallelizable);
        }

        /// <summary>
        /// 境界値分析: ステートメントが BreakOn 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest007()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "BreakOn", "Failure" }
            });

            Assert.AreEqual("Failure", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().BreakLevel.ToString());

            Assert.AreEqual("Timeout", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "BreakOn", "Timeout" },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().BreakLevel.ToString());
        }

        /// <summary>
        /// 境界値分析: ステートメントが GlobalRepeat 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest008()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "GlobalRepeat", "5" }
            });

            Assert.AreEqual(5, statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetExpandedTestNodeInfos().Count);

            Assert.AreEqual(10, statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "GlobalRepeat", "10" },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetExpandedTestNodeInfos().Count);
        }

        /// <summary>
        /// 境界値分析: ステートメントが LocalRepeat 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest009()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "LocalRepeat", "2" }
            });

            Assert.AreEqual(2, statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetExpandedTestNodeInfos().Count);

            Assert.AreEqual(3, statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "LocalRepeat", "3" },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetExpandedTestNodeInfos().Count);
        }

        /// <summary>
        /// 境界値分析: ステートメントが Timeout 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest010()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Timeout", "10" }
            });

            Assert.AreEqual(10, statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().Timeout);

            Assert.AreEqual(20, statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Timeout", "20" },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().Timeout);
        }

        /// <summary>
        /// 境界値分析: ステートメントが BuildType 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest011()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "BuildType", "Debug" }
            });

            Assert.AreEqual("Debug", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().BuildType.ToString());

            Assert.AreEqual("Release", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "BuildType", "Release" },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().BuildType.ToString());
        }

        /// <summary>
        /// 境界値分析: ステートメント中の BuildTypes 要素が指定されたビルドタイプを含まない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest012()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "BuildTypes", new List<object>() { "Foo", "Bar", "Baz" } }
            });

            Assert.AreEqual(0, statement.ProduceTestStatementInfosByBuildTypes(new List<string>()).Count);

            Assert.AreEqual(0, statement.ProduceTestStatementInfosByBuildTypes(new List<string>() { "Qux" }).Count);

            Assert.AreEqual(0, statement.ProduceTestStatementInfosByBuildTypes(new List<string>() { "Develop" }).Count);

            var listContexts = statement.ProduceTestStatementInfosByBuildTypes(new List<string>() { "Bar" })
                                        .Select(x => x.GetListContext(new ListContext()))
                                        .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual("Bar", listContexts[0].BuildTypeName);

            statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "BuildTypes", new List<object>() { "Foo", "Develop", "Bar" } }
            });

            listContexts = statement.ProduceTestStatementInfosByBuildTypes(new List<string>())
                                    .Select(x => x.GetListContext(new ListContext()))
                                    .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual(string.Empty, listContexts[0].BuildTypeName);

            Assert.AreEqual(0, statement.ProduceTestStatementInfosByBuildTypes(new List<string>() { "Qux" }).Count);

            listContexts = statement.ProduceTestStatementInfosByBuildTypes(new List<string>() { "Develop" })
                                    .Select(x => x.GetListContext(new ListContext()))
                                    .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual("Develop", listContexts[0].BuildTypeName);

            listContexts = statement.ProduceTestStatementInfosByBuildTypes(new List<string>() { "Bar" })
                                    .Select(x => x.GetListContext(new ListContext()))
                                    .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual("Bar", listContexts[0].BuildTypeName);

            statement = new TestStatementInfo(new Dictionary<string, object>());

            listContexts = statement.ProduceTestStatementInfosByBuildTypes(new List<string>())
                                    .Select(x => x.GetListContext(new ListContext()))
                                    .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual(string.Empty, listContexts[0].BuildTypeName);

            Assert.AreEqual(0, statement.ProduceTestStatementInfosByBuildTypes(new List<string>() { "Qux" }).Count);

            listContexts = statement.ProduceTestStatementInfosByBuildTypes(new List<string>() { "Develop" })
                                    .Select(x => x.GetListContext(new ListContext()))
                                    .ToArray();

            Assert.AreEqual(1, listContexts.Length);
            Assert.AreEqual("Develop", listContexts[0].BuildTypeName);
        }

        /// <summary>
        /// 境界値分析: ステートメントが Platform 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest013()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Platform", "Win32" }
            });

            Assert.AreEqual("Win32", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().Platform.ToString());

            Assert.AreEqual("Win64", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Platform", "Win64" },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().Platform.ToString());
        }

        /// <summary>
        /// 境界値分析: ステートメントが TargetName 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest014()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "TargetName", "MY-TARGET-NAME-1" }
            });

            Assert.AreEqual("MY-TARGET-NAME-1", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().TargetNamePattern);

            Assert.AreEqual("MY-TARGET-NAME-2", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "TargetName", "MY-TARGET-NAME-2" },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().TargetNamePattern);
        }

        /// <summary>
        /// 境界値分析: ステートメントが TargetInterface 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest015()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "TargetInterface", "MY-TARGET-INTERFACE-1" }
            });

            Assert.AreEqual("MY-TARGET-INTERFACE-1", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().TargetInterfacePattern);

            Assert.AreEqual("MY-TARGET-INTERFACE-2", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "TargetInterface", "MY-TARGET-INTERFACE-2" },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().TargetInterfacePattern);
        }

        /// <summary>
        /// 境界値分析: ステートメントが FailurePatterns 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest016()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "FailurePatterns", new List<object> { "Failed" } }
            });

            Assert.AreEqual("Failed", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().FailurePatterns[0]);

            Assert.AreEqual("Aborted", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "FailurePatterns", new List<object> { "Aborted" } },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().FailurePatterns[0]);
        }

        /// <summary>
        /// 境界値分析: ステートメントが Resources 要素を持たない
        /// </summary>
        [TestMethod]
        public void ProduceTestStatementInfosBoundaryTest017()
        {
            var statement = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Resources", new List<object> { "Hoge.exe" } }
            });

            Assert.AreEqual("Hoge.exe", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().Resources[0]);

            Assert.AreEqual("Fuga.dll", statement.ProduceTestStatementInfosByDocuments(new List<object>()
            {
                new Dictionary<string, object>()
                {
                    { "Resources", new List<object> { "Fuga.dll" } },
                    { "Tests", new List<object>() { new Dictionary<string, object>() } }
                }
            }).First().GetTestNodeInfos().First().Resources[0]);
        }
    }
}
