﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.ControlTarget;
using Nintendo.RunOnTarget;
using Nintendo.ControlTarget.Tmapi;
using Nintendo.Bridge;
using System.IO;
using System.IO.Ports;
using System.Diagnostics;
using System.Threading;
using System.Net;
using System.Net.Sockets;
using System.Collections.Generic;
using System.Text;
using System.Linq;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using TestUtility;


namespace MemoryUsageTest
{

    public struct KernelObjectInfo
    {
        public string name;
        public int used;
        public int peak;
        public int num;
    };

    [TestClass]
    public class MemoryUsageGrabber
    {
        // We expect/require Process name/ID in this format:
        //Process ID=10 (usb)
        private static string GrabProcessNameFromLine(string line)
        {
            string[] tokens = line.Split(new char[] { ' ', '\n' });
            // The last token is our process name. Replace the braces with spaces, trim off all whitespace, and return the process name string.
            return tokens[tokens.Length - 1].Replace('(', ' ').Replace(')', ' ').Trim();
        }

        // We expect/require memory information in this format in the text spew back from the target:
        //Code             1016 KB
        //CodeData         5152 KB
        //Heap                0 KB
        //SharedMemory        0 KB
        //InitialStack       16 KB
        //---
        //TOTAL            6184 KB
        private static int GrabMemoryAmountFromLine(string line)
        {
            string[] tokens = line.Split(new char[] { ' ' });
            return Int32.Parse(tokens[tokens.Length - 2]);
        }

        private static bool IsProcessInProductionList(string processName)
        {
            bool isTracked = false;
            switch (processName.ToLower())
            {
                // Processes that are run on product hardware
                case "account":
                case "am":
                case "arp":
                case "audio":
                case "bcat":
                case "bluetooth":
                case "boot2.prodboot":
                case "bsdsocket":
                case "btm":
                case "bus":
                case "capsrv":
                case "eclct":
                case "erpt":
                case "es":
                case "eupld":
                case "fatal":
                case "friends":
                case "fs":
                case "hid":
                case "lbl":
                case "ldn":
                case "loader":
                case "logmanager.prod":         // only logmanager
                case "logmanager":
                case "ncm":
                case "nfc":
                case "nifm":
                case "nim":
                case "npns":
                case "ns":
                case "nvnflinger":
                case "nvservices":
                case "pcie.withouthb":          // only pcie
                case "pcie":
                case "pctl":
                case "pcv":
                case "ppc":
                case "processmana":
                case "psc":
                case "ptm":
                case "settings":
                case "sm":
                case "spl":
                case "ssl":
                case "tma.stub":                // only tma
                case "usb":
                case "vi":
                case "wlan":

                // Data Files
                // case "fontchinesesimple":
                // case "fontchinesetraditional":
                // case "fontkorean":
                // case "fontnintendoextension":
                // case "fontstandard":
                    isTracked = true;
                    break;

                default:
                    break;
            }
            return isTracked;
        }

        private static bool IsProcessInDevelopmentList(string processName)
        {
            bool isTracked = false;
            switch (processName.ToLower())
            {
                // Processes that are run on development hardware
                case "tma":
                case "dmnt":
                case "cs":
                case "shell":
                case "nvdbgsvc":
                case "snapshotdum":
                    isTracked = true;
                    break;

                default:
                    break;
            }
            return isTracked;
        }

        private static bool IsProcessInOceanRegionList(string processName)
        {
            bool isTracked = false;
            switch (processName.ToLower())
            {
                // Processes that are run on Ocean region.
                case "devmenu":
                case "devoverlayd":
                case "debugmonito":
                    isTracked = true;
                    break;

                default:
                    break;
            }
            return isTracked;
        }

        // We expect/require kernel information in this format in the text spew back from the target:
        //KEvent
        //347 / 0 / 600
        //KInterruptEvent
        //41 / 0 / 100
        //KProcess
        //50 / 0 / 80
        //KThread
        //454 / 0 / 512
        //KPort
        //184 / 0 / 200
        private static KernelObjectInfo GrabKernelAmountsFromLine(string line, string name)
        {
            string[] tokens = line.Split(new char[] { '/' });
            KernelObjectInfo infoblock = new KernelObjectInfo();
            infoblock.name = name;
            infoblock.used = Int32.Parse(tokens[0]);
            infoblock.peak = Int32.Parse(tokens[1]);
            infoblock.num = Int32.Parse(tokens[2]);
            return infoblock;
        }

        private static void LogMemoryUsage(string processName, string memoryType, int usage)
        {
            // Change this formatting depending on how we're logging our results for TC to pick them up.
            Console.WriteLine("##teamcity[buildStatisticValue key='{0}-{1}' value='{2}']", processName, memoryType, usage);
        }

        private static void LogKernelObjectUsage(KernelObjectInfo info)
        {
            // Change this formatting depending on how we're logging our results for TC to pick them up.
            Console.WriteLine("##teamcity[buildStatisticValue key='{0}-used' value='{1}']", info.name, info.used);
            Console.WriteLine("##teamcity[buildStatisticValue key='{0}-peak' value='{1}']", info.name, info.peak);
            Console.WriteLine("##teamcity[buildStatisticValue key='{0}-num' value='{1}']", info.name, info.num);
        }

        [TestMethod]
        public static void GetMemoryUsageInfos(string commandOutput)
        {

            Console.WriteLine("Result of M(emory) command:\n");
            Console.WriteLine(commandOutput); // log the entire memory results spew that we get, so that we can verify we've gotten the whole thing.
            string[] lines = commandOutput.Split(new char[] { '\n' });

            int currentLine = 0;
            string processName = "";
            int memoryUsage = 0;
            int sumOfAllProductionProcessesMemoryUsage = 0;
            int sumOfAllDevelopmentProcessesMemoryUsage = 0;
            const int SharedMemorySizeForNs = 17*1024;
            do
            {
                // skip these lines before they trigger false matches for the rest
                if (lines[currentLine].Contains("0x"))
                {
                    continue;
                }
                if (lines[currentLine].Contains("Process "))
                {
                    processName = GrabProcessNameFromLine(lines[currentLine]);
                }
                if (lines[currentLine].Contains("Code ")) // The space on the end is to avoid matching against "CodeData" lines
                {
                    memoryUsage = GrabMemoryAmountFromLine(lines[currentLine]);
                    LogMemoryUsage(processName, "Code", memoryUsage);
                }
                if (lines[currentLine].Contains("CodeData "))
                {
                    memoryUsage = GrabMemoryAmountFromLine(lines[currentLine]);
                    LogMemoryUsage(processName, "CodeData", memoryUsage);
                }
                if (lines[currentLine].Contains("Heap "))
                {
                    memoryUsage = GrabMemoryAmountFromLine(lines[currentLine]);
                    LogMemoryUsage(processName, "Heap", memoryUsage);
                }
                if (lines[currentLine].Contains("SharedMemory "))
                {
                    memoryUsage = GrabMemoryAmountFromLine(lines[currentLine]);
                    if (processName.ToLower() == "sdb")
                    {
                        // Sdb process uses 17MB shared memory that came from Ocean region.
                        if (memoryUsage < SharedMemorySizeForNs)
                        {
                            throw new InternalTestFailureException(string.Format("FAILED sdb process should use 17MB shared memory ({0})", memoryUsage));
                        }
                    }
                    LogMemoryUsage(processName, "SharedMemory", memoryUsage);
                }
                if (lines[currentLine].Contains("InitialStack "))
                {
                    memoryUsage = GrabMemoryAmountFromLine(lines[currentLine]);
                    LogMemoryUsage(processName, "InitialStack", memoryUsage);
                }
                if (lines[currentLine].Contains("TOTAL"))
                {
                    memoryUsage = GrabMemoryAmountFromLine(lines[currentLine]);
                    LogMemoryUsage(processName, "TOTAL", memoryUsage);
                    if (IsProcessInDevelopmentList(processName))
                    {
                        sumOfAllDevelopmentProcessesMemoryUsage += memoryUsage;
                    }
                    else if (IsProcessInOceanRegionList(processName))
                    {
                        // Do not summarize value
                    }
                    else
                    {
                        // Including unknown new process
                        if (processName.ToLower() == "sdb")
                        {
                            // Sdb process uses 17MB shared memory that came from Ocean region.
                            memoryUsage -= SharedMemorySizeForNs;
                        }
                        sumOfAllProductionProcessesMemoryUsage += memoryUsage;
                    }

                }
                // ... and if nothing matches, increment line number and keep going
            } while (++currentLine < lines.Length);

            //Now that we've added up all of the process' total memory usage, report that.
            LogMemoryUsage("System", "Product-Total", sumOfAllProductionProcessesMemoryUsage);
            LogMemoryUsage("System", "Develop-Total", sumOfAllDevelopmentProcessesMemoryUsage);
            LogMemoryUsage("System", "Total", sumOfAllProductionProcessesMemoryUsage + sumOfAllDevelopmentProcessesMemoryUsage);
        }


        [TestMethod]
        public static void GetThreadUsageInfos(string commandOutput)
        {
            Console.WriteLine("Result of K(ernel) command:\n");
            Console.WriteLine(commandOutput); // log the entire results spew that we get, so that we can verify we've gotten the whole thing.
            string[] lines = commandOutput.Split(new char[] { '\n' });
            int currentLine = 0;

            do
            {
                // If we find one of the Kernel object types we recognize, log its value in the statistics system:
                if ((lines[currentLine].Contains("KEvent")) ||
                    (lines[currentLine].Contains("KInterruptEvent")) ||
                    (lines[currentLine].Contains("KProcess")) ||
                    (lines[currentLine].Contains("KThread")) ||
                    (lines[currentLine].Contains("KPort")) ||
                    (lines[currentLine].Contains("KSharedMemory")) ||
                    (lines[currentLine].Contains("KTransferMemory")) ||
                    (lines[currentLine].Contains("KDeviceAddressSpace")) ||
                    (lines[currentLine].Contains("KDebug")) ||
                    (lines[currentLine].Contains("KSession")) ||
                    (lines[currentLine].Contains("KLightSession")) ||
                    (lines[currentLine].Contains("KLinkedListNode")) ||
                    (lines[currentLine].Contains("KBlockInfo")) ||
                    (lines[currentLine].Contains("KAppMemoryBlock")) ||
                    (lines[currentLine].Contains("KSysMemoryBlock")) ||
                    (lines[currentLine].Contains("KThreadLocalPage")) ||
                    (lines[currentLine].Contains("KObjectName")) ||
                    (lines[currentLine].Contains("KEventInfo")) ||
                    (lines[currentLine].Contains("KSessionRequest")) ||
                    (lines[currentLine].Contains("KResourceLimit")))
                {
                    string name = lines[currentLine].TrimEnd(new char[] { '\r', '\n' });
                    if (name.Contains("usage")) // KBlockInfo usage
                    {
                        name = name.Split()[0];
                        currentLine++; // KBlockInfo has an extra line of "used / peak / num"
                    }
                    currentLine++; // These info blocks span two lines.
                    string[] tokens = lines[currentLine].Split(new char[] { '/' });
                    if ((tokens[0].Contains("used"))&&(tokens[1].Contains("peak"))&&(tokens[2].Contains("num")))
                    {
                        currentLine++;
                    }

                    LogKernelObjectUsage(GrabKernelAmountsFromLine(lines[currentLine], name));
                }

                // ... and if nothing matches, increment line number and keep going
            } while (++currentLine < lines.Length);
        }

        public static string ConnectAndRetreiveCommandOutput(string commandToRun, string nameFromUser)
        {
            var acc = new Nintendo.ControlTarget.Tmapi.TargetManagerAccessor();
            ConnectCommand connect = new Nintendo.ControlTarget.ConnectCommand();
            // Go with the default target unless the user specified a target name.
            // If the user-specified target name doesn't match a detected target, error out.
            string targetIP = "";
            string targetID = "";

            var targets = acc.DetectTargets(TimeSpan.FromSeconds(10));
            foreach (var t in targets)
            {
                Console.WriteLine(t.ToString());
                // If the user specified a target name, grab that target's IP address.
                if (nameFromUser.Length > 0 && t.GetName().Equals(nameFromUser))
                {
                    targetIP = t.GetIpAddress();
                    connect.Target = t.GetName();
                    targetID = t.GetSerialNumber();
                }
            }
            // If the user specified a target name and we couldn't find it, error out.
            if (nameFromUser.Length > 0 && targetIP.Length == 0)
            {
                throw new KeyNotFoundException(string.Format("Could not find target with name {0}", nameFromUser));
            }

            // If they didn't specify a target name, use the default, if there is one.
            if (nameFromUser.Length == 0)
            {
                if (acc.HasDefaultTarget())
                {
                    Console.WriteLine(acc.GetDefaultTargetInfo().ToString());
                    targetIP = acc.GetDefaultTargetInfo().GetIpAddress();
                    connect.Target = acc.GetDefaultTargetInfo().GetName();
                    targetID = acc.GetDefaultTargetInfo().GetSerialNumber();
                }
                else
                {
                    throw new ArgumentNullException("Could not find a default target device, and have no command-line target name.");
                }
            }

            // Connect to target so that the @Shell port activates.
            Console.WriteLine("Connecting to target {0}...", targetID);
            connect.Run();

            string localHost = @"127.0.0.1";
            int shellPort = Nintendo.ControlTarget.Utility.FindPortForService("@Shell", targetID, true);
            if (shellPort < 0)
            {
                throw new ArgumentException("Could not find port for @Shell. Are we connected to a target device?");
            }

            Console.WriteLine("Connecting to listening port on target at {0}... ", targetIP);
            Nintendo.Bridge.Telnet loggingPort = new Nintendo.Bridge.Telnet();
            bool connected = loggingPort.Connect(targetIP, Nintendo.Bridge.Telnet.TARGET_PORT);
            if (!connected)
            {
                throw new InternalTestFailureException(string.Format("FAILED loggingPort.ConnectToTarget({0})", targetIP));
            }

            Console.WriteLine("Connecting to @Shell port at localhost:{0}... ", shellPort);
            Nintendo.Bridge.Telnet commandShell = new Nintendo.Bridge.Telnet();
            connected = commandShell.Connect(localHost, shellPort);
            if (!connected)
            {
                throw new InternalTestFailureException(string.Format("FAILED commandShell.ConnectToTarget({0}, {1})", localHost, shellPort));
            }

            // Send commandToRun to shell on localhost to request memory stats back from listening port on target device.
            Console.WriteLine("Sending {0} to shell... ", commandToRun);
            // We need to keep reading after sending our command or the telnet session on the other end gets annoyed and the target doesn't produce output.
            string commandSnapBack = commandShell.WaitCommand(commandToRun);
            Console.WriteLine("Return from {1} command: {0}", commandSnapBack, commandToRun); // This should always just be "shell#", but it's not necessary to enforce that.

            Console.WriteLine("Distilling thread usage info from target device... ");
            string result = loggingPort.WaitCommand("\n", 1000);

            loggingPort.Exit();
            commandShell.Exit();

            return result;
        }
    };

    public class Program
    {
        public static void Main(string[] args)
        {
            string name = "";
            // before running the test, see if our user gave us a target name to connect to...
            if (args.Length > 0)
            {
                name = args[0];
                Console.WriteLine("Target name to find: {0}", args[0]);
            }
            try
            {
                MemoryUsageGrabber.GetThreadUsageInfos(MemoryUsageGrabber.ConnectAndRetreiveCommandOutput("K\r\n", name));
                MemoryUsageGrabber.GetMemoryUsageInfos(MemoryUsageGrabber.ConnectAndRetreiveCommandOutput("M\r\n", name));
            }
            catch (Exception exception)
            {
                PrintException(exception);
                System.Environment.Exit(1);
            }
        }

        public static void PrintException(Exception exception)
        {
            Console.Error.WriteLine("[ERROR] {0}", exception.Message);
            Console.WriteLine(string.Format("StackTrace: {0}", exception.StackTrace));
        }
    }

}
