﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace PurchaseTest
{
    struct NintendoAccount
    {
        public string id;
        public string password;

        public NintendoAccount(string id, string password)
        {
            this.id = id;
            this.password = password;
        }
    }

    [TestClass]
    public class ExecutionTest
    {
        public TestContext TestContext { get; set; }
        private static string m_TargetName = string.Empty;

        [ClassInitialize]
        public static void TestClassinitialize(TestContext context)
        {
            m_TargetName = (string)context.Properties["TargetName"];
        }

        private bool ExecuteDevMenuCommand(string args)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            return ExecuteDevMenuCommandCommon(args, testPath.GetSigloRoot() + "\\Programs\\Eris\\Outputs\\NX-NXFP2-a64\\TargetTools\\DevMenuCommand\\Release\\DevMenuCommand.nsp", false);
        }

        private bool ExecuteDevMenuCommand(string args, string successPattern)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            return ExecuteDevMenuCommandCommon(args, testPath.GetSigloRoot() + "\\Programs\\Eris\\Outputs\\NX-NXFP2-a64\\TargetTools\\DevMenuCommand\\Release\\DevMenuCommand.nsp", true, successPattern);
        }

        private bool ExecuteDevMenuCommandSystem(string args)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            return ExecuteDevMenuCommandCommon(args, testPath.GetSigloRoot() + "\\Programs\\Eris\\Outputs\\NX-NXFP2-a64\\TargetTools\\DevMenuCommandSystem\\Release\\DevMenuCommandSystem.nsp", false);
        }

        private bool ExecuteDevMenuCommandSystem(string args, string successPattern)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            return ExecuteDevMenuCommandCommon(args, testPath.GetSigloRoot() + "\\Programs\\Eris\\Outputs\\NX-NXFP2-a64\\TargetTools\\DevMenuCommandSystem\\Release\\DevMenuCommandSystem.nsp", true, successPattern);
        }

        private bool ExecuteDevMenuCommandToLaunchApplicationAndExpectOutput(string id, string expectOutput, int timeoutSeconds)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            return ExecuteDevMenuCommandToLaunchApplicationAndExpectOutputCommon(id, expectOutput, timeoutSeconds,
                       testPath.GetSigloRoot() + "\\Programs\\Eris\\Outputs\\NX-NXFP2-a64\\TargetTools\\DevMenuCommand\\Release\\DevMenuCommand.nsp");
        }

        private Tuple<StringBuilder, StringBuilder> SetupProcess(Process process, string filename, string arguments)
        {
            var standardOutput = new StringBuilder();
            var standardError = new StringBuilder();

            process.StartInfo.FileName = filename;
            process.StartInfo.Arguments = arguments;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.StandardOutputEncoding = System.Text.Encoding.UTF8;
            process.OutputDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                standardOutput.AppendLine(e.Data);
            };
            process.ErrorDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                standardError.AppendLine(e.Data);
            };

            return new Tuple<StringBuilder, StringBuilder>(standardOutput, standardError);
        }
        private void RunAndWaitProcess(Process process)
        {
            Console.WriteLine("process start.");
            process.Start();
            process.BeginOutputReadLine();
            process.BeginErrorReadLine();
            Console.WriteLine("process wait for exit.");
            process.WaitForExit();
        }

        private bool ExecuteDevMenuCommandCommon(string args, string ncaPath, bool setSuccessPattern, string successPattern = "")
        {
            Console.WriteLine("ExecuteDevMenuCommandCommon start.(" + args + ")");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\RunOnTarget.exe";
                var pattern = setSuccessPattern ? " --pattern-success-exit \"" + successPattern + "\"" : " --pattern-failure-exit \\[FAILURE\\] ";
                var arguments = " -t " + m_TargetName +
                                " " + ncaPath +
                                pattern +
                                " -v " +
                                "-- " +
                                args;

                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }
        private bool ExecuteDevMenuCommandToLaunchApplicationAndExpectOutputCommon(string id, string expectOutput, int timeoutSeconds, string ncaPath)
        {
            Console.WriteLine("ExecuteDevMenuCommandToLaunchApplicationAndExpectOutputCommon start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\RunOnTargetPrivate.exe";
                var arguments = "run " +
                                " " + ncaPath +
                                " -t " + m_TargetName +
                                " --suppress-polling-process " +
                                " --pattern-success-exit " +
                                expectOutput +
                                " --pattern-failure-exit \\[FAILURE\\] " +
                                " --failure-timeout " +
                                string.Format("{0}", timeoutSeconds) +
                                " -v " +
                                " -- " +
                                " application " +
                                " launch " +
                                id;

                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }

        private bool IsApplicationInstalled(string id, bool withReset = false)
        {
            Console.WriteLine("IsApplicationInstalled start.(" + id + ")");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var ncaPath = testPath.GetSigloRoot()
                    + @"\Programs\Eris\Outputs\NX-NXFP2-a64\TargetTools\DevMenuCommandSystem\Release\DevMenuCommandSystem.nsp";

                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\RunOnTarget.exe";
                var resetOption = withReset ? "--reset" : string.Empty;
                var arguments = " -t " + m_TargetName +
                                " " + ncaPath +
                                " " + resetOption + " " +
                                " --pattern-failure-exit \\[FAILURE\\] " +
                                " -v " +
                                " -- " +
                                " application " +
                                " list ";

                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                if (process.ExitCode != 0)
                {
                    throw new Exception("IsApplicationInstalled Exception: RunOnTarget Failure");
                }

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                if (outputs.Item1.ToString().Contains(id))
                {
                    return true;
                }

                return false;
            }
        }

        private bool ExecuteControlTarget(string args)
        {
            Console.WriteLine($"ControlTarget {args} start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\ControlTarget.exe";
                var arguments = args;
                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }

        private bool RunInstalledApplication(UInt64 applicationId, string successText)
        {
            Console.WriteLine("RunInstalledApplication start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\RunOnTarget.exe";
                var arguments = " -- " + "0x" + applicationId.ToString("x16");

                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return outputs.Item1.ToString().Contains(successText);
            }
        }

        private void LinkAccountAndDevice(int index, string nintendoAccountId, string nintendoAccountPassword, bool deleteAllRights = false)
        {
            Assert.IsTrue(ExecuteDevMenuCommand("account add --register_nsa"));
            Assert.IsTrue(ExecuteDevMenuCommand("account link --index " + index.ToString() + " --id " + nintendoAccountId + " --password " + nintendoAccountPassword));
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop shop-account-status " + index.ToString()));
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop link-device " + index.ToString()));

            if(deleteAllRights)
            {
                Assert.IsTrue(ExecuteDevMenuCommandSystem("shop delete-all-rights " + index.ToString()));
            }
        }

        private void Cleanup()
        {
            Assert.IsTrue(ExecuteDevMenuCommand("application uninstall --all"));
            Assert.IsTrue(ExecuteDevMenuCommandSystem("ticket delete-all"));
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop unlink-device-all"));
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop unregister-device-account"));
            Assert.IsTrue(ExecuteDevMenuCommand("account clear_all"));
        }

        [TestInitialize]
        public void InitializeForPurchaseTest()
        {
            Assert.IsTrue(ExecuteDevMenuCommand("servicediscovery import-all td1pass"));
            Assert.IsTrue(ExecuteControlTarget("reset"));

            Cleanup();
        }

        [TestCleanup]
        public void FinalizeForPurchaseTest()
        {
            Cleanup();
        }

        [TestMethod]
        public void PurchaseAndRunExternalKeyApplication()
        {
            UInt64 applicationId = 0x01004320023dc000;
            int ApplicationCount = 64;

            // id, password
            NintendoAccount account = new NintendoAccount("eshop-test+psgcitd1_2@exmx.nintendo.co.jp", "psgcitd1_2");

            // デバイスアカウントの作成
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop register-device-account"));

            // デバイスリンク
            LinkAccountAndDevice(0, account.id, account.password, true);

            // コンテンツの購入
            for (int i = 0; i < ApplicationCount; i++)
            {
                Assert.IsTrue(ExecuteDevMenuCommandSystem("shop download-demo 0 --id " + (applicationId +(UInt64)i).ToString("x16")));
            }

            // コンテンツのダウンロードが完了するまで待機
            System.Threading.Thread.Sleep(60 * 1000);
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application wait-download-all"));

            // コンテンツの権限確認と起動
            for (int i = 0; i < ApplicationCount; i++)
            {
                Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + (applicationId + (UInt64)i).ToString("x16")));
                Assert.IsTrue(RunInstalledApplication(applicationId + (UInt64)i, "HashCheck : OK"));
            }

            // 全てのチケットを削除
            Assert.IsTrue(ExecuteDevMenuCommandSystem("ticket delete-all"));

            // チケット同期
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop sync-ticket"));

            // コンテンツの権限確認(チケット同期でチケットが再度落ちてきているか確認するためにもう一度実行)
            for (int i = 0; i < ApplicationCount; i++)
            {
                Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + (applicationId + (UInt64)i).ToString("x16")));
            }

            // テストの後処理
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop delete-all-rights 0"));
        }

        [TestMethod]
        public void PurchaseAndRunExternalKeyApplicationByDifferentUser()
        {
            const int MaxAccountCount = 8;

            // id, password
            NintendoAccount[] accounts = new NintendoAccount[]
            {
                new NintendoAccount("eshop-test+psgcitd1_2@exmx.nintendo.co.jp", "psgcitd1_2"),
                new NintendoAccount("eshop-test+psgcitd1_3@exmx.nintendo.co.jp", "psgcitd1_3"),
                new NintendoAccount("eshop-test+psgcitd1_4@exmx.nintendo.co.jp", "psgcitd1_4"),
                new NintendoAccount("eshop-test+psgcitd1_5@exmx.nintendo.co.jp", "psgcitd1_5"),
                new NintendoAccount("eshop-test+psgcitd1_6@exmx.nintendo.co.jp", "psgcitd1_6"),
                new NintendoAccount("eshop-test+psgcitd1_7@exmx.nintendo.co.jp", "psgcitd1_7"),
                new NintendoAccount("eshop-test+psgcitd1_8@exmx.nintendo.co.jp", "psgcitd1_8"),
                new NintendoAccount("eshop-test+psgcitd1_9@exmx.nintendo.co.jp", "psgcitd1_9"),
            };

            UInt64[] applicationId = new UInt64[]
            {
                0x01004320023dc000,
                0x01004320023dc001,
                0x01004320023dc002,
                0x01004320023dc003,
                0x01004320023dc004,
                0x01004320023dc005,
                0x01004320023dc006,
                0x01004320023dc007,
            };

            // デバイスアカウントの作成
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop register-device-account"));

            // デバイスリンク
            for (int i = 0; i < MaxAccountCount; i++)
            {
                LinkAccountAndDevice(i, accounts[i].id, accounts[i].password, true);
            }

            // コンテンツの購入
            for (int i = 0; i < MaxAccountCount; i++)
            {
                Assert.IsTrue(ExecuteDevMenuCommandSystem("shop download-demo " + i.ToString() + " --id " + applicationId[i].ToString("x16")));
            }

            // コンテンツのダウンロードが完了するまで待機
            System.Threading.Thread.Sleep(60 * 1000);
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application wait-download-all"));

            // コンテンツの権限確認と起動
            for (int i = 0; i < MaxAccountCount; i++)
            {
                Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId[i].ToString("x16")));
                Assert.IsTrue(RunInstalledApplication(applicationId[i], "HashCheck : OK"));
            }

            // 全てのチケットを削除
            Assert.IsTrue(ExecuteDevMenuCommandSystem("ticket delete-all"));

            // チケット同期
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop sync-ticket"));

            // コンテンツの権限確認(チケット同期でチケットが再度落ちてきているか確認するためにもう一度実行)
            for (int i = 0; i < MaxAccountCount; i++)
            {
                Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId[i].ToString("x16")));
            }

            // デバイスアンリンクと紐付け解除した NA で購入したコンテンツの権限確認で ResultTicketNotFound が返ってくることを確認
            for (int i = 0; i < MaxAccountCount; i++)
            {
                Assert.IsTrue(ExecuteDevMenuCommandSystem("shop unlink-device " + i.ToString()));
                Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId[i].ToString("x16"), "RightsNotFound"));
            }
        }

        [TestMethod]
        public void InactiveNintendoAccountTest()
        {
            UInt64 applicationId = 0x01004320023dc000;

            // id, password
            NintendoAccount account = new NintendoAccount("eshop-test+psgcitd1_2@exmx.nintendo.co.jp", "psgcitd1_2");

            // デバイスアカウントの作成
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop register-device-account"));

            // デバイスリンク
            LinkAccountAndDevice(0, account.id, account.password, true);

            // コンテンツの購入
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop download-demo 0 --id " + applicationId.ToString("x16")));

            // コンテンツのダウンロードが完了するまで待機
            System.Threading.Thread.Sleep(30 * 1000);
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application wait-download-all"));

            // コンテンツの権限確認と起動
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId.ToString("x16")));
            Assert.IsTrue(RunInstalledApplication(applicationId, "HashCheck : OK"));

            // NA の状態を仮削除状態に変更
            Assert.IsTrue(ExecuteDevMenuCommandSystem("account set --index 0 dbg:user_state=NintendoAccount:StateWithdrawn"));

            // コンテンツの権限確認で ResultInactiveNintendoAccount が返ってくることを確認
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId.ToString("x16"), "InactiveNintendoAccount"));

            // NA の状態を削除状態に変更
            Assert.IsTrue(ExecuteDevMenuCommandSystem("account set --index 0 dbg:user_state=NintendoAccount:StateDeleted"));

            // コンテンツの権限確認で ResultInactiveNintendoAccount が返ってくることを確認
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId.ToString("x16"), "InactiveNintendoAccount"));
        }

        [TestMethod]
        public void MultiInactiveNintendoAccountTest()
        {
            UInt64 applicationId = 0x01004320023dc000;

            // id, password
            NintendoAccount[] accounts = new NintendoAccount[]
            {
                // NA_1
                new NintendoAccount("eshop-test+psgcitd1_2@exmx.nintendo.co.jp", "psgcitd1_2"),
                // NA_2
                new NintendoAccount("eshop-test+psgcitd1_3@exmx.nintendo.co.jp", "psgcitd1_3"),
            };

            // デバイスアカウントの作成
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop register-device-account"));

            // NA_1 をデバイスリンク
            LinkAccountAndDevice(0, accounts[0].id, accounts[0].password, true);

            // NA_1 でコンテンツの購入
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop download-demo 0 --id " + applicationId.ToString("x16")));

            // コンテンツのダウンロードが完了するまで待機
            System.Threading.Thread.Sleep(30 * 1000);
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application wait-download-all"));

            // コンテンツの権限確認と起動
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId.ToString("x16")));
            Assert.IsTrue(RunInstalledApplication(applicationId, "HashCheck : OK"));

            // NA_1 のデバイスアンリンクと購入したコンテンツの権限確認で ResultTicketNotFound が返ってくることを確認
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop unlink-device 0"));
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId.ToString("x16"), "RightsNotFound"));

            // NA_2 をデバイスリンク
            LinkAccountAndDevice(1, accounts[1].id, accounts[1].password);

            // NA_2 でコンテンツの購入
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop download-demo 1 --id " + applicationId.ToString("x16")));

            // コンテンツのダウンロードが完了するまで待機
            System.Threading.Thread.Sleep(30 * 1000);
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application wait-download-all"));

            // NA_1 で購入したアプリケーションが残っているため、チケットが落ちてこないのでこの時点ではコンテンツの権限確認で ResultTicketNotFound が返ってくることを確認
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId.ToString("x16"), "RightsNotFound"));

            // チケット同期
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop sync-ticket"));

            // コンテンツの権限確認と起動(チケット同期をすると起動できるようになることを確認)
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId.ToString("x16")));
            Assert.IsTrue(RunInstalledApplication(applicationId, "HashCheck : OK"));

            // NA_1 を再度デバイスリンク
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop link-device 0"));

            // チケット同期
            Assert.IsTrue(ExecuteDevMenuCommandSystem("shop sync-ticket"));

            // NA_1 の状態を仮削除状態に変更
            Assert.IsTrue(ExecuteDevMenuCommandSystem("account set --index 0 dbg:user_state=NintendoAccount:StateWithdrawn"));

            // コンテンツの権限確認と起動(NA_1 のチケットの権利は利用できない状態だが、NA_2 のチケットは利用できるので起動できる)
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId.ToString("x16")));
            Assert.IsTrue(RunInstalledApplication(applicationId, "HashCheck : OK"));

            // NA_2 の状態を仮削除状態に変更
            Assert.IsTrue(ExecuteDevMenuCommandSystem("account set --index 1 dbg:user_state=NintendoAccount:StateWithdrawn"));

            // コンテンツの権限確認で ResultInactiveNintendoAccount が返ってくることを確認
            Assert.IsTrue(ExecuteDevMenuCommandSystem("application check-launch-rights " + applicationId.ToString("x16"), "InactiveNintendoAccount"));
        }
    }
}
