﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using PackageCreator;
using TestUtility;
using Nintendo.Foundation.IO;

namespace PackageCreatorTest
{
    [TestClass]
    public class UnitTest1
    {
        public TestContext TestContext { get; set; }

        private static readonly string TestDefinitionsDirectory = "PackageData/PackageDefinitions";

        [TestMethod]
        public void CreatePackageTest1()
        {
            TestUtility.TestPath testPath = new TestPath(this.TestContext);

            string rootDir = testPath.GetSigloRoot();
            string sourceDir = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory);
            string source = Path.Combine(sourceDir, "test.package.yml");
            string translateFilePath = Path.Combine(sourceDir, "TestRule.list");

            string packageBaseName = "CreatePackageTest";
            string packageName = packageBaseName + ".zip";
            string packageDir = packageBaseName + "/DummyPackage/DummyFiles";
            var fileList = new List<string>();
            fileList.Add("filename include whitespace.cpp");
            fileList.Add("normalFiles.cpp");
            fileList.Add("size0file.txt");

            this.CreanUp(rootDir);

            Program.Main(new[] { "--root", rootDir, "--source-file", source, "--translate", translateFilePath, "--output-name", packageName });

            ZipFile.ExtractToDirectory(Path.Combine(rootDir, packageName), rootDir);

            foreach (string file in fileList)
            {
                Assert.IsTrue(File.Exists(Path.Combine(rootDir, packageDir, file)));
            }

            Directory.Delete(Path.Combine(rootDir, packageBaseName), true);
        }

        [TestMethod]
        public void CreatePackageTest2()
        {
            TestUtility.TestPath testPath = new TestPath(this.TestContext);

            string rootDir = testPath.GetSigloRoot();
            string source = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory, "test.package.yml");

            string packageBaseName = "CreatePackageTest";
            string packageName = packageBaseName + ".zip";
            string packageDir = packageBaseName + "/Tests/Tools/Resources/PackageData/DummyFiles";
            var fileList = new List<string>();
            fileList.Add("filename include whitespace.cpp");
            fileList.Add("normalFiles.cpp");
            fileList.Add("size0file.txt");

            this.CreanUp(rootDir);

            Program.Main(new[] { "--root", rootDir, "--source-file", source, "--output-name", packageName });

            ZipFile.ExtractToDirectory(Path.Combine(rootDir, packageName), rootDir);

            foreach (string file in fileList)
            {
                Assert.IsTrue(File.Exists(Path.Combine(rootDir, packageDir, file)));
            }

            Directory.Delete(Path.Combine(rootDir, packageBaseName), true);
        }

        [TestMethod]
        public void CreateCommentAndSpaceTest()
        {
            TestUtility.TestPath testPath = new TestPath(this.TestContext);

            string rootDir = testPath.GetSigloRoot();
            string sourceDir = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory);
            string source = Path.Combine(sourceDir, "test.package.yml");
            string translateFilePath = Path.Combine(sourceDir, "TestCommentAndSpace.list");

            string packageBaseName = "CreatePackageTest";
            string packageName = packageBaseName + ".zip";
            string packageDir = packageBaseName + "/DummyPackage/DummyFiles";
            var fileList = new List<string>();
            fileList.Add("filename include whitespace.cpp");
            fileList.Add("normalFiles.cpp");
            fileList.Add("size0file.txt");

            this.CreanUp(rootDir);

            Program.Main(new[] { "--root", rootDir, "--source-file", source, "--translate", translateFilePath, "--output-name", packageName });

            ZipFile.ExtractToDirectory(Path.Combine(rootDir, packageName), rootDir);

            foreach (string file in fileList)
            {
                Assert.IsTrue(File.Exists(Path.Combine(rootDir, packageDir, file)));
            }

            Directory.Delete(Path.Combine(rootDir, packageBaseName), true);
        }

        [TestMethod]
        public void CreateForbiddenWordTest()
        {
            TestUtility.TestPath testPath = new TestPath(this.TestContext);

            string rootDir = testPath.GetSigloRoot();

            string sourceDir = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory);
            string packageFilePath = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory, "test.package.yml");
            string translateFilePath = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory, "TestRule.list");
            string forbiddenFilePath = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory, "TestWord.list");

            string packageBaseName = "CreatePackageTest";
            string packageName = packageBaseName + ".zip";
            string packageDir = packageBaseName + "/DummyPackage/DummyFiles";
            var fileList = new List<string>();
            fileList.Add("filename include whitespace.cpp");
            fileList.Add("normalFiles.cpp");
            fileList.Add("size0file.txt");

            this.CreanUp(rootDir);

            Program.Main(new[] {"--root", rootDir, "--source-file", packageFilePath, "--translate", translateFilePath,
                "--forbidden", forbiddenFilePath, "--output-name", packageName });

            ZipFile.ExtractToDirectory(Path.Combine(rootDir, packageName), rootDir);

            foreach (string file in fileList)
            {
                Assert.IsTrue(File.Exists(Path.Combine(rootDir, packageDir, file)));
            }

            Directory.Delete(Path.Combine(rootDir, packageBaseName), true);
        }

        [TestMethod]
        public void CreateFailureWordTest()
        {
            TestUtility.TestPath testPath = new TestPath(this.TestContext);

            string rootDir = testPath.GetSigloRoot();

            string sourceDir = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory);
            string packageFilePath = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory, "test.package.yml");
            string translateFilePath = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory, "TestFailureRule.list");
            string forbiddenFilePath = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory, "TestWord.list");

            string packageBaseName = "CreatePackageTest";
            string packageName = packageBaseName + ".zip";
            string packageDir = packageBaseName + "/Tests/Tools/Resources/PackageData/DummyFiles";
            var fileList = new List<string>();
            fileList.Add("filename include whitespace.cpp");
            fileList.Add("normalFiles.cpp");
            fileList.Add("size0file.txt");

            this.CreanUp(rootDir);

            try
            {
                PackageCreatorParam param = null;
                CommandLineParser.Default.ParseArgs(
                    new[] {"--root", rootDir, "--source-file", packageFilePath, "--translate", translateFilePath,
                        "--forbidden", forbiddenFilePath, "--output-name", packageName },
                    out param);
                Assert.IsNotNull(param);
                // NGワード検出部分のチェックができればよいので、 PackageCreator.Create のみチェック
                var package = new PackageDefinition(packageFilePath, rootDir);
                PackageCreator.PackageCreator.Create(param, package);
            }
            catch (ArgumentException ex)
            {
                Console.Error.WriteLine(ex.ToString());
                return;
            }
            finally
            {
                if (Directory.Exists(Path.Combine(rootDir, packageBaseName)))
                {
                    Directory.Delete(Path.Combine(rootDir, packageBaseName), true);
                }
            }
            Assert.Fail();
        }

        [TestMethod]
        public void CreatePackageInTheDirectoryTest()
        {
            TestUtility.TestPath testPath = new TestPath(this.TestContext);

            string rootDir = testPath.GetSigloRoot();
            string outputDir = Path.Combine(rootDir, "TestOutputs");

            string source = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory, "test.package.yml");

            string packageBaseName = "CreatePackageTest";
            string packageName = packageBaseName + ".zip";
            string packageDir = packageBaseName + "/Tests/Tools/Resources/PackageData/DummyFiles";
            var fileList = new List<string>();
            fileList.Add("filename include whitespace.cpp");
            fileList.Add("normalFiles.cpp");
            fileList.Add("size0file.txt");

            Program.Main(new[] { "--root", rootDir, "--source-file", source, "--output-dir", outputDir, "--output-name", packageName });

            Assert.IsTrue(Directory.Exists(outputDir));

            ZipFile.ExtractToDirectory(Path.Combine(outputDir, packageName), outputDir);

            foreach (string file in fileList)
            {
                Assert.IsTrue(File.Exists(Path.Combine(outputDir, packageDir, file)));
            }

            Directory.Delete(outputDir, true);
        }

        [TestMethod]
        public void NoCreateZip()
        {
            TestUtility.TestPath testPath = new TestPath(this.TestContext);

            string rootDir = testPath.GetSigloRoot();
            string outputDir = Path.Combine(rootDir, "TestOutputs");

            string source = Path.Combine(testPath.GetToolTestResourceDirectory(), TestDefinitionsDirectory, "test.package.yml");

            string packageBaseName = "CreatePackageTest";
            string packageName = packageBaseName + ".zip";
            string packageDir = packageBaseName + "/Tests/Tools/Resources/PackageData/DummyFiles";
            var fileList = new List<string>();
            fileList.Add("filename include whitespace.cpp");
            fileList.Add("normalFiles.cpp");
            fileList.Add("size0file.txt");

            Program.Main(new[] { "--root", rootDir, "--source-file", source, "--output-dir", outputDir, "--no-create-archives", "--output-name", packageName });

            Assert.IsTrue(!File.Exists(Path.Combine(outputDir, packageName)));
            Assert.IsTrue(Directory.Exists(Path.Combine(outputDir, packageDir)));

            Directory.Delete(outputDir, true);
        }

        public void CreanUp(string root)
        {
            if (!Directory.Exists(root))
            {
                return;
            }

            List<string> artifacts = new List<string>();

            foreach (string extention in new string[] { "*.zip", "*.list", "*.nuspec" })
            {
                artifacts.AddRange(System.IO.Directory.GetFiles(root, extention));
            }

            foreach (string file in artifacts)
            {
                System.IO.File.Delete(file);
            }
        }
    }
}
