﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Security.Cryptography;
using System.Threading.Tasks;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MaxStorageSizeInstallTest
{
    [TestClass]
    public class MaxStorageSizeInstallTest
    {
        public TestContext TestContext { get; set; }
        private static string m_TargetName = string.Empty;

        [ClassInitialize]
        public static void TestClassinitialize(TestContext context)
        {
            m_TargetName = (string)context.Properties["TargetName"];
        }

        private static string[] m_ApplicationIds = {
            "0x0100ce00023d8600",
            "0x0100ce00023d8601",
            "0x0100ce00023d8602",
            "0x0100ce00023d8603",
            "0x0100ce00023d8604",
            "0x0100ce00023d8605",
            "0x0100ce00023d8606",
            "0x0100ce00023d8607",
            "0x0100ce00023d8608",
            "0x0100ce00023d8609",
            "0x0100ce00023d860A",
            "0x0100ce00023d860B",
            "0x0100ce00023d860C",
            "0x0100ce00023d860D",
            "0x0100ce00023d860E",
            "0x0100ce00023d860F"
        };

        private Tuple<StringBuilder, StringBuilder> SetupProcess(Process process, string filename, string arguments, FileStream fsOutput = null, FileStream fsErrorOutput = null)
        {
            var standardOutput = new StringBuilder();
            var standardError = new StringBuilder();

            process.StartInfo.FileName = filename;
            process.StartInfo.Arguments = arguments;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.StandardOutputEncoding = System.Text.Encoding.UTF8;
            process.OutputDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                if (fsOutput != null && e.Data != null)
                {
                    fsOutput.Write(Encoding.UTF8.GetBytes(e.Data), 0, e.Data.Length);
                    fsOutput.Write(Encoding.UTF8.GetBytes("\n"), 0, 1);
                }
                standardOutput.AppendLine(e.Data);
            };
            process.ErrorDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                if (fsErrorOutput != null && e.Data != null)
                {
                    fsErrorOutput.Write(Encoding.UTF8.GetBytes(e.Data), 0, e.Data.Length);
                    fsErrorOutput.Write(Encoding.UTF8.GetBytes("\n"), 0, 1);
                }
                standardError.AppendLine(e.Data);
            };

            return new Tuple<StringBuilder, StringBuilder>(standardOutput, standardError);
        }

        private void RunAndWaitProcess(Process process)
        {
            Console.WriteLine("process start.");
            process.Start();
            process.BeginOutputReadLine();
            process.BeginErrorReadLine();
            Console.WriteLine("process wait for exit.");
            process.WaitForExit();
        }

        private bool ExecuteMakeTestApplication(string arguments, string outputPath)
        {
            Console.WriteLine("[ContentsUploaderTest] ExecuteMakeTestApplication start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\MakeTestApplication\\MakeTestApplication.exe";
                var txtOutput = (outputPath + "\\MakeTestApplicationOutput.txt");
                var txtErrorOutput = (outputPath + "\\MakeTestApplicationErrorOutput.txt");
                var fsOutput = new FileStream(txtOutput, FileMode.Append, FileAccess.Write);
                var fsErrorOutput = new FileStream(txtErrorOutput, FileMode.Append, FileAccess.Write);
                var outputs = SetupProcess(process, filename, " " + arguments, fsOutput, fsErrorOutput);
                RunAndWaitProcess(process);
                fsOutput.Close();
                fsErrorOutput.Close();

                if (process.ExitCode != 0)
                {
                    Console.WriteLine(string.Format("[ContentsUploaderTest] Standard Output: {0}", outputs.Item1.ToString()));
                    Console.WriteLine(string.Format("[ContentsUploaderTest] Standard Error: {0}", outputs.Item2.ToString()));
                    Environment.ExitCode = 1;
                    return false;
                }

                Console.WriteLine(string.Format("[ContentsUploaderTest] Standard Output: {0}", outputs.Item1.ToString()));
                var success = "--ElapsedTime:";
                return outputs.Item1.ToString().Contains(success);
            }
        }

        private bool ExecuteDevMenuCommandSystem(string exeArgs, string nspArgs, string outputPath)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            return ExecuteDevMenuCommandCommon(exeArgs, nspArgs, testPath.GetSigloRoot() + "\\Programs\\Eris\\Outputs\\NX-NXFP2-a64\\TargetTools\\DevMenuCommandSystem\\Develop\\DevMenuCommandSystem.nsp", outputPath);
        }

        private bool ExecuteDevMenuCommandCommon(string exeArgs, string nspArgs, string nspPath, string outputPath)
        {
            Console.WriteLine("ExecuteDevMenuCommandCommon start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\RunOnTargetPrivate.exe";
                var arguments = $"run { nspPath } -t {m_TargetName} { exeArgs } --pattern-failure-exit \\[FAILURE\\] -- { nspArgs }";
                var txtOutput = (outputPath + "\\DevMenuCommandSystemOutput.txt");
                var txtErrorOutput = (outputPath + "\\DevMenuCommandSystemErrorOutput.txt");
                var fsOutput = new FileStream(txtOutput, FileMode.Append, FileAccess.Write);
                var fsErrorOutput = new FileStream(txtErrorOutput, FileMode.Append, FileAccess.Write);
                var outputs = SetupProcess(process, filename, " " + arguments, fsOutput, fsErrorOutput);
                RunAndWaitProcess(process);
                fsOutput.Close();
                fsErrorOutput.Close();

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }

        [TestMethod]
        public void InstallTest()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            DirectoryInfo logOutputDirectory = Directory.CreateDirectory(testPath.GetTestRoot() + "\\Outputs\\AnyCPU\\Tools\\MaxStorageSizeInstallTest\\outputDirectory");
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, "sdcard format", logOutputDirectory.FullName));
            Assert.IsTrue(ExecuteDevMenuCommandSystem("--reset", "application list", logOutputDirectory.FullName));
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, "application storage-size builtin", logOutputDirectory.FullName));
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, "application storage-size sdcard", logOutputDirectory.FullName));

            var xmlPath = testPath.GetTestRoot() + "\\Tools\\Sources\\Tests\\MaxStorageSizeInstallTest\\TestData\\Application_16G_16.xml";
            DirectoryInfo nspOutputDirectory = Directory.CreateDirectory(testPath.GetTestRoot() + "\\Outputs\\AnyCPU\\Tools\\MaxStorageSizeInstallTest\\nspOutputDirectory");
            Assert.IsTrue(ExecuteMakeTestApplication("--config-xml=" + xmlPath + " -o " + nspOutputDirectory.FullName + " -j 8 --small-code", logOutputDirectory.FullName));
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, "application install " + nspOutputDirectory.FullName + " --all --force -s sdcard", logOutputDirectory.FullName));
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, "application list", logOutputDirectory.FullName));
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, "application storage-size builtin", logOutputDirectory.FullName));
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, "application storage-size sdcard", logOutputDirectory.FullName));
            foreach (string id in m_ApplicationIds)
            {
                Assert.IsTrue(ExecuteDevMenuCommandSystem("--pattern-success-exit \"HashCheck : OK\" --pattern-failure-exit \"HashCheck : NG\"", "application launch " + id, logOutputDirectory.FullName));
            }
            nspOutputDirectory.Delete(true);
        }
    }
}
