﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.MakePropertySheets;
using Nintendo.MakePropertySheets.Constructor;
using Nintendo.MakePropertySheets.Constructor.ItemDefinition;
using Nintendo.MakePropertySheets.Constructor.Property;
using Nintendo.MakePropertySheets.Constructor.Target;

namespace Nintendo.MakePropertySheetsTest
{
    [TestClass]
    public class ProjectElementConstructorDependencyResolverTest
    {
        private static readonly Dictionary<string, string> DummyParameters1 = new Dictionary<string, string>() { { "p1", "pv1" } };
        private static readonly Dictionary<string, string> DummyParameters2 = new Dictionary<string, string>() { { "p2", "pv2" } };

        [TestMethod]
        public void TestEmpty()
        {
            var resolver = new ProjectElementConstructorDependencyResolver(Enumerable.Empty<ProjectElement>());

            Assert.IsFalse(resolver.Any());
        }

        [TestMethod]
        public void TestSimpleProperties()
        {
            var elements = new ProjectElement[]
            {
                new ProjectProperty("p1", "v1"),
                new ProjectPropertyGroup(
                    new ProjectProperty("p2", "v2"),
                    new ProjectProperty("p3", "v3")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);

            CollectionAssert.AreEquivalent(elements, resolver.SelectMany(x => x).ToArray());
        }

        [TestMethod]
        public void TestSimpleMetadatas()
        {
            var elements = new ProjectElement[]
            {
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m1", "v1"),
                    new ProjectMetadata("m2", "v2")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);

            CollectionAssert.AreEquivalent(elements, resolver.SelectMany(x => x).ToArray());
        }

        [TestMethod]
        public void TestSimpleTargets()
        {
            var elements = new ProjectElement[]
            {
                new ProjectTarget("target1",
                    new ProjectTask("t1", DummyParameters1),
                    new ProjectTask("t2", DummyParameters2)),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);

            CollectionAssert.AreEquivalent(elements, resolver.SelectMany(x => x).ToArray());
        }

        [TestMethod]
        [ExpectedException(typeof(ImplementationErrorException))]
        public void TestMetadataInItemDefinitionCheck()
        {
            var elements = new ProjectElement[]
            {
                new ProjectMetadata("m1", "v1"),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);

            resolver.SelectMany(x => x).ToArray();
        }

        [TestMethod]
        [ExpectedException(typeof(ImplementationErrorException))]
        public void TestTaskInTargetCheck()
        {
            var elements = new ProjectElement[]
            {
                new ProjectTask("t1", DummyParameters1),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);

            resolver.SelectMany(x => x).ToArray();
        }

        [TestMethod]
        public void TestPropertyDependencies1()
        {
            var elements = new ProjectElement[]
            {
                new ProjectProperty("p1", "$(p3)"),
                new ProjectProperty("p2", "$(p1)"),
                new ProjectProperty("p3", "v3"),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            var resolved = resolver.SelectMany(x => x).ToArray();

            Assert.AreEqual(elements[2], resolved[0]);
            Assert.AreEqual(elements[0], resolved[1]);
            Assert.AreEqual(elements[1], resolved[2]);
        }

        [TestMethod]
        public void TestPropertyDependencies2()
        {
            var elements = new ProjectElement[]
            {
                new ProjectProperty("p1", "v1", "'$(p3)'=='v3'"),
                new ProjectProperty("p2", "v2", "'$(p1)'=='v1'"),
                new ProjectProperty("p3", "v3"),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            var resolved = resolver.SelectMany(x => x).ToArray();

            Assert.AreEqual(elements[2], resolved[0]);
            Assert.AreEqual(elements[0], resolved[1]);
            Assert.AreEqual(elements[1], resolved[2]);
        }

        [TestMethod]
        public void TestPropertyGroupDependencies()
        {
            var elements = new ProjectElement[]
            {
                new ProjectPropertyGroup("'$(p3)'=='v3'",
                    new ProjectProperty("p1", "v1")),
                new ProjectPropertyGroup("'$(p1)'=='v1'",
                    new ProjectProperty("p2", "v2")),
                new ProjectPropertyGroup(
                    new ProjectProperty("p3", "v3")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            var resolved = resolver.SelectMany(x => x).ToArray();

            Assert.AreEqual(elements[2], resolved[0]);
            Assert.AreEqual(elements[0], resolved[1]);
            Assert.AreEqual(elements[1], resolved[2]);
        }

        [TestMethod]
        public void TestPropertyExternalDependency()
        {
            var elements = new ProjectElement[]
            {
                new ProjectProperty("p1", "$(e1)"),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);

            CollectionAssert.AreEquivalent(elements, resolver.SelectMany(x => x).ToArray());
        }

        [TestMethod]
        public void TestPropertySelfReference()
        {
            var elements = new ProjectElement[]
            {
                new ProjectProperty("p1", "v1;$(p1)"),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);

            CollectionAssert.AreEquivalent(elements, resolver.SelectMany(x => x).ToArray());
        }

        [TestMethod]
        [ExpectedException(typeof(ImplementationErrorException))]
        public void TestPropertyCircularDependencyDetection1()
        {
            var elements = new ProjectElement[]
            {
                new ProjectProperty("p1", "$(p2)"),
                new ProjectProperty("p2", "$(p1)"),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            resolver.SelectMany(x => x).ToArray();
        }

        [TestMethod]
        [ExpectedException(typeof(ImplementationErrorException))]
        public void TestPropertyCircularDependencyDetection2()
        {
            var elements = new ProjectElement[]
            {
                new ProjectPropertyGroup("'$(p2)'=='v2'",
                    new ProjectProperty("p1", "v1")),
                new ProjectPropertyGroup("'$(p1)'=='v1'",
                    new ProjectProperty("p2", "v2")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            resolver.SelectMany(x => x).ToArray();
        }

        [TestMethod]
        [ExpectedException(typeof(ImplementationErrorException))]
        public void TestPropertyCircularDependencyDetection3()
        {
            var elements = new ProjectElement[]
            {
                new ProjectPropertyGroup(
                    new ProjectProperty("p1", "$(p2)")),
                new ProjectPropertyGroup(
                    new ProjectProperty("p2", "$(p1)")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            resolver.SelectMany(x => x).ToArray();
        }

        [TestMethod]
        [ExpectedException(typeof(ImplementationErrorException))]
        public void TestPropertyCircularDependencyDetection4()
        {
            var elements = new ProjectElement[]
            {
                new ProjectProperty("p1", "$(p2)"),
                new ProjectPropertyGroup(
                    new ProjectProperty("p2", "v2", "'$(p3)'=='v3'")),
                new ProjectPropertyGroup(
                    new ProjectProperty("p3", "$(p1)")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            resolver.SelectMany(x => x).ToArray();
        }

        [TestMethod]
        public void TestMetadataDependencies1()
        {
            var elements = new ProjectElement[]
            {
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m1", "%(m3)")),
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m2", "%(m1)")),
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m3", "v3")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            var resolved = resolver.SelectMany(x => x).ToArray();

            Assert.AreEqual(elements[2], resolved[0]);
            Assert.AreEqual(elements[0], resolved[1]);
            Assert.AreEqual(elements[1], resolved[2]);
        }

        [TestMethod]
        public void TestMetadataDependencies2()
        {
            var elements = new ProjectElement[]
            {
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m1", "%(i3.m1)")),
                new ProjectItemDefinition("i2",
                    new ProjectMetadata("m1", "%(i1.m1)")),
                new ProjectItemDefinition("i3",
                    new ProjectMetadata("m1", "v1")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            var resolved = resolver.SelectMany(x => x).ToArray();

            Assert.AreEqual(elements[2], resolved[0]);
            Assert.AreEqual(elements[0], resolved[1]);
            Assert.AreEqual(elements[1], resolved[2]);
        }

        [TestMethod]
        public void TestMetadataDependencies3()
        {
            var elements = new ProjectElement[]
            {
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m1", "v1", "'%(m3)'=='v3'")),
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m2", "v2", "'%(m1)'=='v1'")),
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m3", "v3")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            var resolved = resolver.SelectMany(x => x).ToArray();

            Assert.AreEqual(elements[2], resolved[0]);
            Assert.AreEqual(elements[0], resolved[1]);
            Assert.AreEqual(elements[1], resolved[2]);
        }

        [TestMethod]
        public void TestMetadataDependencies4()
        {
            var elements = new ProjectElement[]
            {
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m1", "v1", "'%(i3.m1)'=='v1'")),
                new ProjectItemDefinition("i2",
                    new ProjectMetadata("m1", "v1", "'%(i1.m1)'=='v1'")),
                new ProjectItemDefinition("i3",
                    new ProjectMetadata("m1", "v1")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            var resolved = resolver.SelectMany(x => x).ToArray();

            Assert.AreEqual(elements[2], resolved[0]);
            Assert.AreEqual(elements[0], resolved[1]);
            Assert.AreEqual(elements[1], resolved[2]);
        }

        [TestMethod]
        public void TestItemDefinitionDependencies()
        {
            var elements = new ProjectElement[]
            {
                new ProjectItemDefinition("i1", "'$(p1)'=='v1'",
                    new ProjectMetadata("m1", "v1")),
                new ProjectPropertyGroup(
                    new ProjectProperty("p1", "v1")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            var resolved = resolver.SelectMany(x => x).ToArray();

            Assert.AreEqual(elements[0], resolved[1]);
            Assert.AreEqual(elements[1], resolved[0]);
        }

        [TestMethod]
        public void TestMetadataExternalDependency()
        {
            var elements = new ProjectElement[]
            {
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m1", "$(e1)")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);

            CollectionAssert.AreEquivalent(elements, resolver.SelectMany(x => x).ToArray());
        }

        [TestMethod]
        public void TestMetadataSelfReference()
        {
            var elements = new ProjectElement[]
            {
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m1", "v1;%(m1)")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);

            CollectionAssert.AreEquivalent(elements, resolver.SelectMany(x => x).ToArray());
        }

        [TestMethod]
        [ExpectedException(typeof(ImplementationErrorException))]
        public void TestMetadataCircularDependencyDetection()
        {
            var elements = new ProjectElement[]
            {
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m1", "%(m2)")),
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m2", "%(m1)")),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);
            resolver.SelectMany(x => x).ToArray();
        }

        [TestMethod]
        public void TestCombinedElements()
        {
            var elements = new ProjectElement[]
            {
                new ProjectPropertyGroup(
                    new ProjectProperty("p1", "v1")),
                new ProjectItemDefinition("i1",
                    new ProjectMetadata("m1", "v1")),
                new ProjectTarget("target1",
                    new ProjectTask("t1", DummyParameters1)),
            };

            var resolver = new ProjectElementConstructorDependencyResolver(elements);

            CollectionAssert.AreEquivalent(elements, resolver.SelectMany(x => x).ToArray());
        }
    }
}
