﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeMetaTest
{
    [TestClass]
    public class TestReadMetaFile : TestBase
    {
        private XmlFile descFile;

        public TestReadMetaFile()
        {
            descFile = new XmlFile();
            XmlElement elem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            elem.AddChild(defaultElem);
            XmlElement acidElement = new XmlElement("Acid");
            elem.AddChild(acidElement);

            // Meta ファイルの読み込みのみのテストなので、Acid のバイナリは何でもよい
            acidElement.InnerText = "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABBQ0lEgAEAAAAAAAAAAAAAAQAAAAAAAAD//wAAAAAAAEACAAAAAAAAQAIAAA0AAABQAgAAKAAAAAAAAAAAAAAAhVNlcnZlcgRDbGllKgAAAPdjAAHPAAAAPwgAgD8BAIA/EAAAvwEAAP8XgAD/vwkA/38AAv//AgAAAAAAAAAAAA==";

            descFile.AddLine(elem.GetXml());
            descFile.WriteData();
        }

        [TestMethod]
        public void TestMetaTag()
        {
            // Meta ファイルを読み込める
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // NintendoSdkMeta がルートとなっているファイルを読み込める
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("NintendoSdkMeta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 想定されていないタグを受け取らない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultMetaParam();
                elem.AddChild(defaultElem);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(null, descFile, meta, null, null);
            }

            // 重複を許可しない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                meta.AddLine(elem.GetXml());
                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(null, descFile, meta, null, null);
            }
        }

        [TestMethod]
        public void TestProgramId()
        {
            // 16 進数の ProgramId を読み込みる
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                core.RemoveChild("ProgramId");

                XmlElement programId = new XmlElement("ProgramId");
                core.AddChild(programId);
                programId.InnerText = "0x1";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 10 進数の ProgramId を読み込めない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                core.RemoveChild("ProgramId");

                XmlElement programId = new XmlElement("ProgramId");
                core.AddChild(programId);
                programId.InnerText = "1";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "Core/ProgramId"), descFile, meta, null, null);
            }

            // ProgramId が存在しないとエラー
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                core.RemoveChild("ProgramId");

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Core/ProgramId"), descFile, meta, null, null);
            }

            // 重複を許可しない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                core.RemoveChild("ProgramId");

                XmlElement programId = new XmlElement("ProgramId");
                core.AddChild(programId);
                programId.InnerText = "0x1";
                core.AddChild(programId);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "ProgramId"), descFile, meta, null, null);
            }

            // ProgramId の代わりに ApplicationId が使える
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                core.RemoveChild("ProgramId");

                XmlElement applicationId = new XmlElement("ApplicationId");
                core.AddChild(applicationId);
                applicationId.InnerText = "0x1";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // ProgramId と ApplicationId を重複して使えない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement applicationId = new XmlElement("ApplicationId");
                core.AddChild(applicationId);
                applicationId.InnerText = "0x1";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_InvalidOverlap, "Core/ProgramId", "Core/ApplicationId"), descFile, meta, null, null);
            }


        }

        [TestMethod]
        public void TestIs64BitInstruction()
        {
            const string Is64BitInstructionName = "Is64BitInstruction";
            // 重複を許可しない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                core.AddChild(item);
                item.InnerText = "TRUE";
                core.AddChild(item);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, Is64BitInstructionName), descFile, meta, null, null);
            }

            // True という文字列が受け取れる
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                core.AddChild(item);
                item.InnerText = "TRUE";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // False という文字列が受け取れる
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(Is64BitInstructionName);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                core.AddChild(item);
                item.InnerText = "false";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // True/False 以外の文字列を受け取らない ( True/False の小文字、大文字の違いは気にしない )
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(Is64BitInstructionName);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                core.AddChild(item);
                item.InnerText = "falure";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "Core/" + Is64BitInstructionName), descFile, meta, null, null);
            }
        }

        [TestMethod]
        public void TestProcessAddressSpace()
        {
            const string ProcessAddressSpaceName = "ProcessAddressSpace";

            // 重複を許可しない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                core.AddChild(item);
                item.InnerText = "AddressSpace64Bit";
                core.AddChild(item);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "ProcessAddressSpace"), descFile, meta, null, null);
            }

            // AddressSpace32Bit を受け付ける
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                core.AddChild(item);
                item.InnerText = "AddressSpace32Bit";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // AddressSpace64Bit を受け付ける
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                core.AddChild(item);
                item.InnerText = "AddressSpace64Bit";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // AddressSpace32BitNoReserved を受け付ける
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                core.AddChild(item);
                item.InnerText = "AddressSpace32BitNoReserved";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 異常値
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                core.AddChild(item);
                item.InnerText = "AddressSpace";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_InvalidString, ProcessAddressSpaceName, item.InnerText), descFile, meta, null, null);
            }
        }

        [TestMethod]
        public void TestMainThreadPriority()
        {
            const string MainThreadPriorityName = "MainThreadPriority";

            // 重複を許可しない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                core.AddChild(item);
                item.InnerText = "24";
                core.AddChild(item);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, MainThreadPriorityName), descFile, meta, null, null);
            }

            // 10 進数の数値を受け取れる
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                core.AddChild(item);
                item.InnerText = "24";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 16 進数の数値を受け取れない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                core.AddChild(item);
                item.InnerText = "0x18";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Core/" + MainThreadPriorityName), descFile, meta, null, null);
            }

            // 最小値
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                core.AddChild(item);
                item.InnerText = "0";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 最大値
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                core.AddChild(item);
                item.InnerText = "63";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 最小値 - 1
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                core.AddChild(item);
                item.InnerText = "-1";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Core/" + MainThreadPriorityName), descFile, meta, null, null);
            }

            // 最大値 + 1
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                core.AddChild(item);
                item.InnerText = "64";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_OutOfRange, "Core/" + MainThreadPriorityName, "64"), descFile, meta, null, null);
            }
        }

        [TestMethod]
        public void TestMainThreadCoreNumber()
        {
            const string MainThreadCoreNumberName = "MainThreadCoreNumber";

            // 重複を許可しない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                core.AddChild(item);
                item.InnerText = "24";
                core.AddChild(item);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, MainThreadCoreNumberName), descFile, meta, null, null);
            }

            // 10 進数の数値を受け取れる
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                core.AddChild(item);
                item.InnerText = "1";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 16 進数の数値を受け取れない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                core.AddChild(item);
                item.InnerText = "0x1";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Core/" + MainThreadCoreNumberName), descFile, meta, null, null);
            }

            // 最小値
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                core.AddChild(item);
                item.InnerText = "0";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 最大値
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                core.AddChild(item);
                item.InnerText = "255";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 最小値 - 1
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                core.AddChild(item);
                item.InnerText = "-1";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Core/" + MainThreadCoreNumberName), descFile, meta, null, null);
            }

            // 最大値 + 1
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                core.AddChild(item);
                item.InnerText = "256";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(null, descFile, meta, null, null);
            }
        }

        [TestMethod]
        public void TestMainThreadStackSize()
        {
            const string MainThreadStackSizeName = "MainThreadStackSize";

            // 重複を許可しない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                core.AddChild(item);
                item.InnerText = "0x4000";
                core.AddChild(item);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, MainThreadStackSizeName), descFile, meta, null, null);
            }

            // 10 進数の数値を受け取れない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                core.AddChild(item);
                item.InnerText = "4096";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "Core/" + MainThreadStackSizeName), descFile, meta, null, null);
            }

            // 16 進数の数値を受け取れる
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                core.AddChild(item);
                item.InnerText = "0x1000";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 最小値
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                core.AddChild(item);
                item.InnerText = "0x1000";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 最大値
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                core.AddChild(item);
                item.InnerText = "0xFFFFF000";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 最小値 - 0x1000
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                core.AddChild(item);
                item.InnerText = "0x0";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_Invalid0, "Core/" + MainThreadStackSizeName), descFile, meta, null, null);
            }

            // 最大値 + 0x1000
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                core.AddChild(item);
                item.InnerText = "0x100000000";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(null, descFile, meta, null, null);
            }

            // 0x1000 にアライメントされていないとエラー
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                core.AddChild(item);
                item.InnerText = "0x1001";

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidAlignment, "Core/" + MainThreadStackSizeName, 0x1000), descFile, meta, null, null);
            }
        }

        [TestMethod]
        public void TestVersion()
        {
            // 10 進数の Version を読み込める
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement version = new XmlElement("Version");
                version.InnerText = "1";
                core.AddChild(version);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 16 進数の Version を読み込めない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement version = new XmlElement("Version");
                version.InnerText = "0x1";
                core.AddChild(version);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Core/Version"), descFile, meta, null, null);
            }
        }

        [TestMethod]
        public void TestName()
        {
            // 15byte のUTF-8 の文字列を受け取る
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement name = new XmlElement("Name");
                name.InnerText = "あああああ";
                core.AddChild(name);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 16byte のUTF-8 の文字列を受け取らない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement name = new XmlElement("Name");
                name.InnerText = "あああああa";
                core.AddChild(name);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_InvalidStringLength, "Core/Name", 15), descFile, meta, null, null);
            }
        }

        [TestMethod]
        public void TestProcductCode()
        {
            // 15byte の ASCII の文字列を受け取る
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement name = new XmlElement("ProductCode");
                name.InnerText = "123456789012345";
                core.AddChild(name);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 16byte の ASCII の文字列を受け取らない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement name = new XmlElement("ProductCode");
                name.InnerText = "1234567890123456";
                core.AddChild(name);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_InvalidStringLength, "Core/ProductCode", 15), descFile, meta, null, null);
            }
        }

        [TestMethod]
        public void TestFsControlData()
        {
            // 正常値
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement fs = new XmlElement("FsAccessControlData");
                core.AddChild(fs);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 重複を許可しない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement fs = new XmlElement("FsAccessControlData");
                core.AddChild(fs);
                XmlElement fs2 = new XmlElement("FsAccessControlData");
                core.AddChild(fs2);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "FsAccessControlData"), descFile, meta, null, null);
            }
        }

        [TestMethod, TestCategory("MiddleTest")]
        public void TestSrvControlData()
        {
            XmlElement correctDescElem = TestDataForSrv.MakeSrvEntry("Server", "True");

            // 正常値
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlData");
                core.AddChild(srv);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 重複を許可しない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlData");
                core.AddChild(srv);
                XmlElement srv2 = new XmlElement("SrvAccessControlData");
                core.AddChild(srv2);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "SrvAccessControlData"), descFile, meta, null, null);
            }

            // エントリを登録できる
            SuccessTestSrvAccessControlDataWithSameSad(TestDataForSrv.MakeSrvEntry("Name", "False"));


            // エントリを複数個登録できる
            SuccessTestSrvAccessControlDataWithSameSad(new XmlElement[] {
                TestDataForSrv.MakeSrvEntry("Server", "True"),
                TestDataForSrv.MakeSrvEntry("Client", "False"),
            });

            // IsServer に True を指定できる
            SuccessTestSrvAccessControlDataWithSameSad(new XmlElement[] {
                TestDataForSrv.MakeSrvEntry("True", "True"),
                TestDataForSrv.MakeSrvEntry("true", "true"),
                TestDataForSrv.MakeSrvEntry("TRUE", "TRUE"),
                TestDataForSrv.MakeSrvEntry("tRUE", "tRUE"),
            });

            // IsServer に False を指定できる
            SuccessTestSrvAccessControlDataWithSameSad(new XmlElement[] {
                TestDataForSrv.MakeSrvEntry("False", "False"),
                TestDataForSrv.MakeSrvEntry("false", "false"),
                TestDataForSrv.MakeSrvEntry("FALSE", "FALSE"),
                TestDataForSrv.MakeSrvEntry("FALSe", "FALSe"),
            });

            XmlElement success = TestDataForSrv.MakeSrvEntry("Test", "True");

            // True/False 以外の文字列を受け取らない
            FailTestSrvAccessControlData(
                string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "Entry/IsServer"),
                success,
                TestDataForSrv.MakeSrvEntry("Test", "Failure"));

            // Entry が宣言されている場合、IsServer が指定されていないとエラーになる
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Entry/IsServer"),
                success,
                TestDataForSrv.MakeSrvEntry("Test", null));

            // IsServer を重複して宣言出来ない
            {
                FileManager desc = new FileManager();

                // Default 用
                XmlElement data = new XmlElement("Entry");
                {
                    XmlElement isServer = new XmlElement("IsServer");
                    data.AddChild(isServer);
                    isServer.InnerText = "True";

                    XmlElement isServer2 = new XmlElement("IsServer");
                    data.AddChild(isServer2);
                    isServer2.InnerText = "False";

                    XmlElement name = new XmlElement("Name");
                    data.AddChild(name);
                    name.InnerText = "Test";
                }

                FailTestSrvAccessControlData(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "IsServer"), correctDescElem, data);
            }

            // サービス名を指定できる
            SuccessTestSrvAccessControlDataWithSameSad(TestDataForSrv.MakeSrvEntry("Test", "True"));

            // 長さ 0 のサービス名を許容しない
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Entry/Name"),
                success,
                TestDataForSrv.MakeSrvEntry("", "True"));


            // 8文字の文字列を許容する
            SuccessTestSrvAccessControlDataWithSameSad(TestDataForSrv.MakeSrvEntry("12345678", "True"));

            // 9 文字以上の長さの文字列を許容しない
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_InvalidStringLength, "Entry/Name", 9),
                success,
                TestDataForSrv.MakeSrvEntry("123456789", "True"));

            // Entry が指定されている場合、 Name が指定されていないとエラーになる
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Entry/Name"),
                success,
                TestDataForSrv.MakeSrvEntry(null, "True"));

            // Name を重複して宣言出来ない
            {
                FileManager desc = new FileManager();

                // Default 用
                XmlElement data = new XmlElement("Entry");
                {
                    XmlElement isServer = new XmlElement("IsServer");
                    data.AddChild(isServer);
                    isServer.InnerText = "True";

                    XmlElement name = new XmlElement("Name");
                    data.AddChild(name);
                    name.InnerText = "Test";

                    XmlElement name2 = new XmlElement("Name");
                    data.AddChild(name2);
                    name.InnerText = "Name";
                }

                FailTestSrvAccessControlData(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "Name"), correctDescElem, data);
            }
        }

        [TestMethod]
        public void DescriptorTestForSrvControlData()
        {
            SuccessTestSrvAccessControlData(
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Client", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            SuccessTestSrvAccessControlData(
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Clie*", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            SuccessTestSrvAccessControlData(
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Cl*ie*", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Cl*ient", "False") }
                );

            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_DefaultNotPermitted, "Client", "SrvAccessControlDescriptor"),
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Client", "True") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_DefaultNotPermitted, "Server", "SrvAccessControlDescriptor"),
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Client", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "False"), TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_DefaultNotPermitted, "Server", "SrvAccessControlDescriptor"),
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Ser*", "True"), TestDataForSrv.MakeSrvEntry("Client", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "False"), TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
        }

        [TestMethod, TestCategory("LongTest")]
        public void TestKernelCapabilityData()
        {
            // 正常値
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement kc = new XmlElement("KernelCapabilityData");
                core.AddChild(kc);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                SuccessExecute(descFile, meta, null, null);
            }

            // 重複を許可しない
            {
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                XmlElement kc = new XmlElement("KernelCapabilityData");
                core.AddChild(kc);
                XmlElement kc2 = new XmlElement("KernelCapabilityData");
                core.AddChild(kc2);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "KernelCapabilityData"), descFile, meta, null, null);
            }

            // ThreadInfo を読み込める
            {
                XmlElement success = TestDataForKc.MakeThreadInfoElement("63", "0", "0", "255");
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeThreadInfoElement("0", "0", "0", "0"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeThreadInfoElement("63", "63", "255", "255"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/LowestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("-1", "0", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_ValueShouldBeUnsigned, "LowestPriority", 6),
                    success,
                    TestDataForKc.MakeThreadInfoElement("64", "0", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/LowestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement(null, "0", "0", "0"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/HighestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "-1", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_ValueShouldBeUnsigned, "HighestPriority", 6),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "64", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/HighestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", null, "0", "0"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/MinCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "-1", "0"));
                FailTestKernelCapabilityData(
                    null,
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "256", "0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/MinCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", null, "0"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/MaxCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", "-1"));
                FailTestKernelCapabilityData(
                    null,
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", "256"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/MaxCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", null));

                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_LessThan, "LowestPriority", "HighestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "1", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_LessThan, "MaxCoreNumber", "MinCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "1", "0"));
            }

            // EnableSystemCalls を読み込める
            {
                SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] { TestDataForKc.MakeEnableSystemCall("CreateThread") });
                SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] {
                    TestDataForKc.MakeEnableSystemCall("CreateThread"),
                    TestDataForKc.MakeEnableSystemCall("SleepThread"),
                });
                XmlElement[] success = TestDataForKc.MakeEnableSystemCalls(new string[] { "CreateThread", "SleepThread" });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_DuplicateDefinition, "EnableSystemCalls", "CreateThread"),
                    success,
                    new XmlElement[] {
                        TestDataForKc.MakeEnableSystemCall("CreateThread"),
                        TestDataForKc.MakeEnableSystemCall("CreateThread"),
                    }
                );
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWith, "EnableSystemCalls", "InvalidSvc"),
                    success,
                    new XmlElement[] {
                        TestDataForKc.MakeEnableSystemCall("CreateThread"),
                        TestDataForKc.MakeEnableSystemCall("InvalidSvc"),
                });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWith, "EnableSystemCalls", "1"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("1") });
            }

            // MapMemory を読み込める
            {
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x20000", "0x1000", "RO", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x30000", "0x1000", "RW", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x40000", "0x1000", "Rw", "Static"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x50000", "0x3000", "Ro", "Static"));
                SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] {
                    TestDataForKc.MakeMemoryMap("0x10000", "0x1000", "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x20000", "0x1000", "RO", "Io"),
                    TestDataForKc.MakeMemoryMap("0x30000", "0x2000", "RO", "Io"),
                    TestDataForKc.MakeMemoryMap("0x40000", "0x2000", "RW", "Static"),
                    TestDataForKc.MakeMemoryMap("0x50000", "0x2000", "Ro", "Static"),
                });
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x0", "0x3000", "Ro", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Ro", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0xFFFFFF000", "0x3000", "Ro", "Io"));

                XmlElement[] success = new XmlElement[] {
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Rw", "Static"),
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Ro", "Static"),
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Rw", "Io"),
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Ro", "Io"),
                };
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "MemoryMap/BeginAddress"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("10000", "0x3000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_ValueShouldBeUnsigned, "MemoryMap/BeginAddress", 35),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x1000000000", "0x3000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/BeginAddress"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap(null, "0x3000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_Invalid0, "MemoryMap/Size"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x0", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_InvalidAlign, "MemoryMap/Size", 0x1000),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x1", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    null,
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x100000000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/Size"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", null, "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString_2, "MemoryMap/Permission", "RO", "RW"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "RX", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/Permission"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", null, "Io") });
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString_2, "MemoryMap/Type", "Io", "Static"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Normal") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/Type"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", null) });
            }

            // EnableInterrupts を読み込める
            {
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeEnableInterrupt("10"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeEnableInterrupt("1022"));
                SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] {
                    TestDataForKc.MakeEnableInterrupt("23"),
                    TestDataForKc.MakeEnableInterrupt("45"),
                });
                SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] {
                    TestDataForKc.MakeEnableInterrupt("45"),
                });

                XmlElement success = TestDataForKc.MakeEnableInterrupt("1023");

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "EnableInterrupts"),
                    success,
                    TestDataForKc.MakeEnableInterrupt("0x10"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_DuplicateDefinition, "EnableInterrupts", "45"),
                    new XmlElement[] { success },
                    new XmlElement[] {
                        TestDataForKc.MakeEnableInterrupt("45"),
                        TestDataForKc.MakeEnableInterrupt("45"),
                    }
                );
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_Invalid0, "EnableInterrupts"),
                    success,
                    TestDataForKc.MakeEnableInterrupt("0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "EnableInterrupts"),
                    success,
                    TestDataForKc.MakeEnableInterrupt("-1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfRange, "EnableInterrupts", 1023),
                    success,
                    TestDataForKc.MakeEnableInterrupt("1023"));
            }

            // MiscParams を読み込める
            {
                XmlElement success = TestDataForKc.MakeMiscParams("1");
                SuccessTestKernelCapabilityDataWithSameKc(success);
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscParams("0"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscParams("7"));

                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfRange, "MiscParams/ProgramType", 8),
                    success,
                    TestDataForKc.MakeMiscParams("8"));
                FailTestKernelCapabilityData(
                    null,
                    success,
                    TestDataForKc.MakeMiscParams("-1"));
            }

            // HandleTableSize を読み込める
            {
                XmlElement success = TestDataForKc.MakeHandleTableSize("1023");
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeHandleTableSize("0"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeHandleTableSize("256"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeHandleTableSize("1023"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidValue_Bigger, "KernelCapabilityData/HandleTableSize", 1023),
                    success,
                    TestDataForKc.MakeHandleTableSize("1024"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "KernelCapabilityData/HandleTableSize"),
                    success,
                    TestDataForKc.MakeHandleTableSize("-1"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "KernelCapabilityData/HandleTableSize"),
                    success,
                    TestDataForKc.MakeHandleTableSize("0x1"));
            }

            // MiscFlags を読み込める
            {
                XmlElement success = TestDataForKc.MakeMiscFlags("True", "True");
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags("True", "True"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags("False", "False"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags("True", null));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags(null, "True"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags("FALSE", null));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags(null, "FALSE"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "MiscFlags/EnableDebug"),
                    success,
                    TestDataForKc.MakeMiscFlags("Success", "FALSE"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "MiscFlags/ForceDebug"),
                    success,
                    TestDataForKc.MakeMiscFlags("True", "Fail"));
            }

            // KernelVersion が一致していると成功する
            {
                FileManager outputDesc = new FileManager();
                XmlFile testMeta = new XmlFile();
                MakeDescForKernelCapability(ref outputDesc, null, "3.0");

                MakeMetaForKernelCapability(ref testMeta, null);

                SuccessExecute(outputDesc, testMeta, "3.0");
            }

            // KernelVersion が一致していないと失敗する
            {
                FileManager outputDesc = new FileManager();
                XmlFile testMeta = new XmlFile();
                MakeDescForKernelCapability(ref outputDesc, null, "3.0");

                MakeMetaForKernelCapability(ref testMeta, null);

                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "KernelVersion"), outputDesc, testMeta, "2.0");
            }
        }

        [TestMethod, TestCategory("MiddleTest")]
        public void DescriptorTestForKernelCapability()
        {
            // ThreadInfo
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeThreadInfoElement("63", "0", "0", "4"), TestDataForKc.MakeThreadInfoElement("59", "22", "1", "3"));

                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "ThreadInfo/LowestPriority"),
                    TestDataForKc.MakeThreadInfoElement("1", "0", "0", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "0", "1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "ThreadInfo/HighestPriority"),
                    TestDataForKc.MakeThreadInfoElement("2", "1", "0", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "0", "1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "ThreadInfo/MinCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("2", "0", "1", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "0", "1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "ThreadInfo/MaxCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("2", "0", "1", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "1", "2"));
            }

            // EnableSystemCalls
            {
                SuccessTestKernelCapabilityData(
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("CreateThread"), TestDataForKc.MakeEnableSystemCall("SleepThread") },
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("CreateThread") }
                    );

                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWith, "EnableSystemCalls", "SetHeapSize"),
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("CreateThread"), TestDataForKc.MakeEnableSystemCall("SleepThread") },
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("SetHeapSize") }
                    );
            }

            // MapMemory を読み込める
            {
                SuccessTestKernelCapabilityData(
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x2000", "Ro", "Io"));
                SuccessTestKernelCapabilityData(
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Rw", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x2000", "Rw", "Io"));
                SuccessTestKernelCapabilityData(
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x1000", "Ro", "Io"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWithMemoryMap, "KernelCapabilityData", "MemoryMap", 0x10000, 0x2000, "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x2000", "Rw", "Io"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWithMemoryMap, "KernelCapabilityData", "MemoryMap", 0x10000, 0x4000, "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Rw", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x4000", "Rw", "Io"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWithMemoryMap, "KernelCapabilityData", "MemoryMap", 0x10000, 0x1000, "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Rw", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x1000", "Rw", "Io"));
            }

            // EnableInterrupts を読み込める
            {
                SuccessTestKernelCapabilityData(
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8"), TestDataForKc.MakeEnableInterrupt("9") },
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8") }
                );
                SuccessTestKernelCapabilityData(
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("1023") },
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8"), TestDataForKc.MakeEnableInterrupt("9") }
                );
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWith, "EnableInterrupts",  8),
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("9") },
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8"), TestDataForKc.MakeEnableInterrupt("9") }
                );

            }

            // MiscParams を読み込める
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscParams("0"), TestDataForKc.MakeMiscParams("0"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscParams("1"), TestDataForKc.MakeMiscParams("1"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscParams("7"), TestDataForKc.MakeMiscParams("7"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "MiscParams/ProgramType"),
                    TestDataForKc.MakeMiscParams("1"), TestDataForKc.MakeMiscParams("0"));
            }

            // HandleTableSize を読み込める
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeHandleTableSize("256"), TestDataForKc.MakeHandleTableSize("0"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeHandleTableSize("256"), TestDataForKc.MakeHandleTableSize("10"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "KernelCapabilityData/HandleTableSize"),
                    TestDataForKc.MakeHandleTableSize("256"), TestDataForKc.MakeHandleTableSize("512"));
            }

            // MiscFlags を読み込める
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("True", "True"), TestDataForKc.MakeMiscFlags("True", "False"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags("False", "True"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags("False", "False"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags(null, null));

                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "MiscFlags/EnableDebug"),
                    TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags("True", null));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "MiscFlags/ForceDebug"),
                    TestDataForKc.MakeMiscFlags("True", "False"), TestDataForKc.MakeMiscFlags("True", "True"));
            }
        }


        /* KernelCapability 用のユーティリティ */

        void MakeDescForKernelCapability(ref FileManager outputDesc, XmlElement[] descs, string kernelVersion = null)
        {
            // Desc
            XmlFile descFile = new XmlFile();
            XmlElement descElem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            descElem.AddChild(defaultElem);
            string baseId = Path.Combine(new string[] { GetSvcHeaderDirPath(), BaseIdHeader });

            if (descs != null)
            {
                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                foreach (var elem in descs)
                {
                    kcd.AddChild(elem);
                }

                descElem.AddChild(kcd);
            }

            descFile.AddLine(descElem.GetXml());
            descFile.WriteData();

            CreateDescFile(outputDesc, descFile, new string[] { baseId }, kernelVersion);
        }

        void MakeMetaForKernelCapability(ref XmlFile outputMeta, XmlElement[] metas)
        {
            XmlElement elem = new XmlElement("Meta");
            XmlElement core = new XmlElement("Core");
            elem.AddChild(core);
            XmlElement programId = new XmlElement("ProgramId");
            programId.InnerText = "0x0005000C10000000";
            core.AddChild(programId);

            if (metas != null)
            {
                XmlElement kc = new XmlElement("KernelCapabilityData");
                foreach (var m in metas)
                {
                    kc.AddChild(m);
                }

                core.AddChild(kc);
            }

            outputMeta.AddLine(elem.GetXml());
            outputMeta.WriteData();
        }

        void SuccessTestKernelCapabilityData(XmlElement[] descs, XmlElement[] metas, string kernelVersion = null)
        {
            FileManager outputDesc = new FileManager();
            MakeDescForKernelCapability(ref outputDesc, descs, kernelVersion);

            XmlFile testMeta = new XmlFile();
            MakeMetaForKernelCapability(ref testMeta, metas);

            SuccessExecute(outputDesc, testMeta, kernelVersion);
        }

        void FailTestKernelCapabilityData(string errMsg, XmlElement[] descs, XmlElement[] metas, string kernelVersion = null)
        {
            FileManager outputDesc = new FileManager();
            MakeDescForKernelCapability(ref outputDesc, descs, kernelVersion);

            XmlFile testMeta = new XmlFile();
            MakeMetaForKernelCapability(ref testMeta, metas);


            FailExecute(errMsg, outputDesc, testMeta, kernelVersion);
        }

        // ラッパー
        void SuccessTestKernelCapabilityDescriptor(XmlElement elem)
        {
            SuccessTestKernelCapabilityDescriptor(new XmlElement[] { elem });
        }
        void SuccessTestKernelCapabilityDescriptor(XmlElement[] elems)
        {
            SuccessTestKernelCapabilityData(elems, null);
        }
        void FailTestKernelCapabilityDescriptor(string errMsg, XmlElement elem)
        {
            FailTestKernelCapabilityDescriptor(errMsg, new XmlElement[] { elem });
        }
        void FailTestKernelCapabilityDescriptor(string errMsg, XmlElement[] elems)
        {
            FailTestKernelCapabilityData(errMsg, elems, null);
        }
        void SuccessTestKernelCapabilityDataWithSameKc(XmlElement elem)
        {
            SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] { elem });
        }
        void SuccessTestKernelCapabilityDataWithSameKc(XmlElement[] elems)
        {
            SuccessTestKernelCapabilityData(elems, elems);
        }
        void SuccessTestKernelCapabilityData(XmlElement desc, XmlElement meta)
        {
            SuccessTestKernelCapabilityData(new XmlElement[] { desc }, new XmlElement[] { meta });
        }
        void FailTestKernelCapabilityData(string errMsg, XmlElement desc, XmlElement meta, string kernelVersion = null)
        {
            FailTestKernelCapabilityData(errMsg, new XmlElement[] { desc }, new XmlElement[] { meta }, kernelVersion);
        }

        /* SrviceAccessControl 用のユーティリティ */

        void MakeDescFileForSrvAccessControlData(ref FileManager outputDesc, XmlElement[] descs, string kernelVersion = null)
        {
            XmlFile descFile = new XmlFile();
            XmlElement descElem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            descElem.AddChild(defaultElem);
            string baseId = Path.Combine(new string[] { GetSvcHeaderDirPath(), BaseIdHeader });

            XmlElement sad = new XmlElement("SrvAccessControlDescriptor");
            if (descs != null)
            {
                foreach (var elem in descs)
                {
                    sad.AddChild(elem);
                }
            }
            descElem.AddChild(sad);

            descFile.AddLine(descElem.GetXml());
            descFile.WriteData();

            CreateDescFile(outputDesc, descFile, new string[] { baseId }, kernelVersion);
        }

        void MakeMetaFileForSrvAccessControlData(ref XmlFile outputMeta, XmlElement[] metas)
        {
            XmlElement elem = new XmlElement("Meta");
            XmlElement core = new XmlElement("Core");
            elem.AddChild(core);
            XmlElement programId = new XmlElement("ProgramId");
            programId.InnerText = "0x0005000C10000000";
            core.AddChild(programId);

            if (metas != null)
            {
                XmlElement srv = new XmlElement("SrvAccessControlData");
                foreach (var m in metas)
                {
                    srv.AddChild(m);
                }

                core.AddChild(srv);
            }

            outputMeta.AddLine(elem.GetXml());
            outputMeta.WriteData();
        }

        void SuccessTestSrvAccessControlData(XmlElement[] descs, XmlElement[] metas)
        {
            FileManager desc = new FileManager();
            MakeDescFileForSrvAccessControlData(ref desc, descs);

            XmlFile testMeta = new XmlFile();
            MakeMetaFileForSrvAccessControlData(ref testMeta, metas);

            SuccessExecute(desc, testMeta, null, null);
        }

        void FailTestSrvAccessControlData(string errMsg, XmlElement[] descs, XmlElement[] metas)
        {
            FileManager desc = new FileManager();
            MakeDescFileForSrvAccessControlData(ref desc, descs);

            XmlFile testMeta = new XmlFile();
            MakeMetaFileForSrvAccessControlData(ref testMeta, metas);

            FailExecute(errMsg, desc, testMeta, null, null);
        }

        // ラッパー
        void SuccessTestSrvAccessControlData(XmlElement desc, XmlElement meta)
        {
            SuccessTestKernelCapabilityData(new XmlElement[] { desc }, new XmlElement[] { meta });
        }
        void FailTestSrvAccessControlData(string errMsg, XmlElement desc, XmlElement meta)
        {
            FailTestSrvAccessControlData(errMsg, new XmlElement[] { desc }, new XmlElement[] { meta });
        }
        void SuccessTestSrvAccessControlDataWithSameSad(XmlElement elem)
        {
            SuccessTestSrvAccessControlData(new XmlElement[] { elem }, new XmlElement[] { elem });
        }
        void SuccessTestSrvAccessControlDataWithSameSad(XmlElement[] elems)
        {
            SuccessTestSrvAccessControlData(elems, elems);
        }
    }
}
