﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeMetaTest
{
    [TestClass]
    public class TestMerge : TestBase
    {
        public byte[] GetAciFromNpdm(FileManager npdm)
        {
            byte[] metaBinary = npdm.ReadFile(MetaHeaderSize, MetaOffset);
            uint aciOffset = BitConverter.ToUInt32(metaBinary, MetaAciOffsetOffset);
            uint aciSize = BitConverter.ToUInt32(metaBinary, MetaAciSizeOffset);
            return npdm.ReadFile((int)aciSize, aciOffset);
        }

        [TestMethod]
        public void TestFac()
        {
            // .desc 宣言、.nmeta 空
            {
                FileManager desc = new FileManager();
                {
                    XmlElement flag1 = new XmlElement("FlagPresets");
                    XmlElement flag2 = new XmlElement("FlagPresets");
                    flag1.InnerText = "SystemSaveData";
                    flag2.InnerText = "BisAllRaw";
                    var elems = new XmlElement[] { flag1, flag2 };
                    MakeDesc(ref desc, MakeDescriptorElem(elems, CapabilityType.Fs), MakeDataElem(elems, CapabilityType.Fs), null);
                }

                XmlFile meta = new XmlFile();
                {
                    XmlElement elem = new XmlElement("Meta");
                    XmlElement core = new XmlElement("Core");
                    XmlElement defaultElem = CreateDefaultMetaParam();
                    core.AddChild(defaultElem);
                    elem.AddChild(core);

                    meta.AddLine(elem.GetXml());
                    meta.WriteData();
                }

                FileManager npdmFile = SuccessExecute(desc, meta, null, null);
                byte[] aci = GetAciFromNpdm(npdmFile);

                int facSize = BitConverter.ToInt32(aci, AciFacSizeOffset);
                int facOffset = BitConverter.ToInt32(aci, AciFacOffsetOffset);
                byte[] fac = new byte[facSize];
                Array.Copy(aci, facOffset, fac, 0, facSize);
                var facFlags = BitConverter.ToUInt64(fac, 4);
                Assert.IsTrue(facFlags == 0x0000000000000088);
            }

            // .desc 宣言、.nmeta 宣言
            {
                FileManager desc = new FileManager();
                {
                    XmlElement flag1 = new XmlElement("FlagPresets");
                    XmlElement flag2 = new XmlElement("FlagPresets");
                    flag1.InnerText = "SystemSaveData";
                    flag2.InnerText = "BisAllRaw";
                    var elems = new XmlElement[] { flag1, flag2 };
                    MakeDesc(ref desc, MakeDescriptorElem(elems, CapabilityType.Fs), MakeDataElem(elems, CapabilityType.Fs), null);
                }

                XmlFile meta = new XmlFile();
                {
                    XmlElement elem = new XmlElement("Meta");
                    XmlElement core = new XmlElement("Core");
                    XmlElement fad = new XmlElement("FsAccessControlData");
                    XmlElement flag1 = new XmlElement("FlagPresets");
                    XmlElement defaultElem = CreateDefaultMetaParam();
                    flag1.InnerText = "Debug";
                    fad.AddChild(flag1);
                    core.AddChild(defaultElem);
                    core.AddChild(fad);
                    elem.AddChild(core);

                    meta.AddLine(elem.GetXml());
                    meta.WriteData();
                }

                FileManager npdmFile = SuccessExecute(desc, meta, null, null);
                byte[] aci = GetAciFromNpdm(npdmFile);

                int facSize = BitConverter.ToInt32(aci, AciFacSizeOffset);
                int facOffset = BitConverter.ToInt32(aci, AciFacOffsetOffset);
                byte[] fac = new byte[facSize];
                Array.Copy(aci, facOffset, fac, 0, facSize);
                var facFlags = BitConverter.ToUInt64(fac, 4);
                Assert.IsTrue(facFlags == 0x4000000000000000);
            }
        }

        [TestMethod]
        public void TestSac()
        {
            // .desc 空、.nmeta 宣言
            {
                string name = "Test";
                XmlElement entry = TestDataForSrv.MakeSrvEntry(name, "True");
                FileManager desc = new FileManager();
                MakeDesc(ref desc, MakeDescriptorElem(new XmlElement[] { entry }, CapabilityType.Service), null);

                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);
                core.AddChild(MakeDataElem(new XmlElement[] { entry }, CapabilityType.Service));

                meta.AddLine(elem.GetXml());
                meta.WriteData();

                FileManager npdmFile = SuccessExecute(desc, meta, null, null);
                byte[] aci = GetAciFromNpdm(npdmFile);

                int sacSize = name.Length + 1;
                int sacOffset = BitConverter.ToInt32(aci, AciSacOffsetOffset);
                byte[] entryBinary = new byte[sacSize];
                Array.Copy(aci, sacOffset, entryBinary, 0, sacSize);
                Assert.IsTrue(entryBinary[0] == 0x83);
                Assert.IsTrue(System.Text.Encoding.ASCII.GetString(entryBinary, 1, sacSize - 1) == name);
            }

            // .desc 宣言、.nmeta 空
            {
                string name = "Test";
                XmlElement entry = TestDataForSrv.MakeSrvEntry(name, "True");
                FileManager desc = new FileManager();
                MakeDesc(ref desc,
                    MakeDescriptorElem(new XmlElement[] { entry }, CapabilityType.Service),
                    MakeDataElem(new XmlElement[] { entry }, CapabilityType.Service)
                    );

                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                meta.AddLine(elem.GetXml());
                meta.WriteData();

                FileManager npdmFile = SuccessExecute(desc, meta, null, null);
                byte[] aci = GetAciFromNpdm(npdmFile);

                int sacSize = name.Length + 1;
                int sacOffset = BitConverter.ToInt32(aci, AciSacOffsetOffset);
                byte[] entryBinary = new byte[sacSize];
                Array.Copy(aci, sacOffset, entryBinary, 0, sacSize);
                Assert.IsTrue(entryBinary[0] == 0x83);
                Assert.IsTrue(System.Text.Encoding.ASCII.GetString(entryBinary, 1, sacSize - 1) == name);
            }

            // .desc 宣言、.nmeta 宣言
            {
                string clientName = "Client";
                string serverName = "Server";
                XmlElement client = TestDataForSrv.MakeSrvEntry(clientName, "False");
                XmlElement server = TestDataForSrv.MakeSrvEntry(serverName, "True");
                FileManager desc = new FileManager();
                MakeDesc(ref desc,
                    MakeDescriptorElem(new XmlElement[] { client, server }, CapabilityType.Service),
                    MakeDataElem(new XmlElement[] { client }, CapabilityType.Service)
                    );

                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);
                core.AddChild(MakeDataElem(new XmlElement[] { server }, CapabilityType.Service));

                meta.AddLine(elem.GetXml());
                meta.WriteData();

                FileManager npdmFile = SuccessExecute(desc, meta, null, null);
                byte[] aci = GetAciFromNpdm(npdmFile);

                int sacSize = serverName.Length + 1;
                int sacOffset = BitConverter.ToInt32(aci, AciSacOffsetOffset);
                byte[] entryBinary = new byte[sacSize];
                Array.Copy(aci, sacOffset, entryBinary, 0, sacSize);
                Assert.IsTrue(entryBinary[0] == 0x85);
                Assert.IsTrue(System.Text.Encoding.ASCII.GetString(entryBinary, 1, sacSize - 1) == serverName);
            }
        }

        [TestMethod]
        public void TestKc()
        {
            // .desc 空、.nmeta 宣言
            {
                XmlElement kc = TestDataForKc.MakeHandleTableSize("1023");

                FileManager desc = new FileManager();
                MakeDesc(ref desc, MakeDescriptorElem(new XmlElement[] { kc }, CapabilityType.Kernel), null);

                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);
                core.AddChild(MakeDataElem(new XmlElement[] { kc }, CapabilityType.Kernel));

                meta.AddLine(elem.GetXml());
                meta.WriteData();

                FileManager npdmFile = SuccessExecute(desc, meta, null, null);
                byte[] aci = GetAciFromNpdm(npdmFile);

                int kcOffset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, kcOffset) == 0x3FF7FFF);
            }

            // .desc 宣言、.nmeta 空
            {
                XmlElement kc = TestDataForKc.MakeHandleTableSize("1023");

                FileManager desc = new FileManager();
                MakeDesc(ref desc, MakeDescriptorElem(new XmlElement[] { kc }, CapabilityType.Kernel), MakeDataElem(new XmlElement[] { kc }, CapabilityType.Kernel));

                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                meta.AddLine(elem.GetXml());
                meta.WriteData();

                FileManager npdmFile = SuccessExecute(desc, meta, null, null);
                byte[] aci = GetAciFromNpdm(npdmFile);

                int kcOffset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, kcOffset) == 0x3FF7FFF);
            }

            // .desc 宣言、.nmeta 宣言
            {
                XmlElement big = TestDataForKc.MakeHandleTableSize("1023");
                XmlElement small = TestDataForKc.MakeHandleTableSize("256");

                FileManager desc = new FileManager();
                MakeDesc(ref desc, MakeDescriptorElem(new XmlElement[] { big }, CapabilityType.Kernel), MakeDataElem(new XmlElement[] { small }, CapabilityType.Kernel));

                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);
                core.AddChild(MakeDataElem(new XmlElement[] { big }, CapabilityType.Kernel));

                meta.AddLine(elem.GetXml());
                meta.WriteData();

                FileManager npdmFile = SuccessExecute(desc, meta, null, null);
                byte[] aci = GetAciFromNpdm(npdmFile);

                int kcOffset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, kcOffset) == 0x3FF7FFF);
            }

            // .desc 宣言、.nmeta 宣言
            // エントリ毎に判別される
            {
                XmlElement kcd = TestDataForKc.MakeMiscFlags("True", null);
                XmlElement kc = TestDataForKc.MakeHandleTableSize("1023");

                FileManager desc = new FileManager();
                MakeDesc(ref desc, MakeDescriptorElem(
                    new XmlElement[] { kcd, kc }, CapabilityType.Kernel),
                    MakeDataElem(new XmlElement[] { kcd }, CapabilityType.Kernel));

                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);
                core.AddChild(MakeDataElem(new XmlElement[] { kc }, CapabilityType.Kernel));

                meta.AddLine(elem.GetXml());
                meta.WriteData();

                FileManager npdmFile = SuccessExecute(desc, meta, null, null);
                byte[] aci = GetAciFromNpdm(npdmFile);

                int kcOffset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                int kcSize = BitConverter.ToInt32(aci, AciKcSizeOffset);
                Assert.IsTrue(kcSize == KernelCapabilityEntrySize * 2);
                Assert.IsTrue(BitConverter.ToUInt32(aci, kcOffset) == 0x3FF7FFF);
                Assert.IsTrue(BitConverter.ToUInt32(aci, kcOffset + KernelCapabilityEntrySize) == 0x2FFFF);
            }
        }
    }
}
