﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using MakeAddrRegionHeader;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeAddrRegionHeaderTest
{
    [TestClass]
    public class MakeAddrRegionHeaderTest
    {
        [TestMethod]
        public void SuccessTest()
        {
            var source = @"// タイプ0: 定数
ZERO	 =0x0000_0000 	// 定数

// タイプ1: begin, size 指定
ALL                 begin=0x00000000 size=0xfffff000

// タイプ2: begin, end 指定
KERNEL              begin=0xf0000000 end=0xff000000

// タイプ3: 領域相対指定 (inner|outer) offset, size
KERNEL_HEAP                 KERNEL     inner -0x0100_0000 size=0x0100_0000
HWREG                       KERNEL     outer -0x0010_0000 size=0x0010_0000
KERNEL_SECOND_PAGE          KERNEL     inner +0x0000_1000 size=0x0000_1000
KERNEL_SECOND_TRAILING_PAGE KERNEL     outer +0x0000_1000 size=0x0000_1000

HWREG_GPIO                  HWREG      inner +0x0000_0000 size=0x0000_1000

// タイプ4: 領域相対指定 shift offset
HWREG_UART0                 HWREG_GPIO shift +0x0000_1000

// タイプ5: alias 指定
HWREG_UART                  alias HWREG_UART0
KERNEL_LOAD_ADDRESS         alias KERNEL begin";

            var expected = @"#define NN_KERN_ADDRESS_ZERO NN_KERN_ADDRESS_CONVERT(0x00000000)

#define NN_KERN_ADDRESS_ALL_BEGIN NN_KERN_ADDRESS_CONVERT(0x00000000)
#define NN_KERN_ADDRESS_ALL_END   NN_KERN_ADDRESS_CONVERT(0xfffff000)
#define NN_KERN_ADDRESS_ALL_SIZE  NN_KERN_ADDRESS_CONVERT(0xfffff000)

#define NN_KERN_ADDRESS_KERNEL_BEGIN NN_KERN_ADDRESS_CONVERT(0xf0000000)
#define NN_KERN_ADDRESS_KERNEL_END   NN_KERN_ADDRESS_CONVERT(0xff000000)
#define NN_KERN_ADDRESS_KERNEL_SIZE  NN_KERN_ADDRESS_CONVERT(0x0f000000)

#define NN_KERN_ADDRESS_KERNEL_HEAP_BEGIN NN_KERN_ADDRESS_CONVERT(0xfd000000)
#define NN_KERN_ADDRESS_KERNEL_HEAP_END   NN_KERN_ADDRESS_CONVERT(0xfe000000)
#define NN_KERN_ADDRESS_KERNEL_HEAP_SIZE  NN_KERN_ADDRESS_CONVERT(0x01000000)

#define NN_KERN_ADDRESS_HWREG_BEGIN NN_KERN_ADDRESS_CONVERT(0xefe00000)
#define NN_KERN_ADDRESS_HWREG_END   NN_KERN_ADDRESS_CONVERT(0xeff00000)
#define NN_KERN_ADDRESS_HWREG_SIZE  NN_KERN_ADDRESS_CONVERT(0x00100000)

#define NN_KERN_ADDRESS_KERNEL_SECOND_PAGE_BEGIN NN_KERN_ADDRESS_CONVERT(0xf0001000)
#define NN_KERN_ADDRESS_KERNEL_SECOND_PAGE_END   NN_KERN_ADDRESS_CONVERT(0xf0002000)
#define NN_KERN_ADDRESS_KERNEL_SECOND_PAGE_SIZE  NN_KERN_ADDRESS_CONVERT(0x00001000)

#define NN_KERN_ADDRESS_KERNEL_SECOND_TRAILING_PAGE_BEGIN NN_KERN_ADDRESS_CONVERT(0xff001000)
#define NN_KERN_ADDRESS_KERNEL_SECOND_TRAILING_PAGE_END   NN_KERN_ADDRESS_CONVERT(0xff002000)
#define NN_KERN_ADDRESS_KERNEL_SECOND_TRAILING_PAGE_SIZE  NN_KERN_ADDRESS_CONVERT(0x00001000)

#define NN_KERN_ADDRESS_HWREG_GPIO_BEGIN NN_KERN_ADDRESS_CONVERT(0xefe00000)
#define NN_KERN_ADDRESS_HWREG_GPIO_END   NN_KERN_ADDRESS_CONVERT(0xefe01000)
#define NN_KERN_ADDRESS_HWREG_GPIO_SIZE  NN_KERN_ADDRESS_CONVERT(0x00001000)

#define NN_KERN_ADDRESS_HWREG_UART0_BEGIN NN_KERN_ADDRESS_CONVERT(0xefe01000)
#define NN_KERN_ADDRESS_HWREG_UART0_END   NN_KERN_ADDRESS_CONVERT(0xefe02000)
#define NN_KERN_ADDRESS_HWREG_UART0_SIZE  NN_KERN_ADDRESS_CONVERT(0x00001000)

#define NN_KERN_ADDRESS_HWREG_UART_BEGIN NN_KERN_ADDRESS_CONVERT(0xefe01000)
#define NN_KERN_ADDRESS_HWREG_UART_END   NN_KERN_ADDRESS_CONVERT(0xefe02000)
#define NN_KERN_ADDRESS_HWREG_UART_SIZE  NN_KERN_ADDRESS_CONVERT(0x00001000)

#define NN_KERN_ADDRESS_KERNEL_LOAD_ADDRESS NN_KERN_ADDRESS_CONVERT(0xf0000000)

";

            string actual = RunMakeAddrRegionHeader(source, AddrWidth.Bit32);
            Assert.IsTrue(actual != null);
            Assert.AreEqual(expected, actual);
        }

        [TestMethod]
        public void LastPageTest32()
        {
            var source = @"LAST_PAGE           begin=0xfffffc00 size=0x00000400";

            var expected = @"#define NN_KERN_ADDRESS_LAST_PAGE_BEGIN NN_KERN_ADDRESS_CONVERT(0xfffffc00)
#define NN_KERN_ADDRESS_LAST_PAGE_SIZE  NN_KERN_ADDRESS_CONVERT(0x00000400)

";

            string actual = RunMakeAddrRegionHeader(source, AddrWidth.Bit32);
            Assert.IsTrue(actual != null);
            Assert.AreEqual(expected, actual);
        }

        [TestMethod]
        public void LastPageTest64()
        {
            var source = @"LAST_PAGE           begin=0xffffffff_fffffc00 size=0x00000000_00000400";

            var expected = @"#define NN_KERN_ADDRESS_LAST_PAGE_BEGIN NN_KERN_ADDRESS_CONVERT(0xfffffffffffffc00)
#define NN_KERN_ADDRESS_LAST_PAGE_SIZE  NN_KERN_ADDRESS_CONVERT(0x0000000000000400)

";

            string actual = RunMakeAddrRegionHeader(source, AddrWidth.Bit64);
            Assert.IsTrue(actual != null);
            Assert.AreEqual(expected, actual);
        }

        private static string RunMakeAddrRegionHeader(string source, AddrWidth addrWidth)
        {
            var inputPath = Path.GetTempFileName();
            var outputPath = Path.GetTempFileName();
            try
            {
                File.WriteAllText(inputPath, source);

                var args = new List<string>() {
                    "--input", inputPath,
                    "--output", outputPath,
                    "--prefix", "NN_KERN_ADDRESS_",
                    "--address-convert-macro", "CONVERT",
                };

                if (addrWidth == AddrWidth.Bit64)
                {
                    args.Add("--64");
                }

                var param = MakeAddrRegionHeader.Program.ParseArgs(args.ToArray());
                Assert.IsNotNull(param);
                var maker = new AddrRegionHeaderMaker(param);
                maker.MakeAddrRegionHeader();
                return maker.IsError ? null : File.ReadAllText(outputPath);
            }
            finally
            {
                File.Delete(inputPath);
                File.Delete(outputPath);
            }
        }

        [TestMethod]
        public void SyntaxTest()
        {
            ExecuteTest(AddrWidth.Bit32, new[] {
                // 数値の形式
                TestData.Failure("MinusOne            =-1"),
                TestData.Failure("MinusOne            =-0x00000001"),
                // 名前の形式
                TestData.Failure("1_Value             =0x00000001"),
            });
        }

        [TestMethod]
        public void DuplicatedNameTest()
        {
            ExecuteTest(AddrWidth.Bit32, new[] {
                // 定数の多重定義
                TestData.Success("Constant            =0x00000001"),
                TestData.Failure("Constant            =0x00000001"),
                // 領域の多重定義
                TestData.Success("Region              begin=0x00000000 size=0xffffffff"),
                TestData.Failure("Region              begin=0x00000000 size=0xffffffff"),
                // 領域のシンボルとの衝突
                TestData.Failure("Region_BEGIN        =0x00000001"),
                // 定数との衝突
                TestData.Success("Region2_BEGIN       =0x00000001"),
                TestData.Failure("Region2             begin=0x00000000 size=0xffffffff"),
            });
        }

        [TestMethod]
        public void BoundaryTest_32()
        {
            ExecuteTest(AddrWidth.Bit32, new[] {
                // 数値が大きい
                TestData.Failure("FourG0              =0x0000_0001_0000_0000"),
                TestData.Success("FourGminusOne0      =0x0000_0000_ffff_ffff"),
                // タイプ1: begin, size 指定
                // 終端がオーバーフロー
                TestData.Failure("FourGplusOneEnd1     begin=0x00000002 size=0xffffffff"),
                TestData.Success("FourGEnd1            begin=0x00000001 size=0xffffffff"),
                TestData.Success("FourGminusOneEnd1    begin=0x00000000 size=0xffffffff"),
                // 定数のオーバーフロー
                TestData.Failure("FourGplusOneEnd1END  alias FourGEnd1 end"),
                TestData.Success("FourGminusOneEnd1END alias FourGminusOneEnd1 end"),
                // タイプ2: begin, end 指定
                TestData.Failure("FourGEnd2            begin=0x00000000 end=0x0000_0001_0000_0000"),
                TestData.Failure("Reversed             begin=0x10000000 end=0x0fffffff"),
                TestData.Success("FourGminusOneEnd2    begin=0x00000000 end=0xffffffff"),
                TestData.Success("ZeroSized            begin=0x00000000 end=0x00000000"),

                // タイプ3: 領域相対指定 (inner|outer) offset, size
                TestData.Success("BASE_REGION          begin=0x80000000 size=0x10000000"),
                TestData.Success("BeginOk3a            BASE_REGION inner -0x88000000 size=0x08000000"),
                TestData.Failure("BeginOverflow3a      BASE_REGION inner -0x88000001 size=0x08000000"),
                TestData.Success("BeginOk3b            BASE_REGION inner +0x7fffffff size=0x00000000"),
                TestData.Failure("BeginOverflow3b      BASE_REGION inner +0x80000000 size=0x00000000"),
                TestData.Success("BeginOk3c            BASE_REGION outer -0x78000000 size=0x08000000"),
                TestData.Failure("BeginOverflow3c      BASE_REGION outer -0x78000001 size=0x08000000"),
                TestData.Success("BeginOk3d            BASE_REGION outer +0x6fffffff size=0x00000000"),
                TestData.Failure("BeginOverflow3d      BASE_REGION outer +0x70000000 size=0x00000000"),

                TestData.Success("EndOk3a              BASE_REGION inner +0x78000000 size=0x08000000"),
                TestData.Failure("EndOverflow3a        BASE_REGION inner +0x78000001 size=0x08000000"),
                TestData.Success("EndOk3b              BASE_REGION outer +0x68000000 size=0x08000000"),
                TestData.Failure("EndOverflow3b        BASE_REGION outer +0x68000001 size=0x08000000"),

                // タイプ4: 領域相対指定 shift offset
                TestData.Success("BeginOk4a            BASE_REGION shift -0x80000000"),
                TestData.Failure("BeginOverflow4a      BASE_REGION shift -0x80000001"),
                TestData.Success("EndOk4a              BASE_REGION shift +0x70000000"),
                TestData.Failure("EndOverflow4a        BASE_REGION shift +0x70000001"),
            });
        }

        [TestMethod]
        public void BoundaryTest_64()
        {
            ExecuteTest(AddrWidth.Bit64, new[] {
                // タイプ1: begin, size 指定
                TestData.Failure("SixteenEplusOneEnd1  begin=0x00000000_00000002 size=0xffffffff_ffffffff"),
                TestData.Success("SixteenEEnd1         begin=0x00000000_00000001 size=0xffffffff_ffffffff"),
                TestData.Success("SixteenEminusOneEnd1 begin=0x00000000_00000000 size=0xffffffff_ffffffff"),
                // 定数のオーバーフロー
                TestData.Failure("SixteenEEnd1         alias SixteenEEnd1 end"),
                TestData.Success("SixteenEminusOneEnd1 alias SixteenEminusOneEnd1 end"),
                // タイプ2: begin, end 指定
                TestData.Failure("Reversed             begin=0x10000000_00000000 end=0x0fffffff_ffffffff"),
                TestData.Success("SixteenEminusOneEnd2 begin=0x00000000_00000000 size=0xffffffff_ffffffff"),
                // タイプ3: 領域相対指定 (inner|outer) offset, size
                TestData.Success("BASE_REGION          begin=0x80000000_00000000 size=0x10000000_00000000"),

                TestData.Success("BeginOk3a            BASE_REGION inner -0x88000000_00000000 size=0x08000000_00000000"),
                TestData.Failure("BeginOverflow3a      BASE_REGION inner -0x88000000_00000001 size=0x08000000_00000000"),
                TestData.Success("BeginOk3b            BASE_REGION inner +0x7fffffff_ffffffff size=0x00000000_00000000"),
                TestData.Failure("BeginOverflow3b      BASE_REGION inner +0x80000000_00000000 size=0x00000000_00000000"),
                TestData.Success("BeginOk3c            BASE_REGION outer -0x78000000_00000000 size=0x08000000_00000000"),
                TestData.Failure("BeginOverflow3c      BASE_REGION outer -0x78000000_00000001 size=0x08000000_00000000"),
                TestData.Success("BeginOk3d            BASE_REGION outer +0x6fffffff_ffffffff size=0x00000000_00000000"),
                TestData.Failure("BeginOverflow3d      BASE_REGION outer +0x70000000_00000000 size=0x00000000_00000000"),

                TestData.Success("EndOk3a              BASE_REGION inner +0x78000000_00000000 size=0x08000000_00000000"),
                TestData.Failure("EndOverflow3a        BASE_REGION inner +0x78000000_00000001 size=0x08000000_00000000"),
                TestData.Success("EndOk3b              BASE_REGION outer +0x68000000_00000000 size=0x08000000_00000000"),
                TestData.Failure("EndOverflow3b        BASE_REGION outer +0x68000000_00000001 size=0x08000000_00000000"),

                // タイプ4: 領域相対指定 shift offset
                TestData.Success("BeginOk4a            BASE_REGION shift -0x80000000_00000000"),
                TestData.Failure("BeginOverflow4a      BASE_REGION shift -0x80000000_00000001"),
                TestData.Success("EndOk4a              BASE_REGION shift +0x70000000_00000000"),
                TestData.Failure("EndOverflow4a        BASE_REGION shift +0x70000000_00000001"),
            });
        }

        // テストデータのソース行を評価し、テストデータの指定通りに成功/失敗するかどうかを確認する
        private void ExecuteTest(AddrWidth addrWidth, IEnumerable<TestData> testDataList)
        {
            var evaluationContext = new EvaluationContext();
            var parseContext = new ParseContext(addrWidth, new FilePosition("Test", 1));
            foreach (var testData in testDataList)
            {
                try
                {
                    var statement = StatementParser.Parse(testData.Line, parseContext);
                    statement.Evaluate(evaluationContext);
                    if (!testData.SuccessExpected)
                    {
                        Assert.Fail("失敗しなければなりません。: {0}", testData.Line);
                    }
                }
                catch (MakeAddrRegionHeaderException e)
                {
                    if (testData.SuccessExpected)
                    {
                        Assert.Fail("成功しなければなりません。: {0}, {1}", testData.Line, e.Message);
                    }
                }
            }
        }

        private class TestData
        {
            public string Line { get; private set; }
            public bool SuccessExpected { get; private set; }

            private TestData(string line, bool successExpected)
            {
                this.Line = line;
                this.SuccessExpected = successExpected;
            }

            // 例外を送出しないテストデータの作成
            public static TestData Success(string line)
            {
                return new TestData(line, true);
            }

            // 例外を送出するテストデータの作成
            public static TestData Failure(string line)
            {
                return new TestData(line, false);
            }
        }
    }
}
