﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Linq;
using LECore;
using LECore.Manipulator;
using LECore.Save_Load;
using LECore.Structures;
using LECore.Structures.Core;
using LECore.Structures.Core.Command;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace LECoreTest
{
    /// <summary>
    /// チェックされていない項目（TODO）
    ///
    /// * 部品の上書き設定を変更した際の親シーンへのリソース登録処理（関連：SIGLO-70685）
    ///
    /// </summary>
    [TestClass]
    public class PartsTest
    {
        [TestMethod]
        public void PartsTest_PartsPaneTest()
        {
            // 準備
            {
                // 部品ルートの設定
                SceneManipulator mnp = new SceneManipulator();
                mnp.BindTarget(LayoutEditorCore.Scene);
                mnp.RefreshPartsSubScenes(Path.Combine(AppConstants.NwUnitTestDataRoot, "parts"));

                // ファイル読み込み
                string filePath = Path.Combine(AppConstants.NwUnitTestDataRoot, "parts_overwrite.flyt");
                LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan);
            }

            // 部品の切り替えを行う
            {
                IPane partsPane = LayoutEditorCore.Scene.CurrentISubScene.FindPaneByName("Parts_01");

                float beforeChangeW = partsPane.IPartsLayout.PartsSubScene.FindPaneByName("TextBox_00").Size.X;
                Assert.AreEqual(beforeChangeW, 343.00f);

                // 部品を切り替えて正しく切り替わることを確認する。
                LayoutEditorCore.Scene.LoadAndCachePartsSubSceneIfNeeded("Parts2.flyt");

                PartsLayoutManipulator partsPaneMnp = new PartsLayoutManipulator();
                partsPaneMnp.BindTarget(partsPane.IPartsLayout);

                LayoutEditorCore.Scene.CurrentISubScene.BeginMassiveModify();
                partsPaneMnp.ChangePartsPaneLayout("Parts2.flyt");
                LayoutEditorCore.Scene.CurrentISubScene.EndMassiveModify();

                partsPane = LayoutEditorCore.Scene.CurrentISubScene.FindPaneByName("Parts_01");
                Assert.AreEqual(partsPane.IPartsLayout.PartsSubScene.FindPaneByName("TextBox_00").Size.X, 157.00f);

                // Undo して正しく戻ることを確認する。
                var cmdQueue = (LayoutEditorCore.Scene.CurrentISubScene.ISubSceneCmdQueue as SubSceneCmdQueue);
                cmdQueue.Undo();

                partsPane = LayoutEditorCore.Scene.CurrentISubScene.FindPaneByName("Parts_01");
                Assert.AreEqual(partsPane.IPartsLayout.PartsSubScene.FindPaneByName("TextBox_00").Size.X, beforeChangeW);

                // ボディ、parts_overwrite.flyt をアンロード
                SceneManipulator mnp = new SceneManipulator();
                mnp.BindTarget(LayoutEditorCore.Scene);
                mnp.RemoveSubScene(LayoutEditorCore.Scene.CurrentISubScene);
            }

            // 上書き周り
            {
                PartsLayoutManipulator partsLayoutMnp = new PartsLayoutManipulator();

                string filePath = Path.Combine(AppConstants.NwUnitTestDataRoot, "parts_overwrite.flyt");
                LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan);
                var cmdQueue = (LayoutEditorCore.Scene.CurrentISubScene.ISubSceneCmdQueue as SubSceneCmdQueue);

                IPane partsPane = LayoutEditorCore.Scene.CurrentISubScene.FindPaneByName("Parts_01");

                {
                    // 部品レイアウト側でペインの値を変更
                    IPane originalPane = PartsLayoutHelper.GetOriginalPane(partsPane.IPartsLayout, "TextBox_00");
                    PaneManipulator paneMnp = new PaneManipulator();
                    paneMnp.BindTarget(originalPane);

                    float orgSizeX = originalPane.Size.X;
                    float modSizeX = originalPane.Size.X + 100;
                    paneMnp.Size = new FVec3(modSizeX, originalPane.Size.Y, originalPane.Size.Z);

                    // リロードする
                    PartsLayoutManipulator.ReloadPartsPane(partsPane);

                    // ボディ側でリロードが反映されていることを確認
                    IPane partsTextPane = partsPane.IPartsLayout.PartsSubScene.FindPaneByName("TextBox_00");
                    Assert.AreEqual(partsTextPane.Size.X, modSizeX);

                    // ボディ側で上書きされるペインの値を変更
                    // リロードして、値が変わらないことを確認
                    {
                        partsLayoutMnp.BindTarget(partsPane.IPartsLayout);
                        partsLayoutMnp.ModifyOverwriteParamaters("TextBox_00", PaneKind.Textbox, PartsPropertyUsageOptions.UsePaneSize);

                        // 元の値に戻す
                        paneMnp.Size = new FVec3(orgSizeX, originalPane.Size.Y, originalPane.Size.Z);

                        // リロード
                        PartsLayoutManipulator.ReloadPartsPane(partsPane);
                        // リロード時に、インスタンス作り直しにつきとりなおし。。。
                        partsTextPane = partsPane.IPartsLayout.PartsSubScene.FindPaneByName("TextBox_00");

                        // ボディで上書きしているので、値は「変更した値に」維持されている。
                        Assert.AreEqual(partsTextPane.Size.X, modSizeX);

                        // 上書きをリセット
                        // 正しい値にリセットされることを確認
                        {
                            // 正しい値（元の値）にリセットされることを確認
                            partsLayoutMnp.ModifyOverwriteParamaters("TextBox_00", PaneKind.Textbox, PartsPropertyUsageOptions.None);
                            Assert.AreEqual(partsTextPane.Size.X, orgSizeX);

                            // Undo / Redo
                            {
                                cmdQueue.Undo();
                                Assert.AreEqual(partsTextPane.Size.X, modSizeX);
                                cmdQueue.Redo();
                                Assert.AreEqual(partsTextPane.Size.X, orgSizeX);
                            }
                        }

                        // 部品ペインの貼り付け
                        {
                            // 現状を記録
                            IPane[] srcPaneSet = PaneSetDupulicator.ClonePaneSet(
                                new IPane[] { partsPane },
                                PaneSetDupulicator.Option.CopyAll);

                            // Undo して値が変化していることを確認。
                            cmdQueue.Undo();
                            Assert.AreEqual(partsTextPane.Size.X, modSizeX);

                            // 記録してある状態を貼りつける
                            {
                                LECore.LayoutEditorCore.Scene.CurrentISubScene.BeginMassiveModify();
                                {
                                    var copyOption = new PaneParamaterCopyOption();
                                    copyOption.EnableOption(ParamaterKind.All);
                                    paneMnp.BindTarget(partsPane);
                                    paneMnp.Copy(srcPaneSet[0], copyOption);
                                }
                                LECore.LayoutEditorCore.Scene.CurrentISubScene.EndMassiveModify();
                                // 値が記録してある状態にもどっているはず
                                Assert.AreEqual(partsTextPane.Size.X, orgSizeX);
                            }

                            // Undo / Redo
                            {
                                cmdQueue.Undo();
                                Assert.AreEqual(partsTextPane.Size.X, modSizeX);
                                cmdQueue.Redo();
                                Assert.AreEqual(partsTextPane.Size.X, orgSizeX);
                            }
                        }
                    }
                }

                // ボディ、parts_overwrite.flyt をアンロード
                SceneManipulator sceneMnp = new SceneManipulator();
                sceneMnp.BindTarget(LayoutEditorCore.Scene);
                sceneMnp.RemoveSubScene(LayoutEditorCore.Scene.CurrentISubScene);
            }
        }
    }
}
