﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Linq;
using LECore;
using LECore.Manipulator;
using LECore.Save_Load;
using LECore.Structures;
using LECore.Structures.Core;
using LECore.Structures.Core.Command;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace LECoreTest
{
    [TestClass]
    public class CaptureTest
    {
        [TestMethod]
        public void CaptureTest_CapturePaneTest()
        {
            // 準備
            {
                // 部品ルートの設定
                SceneManipulator mnp = new SceneManipulator();
                mnp.BindTarget(LayoutEditorCore.Scene);
                mnp.RefreshPartsSubScenes(Path.Combine(AppConstants.NwUnitTestDataRoot, "parts"));
            }

            {
                // ファイル読み込み
                string filePath = Path.Combine(AppConstants.NwUnitTestDataRoot, "Parts/Parts.flyt");
                LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan);

                // キャプチャテクスチャの存在を確認する。
                var texMgr = LayoutEditorCore.Scene.CurrentISubScene.ITextureMgr;
                Assert.IsNotNull(texMgr.FindITextureImageByName("P_fb_capture_00"));
                Assert.IsNotNull(texMgr.FindITextureImageByName("C_capture_00"));

                // キャプチャテクスチャの供給元ペインを消す
                SubSceneManipulator mnp = new SubSceneManipulator();
                mnp.BindTarget(LayoutEditorCore.Scene.CurrentISubScene);
                mnp.SelectPanesByPaneName("C_capture_00");
                mnp.SelectPanesByPaneName("P_fb_capture_00");
                mnp.DeleteSelectedPanes();

                // 利用するペインが一つもないので、テクスチャが消えることを確認する
                Assert.IsNull(texMgr.FindITextureImageByName("P_fb_capture_00"));
                // 利用しているペイン(P_pict_00)がある場合、テクスチャが消えないことを確認する
                Assert.IsNotNull(texMgr.FindITextureImageByName("C_capture_00"));

                // C_capture_00 を 利用しているペインを消す。テクスチャーが消える。
                mnp.SelectPanesByPaneName("P_pict_00");
                mnp.DeleteSelectedPanes();
                Assert.IsNull(texMgr.FindITextureImageByName("C_capture_00"));

                // Undo で状態が戻ることを確認する
                var cmdQueue = (LayoutEditorCore.Scene.CurrentISubScene.ISubSceneCmdQueue as SubSceneCmdQueue);
                cmdQueue.Undo();
                Assert.IsNull(texMgr.FindITextureImageByName("P_fb_capture_00"));
                Assert.IsNotNull(texMgr.FindITextureImageByName("C_capture_00"));
                cmdQueue.Undo();
                Assert.IsNotNull(texMgr.FindITextureImageByName("P_fb_capture_00"));
                Assert.IsNotNull(texMgr.FindITextureImageByName("C_capture_00"));

                // TODO:テクスチャを消し、ペインが消えることを確認する（⇒この仕様は不要ではないか？削除できなくてよさそう。）
            }


        }

        [TestMethod]
        public void CaptureTest_PartsTest()
        {
            // 準備
            {
                // 部品ルートの設定
                SceneManipulator mnp = new SceneManipulator();
                mnp.BindTarget(LayoutEditorCore.Scene);
                mnp.RefreshPartsSubScenes(Path.Combine(AppConstants.NwUnitTestDataRoot, "parts"));
            }

            // キャプチャ周りをチェック
            {
                PartsLayoutManipulator partsLayoutMnp = new PartsLayoutManipulator();

                string filePath = Path.Combine(AppConstants.NwUnitTestDataRoot, "parts_overwrite.flyt");
                LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan);

                // 部品レイアウトのテクスチャーが修飾された形で初期化されていることを確認します。
                IPane partsPane = LayoutEditorCore.Scene.CurrentISubScene.FindPaneByName("Parts_01");
                var partsTexMgr = partsPane.IPartsLayout.PartsSubScene.ITextureMgr;
                Assert.IsNotNull(partsTexMgr.FindITextureImageByName("Parts_01%P_fb_capture_00"));
                Assert.IsNotNull(partsTexMgr.FindITextureImageByName("Parts_01%C_capture_00"));

                // 部品ペインのペイン名を変更して、キャプチャテクスチャ名が追従することを確認します。
                PaneManipulator paneMnp = new PaneManipulator();
                paneMnp.BindTarget(partsPane);
                paneMnp.PaneName = "Parts_01_mod";
                Assert.IsNotNull(partsTexMgr.FindITextureImageByName("Parts_01_mod%P_fb_capture_00"));
                Assert.IsNotNull(partsTexMgr.FindITextureImageByName("Parts_01_mod%C_capture_00"));
                // Undo で戻る
                var cmdQueue = (LayoutEditorCore.Scene.CurrentISubScene.ISubSceneCmdQueue as SubSceneCmdQueue);
                cmdQueue.Undo();
                Assert.IsNotNull(partsTexMgr.FindITextureImageByName("Parts_01%P_fb_capture_00"));
                Assert.IsNotNull(partsTexMgr.FindITextureImageByName("Parts_01%C_capture_00"));

                // 上書きされたキャプチャペインの操作が問題なく Undo できるか確認します。
                CaptureManipulator captureMnp = new CaptureManipulator();
                ICapture capturePane = partsPane.IPartsLayout.PartsSubScene.FindPaneByName("C_capture_00").ICapture;
                captureMnp.BindTarget(capturePane);
                captureMnp.FrameBufferCaptureEnabled = false;
                cmdQueue.Undo();
                Assert.AreEqual(capturePane.FrameBufferCaptureEnabled, true);

                // ボディでキャプチャテクスチャーを使っているペインを上書きする
                partsLayoutMnp.BindTarget(partsPane.IPartsLayout);
                partsLayoutMnp.ModifyOverwriteParamaters("P_fb_capture_00", PaneKind.Textbox, PartsPropertyUsageOptions.UsePaneExParamater);
                partsLayoutMnp.ModifyOverwriteParamaters("P_pict_00", PaneKind.Textbox, PartsPropertyUsageOptions.UsePaneExParamater);

                // ボディ側にキャプチャテクスチャーは登録されない(通常のテクスチャーは登録される)
                var bodyTexMgr = LayoutEditorCore.Scene.CurrentISubScene.ITextureMgr;
                Assert.IsNull(bodyTexMgr.FindITextureImageByName("Parts_01%P_fb_capture_00"));
                Assert.IsNull(bodyTexMgr.FindITextureImageByName("Parts_01%C_capture_00"));

                // ボディ、parts_overwrite.flyt をアンロード
                SceneManipulator mnp = new SceneManipulator();
                mnp.BindTarget(LayoutEditorCore.Scene);
                mnp.RemoveSubScene(LayoutEditorCore.Scene.CurrentISubScene);
            }
        }
    }
}
