﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace Nintendo.Hid
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;

    public class DebugPad : BaseModule
    {
        private const byte StateCountMax = 16;

        private Buttons buttons;
        private AnalogStickL analogStickL;
        private AnalogStickR analogStickR;

        internal DebugPad(byte id) : base(id)
        {
            this.buttons = new Buttons(id);
            this.analogStickL = new AnalogStickL(id);
            this.analogStickR = new AnalogStickR(id);

            this.AddSubModule(this.buttons);
            this.AddSubModule(this.analogStickL);
            this.AddSubModule(this.analogStickR);
        }

        private enum SubModuleType : byte
        {
            Buttons = 0x11,
            AnalogStickL = 0x12,
            AnalogStickR = 0x13,
        }

        internal override byte ModuleTypeValue
        {
            get { return (byte)ModuleType.DebugPad; }
        }

        public void PressButtons(params DebugPadButton[] args)
        {
            this.buttons.Press(args);
        }

        public void ReleaseButtons(params DebugPadButton[] args)
        {
            this.buttons.Release(args);
        }

        public void ClearButtons()
        {
            this.buttons.Clear();
        }

        public void MonitorButtons()
        {
            this.buttons.IsMonitored = true;
        }

        public void UnmonitorButtons()
        {
            this.buttons.IsMonitored = false;
        }

        public DebugPadButtonsState[] GetButtonsStates()
        {
            return this.buttons.GetStates();
        }

        public void SetAnalogStickL(short x, short y)
        {
            this.analogStickL.Set(x, y);
        }

        public void ClearAnalogStickL()
        {
            this.analogStickL.Clear();
        }

        public void MonitorAnalogStickL()
        {
            this.analogStickL.IsMonitored = true;
        }

        public void UnmonitorAnalogStickL()
        {
            this.analogStickL.IsMonitored = false;
        }

        public DebugPadAnalogStickState[] GetAnalogStickLStates()
        {
            return this.analogStickL.GetStates();
        }

        public void SetAnalogStickR(short x, short y)
        {
            this.analogStickR.Set(x, y);
        }

        public void ClearAnalogStickR()
        {
            this.analogStickR.Clear();
        }

        public void MonitorAnalogStickR()
        {
            this.analogStickR.IsMonitored = true;
        }

        public void UnmonitorAnalogStickR()
        {
            this.analogStickR.IsMonitored = false;
        }

        public DebugPadAnalogStickState[] GetAnalogStickRStates()
        {
            return this.analogStickR.GetStates();
        }

        private class Buttons : BaseSubModule
        {
            private ushort currAutoInputState = 0;
            private ushort nextAutoInputState = 0;
            private List<DebugPadButtonsState> states = new List<DebugPadButtonsState>();

            public Buttons(byte id) : base(id)
            {
            }

            internal override byte ModuleTypeValue
            {
                get { return (byte)ModuleType.DebugPad; }
            }

            internal override byte SubModuleTypeValue
            {
                get { return (byte)SubModuleType.Buttons; }
            }

            protected override byte StateCountMax
            {
                get { return DebugPad.StateCountMax; }
            }

            protected override bool IsAutoInputChanged
            {
                get { return this.currAutoInputState != this.nextAutoInputState; }
            }

            protected override bool IsAutoInputSet
            {
                get { return this.nextAutoInputState != 0; }
            }

            internal override void Reset()
            {
                this.currAutoInputState = 0;
                this.nextAutoInputState = 0;
                this.states.Clear();
            }

            internal override void Update()
            {
                this.currAutoInputState = this.nextAutoInputState;
                this.states.Clear();
            }

            internal override void AcceptInputState(NetworkBinaryReader reader)
            {
                reader.ReadBytes(2);
                var state = reader.ReadUInt16();
                this.states.Add(new DebugPadButtonsState(state));
            }

            internal void Press(params DebugPadButton[] args)
            {
                foreach (var button in args)
                {
                    this.nextAutoInputState |= (ushort)button;
                }
            }

            internal void Release(params DebugPadButton[] args)
            {
                foreach (var button in args)
                {
                    this.nextAutoInputState &= (ushort)(~button);
                }
            }

            internal void Clear()
            {
                this.nextAutoInputState = 0;
            }

            internal DebugPadButtonsState[] GetStates()
            {
                return this.states.ToArray();
            }

            protected override byte[] GenerateModuleState()
            {
                using (var stream = new MemoryStream())
                using (var writer = new NetworkBinaryWriter(stream))
                {
                    writer.Write((ushort)0);
                    writer.Write((ushort)this.nextAutoInputState);
                    return stream.ToArray();
                }
            }
        }

        private abstract class AnalogStick : BaseSubModule
        {
            private DebugPadAnalogStickState currAutoInputState = null;
            private DebugPadAnalogStickState nextAutoInputState = null;
            private List<DebugPadAnalogStickState> states = new List<DebugPadAnalogStickState>();

            public AnalogStick(byte id) : base(id)
            {
            }

            internal override byte ModuleTypeValue
            {
                get { return (byte)ModuleType.DebugPad; }
            }

            protected override byte StateCountMax
            {
                get { return DebugPad.StateCountMax; }
            }

            protected override bool IsAutoInputChanged
            {
                get
                {
                    if (this.currAutoInputState != this.nextAutoInputState)
                    {
                        return true;
                    }

                    if (this.currAutoInputState != null && !this.currAutoInputState.Equals(this.nextAutoInputState))
                    {
                        return true;
                    }

                    return false;
                }
            }

            protected override bool IsAutoInputSet
            {
                get { return this.nextAutoInputState != null; }
            }

            internal override void Reset()
            {
                this.currAutoInputState = null;
                this.nextAutoInputState = null;
                this.states.Clear();
            }

            internal override void Update()
            {
                this.currAutoInputState = this.nextAutoInputState;
                this.states.Clear();
            }

            internal override void AcceptInputState(NetworkBinaryReader reader)
            {
                var x = reader.ReadUInt16() + short.MinValue;
                var y = reader.ReadUInt16() + short.MinValue;
                this.states.Add(new DebugPadAnalogStickState((short)x, (short)y));
            }

            internal void Set(short x, short y)
            {
                this.nextAutoInputState = new DebugPadAnalogStickState(x, y);
            }

            internal void Clear()
            {
                this.nextAutoInputState = null;
            }

            internal DebugPadAnalogStickState[] GetStates()
            {
                return this.states.ToArray();
            }

            protected override byte[] GenerateModuleState()
            {
                using (var stream = new MemoryStream())
                using (var writer = new NetworkBinaryWriter(stream))
                {
                    writer.Write((ushort)(this.nextAutoInputState.X - short.MinValue));
                    writer.Write((ushort)(this.nextAutoInputState.Y - short.MinValue));
                    return stream.ToArray();
                }
            }
        }

        private class AnalogStickL : AnalogStick
        {
            public AnalogStickL(byte id) : base(id)
            {
            }

            internal override byte SubModuleTypeValue
            {
                get { return (byte)SubModuleType.AnalogStickL; }
            }
        }

        private class AnalogStickR : AnalogStick
        {
            public AnalogStickR(byte id) : base(id)
            {
            }

            internal override byte SubModuleTypeValue
            {
                get { return (byte)SubModuleType.AnalogStickR; }
            }
        }
    }
}
