﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using GitExternalRepository;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Microsoft.Win32;
using System.Collections.Generic;
using System.Collections;

namespace GitExternalRepositoryTest
{
    [TestClass]
    public class UnitTestUtils
    {
        [TestMethod]
        public void TestGetRelativePath()
        {
            var basePath = @"TestGetRelativePath\directory1\";
            var targetPath = @"TestGetRelativePath\directory2";
            var targetPathLarge = @"TESTGETRELATIVEPATH\directory2";    // 大文字小文字の違いを無視した比較が行われることの確認

            Directory.CreateDirectory(basePath);
            Directory.CreateDirectory(targetPath);

            // 相対パス同士
            {
                var relativePath = PathUtility.GetRelativePath(basePath, targetPath);

                Assert.AreEqual(@"../directory2", relativePath);
            }
            {
                var relativePath = PathUtility.GetRelativePath(basePath, targetPathLarge);
                Assert.AreEqual(@"../directory2", relativePath);
            }

            // 絶対パス同士
            {
                var relativePath = PathUtility.GetRelativePath(Path.GetFullPath(basePath), Path.GetFullPath(targetPath));

                Assert.AreEqual(@"../directory2", relativePath);
            }
            {
                var relativePath = PathUtility.GetRelativePath(Path.GetFullPath(basePath), Path.GetFullPath(targetPathLarge));

                Assert.AreEqual(@"../directory2", relativePath);
            }

            // 相対 + 絶対
            {
                var relativePath = PathUtility.GetRelativePath(basePath, Path.GetFullPath(targetPath));

                Assert.AreEqual(@"../directory2", relativePath);
            }
            {
                var relativePath = PathUtility.GetRelativePath(basePath, Path.GetFullPath(targetPathLarge));

                Assert.AreEqual(@"../directory2", relativePath);
            }

            // 絶対 + 相対
            {
                var relativePath = PathUtility.GetRelativePath(Path.GetFullPath(basePath), targetPath);

                Assert.AreEqual(@"../directory2", relativePath);
            }
            {
                var relativePathLarge = PathUtility.GetRelativePath(Path.GetFullPath(basePath), targetPathLarge);

                Assert.AreEqual(@"../directory2", relativePathLarge);
            }
        }

        [TestMethod]
        public void TestPathsAreSame()
        {
            Assert.IsTrue(PathUtility.AreSame(@"c:\a", @"c:\a"));

            // 大文字小文字の異なるパス
            Assert.IsTrue(PathUtility.AreSame(@"c:\a", @"C:\A"));

            // 異なるパス
            Assert.IsFalse(PathUtility.AreSame(@"c:\a", @"d:\A"));
        }

        [TestMethod]
        public void TestPathContainsPath()
        {
            Assert.IsTrue(PathUtility.ContainsPath(@"c:\a\b", @"c:\a"));

            // 大文字小文字の異なるパス
            Assert.IsTrue(PathUtility.ContainsPath(@"c:\a\b", @"C:\A"));

            // 異なるパス
            Assert.IsFalse(PathUtility.ContainsPath(@"c:\a\b", @"d:\A"));
        }

        private class SuparClass
        {
        }

        private class SubClass1 : SuparClass
        {
        }

        private class SubClass2 : SuparClass
        {
        }

        /// <summary>
        /// GetInheritedClasses メソッドを試します。
        /// </summary>
        [TestMethod]
        public void TestGetInheritedClasses()
        {
            var classes = Utils.GetInheritedClasses<SuparClass>();
            var classList = Enumerable.ToList(classes);

            var expected = new List<Type>()
            {
                typeof(SubClass1),
                typeof(SubClass2),
            };

            CollectionAssert.AreEqual(expected, classList);
        }
    }
}
