﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Diagnostics;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.FsAccessLogAnalysis;

namespace FsAccessLogCheckerTest
{
    [TestClass]
    // library に組み込まれたアクセスログとツールの対応チェックをテストします
    public class FsAccessLogSupportTest
    {
        public TestContext TestContext { get; set; }

        public static string TestTempDirectory { get; set; }

        private static string GetMountFunctionListFilePath()
        {
            return Path.Combine(TestTempDirectory, MountFunctionListFileName);
        }
        private static string GetSystemMountFunctionListFilePath()
        {
            return Path.Combine(TestTempDirectory, SystemMountFunctionListFileName);
        }
        private static string GetAccessFunctionListFilePath()
        {
            return Path.Combine(TestTempDirectory, AccessFunctionListFileName);
        }
        private static string GetSystemAccessFunctionListFilePath()
        {
            return Path.Combine(TestTempDirectory, SystemAccessFunctionListFileName);
        }
        private static string GetAllYamlAccessLogFilePath()
        {
            return Path.Combine(TestTempDirectory, AllYamlAccessLogFileName);
        }

        private static int ExecuteProgram(string filePath, string args, out string message, out string errorMessage)
        {
            Process process = new Process();
            process.StartInfo.FileName = filePath;
            process.StartInfo.Arguments = args;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardInput = true;
            process.Start();

            var messageTask = process.StandardOutput.ReadToEndAsync();
            var errorMessageTask = process.StandardError.ReadToEndAsync();
            process.WaitForExit();
            message = messageTask.Result;
            errorMessage = errorMessageTask.Result;

            return process.ExitCode;
        }

        private static int ExecutePythonProgram(TestContext context, string filePath, string args, out string message, out string errorMessage)
        {
            TestUtility.TestPath testPath = new TestUtility.TestPath(context);
            string pythonPath = testPath.GetSigloRoot() + @"\Externals\Binaries\Python36\python.exe";
            return ExecuteProgram(pythonPath, filePath + " " + args, out message, out errorMessage);
        }

        [ClassInitialize]
        public static void TestClassInitialize(TestContext context)
        {
            // アクセスログの抽出ツールを使って情報を取得する
            if (!IsRunPythonTool)
            {
                if (File.Exists(GetMountFunctionListFilePath()) && File.Exists(GetSystemMountFunctionListFilePath()))
                {
                    return;
                }
            }
            TestUtility.TestPath testPath = new TestUtility.TestPath(context);
            string toolDir = testPath.GetSigloRoot() + @"\Tests\Tools\Sources\Tests\FsAccessLogCheckerTest\FsAccessLogSupportTest";
            string[] targetDirs = {
                testPath.GetSigloRoot() + @"\Programs\Chris\Sources\Libraries\fs",
                testPath.GetSigloRoot() + @"\Programs\Eris\Sources\Libraries\fs",
                testPath.GetSigloRoot() + @"\Programs\Eris\Sources\Libraries\bcat",
                testPath.GetSigloRoot() + @"\Programs\Eris\Sources\Libraries\nfp",
                testPath.GetSigloRoot() + @"\Programs\Eris\Sources\Libraries\pdm",
                testPath.GetSigloRoot() + @"\Programs\Iris\Sources\Libraries\album",
            };

            TestTempDirectory = TestUtility.TestPath.GetNewTemporaryDirectoryPath("SIGLO_FSACCESS_TEST");
            TestUtility.TestPath.DeleteDirectoryIfExisted(TestTempDirectory);

            string toolOptions = "-o " + TestTempDirectory;
            foreach (var path in targetDirs)
            {
                toolOptions += " " + path;
            }
            ExtractResult = ExecutePythonProgram(context,
                Path.Combine(toolDir, "extract_accesslog.py"), toolOptions, out ExtractMessage, out ExtractErrorMessage);
            // TestMethod を実行するために、ここで Assert せず、TestExtractToolCheck TestMethod でチェックします。
            //Assert.AreEqual(0, ExtractResult, ExtractErrorMessage);
        }

        [ClassCleanup]
        public static void TestClassCleanup()
        {
            TestUtility.TestPath.DeleteDirectoryIfExisted(TestTempDirectory);
        }

        public static List<string> GetFsAccessLogCheckerSupportMountFunctionList()
        {
            List<string> mountFunctionList = new List<string>();
            foreach (var item in FsFunction.Names.MountAssociation)
            {
                mountFunctionList.Add(item.Key);
            }
            return mountFunctionList;
        }

        public static List<string> GetFsAccessLogCheckerSupportSystemMountFunctionList()
        {
            List<string> mountFunctionList = new List<string>();
            foreach (var item in FsFunction.Names.MountAssociationForSystem)
            {
                mountFunctionList.Add(item.Key);
            }
            return mountFunctionList;
        }

        public static List<string> GetFsAccessLogCheckerSupportAccessFunctionList()
        {
            List<string> accessFunctionList = new List<string>();
            foreach (var name in FsFunction.Names.WriteAccess)
            {
                accessFunctionList.Add(name);
            }
            foreach (var name in FsFunction.Names.ReadAccess)
            {
                accessFunctionList.Add(name);
            }
            foreach (var item in FsFunction.Names.WriteGroupAccess)
            {
                accessFunctionList.Add(item.Key);
            }
            foreach (var item in FsFunction.Names.ReadGroupAccess)
            {
                accessFunctionList.Add(item.Key);
            }
            accessFunctionList.AddRange(FsFunction.Names.HandleOpenAccess);
            accessFunctionList.AddRange(FsFunction.Names.HandleCloseAccess);
            foreach (var name in FsFunction.Names.UnmountAccess)
            {
                accessFunctionList.Add(name);
            }
            return accessFunctionList;
        }
        public static List<string> GetFsAccessLogCheckerSupportSystemAccessFunctionList()
        {
            List<string> accessFunctionList = new List<string>();
            accessFunctionList.Add("QueryApplicationPlayStatisticsForSystem");
            return accessFunctionList;
        }

        private static List<string> FileToList(string path)
        {
            using (StreamReader stream = new StreamReader(path))
            {
                List<string> list = new List<string>();
                while (stream.Peek() > 0)
                {
                    string line = stream.ReadLine();
                    line = line.Trim();
                    list.Add(line);
                }
                return list;
            }
        }

        private static string MakeAssertMessage(string functionType, List<string> listFsAccessLogChecker, List<string> listFsAccessLog)
        {
            StringBuilder builder = new StringBuilder();
            if (listFsAccessLogChecker.Any())
            {
                builder.AppendLine(string.Format("FsAccessLogChecker: not found {0} functions", functionType));
                foreach (var name in listFsAccessLogChecker)
                {
                    builder.AppendLine(name);
                }
            }

            if (listFsAccessLog.Any())
            {
                builder.AppendLine("----------");
                builder.AppendLine(string.Format("FsAccessLog: extracted {0} functions", functionType));
                foreach (var name in listFsAccessLog)
                {
                    builder.AppendLine(name);
                }
            }
            return builder.ToString();
        }

        private static int ExtractResult;
        private static string ExtractMessage;
        private static string ExtractErrorMessage;
        private static string MountFunctionListFileName = "mount.txt";
        private static string SystemMountFunctionListFileName = "system_mount.txt";
        private static string AccessFunctionListFileName = "access.txt";
        private static string SystemAccessFunctionListFileName = "system_access.txt";
        private static string AllYamlAccessLogFileName = "all_yaml_accesslog.txt";

        [TestInitialize]
        public void TestInitialize()
        {
            FsAccessLogCheckerTestUtility.Initialize();
        }

        [TestMethod]
        public void TestExtractToolCheck()
        {
            Assert.AreEqual(0, ExtractResult, ExtractErrorMessage);
        }

        [TestMethod]
        public void TestMountFunction()
        {
            var listFsAccessLog = FileToList(GetMountFunctionListFilePath());
            var listFsAccessLogChecker = GetFsAccessLogCheckerSupportMountFunctionList();
            // Mount はないはず
            listFsAccessLogChecker.Remove("Mount");
            // マウント関数だが、マウント名が内部で付与されるものは、access グループに登録されている。
            listFsAccessLogChecker.Remove("MountDeliveryCacheStorage");

            // 共通するものを削除する
            var remainedFsAccessLog = listFsAccessLog.Except<string>(listFsAccessLogChecker).ToList();
            var remainedFsAccessLogChecker = listFsAccessLogChecker.Except<string>(listFsAccessLog).ToList();

            // どちらのリストも空になる
            Assert.IsTrue(!remainedFsAccessLogChecker.Any() && !remainedFsAccessLog.Any(), MakeAssertMessage("mount", remainedFsAccessLogChecker, remainedFsAccessLog));
        }

        [TestMethod]
        public void TestSystemMountFunction()
        {
            var listFsAccessLog = FileToList(GetSystemMountFunctionListFilePath());
            var listFsAccessLogChecker = GetFsAccessLogCheckerSupportSystemMountFunctionList();
            var listFsAccessLogCheckerAppendList = new List<string>()
            {
                "MountCacheStorage",
            };
            listFsAccessLogChecker.AddRange(listFsAccessLogCheckerAppendList);

            // 共通するものを削除する
            var remainedFsAccessLog = listFsAccessLog.Except<string>(listFsAccessLogChecker).ToList();
            var remainedFsAccessLogChecker = listFsAccessLogChecker.Except<string>(listFsAccessLog).ToList();

            // どちらのリストも空になる
            Assert.IsTrue(!remainedFsAccessLogChecker.Any() && !remainedFsAccessLog.Any(), MakeAssertMessage("system mount", remainedFsAccessLogChecker, remainedFsAccessLog));
        }

        [TestMethod]
        public void TestAccessFunction()
        {
            var listFsAccessLog = FileToList(GetAccessFunctionListFilePath());
            var listFsAccessLogChecker = GetFsAccessLogCheckerSupportAccessFunctionList();
            var listFsAccessLogCheckerForSystem = GetFsAccessLogCheckerSupportSystemAccessFunctionList();

            // FsAccessLogChecker では未定義（未使用）な関数をテスト用に追加する
            var listFsAccessLogCheckerAppendList = new List<string>()
            {
                "EnsureSaveData",
                "GetEntryType",
                "QueryMountAddOnContentCacheSize",
                "QueryMountRomCacheSize",
                "ReadDirectory",
                "SetAllocator",
                "SetSaveDataRootPath",
                "DeliveryCacheDirectory::~DeliveryCacheDirectory",
                "DeliveryCacheFile::~DeliveryCacheFile",
                "nfp::CreateApplicationArea",
                "nfp::Flush",
                "nfp::Mount",
                "nfp::Restore",
                "GetSaveDataSize",
                "CreateCacheStorage",
                "DeleteCacheStorage",
                "GetCacheStorageSize",
                "OpenCacheStorageList",
                "ReadCacheStorageList",
                "CloseCacheStorageList",
                "GetPriorityOnCurrentThread",
                "EnableGlobalFileDataCache",
                "DisableGlobalFileDataCache",
                "DisableIndividualFileDataCache",
            };
            listFsAccessLogChecker.AddRange(listFsAccessLogCheckerAppendList);
            listFsAccessLogChecker = listFsAccessLogChecker.Except(listFsAccessLogCheckerForSystem).ToList();

            // マウント関数だが、マウント名が内部で付与されるものは、access グループに登録されている。
            listFsAccessLogChecker.Add("MountDeliveryCacheStorage");

            // ソースからのリストアップでは commitApiName になるが実際のログでは Commit と CommitSaveData になる
            listFsAccessLog.Add("Commit");
            listFsAccessLog.Add("CommitSaveData");
            listFsAccessLogChecker.Add("commitApiName");

            // 共通するものを削除する
            var remainedFsAccessLog = listFsAccessLog.Except<string>(listFsAccessLogChecker).ToList();
            var remainedFsAccessLogChecker = listFsAccessLogChecker.Except<string>(listFsAccessLog).ToList();

            // どちらのリストも空になる
            Assert.IsTrue(!remainedFsAccessLogChecker.Any() && !remainedFsAccessLog.Any(), MakeAssertMessage("accesslog", remainedFsAccessLogChecker, remainedFsAccessLog));
        }

        [TestMethod]
        public void TestSystemAccessFunction()
        {
            var listFsAccessLog = FileToList(GetSystemAccessFunctionListFilePath());
            var listFsAccessLogChecker = GetFsAccessLogCheckerSupportSystemAccessFunctionList();
            var listFsAccessLogCheckerForAll = GetFsAccessLogCheckerSupportAccessFunctionList();
            listFsAccessLogChecker = listFsAccessLogChecker.Intersect(listFsAccessLogCheckerForAll).ToList();

            // FsAccessLogChecker では未定義（未使用）な関数をテスト用に追加する
            var listFsAccessLogCheckerAppendList = new List<string>()
            {
                "QueryMountRomOnFileCacheSize",
                "QueryMountSystemDataCacheSize",
                "DeleteSaveData",
                "DeleteSystemSaveData",
                "CreateSaveData",
                "CreateBcatSaveData",
                "CreateDeviceSaveData",
                "CreateTemporaryStorage",
                "CreateCacheStorage",
                "CreateSystemSaveData",
                "CreateSystemBcatSaveData",
                "ExtendSaveData",
                "QuerySaveDataTotalSize",
                "QuerySaveDataInternalStorageTotalSize",
                "GetSaveDataOwnerId",
                "GetSaveDataFlags",
                "SetSaveDataFlags",
                "GetSaveDataTimeStamp",
                "SetSaveDataTimeStamp",
                "GetSaveDataAvailableSize",
                "GetSaveDataJournalSize",
                "OpenSaveDataIterator",
                "SaveDataIterator::ReadSaveDataInfo",
                "GetPriorityRawOnCurrentThread",
                "FindSaveDataWithFilter",
                "GetSaveDataCommitId",
                "SetSaveDataCommitId",
            };
            listFsAccessLogChecker.AddRange(listFsAccessLogCheckerAppendList);

            // 共通するものを削除する
            var remainedFsAccessLog = listFsAccessLog.Except<string>(listFsAccessLogChecker).ToList();
            var remainedFsAccessLogChecker = listFsAccessLogChecker.Except<string>(listFsAccessLog).ToList();

            // どちらのリストも空になる
            Assert.IsTrue(!remainedFsAccessLogChecker.Any() && !remainedFsAccessLog.Any(), MakeAssertMessage("system accesslog", remainedFsAccessLogChecker, remainedFsAccessLog));
        }

        [TestMethod]
        public void TestAllPossibleElements()
        {
            using (FileStream stream = new FileStream(GetAllYamlAccessLogFilePath(), FileMode.Open))
            {
                TestListener listener = new TestListener();
                FsAccessLogParser parser = new FsAccessLogParser(stream);
                parser.Listener = listener;
                FsAccessLogAnalyzer analyzer = parser.Parse()[0];
                foreach (var log in listener.LogList)
                {
                    Assert.IsNull(log.ParserError, log.ParserError);
                    Assert.IsNull(log.AnalyzerError, log.AnalyzerError);
                    Assert.IsNull(log.OutputStateError, log.OutputStateError);
                }
            }
        }

        [TestMethod]
        public void CheckTestConfig()
        {
            Assert.IsTrue(IsRunPythonTool);
        }
        private static bool IsRunPythonTool = true;
    }
}
