﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.FsAccessLogAnalysis;

namespace FsAccessLogCheckerTest
{
    // FsGuideline.Check のテストです
    // FsGuideline.Check を直接呼び出すので、マウントターゲットは気にする必要なく
    // LogList すべてを Unknown として検証します
    [TestClass]
    public class FsLogAnalysisLibraryGuidelineTest
    {
        public TestContext TestContext { get; set; }

        public void AddLog(FsAccessLogList list, long start, long end)
        {
            var log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            list.Add(log);
        }

        [TestInitialize]
        public void TestInitialize()
        {
            FsAccessLogCheckerTestUtility.Initialize();
        }

        [TestMethod]
        public void TestAccumulatedLog()
        {
            FsAccessLogList list = new FsAccessLogList();
            AddLog(list, 0, 0);

            var log = new FsAccessLog();
            log.Start = FsGuideline.CheckMillisecondsStep;
            log.End = FsGuideline.CheckMillisecondsStep;
            Assert.IsTrue(FsGuideline.IsAccumulatedLog(list, log));

            // Start ～ Start の期間で取る
            log.Start = 1;
            log.End = FsGuideline.CheckMillisecondsStep;
            Assert.IsFalse(FsGuideline.IsAccumulatedLog(list, log));
        }

        [TestMethod]
        public void TestMountCountFailure()
        {
            TestStream testStream = new TestStream();
            for (int i = 0; i < 601; ++i)
            {
                int period = i * 2;
                testStream.MountRom(period, period + 1, "test");
                testStream.Unmount(period + 1, period + 2, "test");
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.AreEqual(FsGuideline.CheckResult.ErrorType.MountCount, result.ErrorFlags);
        }

        [TestMethod]
        public void TestMountCountFailureAssociationMount()
        {
            TestStream testStream = new TestStream();
            for (int i = 0; i < 600; ++i)
            {
                {
                    int period = i * 2;
                    testStream.MountSaveData(period, period + 1, "test", 1, 1);
                    testStream.Unmount(period + 1, period + 2, "test");
                }
                {
                    int period = i * 2;
                    testStream.MountSaveDataReadOnly(period, period + 1, "test", 1, 1);
                    testStream.Unmount(period + 1, period + 2, "test");
                }
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.AreEqual(FsGuideline.CheckResult.ErrorType.MountCount, result.ErrorFlags);
            Assert.AreEqual("MountSaveData/MountSaveDataReadOnly(UserId: 00000000000000000000000000000001, ApplicationId: 0x1)", result.FailMountCountPerSecondsList.ElementAt(0).Key);
        }

        [TestMethod]
        public void TestMountCountSuccess()
        {
            TestStream testStream = new TestStream();
            for (int i = 0; i < 600; ++i)
            {
                int period = i * 2;
                testStream.MountRom(period, period + 1, "test");
                testStream.Unmount(period + 1, period + 2, "test");
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);
        }

        [TestMethod]
        public void TestMountCountResultFailure()
        {
            TestStream testStream = new TestStream();
            testStream.MountRom(0, 0, "test", 0x1);
            for (int i = 0; i < 600; ++i)
            {
                int period = i * 2;
                testStream.MountRom(period, period + 1, "test");
                testStream.Unmount(period + 1, period + 2, "test");
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);
        }

        [TestMethod]
        public void TestMountCountDifferentUserIdSuccess()
        {
            TestStream testStream = new TestStream();
            for (int i = 0; i < 600; ++i)
            {
                int period = i * 2;
                testStream.MountSaveData(period, period + 1, "test", 1);
                testStream.Unmount(period + 1, period + 2, "test");
            }
            // 閾値を超える Mount 回数だが、UserId が異なるため成功とみなされる
            testStream.MountSaveData(2000, 2001, "test", 2);
            testStream.Unmount(2001, 2002, "test");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);
            Assert.AreEqual(2, result.PassMountCountPerSecondsList.Count);
        }

        [TestMethod]
        public void TestWriteSizeFailure()
        {
            TestStream testStream = new TestStream();
            testStream.WriteFile(0, 1, 0x0000000200734060, 0, 167772164);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.AreEqual(FsGuideline.CheckResult.ErrorType.WriteSize, result.ErrorFlags);
        }

        [TestMethod]
        public void TestWriteSizeSuccess()
        {
            TestStream testStream = new TestStream();
            testStream.WriteFile(0, 1, 0x0000000200734060, 0, 167772159);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);
        }

        [TestMethod]
        public void TestWriteSizeResultFailure()
        {
            TestStream testStream = new TestStream();
            testStream.WriteFile(0, 1, 0x0000000200734060, 0, 167772164, 0x1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);
        }

        [TestMethod]
        public void TestWriteSizeMinus()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { start: 4, end: 5, result: 0x00000000, handle: 0x0000000200734060, function: \"WriteFile\", offset: 0, size: -1 }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);
        }

        [TestMethod]
        public void TestWriteCountFailure()
        {
            TestStream testStream = new TestStream();
            int writeAccessFunctionCount = FsFunction.Names.WriteAccess.Count();
            for (int i = 0; i < (320 / writeAccessFunctionCount) + 1; ++i)
            {
                int period = i * writeAccessFunctionCount;
                foreach (string function in FsFunction.Names.WriteAccess)
                {
                    testStream.Function(period, period + 1, function);
                    ++period;
                }
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.AreEqual(FsGuideline.CheckResult.ErrorType.WriteAccessCount, result.ErrorFlags);
        }

        [TestMethod]
        public void TestWriteCountSuccess()
        {
            TestStream testStream = new TestStream();
            int writeAccessFunctionCount = FsFunction.Names.WriteAccess.Count();
            for (int i = 0; i < (320 / writeAccessFunctionCount); ++i)
            {
                int period = i * writeAccessFunctionCount;
                foreach (string function in FsFunction.Names.WriteAccess)
                {
                    testStream.Function(period, period + 1, function);
                    ++period;
                }
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);
        }

        [TestMethod]
        public void TestWriteCountResultFailure()
        {
            TestStream testStream = new TestStream();
            for (int i = 0; i < 320; ++i)
            {
                testStream.WriteFile(0, 1, 0x0000000200734060, 0, 1);
            }
            testStream.WriteFile(0, 1, 0x0000000200734060, 0, 1, 0x1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);
        }

        [TestMethod]
        public void TestSameAddressReadCountFailure()
        {
            // 以下の様なアクセスのテスト
            //  ====
            //   ===
            //    ==
            //     =
            TestStream testStream = new TestStream();
            testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
            int accessCount = 601;
            ulong fileSize = (ulong)accessCount * 2;
            for (int i = 0; i < accessCount; ++i)
            {
                ulong offset = (ulong)i;
                testStream.ReadFile(i, i + 1, 0x0000000200734060, offset, fileSize - offset);
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.AreEqual(FsGuideline.CheckResult.ErrorType.SameAddressRead, result.ErrorFlags);

            Assert.AreEqual(1, result.FailReadAccessList.Count);
            foreach (var rawInfo in result.FailReadAccessList)
            {
                // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                Assert.AreEqual((ulong)(accessCount - 1), info.Positions.Offset);
                Assert.AreEqual(fileSize - info.Positions.Offset, info.Positions.Size);
                Assert.AreEqual(accessCount, info.Count);
            }

            Assert.AreEqual(accessCount - 1, result.PassReadAccessList.Count);
            foreach (var rawInfo in result.PassReadAccessList)
            {
                // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                Assert.AreEqual((ulong)1, info.Positions.Size);
                Assert.AreEqual((int)(info.Positions.Offset + 1), info.Count);
            }
        }

        [TestMethod]
        public void TestSameAddressReadCountFailureForSystem()
        {
            // 以下の様なアクセスのテスト
            //  ====
            //   ===
            //    ==
            //     =
            TestStream testStream = new TestStream();
            testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
            int accessCount = 151;
            ulong fileSize = (ulong)accessCount * 2;
            for (int i = 0; i < accessCount; ++i)
            {
                ulong offset = (ulong)i;
                testStream.ReadFile(i, i + 1, 0x0000000200734060, offset, fileSize - offset);
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            {
                var result = FsGuideline.Check(analyzer.LogList);
                Assert.IsFalse(result.ForSystem);
                Assert.IsFalse(result.IsVerificationError);

                Assert.AreEqual(0, result.FailReadAccessList.Count);

                Assert.AreEqual(accessCount, result.PassReadAccessList.Count);
                foreach (var rawInfo in result.PassReadAccessList)
                {
                    // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                    FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                    if (info.Positions.Offset == (ulong)(accessCount - 1))
                    {
                        Assert.AreEqual(fileSize - info.Positions.Offset, info.Positions.Size);
                        Assert.AreEqual(accessCount, info.Count);
                    }
                    else
                    {
                        Assert.AreEqual((ulong)1, info.Positions.Size);
                        Assert.AreEqual((int)(info.Positions.Offset + 1), info.Count);
                    }
                }
            }
            {
                var result = FsGuideline.CheckForSystem(analyzer.LogList);
                Assert.IsTrue(result.ForSystem);
                Assert.IsTrue(result.IsVerificationError);

                Assert.AreEqual(1, result.FailReadAccessList.Count);
                foreach (var rawInfo in result.FailReadAccessList)
                {
                    // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                    FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                    Assert.AreEqual((ulong)(accessCount - 1), info.Positions.Offset);
                    Assert.AreEqual(fileSize - info.Positions.Offset, info.Positions.Size);
                    Assert.AreEqual(accessCount, info.Count);
                }

                Assert.AreEqual(accessCount - 1, result.PassReadAccessList.Count);
                foreach (var rawInfo in result.PassReadAccessList)
                {
                    // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                    FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                    Assert.AreEqual((ulong)1, info.Positions.Size);
                    Assert.AreEqual((int)(info.Positions.Offset + 1), info.Count);
                }
            }
        }

        [TestMethod]
        public void TestSameAddressReadCountSuccess()
        {
            // 以下の様なアクセスのテスト
            //  ====
            //   ===
            //    ==
            //     =
            TestStream testStream = new TestStream();
            testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
            int accessCount = 600;
            ulong fileSize = (ulong)accessCount * 2;
            for (int i = 0; i < accessCount; ++i)
            {
                ulong offset = (ulong)i;
                testStream.ReadFile(i, i + 1, 0x0000000200734060, offset, fileSize - offset);
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);

            Assert.AreEqual(0, result.FailReadAccessList.Count);

            Assert.AreEqual(accessCount, result.PassReadAccessList.Count);
            foreach (var rawInfo in result.PassReadAccessList)
            {
                // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                if (info.Positions.Offset == (ulong)(accessCount - 1))
                {
                    Assert.AreEqual(fileSize - info.Positions.Offset, info.Positions.Size);
                    Assert.AreEqual(accessCount, info.Count);
                }
                else
                {
                    Assert.AreEqual((ulong)1, info.Positions.Size);
                    Assert.AreEqual((int)(info.Positions.Offset + 1), info.Count);
                }
            }
        }

        [TestMethod]
        public void TestSameAddressReadCountCamel()
        {
            // 以下の様なアクセスのテスト
            // ========================
            // ========================
            //  ========== ===========
            //   ========   =========
            //    ======     ======
            //     ====       ====
            // ========================
            TestStream testStream = new TestStream();
            testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
            int accessCount = 600;
            ulong fileSize = (ulong)accessCount * 4;
            testStream.ReadFile(0, 1, 0x0000000200734060, 0, fileSize);
            int period = 1;
            for (int i = 0; i < accessCount - 1; ++i)
            {
                ulong offset = (ulong)i;
                testStream.ReadFile(period, period + 1, 0x0000000200734060, offset, fileSize / 2 - offset * 2);
                ++period;
                testStream.ReadFile(period, period + 1, 0x0000000200734060, fileSize / 2 + offset, fileSize / 2 - offset * 2);
                ++period;
            }
            testStream.ReadFile(period, period + 1, 0x0000000200734060, 0, fileSize);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(accessCount * 2 + 1, analyzer.LogList.Count);
            {
                var result = FsGuideline.Check(analyzer.LogList);
                Assert.AreEqual(FsGuideline.CheckResult.ErrorType.SameAddressRead, result.ErrorFlags);

                foreach (var rawInfo in result.PassReadAccessList)
                {
                    // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                    FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                    // 二つの山をまとめて判定するためにオフセットをまとめる
                    var simpleOffset = info.Positions.Offset % (fileSize / 2);
                    if (simpleOffset < fileSize / 4)
                    {
                        // 山の前半部分
                        Assert.AreEqual((int)simpleOffset + 3, info.Count);
                        if (simpleOffset == fileSize / 4 - 2)
                        {
                            // ループの最後に読み込まれる領域
                            Assert.AreEqual((ulong)4, info.Positions.Size);
                        }
                        else
                        {
                            Assert.AreEqual((ulong)1, info.Positions.Size);
                        }
                    }
                    else
                    {
                        // 山の後半部分
                        Assert.AreEqual((int)(fileSize / 2 - simpleOffset + 2), info.Count);
                        Assert.AreEqual((ulong)1, info.Positions.Size);
                    }
                }
            }
            {
                analyzer.LogList.RemoveAt(1);
                var result = FsGuideline.Check(analyzer.LogList);
                Assert.IsFalse(result.IsVerificationError);
            }
        }

        [TestMethod]
        public void TestSameAddressReadCountShift()
        {
            // 大きな Read を1回したあと、1 byte ずつずらしながら 1byte 読むアクセスのテスト
            // 以下の様なアクセスのテスト
            //  ====
            //  =
            //   =
            //    =
            //     =
            TestStream testStream = new TestStream();
            testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
            ulong accessCount = 600;
            testStream.ReadFile(0, 1, 0x0000000200734060, 0, accessCount);
            for (ulong offset = 0; offset < accessCount; ++offset)
            {
                testStream.ReadFile(1, 2, 0x0000000200734060, offset, 1);
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);

            foreach (var rawInfo in result.PassReadAccessList)
            {
                // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                Assert.AreEqual(2, info.Count);
                Assert.AreEqual((ulong)1, info.Positions.Size);
            }
        }

        [TestMethod]
        public void TestSameAddressReadCountSameOffsetAndSize()
        {
            // 同一領域へのアクセスの衝突判定最適化のテスト
            // 以下の様なアクセスのテスト
            //  ============
            //  ============
            //  =
            //   =
            //    =
            //     =
            TestStream testStream = new TestStream();
            testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
            ulong accessCount = 601;
            testStream.ReadFile(0, 1, 0x0000000200734060, 0, 1024);
            testStream.ReadFile(0, 1, 0x0000000200734060, 0, 1024);
            for (ulong offset = 0; offset < accessCount - 2; ++offset)
            {
                testStream.ReadFile(1, 2, 0x0000000200734060, offset, 1);
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);

            foreach (var rawInfo in result.PassReadAccessList)
            {
                // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                if (info.Positions.Offset == accessCount - 2)
                {
                    Assert.AreEqual(2, info.Count);
                    Assert.AreEqual(1024 - info.Positions.Offset, info.Positions.Size);
                }
                else
                {
                    Assert.AreEqual(3, info.Count);
                    Assert.AreEqual((ulong)1, info.Positions.Size);
                }
            }
        }

        [TestMethod]
        public void TestSameAddressReadCountResultFailure()
        {
            // 以下の様なアクセスのテスト
            //  ========
            //   =======
            //    ======
            //     =====
            TestStream testStream = new TestStream();
            testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
            int accessCount = 601;
            ulong fileSize = (ulong)accessCount * 2;
            testStream.ReadFile(0, 1, 0x0000000200734060, 0, fileSize, 0x1);
            for (int i = 1; i < accessCount; ++i)
            {
                ulong offset = (ulong)i;
                testStream.ReadFile(i, i + 1, 0x0000000200734060, offset, fileSize - offset);
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);

            foreach (var rawInfo in result.PassReadAccessList)
            {
                // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                if (info.Positions.Offset == (ulong)(accessCount - 1))
                {
                    Assert.AreEqual(accessCount - 1, info.Count);
                    Assert.AreEqual(fileSize - info.Positions.Offset, info.Positions.Size);
                }
                else
                {
                    Assert.AreEqual((int)(info.Positions.Offset), info.Count);
                    Assert.AreEqual((ulong)1, info.Positions.Size);
                }
            }
        }

        [TestMethod]
        public void TestSameAddressReadCountMinusSize()
        {
            // サイズがマイナスのアクセス + 以下の様なアクセスのテスト
            //  ========
            //   =======
            //    ======
            //     =====
            TestStream testStream = new TestStream();
            testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
            int accessCount = 601;
            ulong fileSize = (ulong)accessCount * 2;
            testStream.Write("FS_ACCESS: { start: 0, end: 1, result: 0x00000000, handle: 0x0000000200734060, function: \"ReadFile\", offset: 0, size: -1 }");
            for (int i = 1; i < accessCount; ++i)
            {
                ulong offset = (ulong)i;
                testStream.ReadFile(i, i + 1, 0x0000000200734060, offset, fileSize - offset);
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);

            foreach (var rawInfo in result.PassReadAccessList)
            {
                // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                if (info.Positions.Offset == (ulong)(accessCount - 1))
                {
                    Assert.AreEqual(accessCount - 1, info.Count);
                    Assert.AreEqual(fileSize - info.Positions.Offset, info.Positions.Size);
                }
                else
                {
                    Assert.AreEqual((int)(info.Positions.Offset), info.Count);
                    Assert.AreEqual((ulong)1, info.Positions.Size);
                }
            }
        }

        [TestMethod]
        public void TestSameAddressReadCountMinusOffset()
        {
            // オフセットがマイナスのアクセス + 以下の様なアクセスのテスト
            //  ========
            //   =======
            //    ======
            //     =====
            TestStream testStream = new TestStream();
            testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
            int accessCount = 601;
            ulong fileSize = (ulong)accessCount * 2;
            testStream.Write("FS_ACCESS: { start: 0, end: 1, result: 0x00000000, handle: 0x0000000200734060, function: \"ReadFile\", offset: -1, size: 4 }");
            for (int i = 1; i < accessCount; ++i)
            {
                ulong offset = (ulong)i;
                testStream.ReadFile(i, i + 1, 0x0000000200734060, offset, fileSize - offset);
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);

            foreach (var rawInfo in result.PassReadAccessList)
            {
                // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                if (info.Positions.Offset == (ulong)(accessCount - 1))
                {
                    Assert.AreEqual(accessCount - 1, info.Count);
                    Assert.AreEqual(fileSize - info.Positions.Offset, info.Positions.Size);
                }
                else
                {
                    Assert.AreEqual((int)(info.Positions.Offset), info.Count);
                    Assert.AreEqual((ulong)1, info.Positions.Size);
                }
            }
        }

        [TestMethod]
        public void TestSameAddressReadMultipleBlock()
        {
            // 以下の様なアクセスを繰り返すテスト
            //  ====
            //  =
            //   =
            //    =
            //     =
            TestStream testStream = new TestStream();
            testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
            int loopCount = 300;
            for (int i = 0; i < loopCount; ++i)
            {
                int start = i * 5;
                testStream.ReadFile(i + 0, i + 1, 0x0000000200734060, 0, 512 * 4);
                testStream.ReadFile(i + 1, i + 2, 0x0000000200734060, 512 * 0, 512);
                testStream.ReadFile(i + 2, i + 3, 0x0000000200734060, 512 * 1, 512);
                testStream.ReadFile(i + 3, i + 4, 0x0000000200734060, 512 * 2, 512);
                testStream.ReadFile(i + 4, i + 5, 0x0000000200734060, 512 * 3, 512);
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);

            foreach (var rawInfo in result.PassReadAccessList)
            {
                // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                Assert.AreEqual(loopCount * 2, info.Count);
                Assert.AreEqual((ulong)512, info.Positions.Size);
            }
        }

        [TestMethod]
        public void TestSameAddressReadSize1Twice()
        {
            ulong fileSize = 3;
            ulong handle = 0x0000000200734060;

            // 以下のようなアクセス
            // ===
            // =   ← i の位置
            //  =  ← j の位置
            for (ulong i = 0; i < fileSize; ++i)
            {
                for (ulong j = 0; j < fileSize; ++j)
                {
                    TestStream testStream = new TestStream();
                    testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
                    testStream.ReadFile(0, 1, handle, 0, fileSize);
                    testStream.ReadFile(1, 2, handle, i, 1);
                    testStream.ReadFile(2, 3, handle, j, 1);
                    testStream.SetUp();

                    FsAccessLogParser parser = new FsAccessLogParser(testStream);
                    FsAccessLogAnalyzer analyzer = parser.Parse()[0];
                    var result = FsGuideline.Check(analyzer.LogList);
                    Assert.IsFalse(result.IsVerificationError);

                    foreach (var rawInfo in result.PassReadAccessList)
                    {
                        // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                        FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                        if (info.Positions.Offset == i || info.Positions.Offset == j)
                        {
                            Assert.AreEqual(i == j ? 3 : 2, info.Count);
                            Assert.AreEqual((ulong)1, info.Positions.Size);
                        }
                        else
                        {
                            Assert.AreEqual(1, info.Count);

                            if (i == j && i != 1)
                            {
                                // 012
                                // ===
                                // =
                                // =
                                // [offset: 1, size: 2, count: 1]
                                Assert.AreEqual((ulong)2, info.Positions.Size);
                            }
                            else
                            {
                                // 012
                                // ===
                                // =
                                //  =
                                // [offset: 2, size: 1, count: 1]
                                Assert.AreEqual((ulong)1, info.Positions.Size);
                            }
                        }
                    }
                }
            }
        }

        [TestMethod]
        public void TestSameAddressReadSize1And2()
        {
            ulong fileSize = 3;
            ulong handle = 0x0000000200734060;

            // 以下のようなアクセス
            // ===
            // =   ← i の位置
            // ==  ← j の位置
            for (ulong i = 0; i < fileSize; ++i)
            {
                for (ulong j = 0; j < fileSize - 1; ++j)
                {
                    bool isIntersect = (i == j) || (i == j + 1);
                    TestStream testStream = new TestStream();
                    testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
                    testStream.ReadFile(0, 1, handle, 0, fileSize);
                    testStream.ReadFile(1, 2, handle, i, 1);
                    testStream.ReadFile(2, 3, handle, j, 2);
                    testStream.SetUp();

                    FsAccessLogParser parser = new FsAccessLogParser(testStream);
                    FsAccessLogAnalyzer analyzer = parser.Parse()[0];
                    var result = FsGuideline.Check(analyzer.LogList);
                    Assert.IsFalse(result.IsVerificationError);

                    foreach (var rawInfo in result.PassReadAccessList)
                    {
                        // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                        FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                        if (info.Positions.Offset == i)
                        {
                            Assert.AreEqual(isIntersect ? 3 : 2, info.Count);
                            Assert.AreEqual((ulong)1, info.Positions.Size);
                        }
                        else if (info.Positions.Offset == j)
                        {
                            Assert.AreEqual(2, info.Count);
                            Assert.AreEqual((ulong)(isIntersect ? 1 : 2), info.Positions.Size);
                        }
                        else
                        {
                            if (info.Positions.Offset == j + 1 && isIntersect)
                            {
                                // 012
                                // ===
                                // =
                                // ==
                                // [offset: 1, size: 1, count: 2]
                                Assert.AreEqual(2, info.Count);
                            }
                            else
                            {
                                Assert.AreEqual(1, info.Count);
                            }
                            Assert.AreEqual((ulong)1, info.Positions.Size);
                        }
                    }
                }
            }
        }

        [TestMethod]
        public void TestSameAddressReadSize2Twice()
        {
            ulong fileSize = 3;
            ulong handle = 0x0000000200734060;

            // 以下のようなアクセス
            // ===
            // ==  ← i の位置
            //  == ← j の位置
            for (ulong i = 0; i < fileSize - 1; ++i)
            {
                for (ulong j = 0; j < fileSize - 1; ++j)
                {
                    TestStream testStream = new TestStream();
                    testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
                    testStream.ReadFile(0, 1, handle, 0, fileSize);
                    testStream.ReadFile(1, 2, handle, i, 2);
                    testStream.ReadFile(2, 3, handle, j, 2);
                    testStream.SetUp();

                    FsAccessLogParser parser = new FsAccessLogParser(testStream);
                    FsAccessLogAnalyzer analyzer = parser.Parse()[0];
                    var result = FsGuideline.Check(analyzer.LogList);
                    Assert.IsFalse(result.IsVerificationError);

                    foreach (var rawInfo in result.PassReadAccessList)
                    {
                        // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                        FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                        if (info.Positions.Offset == i || info.Positions.Offset == j)
                        {
                            if (i == j)
                            {
                                Assert.AreEqual(3, info.Count);
                                Assert.AreEqual((ulong)2, info.Positions.Size);
                            }
                            else
                            {
                                // ===
                                // ==
                                //  ==
                                // [offset: 0, size: 1, count: 2]
                                // [offset: 1, size: 1, count: 3]
                                Assert.AreEqual(info.Positions.Offset == 0 ? 2 : 3, info.Count);
                                Assert.AreEqual((ulong)1, info.Positions.Size);
                            }
                        }
                        else
                        {
                            Assert.AreEqual(i == j ? 1 : 2, info.Count);
                            Assert.AreEqual((ulong)1, info.Positions.Size);
                        }
                    }
                }
            }
        }

        [TestMethod]
        public void TestSameAddressReadSize1And3()
        {
            ulong fileSize = 3;
            ulong handle = 0x0000000200734060;

            // 以下のようなアクセス
            // ===
            // =   ← i の位置
            // ===
            for (ulong i = 0; i < fileSize; ++i)
            {
                TestStream testStream = new TestStream();
                testStream.OpenFile(0, 0, 0x0000000200734060, "test:/test");
                testStream.ReadFile(0, 1, handle, 0, fileSize);
                testStream.ReadFile(1, 2, handle, i, 1);
                testStream.ReadFile(2, 3, handle, 0, fileSize);
                testStream.SetUp();

                FsAccessLogParser parser = new FsAccessLogParser(testStream);
                FsAccessLogAnalyzer analyzer = parser.Parse()[0];
                var result = FsGuideline.Check(analyzer.LogList);
                Assert.IsFalse(result.IsVerificationError);

                foreach (var rawInfo in result.PassReadAccessList)
                {
                    // このテストにおける result.BadReadAccessList は全て SameAddressAccessInfoPath
                    FsGuideline.SameAddressAccessInfoPath info = (FsGuideline.SameAddressAccessInfoPath)rawInfo;
                    if (info.Positions.Offset == i)
                    {
                        Assert.AreEqual(3, info.Count);
                        Assert.AreEqual((ulong)1, info.Positions.Size);
                    }
                    else
                    {
                        Assert.AreEqual(2, info.Count);

                        if (i != 1)
                        {
                            // 012
                            // ===
                            // =
                            // ===
                            // [offset: 1, size: 2, count: 2]
                            Assert.AreEqual((ulong)2, info.Positions.Size);
                        }
                        else
                        {
                            // 012
                            // ===
                            //  =
                            // ===
                            // [offset: 0, size: 1, count: 2], [offset: 2, size: 1, count: 2]
                            Assert.AreEqual((ulong)1, info.Positions.Size);
                        }
                    }
                }
            }
        }

        [TestMethod]
        public void TestSetCheckStep()
        {
            Assert.IsTrue(FsGuideline.SetCheckStep(60));
            Assert.IsTrue(FsGuideline.IsChangeCheckStep());

            Assert.IsTrue(FsGuideline.SetCheckStep(4 * 60));
            Assert.IsTrue(FsGuideline.IsChangeCheckStep());

            Assert.IsTrue(FsGuideline.SetCheckStep(10));
            Assert.IsFalse(FsGuideline.IsChangeCheckStep());

            Assert.IsFalse(FsGuideline.SetCheckStep(4 * 60 + 1));
            Assert.IsFalse(FsGuideline.IsChangeCheckStep());

            Assert.IsFalse(FsGuideline.SetCheckStep(0));
            Assert.IsFalse(FsGuideline.IsChangeCheckStep());

            Assert.IsFalse(FsGuideline.SetCheckStep(-1));
            Assert.IsFalse(FsGuideline.IsChangeCheckStep());
        }

        [TestMethod]
        public void TestEnumerateDeliveryCacheDirectory()
        {
            Func<int, FsGuideline.CheckResult> check = (int count) =>
            {
                return CreateBcatGuidelineCheck(count,
                    (TestStream testStream, int i) =>
                    {
                        testStream.EnumerateDeliveryCacheDirectory(i, i);
                    });
            };
            {
                var result = check(600);
                Assert.IsFalse(result.IsVerificationError);
            }
            {
                var result = check(601);
                Assert.IsTrue(result.ErrorFlags.HasFlag(FsGuideline.CheckResult.ErrorType.SameAddressRead));
            }
        }

        [TestMethod]
        public void TestDeliveryCacheDirectoryOpen()
        {
            Func<int, FsGuideline.CheckResult> check = (int count) =>
            {
                return CreateBcatGuidelineCheck(count,
                    (TestStream testStream, int i) =>
                    {
                        testStream.DeliveryCacheDirectoryOpen(i, i, 1, "test");
                        testStream.DeliveryCacheDirectoryClose(i, i, 1);
                    });
            };
            {
                var result = check(600);
                Assert.IsFalse(result.IsVerificationError);
            }
            {
                var result = check(601);
                Assert.IsTrue(result.ErrorFlags.HasFlag(FsGuideline.CheckResult.ErrorType.SameAddressRead));
            }
        }

        [TestMethod]
        public void TestDeliveryCacheDirectoryOpenWithDestructor()
        {
            Func<int, FsGuideline.CheckResult> check = (int count) =>
            {
                return CreateBcatGuidelineCheck(count,
                    (TestStream testStream, int i) =>
                    {
                        testStream.DeliveryCacheDirectoryOpen(i, i, 1, "test");
                        testStream.DeliveryCacheDirectoryClose(i, i, 1);
                        testStream.DeliveryCacheDirectoryDestructor(i, i, 1);
                    });
            };
            {
                var result = check(600);
                Assert.IsFalse(result.IsVerificationError);
            }
            {
                var result = check(601);
                Assert.IsTrue(result.ErrorFlags.HasFlag(FsGuideline.CheckResult.ErrorType.SameAddressRead));
            }
        }

        [TestMethod]
        public void TestDeliveryCacheDirectoryOpenAndEnumerateDeliveryCacheDirectory()
        {
            Func<int, FsGuideline.CheckResult> check = (int count) =>
            {
                return CreateBcatGuidelineCheck(count,
                    (TestStream testStream, int i) =>
                    {
                        testStream.EnumerateDeliveryCacheDirectory(i, i);
                        testStream.DeliveryCacheDirectoryOpen(i, i, 1, "test");
                        testStream.DeliveryCacheDirectoryClose(i, i, 1);
                    });
            };
            {
                // EnumerateDeliveryCacheDirectory と DeliveryCacheDirectoryOpen は別の箇所を Read する
                var result = check(600);
                Assert.IsFalse(result.IsVerificationError);
            }
            {
                var result = check(601);
                Assert.IsTrue(result.ErrorFlags.HasFlag(FsGuideline.CheckResult.ErrorType.SameAddressRead));
            }
        }

        [TestMethod]
        public void TestDeliveryCacheDirectoryOpenAndRead()
        {
            Func<int, FsGuideline.CheckResult> check = (int count) =>
            {
                return CreateBcatGuidelineCheck(count,
                    (TestStream testStream, int i) =>
                    {
                        if (i == 0)
                        {
                            testStream.DeliveryCacheDirectoryOpen(i, i, 1, "test");
                        }
                        else
                        {
                            testStream.DeliveryCacheDirectoryRead(i, i, 1);
                        }
                    });
            };
            {
                // Open と Read は同じ箇所を Read する
                var result = check(600);
                Assert.IsFalse(result.IsVerificationError);
            }
            {
                var result = check(601);
                Assert.IsTrue(result.ErrorFlags.HasFlag(FsGuideline.CheckResult.ErrorType.SameAddressRead));
            }
        }

        [TestMethod]
        public void TestDeliveryCacheFileOpenAndRead()
        {
            Func<int, FsGuideline.CheckResult> check = (int count) =>
            {
                return CreateBcatGuidelineCheck(count,
                    (TestStream testStream, int i) =>
                    {
                        if (i == 0)
                        {
                            testStream.DeliveryCacheFileOpen(i, i, 1, "test/test.txt");
                        }
                        testStream.DeliveryCacheFileRead(i, i, 1, 0, 1);
                    });
            };
            {
                var result = check(600);
                Assert.IsFalse(result.IsVerificationError);
            }
            {
                var result = check(601);
                Assert.IsTrue(result.ErrorFlags.HasFlag(FsGuideline.CheckResult.ErrorType.SameAddressRead));
            }
        }

        private static FsGuideline.CheckResult CreateBcatGuidelineCheck(int count, Action<TestStream, int> action)
        {
            TestStream testStream = new TestStream();
            testStream.MountDeliveryCacheStorage(0, 0);
            for (int i = 0; i < count; ++i)
            {
                action(testStream, i);
            }
            testStream.UnmountDeliveryCacheStorage(count, count);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            return FsGuideline.Check(analyzer.LogList);
        }

        [TestMethod]
        public void TestNfpMountAndCreate()
        {
            TestStream testStream = new TestStream();
            testStream.NfpMount(0, 1, 0);
            testStream.NfpCreateApplicationArea(0, 1, 0, 16);
            testStream.NfpFlush(0, 1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);
        }

        [TestMethod]
        public void TestNfpMountAndRestore()
        {
            TestStream testStream = new TestStream();
            testStream.NfpMount(0, 1, 0);
            testStream.NfpRestore(0, 1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            var result = FsGuideline.Check(analyzer.LogList);
            Assert.IsFalse(result.IsVerificationError);
        }
    }
}
