﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <eftdemo_ScreenDrawer.h>
#include <eftdemo_Texture.h>

namespace nw      {
namespace eftdemo {


//---------------------------------------------------------------------------
//! @brief        コンストラクタです。
//---------------------------------------------------------------------------
ScreenDrawer::ScreenDrawer()
{
    {
        mWvpMatrix.SetIdentity();

        nw::math::MTX44 projMtx;
        projMtx.SetOrtho(-0.5f, 0.5f, -0.5f, 0.5f, 0.5f, 1.0f);

        nw::math::MTX34 viewMtx;
        viewMtx.SetLookAt(nw::math::VEC3(0.0f, 0.0f, 1.0f), nw::math::VEC3(0.0f, 1.0f, 0.0f), nw::math::VEC3(0.0f, 0.0f, 0.0f));
        const nw::math::MTX44 viewMatrix( viewMtx );

        nw::math::MTX44Mult( &mWvpMatrix, &projMtx, &viewMatrix );
    }

    {
        mWvpMatrixMirror.SetIdentity();

        nw::math::MTX44 projMtx;
        projMtx.SetOrtho(-0.5f, 0.5f, 0.5f, -0.5f, 0.5f, 1.0f);

        nw::math::MTX34 viewMtx;
        viewMtx.SetLookAt(nw::math::VEC3(0.0f, 0.0f, 1.0f), nw::math::VEC3(0.0f, 1.0f, 0.0f), nw::math::VEC3(0.0f, 0.0f, 0.0f));
        const nw::math::MTX44 viewMatrix( viewMtx );

        nw::math::MTX44Mult( &mWvpMatrixMirror, &projMtx, &viewMatrix );
    }
}


//---------------------------------------------------------------------------
//! @brief        終了処理です。
//---------------------------------------------------------------------------
void ScreenDrawer::Finalize()
{
    mPositionVB.Finalize( &mEffectHeap );
    mTexCoordVB.Finalize( &mEffectHeap );

    mShader[SHADER_TYPE_DEPTH_COPY].Finalize( &mEffectHeap );
    mShader[SHADER_TYPE_REDUCE].Finalize( &mEffectHeap );
    mShader[SHADER_TYPE_GAUSS_Y].Finalize( &mEffectHeap );
    mShader[SHADER_TYPE_GAUSS_X].Finalize( &mEffectHeap );
    mShader[SHADER_TYPE_GAUSS].Finalize( &mEffectHeap );
    mShader[SHADER_TYPE_LUMINANCE_SELECTION].Finalize( &mEffectHeap );
    mShader[SHADER_TYPE_COPY].Finalize( &mEffectHeap );
}


//---------------------------------------------------------------------------
//! @brief        初期化処理です。
//---------------------------------------------------------------------------
void ScreenDrawer::Initialize( nw::ut::IAllocator* allocator )
{
#if defined(NW_PLATFORM_WIN32)
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDevice::LoadArg loadVshArg;        loadVshArg.allocator        = allocator;
        nw::dev::FileDevice::LoadArg loadCopyArg;       loadCopyArg.allocator       = allocator;
        nw::dev::FileDevice::LoadArg loadLuminaceArg;   loadLuminaceArg.allocator   = allocator;
        nw::dev::FileDevice::LoadArg loadGaussArg;      loadGaussArg.allocator      = allocator;
        nw::dev::FileDevice::LoadArg loadGaussXArg;     loadGaussXArg.allocator     = allocator;
        nw::dev::FileDevice::LoadArg loadGaussYArg;     loadGaussYArg.allocator     = allocator;
        nw::dev::FileDevice::LoadArg loadReduceArg;     loadReduceArg.allocator     = allocator;
        nw::dev::FileDevice::LoadArg loadDepthCopyArg;  loadDepthCopyArg.allocator  = allocator;

        loadVshArg.path             = "eft/common/eftdemo_ScreenDrawer.vsh";
        u8* shaderVsh               = fileSystem->Load( loadVshArg );
        u32 shaderVshSize           = loadVshArg.readSize;

        loadCopyArg.path            = "eft/common/eftdemo_ScreenDrawer_Copy.fsh";
        u8* shaderCopyFsh           = fileSystem->Load( loadCopyArg );
        u32 shaderCopyFshSize       = loadCopyArg.readSize;

        loadLuminaceArg.path        = "eft/common/eftdemo_ScreenDrawer_LuminanceSelection.fsh";
        u8* shaderLuminanceFsh      = fileSystem->Load( loadLuminaceArg );
        u32 shaderLuminanceFshSize  = loadLuminaceArg.readSize;

        loadGaussArg.path           = "eft/common/eftdemo_ScreenDrawer_Gauss.fsh";
        u8* shaderGaussFsh          = fileSystem->Load( loadGaussArg );
        u32 shaderGaussFshSize      = loadGaussArg.readSize;

        loadGaussXArg.path          = "eft/common/eftdemo_ScreenDrawer_Gauss_X.fsh";
        u8* shaderGaussXFsh         = fileSystem->Load( loadGaussXArg );
        u32 shaderGaussXFshSize     = loadGaussXArg.readSize;

        loadGaussYArg.path          = "eft/common/eftdemo_ScreenDrawer_Gauss_Y.fsh";
        u8* shaderGaussYFsh         = fileSystem->Load( loadGaussYArg );
        u32 shaderGaussYFshSize     = loadGaussYArg.readSize;

        loadReduceArg.path          = "eft/common/eftdemo_ScreenDrawer_Reduce.fsh";
        u8* shaderReduceFsh         = fileSystem->Load( loadReduceArg );
        u32 shaderReduceFshSize     = loadReduceArg.readSize;

        loadDepthCopyArg.path       = "eft/common/eftdemo_ScreenDrawer_DepthCopy.fsh";
        u8* shaderDepthCopyFsh      = fileSystem->Load( loadDepthCopyArg );
        u32 shaderDepthCopyFshSize  = loadDepthCopyArg.readSize;

        Initialize( allocator,
                    shaderVsh,         shaderVshSize,
                    shaderCopyFsh,     shaderCopyFshSize,
                    shaderLuminanceFsh,shaderLuminanceFshSize,
                    shaderGaussFsh,    shaderGaussFshSize,
                    shaderGaussXFsh,   shaderGaussXFshSize,
                    shaderGaussYFsh,   shaderGaussYFshSize,
                    shaderReduceFsh,   shaderReduceFshSize,
                    shaderDepthCopyFsh,shaderDepthCopyFshSize );

        fileSystem->Unload( loadVshArg,         shaderVsh );
        fileSystem->Unload( loadCopyArg,        shaderCopyFsh );
        fileSystem->Unload( loadLuminaceArg,    shaderLuminanceFsh );
        fileSystem->Unload( loadGaussArg,       shaderGaussFsh );
        fileSystem->Unload( loadGaussXArg,      shaderGaussXFsh );
        fileSystem->Unload( loadReduceArg,      shaderReduceFsh );
        fileSystem->Unload( loadDepthCopyArg,   shaderDepthCopyFsh );
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
#endif
#if defined(NW_PLATFORM_CAFE)
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "eft/eftdemo_ScreenDrawer.gsh";
        loadArg.allocator = allocator;
        u8* binary = fileSystem->Load( loadArg );

        Initialize( allocator, binary, loadArg.readSize );

        fileSystem->Unload( loadArg, binary );
    }
#endif
}


//---------------------------------------------------------------------------
//! @brief        初期化処理です。
//---------------------------------------------------------------------------
#if defined( NW_PLATFORM_WIN32 )
void ScreenDrawer::Initialize(  nw::ut::IAllocator* allocator,
                                const u8* screenDrawerShaderVsh,          u32 screenDrawerShaderVshSize,
                                const u8* screenDrawerShaderCopyFsh,      u32 screenDrawerShaderCopyFshSize,
                                const u8* screenDrawerShaderLuminanceFsh, u32 screenDrawerShaderLuminanceFshSize,
                                const u8* screenDrawerShaderGaussFsh,     u32 screenDrawerShaderGaussFshSize,
                                const u8* screenDrawerShaderGaussXFsh,    u32 screenDrawerShaderGaussXFshSize,
                                const u8* screenDrawerShaderGaussYFsh,    u32 screenDrawerShaderGaussYFshSize,
                                const u8* screenDrawerShaderReduceFsh,    u32 screenDrawerShaderReduceFshSize,
                                const u8* screenDrawerShaderDepthCopyFsh, u32 screenDrawerShaderDepthCopyFshSize )
{
    mAllocator = allocator;
    mEffectHeap.SetNwAllocator( mAllocator );

    // コピー
    mShader[SHADER_TYPE_COPY].Compile(
        reinterpret_cast<const GLchar *>( screenDrawerShaderVsh ), screenDrawerShaderVshSize,
        reinterpret_cast<const GLchar *>( screenDrawerShaderCopyFsh ), screenDrawerShaderCopyFshSize );

    // 輝度抽出
    mShader[SHADER_TYPE_LUMINANCE_SELECTION].Compile(
        reinterpret_cast<const GLchar *>( screenDrawerShaderVsh ), screenDrawerShaderVshSize,
        reinterpret_cast<const GLchar *>( screenDrawerShaderLuminanceFsh ), screenDrawerShaderLuminanceFshSize );

    // ガウス
    mShader[SHADER_TYPE_GAUSS].Compile(
        reinterpret_cast<const GLchar *>( screenDrawerShaderVsh ), screenDrawerShaderVshSize,
        reinterpret_cast<const GLchar *>( screenDrawerShaderGaussFsh ), screenDrawerShaderGaussFshSize );


    // ガウス X軸方向
    mShader[SHADER_TYPE_GAUSS_X].Compile(
        reinterpret_cast<const GLchar *>( screenDrawerShaderVsh ), screenDrawerShaderVshSize,
        reinterpret_cast<const GLchar *>( screenDrawerShaderGaussXFsh ), screenDrawerShaderGaussXFshSize );

    // ガウス Y軸方向
    mShader[SHADER_TYPE_GAUSS_Y].Compile(
        reinterpret_cast<const GLchar *>( screenDrawerShaderVsh ), screenDrawerShaderVshSize,
        reinterpret_cast<const GLchar *>( screenDrawerShaderGaussYFsh ), screenDrawerShaderGaussYFshSize );

    // 縮小
        mShader[SHADER_TYPE_REDUCE].Compile(
            reinterpret_cast<const GLchar *>( screenDrawerShaderVsh ), screenDrawerShaderVshSize,
            reinterpret_cast<const GLchar *>( screenDrawerShaderReduceFsh ), screenDrawerShaderReduceFshSize );

    // デプスコピー
        mShader[SHADER_TYPE_DEPTH_COPY].Compile(
            reinterpret_cast<const GLchar *>( screenDrawerShaderVsh ), screenDrawerShaderVshSize,
            reinterpret_cast<const GLchar *>( screenDrawerShaderDepthCopyFsh ), screenDrawerShaderDepthCopyFshSize );

#endif
#if defined( NW_PLATFORM_CAFE )
void ScreenDrawer::Initialize( nw::ut::IAllocator* allocator, void* shaderRes, u32 shaderResSize )
{
    mAllocator = allocator;
    mEffectHeap.SetNwAllocator( mAllocator );

    // コピー
    mShader[SHADER_TYPE_COPY].CreateShader( &mEffectHeap, shaderRes, shaderResSize, SHADER_TYPE_COPY );

    // 輝度抽出
    mShader[SHADER_TYPE_LUMINANCE_SELECTION].CreateShader( &mEffectHeap, shaderRes, shaderResSize, SHADER_TYPE_LUMINANCE_SELECTION );

    // ガウス
    mShader[SHADER_TYPE_GAUSS].CreateShader( &mEffectHeap, shaderRes, shaderResSize, SHADER_TYPE_GAUSS );

    // ガウス X軸方向
    mShader[SHADER_TYPE_GAUSS_X].CreateShader( &mEffectHeap, shaderRes, shaderResSize, SHADER_TYPE_GAUSS_X );

    // ガウス Y軸方向
    mShader[SHADER_TYPE_GAUSS_Y].CreateShader( &mEffectHeap, shaderRes, shaderResSize, SHADER_TYPE_GAUSS_Y );

    // 縮小
    mShader[SHADER_TYPE_REDUCE].CreateShader( &mEffectHeap, shaderRes, shaderResSize, SHADER_TYPE_REDUCE );

    // デプスコピー
    mShader[SHADER_TYPE_DEPTH_COPY].CreateShader( &mEffectHeap, shaderRes, shaderResSize, SHADER_TYPE_DEPTH_COPY );

#endif
    // 描画マトリクス初期化
    _setupOrthoProjectionMatrix();

    // コピー
    mPositionAttr[SHADER_TYPE_COPY]       = mShader[SHADER_TYPE_COPY].GetAttribute( "Position",  0, nw::eft2::FORMAT_32_32_32_FLOAT, 0, false, false );
    mTexCoordAttr[SHADER_TYPE_COPY]       = mShader[SHADER_TYPE_COPY].GetAttribute( "TexCoord0", 1, nw::eft2::FORMAT_32_32_FLOAT,    0, false, false );
    mWVPOffset[SHADER_TYPE_COPY]          = mShader[SHADER_TYPE_COPY].GetVertexUniformLocation( "ViewProjectionMatrix" );
    mTextureLocation[SHADER_TYPE_COPY]    = mShader[SHADER_TYPE_COPY].GetFragmentSamplerLocation( "Texture0" );
    mShader[SHADER_TYPE_COPY].SetupShader( &mEffectHeap );

    // 輝度抽出
    mPositionAttr[SHADER_TYPE_LUMINANCE_SELECTION]       = mShader[SHADER_TYPE_LUMINANCE_SELECTION].GetAttribute( "Position",  0, nw::eft2::FORMAT_32_32_32_FLOAT, 0, false, false );
    mTexCoordAttr[SHADER_TYPE_LUMINANCE_SELECTION]       = mShader[SHADER_TYPE_LUMINANCE_SELECTION].GetAttribute( "TexCoord0", 1, nw::eft2::FORMAT_32_32_FLOAT,    0, false, false );
    mWVPOffset[SHADER_TYPE_LUMINANCE_SELECTION]          = mShader[SHADER_TYPE_LUMINANCE_SELECTION].GetVertexUniformLocation( "ViewProjectionMatrix" );
    mTextureLocation[SHADER_TYPE_LUMINANCE_SELECTION]    = mShader[SHADER_TYPE_LUMINANCE_SELECTION].GetFragmentSamplerLocation( "Texture0" );
    mShader[SHADER_TYPE_LUMINANCE_SELECTION].SetupShader( &mEffectHeap );

    // ガウス
    mPositionAttr[SHADER_TYPE_GAUSS]           = mShader[SHADER_TYPE_GAUSS].GetAttribute( "Position",  0, nw::eft2::FORMAT_32_32_32_FLOAT, 0, false, false );
    mTexCoordAttr[SHADER_TYPE_GAUSS]           = mShader[SHADER_TYPE_GAUSS].GetAttribute( "TexCoord0", 1, nw::eft2::FORMAT_32_32_FLOAT,    0, false, false );
    mWVPOffset[SHADER_TYPE_GAUSS]              = mShader[SHADER_TYPE_GAUSS].GetVertexUniformLocation( "ViewProjectionMatrix" );
    mTextureLocation[SHADER_TYPE_GAUSS]        = mShader[SHADER_TYPE_GAUSS].GetFragmentSamplerLocation( "Texture0" );
    mTextureSize[SHADER_TYPE_GAUSS]            = mShader[SHADER_TYPE_GAUSS].GetFragmentUniformLocation( "TextureSize0" );
    mShader[SHADER_TYPE_GAUSS].SetupShader( &mEffectHeap );

    // ガウス X軸方向
    mPositionAttr[SHADER_TYPE_GAUSS_X]       = mShader[SHADER_TYPE_GAUSS_X].GetAttribute( "Position",  0, nw::eft2::FORMAT_32_32_32_FLOAT, 0, false, false );
    mTexCoordAttr[SHADER_TYPE_GAUSS_X]       = mShader[SHADER_TYPE_GAUSS_X].GetAttribute( "TexCoord0", 1, nw::eft2::FORMAT_32_32_FLOAT,    0, false, false );
    mWVPOffset[SHADER_TYPE_GAUSS_X]          = mShader[SHADER_TYPE_GAUSS_X].GetVertexUniformLocation( "ViewProjectionMatrix" );
    mTextureLocation[SHADER_TYPE_GAUSS_X]    = mShader[SHADER_TYPE_GAUSS_X].GetFragmentSamplerLocation( "Texture0" );
    mTextureSize[SHADER_TYPE_GAUSS_X]        = mShader[SHADER_TYPE_GAUSS_X].GetFragmentUniformLocation( "TextureSize0" );
    mWeight[SHADER_TYPE_GAUSS_X]             = mShader[SHADER_TYPE_GAUSS_X].GetFragmentUniformLocation( "Weight" );
    mShader[SHADER_TYPE_GAUSS_X].SetupShader( &mEffectHeap );

    // ガウス Y軸方向
    mPositionAttr[SHADER_TYPE_GAUSS_Y]       = mShader[SHADER_TYPE_GAUSS_Y].GetAttribute( "Position",  0, nw::eft2::FORMAT_32_32_32_FLOAT, 0, false, false );
    mTexCoordAttr[SHADER_TYPE_GAUSS_Y]       = mShader[SHADER_TYPE_GAUSS_Y].GetAttribute( "TexCoord0", 1, nw::eft2::FORMAT_32_32_FLOAT,    0, false, false );
    mWVPOffset[SHADER_TYPE_GAUSS_Y]          = mShader[SHADER_TYPE_GAUSS_Y].GetVertexUniformLocation( "ViewProjectionMatrix" );
    mTextureLocation[SHADER_TYPE_GAUSS_Y]    = mShader[SHADER_TYPE_GAUSS_Y].GetFragmentSamplerLocation( "Texture0" );
    mTextureSize[SHADER_TYPE_GAUSS_Y]        = mShader[SHADER_TYPE_GAUSS_Y].GetFragmentUniformLocation( "TextureSize0" );
    mWeight[SHADER_TYPE_GAUSS_Y]             = mShader[SHADER_TYPE_GAUSS_Y].GetFragmentUniformLocation( "Weight" );
    mShader[SHADER_TYPE_GAUSS_Y].SetupShader( &mEffectHeap );

    // 縮小
    mPositionAttr[SHADER_TYPE_REDUCE]           = mShader[SHADER_TYPE_REDUCE].GetAttribute( "Position",  0, nw::eft2::FORMAT_32_32_32_FLOAT, 0, false, false );
    mTexCoordAttr[SHADER_TYPE_REDUCE]           = mShader[SHADER_TYPE_REDUCE].GetAttribute( "TexCoord0", 1, nw::eft2::FORMAT_32_32_FLOAT,    0, false, false );
    mWVPOffset[SHADER_TYPE_REDUCE]              = mShader[SHADER_TYPE_REDUCE].GetVertexUniformLocation( "ViewProjectionMatrix" );
    mTextureLocation[SHADER_TYPE_REDUCE]        = mShader[SHADER_TYPE_REDUCE].GetFragmentSamplerLocation( "Texture0" );
    mTextureSize[SHADER_TYPE_REDUCE]            = mShader[SHADER_TYPE_REDUCE].GetFragmentUniformLocation( "TextureSize0" );
    mShader[SHADER_TYPE_REDUCE].SetupShader( &mEffectHeap );

    // デプスコピー
    mPositionAttr[SHADER_TYPE_DEPTH_COPY]       = mShader[SHADER_TYPE_DEPTH_COPY].GetAttribute( "Position",  0, nw::eft2::FORMAT_32_32_32_FLOAT, 0, false, false );
    mTexCoordAttr[SHADER_TYPE_DEPTH_COPY]       = mShader[SHADER_TYPE_DEPTH_COPY].GetAttribute( "TexCoord0", 1, nw::eft2::FORMAT_32_32_FLOAT,    0, false, false );
    mWVPOffset[SHADER_TYPE_DEPTH_COPY]          = mShader[SHADER_TYPE_DEPTH_COPY].GetVertexUniformLocation( "ViewProjectionMatrix" );
    mTextureLocation[SHADER_TYPE_DEPTH_COPY]    = mShader[SHADER_TYPE_DEPTH_COPY].GetFragmentSamplerLocation( "Texture0" );
    mTextureSize[SHADER_TYPE_DEPTH_COPY]        = mShader[SHADER_TYPE_DEPTH_COPY].GetFragmentUniformLocation( "TextureSize0" );
    mShader[SHADER_TYPE_DEPTH_COPY].SetupShader( &mEffectHeap );

    // 頂点アトリビュート初期化
    mPositionVB.Initialize();
    nw::math::VEC3* pos = static_cast<nw::math::VEC3*>( mPositionVB.AllocateVertexBuffer( &mEffectHeap, sizeof(nw::math::VEC3) * 4, 3 ) );
    {
        pos[0].x = -0.5f; pos[0].y =  0.5f; pos[0].z = 0.0f;
        pos[1].x = -0.5f; pos[1].y = -0.5f; pos[1].z = 0.0f;
        pos[2].x = 0.5f;  pos[2].y = -0.5f; pos[2].z = 0.0f;
        pos[3].x = 0.5f;  pos[3].y =  0.5f; pos[3].z = 0.0f;
    }
    mPositionVB.Validate( true );

#if defined( NW_PLATFORM_WIN32 )
    float vBottom = 0.0f, vTop = 1.0f;
#endif
#if defined( NW_PLATFORM_CAFE )
    float vBottom = 1.0f, vTop = 0.0f;
#endif

    mTexCoordVB.Initialize();
    nw::math::VEC2* tex = static_cast<nw::math::VEC2*>( mTexCoordVB.AllocateVertexBuffer( &mEffectHeap, sizeof(nw::math::VEC2) * 4, 2 ) );
    {
        tex[0].x = 0.0f; tex[0].y = vTop;
        tex[1].x = 0.0f; tex[1].y = vBottom;
        tex[2].x = 1.0f; tex[2].y = vBottom;
        tex[3].x = 1.0f; tex[3].y = vTop;
    }
    mTexCoordVB.Validate( true );

    // テクスチャサンプラ
    m_ResTextureSampler.filter      = nw::eft2::EFT_TEXTURE_FILTER_TYPE_LINEAR;
    m_ResTextureSampler.wrapU       = nw::eft2::EFT_TEXTURE_WRAP_TYPE_REPEAT;
    m_ResTextureSampler.wrapV       = nw::eft2::EFT_TEXTURE_WRAP_TYPE_REPEAT;
    m_ResTextureSampler.isSphereMap = false;
    m_ResTextureSampler.mipLevel    = 15.99f;
    m_ResTextureSampler.mipMapBias  = 0.0f;
    mTextureSampler.Initialize( &m_ResTextureSampler );
} // NOLINT(readability/fn_size)


//---------------------------------------------------------------------------
//! @brief        正射影マトリクスを生成します。
//---------------------------------------------------------------------------
void ScreenDrawer::_setupOrthoProjectionMatrix()
{
    {
        mWvpMatrix.SetIdentity();

        nw::math::MTX44 projMtx;
        nw::math::MTX34 viewMtx;

        projMtx.SetIdentity();
        projMtx.SetOrtho(-0.5f, 0.5f, -0.5f, 0.5f, 0.5f, 1.0f);

        viewMtx.SetIdentity();
        viewMtx.SetLookAt( nw::math::VEC3(0.0f, 0.0f, 1.0f), nw::math::VEC3(0.0f, 1.0f, 0.0f), nw::math::VEC3(0.0f, 0.0f, 0.0f) );
        const nw::math::MTX44 viewMatrix( viewMtx );
        nw::math::MTX44Mult( &mWvpMatrix, &projMtx, &viewMatrix );
    }

    {
        mWvpMatrixMirror.SetIdentity();

        nw::math::MTX44 projMtx;
        nw::math::MTX34 viewMtx;

        projMtx.SetIdentity();
        projMtx.SetOrtho(-0.5f, 0.5f, 0.5f, -0.5f, 0.5f, 1.0f);

        viewMtx.SetIdentity();
        viewMtx.SetLookAt( nw::math::VEC3(0.0f, 0.0f, 1.0f), nw::math::VEC3(0.0f, 1.0f, 0.0f), nw::math::VEC3(0.0f, 0.0f, 0.0f) );
        const nw::math::MTX44 viewMatrix( viewMtx );
        nw::math::MTX44Mult( &mWvpMatrixMirror, &projMtx, &viewMatrix );
    }
}


//---------------------------------------------------------------------------
//      テクスチャをセットアップします。
//---------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
void ScreenDrawer::_setupTexture( ShaderType type, GLuint texture )
{
    glActiveTexture( GL_TEXTURE0 );
    glBindTexture( GL_TEXTURE_2D , texture );
    glUniform1i( mTextureLocation[type], 0 );
    if ( texture == 0 )
    {
        return;
    }
    mTextureSampler.Bind( static_cast<u32>( nw::eft2::EFT_INVALID_TEXTURE_LOCATION ), mTextureLocation[type], 0 );
}
#endif
#if defined(NW_PLATFORM_CAFE)
void ScreenDrawer::_setupTexture( ShaderType type, const GX2Texture* texture )
{
    if ( texture == 0 )
    {
        return;
    }
    GX2SetPixelTexture( texture, mTextureLocation[type] );

    GX2InitSamplerXYFilter( (GX2Sampler *)mTextureSampler.GetSampler(),
        GX2_TEX_XY_FILTER_BILINEAR,
        GX2_TEX_XY_FILTER_BILINEAR,
        GX2_TEX_ANISO_1_TO_1 );
    GX2SetPixelSampler( (GX2Sampler *)mTextureSampler.GetSampler(), mTextureLocation[type] );
}
#endif


//---------------------------------------------------------------------------
//! @brief        画面全体に描画します。
//---------------------------------------------------------------------------
void ScreenDrawer::DrawScreen( ShaderType type, Texture* texture, bool mirror )
{
    // シェーダ切り替え
    mShader[type].Bind();

    // 頂点バッファ設定
    mPositionVB.BindBuffer( mPositionAttr[type], sizeof(nw::math::VEC3) * 4,  sizeof(nw::math::VEC3) );
    mTexCoordVB.BindBuffer( mTexCoordAttr[type], sizeof(nw::math::VEC2) * 4,  sizeof(nw::math::VEC2) );

#if defined( NW_PLATFORM_WIN32 )
    // マトリクス設定
    if ( mirror )
    {
        glUniformMatrix4fv( mWVPOffset[type], 1, GL_FALSE, mWvpMatrixMirror.a );
    }
    else
    {
        glUniformMatrix4fv( mWVPOffset[type], 1, GL_FALSE, mWvpMatrix.a );
    }

    // テクスチャサイズ設定
    if ( type == SHADER_TYPE_GAUSS || type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        glUniform2f( mTextureSize[type], static_cast<GLfloat>( texture->GetSize().x ), static_cast<GLfloat>( texture->GetSize().y ) );
    }

    // ガウスフィルタの重み設定
    if ( type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        glUniform1fv( mWeight[type], GAUSS_LENGTH, mWeightValue );
    }

    // テクスチャ設定
    _setupTexture( type, texture->GetGLTextureID() );
#endif
#if defined(NW_PLATFORM_CAFE)
    // マトリクス設定
    if ( mirror )
    {
        GX2SetVertexUniformReg( mWVPOffset[type], 4 * 4, mWvpMatrixMirror );
    }
    else
    {
        GX2SetVertexUniformReg( mWVPOffset[type], 4 * 4, mWvpMatrix );
    }


    // テクスチャサイズ設定
    if ( type == SHADER_TYPE_GAUSS || type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        float size[4] = { texture->GetSize().x, texture->GetSize().y, 0.0f, 0.0f };
        GX2SetPixelUniformReg( mTextureSize[type], 4, size );
    }

    // ガウスフィルタの重み設定
    if ( type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        GX2SetPixelUniformReg( mWeight[type], GAUSS_LENGTH * 4, mWeightValue );
    }

    // テクスチャ設定
    _setupTexture( type, texture->GetGX2Texture() );
#endif

    // 描画
    nw::eft2::Render::DrawPrimitive( nw::eft2::Render::PRIM_TYPE_QUADS, 0, 4 );

    _setupTexture( type, 0 );
}

void ScreenDrawer::DrawScreen( ShaderType type, const nw::eft2::TextureResource* texture, bool mirror )
{
    // シェーダ切り替え
    mShader[type].Bind();

    // 頂点バッファ設定
    mPositionVB.BindBuffer( mPositionAttr[type], sizeof(nw::math::VEC3) * 4,  sizeof(nw::math::VEC3) );
    mTexCoordVB.BindBuffer( mTexCoordAttr[type], sizeof(nw::math::VEC2) * 4,  sizeof(nw::math::VEC2) );

#if defined( NW_PLATFORM_WIN32 )
    // マトリクス設定
    if ( mirror )
    {
        glUniformMatrix4fv( mWVPOffset[type], 1, GL_FALSE, mWvpMatrixMirror.a );
    }
    else
    {
        glUniformMatrix4fv( mWVPOffset[type], 1, GL_FALSE, mWvpMatrix.a );
    }

    // テクスチャサイズ設定
    if ( type == SHADER_TYPE_GAUSS || type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        glUniform2f( mTextureSize[type], static_cast<GLfloat>( texture->GetSizeX() ), static_cast<GLfloat>( texture->GetSizeY() ) );
    }

    // ガウスフィルタの重み設定
    if ( type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        glUniform1fv( mWeight[type], GAUSS_LENGTH, mWeightValue );
    }

    // テクスチャ設定
    _setupTexture( type, texture->GetTexture() );
#endif
#if defined(NW_PLATFORM_CAFE)
    // マトリクス設定
    if ( mirror )
    {
        GX2SetVertexUniformReg( mWVPOffset[type], 4 * 4, mWvpMatrixMirror );
    }
    else
    {
        GX2SetVertexUniformReg( mWVPOffset[type], 4 * 4, mWvpMatrix );
    }


    // テクスチャサイズ設定
    if ( type == SHADER_TYPE_GAUSS || type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        float size[4] = { texture->GetSizeX(), texture->GetSizeY(), 0.0f, 0.0f };
        GX2SetPixelUniformReg( mTextureSize[type], 4, size );
    }

    // ガウスフィルタの重み設定
    if ( type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        GX2SetPixelUniformReg( mWeight[type], GAUSS_LENGTH * 4, mWeightValue );
    }

    // テクスチャ設定
    _setupTexture( type, texture->GetTextureID() );
#endif

    // 描画
    nw::eft2::Render::DrawPrimitive( nw::eft2::Render::PRIM_TYPE_QUADS, 0, 4 );

    _setupTexture( type, 0 );
}
void ScreenDrawer::DrawScreen( ShaderType type, FrameBuffer* frameBuffer, bool mirror )
{
    // シェーダ切り替え
    mShader[type].Bind();

    // 頂点バッファ設定
    mPositionVB.BindBuffer( mPositionAttr[type], sizeof(nw::math::VEC3) * 4,  sizeof(nw::math::VEC3) );
    mTexCoordVB.BindBuffer( mTexCoordAttr[type], sizeof(nw::math::VEC2) * 4,  sizeof(nw::math::VEC2) );

#if defined( NW_PLATFORM_WIN32 )
    // マトリクス設定
    if ( mirror )
    {
        glUniformMatrix4fv( mWVPOffset[type], 1, GL_FALSE, mWvpMatrixMirror.a );
    }
    else
    {
        glUniformMatrix4fv( mWVPOffset[type], 1, GL_FALSE, mWvpMatrix.a );
    }

    // テクスチャサイズ設定
    if ( type == SHADER_TYPE_GAUSS || type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        glUniform2f( mTextureSize[type], frameBuffer->GetSize().x, frameBuffer->GetSize().y );
    }

    // ガウスフィルタの重み設定
    if ( type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        glUniform1fv( mWeight[type], GAUSS_LENGTH, mWeightValue );
    }

    // テクスチャ設定
    _setupTexture( type, frameBuffer->GetColorBufferTexture() );
#endif
#if defined(NW_PLATFORM_CAFE)
    // マトリクス設定
    if ( mirror )
    {
        GX2SetVertexUniformReg( mWVPOffset[type], 4 * 4, mWvpMatrixMirror );
    }
    else
    {
        GX2SetVertexUniformReg( mWVPOffset[type], 4 * 4, mWvpMatrix );
    }

    // テクスチャサイズ設定
    if ( type == SHADER_TYPE_GAUSS || type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        float size[4] = { frameBuffer->GetSize().x, frameBuffer->GetSize().y, 0.0f, 0.0f };
        GX2SetPixelUniformReg( mTextureSize[type], 4, size );
    }

    // ガウスフィルタの重み設定
    if ( type == SHADER_TYPE_GAUSS_X || type == SHADER_TYPE_GAUSS_Y )
    {
        GX2SetPixelUniformReg( mWeight[type], GAUSS_LENGTH * 4, mWeightValue );
    }

    // テクスチャ設定
    _setupTexture( type, frameBuffer->GetColorBufferTexture() );
#endif

    // 描画
    nw::eft2::Render::DrawPrimitive( nw::eft2::Render::PRIM_TYPE_QUADS, 0, 4 );

    _setupTexture( type, 0 );
}


//---------------------------------------------------------------------------
//! @brief        小窓を描画します。
//---------------------------------------------------------------------------
void ScreenDrawer::DrawWipe( FrameBuffer* frameBuffer, f32 left, f32 bottom, f32 right, f32 top )
{
    // シェーダ切り替え
    mShader[SHADER_TYPE_COPY].Bind();

    // 頂点バッファ設定
    mPositionVB.BindBuffer( mPositionAttr[SHADER_TYPE_COPY], sizeof(nw::math::VEC3) * 4,  sizeof(nw::math::VEC3) );
    mTexCoordVB.BindBuffer( mTexCoordAttr[SHADER_TYPE_COPY], sizeof(nw::math::VEC2) * 4,  sizeof(nw::math::VEC2) );

    // マトリクス計算
    nw::math::Matrix44 wvpMatrix;

    float width = right - left;
    float height = top - bottom;

    nw::math::Vector3 pos( left + width / 2.0f, bottom + height / 2.0f, 0.0f );
    nw::math::MTX44Translate(&wvpMatrix, pos);
    nw::math::Vector3 scale( width, height, 1.0f );
    nw::math::MTX44MultScale( &wvpMatrix, wvpMatrix, scale );

#if defined(NW_PLATFORM_WIN32)
    // マトリクス設定
    glUniformMatrix4fv( mWVPOffset[SHADER_TYPE_COPY], 1, GL_FALSE, wvpMatrix.a );
#endif
#if defined(NW_PLATFORM_CAFE)
    // マトリクス設定
    GX2SetVertexUniformReg( mWVPOffset[SHADER_TYPE_COPY], 4 * 4, wvpMatrix.a );
#endif

    // テクスチャ設定
    _setupTexture( SHADER_TYPE_COPY, frameBuffer->GetColorBufferTexture() );

    // 描画
    nw::eft2::Render::DrawPrimitive( nw::eft2::Render::PRIM_TYPE_QUADS, 0, 4 );

    _setupTexture( SHADER_TYPE_COPY, 0 );
}


//---------------------------------------------------------------------------
//! @brief        キューブマップテクスチャを描画します。
//---------------------------------------------------------------------------
//     ┌─┐
//     │+Y│
// ┌─┼─┼─┬─┐
// │-X│-Z│+X│+Z│
// └─┼─┼─┴─┘
//     │-Y│
//     └─┘
//---------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
void ScreenDrawer::DrawCubeMapTexture( GLuint* texture, f32 left, f32 bottom, f32 right, f32 top )
#endif
#if defined(NW_PLATFORM_CAFE)
void ScreenDrawer::DrawCubeMapTexture( GX2Texture* texture, f32 left, f32 bottom, f32 right, f32 top )
#endif
{
    float width = (right - left) / 4.0f;
    float height = (top - bottom) / 3.0f;

    static const nw::math::VEC3 position[] = {
        nw::math::VEC3( left + width * 5.0f / 2.0f, bottom + height * 3.0f / 2.0f, 0.0f ),  // +X
        nw::math::VEC3( left + width * 1.0f / 2.0f, bottom + height * 3.0f / 2.0f, 0.0f ),  // -X
#if defined(NW_PLATFORM_WIN32)
        nw::math::VEC3( left + width * 3.0f / 2.0f, bottom + height * 5.0f / 2.0f, 0.0f ),  // +Y
        nw::math::VEC3( left + width * 3.0f / 2.0f, bottom + height * 1.0f / 2.0f, 0.0f ),  // -Y
#endif
#if defined(NW_PLATFORM_CAFE)
        nw::math::VEC3( left + width * 3.0f / 2.0f, bottom + height * 1.0f / 2.0f, 0.0f ),  // +Y
        nw::math::VEC3( left + width * 3.0f / 2.0f, bottom + height * 5.0f / 2.0f, 0.0f ),  // -Y
#endif
        nw::math::VEC3( left + width * 7.0f / 2.0f, bottom + height * 3.0f / 2.0f, 0.0f ),  // +Z
        nw::math::VEC3( left + width * 3.0f / 2.0f, bottom + height * 3.0f / 2.0f, 0.0f )   // -Z
    };

    // シェーダ切り替え
    mShader[SHADER_TYPE_COPY].Bind();

    // 頂点バッファ設定
    mPositionVB.BindBuffer( mPositionAttr[SHADER_TYPE_COPY], sizeof(nw::math::VEC3) * 4,  sizeof(nw::math::VEC3) );
    mTexCoordVB.BindBuffer( mTexCoordAttr[SHADER_TYPE_COPY], sizeof(nw::math::VEC2) * 4,  sizeof(nw::math::VEC2) );

    for ( u32 i = 0; i < 6; ++i )
    {
        // マトリクス計算
        nw::math::MTX44 wvpMatrix;

        nw::math::Vector3 scale( width, height, 1.0f );

        if ( i != 2 && i != 3 )
        {
            scale.x = -scale.x;
            scale.y = -scale.y;
        }

        nw::math::MTX44Translate(&wvpMatrix, position[i] );
        nw::math::MTX44MultScale( &wvpMatrix, wvpMatrix, scale );

#if defined(NW_PLATFORM_WIN32)
        // マトリクス設定
        glUniformMatrix4fv( mWVPOffset[SHADER_TYPE_COPY], 1, GL_FALSE, wvpMatrix.a );

        // テクスチャ設定
        _setupTexture( SHADER_TYPE_COPY, texture[i] );
#endif
#if defined(NW_PLATFORM_CAFE)
        // マトリクス設定
        GX2SetVertexUniformReg( mWVPOffset[SHADER_TYPE_COPY], 4 * 4, wvpMatrix.a );

        // テクスチャ設定
        _setupTexture( SHADER_TYPE_COPY, &texture[i] );
#endif

        // 描画
        nw::eft2::Render::DrawPrimitive( nw::eft2::Render::PRIM_TYPE_QUADS, 0, 4 );
    }

    _setupTexture( SHADER_TYPE_COPY, 0 );
}

//---------------------------------------------------------------------------
//! @brief        ガウスフィルタの重みを設定します。
//---------------------------------------------------------------------------
void ScreenDrawer::SetWeight( const f32* weight )
{
    for ( u32 i = 0; i < GAUSS_LENGTH; ++i )
    {
#if defined(NW_PLATFORM_WIN32)
        mWeightValue[i] = weight[i];
#endif
#if defined(NW_PLATFORM_CAFE)
        mWeightValue[i * 4] = weight[i];
#endif
    }
}

} // NOLINT(readability/fn_size) // namespace nw::eftdemo
} // namespace nw
