﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/types.h>

namespace nw      {
namespace eftdemo {

enum
{
    EFT_DEMO_NAME_STRING_MAX = 64
};


//------------------------------------------------------------------------------
// 簡易デバッグメニューアイテム
//------------------------------------------------------------------------------
class DebugMenuItem
{
public:
    // 1ページ8アイテムまで。
    enum { DEBUG_MENU_PARAM_MAX = 16 };

    struct Param
    {
        char    name[EFT_DEMO_NAME_STRING_MAX];
        u32     value;

        void Init()
        {
            name[0] = '\0';
            value = 0;
        }
    };

    DebugMenuItem()
    {
        mCurrent = 0;
        mParamName[0] = '\0';
        mParamNum = 0;
        mDstParam = NULL;
        for ( u32 i = 0; i < DEBUG_MENU_PARAM_MAX; i++ )
        {
            mParameter[i].Init();
        }
    }

    void SetItemName( const char* name )
    {
#if defined(NW_PLATFORM_WIN32)
        strcpy_s( mParamName, EFT_DEMO_NAME_STRING_MAX, name );
#else
        strcpy( mParamName, name );
#endif
    }

    void AddItemParam( const char* name, u32 value )
    {
#if defined(NW_PLATFORM_WIN32)
        strcpy_s(  mParameter[mParamNum].name, EFT_DEMO_NAME_STRING_MAX, name );
#else
        strcpy( mParameter[mParamNum].name, name );
#endif
        mParameter[mParamNum].value = value;
        mParamNum++;
    }

    virtual void Behavior( nw::dev::Pad* pad )
    {
        if ( pad->IsTrig( nw::dev::Pad::MASK_RIGHT ) )
        {
            if ( mParamNum > 0 && mCurrent < mParamNum - 1 )
            {
                mCurrent++;
            }
        }
        if ( pad->IsTrig( nw::dev::Pad::MASK_LEFT ) )
        {
            if ( 0 < mCurrent )
            {
                mCurrent--;
            }
        }

        if ( mDstParam )
        {
            *mDstParam = mParameter[mCurrent].value;
        }

        mLastInput = pad->GetTrigMask();
    }

    void SetCurrentIndex( u32 index )
    {
        mCurrent = index;
        if ( mDstParam )
        {
            *mDstParam = mParameter[mCurrent].value;
        }
    }

    void SetDstParamPtr( u32* dst )
    {
        mDstParam = dst;
    }

    void SetValue( u32 value )
    {
        for( u32 i = 0; i < mParamNum; i++ )
        {
            if ( mParameter[i].value == value )
            {
                mCurrent = i;
                return;
            }
        }

        mCurrent = 0;   //error
    }
    virtual u32 GetValue()
    {
        return mParameter[mCurrent].value;
    }

    virtual bool IsEnable()
    {
        if ( mParameter[mCurrent].value == 0 )
        {
            return false;
        }
        else
        {
            return true;
        }
    }

    u32 GetLastInput()
    {
        u32 temp = mLastInput;
        mLastInput = 0;
        return temp;
    }

    virtual void WriteMenuItem( nw::dev::DevTextWriter* writer, f32 posX, f32 posY )
    {
        writer->SetCursor( posX, posY );
        writer->Printf( "%-32s : %s ", mParamName, mParameter[mCurrent].name );
    }

protected:
    char        mParamName[EFT_DEMO_NAME_STRING_MAX];
    u32         mCurrent;
    u32         mParamNum;
    u32*        mDstParam;
    Param       mParameter[DEBUG_MENU_PARAM_MAX];
    u32         mLastInput;
};


//------------------------------------------------------------------------------
// 簡易デバッグメニューページ
//------------------------------------------------------------------------------
class DebugMenuPage
{
public:
    // 1ページ8アイテムまで。
    enum { DEBUG_MENU_ITEM_MAX = 16 };

    DebugMenuPage()
    {
        mCurrent        = 0;
        mPageName[0]    = '\0';
        mItemNum        = 0;
        for ( u32 i = 0; i < DEBUG_MENU_ITEM_MAX; i++ )
        {
            mItem[i] = NULL;
        }
    }

    void SetPageName( const char* name )
    {
#if defined(NW_PLATFORM_WIN32)
        strcpy_s( mPageName, EFT_DEMO_NAME_STRING_MAX, name );
#else
        strcpy( mPageName, name );
#endif
    }

    void AddMenuItem( DebugMenuItem* item )
    {
        mItem[mItemNum] = item;
        mItemNum++;
    }

    void Behavior( nw::dev::Pad* pad )
    {
        if ( pad->IsTrig( nw::dev::Pad::MASK_UP ) )
        {
            if ( mCurrent > 0 )
            {
                mCurrent--;
            }
        }
        if ( pad->IsTrig( nw::dev::Pad::MASK_DOWN ) )
        {
            if ( mCurrent < mItemNum - 1 )
            {
                mCurrent++;
            }
        }
        if ( pad->IsTrig( nw::dev::Pad::MASK_RIGHT ) || pad->IsTrig( nw::dev::Pad::MASK_LEFT  ) )
        {
            if ( mItem[mCurrent] )
            {
                mItem[mCurrent]->Behavior( pad );
            }
        }
    }

    void WritePage( nw::dev::DevTextWriter* writer, f32 posX, f32 posY )
    {
        f32 fontSize = 15.0f;

        writer->SetCursor( posX, posY );
        writer->Printf( "|-[%s]", mPageName ); posY += fontSize;

        for( u32 i = 0; i < mItemNum; i++ )
        {
            if ( mItem[i] )
            {
                writer->SetCursor( posX, posY );
                if ( mCurrent == i )
                {
                    writer->Printf( ">" );
                }
                mItem[i]->WriteMenuItem( writer, posX + 10, posY );
                posY += fontSize;
            }
        }
    }

    const char* GetPageName() const
    {
        return mPageName;
    }

private:
    char            mPageName[EFT_DEMO_NAME_STRING_MAX];
    u32             mCurrent;
    u32             mItemNum;
    DebugMenuItem*  mItem[DEBUG_MENU_ITEM_MAX];
};


//------------------------------------------------------------------------------
// 簡易デバッグメニュー
//------------------------------------------------------------------------------
struct DebugMenu
{
    // 8ページまで。
    enum
    {
        DEBUG_MENU_PAGE_MAX = 8
    };

    DebugMenu()
    {
        mCurrent        = 0;
        mMenuName[0]    = '\0';
        mPageNum        = 0;
        mIsWritePage    = false;
        for ( u32 i = 0; i < DEBUG_MENU_PAGE_MAX; i++ )
        {
            mPage[i] = NULL;
        }
    }

    void SetMenuName( const char* name )
    {
#if defined(NW_PLATFORM_WIN32)
        strcpy_s( mMenuName, EFT_DEMO_NAME_STRING_MAX, name );
#else
        strcpy( mMenuName, name );
#endif
    }

    void AddMenuPage( DebugMenuPage* page )
    {
        mPage[mPageNum] = page;
        mPageNum++;
    }


    void SetCurrentPage( const char* pageName )
    {
        for( u32 i = 0; i < mPageNum; i++ )
        {
            if ( mPage[i] && strcmp( mPage[i]->GetPageName(), pageName ) == 0 )
            {
                mCurrent = i;
            }
        }

        mIsWritePage = true;
    }

    const char* GetCurrentPage()
    {
        return mPage[mCurrent]->GetPageName();
    }

    void Behavior( nw::dev::Pad* pad )
    {
        if ( !mIsWritePage )
        {
            if ( pad->IsTrig( nw::dev::Pad::MASK_UP ) )
            {
                if ( mCurrent > 0 )
                {
                    mCurrent--;
                }
            }
            if ( pad->IsTrig( nw::dev::Pad::MASK_DOWN ) )
            {
                if ( mCurrent < mPageNum - 1 )
                {
                    mCurrent++;
                }
            }

            if ( pad->IsTrig( nw::dev::Pad::MASK_A ) )
            {
                mIsWritePage = true;
            }
        }
        else
        {
            if ( mPage[mCurrent] )
            {
                mPage[mCurrent]->Behavior( pad );
            }

            if ( pad->IsTrig( nw::dev::Pad::MASK_B ) )
            {
                mIsWritePage = false;
            }
        }
    }

    void WritePage( nw::dev::DevTextWriter* writer, f32 posX, f32 posY )
    {
        f32 fontSize = 15.0f;

        writer->SetCursor( posX, posY );
        writer->Printf( "[%s]", mMenuName ); posY += fontSize;

        if ( !mIsWritePage )
        {
            for( u32 i = 0; i < mPageNum; i++ )
            {
                if ( mPage[i] )
                {
                    writer->SetCursor( posX, posY );
                    if ( mCurrent == i )
                    {
                        writer->Printf( ">" );
                    }
                    writer->SetCursor( posX + 10, posY );
                    writer->Printf( "%-32s", mPage[i]->GetPageName() );
                    posY += fontSize;
                }
            }
        }
        else
        {
            if ( mPage[mCurrent] )
            {
                mPage[mCurrent]->WritePage( writer, posX + 10, posY );
            }
        }
    }

private:
    char                mMenuName[EFT_DEMO_NAME_STRING_MAX];
    u32                 mCurrent;
    u32                 mPageNum;
    bool                mIsWritePage;
    DebugMenuPage*      mPage[DEBUG_MENU_PAGE_MAX];
};


//------------------------------------------------------------------------------
// 簡易デバッグメニュー On/Offアイテム
//------------------------------------------------------------------------------
class DebugMenuItemOnOff : public DebugMenuItem
{
public:
    DebugMenuItemOnOff() : DebugMenuItem()
    {
        AddItemParam( "OFF", 0 );
        AddItemParam( "ON", 1 );
    }
};


//------------------------------------------------------------------------------
// 簡易デバッグメニュー uintカウンタアイテム
//------------------------------------------------------------------------------
class DebugMenuItemCounter : public DebugMenuItem
{
public:
    DebugMenuItemCounter() : DebugMenuItem()
    {
        mParam = 0;
        mMin   = 0;
        mMax   = 1000000;
    }

    void SetRange( int min, int max )
    {
        mMin    = min;
        mMax    = max;
    }

    void SetIntParam( int param )
    {
        mParam = param;
    }

    void AddCounter()
    {
        mParam++;
    }

    virtual void Behavior( nw::dev::Pad* pad )
    {
        if ( pad->IsTrig( nw::dev::Pad::MASK_RIGHT ) )
        {
            if ( mParam < mMax )
            {
                mParam++;
            }
        }
        if ( pad->IsTrig( nw::dev::Pad::MASK_LEFT ) )
        {
            if ( mParam > mMin )
            {
                mParam--;
            }
        }
        mLastInput = pad->GetTrigMask();
    }

    virtual void WriteMenuItem( nw::dev::DevTextWriter* writer, f32 posX, f32 posY )
    {
        writer->SetCursor( posX, posY );
        writer->Printf( "%-32s : %d ", mParamName, mParam );
    }

    virtual u32 GetValue()
    {
        return mParam;
    }

private:
    s32     mParam;
    s32     mMin;
    s32     mMax;
};


//------------------------------------------------------------------------------
// 簡易デバッグメニュー floatカウンタアイテム
//------------------------------------------------------------------------------
class DebugMenuItemFloatCounter : public DebugMenuItem
{
public:
    DebugMenuItemFloatCounter() : DebugMenuItem()
    {
        mParam = 10;
        mMin   = 0;
        mMax   = 100;
    }

    void SetRange( f32 min, f32 max )
    {
        mMin    = static_cast<u32>( min * 10 );
        mMax    = static_cast<u32>( max * 10 );
    }

    void SetFloatParam( f32 param )
    {
        mParam = static_cast<u32>( param * 10 );
    }

    virtual void Behavior( nw::dev::Pad* pad )
    {
        if ( pad->IsTrig( nw::dev::Pad::MASK_RIGHT ) )
        {
            if ( mParam < mMax )
            {
                mParam++;
            }
        }
        if ( pad->IsTrig( nw::dev::Pad::MASK_LEFT ) )
        {
            if ( mParam > mMin )
            {
                mParam--;
            }
        }
        mLastInput = pad->GetTrigMask();
    }

    virtual void WriteMenuItem( nw::dev::DevTextWriter* writer, f32 posX, f32 posY )
    {
        writer->SetCursor( posX, posY );
        writer->Printf( "%-32s : %f ", mParamName, static_cast<f32>( mParam ) / 10.f );
    }

    f32 GetFloatValue()
    {
        return static_cast<f32>( mParam ) / 10.f;
    }

private:
    u32     mParam;
    u32     mMin;
    u32     mMax;
};

//------------------------------------------------------------------------------
// 簡易デバッグメニュー 文字列表示アイテム
//------------------------------------------------------------------------------
class DebugMenuItemStringParam : public DebugMenuItem
{
public:
    DebugMenuItemStringParam() : nw::eftdemo::DebugMenuItem()
    {
        mParam[0] = '\0';
    }

    void SetStringParam( const char* param )
    {
#if defined(NW_PLATFORM_WIN32)
        strcpy_s( mParam, EFT_DEMO_NAME_STRING_MAX, param );
#else
        strcpy( mParam, param );
#endif
    }

    virtual void WriteMenuItem( nw::dev::DevTextWriter* writer, f32 posX, f32 posY )
    {
        writer->SetCursor( posX, posY );
        writer->Printf( "%-32s : %s ", mParamName, mParam );
    }

private:
    char         mParam[EFT_DEMO_NAME_STRING_MAX];
};

//------------------------------------------------------------------------------
// 簡易デバッグメニュー float表示アイテム
//------------------------------------------------------------------------------
class DebugMenuItemFloatParam : public DebugMenuItem
{
public:
    DebugMenuItemFloatParam() : nw::eftdemo::DebugMenuItem()
    {
        mParam = 0.0f;
    }

    void SetFloatParam( f32 param )
    {
        mParam = param;
    }

    virtual void WriteMenuItem( nw::dev::DevTextWriter* writer, f32 posX, f32 posY )
    {
        writer->SetCursor( posX, posY );
        writer->Printf( "%-32s : %f ", mParamName, mParam );
    }

private:
    f32         mParam;
};


} // namespace eftdemo
} // namespace nw
