﻿using System;
using System.Collections.Generic;
using System.Threading;
using System.Web.Script.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;

namespace DevMenuCommandTestNuiShell
{
    [TestClass]
    public class TestSdCard : TestBase
    {
        public const string MsgMounted = "Sdcard can be used.";
        public const string MsgDatabaseCorrupted = "Database of sdcard is corrupted. Please cleanup sdcard and reboot the system.";
        public const string MsgFileSystemCorrupted = "FileSystem of sdcard is corrupted. Please format sdcard on PC.";
        public const string MsgNeedsSystemUpdate = "Sdcard is formated with exFAT. Please update the sytem.";
        public const string MsgNoOwnerShip = "SdCard is initialized on the diffrent machine. Please cleanup sdcard and reboot the system.";
        public const string MsgNotInserted = "Sdcard is not inserted.";
        public const string MsgNotMounted = "Sdcard can be used, if the system reboots.";
        public const string MsgNotEnoughSpace = "There is not enough space in sdcard.";
        public const string MsgAccessFailed = "Sdcard cannot be accessed.";
        public const string MsgMountUnexpected = "Unexpected error occurs while mounting sdcard. result=0x\\d+";

        public override void DoCleanup()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            command.Execute(new string[] {
                "debug enable-mount-sdcard",
                "debug set-integer-fwdbg --name ns.sdcard --key compare_sdcard 0",
                "sdcard format",
            });
            command.ResetAndExecute(new string[] {
                "application uninstall --all",
            });

            // 確認
            CheckSdCardStatus(command, MsgMounted);
        }

        private void CheckSdCardStatusImpl(DevMenuCommandSystem command, string status, bool doReset = false)
        {
            if (doReset)
            {
                Assert.IsTrue(command.ResetAndExecute(new string[] {
                    "sdcard status",
                }, status));
            }
            else
            {
                Assert.IsTrue(command.Execute(new string[] {
                    "sdcard status",
                }, status));
            }
        }

        private void CheckSdCardStatus(DevMenuCommandSystem command, string status)
        {
            CheckSdCardStatusImpl(command, status, false);
        }

        private void ResetAndCheckSdCardStatus(DevMenuCommandSystem command, string status)
        {
            CheckSdCardStatusImpl(command, status, true);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestCleanupRedundantSdCard()
        {
            var appv0 = m_Maker.MakeApplication(0x0100394000059000, 0);
            var appv1 = m_Maker.MakeApplication(0x0100394000059000, 1);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "application install " + appv1.Path + " -s sdcard",
                "application install-redundant-entity " + appv0.Path + " -s sdcard",
                "application install-redundant-entity " + appv1.Path + " -s builtin",
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database -s builtin",
            }, new string[] {
                @"\""id\"": \""0x0100394000059000\"",",
                @"\""version\"": 65536,"
            }));
            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database -s sdcard",
            }, new string[] {
                @"\""id\"": \""0x0100394000059000\"",",
                @"\""version\"": 0,",
                @"\""version\"": 65536,"
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "application delete-redundant-entity",
                "application list-content-meta-database -s builtin",
            },
                @"\[\]"
            ));

            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database -s sdcard",
            }, new string[] {
                @"\""id\"": \""0x0100394000059000\"",",
                @"\""version\"": 65536,"
            }));
            Assert.IsFalse(command.Execute(new string[] {
                "application list-content-meta-database -s sdcard",
            }, new string[] {
                @"\""version\"": 0,"
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "application list-record 0x0100394000059000",
                "application verify 0x0100394000059000",
            }));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestDetachSdCardWhileInstall()
        {
            var app = m_Maker.MakeApplication(0x0100394000059000, 0, 64 * 1024 * 1024);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsFalse(command.Execute(new string[] {
                "application install " + app.Path + " -s sdcard --detach-sdcard-after 2000",
            }));

            // SD カード抜きでインストールに失敗しても次のコマンドが実行できる
            Assert.IsTrue(command.Execute(new string[] {
                "application list-view " + app.Id,
            }));

            // 再起動後にインストールが成功していない
            Assert.IsFalse(command.ResetAndExecute(new string[] {
                "application list-record " + app.Id,
            }));

            // 再起動後に再度 SD カードにインストールできる
            Assert.IsTrue(command.Execute(new string[] {
                "application install " + app.Path + " -s sdcard",
                "application verify " + app.Id,
                "application uninstall " + app.Id,
            }));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestCleanupSdCardAfterFactoryReset()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "factoryreset do",
            }));

            // 本体初期化 -> 再起動後は使えない
            ResetAndCheckSdCardStatus(command, MsgNoOwnerShip);

            Assert.IsTrue(command.Execute(new string[] {
                "filesystem list-all sdcard:/"
            }, new string[] {
                "Contents/",
                "registered/",
                "placehld/",
                "private",
                "Album/",
                "save/",
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "sdcard cleanup"    // クリーンアップできる
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "filesystem list-all sdcard:/"
            }, new string[] {
                "Album/",
            }));

            Assert.IsFalse(command.Execute(new string[] {
                "filesystem list-all sdcard:/"
            }, new string[] {
                "Contents/",
                "registered/",
                "placehld/",
                "private",
                "save/",
            }));

            ResetAndCheckSdCardStatus(command, MsgMounted);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestResultMountUnexpected()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            var app = m_Maker.MakeApplication(0x0100394000059000);

            Assert.IsTrue(command.Execute(new string[] {
                "sdcard detach",
                "filesystem delete-directory /Nintendo/Contents -s sdcard",
                "filesystem create-file /Nintendo/Contents -s sdcard --src " + app.Path
            }));

            ResetAndCheckSdCardStatus(command, MsgMountUnexpected);

            Assert.IsTrue(command.Execute(new string[] {
                "debug disable-mount-sdcard",
            }));

            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "sdcard insert",        // イベントの強制発行
            }));

            CheckSdCardStatus(command, MsgMountUnexpected);

            Assert.IsTrue(command.Execute(new string[] {
                "sdcard format",
                "debug enable-mount-sdcard",
            }));

            // フォーマット直後の再起動で使える状態
            ResetAndCheckSdCardStatus(command, MsgMounted);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestResultDatabaseCorrupted()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "sdcard detach",
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "filesystem delete-directory -s sdcard /Nintendo/Contents/registered"
            }));

            // 起動前から挿入されている場合
            ResetAndCheckSdCardStatus(command, MsgDatabaseCorrupted);

            Assert.IsTrue(command.Execute(new string[] {
                "debug disable-mount-sdcard",
            }));

            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "sdcard insert",        // イベントの強制発行
            }));

            // 起動後に挿入された場合
            CheckSdCardStatus(command, MsgDatabaseCorrupted);

            Assert.IsTrue(command.Execute(new string[] {
                "sdcard cleanup",
                "debug enable-mount-sdcard",
            }));

            // cleanup 直後の再起動で使える状態
            ResetAndCheckSdCardStatus(command, MsgMounted);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestAfterBoot()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "debug disable-mount-sdcard",
            }));

            ResetAndCheckSdCardStatus(command, MsgNotInserted);

            // 起動後に挿入をエミュレート
            Assert.IsTrue(command.Execute(new string[] {
                "sdcard insert",        // イベントの強制発行
            }));

            CheckSdCardStatus(command, MsgNotMounted);

            // 再起動後は利用できる
            Assert.IsTrue(command.Execute(new string[] {
                "debug enable-mount-sdcard",
            }));

            ResetAndCheckSdCardStatus(command, MsgMounted);

            // SD カードを抜去すると利用できない
            Assert.IsTrue(command.Execute(new string[] {
                "sdcard remove",        // イベントの強制発行
            }));

            CheckSdCardStatus(command, MsgNotInserted);

            // 挿入エミュレート
            Assert.IsTrue(command.Execute(new string[] {
                "sdcard insert",        // イベントの強制発行
            }));

            CheckSdCardStatus(command, MsgNotInserted);

            Assert.IsTrue(command.Execute(new string[] {
                "debug enable-mount-sdcard",
            }));

            ResetAndCheckSdCardStatus(command, MsgMounted);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestInsertAndRemoveRepeatedly()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            // Attach 状態で挿入イベントを複数回発行する
            // SD カードが使える Attach 状態だけど、CheckMountSdCardState は更新されてしまう
            // この振る舞いはコマンドの使い方が間違っているので、仕様通りとする
            for (int i = 0; i < 3; i++)
            {
                Assert.IsTrue(command.Execute(new string[] {
                    "sdcard insert",        // イベントの強制発行
                }));
                CheckSdCardStatus(command, MsgNotMounted);
            }

            // Attach 状態であれば Detach 出来る
            Assert.IsTrue(command.Execute(new string[] {
                "sdcard detach",        // イベントの強制発行
            }));
            CheckSdCardStatus(command, MsgNotInserted);

            // Detach 状態で挿抜しても変わらない
            for (int i = 0; i < 2; i++)
            {
                Assert.IsTrue(command.Execute(new string[] {
                    "sdcard insert",        // イベントの強制発行
                }));
                CheckSdCardStatus(command, MsgNotInserted);

                Assert.IsTrue(command.Execute(new string[] {
                    "sdcard remove",        // イベントの強制発行
                }));
                CheckSdCardStatus(command, MsgNotInserted);
            }

            Assert.IsTrue(command.Execute(new string[] {
                "debug disable-mount-sdcard",
            }));

            ResetAndCheckSdCardStatus(command, MsgNotInserted);

            // 繰り返し挿抜できる
            for (int i = 0; i < 3; i++)
            {
                Assert.IsTrue(command.Execute(new string[] {
                    "sdcard insert",        // イベントの強制発行
                }));
                CheckSdCardStatus(command, MsgNotMounted);

                Assert.IsTrue(command.Execute(new string[] {
                    "sdcard remove",        // イベントの強制発行
                }));
                CheckSdCardStatus(command, MsgNotInserted);
            }

            // 挿入を繰り返し出来る
            for (int i = 0; i < 3; i++)
            {
                Assert.IsTrue(command.Execute(new string[] {
                    "sdcard insert",        // イベントの強制発行
                }));
                CheckSdCardStatus(command, MsgNotMounted);
            }

            // 抜去を繰り返し出来る
            for (int i = 0; i < 3; i++)
            {
                Assert.IsTrue(command.Execute(new string[] {
                    "sdcard remove",        // イベントの強制発行
                }));
                CheckSdCardStatus(command, MsgNotInserted);
            }
        }

        private void MakeIncompleteSdCard(DevMenuCommandSystem command)
        {
            // プリイン状態を強制的に作り出す
            Assert.IsTrue(command.Execute(new string[] {
                "sdcard detach",
            }));
            Assert.IsTrue(command.Execute(new string[] {
                "filesystem delete-directory -s sdcard /Nintendo/save",
                "filesystem delete-directory -s sdcard /Nintendo/Contents/registered",
                "filesystem delete-directory -s sdcard /Nintendo/Contents/placehld",
            }));
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void TestIncompleteSdCard()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            MakeIncompleteSdCard(command);

            Assert.IsTrue(command.Execute(new string[] {
                "debug disable-mount-sdcard",
            }));

            // 起動後に挿入したときに、再起動したら使えると判断する
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "sdcard insert"
            }));
            CheckSdCardStatus(command, MsgNotMounted);
            Assert.IsTrue(command.Execute(new string[] {
                "debug enable-mount-sdcard",
            }));

            // 再起動したら利用できる
            ResetAndCheckSdCardStatus(command, MsgMounted);

            MakeIncompleteSdCard(command);

            // 起動時からプリイン状態でも再起動したら利用できる
            ResetAndCheckSdCardStatus(command, MsgMounted);

            // 他の本体で初期化されたプリイン状態の SD カードが正しくハンドリングできる
            MakeIncompleteSdCard(command);

            Assert.IsTrue(command.Execute(new string[] {
                "factoryreset do",
            }));

            ResetAndCheckSdCardStatus(command, MsgNoOwnerShip);

            Assert.IsTrue(command.Execute(new string[] {
                "sdcard cleanup"
            }));

            ResetAndCheckSdCardStatus(command, MsgMounted);

            // save ディレクトリがないだけの場合は、 DatabaseCorrupted
            Assert.IsTrue(command.Execute(new string[] {
                "sdcard detach",
            }));
            Assert.IsTrue(command.Execute(new string[] {
                "filesystem delete-directory -s sdcard /Nintendo/save",
            }));
            ResetAndCheckSdCardStatus(command, MsgDatabaseCorrupted);

            // registered だけが存在する場合も、 DatabaseCorrupted
            Assert.IsTrue(command.Execute(new string[] {
                "filesystem delete-directory -s sdcard /Nintendo/Contents/placehld",
            }));
            ResetAndCheckSdCardStatus(command, MsgDatabaseCorrupted);

            // registered だけが存在しない場合も、 DatabaseCorrupted
            Assert.IsTrue(command.Execute(new string[] {
                "sdcard cleanup"
            }));

            ResetAndCheckSdCardStatus(command, MsgMounted);
            Assert.IsTrue(command.Execute(new string[] {
                "filesystem delete-directory -s sdcard /Nintendo/Contents/registered",
            }));
            ResetAndCheckSdCardStatus(command, MsgDatabaseCorrupted);

            // DatabaseCorrupted に関する他の組み合わせは時間が掛かるので省略

            // 容量一杯のケース
            Assert.IsTrue(command.Execute(new string[] {
                "sdcard cleanup"
            }));
            ResetAndCheckSdCardStatus(command, MsgMounted);
            Assert.IsTrue(command.Execute(new string[] {
                "debug disable-mount-sdcard",
            }));
            ResetAndCheckSdCardStatus(command, MsgNotInserted);
            MakeIncompleteSdCard(command);
            Assert.IsTrue(command.Execute(new string[] {
                "debug fill-sdcard-free-space"
            }));

            // 容量一杯でも、正常なプリイン状態であれば NotMounted と判定される
            Assert.IsTrue(command.Execute(new string[] {
                "sdcard insert"
            }));
            CheckSdCardStatus(command, MsgNotMounted);
            Assert.IsTrue(command.Execute(new string[] {
                "debug enable-mount-sdcard",
            }));

            // プリイン状態の SD カードが容量一杯だった場合は容量不足エラー
            ResetAndCheckSdCardStatus(command, MsgNotEnoughSpace);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestCleanupWhenDifferentSdCardIsInserted()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var appv1 = m_Maker.MakeApplication(id, 1);

            CheckSdCardStatus(command, MsgMounted);

            Assert.IsTrue(command.Execute(new string[] {
                "application create-place-holder -s sdcard",
                "application list-place-holder -s sdcard"
            }));
            var placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
            Assert.IsTrue(placeHolderList.Count > 0);

            Assert.IsTrue(command.Execute(new string[] {
                "application install " + appv1.Path + " -s sdcard",
                "application install-redundant-entity " + appv0.Path + " -s sdcard",
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database -s sdcard",
            }, new string[] {
                @"\""id\"": \""0x0100394000059000\"",",
                @"\""version\"": 0,",
                @"\""version\"": 65536,"
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "debug set-integer-fwdbg --name ns.sdcard --key compare_sdcard 2", // 違うカードとする
            }));

            Reboot(command);

            Assert.IsTrue(command.Execute(new string[] {
                "application list-place-holder -s sdcard"
            }));
            placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
            Assert.IsTrue(placeHolderList.Count == 0);

            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database -s sdcard",
            }, new string[] {
                @"\""id\"": \""0x0100394000059000\"",",
                @"\""version\"": 65536,"
            }));
            Assert.IsFalse(command.Execute(new string[] {
                "application list-content-meta-database -s sdcard",
            }, new string[] {
                @"\""version\"": 0,"
            }));
        }
    }
}
