﻿using System;
using System.Threading;
using System.Linq;
using System.Collections.Generic;
using System.Web.Script.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;


namespace DevMenuCommandTestNuiShell
{
    [TestClass]
    public class TestSaveData : TestBase
    {
        public override void DoCleanup()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            command.Execute(new string[] {
                "debug enable-mount-sdcard",
                "sdcard format",
                "debug disable-cleanup-application-cache-storage"
            });
            command.ResetAndExecute(new string[] {
                "application uninstall --all",
                "savedata delete all",
                "application reset-required-version --all",
            });
        }

        private StorageSize GetStorageSize(DevMenuCommandSystem command, string storage = "sdcard")
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("application storage-size {0}", storage)
            }));
            return StorageSize.Deserialize(command.LastOutput);
        }

        private IEnumerable<SaveDataInfoForJson> GetSaveDataInfoList(DevMenuCommandSystem command, ulong id)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                "savedata list-detail"
            }));
            var saveDataList = SaveDataInfoForJson.Deserialize(command.LastOutput);
            return saveDataList.Where(x => !string.IsNullOrEmpty(x.applicationId) && (x.applicationId == string.Format("0x{0:x16}", id)));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestCleanupCacheStorage()
        {

            //
            // 強制電源断でクリーンナップが掛かるかどうかを確認
            //

            var command = new DevMenuCommandSystem(this.TestContext);
            Assert.IsTrue(command.Execute(new string[] {
                "debug enable-cleanup-application-cache-storage"
            }));
            Reboot(command);

            ulong id = 0x0100394000059000;
            string idString = string.Format("0x{0:X16}", id);
            // TODO: bcat storage と cache storage を指定する
            var app = m_Maker.MakeApplication(id, additionalOptions: "--cache-storage-size 0xC000 --cache-storage-journal-size 0xC000");

            Assert.IsTrue(command.Execute(new string[]
            {
                "application install " + app.Path
            }));

            // 起動しないとセーブデータは作成されない
            var appSaveDataInfoList = GetSaveDataInfoList(command, id);
            Assert.IsTrue(appSaveDataInfoList == null || appSaveDataInfoList.Count() == 0);

            var initSize = GetStorageSize(command);

            // 起動することでセーブデータを作成する
            SuccessLaunchApplication(command, id);

            appSaveDataInfoList = GetSaveDataInfoList(command, id);
            Assert.AreEqual(1, appSaveDataInfoList.Count());
            Assert.IsTrue(appSaveDataInfoList.Any(x => x.type == "Cache"));
            Assert.IsTrue(initSize.free > GetStorageSize(command).free);

            // SD カードを抜いてからアプリを削除する
            Assert.IsTrue(command.Execute(new string[]
            {
                "sdcard remove",
                string.Format("application uninstall {0}", idString)
            }));

            // 再起動して、クリーンナップされるか確認する
            // サイズを合わせるため再インストール
            Assert.IsTrue(command.ResetAndExecute(new string[]
            {
                "application install " + app.Path
            }));

            appSaveDataInfoList = GetSaveDataInfoList(command, id);
            Assert.IsTrue(appSaveDataInfoList == null || appSaveDataInfoList.Count() == 0);
            Assert.AreEqual(initSize.free, GetStorageSize(command).free);

            //
            // SD カードの抜き差しでクリーンナップが掛かるかどうかを確認
            //

            SuccessLaunchApplication(command, id);

            appSaveDataInfoList = GetSaveDataInfoList(command, id);
            Assert.AreEqual(1, appSaveDataInfoList.Count());
            Assert.IsTrue(appSaveDataInfoList.Any(x => x.type == "Cache"));
            Assert.IsTrue(initSize.free > GetStorageSize(command).free);

            command.Execute(new string[]
            {
                "debug disable-mount-sdcard",
            });

            Reboot(command);

            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("application uninstall {0}", idString),
                "debug enable-mount-sdcard",
            }));

            // 再起動して、クリーンナップされるか確認する
            Reboot(command);

            // サイズを合わせるため再インストール
            Assert.IsTrue(command.Execute(new string[]
            {
                "application install " + app.Path
            }));

            appSaveDataInfoList = GetSaveDataInfoList(command, id);
            Assert.IsTrue(appSaveDataInfoList == null || appSaveDataInfoList.Count() == 0);
            Assert.AreEqual(initSize.free, GetStorageSize(command).free);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestCleanupCacheStorageOnFactoryResetWithoutUserSaveData()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "debug enable-cleanup-application-cache-storage"
            }));
            Reboot(command);

            ulong id = 0x0100394000059000;
            var app = m_Maker.MakeApplication(id, additionalOptions: "--cache-storage-size 0xC000 --cache-storage-journal-size 0xC000");

            // NAND にキャッシュストレージを作成する

            Assert.IsTrue(command.Execute(new string[]
            {
                "sdcard remove",
                "application install " + app.Path
            }));

            var nandInitSize = GetStorageSize(command, "builtin");
            SuccessLaunchApplication(command, id);

            var nandSaveDataInfoList = GetSaveDataInfoList(command, id);
            Assert.AreEqual(1, nandSaveDataInfoList.Count());
            Assert.IsTrue(nandSaveDataInfoList.Any(x => x.type == "Cache"));
            Assert.IsTrue(nandInitSize.free > GetStorageSize(command, "builtin").free);

            // ユーザーのセーブデータを残して、本体初期化
            Assert.IsTrue(command.Execute(new string[] {
                "debug enable-force-maintenance-mode",
            }));

            Reboot(command);

            Assert.IsTrue(command.Execute(new string[] {
                "factoryreset do-without-usersavedata",
                "debug disable-force-maintenance-mode",
            }));

            Reboot(command);

            nandSaveDataInfoList = GetSaveDataInfoList(command, id);
            Assert.AreEqual(0, nandSaveDataInfoList.Count());
        }
    }
}
