﻿using System;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;

using ContentInfo = DevMenuCommandTest.SystemProgramList;
using BlackListRecord = DevMenuCommandTest.MakeNsData.BlackListRecord;

namespace DevMenuCommandTestNuiShell
{
    [TestClass]
    public class TestApplicationBlackList : TestBase
    {
        public static ContentInfo BlackListInfo { get; set; } = null;

        public override void DoCleanup()
        {
            var command = new DevMenuCommandSystem(this.TestContext);
            var blacklist = MakeBlackListNsp(command, "blacklist0");

            command.Execute(new string[] {
                $"debug set-blacklist 0 --version 0",
                $"systemprogram install {blacklist.Path}",
            });
            command.ResetAndExecute(new string[] {
                $"application uninstall --all",
                $"application reset-required-version --all",
            });
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestBlackListForDebug()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            // make contents
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path);

            // blacklist: none
            Assert.IsTrue(command.Execute(new string[]
            {
                $"application install {appv0.Path}"
            }));
            SuccessLaunchApplication(command, id);

            // blacklist: 0x0100394000059000,0
            Assert.IsTrue(command.Execute(new string[] {
                $"debug set-blacklist 0x{id:x16} --version 0",
            }));
            Reboot(command);
            FailureLaunchApplication(command, id, result: 0x0004de10);

            Assert.IsTrue(command.Execute(new string[]
            {
                $"application install {patchv1.Path}"
            }));
            SuccessLaunchApplication(command, id);

            // blacklist: 0x0100394000059000,65536
            Assert.IsTrue(command.Execute(new string[] {
                $"debug set-blacklist 0x{id:x16} --version 65536",
            }));
            Reboot(command);
            FailureLaunchApplication(command, id, result: 0x0004de10);

            // blacklist: none
            command.Execute(new string[] {
                $"debug set-blacklist 0 --version 0",
            });
            Reboot(command);
            SuccessLaunchApplication(command, id);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestBlackListSystemData()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            // make contents
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path);
            var patchv2 = m_Maker.MakePatch(id, 2, appv0.Path);

            // make blacklist
            var blacklist = MakeBlackListNsp(
                command, "blacklist1",
                new BlackListRecord[] { new BlackListRecord(id, 65536) });

            // blacklist: none
            Assert.IsTrue(command.Execute(new string[]
            {
                $"application install {appv0.Path}"
            }));
            SuccessLaunchApplication(command, id);

            // blacklist: 0x0100394000059000,65536 (before notification)
            Assert.IsTrue(command.Execute(new string[]
            {
                $"systemprogram install {blacklist.Path}"
            }));
            SuccessLaunchApplication(command, id);

            // blacklist: 0x0100394000059000,65536 (after notification)
            Assert.IsTrue(command.Execute(new string[]
            {
                $"systemupdate notify-systemdata-update"
            }));
            FailureLaunchApplication(command, id, result: 0x0004de10);

            Assert.IsTrue(command.Execute(new string[]
            {
                $"application install {patchv1.Path}"
            }));
            FailureLaunchApplication(command, id, result: 0x0004de10);

            Assert.IsTrue(command.Execute(new string[]
            {
                $"application install {patchv2.Path}"
            }));
            SuccessLaunchApplication(command, id);
        }

        [TestMethod]
        [Timeout(960 * 1000)]
        public void TestBlackListSystemDataMax()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            const int BlackListCountMax = 32;
            const ulong ApplicationIdFirst = 0x0100394000059000;
            const ulong ApplicationIdLast = ApplicationIdFirst + BlackListCountMax;

            // make blacklist
            {
                var list = new List<BlackListRecord>();
                for (var id = ApplicationIdFirst; id < ApplicationIdLast; id++)
                {
                    list.Add(new BlackListRecord(id, 0));
                }

                var blacklist = MakeBlackListNsp(command, $"blacklist{BlackListCountMax}", list);
                Assert.IsTrue(command.Execute(new string[]
                {
                    $"systemprogram install {blacklist.Path}",
                    $"systemupdate notify-systemdata-update"
                }));
            }

            // blacklist: 0x0100394000059000 ～ 0x010039400005901f,0 (after notification)
            for (var id = ApplicationIdFirst; id < ApplicationIdLast; id++)
            {
                var appv0 = m_Maker.MakeApplication(id, 0);

                Assert.IsTrue(command.Execute(new string[]
                {
                    $"application install {appv0.Path}"
                }));
                FailureLaunchApplication(command, id, result: 0x0004de10);
            }
            {
                var id = ApplicationIdLast;
                var appv0 = m_Maker.MakeApplication(id, 0);

                Assert.IsTrue(command.Execute(new string[]
                {
                    $"application install {appv0.Path}"
                }));
                SuccessLaunchApplication(command, id);
            }
        }

        private ContentInfo GetContentInfo(DevMenuCommandSystem command, string name)
        {
            Assert.IsTrue(command.Execute(new string[] {
                "systemprogram list"
            }));
            return SystemProgramList.Deserialize(command.LastOutput).Find(x => x.name == name);
        }

        private NspInfo MakeBlackListNsp(DevMenuCommandSystem command, string name, IEnumerable<BlackListRecord> list = null)
        {
            if (BlackListInfo == null)
            {
                var info = GetContentInfo(command, "ApplicationBlackList");
                if (string.IsNullOrEmpty(info.id))
                {
                    throw new Exception("Not found ApplicationBlackList.");
                }
                BlackListInfo = info;
            }

            var output = m_Maker.GetOutputDirectory().Replace("\"", "");
            var nsdata = new MakeNsData(this.TestContext);
            return nsdata.MakeBlackListNsp(output, name, BlackListInfo.id, BlackListInfo.version, list);
        }
    }
}
