﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Web.Script.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;


namespace DevMenuCommandTest
{
    /*
     * システム更新が伴うテストを追加するためのクラス
     */
    [TestClass]
    public class SystemTest : TestBase
    {
        private const ulong TestAppId = 0x0100ccccaaaa0000;
        private const string TestAppIdString = "0x0100ccccaaaa0000";
        private const ulong TestSystemUpdateId = 0x0100ccccbbbb0000;
        private const string TestSystemUpdateIdString = "0x0100ccccbbbb0000";
        private const ulong TestSystemDataId = 0x0100cccccccc0000;
        private const string TestSystemDataIdString = "0x0100cccccccc0000";
        private const ulong TestSystemDataWithExFatFlagId = 0x0100cccccccd0000;
        private const string TestSystemDataWithExFatFlagIdString = "0x0100cccccccd0000";

        private const int MaxSystemVersion = 2;
        private const int IncludingExFatVersion = MaxSystemVersion;

        private static List<NspInfo> AppList;
        private static List<NspInfo> SystemDataList;
        private static List<NspInfo> SystemUpdateList;
        private static NspInfo SystemDataWithExFatFlag;

        [ClassInitialize]
        public static void ClassInitialize(TestContext context)
        {
            SetupTestEnvironment(context);

            AppList = new List<NspInfo>();
            SystemDataList = new List<NspInfo>();
            SystemUpdateList = new List<NspInfo>();

            AppList.Add(AppMaker.MakeApplication(TestAppId, additionalOptions: "--small-code"));
            AppList.Add(AppMaker.MakePatch(TestAppId, 1, AppList[0].Path, requiredSystemVersion: 1, additionalOptions: "--small-code"));

            for (int i = 0; i < MaxSystemVersion; i++)
            {
                SystemDataList.Add(AppMaker.MakeSystemData(TestSystemDataId, version: i));
                SystemUpdateList.Add(SysMaker.MakeTestSystemUpdateMeta(TestSystemUpdateId, (UInt32)i, new MakeTestSystemContent.ContentMetaInfo[]
                {
                    new MakeTestSystemContent.ContentMetaInfo(SystemDataList[i], (UInt32)i << 16),
                }));
            }

            var systemProgramList = GetSystemProgramList(HostCommand);
            var exFatBootImage = systemProgramList.Single(x => x.id.ToLower() == ExFatBootImageIdString);

            SystemDataWithExFatFlag = AppMaker.MakeSystemData(TestSystemDataWithExFatFlagId);
            SystemDataList.Add(AppMaker.MakeSystemData(TestSystemDataId, version: IncludingExFatVersion));
            SystemUpdateList.Add(SysMaker.MakeTestSystemUpdateMeta(TestSystemUpdateId, (UInt32)IncludingExFatVersion, new MakeTestSystemContent.ContentMetaInfo[]
            {
                new MakeTestSystemContent.ContentMetaInfo(SystemDataList[IncludingExFatVersion], (UInt32)IncludingExFatVersion << 16),
                new MakeTestSystemContent.ContentMetaInfo(ExFatBootImageId, exFatBootImage.type, exFatBootImage.version, attribute: 1),
                new MakeTestSystemContent.ContentMetaInfo(SystemDataWithExFatFlag, 0, attribute: 1),
            }));

            CleanupImpl(context);
        }

        public static void CleanupImpl(TestContext context)
        {
            var cleanupCommandList = new string[]
            {
                "localcontentshare set-system-update-id 0x0",
                string.Format("systemprogram uninstall {0}", TestSystemUpdateIdString),
                string.Format("systemprogram uninstall {0}", TestSystemDataIdString),
                string.Format("systemprogram uninstall {0}", TestSystemDataWithExFatFlagIdString),
            };

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                command.Execute(cleanupCommandList, ignoreError: true);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                command.Execute(cleanupCommandList, ignoreError: true);
            });

            hostTask.Wait();
            clientTask.Wait();

            CleanupBase(context);
        }

        [TestCleanup]
        public void TestCleanup()
        {
            CleanupImpl(this.TestContext);
        }

        private void SwitchToTestSystemImpl(DevMenuCommandSystem command, NspInfo systemUpdate, IEnumerable<NspInfo> systemDataList)
        {
            // 前の環境をリセット
            command.Execute(new string[]
            {
                "localcontentshare set-system-update-id 0x0",
                "systemprogram uninstall " + TestSystemUpdateIdString,
                "systemprogram uninstall " + TestSystemDataIdString,
                "systemprogram uninstall " + TestSystemDataWithExFatFlagIdString,
            }, ignoreError: true);

            var args = new List<string>();
            args.Add(string.Format("localcontentshare set-system-update-id {0}", TestSystemUpdateIdString));
            args.Add("systemprogram install " + systemUpdate.Path);
            args.AddRange(systemDataList.Select(x => "systemprogram install " + x.Path));
            Assert.IsTrue(command.Execute(args));

            ForceReset(command);
        }

        private void SwitchToTestSystem(DevMenuCommandSystem command, int systemVersion)
        {
            var systemUpdate = SystemUpdateList[systemVersion];
            var systemDataList = new List<NspInfo>();
            systemDataList.Add(SystemDataList[systemVersion]);
            if (systemVersion == IncludingExFatVersion)
            {
                systemDataList.Add(SystemDataWithExFatFlag);
            }

            SwitchToTestSystemImpl(command, systemUpdate, systemDataList);
        }

        private void InstallTestApplication(DevMenuCommandSystem command, int version, string appStorage = null, string patchStorage = null)
        {
            InstallTestApplicationImpl(command, version, AppList, appStorage, patchStorage);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void SendSystemTest()
        {
            var barrier = new Barrier(2);
            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                SwitchToTestSystem(command, IncludingExFatVersion);
                InstallTestApplication(command, 1);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application 0x{0:X16} --timeout 300", TestAppId)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }
            });

            var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
            if (File.Exists(resumeFile))
            {
                File.Delete(resumeFile);
            }
            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                SwitchToTestSystem(command, 0);
                InstallTestApplication(command, 0);

                Assert.IsTrue(!File.Exists(resumeFile));

                barrier.SignalAndWait();

                var systemProgramList = GetSystemProgramList(command);
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == 0));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == 0));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == ExFatBootImageIdString));
                Assert.IsFalse(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataWithExFatFlagIdString));


                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 300", TestAppIdString, resumeFile)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Suspened, output.state);
                    Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, output.reason);
                    Assert.IsTrue(!string.IsNullOrEmpty(output.context));
                    Assert.IsTrue(File.Exists(resumeFile));
                }

                systemProgramList = GetSystemProgramList(command);
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == IncludingExFatVersion));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == IncludingExFatVersion << 16));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == ExFatBootImageIdString));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataWithExFatFlagIdString));

                Assert.IsTrue(command.ResetAndExecute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --resume \"{1}\" --timeout 300", TestAppIdString, resumeFile)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(1u << 16, view[0].version);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void SendSystemFromHasExFatToNoExFat()
        {
            using (var recover = new ExFatRecovery(ClientCommand, TestContext))
            {
                var barrier = new Barrier(2);
                var hostTask = Task.Run(() =>
                {
                    var command = HostCommand;

                    SwitchToTestSystem(command, IncludingExFatVersion);
                    InstallTestApplication(command, 1);

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }
                });

                var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
                if (File.Exists(resumeFile))
                {
                    File.Delete(resumeFile);
                }
                var clientTask = Task.Run(() =>
                {
                    var command = ClientCommand;

                    UninstallExFat(command, false);
                    SwitchToTestSystem(command, 0);

                    var systemProgramList = GetSystemProgramList(command);
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == 0));
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == 0));
                    Assert.IsFalse(systemProgramList.Any(x => x.id.ToLower() == ExFatBootImageIdString));
                    Assert.IsFalse(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataWithExFatFlagIdString));

                    InstallTestApplication(command, 0);
                    Assert.IsTrue(!File.Exists(resumeFile));

                    barrier.SignalAndWait();
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 300", TestAppIdString, resumeFile)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Suspened, output.state);
                        Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, output.reason);
                        Assert.IsTrue(!string.IsNullOrEmpty(output.context));
                        Assert.IsTrue(File.Exists(resumeFile));
                    }

                    systemProgramList = GetSystemProgramList(command);
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == IncludingExFatVersion));
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == IncludingExFatVersion << 16));
                    Assert.IsFalse(systemProgramList.Any(x => x.id.ToLower() == ExFatBootImageIdString));
                    Assert.IsFalse(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataWithExFatFlagIdString));

                    Assert.IsTrue(command.ResetAndExecute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --resume \"{1}\" --timeout 300", TestAppIdString, resumeFile)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-view"
                    }));
                    var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual(1, view.Count);
                    Assert.AreEqual(1u << 16, view[0].version);
                });

                hostTask.Wait();
                clientTask.Wait();
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void SendSystemFromNoExFatToHasExFat()
        {
            using (var recover = new ExFatRecovery(HostCommand, TestContext))
            {
                var barrier = new Barrier(2);
                var hostTask = Task.Run(() =>
                {
                    var command = HostCommand;
                    UninstallExFat(command, false);
                    SwitchToTestSystem(command, 1);

                    InstallTestApplication(command, 1);

                    barrier.SignalAndWait();
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }
                });

                var clientTask = Task.Run(() =>
                {
                    var command = ClientCommand;
                    SwitchToTestSystem(command, 0);
                    InstallTestApplication(command, 0);

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, output.state);
                        Assert.AreEqual((UInt32)LcsFailureReason.CannotUpdateSystem, output.reason);
                    }
                });

                hostTask.Wait();
                clientTask.Wait();
            }
        }

        [TestMethod]
        [Timeout(300 * 1000) ]
        // RequiredSystemVersion は満たしているパッチを送るけど、本体更新が走るのを確認する
        public void TestApplicationDeliveryProtocolVersion()
        {
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-application-delivery-protocol-version 2",
                    "localcontentshare set-acceptable-application-delivery-protocol-version 2",
                }));
                SwitchToTestSystem(command, 2);
                InstallTestApplication(command, 1);

                barrier.SignalAndWait();
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }

                // 下位互換範囲内であれば、本体更新は発生しない
                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-acceptable-application-delivery-protocol-version 1",
                }));
                ForceReset(command);

                barrier.SignalAndWait();
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }
            });

            var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
            if (File.Exists(resumeFile))
            {
                File.Delete(resumeFile);
            }
            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-application-delivery-protocol-version 1",
                    "localcontentshare set-acceptable-application-delivery-protocol-version 1",
                }));
                SwitchToTestSystem(command, 1);
                InstallTestApplication(command, 0);

                var systemProgramList = GetSystemProgramList(command);
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == 1));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == 1 << 16));

                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare get-system-delivery-info"
                }));
                {
                    var systemDeliveryInfo = SystemDeliveryInfoForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual(1u, systemDeliveryInfo.applicationDeliveryProtocolVersion);
                    Assert.AreEqual(1u, systemDeliveryInfo.systemDeliveryProtocolVersion);
                }

                Assert.IsTrue(!File.Exists(resumeFile));
                barrier.SignalAndWait();
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 300", TestAppIdString, resumeFile)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Suspened, output.state);
                    Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, output.reason);
                    Assert.IsTrue(!string.IsNullOrEmpty(output.context));
                    Assert.IsTrue(File.Exists(resumeFile));
                }

                systemProgramList = GetSystemProgramList(command);
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == 2));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == 2 << 16));

                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-application-delivery-protocol-version 2",
                    "localcontentshare set-acceptable-application-delivery-protocol-version 2",
                }));

                ForceReset(command);

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --resume \"{1}\" --timeout 300", TestAppId, resumeFile)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(1u << 16, view[0].version);

                // 下位互換範囲内であれば、本体更新は発生しない
                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-application-delivery-protocol-version 1",
                    "localcontentshare set-acceptable-application-delivery-protocol-version 1",
                }));
                SwitchToTestSystem(command, 1);

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("application delete-entity {0} --patch", TestAppIdString)
                }));

                barrier.SignalAndWait();
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 300", TestAppIdString),
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(1u << 16, view[0].version);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        // RequiredSystemVersion は満たしているパッチを送るけど、本体更新が出来ずに失敗する
        public void CannotUpdateSystemForTestApplicationDeliveryProtocolVersion()
        {
            var barrier = new Barrier(2);

            using (var recover = new ExFatRecovery(HostCommand, TestContext))
            {
                var hostTask = Task.Run(() =>
                {
                    var command = HostCommand;

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "localcontentshare set-application-delivery-protocol-version 2",
                        "localcontentshare set-acceptable-application-delivery-protocol-version 2",
                    }));
                    UninstallExFat(command, false);
                    SwitchToTestSystem(command, 2);
                    InstallTestApplication(command, 1);

                    barrier.SignalAndWait();
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }
                });

                var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
                if (File.Exists(resumeFile))
                {
                    File.Delete(resumeFile);
                }
                var clientTask = Task.Run(() =>
                {
                    var command = ClientCommand;

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "localcontentshare set-application-delivery-protocol-version 1",
                        "localcontentshare set-acceptable-application-delivery-protocol-version 1",
                    }));
                    SwitchToTestSystem(command, 1);
                    InstallTestApplication(command, 0);

                    var systemProgramList = GetSystemProgramList(command);
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == 1));
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == 1 << 16));

                    Assert.IsTrue(!File.Exists(resumeFile));
                    barrier.SignalAndWait();
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 300", TestAppIdString, resumeFile)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, output.state);
                        Assert.AreEqual((UInt32)LcsFailureReason.CannotUpdateSystem, output.reason);
                    }
                });

                hostTask.Wait();
                clientTask.Wait();
            }
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        // SystemDeliveryProtocolVersion の場合は、セッションにすら参加できない
        public void TestSystemDeliveryProtocolVersion()
        {
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                // 下位互換がない
                var command = HostCommand;
                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-system-delivery-protocol-version 2",
                }));
                SwitchToTestSystem(command, 1);
                InstallTestApplication(command, 1);

                barrier.SignalAndWait();
                Assert.IsTrue(command.FailureExecute(
                    string.Format("localcontentshare send-application {0} --timeout 10", TestAppIdString),
                    "Timeout"
                ));

                // 上位互換がない
                barrier.SignalAndWait();
                Assert.IsTrue(command.FailureExecute(
                    string.Format("localcontentshare send-application {0} --timeout 20", TestAppIdString),
                    "Timeout"
                ));
            });

            var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
            if (File.Exists(resumeFile))
            {
                File.Delete(resumeFile);
            }
            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-system-delivery-protocol-version 1",
                }));
                SwitchToTestSystem(command, 0);
                InstallTestApplication(command, 0);
                Assert.IsTrue(!File.Exists(resumeFile));

                // 下位互換がない
                barrier.SignalAndWait();

                Assert.IsTrue(command.FailureExecute(
                    string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 10", TestAppIdString, resumeFile),
                    "Unexpected error. result = 0x0001d2d3" // lcs::ResultLowerSystemVersion
                ));

                // 上位互換がない
                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-system-delivery-protocol-version 3",
                }));
                ForceReset(command);


                barrier.SignalAndWait();

                Assert.IsTrue(command.FailureExecute(
                    string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 10", TestAppIdString, resumeFile),
                    "Unexpected error. result = 0x0001d4d3" // lcs::ResultHigherSystemVersion
                ));
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void CleanupPlaceHolderToSendSystem()
        {
            var barrier = new Barrier(2);

            uint systemVersion = MaxSystemVersion + 1; // 使っていないバージョンを使う

            // 転送時間を稼ぐため大き目の SystemData を参照する SystemUpdate を作る
            var bigSystemData = AppMaker.MakeSystemData(TestSystemDataId, size: 0x5000000, version: (int)systemVersion);

            var bigSystemUpdate = SysMaker.MakeTestSystemUpdateMeta(TestSystemUpdateId, systemVersion, new MakeTestSystemContent.ContentMetaInfo[]
            {
                new MakeTestSystemContent.ContentMetaInfo(bigSystemData, systemVersion << 16),
            });

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare set-system-update-id {0}", TestSystemUpdateIdString),
                    "systemprogram install " + bigSystemData.Path,
                    "systemprogram install " + bigSystemUpdate.Path
                }));
                ForceReset(command);
                InstallTestApplication(command, 1);

                barrier.SignalAndWait();

                // クライアントは再起動したあと戻ってこないので、適当にタイムアウトさせる
                command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 30", TestAppIdString)
                });
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                SwitchToTestSystem(command, 0);
                InstallTestApplication(command, 0);

                //
                // BuiltInSystem の PlaceHolder がクリーンナップされる
                //
                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-place-holder -s system"
                }));
                var placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
                Assert.AreEqual(0, placeHolderList.Count);

                DevMenuCommandBase.RunOnTargetOptions options = new DevMenuCommandBase.RunOnTargetOptions();
                options.NoWait = true;
                command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 30", TestAppIdString)
                }, options: options);

                // 先に受信操作を行ってから、送信を行わせる
                barrier.SignalAndWait();

                bool hasPlaceHolder = false;
                while (!hasPlaceHolder)
                {
                    options.NoWait = false;
                    options.SuppressAutoKill = true;
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-place-holder -s system"
                    }, options: options));
                    placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
                    hasPlaceHolder = placeHolderList.Count > 0;
                }

                // コマンドの途中で強制電源断
                ForceReset(command);

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-place-holder -s system"
                }));
                placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
                Assert.AreEqual(0, placeHolderList.Count);

                var systemProgramList = GetSystemProgramList(command);
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == 0));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == 0));
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void TestRequiredSystemVersionToDeliver()
        {
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-required-system-version 2",
                }));
                SwitchToTestSystem(command, 2);
                InstallTestApplication(command, 1);

                barrier.SignalAndWait();
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString),
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                // 送信者がローカルコンテンツ配信必須システムバージョンを満たしていない場合、先に本体更新が入る

                var resumeFileForHost = Path.Combine(SysMaker.OutputDirPath, "resumeForHost.txt");
                if (File.Exists(resumeFileForHost))
                {
                    File.Delete(resumeFileForHost);
                }

                SwitchToTestSystem(command, 1);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application {0} --output-resume-file \"{1}\" --timeout 300", TestAppIdString, resumeFileForHost),
                }));
                outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Suspened, outputs.state);
                Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, outputs.reason);

                var systemProgramList = GetSystemProgramList(command);
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == 2));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == 2 << 16));

                Assert.IsTrue(command.ResetAndExecute(new string[]
                {
                    string.Format("localcontentshare send-application {0} --resume \"{1}\" --timeout 300", TestAppIdString, resumeFileForHost),
                }));
                outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-required-system-version 2",
                }));
                // RequiredSystemVersion を満たすために、ver.1 のシステムを入れる
                SwitchToTestSystem(command, 1);
                InstallTestApplication(command, 0);

                var resumeFileForClient = Path.Combine(SysMaker.OutputDirPath, "resumeForClient.txt");
                if (File.Exists(resumeFileForClient))
                {
                    File.Delete(resumeFileForClient);
                }

                barrier.SignalAndWait();

                // RequiredSystemVersion は満たしているけど、
                // ローカルコンテンツ配信必須システムバージョンを満たしていない場合に本体更新が掛かるかをテストする
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 300", TestAppIdString, resumeFileForClient),
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Suspened, output.state);
                    Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, output.reason);
                    Assert.IsTrue(!string.IsNullOrEmpty(output.context));
                    Assert.IsTrue(File.Exists(resumeFileForClient));
                }

                var systemProgramList = GetSystemProgramList(command);
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == 2));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == 2 << 16));

                Assert.IsTrue(command.ResetAndExecute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --resume \"{1}\" --timeout 300", TestAppId, resumeFileForClient)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(1u << 16, view[0].version);

                // 送信者がローカルコンテンツ配信必須システムバージョンを満たしていない場合

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application uninstall " + TestAppIdString,
                }));
                InstallTestApplication(command, 0);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 600", TestAppIdString)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(1u << 16, view[0].version);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void CannotUpdateSystemForRequiredSystemVersionToDeliver()
        {
            var barrier = new Barrier(2);

            var resumeFileForHost = Path.Combine(SysMaker.OutputDirPath, "resumeForHost.txt");
            if (File.Exists(resumeFileForHost))
            {
                File.Delete(resumeFileForHost);
            }
            var hostTask = Task.Run(() =>
            {
                using (var recover = new ExFatRecovery(HostCommand, TestContext))
                {
                    var command = HostCommand;
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "localcontentshare set-required-system-version 2",
                    }));
                    UninstallExFat(command);
                    SwitchToTestSystem(command, 2);
                    InstallTestApplication(command, 1);

                    barrier.SignalAndWait();
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString),
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
                }
            });

            var resumeFileForClient = Path.Combine(SysMaker.OutputDirPath, "resumeForClient.txt");
            if (File.Exists(resumeFileForClient))
            {
                File.Delete(resumeFileForClient);
            }
            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-required-system-version 2",
                }));
                // RequiredSystemVersion を満たすために、ver.1 のシステムを入れる
                SwitchToTestSystem(command, 1);
                InstallTestApplication(command, 0);
                Assert.IsTrue(!File.Exists(resumeFileForClient));

                barrier.SignalAndWait();

                // RequiredSystemVersion は満たしているけど、
                // ローカルコンテンツ配信必須システムバージョンを満たせず、LUP 出来ない場合
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 180", TestAppIdString, resumeFileForClient),
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, output.state);
                    Assert.AreEqual((UInt32)LcsFailureReason.CannotUpdateSystem, output.reason);
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void SendCorruptSystemTest()
        {
            var barrier = new Barrier(2);
            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                SwitchToTestSystem(command, IncludingExFatVersion);
                InstallTestApplication(command, 1);

                // SystemData を破損させる
                var dummyFile = Path.Combine(AppMaker.GetOutputDirectory().Replace("\"", ""), "Dummy.nca");
                if (File.Exists(dummyFile))
                {
                    File.Delete(dummyFile);
                }
                using (var sw = File.CreateText(dummyFile))
                {
                    sw.WriteLine("CorruptData");
                }

                AuthoringTool authoringTool = new AuthoringTool(this.TestContext);
                var exeResult = authoringTool.Execute("list " + SystemDataList[IncludingExFatVersion].Path + " --keyconfig " + authoringTool.KeyConfigFilePath);
                Assert.IsTrue(exeResult.Item1);
                var stdout = exeResult.Item2.ToString();
                string ncaName = null;
                foreach (var line in stdout.Split(new string[] { Environment.NewLine }, StringSplitOptions.None))
                {
                    if (line.IndexOf("SystemData.dat") >= 0)
                    {
                        ncaName = line.Substring(0, line.IndexOf("/"));
                        break;
                    }
                }
                Assert.IsTrue(!string.IsNullOrEmpty(ncaName));

                Assert.IsTrue(command.Execute(new string[]
                {
                    "filesystem delete-file /Contents/registered/" + ncaName + " -s system",
                    "filesystem create-file /Contents/registered/" + ncaName + " --src " + dummyFile + " -s system",
                }));


                barrier.SignalAndWait();

                // タイミングによって、返ってくる結果が変わるので、結果を見ないようにする
                command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application 0x{0:X16} --timeout 300", TestAppId)
                });
            });

            var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
            if (File.Exists(resumeFile))
            {
                File.Delete(resumeFile);
            }
            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                SwitchToTestSystem(command, 0);
                InstallTestApplication(command, 0);

                Assert.IsTrue(!File.Exists(resumeFile));

                var systemProgramList = GetSystemProgramList(command);
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == 0));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == 0));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == ExFatBootImageIdString));
                Assert.IsFalse(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataWithExFatFlagIdString));

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 300", TestAppIdString, resumeFile)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, output.state);
                    Assert.AreEqual((UInt32)LcsFailureReason.UnknownError, output.reason);
                }

                systemProgramList = GetSystemProgramList(command);
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == 0));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == 0));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == ExFatBootImageIdString));
                Assert.IsFalse(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataWithExFatFlagIdString));
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(1200 * 1000)]
        public void SendManySystemData()
        {
            ulong sysDataIdBase = 0x0100ffffaaaa0000;
            // TORIAEZU:    コンテンツの作成やインストールに割と時間が掛かるので、ほどほどの大きさにとどめておく
            //              64 + 1 で一応境界値 + 1 みたいな値にしておく
            const int MaxSystemDataCount = 65;

            var systemDataList = new List<NspInfo>();
            var contentMetaInfoList = new List<MakeTestSystemContent.ContentMetaInfo>();

            for (int i = 0; i < MaxSystemDataCount; i++)
            {
                systemDataList.Add(AppMaker.MakeSystemData(sysDataIdBase + (ulong)i));
                contentMetaInfoList.Add(new MakeTestSystemContent.ContentMetaInfo(systemDataList[i], 0));
            }

            ulong systemUpdateMetaId = 0x0100ffffbbbb0000;
            var systemUpdateV0 = SysMaker.MakeTestSystemUpdateMeta(systemUpdateMetaId, 0, new MakeTestSystemContent.ContentMetaInfo[] { });
            var systemUpdateV1 = SysMaker.MakeTestSystemUpdateMeta(systemUpdateMetaId, 1, contentMetaInfoList);

            Action<DevMenuCommandSystem> clearTestEnvironment = (DevMenuCommandSystem command) =>
            {
                var commandList = new List<string>();
                commandList.Add("localcontentshare set-system-update-id 0x0");
                commandList.Add(string.Format("systemprogram uninstall 0x{0:x16}", systemUpdateMetaId));
                command.Execute(commandList);
                commandList.Clear();
                // コマンドが長いと失敗するので、5個ずつインストールする
                foreach (var systemData in systemDataList)
                {
                    if (commandList.Count >= 5)
                    {
                        command.Execute(commandList);
                        commandList.Clear();
                    }
                    commandList.Add(string.Format("systemprogram uninstall 0x{0}", systemData.Id));
                }
                if (commandList.Count > 0)
                {
                    command.Execute(commandList);
                }
                ForceReset(command);
            };

            ulong appId = 0x0100ffffaaab0000;
            var appv0 = AppMaker.MakeApplication(appId, requiredSystemVersion: 0, additionalOptions: "--small-code");
            var patchv1 = AppMaker.MakePatch(appId, 1, appv0.Path, requiredSystemVersion: 1, additionalOptions: "--small-code");

            var barrier = new Barrier(2);
            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                using (var _ = new ScopeExit(() => { clearTestEnvironment(command); }))
                {
                    {
                        var commandList = new List<string>();
                        commandList.Add(string.Format("localcontentshare set-system-update-id 0x{0:X16}", systemUpdateMetaId));
                        commandList.Add("systemprogram install " + systemUpdateV1.Path);
                        Assert.IsTrue(command.Execute(commandList));
                        commandList.Clear();
                        // コマンドが長いと失敗するので、5個ずつインストールする
                        foreach (var systemData in systemDataList)
                        {
                            if (commandList.Count >= 5)
                            {
                                Assert.IsTrue(command.Execute(commandList));
                                commandList.Clear();
                            }
                            commandList.Add("systemprogram install " + systemData.Path);
                        }
                        if (commandList.Count > 0)
                        {
                            Assert.IsTrue(command.Execute(commandList));
                        }
                    }
                    ForceReset(command);

                    var systemProgramList = GetSystemProgramList(command);
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == string.Format("0x{0:x16}", systemUpdateMetaId) && x.version == 1));
                    Assert.IsTrue(systemDataList.TrueForAll(x => systemProgramList.Any(y => y.id.ToLower() == string.Format("0x{0}", x.Id).ToLower())));

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application install " + appv0.Path,
                        "patch install " + patchv1.Path
                    }));

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application 0x{0:X16} --timeout 600", appId)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }
               }
            });


            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                using (var _ = new ScopeExit(() => { clearTestEnvironment(command); }))
                {
                    var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
                    if (File.Exists(resumeFile))
                    {
                        File.Delete(resumeFile);
                    }

                    var commandList = new List<string>();
                    commandList.Add(string.Format("localcontentshare set-system-update-id 0x{0:X16}", systemUpdateMetaId));
                    commandList.Add("systemprogram install " + systemUpdateV0.Path);
                    Assert.IsTrue(command.Execute(commandList));
                    ForceReset(command);

                    var systemProgramList = GetSystemProgramList(command);
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == string.Format("0x{0:x16}", systemUpdateMetaId) && x.version == 0));
                    Assert.IsTrue(!systemDataList.Any(x => systemProgramList.Any(y => y.id.ToLower() == string.Format("0x{0}", x.Id).ToLower())));

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application install " + appv0.Path
                    }));

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application 0x{0:x16} --output-resume-file \"{1}\" --timeout 600", appId, resumeFile)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Suspened, output.state);
                        Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, output.reason);
                        Assert.IsTrue(File.Exists(resumeFile));
                    }

                    systemProgramList = GetSystemProgramList(command);
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == string.Format("0x{0:x16}", systemUpdateMetaId) && x.version == 1));
                    Assert.IsTrue(systemDataList.TrueForAll(x => systemProgramList.Any(y => y.id.ToLower() == string.Format("0x{0}", x.Id).ToLower())));

                    Reboot(command);

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application 0x{0:x16} --resume \"{1}\" --timeout 300", appId, resumeFile)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-view"
                    }));
                    var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual(1, view.Count);
                    Assert.AreEqual(1u << 16, view[0].version);
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(1200 * 1000)]
        public void TestErrorContextForSystem()
        {
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                SwitchToTestSystem(command, IncludingExFatVersion);
                InstallTestApplication(command, 1);

                barrier.SignalAndWait();

                command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 60", TestAppIdString)
                });
                // 結果は見ない


                UInt32 resultValue = 0x202;
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare set-async-task-result 0x{0:x16}", resultValue),
                }));
                Reboot(command);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 60", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, outputs.state);
                Assert.AreEqual((int)ErrorContextType.LocalContentShare, outputs.errorContext.type);
                var errorContext = outputs.errorContext.localContentShare;
                Assert.AreEqual("0x0000000000000000", errorContext.applicationId.ToLower());
                Assert.AreEqual(resultValue, errorContext.resultInnerValue);
                Assert.AreNotEqual(0u, errorContext.ip);
                Assert.IsTrue(errorContext.isSender);
                Assert.IsTrue(!errorContext.isApplicationRequest);
                Assert.IsTrue(errorContext.hasExFatDriver);
                Assert.AreEqual(1, errorContext.numKey);
                Assert.AreEqual(TestSystemUpdateId, errorContext.keyList[0].id);
                Assert.AreEqual((UInt32)IncludingExFatVersion, errorContext.keyList[0].version);
                Assert.AreEqual(3, errorContext.keyList[0].type);
                Assert.AreEqual(0, errorContext.keyList[0].installType);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                using (var recover = new ExFatRecovery(ClientCommand, TestContext))
                {
                    UninstallExFat(command, false);
                    SwitchToTestSystem(command, 0);

                    InstallTestApplication(command, 0);

                    UInt32 resultValue = 0x202;
                    Assert.IsTrue(command.Execute(new string[] {
                        string.Format("localcontentshare set-async-task-result 0x{0:x16}", resultValue),
                    }));
                    Reboot(command);

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 60", TestAppIdString)
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, outputs.state);
                    Assert.AreEqual((int)ErrorContextType.LocalContentShare, outputs.errorContext.type);
                    var errorContext = outputs.errorContext.localContentShare;
                    Assert.AreEqual("0x0000000000000000", errorContext.applicationId.ToLower());
                    Assert.AreEqual(resultValue, errorContext.resultInnerValue);
                    Assert.AreNotEqual(0u, errorContext.ip);
                    Assert.IsTrue(!errorContext.isSender);
                    Assert.IsTrue(!errorContext.isApplicationRequest);
                    Assert.IsTrue(!errorContext.hasExFatDriver);
                    Assert.AreEqual(1, errorContext.numKey);
                    Assert.AreEqual(TestSystemUpdateId, errorContext.keyList[0].id);
                    Assert.AreEqual((UInt32)IncludingExFatVersion, errorContext.keyList[0].version);
                    Assert.AreEqual(3, errorContext.keyList[0].type);
                    Assert.AreEqual(0, errorContext.keyList[0].installType);

                    resultValue = 0;
                    Assert.IsTrue(command.Execute(new string[] {
                        string.Format("localcontentshare set-async-task-result 0x{0:x16}", resultValue),
                    }));
                    Reboot(command);

                    barrier.SignalAndWait();

                    command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 60", TestAppIdString)
                    });
                    // 結果は見ない
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void TestMaxReleaseVersion()
        {
            // ReleaseVersion の最大値を設定
            var maxReleaseVersion = UInt16.MaxValue;
            var maxVersion = (UInt32)maxReleaseVersion << 16;
            var maxReleaseVersionSystemData = AppMaker.MakeSystemData(TestSystemDataId, version: maxReleaseVersion);
            var maxReleaseVersionSystemUpdate = SysMaker.MakeTestSystemUpdateMeta(TestSystemUpdateId, maxVersion, new MakeTestSystemContent.ContentMetaInfo[]
            {
                new MakeTestSystemContent.ContentMetaInfo(maxReleaseVersionSystemData, maxVersion),
            });
            var maxReleaseAppId = TestAppId + 0x2000;
            var maxReleaseApp = AppMaker.MakeApplication(maxReleaseAppId);
            var maxReleasePatch = AppMaker.MakePatch(maxReleaseAppId, maxReleaseVersion, maxReleaseApp.Path, requiredSystemVersion: maxVersion);

            var barrier = new Barrier(2);
            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                SwitchToTestSystemImpl(command, maxReleaseVersionSystemUpdate, new NspInfo[] { maxReleaseVersionSystemData });
                Assert.IsTrue(command.Execute(new string[]
                {
                    "application install " + maxReleaseApp.Path,
                    "patch install " + maxReleasePatch.Path,
                }));

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application 0x{0:X16} --timeout 300", maxReleaseAppId)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }
            });

            var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
            if (File.Exists(resumeFile))
            {
                File.Delete(resumeFile);
            }
            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                SwitchToTestSystem(command, 0);
                Assert.IsTrue(command.Execute(new string[]
                {
                    "application install " + maxReleaseApp.Path,
                }));

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application 0x{0:X16} --output-resume-file \"{1}\" --timeout 300", maxReleaseAppId, resumeFile)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Suspened, output.state);
                    Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, output.reason);
                    Assert.IsTrue(!string.IsNullOrEmpty(output.context));
                    Assert.IsTrue(File.Exists(resumeFile));
                }

                var systemProgramList = GetSystemProgramList(command);
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == maxVersion));
                Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemDataIdString && x.version == maxVersion));

                Assert.IsTrue(command.ResetAndExecute(new string[]
                {
                    string.Format("localcontentshare receive-application {0:X16} --resume \"{1}\" --timeout 300", maxReleaseAppId, resumeFile)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(maxVersion, view[0].version);
            });

            hostTask.Wait();
            clientTask.Wait();
        }
    }
}
