﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CodingCheckerUtil;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace CodingCheckerUtilTest
{
    [TestClass]
    public class ContentFormatTest
    {
        public TestContext TestContext { get; set; }

        private const string LineBreakLf = "\nHello\nWorld\n";
        private const string LineBreakCr = "\rHello\rWorld\r";
        private const string LineBreakCrlf = "\r\nHello\r\nWorld\r\n";
        private const string LineBreakMixed = "\nHello\rWorld\r\n";

        [TestMethod]
        public void FormatLineBreakTest()
        {
            // 各改行コードが CRLF へと変換されることを確認
            string actual;
            actual = string.Copy(LineBreakLf);
            Assert.IsTrue(ContentFormatter.Format(ref actual).IsFormatted);
            Assert.AreEqual(LineBreakCrlf, actual);

            actual = string.Copy(LineBreakCr);
            Assert.IsTrue(ContentFormatter.Format(ref actual).IsFormatted);
            Assert.AreEqual(LineBreakCrlf, actual);

            actual = string.Copy(LineBreakCrlf);
            Assert.IsFalse(ContentFormatter.Format(ref actual).IsFormatted);
            Assert.AreEqual(LineBreakCrlf, actual);

            actual = string.Copy(LineBreakMixed);
            Assert.IsTrue(ContentFormatter.Format(ref actual).IsFormatted);
            Assert.AreEqual(LineBreakCrlf, actual);
        }

        [TestMethod]
        public void FormatTabToSpaceTest()
        {
            // 行頭タブのスペースへの変換
            var target = "\tLine1\r\n\t\tLine2\r\n";
            var expected = "    Line1\r\n        Line2\r\n";
            Assert.IsTrue(ContentFormatter.Format(ref target).IsFormatted);
            Assert.AreEqual(expected, target);

            // 行頭タブのみ変換し、行中タブは変換しない
            target = "Line\t1\r\n\tLine2\r\n";
            expected = "Line\t1\r\n    Line2\r\n";
            Assert.IsTrue(ContentFormatter.Format(ref target).IsFormatted);
            Assert.AreEqual(expected, target);

            // 行頭タブがなければ何も行わない
            target = "value1\tvalue2\tvalue3\r\n    Line2\r\n";
            expected = "value1\tvalue2\tvalue3\r\n    Line2\r\n";
            Assert.IsFalse(ContentFormatter.Format(ref target).IsFormatted);
            Assert.AreEqual(expected, target);
        }

        [TestMethod]
        public void FormatFootSpaceTest()
        {
            // 行末のスペースを取り除く
            var target = "Line1    \r\nLine2\r\n";
            var expected = "Line1\r\nLine2\r\n";
            Assert.IsTrue(ContentFormatter.Format(ref target).IsFormatted);
            Assert.AreEqual(expected, target);

            // スペースのみの行は空行になる
            target = "Line1\r\n        \r\nLine3\r\n";
            expected = "Line1\r\n\r\nLine3\r\n";
            Assert.IsTrue(ContentFormatter.Format(ref target).IsFormatted);
            Assert.AreEqual(expected, target);

            // 行頭、行中スペースは変換しない
            target = "    Line1    \r\nLine2    WithTail\r\n";
            expected = "    Line1\r\nLine2    WithTail\r\n";
            Assert.IsTrue(ContentFormatter.Format(ref target).IsFormatted);
            Assert.AreEqual(expected, target);

            // 行末スペースがなければ何も行わない
            target = "    Line1\r\n\r\nLine3    WithTail\r\nLine4\r\n";
            expected = "    Line1\r\n\r\nLine3    WithTail\r\nLine4\r\n";
            Assert.IsFalse(ContentFormatter.Format(ref target).IsFormatted);
            Assert.AreEqual(expected, target);
        }

        [TestMethod]
        public void FormatFootLineBreakTest()
        {
            // 終端が改行でなければ付加する1
            var target = "Line1\r\nLine2\r\nLine3";
            var expected = "Line1\r\nLine2\r\nLine3\r\n";
            Assert.IsTrue(ContentFormatter.Format(ref target).IsFormatted);
            Assert.AreEqual(expected, target);

            // 終端が改行でなければ付加する2
            target = "Line1\r\nLine2\r\n\r\nLine3";
            expected = "Line1\r\nLine2\r\n\r\nLine3\r\n";
            Assert.IsTrue(ContentFormatter.Format(ref target).IsFormatted);
            Assert.AreEqual(expected, target);

            // 終端が改行ならば何も行わない1
            target = "Line1\r\nLine2\r\nLine3\r\n";
            expected = "Line1\r\nLine2\r\nLine3\r\n";
            Assert.IsFalse(ContentFormatter.Format(ref target).IsFormatted);
            Assert.AreEqual(expected, target);

            // 終端が改行ならば何も行わない2
            target = "Line1\r\nLine2\r\nLine3\r\n\r\n";
            expected = "Line1\r\nLine2\r\nLine3\r\n\r\n";
            Assert.IsFalse(ContentFormatter.Format(ref target).IsFormatted);
            Assert.AreEqual(expected, target);
        }
    }
}
