﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.IO;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace CodeCheckerTest
{
    [TestClass]
    public class CodeCheckerExecutionTest
    {
        public TestContext TestContext { get; set; }

        [TestMethod]
        public void ExecutionTest()
        {
            string testDir = Path.Combine(TestContext.TestRunDirectory, nameof(ExecutionTest));
            Directory.CreateDirectory(testDir);
            string testFile = Path.Combine(testDir, "test.txt");

            // OK
            string testText = "test.\r\n";
            File.WriteAllText(testFile, testText, new UTF8Encoding(true));
            Assert.AreEqual(0, ExecuteCodeChecker(testFile));

            // BOM チェック違反（UTF8 BOM 無し）
            testText = "test.\r\n";
            File.WriteAllText(testFile, testText, new UTF8Encoding(false));
            Assert.AreEqual(1, ExecuteCodeChecker(testFile));

            // コーディングチェック違反（行末に空白）
            testText = "test.    \r\n";
            File.WriteAllText(testFile, testText, new UTF8Encoding(false));
            Assert.AreEqual(1, ExecuteCodeChecker(testFile));
        }

        [TestMethod]
        public void ExecuteRecursiveTest()
        {
            string testDir = Path.Combine(TestContext.TestRunDirectory, nameof(ExecuteRecursiveTest));
            if (Directory.Exists(testDir))
            {
                Directory.Delete(testDir, true);
            }
            Directory.CreateDirectory(testDir);

            string testFile = Path.Combine(testDir, "test.txt");
            string testText = "test.\r\n";
            File.WriteAllText(testFile, testText, new UTF8Encoding(true));
            Assert.AreEqual(0, ExecuteCodeChecker(testDir, true));

            string testFile2 = Path.Combine(testDir, "test2.txt");
            testText = "test.    \r\n";
            File.WriteAllText(testFile, testText, new UTF8Encoding(true));
            Assert.AreEqual(1, ExecuteCodeChecker(testDir, true));
        }

        [TestMethod]
        public void ExecuteWithTargetExtensionListTest()
        {
            string testDir = Path.Combine(TestContext.TestRunDirectory, nameof(ExecuteWithTargetExtensionListTest));
            Directory.CreateDirectory(testDir);

            string testTextFile = Path.Combine(testDir, "test.txt");
            string testCppFile = Path.Combine(testDir, "test.cpp");
            string testText = "test.   \r\n";
            File.WriteAllText(testTextFile, testText, new UTF8Encoding(true));
            File.WriteAllText(testCppFile, testText, new UTF8Encoding(true));

            // .txt は BOM チェックのみ、 .cpp は全項目チェックを行う
            string targetExtensionList = Path.Combine(testDir, "TestTargetExtensions.list");
            string targetExtensionText = ".txt\tCheckBOM\r\n.cpp\tAll\r\n";
            File.WriteAllText(targetExtensionList, targetExtensionText, new UTF8Encoding(true));

            Assert.AreEqual(0, ExecuteCodeChecker(testTextFile, false, targetExtensionList));
            Assert.AreEqual(1, ExecuteCodeChecker(testCppFile, false, targetExtensionList));
        }

        private static int ExecuteCodeChecker(string path, bool recurse = false, string extensionList = "")
        {
            string args = string.Join(" ",
                new string[]
                {
                    $"\"{path}\"",
                    (recurse ? "--recurse" : ""),
                    (!string.IsNullOrEmpty(extensionList) ? $"--target_extension_list \"{extensionList}\"" : "")
                });

            var psi = new ProcessStartInfo()
            {
                FileName = Path.Combine(Environment.CurrentDirectory, "CodeChecker.exe"),
                Arguments = args,
                CreateNoWindow = true,
                UseShellExecute = false,
                RedirectStandardInput = false,
                RedirectStandardOutput = false,
                RedirectStandardError = false,
            };

            using (var p = Process.Start(psi))
            {
                p.WaitForExit();
                return p.ExitCode;
            }
        }
    }
}
