﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;
using System.Threading;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using TestUtility;

namespace AuthoringToolsTest
{
    public class StopWatch
    {
        [DllImport("kernel32.dll")]
        private static extern bool QueryPerformanceCounter(ref long lpPerformanceCount);
        [DllImport("kernel32.dll")]
        private static extern bool QueryPerformanceFrequency(ref long lpFrequency);

        private long startCounter;

        public void Start()
        {
            QueryPerformanceCounter(ref startCounter);
        }

        public double ElapsedMilliSec()
        {
            long stopCounter = 0;
            QueryPerformanceCounter(ref stopCounter);
            long frequency = 0;
            QueryPerformanceFrequency(ref frequency);
            return (double)(stopCounter - startCounter) * 1000.0 / frequency;
        }
    }

    public class AesCtrExBuildLogParser
    {
        public class EntryEntry
        {
            public long TableAddress { get; private set; }
            public long Offset { get; private set; }
            public uint Generation { get; private set; }

            public EntryEntry(long address, long offset, uint generation)
            {
                TableAddress = address;
                Offset = offset;
                Generation = generation;
            }
        }

        public IList<EntryEntry> EntryEntries
        {
            get; private set;
        } = new List<EntryEntry>();

        private AesCtrExBuildLogParser()
        {
        }

        public static AesCtrExBuildLogParser Parse(string path)
        {
            var parser = new AesCtrExBuildLogParser();

            FileStream stream = null;
            try
            {
                stream = new FileStream(path, FileMode.Open, FileAccess.Read);
                using (var sr = new StreamReader(stream))
                {
                    // FIXME : L1 とか

                    // Entry
                    {
                        var regex = new Regex(@"0x([\da-fA-F]+)\|Entry\|\(index:\d+ count:\d+\)");
                        while (!sr.EndOfStream)
                        {
                            var line = sr.ReadLine();
                            if (regex.IsMatch(line))
                            {
                                break;
                            }
                        }
                    }
                    {
                        var regex = new Regex(@"0x([\da-fA-F]+)\|\s+\|0x([\da-fA-F]+)\s+(\d+)");
                        while (!sr.EndOfStream)
                        {
                            var line = sr.ReadLine();
                            var match = regex.Match(line);
                            if (match.Success)
                            {
                                parser.EntryEntries.Add(new EntryEntry(
                                    long.Parse(match.Groups[1].Value, System.Globalization.NumberStyles.HexNumber),
                                    long.Parse(match.Groups[2].Value, System.Globalization.NumberStyles.HexNumber),
                                    uint.Parse(match.Groups[3].Value)));
                            }
                        }
                    }
                }
            }
            finally
            {
                if (stream != null)
                {
                    stream.Dispose();
                }
            }

            return parser;
        }

    }

    public class IndirectStorageBuildLogParser
    {
        public class EntryEntry
        {
            public long TableAddress { get; private set; }
            public long VirtualOffset { get; private set; }
            public long PhisicalOffset { get; private set; }
            public int StorageIndex { get; private set; }

            public EntryEntry(long address, long virtualOffset,  long phisycalOffset, int storageIndex)
            {
                TableAddress = address;
                VirtualOffset = virtualOffset;
                PhisicalOffset = phisycalOffset;
                StorageIndex = storageIndex;
            }
        }
        public IList<EntryEntry> EntryEntries
        {
            get; private set;
        } = new List<EntryEntry>();

        private IndirectStorageBuildLogParser()
        {

        }

        public static IndirectStorageBuildLogParser Parse(string path)
        {
            var parser = new IndirectStorageBuildLogParser();

            FileStream stream = null;
            try
            {
                stream = new FileStream(path, FileMode.Open, FileAccess.Read);
                using (var sr = new StreamReader(stream))
                {
                    // FIXME : L1 とか

                    // Entry
                    {
                        var regex = new Regex(@"0x([\da-fA-F]+)\|Entry\|\(index:\d+ count:\d+\)");
                        while (!sr.EndOfStream)
                        {
                            var line = sr.ReadLine();
                            if (regex.IsMatch(line))
                            {
                                break;
                            }
                        }
                    }
                    {
                        var regex = new Regex(@"0x([\da-fA-F]+)\|\s+\|0x([\da-fA-F]+)\s+0x([\da-fA-F]+)\s+(\d+)");
                        while (!sr.EndOfStream)
                        {
                            var line = sr.ReadLine();
                            var match = regex.Match(line);
                            if (match.Success)
                            {
                                parser.EntryEntries.Add(new EntryEntry(
                                    long.Parse(match.Groups[1].Value, System.Globalization.NumberStyles.HexNumber),
                                    long.Parse(match.Groups[2].Value, System.Globalization.NumberStyles.HexNumber),
                                    long.Parse(match.Groups[3].Value, System.Globalization.NumberStyles.HexNumber),
                                    int.Parse(match.Groups[4].Value)));
                            }
                        }
                    }
                }
            }
            finally
            {
                if (stream != null)
                {
                    stream.Dispose();
                }
            }

            return parser;
        }
    }

    public class DeltaBuildLogParser
    {
        public class CommandEntry
        {
            public long CommandSize { get; private set; }
            public long Offset { get; private set; }
            public long Size { get; private set; }

            public CommandEntry(long commandSize, long offset, long size)
            {
                CommandSize = commandSize;
                Offset = offset;
                Size = size;
            }
        }

        public class DeltaEntry
        {
            public long Offset { get; private set; }
            public long Size { get; private set; }

            public DeltaEntry(long offset, long size)
            {
                Offset = offset;
                Size = size;
            }
        }

        public IList<CommandEntry> CommandEntries
        {
            get; private set;
        } = new List<CommandEntry>();

        public IList<DeltaEntry> DeltaEntries
        {
            get; private set;
        } = new List<DeltaEntry>();

        public string TargetContentType
        {
            get; private set;
        } = string.Empty;

        private DeltaBuildLogParser()
        {
        }

        private void MergeList()
        {
            long offset = 0;
            long size = 0;
            foreach (var command in CommandEntries)
            {
                if (command.Offset == 0)
                {
                    size += command.Size;
                }
                else
                {
                    if (size > 0)
                    {
                        DeltaEntries.Add(new DeltaEntry(offset, size));
                    }

                    offset += size + command.Offset;
                    size = command.Size;
                }
            }

            if (size > 0)
            {
                DeltaEntries.Add(new DeltaEntry(offset, size));
            }
        }

        public static DeltaBuildLogParser Parse(string path)
        {
            var parser = new DeltaBuildLogParser();

            FileStream stream = null;
            try
            {
                stream = new FileStream(path, FileMode.Open, FileAccess.Read);
                using (var sr = new StreamReader(stream))
                {
                    // 1行目 : TargetContentType
                    parser.TargetContentType = sr.ReadLine().Split(',')[1];

                    // 2行目 : カラム
                    sr.ReadLine();

                    // 3行目～ : データ
                    while (!sr.EndOfStream)
                    {
                        var columns = sr.ReadLine().Split(',');
                        if (columns.Length == 0)
                        {
                            break;
                        }

                        // 0 - コマンド名
                        // 1 - コマンド値
                        // 2 - コマンドサイズ
                        // 3 - オフセット
                        // 4 - サイズ
                        // 5 - 備考
                        parser.CommandEntries.Add(new CommandEntry(long.Parse(columns[2]), long.Parse(columns[3]), long.Parse(columns[4])));
                    }
                }

                parser.MergeList();
            }
            finally
            {
                if (stream != null)
                {
                    stream.Dispose();
                }
            }

            return parser;
        }
    }

    class DisposeCheckStream : FileStream
    {
        public class DisposedValue
        {
            public bool Value;
        }

        public DisposeCheckStream(string path, DisposedValue disposed)
            : base(path, FileMode.Open, FileAccess.Read)
        {
            Disposed = disposed;
            Disposed.Value = false;
        }

        ~DisposeCheckStream()
        {
            Dispose(false);
        }

        private DisposedValue Disposed = null;

        public new void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected override void Dispose(bool disposing)
        {
            if (!this.Disposed.Value)
            {
                if (disposing)
                {
                    base.Dispose(disposing);
                }

                Disposed.Value = true;
            }
        }
    }

    public class Utils
    {
        public static void CheckReturnException(Exception ex, Func<bool> func)
        {
            bool resultOK = false;
            try
            {
                func();
            }
            catch (Exception e)
            {
                Assert.IsTrue(e.GetType() == ex.GetType());
                if (Thread.CurrentThread.CurrentCulture.Name == "ja-JP")
                {
                    Assert.IsTrue(e.Message == ex.Message);
                }
                resultOK = true;
            }
            Assert.IsTrue(resultOK);
        }

        public static byte[] GetTestBuffer(int size)
        {
            byte[] data = new byte[size];
            for (int i = 0; i < size; i++)
            {
                data[i] = (byte)i;
            }
            return data;
        }

        public static void CheckBufferEquality(byte[] data1, int offset1, byte[] data2, int offset2, int size)
        {
            Assert.IsTrue(data1.Length >= size + offset1);
            Assert.IsTrue(data2.Length >= size + offset2);
            for (int i = 0; i < size; i++)
            {
                Assert.IsTrue(data1[i + offset1] == data2[i + offset2]);
            }
        }

        public static void WriteLine(string message)
        {
            System.Diagnostics.Debug.WriteLine(message);
            Console.WriteLine(message);
        }

        public static void WriteLineToError(string message)
        {
            System.Diagnostics.Debug.WriteLine(message);
            Console.Error.WriteLine(message);
        }

        public static void DeleteDirectoryIfExisted(string path)
        {
            if (Directory.Exists(path))
            {
                DeleteDirectory(path, true);
            }
        }

        public static void DeleteIncludingJunctionDirectoryIfExisted(string path)
        {
            if (Directory.Exists(path))
            {
                DeleteIncludingJunctionDirectory(path);
                DeleteDirectoryIfExisted(path);
            }
        }

        private static void ForceDeleteDirectory(DirectoryInfo rootDirectoryInfo, bool isRecursive)
        {
            rootDirectoryInfo.Attributes = FileAttributes.Directory;
            if (isRecursive)
            {
                foreach (FileInfo fi in rootDirectoryInfo.GetFiles())
                {
                    fi.Attributes = FileAttributes.Normal;
                }
                foreach (DirectoryInfo di in rootDirectoryInfo.GetDirectories())
                {
                    ForceDeleteDirectory(di, isRecursive);
                }
            }
            rootDirectoryInfo.Delete(isRecursive);
        }

        private static void DeleteDirectory(string path, bool isRecursive, bool isForce = true)
        {
            DirectoryInfo rootDirectoryInfo = new DirectoryInfo(path);
            if (isForce)
            {
                ForceDeleteDirectory(rootDirectoryInfo, isRecursive);
            }
            else
            {
                Directory.Delete(path, isRecursive);
            }

            // Directory.Delete は非同期なので、削除されるまで待つ
            const int WaitForDeleteDirectoryTimeOutMilliSec = 1000 * 3;
            const int WaitForDeleteDirectoryWaitUnitMilliSec = 100;
            for (int waitMilliSec = 0; waitMilliSec < WaitForDeleteDirectoryTimeOutMilliSec; waitMilliSec += WaitForDeleteDirectoryWaitUnitMilliSec)
            {
                if (!Directory.Exists(path))
                {
                    break;
                }
                Thread.Sleep(WaitForDeleteDirectoryWaitUnitMilliSec);
            }
        }

        private static void DeleteIncludingJunctionDirectory(string path)
        {
            var attributes = File.GetAttributes(path);

            if ((attributes & FileAttributes.ReparsePoint) == FileAttributes.ReparsePoint)
            {
                // ジャンクション等は自身だけ削除する。中身は元フォルダの状況に依存させる
                DeleteDirectory(path, false);
            }
            else
            {
                foreach (var dirPath in Directory.EnumerateDirectories(path))
                {
                    DeleteIncludingJunctionDirectory(dirPath);
                }
            }
        }

        public static int GenerateRandomSeed()
        {
            return Guid.NewGuid().GetHashCode();
        }
    }

    internal static class MakeMetaUtil
    {
        internal static void MakeNpdm(TestPath testPath, string outFile, string metaFile, string descFile)
        {
            Process process = new Process();

            process.StartInfo.FileName = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\MakeMeta\\MakeMeta.exe";
            process.StartInfo.Arguments = String.Format("-o {0} --meta {1} --desc {2} --no_check_programid", outFile, metaFile, descFile);
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardError = true;
            process.Start();

            string errorMsg = process.StandardError.ReadToEnd();
            process.WaitForExit();
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
        }
    }

    public class FilterTestUtil
    {
        public class GenerateSampleFiles
        {
            public string[] fdfPathList = new string[11];//!! フィルターのテスト追加時はここの値をいじる必要あり
            public int validfdfPathListNum = 9;//!! フィルターのテスト追加時はここの値をいじる必要あり。上記のうち Parsefdf に成功する数
            public string testDirPath;
            public string[] fileTree;

            public int folderCount = 0;
            public int fileCount = 0;

            public int fdfTest0FileRemoveCount = 0;
            public int fdfTest0FileRemoveExceptionCount = 0;
            public int fdfTest3FileRemainCount = 0;
            public int fdfTest4FilterRuleCount = 0;
            public int fdfTest6FileRemainCount = 0;

            public GenerateSampleFiles() { }

            public void Generate(string rootPath)
            {
                // TODO: Directory 削除が非同期の為、対処する必要がある？

                for (int i = 0; i < fdfPathList.Length; i++)
                {
                    fdfPathList[i] = Path.Combine(Directory.GetCurrentDirectory(), string.Format("filter{0}.fdf", i));
                }

                string[] fdfTextList = new string[fdfPathList.Length];
                // filter0.ddf からは filter1.ddf を include, また自分自身を include（無視するはず）
                // WINDIR は環境変数の置換テスト、TestAuthoringTool は絶対パスを除外しないかのテスト, folder3\\ は円マーク区切りのテスト, ';'はコメント行のテスト
                //!! フィルターのテスト追加時はここを追加
                fdfTextList[0] = @"
-""@<WINDIR>/@<WINDIR>/hoge""
-""TestAuthoringTool""
-"".*\.txt$""
-""folder3\\.*""
#include ""./filter1.fdf""
#include ""@<AUTHORING_TOOL_WORK_DIR>/filter0.fdf""
#include ""filter0.fdf""
+""test.txt$""
;comment line
;+"".*""
-""^folder1/folderA""
-""folderB/fuga""
";
                fdfTextList[1] = @"
-""foo.org$""
+""var.org$""
";
                // 1 + 2 の remove のみ（明らかに含まれない WINDIR とかは除外）；つまり remove 例外（'+'）を除いたもの
                fdfTextList[2] = @"
-"".*\.txt$""
-""folder3/.*""
-""foo.org$""
-""^folder1/folderA""
-""folderB/fuga""
";
                // テスト用のファイルツリー構造
                fileTree = new string[] {
                        "root.txt",                 // -
                        "!root.bin",
                        "folder1/",
                        "folder1/B1.txt",           // -
                        "folder1/B1.foo.org",       // -
                        "folder1/B1.var.org",       // +
                        "folder1/folder2/",
                        "folder1/folder2/B2.txt",   // -
                        "folder1/folder2/folder3/", // - [dir]
                        "folder1/folder2/folder3/B3.bin",   // (-)
                        "folder1/folder2/folder3/folder4/", // (-) [dir]
                        "folder1/folder2/folder3/folder4/test.txt",  // (-) (-) +
                        "folder1/folderA/",         // - [dir]
                        "folder1/folderA/hoge.bin", // (-)
                        "folder1/folderB/",
                        "folder1/folderB/fuga.bin", // -
                        "^folder/",
                        "^folder/^piyo.bin",
                    };
                // fdfTextList[0]～[2]：フィルタ例外数の正解データ（！手動入力）
                fdfTest0FileRemoveCount = 8;
                fdfTest0FileRemoveExceptionCount = 1;   // (B1.var.org は元々除外ではない)

                // 全削除のテスト
                fdfTextList[3] = @"
-"".*""
+"".*\.bin$""
";
                // 相対パスのテスト
                fdfTextList[4] = @"
#include ""./child/filter5.fdf""
-""key4""
";
                fdfTextList[5] = @"
#include ""../filter4.fdf""
-""key5""
";
                //固定パス絶対追加のテスト
                fdfTextList[6] = @"
*+""folder1/folder2""
*+""folder1/folderB/fuga.bin""
-"".*B.*txt""
";
                // 空のファイルでも動作する
                fdfTextList[7] = @"
";
                // 空のファイルを include しても動作する
                fdfTextList[8] = @"
#include ""./filter7.fdf""
";
                //*+ が途中で入ってる場合に弾くテスト
                fdfTextList[9] = @"
-"".*""
*+""folder1/folder2""
";
                //*+ が途中で入ってる場合に弾くテスト(include でも弾けること確認)
                fdfTextList[10] = @"
*+""folder1""
#include ""./filter9.fdf""
";

                fdfTest4FilterRuleCount = 2;
                string fdfChildPath = Path.Combine(Directory.GetCurrentDirectory(), "child");
                Directory.CreateDirectory(fdfChildPath);
                fdfPathList[5] = Path.Combine(fdfChildPath, string.Format("filter{0}.fdf", 5));

                // fdfTextList[3]：フィルタによる残り数の正解データ（！手動入力）
                fdfTest3FileRemainCount = 5;
                fdfTest6FileRemainCount = 3;

                // フィルタファイルの作成
                for (int i = 0; i < fdfPathList.Length; i++)
                {
                    File.WriteAllText(fdfPathList[i], fdfTextList[i], System.Text.Encoding.UTF8);
                }

                // テスト用のフォルダ・ファイルを作成
                Directory.CreateDirectory(rootPath);
                testDirPath = Path.Combine(rootPath, "FilterTest");
                Utils.DeleteDirectoryIfExisted(testDirPath);
                Directory.CreateDirectory(testDirPath);

                // フォルダ・ファイルを実際に作成する
                foreach (var filePath in fileTree)
                {
                    var path = Path.Combine(testDirPath, filePath);

                    // フォルダ
                    if (path.EndsWith("/"))
                    {
                        Directory.CreateDirectory(path);
                        folderCount++;
                    }
                    else
                    {
                        using (FileStream stream = File.Create(path))
                        {
                            int fileSize = 1024;
                            byte[] data = new byte[fileSize];
                            for (int i = 0; i < fileSize; i++)
                            {
                                data[i] = (byte)i;
                            }
                            stream.Write(data, 0, fileSize);
                        }
                        fileCount++;
                    }
                }
            }
        }
    }
}
