﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>

#include <FileSystemMetaLibrary/IndirectStorageStreamUtil.h>

using namespace System::Collections::Generic;
using namespace System::Linq;
using namespace Microsoft::VisualStudio::TestTools::UnitTesting;
using namespace Nintendo::Authoring::FileSystemMetaLibrary;

namespace AuthoringToolsNativeTest
{

    private ref class TestReadOnlySource : public IReadOnlySource
    {
    public:
        TestReadOnlySource()
        {
            m_AccessLog = gcnew List<Tuple<IntPtr, Int64, Int32>^>();
            m_SizeRead = 0;
            m_Size = 0;
        }

        virtual Int32 Read(IntPtr address, Int64 offset, Int32 size)
        {
            m_AccessLog->Add(gcnew Tuple<IntPtr, Int64, Int32>(address, offset, size));
            return m_SizeRead;
        }

        virtual Int64 GetSize()
        {
            return m_Size;
        }

        void SetSizeRead(int sizeRead)
        {
            m_SizeRead = sizeRead;
        }

        void SetSize(int64_t size)
        {
            m_Size = size;
        }

        IList<Tuple<IntPtr, Int64, Int32>^>^ GetAccessLog()
        {
            return m_AccessLog;
        }

        void Reset()
        {
            m_AccessLog->Clear();
            m_SizeRead = 0;
            m_Size = 0;
        }

    private:
        IList<Tuple<IntPtr, Int64, Int32>^>^ m_AccessLog;
        Int32 m_SizeRead;
        Int64 m_Size;
    };

    [TestClassAttribute]
    public ref class ManagedSourceStorageTest
    {
    public:
        [TestMethodAttribute]
        void TestRead()
        {
            auto source = gcnew TestReadOnlySource();

            // 正常系: nullptr に対してサイズ 0 の読み出し
            {
                source->SetSizeRead(0);

                ManagedSourceStorage storage(source);
                auto result = storage.Read(0, nullptr, 0);

                Assert::AreEqual(1, source->GetAccessLog()->Count);
                Assert::AreEqual(IntPtr::Zero, source->GetAccessLog()[0]->Item1);
                Assert::AreEqual(0LL, source->GetAccessLog()[0]->Item2);
                Assert::AreEqual(0, source->GetAccessLog()[0]->Item3);
                Assert::IsTrue(result.IsSuccess());

                source->Reset();
            }

            // 正常系: 適当なバッファーに読み出し
            {
                source->SetSizeRead(3);

                char buffer[4];
                std::memset(buffer, 0xFE, sizeof(buffer));

                ManagedSourceStorage storage(source);
                auto result = storage.Read(1, buffer, 4);

                Assert::AreEqual(1, source->GetAccessLog()->Count);
                Assert::AreEqual(IntPtr(buffer), source->GetAccessLog()[0]->Item1);
                Assert::AreEqual(1LL, source->GetAccessLog()[0]->Item2);
                Assert::AreEqual(4, source->GetAccessLog()[0]->Item3);
                Assert::IsTrue(result.IsSuccess());

                // 範囲外が 0 フィルされることを確認
                Assert::AreEqual(static_cast<byte>(0xFE), static_cast<byte>(buffer[2]));
                Assert::AreEqual(static_cast<byte>(0x00), static_cast<byte>(buffer[3]));

                source->Reset();
            }

            // 異常系: ソースが要求より大きいサイズを返す
            {
                source->SetSizeRead(10);

                char buffer[4];
                std::memset(buffer, 0xFE, sizeof(buffer));

                ManagedSourceStorage storage(source);
                auto result = storage.Read(1, buffer, 4);

                Assert::AreEqual(1, source->GetAccessLog()->Count);
                Assert::IsTrue(result.IsSuccess());

                source->Reset();
            }

            // 異常系: サイズに負を指定
            {
                char buffer[4];
                std::memset(buffer, 0xFE, sizeof(buffer));

                ManagedSourceStorage storage(source);
                auto result = storage.Read(1, buffer, -1);

                Assert::AreEqual(0, source->GetAccessLog()->Count);
                Assert::IsTrue(result.IsFailure());

                source->Reset();
            }

            GC::KeepAlive(source);
        }
    };

}
