﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Collections.Generic;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Xml.Serialization;
using System.Xml;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.ApplicationControlProperty;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.AuthoringTool;
using Nintendo.Authoring.ETicketLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;
using TestUtility;

namespace AuthoringToolsTest
{
    [TestClass]
    public class ExcecutionTest : ExcecutionTestBase
    {
        [TestInitialize]
        public void ForceGC()
        {
            ForceGCImpl();
            Environment.SetEnvironmentVariable("AOC_TEST_DATA_PATH", "data", EnvironmentVariableTarget.Process);
        }

        private void CheckAocResultXml(string xmlPath, string nspPath, List<UInt64> idList)
        {
            CheckXmlEncoding(xmlPath);

            ResultModel model;
            using (var fs = new FileStream(xmlPath, FileMode.Open, FileAccess.Read))
            {
                var serializer = new XmlSerializer(typeof(ResultModel));
                model = (ResultModel)serializer.Deserialize(fs);
            }

            Assert.IsTrue(model.Code == "Pass");
            Assert.IsTrue(model.ErrorMessage == string.Empty);

            using (var fs = new FileStream(nspPath, FileMode.Open, FileAccess.Read))
            {
                Assert.IsTrue(model.Size == fs.Length);
                var hashCalculator = new SHA256CryptoServiceProvider();
                var hash = "0x" + BitConverter.ToString(hashCalculator.ComputeHash(fs), 0, 32).Replace("-", string.Empty).ToLower();
                Assert.IsTrue(model.Hash == hash);
            }

            Assert.IsTrue(model.ContentMetaList.Count == idList.Count);
            foreach(var id in idList)
            {
                Assert.IsTrue(model.ContentMetaList.Any((cm) => Convert.ToUInt64(cm.Id, 16) == id));
            }

            // TORIAEZU
            Assert.IsTrue(model.Date != string.Empty);
            Assert.IsTrue(model.Command != string.Empty);
            Assert.IsTrue(model.ToolVersion != string.Empty);
            Assert.IsTrue(model.ContentMetaList.Count == idList.Count);
            foreach (var cnmtModel in model.ContentMetaList)
            {
                Assert.IsTrue(!string.IsNullOrEmpty((cnmtModel as AddOnContentContentMetaModel)?.Tag));
            }
        }

        private void CheckDataContents(string nspPath, bool expectPublic, KeyConfiguration config)
        {
            ForceGC();

            using (var fs = new FileStream(nspPath, FileMode.Open, FileAccess.Read))
            {
                var nspReader = new NintendoSubmissionPackageReader(fs);
                List<ulong> dataContentFound = new List<ulong>();
                List<ulong> metaFound = new List<ulong>();
                foreach (var info in nspReader.ListFileInfo().FindAll(x => Path.GetExtension(x.Item1) == ".nca"))
                {
                    var ncaReader = nspReader.OpenNintendoContentArchiveReader(info.Item1, new NcaKeyGenerator(config));
                    var programId = ncaReader.GetProgramId();
                    if (ncaReader.GetContentType() == NintendoContentFileSystemMetaConstant.ContentTypeMeta)
                    {
                        metaFound.Add(programId);
                        continue;
                    }
                    if (ncaReader.GetContentType() == NintendoContentFileSystemMetaConstant.ContentTypeData)
                    {
                        Assert.IsFalse(dataContentFound.Contains(programId));
                        Assert.IsFalse(expectPublic);
                        dataContentFound.Add(programId);
                    }
                    if (ncaReader.GetContentType() == NintendoContentFileSystemMetaConstant.ContentTypePublicData)
                    {
                        Assert.IsFalse(dataContentFound.Contains(programId));
                        Assert.IsTrue(expectPublic);
                        dataContentFound.Add(programId);
                    }
                }
                Assert.IsTrue(metaFound.All(p => dataContentFound.Contains(p)));
                Assert.IsTrue(metaFound.Count == dataContentFound.Count);
            }
        }

        private void CheckAocGeneration(string nspPath, KeyConfiguration config)
        {
            ForceGC();

            using (var nspReader = new NintendoSubmissionPackageReader(nspPath))
            {
                var nmetas = nspReader.ListFileInfo().Where(x => Path.GetExtension(x.Item1) == ".cnmt.xml");
                foreach (var nmeta in nmetas)
                {
                    var model = ArchiveReconstructionUtils.ReadModel(nspReader, nmeta.Item1, nmeta.Item2);
                    foreach (var ncaReader in model.ContentList.Where(x => x.Type == "Data").Select(x => nspReader.OpenNintendoContentArchiveReader(x.Id + ".nca", new NcaKeyGenerator(config))))
                    {
                        foreach (var fsIndex in ncaReader.GetExistentFsIndices())
                        {
                            var fsInfo = ncaReader.GetFsHeaderInfo(fsIndex);
                            Assert.AreEqual(model.Version, fsInfo.GetGeneration());
                        }
                    }
                }
            }
        }

        private void CheckAocGenerationUpdated(string nsp1Path, List<uint> expectedVersion1List,
                                               string nsp2Path, List<uint> expectedVersion2List, KeyConfiguration config)
        {
            // expectedVersionXList には、nsp に含まれるデータコンテンツ nca の期待値と順番を揃えて入力すること

            ForceGC();

            var contentName1List = new List<string>();
            var contentName2List = new List<string>();

            Action<string, List<uint>, List<string>> checkNsp = (nspPath, expectedVersionList, contentNameList) =>
            {
                using (var nspReader = new NintendoSubmissionPackageReader(nspPath))
                {
                    int i = 0;
                    foreach (var info in nspReader.ListFileInfo().FindAll(x => Path.GetExtension(x.Item1) == ".nca"))
                    {
                        using (var ncaReader = nspReader.OpenNintendoContentArchiveReader(info.Item1, new NcaKeyGenerator(config)))
                        {
                            if (ncaReader.GetContentType() == NintendoContentFileSystemMetaConstant.ContentTypePublicData)
                            {
                                var expectedVersion = expectedVersionList[i];
                                contentNameList.Add(info.Item1);
                                foreach (var fsIndex in ncaReader.GetExistentFsIndices())
                                {
                                    var fsInfo = ncaReader.GetFsHeaderInfo(fsIndex);
                                    Assert.AreEqual(expectedVersion, fsInfo.GetGeneration());
                                }
                                i++;
                            }
                        }
                    }
                }
            };

            checkNsp(nsp1Path, expectedVersion1List, contentName1List);
            checkNsp(nsp2Path, expectedVersion2List, contentName2List);

            Trace.Assert(expectedVersion1List.Count == expectedVersion2List.Count);
            Trace.Assert(contentName1List.Count == contentName2List.Count);

            for (int i = 0; i < expectedVersion1List.Count; i++)
            {
                if (expectedVersion1List[i] == expectedVersion2List[i])
                {
                    Assert.AreEqual(contentName1List[i], contentName2List[i]);
                }
                else
                {
                    Assert.AreNotEqual(contentName1List[i], contentName2List[i]);
                }
            }
        }

        public static void CreateTestData(string filePath, int fileSize)
        {
            using (FileStream stream = File.Create(filePath))
            {
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }
        }

        private void CheckTicketInNsp(string nspPath, KeyConfiguration config)
        {
            ForceGC();

            using (var fs = new FileStream(nspPath, FileMode.Open, FileAccess.Read))
            {
                var nspReader = new NintendoSubmissionPackageReader(fs);
                // コンテンツ ID とタイプを保持
                var contentList = new List<Tuple<string, string>>();
                foreach (var file in nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".cnmt.xml")))
                {
                    var model = ReadXmlFromNsp<ContentMetaModel>(nspReader, file.Item1, file.Item2);
                    foreach (var contentInfo in model.ContentList)
                    {
                        contentList.Add(Tuple.Create(contentInfo.Id, contentInfo.Type));
                    }
                    Assert.IsTrue(model.Type == "Application" ||
                                  model.Type == "Patch" ||
                                  model.Type == "AddOnContent");
                }
                // 有効な rightsId が設定されたコンテンツを探す
                foreach (var info in nspReader.ListFileInfo().FindAll(x => Path.GetExtension(x.Item1) == ".nca"))
                {
                    var ncaName = info.Item1;
                    var ncaReader = nspReader.OpenNintendoContentArchiveReader(ncaName, new NcaKeyGenerator(config));
                    var rightsId = ncaReader.GetRightsId();
                    if (TicketUtility.IsValidRightsId(rightsId))
                    {
                        // 内部鍵を持っていない
                        Assert.IsFalse(ncaReader.HasValidInternalKey());
                        // コンテンツタイプが Program or Html or Data (Public)
                        {
                            var content = contentList.Where(x => ncaName.StartsWith(x.Item1));
                            Assert.IsTrue(content.Single().Item2 == "Program" ||
                                          content.Single().Item2 == "HtmlDocument" ||
                                          content.Single().Item2 == "Data");
                        }
                        // rightsId から始まるコモンチケットが nsp 入っている
                        Assert.IsTrue(nspReader.ListFileInfo().Where(x => x.Item1 == TicketUtility.CreateRightsIdText(rightsId) + ".tik").Any());
                    }
                    else
                    {
                        Assert.IsTrue(ncaReader.HasValidInternalKey());
                        var content = contentList.Where(x => ncaName.StartsWith(x.Item1));
                        Assert.IsTrue(content.Single().Item2 != "Program" &&
                                      content.Single().Item2 != "HtmlDocument" &&
                                      content.Single().Item2 != "Data");
                    }
                }
            }
        }

        private void WriteTmpMetaFileFooter(StreamWriter sw, bool isOldRoot)
        {
            sw.WriteLine("</Application>");
            if (isOldRoot)
            {
                sw.WriteLine("</Meta>");
            }
            else
            {
                sw.WriteLine("</NintendoSdkMeta>");
            }
        }

        private string GetIconContainApplicationMetaFile(string metaDir)
        {
            return Path.Combine(metaDir, "describe_all_with_icon.nmeta");
        }

        private void CheckGetNspProperty(string nspPath, string keyConfigFilePath = null)
        {
            var command = "getnspproperty " + nspPath;
            if (keyConfigFilePath != null)
            {
                command += " --keyconfig " + keyConfigFilePath;
            }
            string errorMsg = ExecuteProgram(command);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            command = "getnspproperty --show-apply-difference " + nspPath;
            errorMsg = ExecuteProgram(command);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
        }

        private string ExtractFilePathFromNsp(string nspPath, string fileName)
        {
            string stdout = ExecuteProgram(string.Format("list {0}", nspPath), true);
            Assert.IsTrue(!string.IsNullOrEmpty(stdout));
            var lines = stdout.Split();
            foreach (var line in lines)
            {
                if (line.IndexOf(fileName) >= 0)
                {
                    return line;
                }
            }

            return null;
        }

        private void ExtractFileFromNsp(string outputDir, string nspPath, string fileName)
        {
            var filePath = ExtractFilePathFromNsp(nspPath, fileName);
            Assert.IsTrue(filePath != null);

            string error = ExecuteProgram(string.Format("extract -o {0} --target {1} {2}", outputDir, filePath, nspPath));
            Assert.IsTrue(string.IsNullOrEmpty(error), error);
        }

        private void CompareTextFiles(string file1, string file2)
        {
            var f1Contents = File.ReadAllBytes(file1);
            var f2Contents = File.ReadAllBytes(file2);

            Assert.IsTrue(f1Contents.Count() == f2Contents.Count());
            Assert.IsTrue(f1Contents.SequenceEqual(f2Contents));
        }

        [TestMethod]
        public void TestLargeAddressAware()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var toolPath = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\AuthoringTool\\AuthoringTool.exe";

            var startInfo = new ProcessStartInfo();
            startInfo.FileName = System.Environment.GetEnvironmentVariable("ComSpec");
            startInfo.CreateNoWindow = true;
            startInfo.UseShellExecute = false;
            startInfo.RedirectStandardOutput = true;
            startInfo.RedirectStandardError = false;

            var vcToolPath = Environment.GetEnvironmentVariable("VS140COMNTOOLS");
            startInfo.WorkingDirectory = vcToolPath;

            startInfo.Arguments = string.Format("/c vsvars32.bat & dumpbin.exe /headers \"{0}\"", toolPath);
            Process process = Process.Start(startInfo);

            string stdOut = process.StandardOutput.ReadToEnd();
            Assert.IsTrue(stdOut.IndexOf("Application can handle large (>2GB) addresses") >= 0);

            process.WaitForExit();
            Assert.IsTrue(process.ExitCode == 0);
        }

        [TestMethod]
        public void TestExecutionCreateNca()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            string outputFile = Path.Combine(outputDir, "test.nca");
            string metaFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.aarch64.lp64.nmeta";
            string descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            string testDir = Path.Combine(outputDir, "test/");
            string testFile = testDir + "data.dat";
            string emptyDir = testDir + "empty";
            string testExtractDir = Path.Combine(outputDir, "extract");
            string testExtractEntryDir = Path.Combine(outputDir, "extractEntry");
            string testReplaceDir = Path.Combine(outputDir, "replace");

            SafeDeleteDirectory(testExtractDir);
            Directory.CreateDirectory(testDir);
            Directory.CreateDirectory(emptyDir);
            using (FileStream stream = File.Create(testFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            string testNpdm = testDir + "main.npdm";
            MakeNpdm(testNpdm, metaFile, descFile);

            // program
            {
                File.Delete(outputFile);
                string errorMsg = ExecuteProgram(string.Format("createnca -o {0} --meta {1} --desc {2} --program {3}", outputFile, metaFile, descFile, testDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(outputFile));

                errorMsg = ExecuteProgram(string.Format("list {0}", outputFile));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                TestExtractWithNca(outputFile, testDir, testExtractDir, 1);

                File.Delete(outputFile);
            }

            // program(rom)
            {
                File.Delete(outputFile);
                string errorMsg = ExecuteProgram(string.Format("createnca -o {0} --meta {1} --desc {2} --program {3} {3}", outputFile, metaFile, descFile, testDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(outputFile));

                errorMsg = ExecuteProgram(string.Format("list {0}", outputFile));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                TestExtractWithNca(outputFile, testDir, testExtractDir, 2);
                foreach (var test in ExtractEntryTestList)
                {
                    Assert.IsTrue(TestExtractEntry(outputFile, testExtractEntryDir, test.Item1) == test.Item2);
                }

                // first: pattern, second: isProgramContent
                Tuple<Regex, bool>[] ReplaceEntryInNcaTestList = new Tuple<Regex, bool>[] {
                    new Tuple<Regex, bool>(new Regex(@"fs0/data.dat"), true),
                    new Tuple<Regex, bool>(new Regex(@"fs1/main.npdm"), true),
                };
                var testCombination = new List<int[]>() { new int[] { 0 }, Enumerable.Range(0, ReplaceEntryInNcaTestList.Count()).ToArray() };
                TestReplaceWithNspOrNca(outputFile, testReplaceDir, testExtractDir, testExtractEntryDir, ReplaceEntryInNcaTestList, testCombination);

                File.Delete(outputFile);
            }

            // program(0 file rom)
            {
                File.Delete(outputFile);
                string errorMsg = ExecuteProgram(string.Format("createnca -o {0} --meta {1} --desc {2} --program {3} {4}", outputFile, metaFile, descFile, testDir, emptyDir));
                Assert.IsTrue(errorMsg.IndexOf("invalid format .adf file. At least one file must be included in the data region.") >= 0, errorMsg);
            }
        }

        // --filter, --only-adf を Main.Program 経由でテスト
        [TestMethod]
        public void TestExecutionCreateNcaFiltered()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            string outputFile = Path.Combine(outputDir, "test.nca");
            string outputAdfFile = Path.Combine(outputDir, "test.nca.adf");
            string outputRomAdfFile = Path.Combine(outputDir, "test.nca.rom.adf");
            string outputCodeAdfFile = Path.Combine(outputDir, "test.nca.code.adf");

            string metaFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.aarch64.lp64.nmeta";
            string descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            string testDir = Path.Combine(outputDir, "FilterTest");

            var sampleFiles = new FilterTestUtil.GenerateSampleFiles();
            sampleFiles.Generate(outputDir);

            string testNpdm = testDir + "/main.npdm";
            MakeNpdm(testNpdm, metaFile, descFile);

            // program
            {
                File.Delete(outputFile);
                File.Delete(outputAdfFile);
                File.Delete(outputCodeAdfFile);

                Program.Main(string.Format("createnca -o {0} --meta {1} --desc {2} --program {3} --filter {4} --only-adf",
                    outputFile, metaFile, descFile, testDir, sampleFiles.fdfPathList[0]).Split());
                Assert.IsTrue(File.Exists(outputFile) == false);
                Assert.IsTrue(File.Exists(outputAdfFile));
                Assert.IsTrue(File.Exists(outputCodeAdfFile));
                // Todo: Read Verify
                File.Delete(outputFile);
            }

            // program(rom)
            {
                File.Delete(outputFile);
                File.Delete(outputAdfFile);
                File.Delete(outputRomAdfFile);
                File.Delete(outputCodeAdfFile);

                Program.Main(string.Format("createnca -o {0} --meta {1} --desc {2} --program {3} {3} --filter {4} --only-adf",
                    outputFile, metaFile, descFile, testDir, sampleFiles.fdfPathList[0]).Split());
                Assert.IsTrue(File.Exists(outputFile) == false);
                Assert.IsTrue(File.Exists(outputAdfFile));
                Assert.IsTrue(File.Exists(outputRomAdfFile));
                Assert.IsTrue(File.Exists(outputCodeAdfFile));

                int numEntries = new RomFsAdfReader(outputRomAdfFile).GetFileSystemInfo().entries.Count;
                Assert.IsTrue(numEntries == (1 + sampleFiles.fileCount - sampleFiles.fdfTest0FileRemoveCount + sampleFiles.fdfTest0FileRemoveExceptionCount));

                // Todo: Read Verify
                File.Delete(outputFile);
            }
        }

        // ソートされたファイルリストが出力されることを --only-adf、Main.Program 経由でテスト
        [TestMethod]
        public void TestExecutionCreateNcaRomFsSorted()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            string outputFile = Path.Combine(outputDir, "test.nca");
            string outputAdfFile = Path.Combine(outputDir, "test.nca.adf");
            string outputRomAdfFile = Path.Combine(outputDir, "test.nca.rom.adf");
            string outputCodeAdfFile = Path.Combine(outputDir, "test.nca.code.adf");

            string metaFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.aarch64.lp64.nmeta";
            string descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            string testDir = Path.Combine(outputDir, "SortTest");

            SafeDeleteDirectory(testDir);
            Directory.CreateDirectory(testDir);

            {
                var list = new string[]
                {
                    "data_ä.dat",
                    @"data_b.txt\data_b.txt",
                    "data_c.txt",
                    "data_a.txt",
                    "data_1.txt"
                };
                foreach (var name in list)
                {
                    string testFile = testDir + "/" + name;

                    var testFileDirectory = Path.GetDirectoryName(testFile);
                    if (!Directory.Exists(testFileDirectory))
                    {
                        Directory.CreateDirectory(testFileDirectory);
                    }

                    using (var stream = File.Create(testFile))
                    {
                        stream.SetLength(1024);
                    }
                }

                foreach (var path in Directory.EnumerateFiles(testDir, "*.*", SearchOption.AllDirectories))
                {
                    Utils.WriteLine(path);
                }
            }

            string testNpdm = testDir + "/main.npdm";
            MakeNpdm(testNpdm, metaFile, descFile);

            // program
            {
                File.Delete(outputFile);
                File.Delete(outputAdfFile);
                File.Delete(outputRomAdfFile);
                File.Delete(outputCodeAdfFile);

                Program.Main(string.Format("createnca -o {0} --meta {1} --desc {2} --program {3} {3} --only-adf",
                    outputFile, metaFile, descFile, testDir).Split());
                Assert.IsTrue(File.Exists(outputFile) == false);
                Assert.IsTrue(File.Exists(outputAdfFile));
                Assert.IsTrue(File.Exists(outputRomAdfFile));
                Assert.IsTrue(File.Exists(outputCodeAdfFile));

                File.Delete(outputFile);

                var list = new string[]
                {
                    "data_1.txt",
                    "data_a.txt",
                    "data_b.txt/data_b.txt",
                    "data_c.txt",
                    "data_ä.dat"
                };

                var index = 0;
                using (var stream = new StreamReader(File.OpenRead(outputRomAdfFile), System.Text.Encoding.UTF8))
                {
                    while (!stream.EndOfStream)
                    {
                        var line = stream.ReadLine();
                        if (index < list.Length)
                        {
                            if (line.Contains(string.Format("name : \"{0}\"", list[index])))
                            {
                                ++index;
                            }
                        }
                    }
                }

                Assert.AreEqual(list.Length, index);
            }
        }

        // nca の追加テスト
        [TestMethod]
        public void TestExecutionReconstractArchive()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var invalidIconDir = testSourceDir + "\\Icon\\Invalid";
            var mainVisualDir = testSourceDir + "\\MainVisual";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var testExtractDir = Path.Combine(outputDir, "extract");
            var testExtractEntryDir = Path.Combine(outputDir, "extractEntry");
            var testReplaceDir = Path.Combine(outputDir, "replace");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(testExtractDir);
            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }
            var metaFile = Directory.EnumerateFiles(metaDir).First();

            var legalInformationZip = MakeLegalInfoZipfile(outputDir);

            // create full nsp
            string fullNspPath = "";
            {
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile);

                fullNspPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".full.nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".bmp";
                var mainVisualPath = mainVisualDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".png";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4} --html-document {5} --legal-information {6} --save-adf",
                    fullNspPath,
                    metaFile,
                    descFile,
                    dataDir,
                    iconPath,
                    dataDir,
                    legalInformationZip));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(fullNspPath));
                VerifyNsp(fullNspPath, keyConfigFile);
            }

            // create base nsp
            string baseNspPath = "";
            {
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile);

                baseNspPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".bmp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4} --save-adf",
                    baseNspPath,
                    metaFile,
                    descFile,
                    dataDir,
                    iconPath));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(baseNspPath));
                VerifyNsp(baseNspPath, keyConfigFile);
            }

            // 追加のテスト
            var errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", fullNspPath, testExtractDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            var entryList = GetEntryFilePathList(testExtractDir);
            var targetName = "legalinfo.xml";
            var ncaName = entryList.Find(entry => entry.EndsWith("/" + targetName)).Replace("/fs0/" + targetName, "");

            string tempReplaceDir = Path.Combine(outputDir, "replace_nca");
            foreach (var content in new string[] { "Program", "Control", "Data", "PublicData", "Meta", "LegalInformation", "Hoge", "" })
            {
                var addNcaTargetFailure = new Tuple<string, string>(baseNspPath, string.Format("null.nca@{0}", content));
                TestReplaceNcaInNsp(fullNspPath, tempReplaceDir, ncaName, targetName, addNcaTargetFailure, content != "LegalInformation");
            }
        }

        public TestMetaFile MakeIconPathMetaFile(string iconPath, string outputDir = null)
        {
            TestMetaFile metaFile = new TestMetaFile();
            if (outputDir != null)
            {
                metaFile.OutputDir = outputDir;
            }
            metaFile.AddApplicationTags("<Title>");
            metaFile.AddApplicationTags("<Language>AmericanEnglish</Language>");
            metaFile.AddApplicationTags("<Name>Name</Name>");
            metaFile.AddApplicationTags("<Publisher>Publisher</Publisher>");
            metaFile.AddApplicationTags("</Title>");
            metaFile.AddApplicationTags("<Icon>");
            metaFile.AddApplicationTags("<Language>AmericanEnglish</Language>");
            metaFile.AddApplicationTags(string.Format("<IconPath>{0}</IconPath>", iconPath));
            metaFile.AddApplicationTags("</Icon>");
            metaFile.Write();

            return metaFile;
        }

        public TestMetaFile MakeIconPathMetaFileUsingEnvironmentVariable(string environmentVariableName, string outputDir = null)
        {
            TestMetaFile metaFile = new TestMetaFile();
            if (outputDir != null)
            {
                metaFile.OutputDir = outputDir;
            }
            metaFile.AddApplicationTags("<Title>");
            metaFile.AddApplicationTags("<Language>AmericanEnglish</Language>");
            metaFile.AddApplicationTags("<Name>Name</Name>");
            metaFile.AddApplicationTags("<Publisher>Publisher</Publisher>");
            metaFile.AddApplicationTags("</Title>");
            metaFile.AddApplicationTags("<Icon>");
            metaFile.AddApplicationTags("<Language>AmericanEnglish</Language>");
            metaFile.AddApplicationTags(string.Format("<IconPath UseEnvironmentVariable=\"true\">%{0}%</IconPath>", environmentVariableName));
            metaFile.AddApplicationTags("</Icon>");
            metaFile.Write();

            return metaFile;
        }

        public void OutputReplaceList(string replaceRuleListFilePath, List<Tuple<string, string>> replaceRuleList)
        {
            using (var fw = new System.IO.StreamWriter(replaceRuleListFilePath))
            {
                for (int i = 0; i < replaceRuleList.Count; i++)
                {
                    fw.Write(string.Format("{0}\t{1}\n", replaceRuleList[i].Item1, replaceRuleList[i].Item2));
                }
            }
        }

        [TestMethod]
        public void TestConvertIcon()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var testConvertDir = Path.Combine(outputDir, "convert");

            SafeDeleteDirectory(testConvertDir);
            Directory.CreateDirectory(testConvertDir);

            var metaFile = Directory.EnumerateFiles(metaDir).First();
            var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".bmp";
            var iconName = Path.GetFileName(iconPath);

            var errorMsg = ExecuteProgram(string.Format("converticon {0} -o {1}", iconPath, testConvertDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            var iconRawOutPath = Path.Combine(testConvertDir, iconName.Replace(".bmp", ".raw.jpg"));
            var iconNxOutPath = Path.Combine(testConvertDir, iconName.Replace(".bmp", ".nx.jpg"));

            Assert.IsTrue(File.Exists(iconRawOutPath));
            Assert.IsTrue(File.Exists(iconNxOutPath));

            var orgFileSize = new System.IO.FileInfo(iconPath).Length;
            var rawFileSize = new System.IO.FileInfo(iconRawOutPath).Length;
            var nxFileSize = new System.IO.FileInfo(iconNxOutPath).Length;
            Assert.IsTrue(orgFileSize > rawFileSize && rawFileSize > nxFileSize);
        }

        [TestMethod]
        public void TestReplaceTarget()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var invalidIconDir = testSourceDir + "\\Icon\\Invalid";
            var mainVisualDir = testSourceDir + "\\MainVisual";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var testExtractDir = Path.Combine(outputDir, "extract");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            var metaFile = GetIconContainApplicationMetaFile(metaDir);
            string testNpdm = dataDir + "\\main.npdm";
            MakeNpdm(testNpdm, metaFile, descFile);

            var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
            var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".bmp";

            var error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type {1} --meta {2} --desc {3} --program {4} --save-adf --keyconfig {5}",
                outputPath,
                "Application",
                metaFile,
                descFile,
                dataDir,
                keyConfigFile));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(File.Exists(outputPath));

            // 一度全部 extract する
            SafeDeleteDirectory(testExtractDir);
            var errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", outputPath, testExtractDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            var allFileList = new List<string>(System.IO.Directory.GetFiles(testExtractDir, "*", System.IO.SearchOption.AllDirectories)).Select(x => x.Replace(@"\", "/"));

            var stdMsg = ExecuteProgram(string.Format("list {0} --targetregex \"{1}\"", outputPath, "."), true);
            var listedEntryList = ParseListStdOutput(stdMsg);

            var sampleEntryPath = listedEntryList.Where(path => path.IndexOf(".nca/") >= 0).First();
            var sampleFilePath = allFileList.First();

            {
                // 差替えるものがない時（失敗を想定）
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} hogehoge {2}", outputPath, outputDir, sampleFilePath));
                Assert.IsTrue(errorMsg.IndexOf("nothing was replaced.") >= 0);

                // 単一エントリへの複数アクション指定（失敗を想定）
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} {2}|{2} {3}|{3}", outputPath, outputDir, sampleEntryPath, sampleFilePath));
                Assert.IsTrue(errorMsg.IndexOf("multiple actions are specified for one entry.") >= 0);

                // 直下のアイコンエントリの差し替え指定ミス（失敗を想定）
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} null.icon@Japanese {2}", outputPath, outputDir, sampleFilePath));
                Assert.IsTrue(errorMsg.IndexOf("format for adding icon should be:") >= 0);
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} null.icon@csharp@nx {2}", outputPath, outputDir, sampleFilePath));
                Assert.IsTrue(errorMsg.IndexOf("unknown language") >= 0);
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} null.icon@japanese@nx {2}", outputPath, outputDir, sampleFilePath));
                Assert.IsTrue(errorMsg.IndexOf("unknown language") >= 0);

                // 複数エントリ指定の数間違い
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} {3}|hogehoge {4}", outputPath, outputDir, descFile, sampleEntryPath, sampleFilePath));
                Assert.IsTrue(errorMsg.IndexOf("Number of input file and replace target do not match.") >= 0);
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} {3} {4}|{4}", outputPath, outputDir, descFile, sampleEntryPath, sampleFilePath));
                Assert.IsTrue(errorMsg.IndexOf("Number of input file and replace target do not match.") >= 0);

                // 複数エントリの差替えで片方だけ失敗
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} {3}|hogehoge {4}|{4}", outputPath, outputDir, descFile, sampleEntryPath, sampleFilePath));
                Assert.IsTrue(errorMsg.IndexOf("entry to be replaced not found (replaced 1 entries while 2 rules are specified).") >= 0, errorMsg);
            }

            // ** エントリの追加（単数）
            int entryRandomizeIndex = 0;
            sampleFilePath = allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count());
            var addedEntry = sampleEntryPath + "2.dat";
            errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} add:{3} {4}", outputPath, outputDir, descFile, addedEntry, sampleFilePath));
            Assert.IsTrue(error == string.Empty, error);
            // （extract できるかチェック）
            var replacedNspPath = outputPath.Replace(".nsp", "_replaced.nsp");
            var testExtractRepDir = Path.Combine(outputDir, "replacedExtract");
            SafeDeleteDirectory(testExtractRepDir);
            errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", replacedNspPath, testExtractRepDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            // （内容の確認）
            var extractedFileList = new List<string>(System.IO.Directory.GetFiles(testExtractRepDir, "*", System.IO.SearchOption.AllDirectories)).Select(x => x.Replace(@"\", "/"));
            Assert.IsTrue(extractedFileList.Where(path => path.EndsWith("2.dat")).Any());
            Assert.IsTrue(extractedFileList.Count() == (listedEntryList.Where(path => !path.EndsWith(".nca")).Count() + 1));

            // ** エントリの削除（単数）
            var delEntry = listedEntryList.Where(path => path.EndsWith("/data.dat")).Single();
            {
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} del:{3} null", outputPath, outputDir, descFile, delEntry));
                Assert.IsTrue(error == string.Empty, error);
                SafeDeleteDirectory(testExtractRepDir);
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", replacedNspPath, testExtractRepDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                extractedFileList = new List<string>(System.IO.Directory.GetFiles(testExtractRepDir, "*", System.IO.SearchOption.AllDirectories)).Select(x => x.Replace(@"\", "/"));
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith("data.dat")).Any() == false, "deleted entry still exist");
                Assert.IsTrue(extractedFileList.Count() == (listedEntryList.Where(path => !path.EndsWith(".nca")).Count() - 1));
            }

            // *** エントリの追加と削除と差し替え（複数、直下アイコンエントリ含む）
            string replaceRuleListFilePath = Path.Combine(outputDir, "replaceRuleFilePath.csv");
            // 追加は挿入なので削除よりテスト項目を多く設ける（先頭への追加、単数の追加、複数の追加、末尾への追加）
            var replaceRuleList = new List<Tuple<string, string>> {
                new Tuple<string, string>("add:" + Regex.Replace(sampleEntryPath, "[^/]*$", "") + "abc.dat", allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                new Tuple<string, string>("add:" + Regex.Replace(sampleEntryPath, "[^/]*$", "") + "abd.dat", allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                new Tuple<string, string>("add:" + sampleEntryPath + "a2c.dat", allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                new Tuple<string, string>("add:" + sampleEntryPath + "ac.dat", allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                new Tuple<string, string>("del:" + delEntry, "null"),
                new Tuple<string, string>("add:" + sampleEntryPath + "a1c.dat", allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                new Tuple<string, string>("add:" + sampleEntryPath + "a1.dat", allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                new Tuple<string, string>("add:" + Regex.Replace(sampleEntryPath, "[^/]*$", "") + "zzy.dat", allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                new Tuple<string, string>("add:" + Regex.Replace(sampleEntryPath, "[^/]*$", "") + "zzz.dat", allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
            };
            OutputReplaceList(replaceRuleListFilePath, replaceRuleList);
            var delEntryCount = replaceRuleList.Where(pair => pair.Item1.StartsWith("del:")).Count();
            int entryCountDiff = replaceRuleList.Count - 2 * delEntryCount;
            {
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} {3}", outputPath, outputDir, descFile, replaceRuleListFilePath));
                Assert.IsTrue(error == string.Empty, error);
                SafeDeleteDirectory(testExtractRepDir);
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", replacedNspPath, testExtractRepDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                extractedFileList = new List<string>(System.IO.Directory.GetFiles(testExtractRepDir, "*", System.IO.SearchOption.AllDirectories)).Select(x => x.Replace(@"\", "/"));
                Assert.IsTrue(extractedFileList.Count() == (listedEntryList.Where(path => !path.EndsWith(".nca")).Count() + entryCountDiff));
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith("abc.dat")).Any());
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith("abd.dat")).Any());
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith("a2c.dat")).Any());
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith("data.dat")).Any() == false, "deleted entry still exist");
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith("zzy.dat")).Any());
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith("zzz.dat")).Any());
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith(".nx.Japanese.jpg")).Any());
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith(".raw.Japanese.jpg")).Any());
            }

            // （ファイル順序が正しいことの確認）
            var ncaName = Regex.Replace(sampleEntryPath, @"/.*", "");
            stdMsg = ExecuteProgram(string.Format("list {0} --targetregex \"{1}\"", replacedNspPath, "."), true);
            var repListedEntryList = ParseListStdOutput(stdMsg);
            var repListedTargetEntryList = ParseListStdOutput(stdMsg).Where(path => path.StartsWith(ncaName));
            var sortedTargetEntryList = new List<string>(repListedTargetEntryList);
            sortedTargetEntryList.Sort((a, b) => string.CompareOrdinal(a, b));
            Assert.IsTrue(repListedTargetEntryList.SequenceEqual(sortedTargetEntryList), "entry order is wrong");

            // 削除の確認
            {
                var replacedSampleEntryPathArray = repListedEntryList.Where(path => path.IndexOf("data.") >= 0);
                var jpnIconPathList = repListedEntryList.Where(path => path.IndexOf("Japanese") >= 0);
                replaceRuleList = new List<Tuple<string, string>> {
                    new Tuple<string, string>("del:" + replacedSampleEntryPathArray.ElementAt(0), allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                    new Tuple<string, string>("add:" + replacedSampleEntryPathArray.ElementAt(0) + "a3c.dat", allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                    new Tuple<string, string>("del:" + replacedSampleEntryPathArray.ElementAt(1), allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                    new Tuple<string, string>("del:" + jpnIconPathList.ElementAt(0), allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                    new Tuple<string, string>("del:" + jpnIconPathList.ElementAt(1), allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                };
                delEntryCount = replaceRuleList.Where(pair => pair.Item1.StartsWith("del:")).Count();
                entryCountDiff = replaceRuleList.Count - 2 * delEntryCount;
                OutputReplaceList(replaceRuleListFilePath, replaceRuleList);
                var targetNspPath = replacedNspPath;
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} {3} --ignore-unpublishable-error", targetNspPath, outputDir, descFile, replaceRuleListFilePath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                var targetReplacedNspPath = targetNspPath.Replace(".nsp", "_replaced.nsp");
                SafeDeleteDirectory(testExtractRepDir);
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", targetReplacedNspPath, testExtractRepDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                extractedFileList = new List<string>(System.IO.Directory.GetFiles(testExtractRepDir, "*", System.IO.SearchOption.AllDirectories)).Select(x => x.Replace(@"\", "/"));
                Assert.IsTrue(extractedFileList.Count() == (repListedEntryList.Where(path => ! path.EndsWith(".nca")).Count() + entryCountDiff));
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith("a3c.dat")).Any());
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith(".nx.Japanese.jpg")).Any() == false, "deleted entry still exist");
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith(".raw.Japanese.jpg")).Any() == false, "deleted entry still exist");
            }

            // nca のエントリの追加と削除
            {
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1} --target \"{2}\"", outputPath, outputDir, ncaName));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                var ncaPath = Path.Combine(outputDir, ncaName);

                replaceRuleList = new List<Tuple<string, string>> {
                    new Tuple<string, string>("add:" + "fs0/da2c.dat", allFileList.ElementAt(entryRandomizeIndex++ % allFileList.Count())),
                    new Tuple<string, string>("del:" + "fs0/data.dat", "null"),
                };
                OutputReplaceList(replaceRuleListFilePath, replaceRuleList);
                errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} {3}", ncaPath, outputDir, descFile, replaceRuleListFilePath));
                Assert.IsTrue(error == string.Empty, error);
                var replacedNcaPath = ncaPath.Replace(".nca", "_replaced.nca");
                SafeDeleteDirectory(testExtractRepDir);
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", replacedNcaPath, testExtractRepDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                extractedFileList = new List<string>(System.IO.Directory.GetFiles(testExtractRepDir, "*", System.IO.SearchOption.AllDirectories)).Select(x => x.Replace(@"\", "/"));
                stdMsg = ExecuteProgram(string.Format("list {0} --targetregex \"{1}\"", ncaPath, "."), true);
                var listedNcaEntryList = ParseListStdOutput(stdMsg);
                Assert.IsTrue(extractedFileList.Count() == (listedNcaEntryList.Count()));
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith("da2c.dat")).Any());
                Assert.IsTrue(extractedFileList.Where(path => path.EndsWith("data.dat")).Any() == false, "deleted entry still exist");
            }
        }

        [TestMethod]
        public void TestReplaceFileAlignment()
        {
            var testPath = new TestPath(TestContext);
            var rootPath = testPath.GetSigloRoot();
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var codeDir = Path.Combine(outputDir, "code");
            var dataDir = Path.Combine(outputDir, "data");
            var htmlDir = Path.Combine(outputDir, "html");
            var metaFile = Path.Combine(rootPath, "Programs\\Iris\\Resources\\SpecFiles\\Application.aarch64.lp64.nmeta");
            var descFile = Path.Combine(rootPath, "Programs\\Iris\\Resources\\SpecFiles\\Application.desc");
            var config = new AuthoringConfiguration();
            config.KeyConfigFilePath = Path.Combine(rootPath, "Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources\\AuthoringToolsTest.keyconfig.xml");
            var testExtractDir = Path.Combine(outputDir, "extract");

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(codeDir);
            Directory.CreateDirectory(dataDir);
            Directory.CreateDirectory(htmlDir);

            Action<string, int> CreateFile = (path, size) =>
            {
                using (FileStream stream = File.Create(path))
                {
                    byte[] data = new byte[size];
                    for (int i = 0; i < size; i++)
                    {
                        data[i] = (byte)i;
                    }
                    stream.Write(data, 0, size);
                }
            };

            var codeFile = Path.Combine(codeDir, "code.dat");
            var dataFiles = Enumerable.Range(0, 10).Select(n => Path.Combine(dataDir, string.Format("data{0}.dat", n))).ToArray();
            var htmlFiles = Enumerable.Range(10, 10).Select(n => Path.Combine(htmlDir, string.Format("html{0}.dat", n))).ToArray();

            CreateFile(codeFile, 1023);
            for (int i = 0; i < dataFiles.Length; ++i)
            {
                CreateFile(dataFiles[i], 1024 + i);
            }
            for (int i = 0; i < htmlFiles.Length; ++i)
            {
                CreateFile(htmlFiles[i], 1024 + i);
            }

            MakeNpdm(Path.Combine(codeDir, "main.npdm"), metaFile, descFile);

            var outputPath = Path.Combine(outputDir, "output.nsp");
            var replacePath = outputPath.Replace(".nsp", "_replaced.nsp");

            var error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type {1} --meta {2} --desc {3} --program {4} {5} --html-document {6} --save-adf --keyconfig {7}",
                outputPath,
                "Application",
                metaFile,
                descFile,
                codeDir,
                dataDir,
                htmlDir,
                config.KeyConfigFilePath));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(File.Exists(outputPath));
            List<TestFileInfo> fileList = CheckRomFsFileAlignment(outputPath, config.GetKeyConfiguration());

            // 一度全部 extract する
            SafeDeleteDirectory(testExtractDir);
            var errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", outputPath, testExtractDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            var random = new Random();
            var dataPattern = new Regex(@".*\.nca/.*/data\d+\.dat");
            var htmlPattern = new Regex(@".*\.nca/.*/html\d+\.dat");

            var allEntries = ParseListStdOutput(ExecuteProgram(string.Format("list {0} --targetregex \"{1}\"", outputPath, "."), true));
            var dataEntries = allEntries.Where(path => dataPattern.Match(path).Success);
            var htmlEntries = allEntries.Where(path => htmlPattern.Match(path).Success);

            var addDataEntry = dataEntries.ElementAt(random.Next(dataEntries.Count() - 1)) + ".dat";
            var addDataFilePath = Path.Combine(dataDir, Path.GetFileName(addDataEntry));
            CreateFile(addDataFilePath, 1024 + 17);
            var addHtmlEntry = htmlEntries.ElementAt(random.Next(htmlEntries.Count() - 1)) + ".dat";
            var addHtmlFilePath = Path.Combine(htmlDir, Path.GetFileName(addHtmlEntry));
            CreateFile(addHtmlFilePath, 1024 + 23);

            // extract したファイルの同値判定
            Func<IEnumerable<string>, List<TestFileInfo>, List<TestFileInfo>> CompareExtractFiles = (paths, oldFileList) =>
            {
                File.Delete(outputPath);
                File.Move(replacePath, outputPath);

                var newFileList = CheckRomFsFileAlignment(outputPath, config.GetKeyConfiguration());
                for (int i = 0; i < newFileList.Count; ++i)
                {
                    var newFileInfo = newFileList[i];

                    // ファイル名がソートされているか確認
                    if (0 < i && newFileList[i - 1].IsSamePartition(newFileInfo))
                    {
                        Assert.IsTrue(string.CompareOrdinal(newFileList[i - 1].FileName, newFileInfo.FileName) < 0);
                    }

                    // ファイルのアライメントを調査
                    var oldFileInfos = oldFileList.Where(fileInfo => fileInfo.FileName == newFileInfo.FileName);
                    Debug.Assert(oldFileInfos.Count() <= 1);

                    // 新規追加のファイル
                    if (oldFileInfos.Count() == 0)
                    {
                        if (newFileInfo.IsRomFs)
                        {
                            Assert.IsTrue(BitUtility.IsAligned(newFileInfo.Offset, RomFsAdfWriter.AlignmentSize));
                        }
                    }
                    // 既存のファイル
                    else
                    {
                        var oldFileInfo = oldFileInfos.First();
                        Assert.AreEqual(oldFileInfo.Offset % 512, newFileInfo.Offset % 512);
                    }
                }

                SafeDeleteDirectory(testExtractDir);

                string msg = ExecuteProgram(string.Format("extract {0} -o {1}", outputPath, testExtractDir));
                Assert.IsTrue(msg == string.Empty, msg);

                var allFiles = Directory.GetFiles(testExtractDir, "*.dat", SearchOption.AllDirectories);
                foreach (var path1 in paths)
                {
                    var path2 = allFiles.Where(path => path.EndsWith(Path.GetFileName(path1))).Single();
                    Assert.IsTrue(File.ReadAllBytes(path1).SequenceEqual(File.ReadAllBytes(path2)));
                }

                return newFileList;
            };

            // エントリの追加
            errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} add:{3} {4}", outputPath, outputDir, descFile, addDataEntry, addDataFilePath));
            Assert.IsTrue(error == string.Empty, error);
            fileList = CompareExtractFiles(dataFiles.Union(new string[] { addDataFilePath }), fileList);
            Assert.AreNotEqual(fileList.Find(info => addDataEntry.EndsWith(info.FileName)).FileName, null);
            errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} add:{3} {4}", outputPath, outputDir, descFile, addHtmlEntry, addHtmlFilePath));
            Assert.IsTrue(error == string.Empty, error);
            fileList = CompareExtractFiles(htmlFiles.Union(new string[] { addHtmlFilePath }), fileList);
            Assert.AreNotEqual(fileList.Find(info => addHtmlEntry.EndsWith(info.FileName)).FileName, null);

            allEntries = ParseListStdOutput(ExecuteProgram(string.Format("list {0} --targetregex \"{1}\"", outputPath, "."), true));
            dataEntries = allEntries.Where(path => dataPattern.Match(path).Success);
            htmlEntries = allEntries.Where(path => htmlPattern.Match(path).Success);

            var deleteDataEntry = dataEntries.Where(path => path.EndsWith(".dat.dat")).Single();
            var deleteHtmlEntry = htmlEntries.Where(path => path.EndsWith(".dat.dat")).Single();

            // エントリの削除
            errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} del:{3} null", outputPath, outputDir, descFile, deleteDataEntry));
            Assert.IsTrue(error == string.Empty, error);
            fileList = CompareExtractFiles(dataFiles, fileList);
            Assert.AreEqual(fileList.Find(info => deleteDataEntry.EndsWith(info.FileName)).FileName, null);
            errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} del:{3} null", outputPath, outputDir, descFile, deleteHtmlEntry));
            Assert.IsTrue(error == string.Empty, error);
            fileList = CompareExtractFiles(htmlFiles, fileList);
            Assert.AreEqual(fileList.Find(info => deleteHtmlEntry.EndsWith(info.FileName)).FileName, null);

            allEntries = ParseListStdOutput(ExecuteProgram(string.Format("list {0} --targetregex \"{1}\"", outputPath, "."), true));
            dataEntries = allEntries.Where(path => dataPattern.Match(path).Success);
            htmlEntries = allEntries.Where(path => htmlPattern.Match(path).Success);

            var replaceDataEntry = dataEntries.ElementAt(random.Next(dataEntries.Count()));
            var replaceDataFilePath = Path.Combine(dataDir, Path.GetFileName(replaceDataEntry));
            CreateFile(replaceDataFilePath, 256 - 11);
            var replaceHtmlEntry = htmlEntries.ElementAt(random.Next(htmlEntries.Count()));
            var replaceHtmlFilePath = Path.Combine(htmlDir, Path.GetFileName(replaceHtmlEntry));
            CreateFile(replaceHtmlFilePath, 2048 + 11);

            // エントリの差し替え
            errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} {3} {4}", outputPath, outputDir, descFile, replaceDataEntry, replaceDataFilePath));
            Assert.IsTrue(error == string.Empty, error);
            CompareExtractFiles(dataFiles, fileList);
            errorMsg = ExecuteProgram(string.Format("replace {0} -o {1} --desc {2} {3} {4}", outputPath, outputDir, descFile, replaceHtmlEntry, replaceHtmlFilePath));
            Assert.IsTrue(error == string.Empty, error);
            CompareExtractFiles(htmlFiles, fileList);
        }

        [TestMethod]
        public void TestExtractNsaveTarget()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var savedataDir = testSourceDir + "\\SaveData";
            var savedataFile = savedataDir + "\\20170103034112_01004b9000490000.nsave";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var testExtractDir = Path.Combine(outputDir, "extractnsave");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(testExtractDir);

            // 一度全部 extractnsagve する
            var errorMsg = ExecuteProgram(string.Format("extractnsave {0} -o {1}", savedataFile, testExtractDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            // 出力されるファイル一覧
            var expectedFileList = new string[]
            {
                "__BackupSaveDataInfo.xml",
                "20170103034112_01004b9000490000/user/01004b9000490000/0002/file1",
                "20170103034112_01004b9000490000/user/01004b9000490000/0002/file2",
                "20170103034112_01004b9000490000/user/01004b9000490000/0002/Directory1/file1",
                "20170103034112_01004b9000490000/user/01004b9000490000/0002/Directory1/file2",
                "20170103034112_01004b9000490000/user/01004b9000490000/0002/Directory3/dir1/dir2/file1"
            };

            var fileList = new List<string>(System.IO.Directory.GetFiles(testExtractDir, "*", System.IO.SearchOption.AllDirectories));
            Assert.IsTrue(fileList.Count == expectedFileList.Count());
            foreach (var expectedFile in expectedFileList)
            {
                int index = fileList.FindIndex(x => x.Replace("\\", "/").EndsWith(expectedFile));
                Assert.IsTrue(index >= 0);
                fileList.RemoveAt(index);
            }
            Assert.IsTrue(fileList.Count == 0);

            // 出力されるディレクトリ一覧
            var expectedDirList = new string[]
            {
                "20170103034112_01004b9000490000",
                "20170103034112_01004b9000490000/user",
                "20170103034112_01004b9000490000/user/01004b9000490000",
                "20170103034112_01004b9000490000/user/01004b9000490000/0002",
                "20170103034112_01004b9000490000/user/01004b9000490000/0002/Directory1",
                "20170103034112_01004b9000490000/user/01004b9000490000/0002/Directory2",
                "20170103034112_01004b9000490000/user/01004b9000490000/0002/Directory3",
                "20170103034112_01004b9000490000/user/01004b9000490000/0002/Directory3/dir1",
                "20170103034112_01004b9000490000/user/01004b9000490000/0002/Directory3/dir1/dir2"
            };

            var dirList = new List<string>(System.IO.Directory.GetDirectories(testExtractDir, "*", System.IO.SearchOption.AllDirectories));
            Assert.IsTrue(dirList.Count == expectedDirList.Count());
            foreach (var expectedDir in expectedDirList)
            {
                int index = dirList.FindIndex(x => x.Replace("\\", "/").EndsWith(expectedDir));
                Assert.IsTrue(index >= 0);
                dirList.RemoveAt(index);
            }
            Assert.IsTrue(dirList.Count == 0);
            SafeDeleteDirectory(testExtractDir);
        }

        [TestMethod]
        public void TestExtractTarget()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var invalidIconDir = testSourceDir + "\\Icon\\Invalid";
            var mainVisualDir = testSourceDir + "\\MainVisual";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var testExtractDir = Path.Combine(outputDir, "extract");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            var metaFile = GetIconContainApplicationMetaFile(metaDir);
            string testNpdm = dataDir + "\\main.npdm";
            MakeNpdm(testNpdm, metaFile, descFile);

            var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
            var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".bmp";

            var error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type {1} --meta {2} --desc {3} --program {4} --save-adf --keyconfig {5}",
                outputPath,
                "Application",
                metaFile,
                descFile,
                dataDir,
                keyConfigFile));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(File.Exists(outputPath));

            // 一度全部 extract する
            var errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", outputPath, testExtractDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            var allFileList = new List<string>(System.IO.Directory.GetFiles(testExtractDir, "*", System.IO.SearchOption.AllDirectories)).Select(x => x.Replace(@"\", "/"));

            SafeDeleteDirectory(testExtractDir);

            // パターンと照合（pair の first が正規表現、second がそれで出てくるファイルの拡張子（重複分は複数指定する必要がある））
            var TestPatternList = new List<Pair<string, string[]>> {
                new Pair<string, string[]>(".*\\.xml$", new string[] { ".cnmt.xml", "programinfo.xml", "nacp.xml", "cardspec.xml", "authoringtoolinfo.xml" }),
                new Pair<string, string[]>("main", new string[] { "fs0/main.npdm" }),
                new Pair<string, string[]>("\\w[pP].m", new string[] { "fs0/main.npdm" }),
                new Pair<string, string[]>("fs0/main.npdm$", new string[] { "fs0/main.npdm" }),
                new Pair<string, string[]>("^fs0/main.npdm$", null),
                new Pair<string, string[]>("fs0/main.npd$", null),
                new Pair<string, string[]>(".", allFileList.ToArray())
            };

            foreach(var TestPattern in TestPatternList)
            {
                CheckExtractTarget(outputPath, testExtractDir, TestPattern.first, TestPattern.second);
                CheckListTarget(outputPath, TestPattern.first, TestPattern.second);
            }

            // nca のテスト
            var ncaName = allFileList.Where(x => x.EndsWith("main.npdm")).Single().Replace("\\", "/").Replace("/fs0/main.npdm", "");
            ncaName = Regex.Replace(ncaName, @".*/", "");

            errorMsg = ExecuteProgram(string.Format("extract {0} -o {1} --target \"{2}\"", outputPath, testExtractDir, ncaName));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            var fileList = new List<string>(System.IO.Directory.GetFiles(testExtractDir, "*", System.IO.SearchOption.AllDirectories));
            var ncaFilePath = fileList.Single();

            // 一度全部 extract する
            var testNcaExtractDir = Path.Combine(testExtractDir, "nca");
            errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", ncaFilePath, testNcaExtractDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            allFileList = new List<string>(System.IO.Directory.GetFiles(testNcaExtractDir, "*", System.IO.SearchOption.AllDirectories)).Select(x => x.Replace(@"\", "/"));

            SafeDeleteDirectory(testNcaExtractDir);

            var TestPatternNcaList = new List<Pair<string, string[]>> {
                new Pair<string, string[]>("main", new string[] { "fs0/main.npdm" }),
                new Pair<string, string[]>("\\w[pP].m", new string[] { "fs0/main.npdm" }),
                new Pair<string, string[]>("fs0/main.npdm$", new string[] { "fs0/main.npdm" }),
                new Pair<string, string[]>("^fs0/main.npd$", null),
                new Pair<string, string[]>(".", allFileList.ToArray())
            };
            foreach (var TestPattern in TestPatternNcaList)
            {
                CheckExtractTarget(ncaFilePath, testNcaExtractDir, TestPattern.first, TestPattern.second);
                CheckListTarget(ncaFilePath, TestPattern.first, TestPattern.second);
            }

            SafeDeleteDirectory(testExtractDir);
        }

        private void CheckExtractTarget(string targetPath, string testExtractDir, string pattern, string[] expectedFileList)
        {
            // Extract のテスト
            var errorMsg = ExecuteProgram(string.Format("extract {0} -o {1} --targetregex \"{2}\"", targetPath, testExtractDir, pattern));
            if (expectedFileList == null)
            {
                Assert.IsTrue(errorMsg.IndexOf("not found") >= 0);
                return;
            }
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            var fileList = new List<string>(System.IO.Directory.GetFiles(testExtractDir, "*", System.IO.SearchOption.AllDirectories));
            Assert.IsTrue(fileList.Count == expectedFileList.Count());
            foreach (var expectedFile in expectedFileList)
            {
                int index = fileList.FindIndex(x => x.Replace("\\", "/").EndsWith(expectedFile));
                Assert.IsTrue(index >= 0);
                fileList.RemoveAt(index);
            }
            Assert.IsTrue(fileList.Count == 0);
            SafeDeleteDirectory(testExtractDir);
        }

        private List<string> ParseListStdOutput(string listOutput)
        {
            var stdMsg = listOutput.Replace("\r", "");
            stdMsg = Regex.Replace(stdMsg, @"\s+\(\d+ byte\)[\r\n]+", "\n", RegexOptions.Singleline) + "\n";    // byte 数を削除
            stdMsg = Regex.Replace(stdMsg, @"-+[\r\n]+", "", RegexOptions.Singleline).Replace(Environment.NewLine, ""); // 区切り文字列を削除
            stdMsg = Regex.Replace(stdMsg, @"^\s+$[\r\n]*", "", RegexOptions.Multiline);    // 空白行を削除
            return new List<string>(stdMsg.TrimEnd('\n').Split('\n'));
        }

        private void CheckListTarget(string targetPath, string pattern, string[] expectedFileList)
        {
            // List のテスト
            // expectedFileList に nca が含まれたら分解して足す必要がある（list の仕様）
            List<string> expectedListResult = null;
            if(expectedFileList != null)
            {
                expectedListResult = new List<string>(expectedFileList);
                var additionalNcaList = new HashSet<string>(expectedFileList.Where(x => x.IndexOf(".nca/") >= 0).Select(x => Regex.Replace(x, @"\.nca/.*", ".nca")));
                expectedListResult.AddRange(additionalNcaList);
            }

            var stdMsg = ExecuteProgram(string.Format("list {0} --targetregex \"{1}\"", targetPath, pattern), true);
            var listedEntryList = ParseListStdOutput(stdMsg);
            if (expectedListResult == null)
            {
                Assert.IsTrue(listedEntryList.Count() == 1 && String.IsNullOrEmpty(listedEntryList.Single()));
                return;
            }
            Assert.IsTrue(listedEntryList.Count() == expectedListResult.Count());

            foreach (var expectedFile in expectedListResult)
            {
                int index = listedEntryList.FindIndex(x => x.EndsWith(Regex.Replace(expectedFile, ".*/", "")));
                Assert.IsTrue(index >= 0);
                listedEntryList.RemoveAt(index);
            }
        }

        [TestMethod]
        public void TestAcidCheckWithMetaType()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var invalidIconDir = testSourceDir + "\\Icon\\Invalid";
            var mainVisualDir = testSourceDir + "\\MainVisual";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var testExtractDir = Path.Combine(outputDir, "extract");
            var testExtractEntryDir = Path.Combine(outputDir, "extractEntry");
            var testReplaceDir = Path.Combine(outputDir, "replace");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            var metaFile = GetIconContainApplicationMetaFile(metaDir);
            foreach (var isMissingDescTag in new bool[] { false, true })
            {
                foreach (var targetMetaType in new string[] { "Application", "SystemProgram" })
                {
                    string testNpdm = dataDir + "\\main.npdm";
                    MakeNpdm(testNpdm, metaFile, descFile);

                    var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
                    var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".bmp";
                    var keyConfigOption = targetMetaType == "SystemProgram" ? string.Format("--keyconfig {0}", keyConfigFile) : "";

                    var error = ExecuteProgram(string.Format(
                        "creatensp -o {0} --type {1} --meta {2} --desc {3} --program {4} --save-adf {5}",
                        outputPath,
                        targetMetaType,
                        metaFile,
                        descFile,
                        dataDir,
                        keyConfigOption));
                    Assert.IsTrue(error == string.Empty, error);
                    Assert.IsTrue(File.Exists(outputPath));
                    VerifyNsp(outputPath, keyConfigFile);

                    // programinfo.xml を extract する
                    SafeDeleteDirectory(testExtractDir);
                    Directory.CreateDirectory(testExtractDir);
                    string errorMsg = ExecuteProgram(string.Format("extractnsp {0} -o {1}", outputPath, testExtractDir));
                    Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                    var entryList = GetEntryFilePathList(testExtractDir);
                    var targetProgramInfoXml = entryList.Find(entry => entry.EndsWith("programinfo.xml"));
                    var targetProgramInfoXmlFilePath = Path.Combine(testExtractDir, targetProgramInfoXml);

                    // SaveDataOwnerIds が programinfo.xml に含まれるか確認する
                    {
                        ProgramInfoModel model;
                        using (var fs = new FileStream(targetProgramInfoXmlFilePath, FileMode.Open, FileAccess.Read))
                        {
                            var serializer = new XmlSerializer(typeof(ProgramInfoModel));
                            model = (ProgramInfoModel)serializer.Deserialize(fs);
                        }

                        var saveDataOwnerIds = model.FsAccessControlData.Entries;
                        var expectedSaveDataOwnerIds = new SaveDataOwnerIdsModel[]
                        {
                            new SaveDataOwnerIdsModel() { Accessibility = "ReadWrite", Id = "0x01ffff0000001008" },
                            new SaveDataOwnerIdsModel() { Accessibility = "Read", Id = "0x01ffff0000001009" },
                            new SaveDataOwnerIdsModel() { Accessibility = "Write", Id = "0x01ffff000000100a" },
                        };

                        Assert.IsTrue(saveDataOwnerIds.Count == expectedSaveDataOwnerIds.Count());
                        for (int i = 0; i < saveDataOwnerIds.Count; i++)
                        {
                            Assert.IsTrue(saveDataOwnerIds[i].Accessibility == expectedSaveDataOwnerIds[i].Accessibility);
                            Assert.IsTrue(saveDataOwnerIds[i].Id == expectedSaveDataOwnerIds[i].Id);
                        }
                    }

                    // programinfo.xml の ACID を値変更 or 削除した nsp を作成する
                    {
                        XmlDocument doc = new XmlDocument();
                        doc.Load(targetProgramInfoXmlFilePath);
                        XmlNode nodes = doc.SelectSingleNode("//ProgramInfo");
                        if(isMissingDescTag)
                        {
                            nodes.RemoveChild(nodes.SelectSingleNode("Desc"));
                        }
                        else
                        {
                            nodes.SelectSingleNode("Desc").InnerText = "zb+mktP3bwEBgYrF3sgxcq0gwKJzVwCSr4ApaZDU31dQNpXwJdREQOifJ9aKukjJamac9YV7Y9QL6+aKzvXKyb8qz9pHoXNomOLtZu++hygVOL7XFO23osGkIY17Npj1xuMW+wkK6retnWVvyvh8ie2VxN4NGCDauDlkW/HrhpWlTQw5KEmJiPLFtftIRnqrSrDJaST++EWzfKolD5k1kEHXGrsQwUyKsVMoe7XC52Hlghs9qYXFgZhgv26ilRirSiRq4LFab+POJuD9GOd+KJz6Y+okDulOfpX1iKY2WZV31tiSgzNfdsM46SiX6Y2pE9FVemkc5E5TIHj4loDPO5v1aNmrLVIro0pIfU4E9VvbGu+aR+W+eD7YD1f0i6YkEiVI/3BizniY2D+mpQ57Jgso5ntn3PEO1ynPiUy937f0dEQG9A10coHBCmRBc30Y7XYesQovUc6JR6KXe9E5UBzYflLWf/lyRkgyvOvas82MuXopDZTtp0/CDF4IlfYItINpPkGHBq8OpGqnIjONDSk8MhndA0xDDihn4EEh400fVvBZnkVWMGOYklci4IhyOG+aREu9VlAdDdnUuUk6AIYQsL5fz4/Q7K50OlNtBhfMSZgr/Erg0HIFYado0lVL2V4rhqehbozeUShB0GVstyHQ1uDQtcLY+u+/JoMXu3NBQ0lElAIAAAAAAAABAAAAAAABAAAAAAH/////////AUACAAAsAAAAcAIAAPoAAABwAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0HYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQRic2Q6dQVic2RjZmcEY3NybmcHZnJpZW5kOnUGZnNwLXNydgJoaWQCaHRjB2h0Yzp0ZW52A2h0Y3MFaHdvcHVzAmlycwRsZG46dQVsZHI6cm8BbG0EbWlpOnUDbW06dQduZmM6bWY6dQduZmM6dXNlcgduZnA6dXNlcgVuaWZtOnUEbnNkOnUCbnRjBG52ZHJ2A3BjdGwDcGw6dQZwcmVwbzp1AnNldAdzZmRuc3JlcwJzc2wFdGltZTp1A3RzcG0Ddmk6dQAAAAAAALdzAALP//8f7/9/MI8BAEAPAABw/18AAP8/GAD/fwAC//8CAA==";
                        }
                        doc.Save(targetProgramInfoXmlFilePath);
                    }

                    // desc 違いの replace でエラーになることを確認
                    if (targetMetaType == "Application" && !isMissingDescTag)
                    {
                        var noFlagDescFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\PerformanceEvaluationApplication.desc";
                        var stdMsg = ExecuteProgram(string.Format("list {0} --targetregex \"{1}\"", outputPath, "/data.dat$"), true);
                        var datPath = ParseListStdOutput(stdMsg).Single();

                        // main.npdm 自体の replace
                        {
                            stdMsg = ExecuteProgram(string.Format("list {0} --targetregex \"{1}\"", outputPath, "/main.npdm$"), true);
                            var targetMainNpdm = ParseListStdOutput(stdMsg).Single();

                            errorMsg = ExecuteProgram(string.Format("extract {0} --target {1} -o {2}", outputPath, targetMainNpdm, testReplaceDir));
                            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                            var targetMainNpdmFilePath = Path.Combine(testReplaceDir, "main.npdm");

                            // desc 違いは失敗し、desc が同じなら成功
                            errorMsg = ExecuteProgram(string.Format("replace {0} {1} {2} --desc {3} -o {4} {5}", outputPath, targetMainNpdm, targetMainNpdmFilePath, noFlagDescFile, testReplaceDir, keyConfigOption));
                            Assert.IsTrue(errorMsg.IndexOf("Wrong desc file. Acid in the desc file and main.npdm don't match.") >= 0, errorMsg);
                            errorMsg = ExecuteProgram(string.Format("replace {0} {1} {2} --desc {3} -o {4} {5}", outputPath, targetMainNpdm, targetMainNpdmFilePath, descFile, testReplaceDir, keyConfigOption));
                            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                        }

                        // それ以外は例外
                        errorMsg = ExecuteProgram(string.Format("replace {0} {1} {2} --desc {3} -o {4} {5}", outputPath, datPath, targetProgramInfoXmlFilePath, noFlagDescFile, testReplaceDir, keyConfigOption));
                        Assert.IsTrue(errorMsg.IndexOf("Wrong desc file. Acid in the desc file and main.npdm don't match.") >= 0, errorMsg);
                    }

                    // createfs で nsp を作成する
                    var modifiedNspPath = outputPath.Replace(".nsp", "_replaced.nsp");
                    errorMsg = ExecuteProgram(string.Format("createfs -o {0} --format partitionfs {1}", modifiedNspPath, testExtractDir));
                    Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                    // replace で ACID のチェックを発生させる
                    {
                        var testExtractDir2 = Path.Combine(outputDir, "extract2");
                        SafeDeleteDirectory(testExtractDir2);
                        errorMsg = ExecuteProgram(string.Format("extract {0} -o {1} {2}", outputPath, testExtractDir2, keyConfigOption));
                        Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                        entryList = GetEntryFilePathList(testExtractDir2);
                        var testData = entryList.Find(entry => entry.EndsWith("/data.dat"));
                        var testDataFilePath = Path.Combine(testExtractDir2, testData);
                        // データを加工
                        byte[] data = File.ReadAllBytes(testDataFilePath);
                        data[0] += 1;
                        File.WriteAllBytes(testDataFilePath, data);

                        errorMsg = ExecuteProgram(string.Format("replace {0} {1} {2} --desc {3} -o {4} {5}", modifiedNspPath, testData, testDataFilePath, descFile, testReplaceDir, keyConfigOption));
                    }

                    if (targetMetaType == "SystemProgram" && isMissingDescTag)
                    {
                        Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                    }
                    else
                    {
                        Assert.IsTrue(errorMsg.IndexOf("Invalid programinfo. Acids between programinfo.xml and main.npdm don't match.") >= 0, errorMsg);
                    }
                }
            }
        }

        [TestMethod]
        public void TestAcidCheckWithPatch()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var metaFile = testSourceDir + "\\ApplicationMeta\\describe_all.nmeta";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var descFileV1 = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\PerformanceEvaluationApplication.desc";
            var iconPath = testSourceDir + "\\Icon\\describe_all.bmp";

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            var outputPathV0 = outputDir + "\\v0.nsp";
            var outputPathV1 = outputDir + "\\v1.nsp";
            var outputPathV1Patch = outputDir + "\\v1.patch.nsp";

            // v0
            {
                var npdm = dataDir + "\\main.npdm";
                MakeNpdm(npdm, metaFile, descFile);
                var error = ExecuteProgram(string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4}", outputPathV0, metaFile, descFile, dataDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPathV0));
            }

            // v1
            {
                var metaFileV1 = outputDir + "\\describe_all_v1.nmeta";
                MakeSpecifiedVersionMetaFile(metaFile, metaFileV1, 1);
                var npdm = dataDir + "\\main.npdm";
                MakeNpdm(npdm, metaFileV1, descFileV1);
                var error = ExecuteProgram(string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4}", outputPathV1, metaFileV1, descFileV1, dataDir ,iconPath));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPathV1));
            }

            // makepatch
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", outputPathV1Patch, descFileV1, outputPathV0, outputPathV1));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPathV1Patch));
            }

            // makepatch で v1 と違う desc ファイルは指定できない
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", outputPathV1Patch, descFile, outputPathV0, outputPathV1));
                Assert.IsTrue(error.IndexOf("Wrong desc file. Acid in the desc file and main.npdm don't match.") >= 0, error);
            }
        }

        [TestMethod]
        public void TestExecutionCreateApplicationNsp()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var invalidIconDir = testSourceDir + "\\Icon\\Invalid";
            var mainVisualDir = testSourceDir + "\\MainVisual";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var testExtractDir = Path.Combine(outputDir, "extract");
            var testExtractEntryDir = Path.Combine(outputDir, "extractEntry");
            var testReplaceDir = Path.Combine(outputDir, "replace");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(testExtractDir);
            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            bool once = true;
            foreach (var file in Directory.EnumerateFiles(metaDir))
            {
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, file, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";

                var format = "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf";
                if (string.Compare(Path.GetFileName(file), "no_application.nmeta") == 0)
                {
                    // Applicationエントリがないnmetaを使用した場合、アイコンを設定するとエラーとなるので--Iconオプションを付けない
                    iconPath = null;
                }
                else
                {
                    format += " --icon AmericanEnglish {4} Japanese {4}";
                }

                var error = ExecuteProgram(string.Format(
                        format,
                        outputPath,
                        file,
                        descFile,
                        dataDir,
                        iconPath));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));

                if (string.Compare(Path.GetFileName(file), "describe_all.nmeta") == 0 ||
                    string.Compare(Path.GetFileName(file), "describe_all_with_icon.nmeta") == 0)
                {
                    CheckRequiredSystemVersion(outputPath, 0x1);
                }
                else
                {
                    CheckRequiredSystemVersion(outputPath, NintendoContentMeta.GetRequiredSystemVersion());
                }

                if (once)
                {
                    // extract -> creatensp して一致を確認

                    Directory.CreateDirectory(testExtractDir);
                    error = ExecuteProgram(string.Format(
                        "extractnsp -o {0} {1}",
                        testExtractDir,
                        outputPath
                        ));
                    Assert.IsTrue(error == string.Empty, error);

                    string programNcaPath;
                    using (var nsp = new NintendoSubmissionPackageReader(outputPath))
                    {
                        var meta = ArchiveReconstructionUtils.ReadContentMetaInNsp(nsp).First();
                        var content = meta.ContentList.Find(v => v.Type == "Program");
                        programNcaPath = Path.Combine(testExtractDir, content.Id + ".nca");
                    }

                    var outputNspFromNcaPath = outputPath.Replace(".nsp", "_fromNca.nsp");

                    format = "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf";
                    if (string.Compare(Path.GetFileName(file), "no_application.nmeta") == 0)
                    {
                        // Applicationエントリがないnmetaを使用した場合、アイコンを設定するとエラーとなるので--Iconオプションを付けない
                        iconPath = null;
                    }
                    else
                    {
                        format += " --icon AmericanEnglish {4} Japanese {4}";
                    }

                    error = ExecuteProgram(string.Format(
                        format,
                        outputNspFromNcaPath,
                        file,
                        descFile,
                        programNcaPath,
                        iconPath));
                    Assert.IsTrue(error == string.Empty, error);
                    Assert.IsTrue(File.Exists(outputNspFromNcaPath));
                    VerifyNsp(outputNspFromNcaPath, keyConfigFile);

                    CheckGetNspProperty(outputPath);

                    var beforeDir = testExtractDir + "\\before";
                    var afterDir = testExtractDir + "\\after";
                    Directory.CreateDirectory(beforeDir);
                    Directory.CreateDirectory(afterDir);

                    error = ExecuteProgram(string.Format("extract {0} -o {1} {2}", outputPath, beforeDir, keyConfigFile));
                    Assert.IsTrue(error == string.Empty, error);
                    error = ExecuteProgram(string.Format("extract {0} -o {1} {2}", outputNspFromNcaPath, afterDir, keyConfigFile));
                    Assert.IsTrue(error == string.Empty, error);

                    var fileNum1 = Directory.GetFiles(beforeDir, "*", SearchOption.AllDirectories);
                    var fileNum2 = Directory.GetFiles(afterDir, "*", SearchOption.AllDirectories);
                    foreach (var beforeFile in Directory.EnumerateFiles(beforeDir, "*", SearchOption.AllDirectories))
                    {
                        var afterFile = beforeFile.Replace("before", "after");
                        Assert.IsTrue(File.Exists(afterFile));
                        Assert.IsTrue(File.Exists(beforeFile));

                        if (beforeFile.EndsWith("programinfo.xml"))
                        {
                            // ncaから作成したnspではprograminfo.xmlのDescFileNameが設定されないので比較しない
                            continue;
                        }
                        CheckFileDiff(beforeFile, afterFile, 0);
                    }

                    SafeDeleteDirectory(testExtractDir);
                    Directory.CreateDirectory(testExtractDir);
                }

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);

                // Extract のテスト
                if (once)
                {
                    once = false;
                    string[] entryRuleStringArray = new string[] {
                        @".*\.nca/fs0/data\.dat",
                        @".*\.nca/fs0/main\.npdm",
                        @".*\.nca/fs2/NintendoLogo\.png",
                        @".*\.nca/fs2/StartupMovie\.gif",
                        @".*\.nca/fs0/control\.nacp",
                        @".*\.nca/fs0/icon_AmericanEnglish\.dat",
                        @".*\.nca/fs0/icon_Japanese\.dat",
                        @".*\.cnmt\.nca/fs0/Application_0005000c10000001.cnmt",
                        @".*\.cnmt.xml",
                        @".*\.nacp.xml",
                        @".*\.programinfo.xml",
                        @"cardspec.xml",
                        @".*\.nx\.AmericanEnglish\.jpg",
                        @".*\.nx\.Japanese\.jpg",
                        @".*\.raw\.AmericanEnglish\.jpg",
                        @".*\.raw\.Japanese\.jpg",
                        @"authoringtoolinfo.xml",
                    };
                    TestExtractWithNsp(outputPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);
                    TestNacpFile(testExtractDir);

                    // first: pattern, second: isProgramContent
                    Tuple<Regex, bool>[] ReplaceEntryInNspTestList = new Tuple<Regex, bool>[] {
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/data.dat"), true),
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/control.nacp"), false),
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/Application_0005000c10000001.cnmt"), false),
                        new Tuple<Regex, bool>(new Regex(@".*\.raw\.Japanese\.jpg"), false)
                    };
                    var testCombination = new List<int[]>() { new int[] { 2 }, new int[] { 3 }, new int[] { 0, 1 },
                        Enumerable.Range(0, ReplaceEntryInNspTestList.Count()).ToArray() };
                    TestReplaceWithNspOrNca(outputPath, testReplaceDir, testExtractDir, testExtractEntryDir, ReplaceEntryInNspTestList, testCombination);

                    // nca 差し替え
                    var entryList = GetEntryFilePathList(testExtractDir);
                    var targetName = "data.dat";
                    var ncaName = entryList.Find(entry => entry.EndsWith(targetName)).Replace("/fs0/" + targetName, "");
                    string tempReplaceDir = Path.Combine(outputDir, "replace_nca");

                    TestReplaceNcaInNsp(outputPath, tempReplaceDir, ncaName, targetName);
                }

                TestContentMeta testMeta = new TestContentMeta();
                testMeta.SetFromApplicationMetaFile(file, "/NintendoSdkMeta", 128); // Application = 128
                CheckContentMeta(outputPath, "Application", testMeta, keyConfigFile);

                TestCreateNspMeta(outputPath, testExtractDir, file, "Application", iconPath);

                // 製品化
                {
                    errorMsg = ExecuteProgram(string.Format("prodencryption --no-check --keyconfig {0} --requiredSystemVersion {3} -o {1} {2}", keyConfigFile, outputDir, outputPath, NintendoContentMeta.GetRequiredSystemVersion() + 1));
                    Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                    CheckResultXml(outputDir, outputPath);
                    CheckRequiredSystemVersion(outputDir, outputPath, NintendoContentMeta.GetRequiredSystemVersion() + 1);

                    ExtractAndShowNsp(outputPath.Replace(".nsp", "_prod.nsp"));

                    // requiredSystemVersion
                    errorMsg = ExecuteProgram(string.Format("prodencryption --no-check --keyconfig {0} -o {1} {2}", keyConfigFile, outputDir, outputPath));
                    Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                    if (string.Compare(Path.GetFileName(file), "describe_all.nmeta") == 0 ||
                        string.Compare(Path.GetFileName(file), "describe_all_with_icon.nmeta") == 0)
                    {
                        CheckRequiredSystemVersion(outputDir, outputPath, 0x1);
                    }
                    else
                    {
                        CheckRequiredSystemVersion(outputDir, outputPath, NintendoContentMeta.GetRequiredSystemVersion());
                    }

                    if (string.Compare(Path.GetFileName(file), "describe_all.nmeta") != 0 &&
                        string.Compare(Path.GetFileName(file), "describe_all_with_icon.nmeta") != 0)
                    {
                        errorMsg = ExecuteProgram(string.Format("prodencryption --no-check --keyconfig {0} --requiredSystemVersion {3} -o {1} {2}", keyConfigFile, outputDir, outputPath, NintendoContentMeta.GetRequiredSystemVersion() - 1));
                        Assert.IsTrue(errorMsg.IndexOf(string.Format("Specified required system version (= {0}) is older than that of application", NintendoContentMeta.GetRequiredSystemVersion() - 1)) >= 0);
                    }
                }
            }

            // データディレクトリが指定されている nsp
            var targetAppMeta = new List<string>()
            {
                metaDir + "\\describe_all_with_icon.nmeta",
                metaDir + "\\no_application.nmeta"
            };
            foreach (var file in targetAppMeta)
            {
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, file, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";

                var format = "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {3} --save-adf";
                if (string.Compare(Path.GetFileName(file), "no_application.nmeta") == 0)
                {
                    // Applicationエントリがないnmetaを使用した場合、アイコンを設定するとエラーとなるので--Iconオプションを付けない
                    iconPath = null;
                }
                else
                {
                    format += " --icon AmericanEnglish {4} Japanese {4}";
                }

                var error = ExecuteProgram(string.Format(
                    format,
                    outputPath,
                    file,
                    descFile,
                    dataDir,
                    iconPath));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);

                // Extract のテスト
                if (once)
                {
                    once = false;
                    string[] entryRuleStringArray = new string[] {
                        @".*\.nca/fs0/data\.dat",
                        @".*\.nca/fs0/main\.npdm",
                        @".*\.nca/fs0/control\.nacp",
                        @".*\.nca/fs0/icon_AmericanEnglish\.dat",
                        @".*\.nca/fs0/icon_Japanese\.dat",
                        @".*\.nca/fs1/data\.dat",
                        @".*\.nca/fs1/main\.npdm",
                        @".*\.nca/fs2/NintendoLogo\.png",
                        @".*\.nca/fs2/StartupMovie\.gif",
                        @".*\.cnmt\.nca/fs0/Application_0005000c10000001.cnmt",
                        @".*\.cnmt.xml",
                        @".*\.nacp.xml",
                        @"cardspec.xml",
                        @"[0-9a-fA-F]*\.tik",
                        @"[0-9a-fA-F]*\.cert",
                        @".*\.nx\.AmericanEnglish\.jpg",
                        @".*\.nx\.Japanese\.jpg",
                        @".*\.raw\.AmericanEnglish\.jpg",
                        @".*\.raw\.Japanese\.jpg",
                        @"authoringtoolinfo.xml",
                    };
                    TestExtractWithNsp(outputPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);
                    TestNacpFile(testExtractDir);

                    // first: pattern, second: isProgramContent
                    Tuple<Regex, bool>[] ReplaceEntryInNspTestList = new Tuple<Regex, bool>[] {
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/data.dat"), true),
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/control.nacp"), false),
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/Application_0005000c10000001.cnmt"), false),
                    };
                    var testCombination = new List<int[]>() { Enumerable.Range(0, ReplaceEntryInNspTestList.Count()).ToArray() };
                    TestReplaceWithNspOrNca(outputPath, testReplaceDir, testExtractDir, testExtractEntryDir, ReplaceEntryInNspTestList, testCombination);
                }

                TestContentMeta testMeta = new TestContentMeta();
                testMeta.SetFromApplicationMetaFile(file, "/NintendoSdkMeta", 128); // Application = 128
                CheckContentMeta(outputPath, "Application", testMeta, keyConfigFile);

                TestCreateNspMeta(outputPath, testExtractDir, file, "Application", iconPath);

                // 製品化
                errorMsg = ExecuteProgram(string.Format("prodencryption --no-check --keyconfig {0} --requiredSystemVersion {3} -o {1} {2}", keyConfigFile, outputDir, outputPath, NintendoContentMeta.GetRequiredSystemVersion() + 1));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                CheckResultXml(outputDir, outputPath);
                CheckRequiredSystemVersion(outputDir, outputPath, NintendoContentMeta.GetRequiredSystemVersion() + 1);

                ExtractAndShowNsp(outputPath.Replace(".nsp", "_prod.nsp"));
            }

            // フィルタ付きでテスト
            var sampleFiles = new FilterTestUtil.GenerateSampleFiles();
            sampleFiles.Generate(outputDir);

            // testIndex: 0 は通常のフィルタ、testIndex: 3 は一時全削除フィルタ
            foreach (var testIndex in new int[] { 0, 3 })
            {
                {
                    var file = GetDefaultApplicationMetaFile(metaDir);
                    string testNpdm = dataDir + "\\main.npdm";
                    MakeNpdm(testNpdm, file, descFile);

                    // TestExecutionCreateNcaFiltered で作られている可能性があるので FilterTest 側にも作っておく
                    MakeNpdm(sampleFiles.testDirPath + "\\main.npdm", file, descFile);

                    var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                    var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";
                    var mainVisualPath = mainVisualDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".png";

                    var format = "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {4} --save-adf --filter {5} --icon AmericanEnglish {6} Japanese {6}";

                    string command = string.Format(format,
                        FilterDescription.FormatFilePathForFilter(outputPath, Directory.GetCurrentDirectory()),
                        file,
                        descFile,
                        dataDir,
                        sampleFiles.testDirPath,
                        sampleFiles.fdfPathList[testIndex],
                        iconPath);
                    var error = ExecuteProgram(command);
                    Assert.IsTrue(error == string.Empty, error);
                    Assert.IsTrue(File.Exists(outputPath));
                    VerifyNsp(outputPath, keyConfigFile);

                    string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                    Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                    ExtractAndShowNsp(outputPath);

                    var outputRomAdfFile = outputPath + ".c0.Program.nca.rom.adf";
                    int numEntries = new RomFsAdfReader(outputRomAdfFile).GetFileSystemInfo().entries.Count;
                    if (testIndex == 0)
                    {
                        Assert.IsTrue(numEntries == (1 + sampleFiles.fileCount - sampleFiles.fdfTest0FileRemoveCount + sampleFiles.fdfTest0FileRemoveExceptionCount));
                    }
                    else if (testIndex == 3)
                    {
                        Assert.IsTrue(numEntries == sampleFiles.fdfTest3FileRemainCount);
                    }
                }
            }

            // NX アイコン直接指定
            {
                var file = GetDefaultApplicationMetaFile(metaDir);
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, file, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";
                var nxIconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".jpg";
                File.Delete(outputPath);

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4} --nx-icon AmericanEnglish {5} Japanese {5} --save-adf",
                    outputPath,
                    file,
                    descFile,
                    dataDir,
                    iconPath,
                    nxIconPath));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);
            }

            // アイコンが NG の場合
            {
                var iconSetList = new List<Tuple<string, string>>();
                iconSetList.Add(new Tuple<string, string>(invalidIconDir + "\\invalid_size.bmp", iconDir + "\\describe_all.jpg"));
                iconSetList.Add(new Tuple<string, string>(iconDir + "\\describe_all.bmp", invalidIconDir + "\\nx_progressive.jpg"));

                foreach (var iconSet in iconSetList)
                {
                    var file = GetDefaultApplicationMetaFile(metaDir);
                    string testNpdm = dataDir + "\\main.npdm";
                    MakeNpdm(testNpdm, file, descFile);

                    var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                    var iconPath = iconSet.Item1;
                    var nxIconPath = iconSet.Item2;
                    string standardError;
                    string standardOut;
                    ExecuteProgram(out standardError, out standardOut, string.Format(
                        "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4} --nx-icon AmericanEnglish {5} Japanese {5} --save-adf",
                        outputPath,
                        file,
                        descFile,
                        dataDir,
                        iconPath,
                        nxIconPath));
                    Assert.IsTrue(standardError != string.Empty, standardError);
                    if (Path.GetFileName(nxIconPath) == "nx_progressive.jpg")
                    {
                        // NXアイコンがプログレッシブJpegの場合はUnpublishableErrorが発生してnspは作成されない
                        Assert.IsTrue(standardOut.Contains("[Error] (Issue 10-815)"));
                        Assert.IsFalse(File.Exists(outputPath));
                    }
                    else
                    {
                        VerifyNsp(outputPath, keyConfigFile);
                    }
                }
            }

            // 大きなサイズのアイコン
            {
                var iconSetList = new List<Tuple<string, string>>();
                iconSetList.Add(new Tuple<string, string>(iconDir + "\\describe_all.bmp", iconDir + "\\large_nx_icon.jpg"));

                foreach (var iconSet in iconSetList)
                {
                    var file = GetDefaultApplicationMetaFile(metaDir);
                    string testNpdm = dataDir + "\\main.npdm";
                    MakeNpdm(testNpdm, file, descFile);

                    var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                    var iconPath = iconSet.Item1;
                    var nxIconPath = iconSet.Item2;
                    File.Delete(outputPath);

                    var error = ExecuteProgram(string.Format(
                        "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4} --nx-icon AmericanEnglish {5} Japanese {5} --nx-icon-max-size 131072 --save-adf",
                        outputPath,
                        file,
                        descFile,
                        dataDir,
                        iconPath,
                        nxIconPath));
                    if (string.Compare(Path.GetFileName(file), "no_application.nmeta") == 0)
                    {
                        // Applicationエントリがないnmetaを使用した場合、アイコンを設定するとエラーとなる
                        // iconオプションのテストなのでiconを指定しないテストは実施しない
                        Assert.IsTrue(error.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0, error);
                        Assert.IsFalse(File.Exists(outputPath));
                        continue;
                    }
                    else
                    {
                        Assert.IsTrue(error == string.Empty, error);
                        Assert.IsTrue(File.Exists(outputPath));
                    }
                    VerifyNsp(outputPath, keyConfigFile);

                    var expectFailedError = ExecuteProgram(string.Format(
                        "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4} --nx-icon AmericanEnglish {5} Japanese {5} --save-adf",
                        outputPath,
                        file,
                        descFile,
                        dataDir,
                        iconPath,
                        nxIconPath));
                    Assert.IsTrue(expectFailedError != string.Empty, expectFailedError);
                }
            }

            {
                var file = GetDefaultApplicationMetaFile(metaDir);
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, file, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";
                var nxIconPath = iconDir + "\\describe_all.jpg";
                File.Delete(outputPath);

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4} --nx-icon AmericanEnglish {5} Japanese {5} --save-adf",
                    outputPath,
                    file,
                    descFile,
                    dataDir,
                    iconPath,
                    nxIconPath));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                VerifyNsp(outputPath, null);
            }

            // --nx-icon にはあるけど、 --icon には無い言語があったときにはエラー
            {
                string metaFile = GetDefaultApplicationMetaFile(metaDir);
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".bmp";
                var nxIconPath = iconDir + "\\describe_all.jpg";
                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon French {4} --nx-icon AmericanEnglish {5} Japanese {5} --save-adf",
                    outputPath,
                    metaFile,
                    descFile,
                    dataDir,
                    iconPath,
                    nxIconPath));
                Assert.IsTrue(error.IndexOf("Languages (AmericanEnglish, Japanese) are specified by --nx-icons, but they are not specified by --icon option.") >= 0, error);
            }

            // meta ファイルに icon のパスを指定
            {
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(GetDefaultApplicationMetaFile(metaDir)) + ".bmp";
                TestMetaFile metaFile = MakeIconPathMetaFile(iconPath, outputDir);

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                    outputPath,
                    metaFile.GetFilePath(),
                    descFile,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                // Extract のテスト
                string[] entryRuleStringArray = new string[] {
                    @".*\.nca/fs0/data\.dat",
                    @".*\.nca/fs0/main\.npdm",
                    @".*\.nca/fs2/NintendoLogo\.png",
                    @".*\.nca/fs2/StartupMovie\.gif",
                    @".*\.nca/fs0/control\.nacp",
                    @".*\.nca/fs0/icon_AmericanEnglish\.dat",
                    @".*\.cnmt\.nca/fs0/Application_0100000000002802.cnmt",
                    @".*\.cnmt.xml",
                    @".*\.nacp.xml",
                    @".*\.programinfo.xml",
                    @"cardspec.xml",
                    @".*\.nx\.AmericanEnglish\.jpg",
                    @".*\.raw\.AmericanEnglish\.jpg",
                    @"authoringtoolinfo.xml",
                };
                SafeDeleteDirectory(testExtractDir);
                TestExtractWithNsp(outputPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);
                TestNacpFile(testExtractDir);
            }

            // meta ファイルに icon のパスを指定(環境変数利用)
            {
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(GetDefaultApplicationMetaFile(metaDir)) + ".bmp";
                var envVariableName = "TEST_ICON_PATH";
                TestMetaFile metaFile = MakeIconPathMetaFileUsingEnvironmentVariable(envVariableName, outputDir);
                Environment.SetEnvironmentVariable(envVariableName, iconPath, EnvironmentVariableTarget.Process);

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                    outputPath,
                    metaFile.GetFilePath(),
                    descFile,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                // Extract のテスト
                string[] entryRuleStringArray = new string[] {
                    @".*\.nca/fs0/data\.dat",
                    @".*\.nca/fs0/main\.npdm",
                    @".*\.nca/fs2/NintendoLogo\.png",
                    @".*\.nca/fs2/StartupMovie\.gif",
                    @".*\.nca/fs0/control\.nacp",
                    @".*\.nca/fs0/icon_AmericanEnglish\.dat",
                    @".*\.cnmt\.nca/fs0/Application_0100000000002802.cnmt",
                    @".*\.cnmt.xml",
                    @".*\.nacp.xml",
                    @".*\.programinfo.xml",
                    @"cardspec.xml",
                    @".*\.nx\.AmericanEnglish\.jpg",
                    @".*\.raw\.AmericanEnglish\.jpg",
                    @"authoringtoolinfo.xml",
                };
                SafeDeleteDirectory(testExtractDir);
                TestExtractWithNsp(outputPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);
                TestNacpFile(testExtractDir);
            }

            // meta ファイルに icon のパス指定がある場合に、 --icon オプションが無視される
            {
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(GetDefaultApplicationMetaFile(metaDir)) + ".bmp";
                TestMetaFile metaFile = MakeIconPathMetaFile(iconPath, outputDir);

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --icon French {4}",
                    outputPath,
                    metaFile.GetFilePath(),
                    descFile,
                    dataDir,
                    iconPath));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                // Extract のテスト
                string[] entryRuleStringArray = new string[] {
                    @".*\.nca/fs0/data\.dat",
                    @".*\.nca/fs0/main\.npdm",
                    @".*\.nca/fs2/NintendoLogo\.png",
                    @".*\.nca/fs2/StartupMovie\.gif",
                    @".*\.nca/fs0/control\.nacp",
                    @".*\.nca/fs0/icon_AmericanEnglish\.dat",
                    @".*\.cnmt\.nca/fs0/Application_0100000000002802.cnmt",
                    @".*\.cnmt.xml",
                    @".*\.nacp.xml",
                    @".*\.programinfo.xml",
                    @"cardspec.xml",
                    @".*\.nx\.AmericanEnglish\.jpg",
                    @".*\.raw\.AmericanEnglish\.jpg",
                    @"authoringtoolinfo.xml",
                };
                SafeDeleteDirectory(testExtractDir);
                TestExtractWithNsp(outputPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);
                TestNacpFile(testExtractDir);
            }

            // title の言語に対応する icon が無いと error-unpublishable でエラーになる
            {
                var legalInfo = MakeLegalInfoZipfile(outputDir);
                // meta ファイルに --icon で指定した言語に対応するタイトルがない
                {
                    var file = GetDefaultApplicationMetaFile(metaDir);
                    var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(GetDefaultApplicationMetaFile(metaDir)) + ".bmp";
                    string testNpdm = dataDir + "\\main.npdm";
                    MakeNpdm(testNpdm, file, descFile);

                    var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";

                    var error = ExecuteProgram(string.Format(
                        "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --icon AmericanEnglish {4} Japanese {4} French {4} German {4} --error-unpublishable --legal-information {5}",
                        outputPath,
                        file,
                        descFile,
                        dataDir,
                        iconPath,
                        legalInfo));
                    Assert.IsTrue(error.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0, error);
                }

                // meta ファイルのタイトルの中にアイコンのパスがない
                {
                    var file = GetDefaultApplicationMetaFile(metaDir);
                    string testNpdm = dataDir + "\\main.npdm";
                    MakeNpdm(testNpdm, file, descFile);

                    var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";

                    var error = ExecuteProgram(string.Format(
                        "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --error-unpublishable --legal-information {4}",
                        outputPath,
                        file,
                        descFile,
                        dataDir,
                        legalInfo));
                    Assert.IsTrue(error.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0, error);
                }
            }

            // RequiredSystemVersionが正しいかを確認するテスト
        }

        [TestMethod]
        public void TestExecutionCreateApplicationWithFilter()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var fdfSpecifiedMetaFile = env.SourceDir + "\\FilterTest\\filter_specified.nmeta";
            var fdfFileName = "application.fdf";
            var fdfNotSpecifiedMetaFile = env.SourceDir + "\\FilterTest\\filter_not_specified.nmeta";

            var iconPath = env.SourceDir + "\\Icon\\describe_all.bmp";
            var legalInformationZip = MakeLegalInfoZipfile(env.OutputDir);
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");
            MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

            var dataDir = Path.Combine(env.OutputDir, "data");
            SafeDeleteDirectory(dataDir);
            Directory.CreateDirectory(dataDir);
            const int DataCount = 3;
            for(int i = 0; i < DataCount; ++i)
            {
                var filePath = Path.Combine(dataDir, string.Format("Data_{0}.bin", i));
                using (FileStream stream = File.Create(filePath))
                {
                    int fileSize = 1024;
                    byte[] data = new byte[fileSize];
                    for (int j = 0; j < fileSize; j++)
                    {
                        data[j] = (byte)j;
                    }
                    stream.Write(data, 0, fileSize);
                }
            }

            var fdfSrcPath = Path.Combine(env.SourceDir, "FilterTest\\" + fdfFileName);
            var fdfDestPath = Path.Combine(Directory.GetCurrentDirectory(), fdfFileName);
            File.Copy(fdfSrcPath, fdfDestPath, true);

            Action<string, string, string> createNsp = (nsp, metaFile, addCmd) =>
            {
                var cmd = string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {5} --icon AmericanEnglish {4} Japanese {4}", nsp, metaFile, env.DefaultDescFile, env.TestCodeDir, iconPath, dataDir) + addCmd;
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(nsp, null);


                var patternList = new string[]
                {
                    @".*\.nca/fs0/code\.dat",
                    @".*\.nca/fs0/main\.npdm",
                    @".*\.nca/fs1/Data_0\.bin",
                    @".*\.nca/fs1/Data_2\.bin",
                    @".*\.nca/fs2/NintendoLogo\.png",
                    @".*\.nca/fs2/StartupMovie\.gif",
                    @".*\.nca/fs0/control\.nacp",
                    @".*\.nca/fs0/icon_AmericanEnglish\.dat",
                    @".*\.nca/fs0/icon_Japanese\.dat",
                    @".*\.cnmt\.nca/fs0/Application_0005000c10000001.cnmt",
                    @".*\.cnmt.xml",
                    @".*\.nacp.xml",
                    @".*\.programinfo.xml",
                    @"cardspec.xml",
                    @".*\.nx\.AmericanEnglish\.jpg",
                    @".*\.raw\.AmericanEnglish\.jpg",
                    @".*\.nx\.Japanese\.jpg",
                    @".*\.raw\.Japanese\.jpg",
                    @"authoringtoolinfo.xml",
                };

                var extractDir = Path.Combine(env.OutputDir, "\\extract");
                var extractEntryDir = Path.Combine(env.OutputDir, "\\extractEntry");
                SafeDeleteDirectory(extractDir);
                SafeDeleteDirectory(extractEntryDir);
                TestExtractWithNsp(nsp, extractDir, extractEntryDir, patternList);
            };

            Action<string, string, string> createNspExpectError = (nsp, metaFile, addCmd) =>
            {
                var cmd = string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {5} --icon AmericanEnglish {4} Japanese {4}", nsp, metaFile, env.DefaultDescFile, env.TestCodeDir, iconPath, dataDir) + addCmd;
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                var errorMsg = ExecuteProgram(cmd);
                Assert.IsTrue(errorMsg.Contains("It is impossible to specify fdf file from both nmeta and command line option."));
            };


            // FDF のパスを を nmeta の中で指定
            var filterSpecified = env.OutputDir + "\\filterSpecified.nsp";
            createNsp(filterSpecified, fdfSpecifiedMetaFile, " --keygeneration 0");

            // FDF のパスをコマンドラインで指定
            var filterNotSpecified = env.OutputDir + "\\filterNotSpecified.nsp";
            createNsp(filterNotSpecified, fdfNotSpecifiedMetaFile, string.Format(" --keygeneration 0 --filter {0}", fdfDestPath));

            // FDF を nmeta, コマンドライン両方で指定 ==> エラー
            createNspExpectError(filterSpecified, fdfSpecifiedMetaFile,
                string.Format(" --keygeneration 0 --filter {0}", fdfDestPath));
        }

        [TestMethod]
        public void TestExecutionCreateApplicationNspIncludingEmptyDirectory()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var metaFilePath = env.SourceDir + "\\FilterTest\\filter_not_specified.nmeta";
            var fdfFilePath = env.SourceDir + "\\FilterTest\\application.fdf";

            var iconPath = env.SourceDir + "\\Icon\\describe_all.bmp";
            var legalInformationZip = MakeLegalInfoZipfile(env.OutputDir);
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");
            MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

            var dataDir = Path.Combine(env.OutputDir, "data");
            SafeDeleteDirectory(dataDir);
            Directory.CreateDirectory(dataDir);

            // 下記構成のツリーを作成。フィルタで Data_1.bin は除外。
            // - data/0/file/Data_0.bin
            // - data/1/file/Data_1.bin
            // - data/2/file/Data_2.bin
            // - data/3/file/
            const int DirectoryCount = 4;
            for(int i = 0; i < DirectoryCount; ++i)
            {
                var dirPath = Path.Combine(dataDir, string.Format("{0}", i));
                Directory.CreateDirectory(dirPath);
                var dirSubPath = Path.Combine(dirPath, "file");
                Directory.CreateDirectory(dirSubPath);
            }

            const int DataCount = 3;
            for(int i = 0; i < DataCount; ++i)
            {
                var filePath = Path.Combine(dataDir, string.Format("{0}/file/Data_{0}.bin", i));
                using (FileStream stream = File.Create(filePath))
                {
                    int fileSize = 1024;
                    byte[] data = new byte[fileSize];
                    for (int j = 0; j < fileSize; j++)
                    {
                        data[j] = (byte)j;
                    }
                    stream.Write(data, 0, fileSize);
                }
            }

            Func<string, string, string, string> createNsp = (nsp, metaFile, addCmd) =>
            {
                var cmd = string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {5} --icon AmericanEnglish {4} Japanese {4}", nsp, metaFile, env.DefaultDescFile, env.TestCodeDir, iconPath, dataDir) + addCmd;
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                return ExecuteProgram(cmd, true);
            };


            var outputNsp = env.OutputDir + "\\output.nsp";
            var standardOut = createNsp(outputNsp, metaFilePath, string.Format(" --keygeneration 0 --filter {0}", fdfFilePath));
            Assert.IsTrue(standardOut.Contains("data/1' is not archived because it is empty or filtered."));
            Assert.IsTrue(standardOut.Contains("data/3' is not archived because it is empty or filtered."));
            Assert.IsTrue(standardOut.Contains("data/1/file' is not archived because it is empty or filtered."));
            Assert.IsTrue(standardOut.Contains("data/3/file' is not archived because it is empty or filtered."));

            // --no-check-dir-warning オプションを指定
            standardOut = createNsp(outputNsp, metaFilePath, string.Format(" --keygeneration 0 --filter {0} --no-check-dir-warning", fdfFilePath));
            Assert.IsFalse(standardOut.Contains("is not created in nsp. The directory has no file which should be included."));
        }


        [TestMethod]
        public void ErrorExecutionCreateApplicationNspWithVersionAndReleaseVersion()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaFilePath = testSourceDir + "\\ErrorVersion\\application.nmeta";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            string testNpdm = dataDir + "\\main.npdm";
            MakeNpdm(testNpdm, metaFilePath, descFile);

            var outputPath = outputDir + "\\" + Path.GetFileName(metaFilePath) + ".nsp";

            var error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                outputPath,
                metaFilePath,
                descFile,
                dataDir
                ));
            Assert.IsTrue(error.IndexOf("<Version> cannot be specified with <ReleaseVersion>") >= 0);
        }

        [TestMethod]
        public void ErrorExecutionCreateApplicationNsp()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var properMetaFilePath = env.SourceDir + "\\ApplicationMeta\\describe_all.nmeta";
            var dataDir = env.OutputDir + "\\program";
            var iconPath = env.SourceDir + "\\Icon\\describe_all.bmp";

            SafeDeleteDirectory(env.OutputDir);

            Directory.CreateDirectory(dataDir);
            string testNpdm = dataDir + "\\main.npdm";
            MakeNpdm(testNpdm, properMetaFilePath, env.DefaultDescFile);

            var outputPath = env.OutputDir + "\\" + Path.GetFileName(properMetaFilePath) + ".nsp";

            Action<string, string> executeProgram = (invalidMetaFilePath, errorMessage) =>
            {
                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --icon AmericanEnglish {4} Japanese {4}",
                    outputPath,
                    invalidMetaFilePath,
                    env.DefaultDescFile,
                    dataDir,
                    iconPath
                    ));
                Assert.IsTrue(error.IndexOf(errorMessage) >= 0);
            };
            executeProgram(env.SourceDir + "\\ErrorProgramId\\describe_all.nmeta", "program ID specified in .nmeta file differ from the one used to build code.");

            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_SmallUserAccountSaveDataSize.nmeta", "UserAccountSaveDataSize is too small.It must be larger than 0xc000");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_SmallUserAccountSaveDataJournalSize.nmeta", "UserAccountSaveDataJournalSize is too small.It must be larger than 0xc000");

            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_SmallDeviceSaveDataSize.nmeta", "DeviceSaveDataSize is too small.It must be larger than 0xc000");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_SmallDeviceSaveDataJournalSize.nmeta", "DeviceSaveDataJournalSize is too small.It must be larger than 0xc000");

            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_SmallBcatDeliveryCacheStorageSize.nmeta", "BcatDeliveryCacheStorageSize is too small.It must be larger than 0xc000");

            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_SmallCacheStorageSize.nmeta", "CacheStorageSize is too small.It must be larger than 0xc000");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_SmallCacheStorageJournalSize.nmeta", "CacheStorageJournalSize is too small.It must be larger than 0xc000");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_SmallCacheStorageDataAndJournalSize.nmeta", "CacheStorageDataAndJournalSizeMax is too small.It must be larger than 0x18000");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_SmallTemporaryStorageSize.nmeta", "TemporaryStorageSize is too small.It must be larger than 0xc000");


            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_LargeUserAccountSaveData.nmeta", "UserAccountSaveDataSize or UserAccountSaveDataJournalSize is too large.");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_LargeDeviceSaveDataSize.nmeta", "DeviceSaveDataSize or DeviceSaveDataJournalSize is too large.");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_LargeBcatDeliveryCacheStorageSize.nmeta", "BcatDeliveryCacheStorageSize is too large.");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_LargeCacheStorageSize.nmeta", "CacheStorageSize or CacheStorageJournalSize is too large.");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_LargeCacheStorageIndexMax.nmeta", "CacheStorageIndexMax is too small.It must be larger than 0x0.");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_LargeTemporaryStorageSize.nmeta", "TemporaryStorageSize is too large.");

            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_CacheStorage_NeedCacheStorageSize.nmeta", "CacheStorageAvailableSize is necessary.");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_CacheStorage_NeedCacheStorageJournalSize.nmeta", "CacheStorageJournalSize is necessary.");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_CacheStorage_NeedCacheStorageTotalSize.nmeta", "CacheStorageDataAndJournalSizeMax is necessary.");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_CacheStorage_NeedCacheStorageIndexMax.nmeta", "CacheStorageIndexMax is necessary.");

            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_CacheStorage_UnnecessaryCacheStorageSize.nmeta", "Only either CacheStorageSize/CacheStorageJournalSize or CacheStorageDataAndJournalSizeMax/CacheStorageIndexMax can be set.");
            executeProgram(env.SourceDir + "\\ErrorSaveDataSize\\describe_all_CacheStorage_UnnecessaryCacheStorageTotalSize.nmeta", "Only either CacheStorageSize/CacheStorageJournalSize or CacheStorageDataAndJournalSizeMax/CacheStorageIndexMax can be set.");
        }

        [TestMethod]
        public void ErrorExecutionProdEncryptionNotAllowed()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaFilePath = testSourceDir + "\\ApplicationMeta\\describe_all_with_icon.nmeta";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataNoSignDir = outputDir + "\\program1";
            var dataNoFlagDir = outputDir + "\\program2";
            var noSignDescFile = testPath.GetSigloRoot() + "\\Tests\\Common\\Resources\\TestApplication.autogen.desc";
            var noFlagDescFile = testSourceDir + "\\ErrorProdEncription\\SignedButNoProductionFlagForTestOnly.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";

            SafeDeleteDirectory(outputDir);

            // 署名のついていない desc で失敗
            Directory.CreateDirectory(dataNoSignDir);
            {
                string testNpdm = dataNoSignDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFilePath, noSignDescFile);
            }

            // 署名はついているが製品化フラグがついていない desc で失敗
            Directory.CreateDirectory(dataNoFlagDir);
            {
                string testNpdm = dataNoFlagDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFilePath, noFlagDescFile);
            }

            var testList = new List<Tuple<string, string, string>>() {
                Tuple.Create(noSignDescFile, dataNoSignDir, "Failed to verify acid in main.npdm."),
                Tuple.Create(noFlagDescFile, dataNoFlagDir, "This application is not allowed to be prodencrypted.")
            };

            foreach (var testElem in testList)
            {
                Console.WriteLine();
                var outputPath = outputDir + "\\" + Path.GetFileName(metaFilePath) + ".nsp";
                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                    outputPath,
                    metaFilePath,
                    testElem.Item1,
                    testElem.Item2
                    ));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(outputPath, keyConfigFile);

                error = ExecuteProgram(string.Format("prodencryption --no-check --keyconfig {0} -o {1} {2}", keyConfigFile, outputDir, outputPath));
                Assert.IsTrue(error.IndexOf(testElem.Item3) >= 0);
            }
        }

        [TestMethod]
        public void TestCardSizeAutoSet()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);
            var metaFileAuto = env.SourceDir + "\\ApplicationMeta\\version.nmeta";
            var iconPath = env.SourceDir + "\\Icon\\describe_all.bmp";

            Action<string, string, string> createNsp = (nsp, metaFile, addOption) =>
            {
                var cmd = string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {3} --icon AmericanEnglish {4} Japanese {4} {5}", nsp, metaFile, env.DefaultDescFile, env.TestCodeDir, iconPath, addOption);
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(nsp, null);
            };

            Action<string, int, bool> checkCardXml = (nsp, expectedCardSize, expectedAutoSetSize) =>
            {
                using (var nspReader = new NintendoSubmissionPackageReader(nsp))
                {
                    var cardSpecXml = nspReader.ListFileInfo().Where(x => x.Item1 == "cardspec.xml").Single();
                    var model = ArchiveReconstructionUtils.ReadXml<CardSpecModel>(nspReader, cardSpecXml.Item1, cardSpecXml.Item2);
                    Assert.AreEqual(expectedCardSize, Convert.ToInt32(model.Size));
                    Assert.AreEqual(expectedAutoSetSize, model.AutoSetSize);
                }
            };

            Action<string> createXci = (nsp) =>
            {
                var cmd = string.Format("prodencryption -o {0} --gamecard --no-xcie --no-padding --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, nsp);
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
            };

            Action<string, int> checkXciResultXml = (nsp, expectedCardSize) =>
            {
                var xml = nsp.Replace(".nsp", "_prod.xci.result.xml");
                CheckXmlEncoding(xml);
                ResultModel model;
                using (var fs = new FileStream(xml, FileMode.Open, FileAccess.Read))
                {
                    var serializer = new XmlSerializer(typeof(ResultModel));
                    model = (ResultModel)serializer.Deserialize(fs);
                }
                Assert.IsTrue(model.CardHeader != null);
                Assert.AreEqual(model.CardHeader.RomSize, XciUtils.GetBytesString(XciInfo.ConvertRomSizeToRomSizeByte(expectedCardSize)));
            };

            // app v0 (program, nacp)
            var v0 = env.OutputDir + "\\v0.nsp";
            createNsp(v0, metaFileAuto, string.Empty); // 自動設定では 2 GB 以下にならない
            checkCardXml(v0, 2, true);
            createXci(v0);
            checkXciResultXml(v0, 2);

            var metaFileManual = env.OutputDir + "\\CardSpecSize1.nmeta";
            var meta = @"<?xml version=""1.0""?>
                         <NintendoSdkMeta>
                           <Core>
                             <ApplicationId>0x0005000C10000001</ApplicationId>
                           </Core>
                           <Application>
                             <ReleaseVersion>1</ReleaseVersion>
                             <Title>
                               <Language>AmericanEnglish</Language>
                               <Name>Title</Name>
                               <Publisher>Publisher</Publisher>
                             </Title>
                             <DisplayVersion>1.0.0</DisplayVersion>
                             <SupportedLanguage>Japanese</SupportedLanguage>
                             <SupportedLanguage>AmericanEnglish</SupportedLanguage>
                           </Application>
                           <CardSpec>
                             <Size>1</Size>
                             <ClockRate>25</ClockRate>
                           </CardSpec>
                         </NintendoSdkMeta>";
            File.WriteAllText(metaFileManual, meta);
            createNsp(v0, metaFileManual, " --ignore-unpublishable-error"); // 手動設定で 1 GB には一応できる（製品化も可能）
            checkCardXml(v0, 1, false);
            createXci(v0);
            checkXciResultXml(v0, 1);
        }

        [TestMethod]
        public void TestExecutionCreateApplicationDocumentNsp()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var mainVisualDir = testSourceDir + "\\MainVisual";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var testExtractDir = Path.Combine(outputDir, "extract");
            var accessibleUrlsDir = outputDir + "\\accessible-urls";
            var accessibleUrlsFile = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");
            var accessibleUrlsFromOutputDirPath = "accessible-urls/accessible-urls.txt";
            var htmlDocumentXmlFile = Path.Combine(outputDir, "htmldoc.xml");
            var emptyHtmlDocumentXmlFile = Path.Combine(outputDir, "empty_htmldoc.xml");
            string emptyFile = Path.Combine(outputDir, "empty.txt");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            MakeAccessibleUrlsFile(accessibleUrlsFile);
            MakeHtmlDocumentXmlFile(htmlDocumentXmlFile);
            MakeEmptyHtmlDocumentXmlFile(emptyHtmlDocumentXmlFile);

            using (FileStream stream = File.Create(emptyFile))
            {
            }

            var legalInformationZip = MakeLegalInfoZipfile(outputDir);

            var file = GetDefaultApplicationMetaFile(metaDir);
            {
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, file, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";
                var mainVisualPath = mainVisualDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".png";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4} --html-document {5} --legal-information {6} --save-adf",
                    outputPath,
                    file,
                    descFile,
                    dataDir,
                    iconPath,
                    dataDir,
                    legalInformationZip));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, null);

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);

                TestCreateNspMeta(outputPath, testExtractDir, file, "Application", iconPath);

                TestLegalInformation(outputPath, testExtractDir);
            }

            // html-document を meta ファイルに指定できる
            {
                TestMetaFile metaFile = new TestMetaFile();
                metaFile.AddApplicationTags(string.Format("<HtmlDocumentPath>{0}</HtmlDocumentPath>", dataDir));
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                    outputPath,
                    metaFile.GetFilePath(),
                    descFile,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, null);

                TestHtmlDocumentDir(outputPath, testExtractDir, dataDir, emptyHtmlDocumentXmlFile, true);
            }

            // html-document を meta ファイルとオプションで指定した時には meta ファイルを優先する
            {
                string testDirName = "HtmlDocumentTestDir";
                string testDir = Path.Combine(outputDir, testDirName);
                SafeDeleteDirectory(testDir);
                Directory.CreateDirectory(testDir);

                string testFileName = Path.Combine(testDir, "nothing.html");
                using (var fs = File.OpenWrite(testFileName))
                {
                    var sw = new StreamWriter(fs);
                    sw.WriteLine("Nothing.");
                    sw.Flush();
                }

                TestMetaFile metaFile = new TestMetaFile();
                metaFile.OutputDir = outputDir;
                metaFile.AddApplicationTags(string.Format("<HtmlDocumentPath>./{0}</HtmlDocumentPath>", testDirName)); //meta ファイルからの間接参照
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --html-document {4}",
                    outputPath,
                    metaFile.GetFilePath(),
                    descFile,
                    dataDir,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, null);

                TestHtmlDocumentDir(outputPath, testExtractDir, testDir, emptyHtmlDocumentXmlFile, true);
            }

            // legal-information を meta ファイルに指定できる
            {
                TestMetaFile metaFile = new TestMetaFile();
                metaFile.AddApplicationTags(string.Format("<LegalInformationFilePath>{0}</LegalInformationFilePath>", legalInformationZip));
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                    outputPath,
                    metaFile.GetFilePath(),
                    descFile,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, null);

                TestLegalInformation(outputPath, testExtractDir);
            }

            // legal-information を meta ファイルとオプションで指定したときには meta ファイルを優先する
            {
                var invalidZipFilePath = MakeLegalInfoZipfile(outputDir, false);
                TestMetaFile metaFile = new TestMetaFile();
                metaFile.OutputDir = outputDir;
                metaFile.AddApplicationTags(string.Format("<LegalInformationFilePath>./{0}</LegalInformationFilePath>", Path.GetFileName(legalInformationZip))); // 間接参照
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --legal-information {4}",
                    outputPath,
                    metaFile.GetFilePath(),
                    descFile,
                    dataDir,
                    invalidZipFilePath));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, null);

                TestLegalInformation(outputPath, testExtractDir);
            }

            // accessible-urls を meta ファイルに指定できる
            {
                TestMetaFile metaFile = new TestMetaFile();
                metaFile.AddApplicationTags(string.Format("<AccessibleUrlsFilePath>{0}</AccessibleUrlsFilePath>", accessibleUrlsFile));
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                    outputPath,
                    metaFile.GetFilePath(),
                    descFile,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, null);

                TestHtmlDocumentDir(outputPath, testExtractDir, accessibleUrlsDir, htmlDocumentXmlFile, false);
            }

            // 空の accessible-urls を指定できる
            {
                TestMetaFile metaFile = new TestMetaFile();
                metaFile.AddApplicationTags(string.Format("<AccessibleUrlsFilePath>{0}</AccessibleUrlsFilePath>", emptyFile));
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                    outputPath,
                    metaFile.GetFilePath(),
                    descFile,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, null);

                TestHtmlDocumentDir(outputPath, testExtractDir, accessibleUrlsDir, emptyHtmlDocumentXmlFile, false);
            }

            // accessible-urls を meta ファイルとオプションで指定したときには meta ファイルを優先する
            {
                TestMetaFile metaFile = new TestMetaFile();
                metaFile.OutputDir = outputDir;
                metaFile.AddApplicationTags(string.Format("<AccessibleUrlsFilePath>{0}</AccessibleUrlsFilePath>", accessibleUrlsFromOutputDirPath));
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --accessible-urls {4}",
                    outputPath,
                    metaFile.GetFilePath(),
                    descFile,
                    dataDir,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, null);

                TestHtmlDocumentDir(outputPath, testExtractDir, accessibleUrlsDir, htmlDocumentXmlFile, false);
            }

            // accessible-urls と html-document を meta ファイルに記載した場合、両者が nsp 内に存在する
            {
                string testHtmlDirName = "HtmlDocumentTestDir";
                string testHtmlDir = Path.Combine(outputDir, testHtmlDirName);
                SafeDeleteDirectory(testHtmlDir);
                Directory.CreateDirectory(testHtmlDir);

                string testFileName = Path.Combine(testHtmlDir, "nothing.html");
                File.WriteAllText(testFileName, "Nothing");

                TestMetaFile metaFile = new TestMetaFile();
                metaFile.OutputDir = outputDir;
                metaFile.AddApplicationTags(string.Format("<HtmlDocumentPath>{0}</HtmlDocumentPath>", testHtmlDirName));
                metaFile.AddApplicationTags(string.Format("<AccessibleUrlsFilePath>{0}</AccessibleUrlsFilePath>", accessibleUrlsFromOutputDirPath));
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                    outputPath,
                    metaFile.GetFilePath(),
                    descFile,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, null);

                TestHtmlDocumentDir(outputPath, testExtractDir, accessibleUrlsDir, htmlDocumentXmlFile, false);
                TestHtmlDocumentDir(outputPath, testExtractDir, testHtmlDir, htmlDocumentXmlFile, true);
            }

            File.Delete(legalInformationZip);
        }

        [TestMethod]
        public void TestExecutionCreateApplicationDocumentNspd()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var testExtractDir = Path.Combine(outputDir, "extract");
            var accessibleUrlsDir = outputDir + "\\accessible-urls";
            var accessibleUrlsFile = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");
            var accessibleUrlsFromOutputDirPath = "accessible-urls/accessible-urls.txt";
            var htmlDocumentBaseDir = "htmlDocument0.ncd\\data";
            var legalInfoBaseDir = "legalInformation0.ncd\\data";
            var dataHtmlDocumentDir = "html-document";
            var dataAccessibleUrlsDir = "accessible-urls";

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            MakeAccessibleUrlsFile(accessibleUrlsFile);

            var legalInformationZip = MakeLegalInfoZipfile(outputDir);

            var file = GetIconContainApplicationMetaFile(metaDir);
            {
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, file, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nspd";

                var error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2} --html-document {3} --legal-information {4} --accessible-urls {5}",
                    outputPath,
                    file,
                    dataDir,
                    dataDir,
                    legalInformationZip,
                    accessibleUrlsDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(Directory.Exists(outputPath));

                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, dataHtmlDocumentDir, "data.dat" })));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, dataAccessibleUrlsDir, "accessible-urls.txt" })));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, legalInfoBaseDir, TestZipFileName })));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, legalInfoBaseDir, LegalInfoXmlFileName })));
            }

            // html-document を meta ファイルに指定できる
            {
                TestMetaFile metaFile = new TestMetaFile();
                metaFile.AddApplicationTags(string.Format("<HtmlDocumentPath>{0}</HtmlDocumentPath>", dataDir));
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nspd";

                var error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2}",
                    outputPath,
                    metaFile.GetFilePath(),
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(Directory.Exists(outputPath));

                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, "html-document", "data.dat" })));
            }

            // html-document を meta ファイルとオプションで指定した時には meta ファイルを優先する
            {
                string testDirName = "HtmlDocumentTestDir";
                string testDir = Path.Combine(outputDir, testDirName);
                SafeDeleteDirectory(testDir);
                Directory.CreateDirectory(testDir);

                string testFileName = Path.Combine(testDir, "nothing.html");
                using (var fs = File.OpenWrite(testFileName))
                {
                    var sw = new StreamWriter(fs);
                    sw.WriteLine("Nothing.");
                    sw.Flush();
                }

                TestMetaFile metaFile = new TestMetaFile();
                metaFile.OutputDir = outputDir;
                metaFile.AddApplicationTags(string.Format("<HtmlDocumentPath>./{0}</HtmlDocumentPath>", testDirName)); //meta ファイルからの間接参照
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nspd";

                var error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2} --save-adf --html-document {3}",
                    outputPath,
                    metaFile.GetFilePath(),
                    dataDir,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(Directory.Exists(outputPath));

                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, "html-document", "nothing.html" })));
            }

            // legal-information を meta ファイルに指定できる
            {
                TestMetaFile metaFile = new TestMetaFile();
                metaFile.AddApplicationTags(string.Format("<LegalInformationFilePath>{0}</LegalInformationFilePath>", legalInformationZip));
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nspd";

                var error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2}",
                    outputPath,
                    metaFile.GetFilePath(),
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(Directory.Exists(outputPath));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, legalInfoBaseDir, LegalInfoXmlFileName })));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, legalInfoBaseDir, TestZipFileName })));
            }

            // legal-information を meta ファイルとオプションで指定したときには meta ファイルを優先する
            {
                var invalidZipFilePath = MakeLegalInfoZipfile(outputDir, false);
                TestMetaFile metaFile = new TestMetaFile();
                metaFile.OutputDir = outputDir;
                metaFile.AddApplicationTags(string.Format("<LegalInformationFilePath>./{0}</LegalInformationFilePath>", Path.GetFileName(legalInformationZip))); // 間接参照
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nspd";

                var error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2} --legal-information {3}",
                    outputPath,
                    metaFile.GetFilePath(),
                    dataDir,
                    invalidZipFilePath));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(Directory.Exists(outputPath));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, legalInfoBaseDir, LegalInfoXmlFileName })));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, legalInfoBaseDir, TestZipFileName })));
            }

            // accessible-urls を meta ファイルに指定できる
            {
                TestMetaFile metaFile = new TestMetaFile();
                metaFile.AddApplicationTags(string.Format("<AccessibleUrlsFilePath>{0}</AccessibleUrlsFilePath>", accessibleUrlsFile));
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nspd";

                var error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2}",
                    outputPath,
                    metaFile.GetFilePath(),
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(Directory.Exists(outputPath));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, dataAccessibleUrlsDir, "accessible-urls.txt" })));
            }

            // accessible-urls を meta ファイルとオプションで指定したときには meta ファイルを優先する
            {
                TestMetaFile metaFile = new TestMetaFile();
                metaFile.OutputDir = outputDir;
                metaFile.AddApplicationTags(string.Format("<AccessibleUrlsFilePath>{0}</AccessibleUrlsFilePath>", accessibleUrlsFromOutputDirPath));
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nspd";

                var error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2} --accessible-urls {3}",
                    outputPath,
                    metaFile.GetFilePath(),
                    dataDir,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(Directory.Exists(outputPath));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, dataAccessibleUrlsDir, "accessible-urls.txt" })));
            }

            // accessible-urls と html-document を meta ファイルに記載した場合、両者が nsp 内に存在する
            {
                string testHtmlDirName = "HtmlDocumentTestDir";
                string testHtmlDir = Path.Combine(outputDir, testHtmlDirName);
                SafeDeleteDirectory(testHtmlDir);
                Directory.CreateDirectory(testHtmlDir);

                string testFileName = Path.Combine(testHtmlDir, "nothing.html");
                File.WriteAllText(testFileName, "Nothing");

                TestMetaFile metaFile = new TestMetaFile();
                metaFile.OutputDir = outputDir;
                metaFile.AddApplicationTags(string.Format("<HtmlDocumentPath>{0}</HtmlDocumentPath>", testHtmlDirName));
                metaFile.AddApplicationTags(string.Format("<AccessibleUrlsFilePath>{0}</AccessibleUrlsFilePath>", accessibleUrlsFromOutputDirPath));
                metaFile.Write();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nspd";

                var error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2}",
                    outputPath,
                    metaFile.GetFilePath(),
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(Directory.Exists(outputPath));

                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, dataHtmlDocumentDir, "nothing.html" })));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, dataAccessibleUrlsDir, "accessible-urls.txt" })));
            }

            // accessible-urls が readonly だった場合も再生成出来る
            {
                string testHtmlDirName = "HtmlDocumentTestDir";
                string testHtmlDir = Path.Combine(outputDir, testHtmlDirName);
                SafeDeleteDirectory(testHtmlDir);
                Directory.CreateDirectory(testHtmlDir);

                string testFileName = Path.Combine(testHtmlDir, "nothing.html");
                File.WriteAllText(testFileName, "Nothing");

                TestMetaFile metaFile = new TestMetaFile();
                metaFile.OutputDir = outputDir;
                metaFile.AddApplicationTags(string.Format("<HtmlDocumentPath>{0}</HtmlDocumentPath>", testHtmlDirName));
                metaFile.AddApplicationTags(string.Format("<AccessibleUrlsFilePath>{0}</AccessibleUrlsFilePath>", accessibleUrlsFromOutputDirPath));
                metaFile.Write();
                FileInfo fi = new FileInfo(accessibleUrlsFile);
                fi.IsReadOnly = true;

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile.GetFilePath(), descFile);

                var outputPath = outputDir + "\\" + metaFile.FileName + ".nspd";

                var error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2}",
                    outputPath,
                    metaFile.GetFilePath(),
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(Directory.Exists(outputPath));

                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, dataHtmlDocumentDir, "nothing.html" })));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, dataAccessibleUrlsDir, "accessible-urls.txt" })));

                error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2}",
                    outputPath,
                    metaFile.GetFilePath(),
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(Directory.Exists(outputPath));

                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, dataHtmlDocumentDir, "nothing.html" })));
                Assert.IsTrue(File.Exists(Path.Combine(new string[] { outputPath, htmlDocumentBaseDir, dataAccessibleUrlsDir, "accessible-urls.txt" })));

                fi.IsReadOnly = false;
            }
            File.Delete(legalInformationZip);
        }

        [TestMethod]
        public void TestCreateNspNoLegalInfoXml()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var mainVisualDir = testSourceDir + "\\MainVisual";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var testExtractDir = Path.Combine(outputDir, "extract");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            var legalInformationZip = MakeLegalInfoZipfile(outputDir, false);

            var file = GetDefaultApplicationMetaFile(metaDir);
            {
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, file, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";
                var mainVisualPath = mainVisualDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".png";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4} --html-document {5} --legal-information {6} --save-adf",
                    outputPath,
                    file,
                    descFile,
                    dataDir,
                    iconPath,
                    dataDir,
                    legalInformationZip));
                Assert.IsTrue(error.IndexOf("legalinfo.xml is not found in the zip file.") >= 0);
            }

            File.Delete(legalInformationZip);
        }


        [TestMethod]
        public void TestExecutionCreateApplicationLegalInformationWithDirectory()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var mainVisualDir = testSourceDir + "\\MainVisual";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var legalDir = outputDir + "\\legal";
            var testExtractDir = Path.Combine(outputDir, "extract");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            Directory.CreateDirectory(legalDir);
            CreateLegalInfoXml(legalDir);

            var file = GetDefaultApplicationMetaFile(metaDir);
            {
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, file, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";
                var mainVisualPath = mainVisualDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".png";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4} --html-document {5} --legal-information-dir {6} --save-adf",
                    outputPath,
                    file,
                    descFile,
                    dataDir,
                    iconPath,
                    dataDir,
                    legalDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, null);

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);

                TestCreateNspMeta(outputPath, testExtractDir, file, "Application", iconPath);
            }
        }

        [TestMethod]
        public void TestCreateNspdNoXmlFile()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var mainVisualDir = testSourceDir + "\\MainVisual";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var testExtractDir = Path.Combine(outputDir, "extract");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            var file = GetDefaultApplicationMetaFile(metaDir);
            {
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, file, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";
                var mainVisualPath = mainVisualDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".png";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --icon AmericanEnglish {4} Japanese {4} --html-document {5} --legal-information-dir {6} --save-adf",
                    outputPath,
                    file,
                    descFile,
                    dataDir,
                    iconPath,
                    dataDir,
                    dataDir));
                Assert.IsTrue(error.IndexOf("legalinfo.xml is not found in the zip file.") >= 0);
            }
        }

        [TestMethod]
        public void TestExecutionCreateNspWithErrorUnpublishable()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var metaFile = GetDefaultApplicationMetaFile(metaDir);
            var iconDir = testSourceDir + "\\Icon";
            var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".bmp";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            string testNpdm = dataDir + "\\main.npdm";
            MakeNpdm(testNpdm, metaFile, descFile);

            var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";

            File.Delete(outputPath);
            string error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --error-unpublishable --icon AmericanEnglish {4} Japanese {4} ",
                outputPath,
                metaFile,
                descFile,
                dataDir,
                iconPath));
            Assert.IsTrue(error.IndexOf("[Error] Found one or more unpublishable error in nsp. The nsp will be deleted automatically.") >= 0, error);
            Assert.IsFalse(File.Exists(outputPath));

            metaFile = GetIconContainApplicationMetaFile(metaDir);

            File.Delete(outputPath);
            error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                outputPath,
                metaFile,
                descFile,
                dataDir));
            Assert.IsTrue(error == null || error == string.Empty, error);
            Assert.IsTrue(File.Exists(outputPath));

            File.Delete(outputPath);
            error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type SystemProgram --meta {1} --desc {2} --program {3} --save-adf --keyconfig {4} --error-unpublishable",
                outputPath,
                metaFile,
                descFile,
                dataDir,
                keyConfigFile));
            Assert.IsTrue(error == null || error == string.Empty, error);
            Assert.IsTrue(File.Exists(outputPath));

            File.Delete(outputPath);
            error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type SystemProgram --meta {1} --desc {2} --program {3} --save-adf --keyconfig {4}",
                outputPath,
                metaFile,
                descFile,
                dataDir,
                keyConfigFile));
            Assert.IsTrue(error == null || error == string.Empty, error);
            Assert.IsTrue(File.Exists(outputPath));

            // SupportedLanguage が設定されていないとエラーになる
            {
                TestMetaFile iconMetaFile = MakeIconPathMetaFile(iconPath);

                MakeNpdm(testNpdm, iconMetaFile.GetFilePath(), descFile);

                File.Delete(outputPath);
                error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --keyconfig {4} --error-unpublishable",
                    outputPath,
                    iconMetaFile.GetFilePath(),
                    descFile,
                    dataDir,
                    keyConfigFile));
                Assert.IsTrue(error.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0, error);
                Assert.IsFalse(File.Exists(outputPath));
            }
        }

        [TestMethod]
        public void TestExecutionCreateNspWithoutErrorUnpublishableOption()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var testDir = Path.Combine(outputDir, "test");
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var iconPath = testSourceDir + "\\Icon\\describe_all.bmp";

            SafeDeleteDirectory(outputDir);
            SafeDeleteDirectory(testDir);
            SafeDeleteDirectory(dataDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            Directory.CreateDirectory(testDir);

            Action<string[], string, string> testUnpublishableErrorWithoutOption = delegate (string[] metaLines, string metaFile, string issueString)
            {
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);
                Assert.IsTrue(error.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0, error);
                Assert.IsFalse(File.Exists(outputPath));

                var standardOut = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir, true);
                Assert.IsTrue(standardOut.Contains(issueString));
            };

            // Titleエントリに対応するIconエントリがない
            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Name</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                };
                string metaFile = testDir + "TitleEntry_Only.nmeta";
                testUnpublishableErrorWithoutOption(metaLines, metaFile, "[Error] (Issue 10-608)");
            }

            // Iconエントリに対応するTitleエントリがない
            {
                string[] metaLines = new string[]
                {
                    "<Icon>",
                    "<Language>AmericanEnglish</Language>",
                    string.Format("<IconPath>{0}</IconPath>",iconPath),
                    "</Icon>",
                };
                string metaFile = testDir + "IconEntry_Only.nmeta";
                testUnpublishableErrorWithoutOption(metaLines, metaFile, "[Error] (Issue 10-802)");
            }

            // Titleエントリに不正な文字が指定されている
            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Name\u7b80\u4f53\u5b57</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                };
                string metaFile = testDir + "Invalid_Character.nmeta";
                testUnpublishableErrorWithoutOption(metaLines, metaFile, "[Error] (Issue 10-808)");
            }

            // IsbnエントリにAscii以外な文字が指定されている
            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Name</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<Isbn>\uFF11</Isbn>",
                };
                string metaFile = testDir + "Invalid_Character.nmeta";
                testUnpublishableErrorWithoutOption(metaLines, metaFile, "[Error] (Issue 10-809)");
            }

            // IsbnエントリにAscii以外な文字が指定されている
            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Name</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<DisplayVersion>\uFF11</DisplayVersion>",
                };
                string metaFile = testDir + "Invalid_Character.nmeta";
                testUnpublishableErrorWithoutOption(metaLines, metaFile, "[Error] (Issue 10-809)");
            }

            // IsbnエントリにAscii以外な文字が指定されている
            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Name</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<ApplicationErrorCodeCategory>\uFF11</ApplicationErrorCodeCategory>",
                };
                string metaFile = testDir + "Invalid_Character.nmeta";
                testUnpublishableErrorWithoutOption(metaLines, metaFile, "[Error] (Issue 10-809)");
            }

            // Bcat情報がBcatDeliveryCacheStorageSizeしか無い
            {
                string[] metaLines = new string[]
                {
                    "<BcatDeliveryCacheStorageSize>0x500000</BcatDeliveryCacheStorageSize>",
                };
                string metaFile = testDir + "BcatDeliveryCacheStorageSizeEntry_Only.nmeta";
                testUnpublishableErrorWithoutOption(metaLines, metaFile, "[Error] (Issue 10-803)");
            }

            // Bcat情報がBcatPassphraseしか無い
            {
                string[] metaLines = new string[]
                {
                    "<BcatPassphrase>0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef0</BcatPassphrase>",
                };
                string metaFile = testDir + "BcatPassphraseEntry_Only.nmeta";
                testUnpublishableErrorWithoutOption(metaLines, metaFile, "[Error] (Issue 10-803)");
            }

            // Applicationエントリが空
            {
                string metaFile = testDir + "ApplicationEntry_empty.nmeta";

                MakeTestMetaFile(metaFile, new string[] { });

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));

                var standardOut = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir, true);
                Assert.IsTrue(!standardOut.Contains("[Error]")); // Errorが発生していない
                Assert.IsTrue(standardOut.Contains("[Warning] (Issue 10-010)")); // 不正なnspファイル（タイトル名なし）
                Assert.IsTrue(standardOut.Contains("[Warning] (Issue 10-011)")); // 不正なnspファイル（パブリッシャー名なし）
                Assert.IsTrue(standardOut.Contains("[Warning] (Issue 10-801)")); // 不正なnspファイル（タイトル言語なし）
                Assert.IsTrue(standardOut.Contains("[Warning] (Issue 10-012)")); // 不正なnspファイル（バージョン表記なし）
                Assert.IsTrue(standardOut.Contains("[Warning] (Issue 10-014)")); // 不正なnspファイル（予期しない対応言語情報）
                Assert.IsTrue(standardOut.Contains("[Warning] (Issue 10-607)")); // 不正なnspファイル（アイコンが存在しない）

                File.Delete(outputPath);
            }

            // --error-unpublishable指定時にWarningの項目は --error-unpublishable 未指定時に無視される
            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Test0Test</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<Icon>",
                    "<Language>AmericanEnglish</Language>",
                    string.Format("<IconPath>{0}</IconPath>",iconPath),
                    "</Icon>",
                    "<SupportedLanguage>AmericanEnglish</SupportedLanguage>",
                    "<DisplayVersion>1.0.0</DisplayVersion>",
                    "<UserAccountSaveDataSizeMax>0x100000</UserAccountSaveDataSizeMax>", // --error-unpublishable指定時はWarningになる項目
                };
                string metaFile = testDir + "Warning_Delete.nmeta";

                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));

                var standardOut = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir, true);
                Assert.IsTrue(!standardOut.Contains("UnpublishableError:")); // UnpublishableError:が発生していない

                File.Delete(outputPath);
            }

            // Error,Warningのないnmeta
            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Test0Test</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<Icon>",
                    "<Language>AmericanEnglish</Language>",
                    string.Format("<IconPath>{0}</IconPath>",iconPath),
                    "</Icon>",
                    "<SupportedLanguage>AmericanEnglish</SupportedLanguage>",
                    "<DisplayVersion>1.0.0</DisplayVersion>",
                };
                string metaFile = testDir + "UnpublishableError_Nothing.nmeta";

                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));

                var standardOut = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir, true);
                Assert.IsTrue(!standardOut.Contains("UnpublishableError:")); // UnpublishableError:が発生していない

                File.Delete(outputPath);
            }

            SafeDeleteDirectory(outputDir);
            SafeDeleteDirectory(testDir);
            SafeDeleteDirectory(dataDir);
        }

        [TestMethod]
        public void TestExecutionCreateNspWithOffsetCheck()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var codeDir = outputDir + "\\program";
            var dataDir = outputDir + "\\data";
            var logoDir = outputDir + "\\logo";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var testExtractDir = Path.Combine(outputDir, "extract");
            var testExtractEntryDir = Path.Combine(outputDir, "extractEntry");

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            var applicationMetaPath = GetDefaultApplicationMetaFile(testSourceDir + "\\ApplicationMeta");

            var aocMetaPath = Path.Combine(outputDir, Path.GetFileName(testSourceDir + "\\AddOnContentMeta\\describe_addoncontent_id.nmeta"));
            File.Copy(testSourceDir + "\\AddOnContentMeta\\describe_addoncontent_id.nmeta", aocMetaPath);

            {
                var logoFile = testPath.GetSigloRoot() + "\\Tests\\Patch\\NactBuild\\logo.png";
                Directory.CreateDirectory(logoDir);
                File.Copy(logoFile, logoDir + "/" + Path.GetFileName(logoFile));
            }

            // 以下では コンテントメタタイプ x リソース有無 x コンテントタイプ の組み合わせで
            // 16KiB アラインするかどうかチェックする

            // コンテントメタタイプによる確認
            //
            // 0: コンテントメタタイプ
            // 1: メタファイルパス
            // 2: 16KiB アラインするメタタイプかどうか?
            var metaTypes = new Tuple<string, string, bool>[]
            {
                new Tuple<string, string, bool>(
                    NintendoContentMetaConstant.ContentMetaTypeApplication,
                    applicationMetaPath,
                    true),
                new Tuple<string, string, bool>(
                    NintendoContentMetaConstant.ContentMetaTypePatch,
                    applicationMetaPath,
                    true),
                new Tuple<string, string, bool>(
                    NintendoContentMetaConstant.ContentMetaTypeAddOnContent,
                    aocMetaPath,
                    true),
                new Tuple<string, string, bool>(
                    NintendoContentMetaConstant.ContentMetaTypeSystemData,
                    testSourceDir + "\\SystemDataMeta\\default.nmeta",
                    false),
                new Tuple<string, string, bool>(
                    NintendoContentMetaConstant.ContentMetaTypeSystemProgram,
                    applicationMetaPath,
                    false),
            };

            foreach (var metaType in metaTypes)
            {
                // リソースデータを含む場合と含まない場合どちらも確認
                // (どちらも結果は同じ)
                foreach (var hasData in new bool[] { false, true })
                {
                    Utils.WriteLine(metaType.Item1 + " " + (hasData ? "データ有り" : "データ無し"));

                    var metaFile = metaType.Item2;
                    var iconPath = testSourceDir + "\\Icon\\describe_all.bmp";

                    Utils.DeleteDirectoryIfExisted(codeDir);
                    Directory.CreateDirectory(codeDir);

                    if (metaType.Item1 == NintendoContentMetaConstant.ContentMetaTypeApplication
                        || metaType.Item1 == NintendoContentMetaConstant.ContentMetaTypeSystemProgram
                        || metaType.Item1 == NintendoContentMetaConstant.ContentMetaTypePatch)
                    {
                        string testNpdm = codeDir + "\\main.npdm";
                        MakeNpdm(testNpdm, metaFile, descFile);
                    }

                    List<string> expectedEntiesArray = null;

                    switch (metaType.Item1)
                    {
                        case NintendoContentMetaConstant.ContentMetaTypeApplication:
                            expectedEntiesArray = new List<string>()
                            {
                                @".*\.nca/fs0/html-document/test\.bin",
                                @".*\.nca/fs0/main\.npdm",
                                @".*\.nca/fs2/NintendoLogo\.png",
                                @".*\.nca/fs2/StartupMovie.gif",
                                @".*\.nca/fs0/control\.nacp",
                                @".*\.nx.AmericanEnglish.jpg",
                                @".*\.raw.AmericanEnglish.jpg",
                                @".*\.nx.Japanese.jpg",
                                @".*\.raw.Japanese.jpg",
                                @".*\.nca/fs0/icon_AmericanEnglish.dat",
                                @".*\.nca/fs0/icon_Japanese.dat",
                                @".*\.cnmt\.nca/fs0/Application_0005000c10000001.cnmt",
                                @".*\.cnmt.xml",
                                @".*\.nacp.xml",
                                @".*\.programinfo\.xml",
                                @".*\.htmldocument\.xml",
                                @"cardspec\.xml",
                                @"authoringtoolinfo.xml",
                            };

                            if (hasData)
                            {
                                expectedEntiesArray.Add(@".*\.nca/fs1/test\.bin");
                            }
                            break;

                        case NintendoContentMetaConstant.ContentMetaTypePatch:
                            // Logo は original から読み出されるけど、extract では出ない
                            expectedEntiesArray = new List<string>()
                            {
                                @".*\.nca/fs0/html-document/test\.bin",
                                @".*\.nca/fs0/main\.npdm",
                                @".*\.nca/fs0/control\.nacp",
                                @".*\.nx.AmericanEnglish.jpg",
                                @".*\.raw.AmericanEnglish.jpg",
                                @".*\.nx.Japanese.jpg",
                                @".*\.raw.Japanese.jpg",
                                @".*\.nca/fs0/icon_AmericanEnglish.dat",
                                @".*\.nca/fs0/icon_Japanese.dat",
                                @".*\.cnmt\.nca/fs0/Patch_0005000c10000801.cnmt",
                                @".*\.cnmt.xml",
                                @".*\.nacp.xml",
                                @".*\.programinfo\.xml",
                                @".*\.htmldocument\.xml",
                                @"cardspec\.xml",
                                @"0005000c10000801000000000000000\d\.cert",
                                @"0005000c10000801000000000000000\d\.tik",
                                @"authoringtoolinfo.xml",
                            };

                            if (hasData)
                            {
                                expectedEntiesArray.Add(@".*\.nca/fs1/test\.bin");
                            }
                            break;

                        case NintendoContentMetaConstant.ContentMetaTypeSystemProgram:
                            expectedEntiesArray = new List<string>()
                            {
                                @".*\.nca/fs1/test\.bin",
                                @".*\.nca/fs0/html-document/test\.bin",
                                @".*\.nca/fs0/main\.npdm",
                                @".*\.nca/fs2/logo\.png",
                                @".*\.cnmt\.nca/fs0/SystemProgram_0005000c10000001.cnmt",
                                @".*\.cnmt.xml",
                                @".*\.programinfo\.xml",
                                @".*\.htmldocument\.xml",
                                @"authoringtoolinfo.xml",
                            };
                            break;

                        case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                            expectedEntiesArray = new List<string>()
                            {
                                @".*\.cnmt\.nca/fs0/AddOnContent_0005000c10001004.cnmt",
                                @".*\.cnmt.xml",
                                @".*\.nca/fs0/test\.bin",
                                @"0005000c10001004000000000000000\d\.cert",
                                @"0005000c10001004000000000000000\d\.tik",
                                @"authoringtoolinfo.xml",
                            };
                            break;

                        case NintendoContentMetaConstant.ContentMetaTypeSystemData:
                            expectedEntiesArray = new List<string>()
                            {
                                @".*\.cnmt\.nca/fs0/SystemData_0005000c10000001.cnmt",
                                @".*\.cnmt.xml",
                                @".*\.nca/fs0/test\.bin",
                                @"authoringtoolinfo.xml",
                            };
                            break;
                    }

                    Utils.DeleteDirectoryIfExisted(dataDir);
                    if (hasData)
                    {
                        Directory.CreateDirectory(dataDir);
                        using (var file = new FileStream(dataDir + @"\test.bin", FileMode.Create, FileAccess.Write))
                        {
                            file.SetLength(10 * 1024 * 1024);
                        }
                    }
                    else
                    {
                        Directory.CreateDirectory(dataDir);
                        using (var file = new FileStream(dataDir + @"\test.bin", FileMode.Create, FileAccess.Write))
                        {
                            file.SetLength(1);
                        }
                    }

                    var content = string.Empty;
                    switch (metaType.Item1)
                    {
                        case NintendoContentMetaConstant.ContentMetaTypeApplication:
                            content = string.Format("--program {0} {1} --html-document {2}", codeDir, hasData ? dataDir : string.Empty, dataDir);
                            break;

                        case NintendoContentMetaConstant.ContentMetaTypePatch:
                            content = string.Format("--program {0} {1} --html-document {2}", codeDir, hasData ? dataDir : string.Empty, dataDir);
                            break;

                        case NintendoContentMetaConstant.ContentMetaTypeSystemProgram:
                            content = string.Format("--program {0} {1} {2} --html-document {3}", codeDir, dataDir, logoDir, dataDir);
                            break;

                        case NintendoContentMetaConstant.ContentMetaTypeSystemData:
                        case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                            content = string.Format("--data {0}", dataDir);
                            break;

                        default:
                            Assert.Fail("未知のメタタイプ");
                            break;
                    }

                    var metaTypeForApplicationNsp = metaType.Item1;
                    if (metaTypeForApplicationNsp == NintendoContentMetaConstant.ContentMetaTypePatch)
                    {
                        metaTypeForApplicationNsp = NintendoContentMetaConstant.ContentMetaTypeApplication;
                    }

                    var outputPath = outputDir + (hasData ? @"\test" : @"\test_no_data");
                    outputPath += "_" + metaType.Item1;
                    outputPath += ".nsp";

                    string format = "creatensp -o {0} --type {1} --meta {2} --desc {3} {4} --save-adf --keyconfig {5}";
                    if (metaTypeForApplicationNsp == NintendoContentMetaConstant.ContentMetaTypeApplication ||
                        metaTypeForApplicationNsp == NintendoContentMetaConstant.ContentMetaTypePatch)
                    {
                        format += " --icon AmericanEnglish {6} Japanese {6}";
                    }

                    var error = ExecuteProgram(
                        string.Format(format,
                            outputPath,
                            metaTypeForApplicationNsp,
                            metaFile,
                            descFile,
                            content,
                            keyConfigFile,
                            iconPath));
                    Assert.AreEqual(error, string.Empty);
                    VerifyNsp(outputPath, keyConfigFile);

                    string originalPath = null;
                    if (metaType.Item1 == NintendoContentMetaConstant.ContentMetaTypePatch)
                    {
                        var currentPath = outputDir + "\\tmp.nsp";
                        {
                            var patchMetaFile = outputDir + "\\tmp" + (hasData ? "_data" : "_no_data") + ".nmeta";
                            MakeSpecifiedVersionMetaFile(metaFile, patchMetaFile, 1);

                            {
                                string testNpdm = codeDir + "\\main.npdm";
                                File.Delete(testNpdm);
                                MakeNpdm(testNpdm, patchMetaFile, descFile);
                            }

                            if (hasData)
                            {
                                using (var file = new FileStream(dataDir + @"\test.bin", FileMode.Create, FileAccess.Write))
                                {
                                    var data = new byte[10 * 1024 * 1024];
                                    new Random().NextBytes(data);
                                    file.Write(data, 0, data.Length);
                                }
                            }

                            error = ExecuteProgram(string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} {3} --save-adf --keyconfig {4} --icon AmericanEnglish {5} Japanese {5}",
                                currentPath,
                                patchMetaFile,
                                descFile,
                                string.Format("--program {0} {1} --html-document {2}", codeDir, hasData ? dataDir : string.Empty, dataDir),
                                keyConfigFile,
                                iconPath));
                            Assert.AreEqual(error, string.Empty);
                            VerifyNsp(currentPath, keyConfigFile);
                        }

                        originalPath = outputPath;
                        outputPath = outputDir + (hasData ? @"\patch" : @"\patch_no_data") + ".nsp";
                        error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3} --keyconfig {4}",
                            outputPath,
                            descFile,
                            originalPath,
                            currentPath,
                            keyConfigFile));
                        Assert.AreEqual(error, string.Empty);
                        VerifyPatch(originalPath, outputPath, keyConfigFile);
                    }

                    TestExtractWithNsp(originalPath, outputPath, testExtractDir, testExtractEntryDir, expectedEntiesArray.ToArray(), keyConfigFile);
                    SafeDeleteDirectory(testExtractDir);
                    SafeDeleteDirectory(testExtractEntryDir);

                    using (var nsp = new NintendoSubmissionPackageReader(outputPath))
                    {
                        var meta = nsp.ListFileInfo().Find(v => v.Item1.EndsWith(".cnmt.xml"));
                        var model = ArchiveReconstructionUtils.ReadModel(nsp, meta.Item1, meta.Item2);

                        // コンテントタイプによる確認
                        //
                        // 0: コンテントタイプ
                        // 1: nca 拡張子
                        // 2: 先頭パーティション インデックス
                        // 3: 16KiB アラインするメタタイプかどうか?
                        Tuple<string, string, int, bool>[] contentsTypes = null;

                        switch (metaType.Item1)
                        {
                            case NintendoContentMetaConstant.ContentMetaTypeApplication:
                            case NintendoContentMetaConstant.ContentMetaTypeSystemProgram:
                                contentsTypes = new Tuple<string, string, int, bool>[]
                                {
                                    new Tuple<string, string, int, bool>("Program", ".nca", (int)NintendoContentArchivePartitionType.Logo, true),
                                    new Tuple<string, string, int, bool>("HtmlDocument", ".nca", 0, true),
                                    new Tuple<string, string, int, bool>("Meta", ".cnmt.nca", 0, false),
                                };
                                break;

                            case NintendoContentMetaConstant.ContentMetaTypePatch:
                                contentsTypes = new Tuple<string, string, int, bool>[]
                                {
                                    new Tuple<string, string, int, bool>("Program", ".nca", hasData ? (int)NintendoContentArchivePartitionType.Data : (int)NintendoContentArchivePartitionType.Code, true),
                                    new Tuple<string, string, int, bool>("HtmlDocument", ".nca", 0, true),
                                    new Tuple<string, string, int, bool>("Meta", ".cnmt.nca", 0, false),
                                };
                                break;

                            case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                            case NintendoContentMetaConstant.ContentMetaTypeSystemData:
                                contentsTypes = new Tuple<string, string, int, bool>[]
                                {
                                    new Tuple<string, string, int, bool>("Data", ".nca", 0, true),
                                    new Tuple<string, string, int, bool>("Meta", ".cnmt.nca", 0, false),
                                };
                                break;
                        }

                        foreach (var contentType in contentsTypes)
                        {
                            var align16KiB = metaType.Item3 && contentType.Item4;
                            Utils.WriteLine(" + " + contentType.Item1 + " -> " + (align16KiB ? "16KiB アラインあり" : "16KiB アラインなし"));

                            var contentMeta = model.ContentList.Find(v => v.Type == contentType.Item1);

                            var config = new AuthoringConfiguration();
                            config.KeyConfigFilePath = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";

                            byte[] firstPatitionData = null;
                            long firstPartionOffset = 0;
                            using (var nca = nsp.OpenNintendoContentArchiveReader(contentMeta.Id + contentType.Item2, new NcaKeyGenerator(config.GetKeyConfiguration())))
                            {
                                firstPartionOffset = nca.GetFsStartOffset(contentType.Item3);
                                if (align16KiB)
                                {
                                    Assert.AreEqual(16 * 1024, firstPartionOffset);
                                }
                                else
                                {
                                    Assert.AreEqual(3 * 1024, firstPartionOffset);
                                }

                                if (metaType.Item1 == NintendoContentMetaConstant.ContentMetaTypePatch)
                                {
                                    // SetExternalKey の ref はいずれ外されるので、newNca の後始末は考えなくてよい
                                    NintendoContentArchiveReader newNca = nca;
                                    TicketUtility.SetExternalKey(ref newNca, nsp);
                                }

                                using (var fs = nca.OpenFsDataStorageArchiveReader(contentType.Item3))
                                {
                                    firstPatitionData = fs.Read(0, (int)fs.GetSize());
                                    Assert.IsTrue(firstPatitionData.Length > 0);

                                    if (contentType.Item1 == "Program")
                                    {
                                        if (metaType.Item1 == NintendoContentMetaConstant.ContentMetaTypePatch)
                                        {
                                            // パッチはデータとコードしかないのでデータがある場合だけ見る
                                            if (hasData)
                                            {
                                                Assert.AreNotEqual(0, firstPatitionData.Length % (16 * 1024));

                                                // アプリと異なり、コードは 16KiB にアラインされない
                                                var offset = nca.GetFsStartOffset((int)NintendoContentArchivePartitionType.Code);
                                                Assert.AreNotEqual(0, offset);
                                                Assert.AreNotEqual(0, offset % (16 * 1024));
                                                Assert.AreEqual(0, offset % 512);
                                            }
                                        }
                                        else if (metaType.Item1 == NintendoContentMetaConstant.ContentMetaTypeApplication)
                                        {
                                            Assert.AreNotEqual(0, firstPatitionData.Length % (16 * 1024));

                                            var offset = nca.GetFsStartOffset((int)NintendoContentArchivePartitionType.Data);
                                            if (hasData)
                                            {
                                                Assert.AreNotEqual(0, offset);
                                                Assert.AreEqual(0, offset % (16 * 1024));

                                                using (var data = nca.OpenFsDataStorageArchiveReader((int)NintendoContentArchivePartitionType.Data))
                                                {
                                                    Assert.AreNotEqual(0, data.GetSize() % (16 * 1024));
                                                }
                                            }
                                            else
                                            {
                                                Assert.AreEqual(0, offset);
                                            }

                                            // パッチと異なり、コードは 16KiB にアラインされる
                                            offset = nca.GetFsStartOffset((int)NintendoContentArchivePartitionType.Code);
                                            Assert.AreNotEqual(0, offset);
                                            Assert.AreEqual(0, offset % (16 * 1024));
                                        }
                                        else
                                        {
                                            var offset = nca.GetFsStartOffset((int)NintendoContentArchivePartitionType.Data);
                                            Assert.AreNotEqual(0, offset);
                                            Assert.AreNotEqual(0, offset % (16 * 1024));
                                            Assert.AreEqual(0, offset % 512);

                                            offset = nca.GetFsStartOffset((int)NintendoContentArchivePartitionType.Code);
                                            Assert.AreNotEqual(0, offset);
                                            Assert.AreNotEqual(0, offset % (16 * 1024));
                                            Assert.AreEqual(0, offset % 512);
                                        }
                                    }
                                }
                            }

                            if (contentType.Item3 == (int)NintendoContentArchivePartitionType.Logo)
                            {
                                // ロゴは暗号化されないので生で読んでも一致する
                                var rawData = nsp.ReadFile(contentMeta.Id + ".nca", firstPartionOffset, firstPatitionData.Length);
                                Assert.IsTrue(firstPatitionData.SequenceEqual(rawData));
                            }
                        }
                    }
                }
            }
        }

        [TestMethod]
        public void TestExecutionCreatePatchNsp()
        {
            // TODO:  --type PatchはLegacyなパッチを作る処理で、obsoleteですので将来的に削除する予定です。
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\PatchMeta";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var metaFile = metaDir + "\\use_default.nmeta";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var testExtractDir = Path.Combine(outputDir, "extract");
            var config = new AuthoringConfiguration();

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            foreach (var file in Directory.EnumerateFiles(metaDir))
            {
                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile); // W/A: MakeMeta が PatchId に対応していない

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Patch --meta {1} --desc {2} --program {3} --save-adf",
                    outputPath,
                    file,
                    descFile,
                    dataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, null);

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);

                CheckGetNspProperty(outputPath);

                TestCreateNspMeta(outputPath, testExtractDir, file, "Patch", null);
            }
        }

        [TestMethod]
        public void TestExecutionCreateAddOnContentNspProd()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";

            var aocExcludeMetaDir = testSourceDir + "\\AddOnContentMetaWithExcludeList";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var testExtractDir = Path.Combine(outputDir, "extract");
            var testExtractEntryDir = Path.Combine(outputDir, "extractEntry");

            var aocDataDir = outputDir + "\\data";
            var aocDataFile = aocDataDir + "\\data.dat";

            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var config = new AuthoringConfiguration();

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(testExtractDir);
            Directory.CreateDirectory(dataDir);
            Directory.CreateDirectory(aocDataDir);
            CreateTestData(dataFile, 1024);
            CreateTestData(aocDataFile, 1024);

            // AddOnContent(除外ファイル、鍵世代ファイル使用)
            {
                ForceGC();
                // meta ファイル内のパスが meta ファイルからの間接参照なので、outputDir に meta ファイルをコピーする
                var nmetaName = "use_default_multiple.nmeta";
                var nmetaFile = Path.Combine(aocExcludeMetaDir, nmetaName);
                var metaFile = Path.Combine(outputDir, nmetaName);
                File.Copy(nmetaFile, metaFile);

                var excludeListPath = Path.Combine(aocExcludeMetaDir, "exclude.xml");
                var keygenListPath = Path.Combine(aocExcludeMetaDir, "keygen.xml");

                var outputPath = outputDir + "\\" + Path.GetFileName(nmetaName) + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type AddOnContent --keygeneration 2 --meta {1}",
                    outputPath,
                    metaFile));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);
                CheckAocGeneration(outputPath, config.GetKeyConfiguration());

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);

                string[] entryRuleStringArray;
                entryRuleStringArray = new string[] {
                        @".*\.nca/fs0/data\.dat",
                        @".*\.cnmt.xml",
                        @".*\.nca/fs0/data\.dat",
                        @".*\.cnmt.xml",
                        @".*\.nca/fs0/data\.dat",
                        @".*\.cnmt.xml",
                        @".*\.nca/fs0/data\.dat",
                        @".*\.cnmt.xml",
                        @".*\.nca/fs0/data\.dat",
                        @".*\.cnmt.xml",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c10001001.cnmt",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c10001002.cnmt",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c10001003.cnmt",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c10001004.cnmt",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c10001005.cnmt",
                        @"[0-9a-fA-F]*\.tik",
                        @"[0-9a-fA-F]*\.cert",
                        @"[0-9a-fA-F]*\.tik",
                        @"[0-9a-fA-F]*\.cert",
                        @"[0-9a-fA-F]*\.tik",
                        @"[0-9a-fA-F]*\.cert",
                        @"[0-9a-fA-F]*\.tik",
                        @"[0-9a-fA-F]*\.cert",
                        @"[0-9a-fA-F]*\.tik",
                        @"[0-9a-fA-F]*\.cert",
                        @"authoringtoolinfo.xml",
                    };

                TestExtractWithNsp(outputPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);
                SafeDeleteDirectory(testExtractDir);
                SafeDeleteDirectory(testExtractEntryDir);

                errorMsg = ExecuteProgram(string.Format("prodencryption --no-check --no-nspu --keyconfig {0} -o {1} --exclude-list {2} --keygen-list {3} {4}", keyConfigFile, outputDir, excludeListPath, keygenListPath, outputPath));
                Assert.IsTrue(error == string.Empty, error);

                var nspPath = outputDir + "\\" + Path.GetFileName(nmetaName) + "_prod.nsp";
                var xmlPath = outputDir + "\\" + Path.GetFileName(nmetaName) + "_prod.nsp.result.xml";
                List<UInt64> idList = new List<ulong>()
                {
                    0x0005000c10001001,
                    0x0005000c10001003,
                    0x0005000c10001005,
                };
                CheckAocResultXml(xmlPath, nspPath, idList);
                CheckNcaKeyGeneration(nspPath, new List<byte>() { 2, 2, 0, 0, 0, 0 });

                // --desired-safe-keygen
                errorMsg = ExecuteProgram(string.Format("prodencryption --desired-safe-keygen 3 --no-check --no-nspu --keyconfig {0} -o {1} --exclude-list {2} --keygen-list {3} {4}", keyConfigFile, outputDir, excludeListPath, keygenListPath, outputPath));
                Assert.IsTrue(error == string.Empty, error);

                CheckAocResultXml(xmlPath, nspPath, idList);
                CheckNcaKeyGeneration(nspPath, new List<byte>() { 3, 3, 0, 0, 0, 0 });
            }
        }

        [TestMethod]
        public void ErrorExecutionCreateAddOnContentNsp()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ErrorAddOnContent";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\data";
            var dataFile = dataDir + "\\data.dat";
            var keyConfigFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";
            var config = new AuthoringConfiguration();
            config.KeyConfigFilePath = keyConfigFile;
            SafeDeleteDirectory(dataDir);
            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            foreach (var file in Directory.EnumerateFiles(metaDir))
            {
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var tmpMetaFile = Path.Combine(outputDir, Path.GetFileName(file));
                File.Copy(file, tmpMetaFile);

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type AddOnContent --meta {1} --data {2} --save-adf",
                    outputPath,
                    tmpMetaFile,
                    dataDir));
                Assert.IsTrue(error != string.Empty, error);
            }
        }

        [TestMethod]
        public void TestExecutionCreateAddOnContentNsp()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\AddOnContentMeta";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\data";
            var dataFile = dataDir + "\\data.dat";
            var fdfFile = dataDir + "\\data.fdf";
            var keyConfigFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";
            var config = new AuthoringConfiguration();
            var testExtractDir = Path.Combine(outputDir, "extract");
            var testExtractEntryDir = Path.Combine(outputDir, "extractEntry");

            config.KeyConfigFilePath = keyConfigFile;
            SafeDeleteDirectory(dataDir);
            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            foreach (var file in Directory.EnumerateFiles(metaDir, "*.nmeta"))
            {
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var tmpMetaFile = Path.Combine(outputDir, Path.GetFileName(file));
                File.Copy(file, tmpMetaFile);

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type AddOnContent --meta {1} --save-adf --ignore-unpublishable-error",
                    outputPath,
                    tmpMetaFile));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);
                CheckAocGeneration(outputPath, config.GetKeyConfiguration());

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);
                CheckGetNspProperty(outputPath);
                CheckDataContents(outputPath, true, config.GetKeyConfiguration());

                ExtractAndShowNsp(outputPath);

                string[] entryRuleStringArray;
                entryRuleStringArray = new string[] {
                        @".*\.nca/fs0/data\.dat",
                        @".*\.cnmt.xml",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c10001004.cnmt",
                        @"[0-9a-fA-F]*\.tik",
                        @"[0-9a-fA-F]*\.cert",
                        @"authoringtoolinfo.xml",
                    };

                TestExtractWithNsp(outputPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);
                SafeDeleteDirectory(testExtractDir);
                SafeDeleteDirectory(testExtractEntryDir);
            }

            // 内部鍵版
            foreach (var file in Directory.EnumerateFiles(metaDir, "*.nmeta"))
            {
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var tmpMetaFile = Path.Combine(outputDir, Path.GetFileName(file));

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type AddOnContent --meta {1} --save-adf --no-ticket --ignore-unpublishable-error",
                    outputPath,
                    tmpMetaFile));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);
                CheckAocGeneration(outputPath, config.GetKeyConfiguration());

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);
                CheckGetNspProperty(outputPath);
                CheckDataContents(outputPath, true, config.GetKeyConfiguration());

                ExtractAndShowNsp(outputPath);

                string[] entryRuleStringArray;
                entryRuleStringArray = new string[] {
                        @".*\.nca/fs0/data\.dat",
                        @".*\.cnmt.xml",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c10001004.cnmt",
                        @"authoringtoolinfo.xml",
                    };

                TestExtractWithNsp(outputPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);
                SafeDeleteDirectory(testExtractDir);
                SafeDeleteDirectory(testExtractEntryDir);
            }

            // --error-unpublishable
            {
                var file = Directory.EnumerateFiles(metaDir, "use_default.nmeta").First();
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var tmpMetaFile = Path.Combine(outputDir, Path.GetFileName(file));
                File.Copy(file, tmpMetaFile, true);

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type AddOnContent --meta {1} --save-adf --error-unpublishable",
                    outputPath,
                    tmpMetaFile));
                Assert.IsFalse(error == string.Empty, error);
                VerifyNsp(outputPath, keyConfigFile);
            }

            // --error-unpublishable
            {
                var file = Directory.EnumerateFiles(metaDir, "describe_addoncontent_id.nmeta").First();
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var tmpMetaFile = Path.Combine(outputDir, Path.GetFileName(file));
                File.Copy(file, tmpMetaFile, true);

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type AddOnContent --meta {1} --save-adf --error-unpublishable",
                    outputPath,
                    tmpMetaFile));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);
            }

        }

        [TestMethod]
        public void TestExecutionCreateMultipleAddOnContentNsp()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\MultipleAddOnContentMeta";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\data";
            var dataFile = dataDir + "\\data.dat";
            var dataFile3 = dataDir + "\\data3.dat";
            var dataFile4 = dataDir + "\\data4.dat";

            var fdfFile3 = outputDir + "\\data3.fdf";
            var fdfFile4 = outputDir + "\\data4.fdf";
            var keyConfigFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";
            var config = new AuthoringConfiguration();
            var testExtractDir = Path.Combine(outputDir, "extract");
            var testExtractEntryDir = Path.Combine(outputDir, "extractEntry");

            config.KeyConfigFilePath = keyConfigFile;
            SafeDeleteDirectory(dataDir);
            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            string[] dataFiles = { dataFile, dataFile3, dataFile4 };
            foreach(var dataFilePath in dataFiles)
            {
                using (FileStream stream = File.Create(dataFilePath))
                {
                    int fileSize = 1024;
                    byte[] data = new byte[fileSize];
                    for (int i = 0; i < fileSize; i++)
                    {
                        data[i] = (byte)i;
                    }
                    stream.Write(data, 0, fileSize);
                }
            }

            using (FileStream stream = File.Create(fdfFile3))
            using (var writer = new StreamWriter(stream))
            {
                writer.WriteLine(string.Format("-\"data3.*\""));
            }

            using (FileStream stream = File.Create(fdfFile4))
            using (var writer = new StreamWriter(stream))
            {
                writer.WriteLine(string.Format("-\"data4.*\""));
            }

            foreach (var file in Directory.EnumerateFiles(metaDir, "*.nmeta"))
            {
                if(Path.GetFileName(file) == "use_default_multiple_no_adf.nmeta")
                {
                    continue;
                }
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var tmpMetaFile = Path.Combine(outputDir, Path.GetFileName(file));
                File.Copy(file, tmpMetaFile);

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type AddOnContent --meta {1} --save-adf",
                    outputPath,
                    tmpMetaFile));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);
                CheckAocGeneration(outputPath, config.GetKeyConfiguration());

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);
                CheckGetNspProperty(outputPath);
                CheckDataContents(outputPath, true, config.GetKeyConfiguration());

                ExtractAndShowNsp(outputPath);

                string[] entryRuleStringArray;
                entryRuleStringArray = new string[] {
                        @".*\.nca/fs0/data\.dat",
                        @".*\.nca/fs0/data3\.dat",
                        @".*\.cnmt.xml",
                        @".*\.nca/fs0/data\.dat",
                        @".*\.nca/fs0/data4\.dat",
                        @".*\.cnmt.xml",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c10001003.cnmt",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c10001004.cnmt",
                        @"[0-9a-fA-F]*\.tik",
                        @"[0-9a-fA-F]*\.cert",
                        @"[0-9a-fA-F]*\.tik",
                        @"[0-9a-fA-F]*\.cert",
                        @"authoringtoolinfo.xml",
                    };

                TestExtractWithNsp(outputPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);
                SafeDeleteDirectory(testExtractDir);
                SafeDeleteDirectory(testExtractEntryDir);
            }

            // --save-adf なしで adf が削除されることの確認
            {
                var file = Directory.EnumerateFiles(metaDir, "use_default_multiple_no_adf.nmeta").First();
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var tmpMetaFile = Path.Combine(outputDir, Path.GetFileName(file));
                File.Copy(file, tmpMetaFile);

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type AddOnContent --meta {1}",
                    outputPath,
                    tmpMetaFile));
                Assert.IsTrue(error == string.Empty, error);

                var foundAdf = Directory.EnumerateFiles(outputDir).Any((f) =>
                {
                    return Regex.IsMatch(f, @"use_default_multiple_no_adf\..+\.adf");
                });
                Assert.IsFalse(foundAdf);
            }

            // Version によって正しく Generation 設定されているか、コンテンツハッシュが変わっていないかどうか確認
            {
                var nspA1Path = outputDir + "\\describe_addoncontent_id_multiple.nmeta.nsp";
                var nspA2Path = outputDir + "\\describe_addoncontent_id_multiple2.nmeta.nsp";
                var nspB1Path = outputDir + "\\use_default_multiple.nmeta.nsp";
                var nspB2Path = outputDir + "\\use_default_multiple2.nmeta.nsp";

                CheckAocGenerationUpdated(nspA1Path, new List<uint>() { 1 * 65536, 1 * 65536 },
                                          nspA2Path, new List<uint>() { 1 * 65536, 2 * 65536 }, config.GetKeyConfiguration());

                CheckAocGenerationUpdated(nspB1Path, new List<uint>() { 4 * 65536, 4 * 65536 },
                                          nspB2Path, new List<uint>() { 5 * 65536, 4 * 65536 }, config.GetKeyConfiguration());
            }
        }

        [TestMethod]
        public void TestExecutionCreateMultipleAddOnContentNspWithErrorUnpublishable()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";

            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\data";
            var dataFile = dataDir + "\\data.dat";

            SafeDeleteDirectory(dataDir);
            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(outputDir);
            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            // 複数のコンテンツを含むAocでコンテンツ毎にエラーが出力されることの確認
            {
                Func<string ,string, int, int> checkResult = delegate (string type, string message, int offset)
                {
                    // コンテンツ毎にエラーが出力されていることを確認
                    Assert.IsTrue(message != string.Empty);
                    // Index 3 のコンテンツでタグ重複のエラーが発生する
                    Assert.IsTrue((offset = message.IndexOf("Index:3", offset)) >= 0);
                    Assert.IsTrue((offset = message.IndexOf("[Error] (Issue 10-612)", offset + 1)) >= 0);
                    Assert.IsTrue((offset = message.IndexOf("Tag : DuplicateName", offset + 1)) >= 0);
                    // Index 4 のコンテンツはエラーなし
                    Assert.IsTrue((offset = message.IndexOf("Index:4", offset + 1)) >= 0);
                    Assert.IsTrue(message.IndexOf("[Error]", offset + 1) > message.IndexOf("Index:", offset + 1));
                    // Index 5 のコンテンツは存在しない
                    Assert.IsFalse(message.Contains("Index:5"));
                    // Index が存在しないコンテンツで、Index未指定+タグ名が128より大きいエラーが発生する
                    if (type == "nmeta")
                    {
                        Assert.IsTrue((offset = message.IndexOf("Index:\r\n", offset + 1)) >= 0);
                        Assert.IsTrue((offset = message.IndexOf("[Error] (Issue 10-613)", offset + 1)) >= 0);
                        Assert.IsTrue((offset = message.IndexOf("[Error] (Issue 10-814)", offset + 1)) >= 0);
                    }
                    else
                    {
                        // nspではindexが指定されていないことはありえないので期待値を変える
                        Assert.IsTrue((offset = message.IndexOf("Index:0", offset + 1)) >= 0);
                        Assert.IsTrue((offset = message.IndexOf("[Error] (Issue 10-613)", offset + 1)) >= 0);
                        Assert.IsFalse((message.IndexOf("[Error] (Issue 10-814)", offset + 1)) >= 0);
                    }
                    // Index 7 のコンテンツでタグ重複のエラーが発生する
                    Assert.IsTrue((offset = message.IndexOf("Index:7", offset + 1)) >= 0);
                    Assert.IsTrue((offset = message.IndexOf("[Error] (Issue 10-612)", offset + 1)) >= 0);
                    Assert.IsTrue((offset = message.IndexOf("Tag : DuplicateName", offset + 1)) >= 0);
                    return offset;
                };

                var metaFile = outputDir + "\\describe_aoc_multi_content_error.nmeta";
                var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
                var meta = @"<?xml version=""1.0""?>
                            <NintendoSdkMeta>
                              <AddOnContent>
                                <Index>3</Index>
                                <ApplicationId>0x0005000c10000000</ApplicationId>
                                <RequiredApplicationReleaseVersion>1</RequiredApplicationReleaseVersion>
                                <Tag>DuplicateName</Tag>
                                <DataPath>data</DataPath>
                                <ReleaseVersion>1</ReleaseVersion>
                                <PrivateVersion>1</PrivateVersion>
                              </AddOnContent>
                              <AddOnContent>
                                <Index>4</Index>
                                <ApplicationId>0x0005000c10000000</ApplicationId>
                                <RequiredApplicationReleaseVersion>1</RequiredApplicationReleaseVersion>
                                <Tag>UniqueName</Tag>
                                <DataPath>data</DataPath>
                                <ReleaseVersion>1</ReleaseVersion>
                              </AddOnContent>
                              <AddOnContent>
                                <ApplicationId>0x0005000c10000000</ApplicationId>
                                <RequiredApplicationReleaseVersion>2</RequiredApplicationReleaseVersion>
                                <Tag>123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789</Tag>
                                <DataPath>data</DataPath>
                                <ReleaseVersion>1</ReleaseVersion>
                              </AddOnContent>
                              <AddOnContent>
                                <Id>0x0005000c10001007</Id>
                                <ApplicationId>0x0005000c10000000</ApplicationId>
                                <RequiredApplicationReleaseVersion>2</RequiredApplicationReleaseVersion>
                                <Tag>DuplicateName</Tag>
                                <DataPath>data</DataPath>
                                <ReleaseVersion>1</ReleaseVersion>
                              </AddOnContent>
                            </NintendoSdkMeta>";
                File.WriteAllText(metaFile, meta);

                string standardError;
                string standardOut;
                ExecuteProgram(out standardError, out standardOut, string.Format(
                    "creatensp -o {0} --type AddOnContent --meta {1} --save-adf --error-unpublishable",
                    outputPath,
                    metaFile));
                Assert.IsTrue(standardError.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0, standardError);
                Assert.IsFalse(File.Exists(outputPath));
                int index = 0;
                index = checkResult("nmeta", standardOut, index);
                Assert.IsFalse(standardOut.IndexOf("UnpublishableError", index) >= 0);

                // --ignore-unpublishable-errorオプションを指定した時にエラーは出力されるが例外が発生しないことを確認
                ExecuteProgram(out standardError, out standardOut, string.Format(
                    "creatensp -o {0} --type AddOnContent --meta {1} --save-adf --error-unpublishable --ignore-unpublishable-error",
                    outputPath,
                    metaFile));
                Assert.IsTrue(standardError == string.Empty);
                Assert.IsTrue(File.Exists(outputPath));

                // nmetaのチェック時と、nspのチェック時で2回警告が検出される
                index = 0;
                index = checkResult("nmeta", standardOut, index);
                Assert.IsTrue(standardOut.IndexOf("UnpublishableError", index) >= 0);
                index = checkResult("nsp", standardOut, index);
                Assert.IsFalse(standardOut.IndexOf("UnpublishableError", index) >= 0);
            }

            SafeDeleteDirectory(dataDir);
            SafeDeleteDirectory(outputDir);
        }

        [TestMethod]
        public void TestExecutionCreateSystemDataNsp()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\SystemDataMeta";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\data";
            var dataFile = dataDir + "\\data.dat";
            var keyConfigFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";
            var config = new AuthoringConfiguration();
            config.KeyConfigFilePath = keyConfigFile;
            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            foreach (var file in Directory.EnumerateFiles(metaDir))
            {
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type SystemData --meta {1} --data {2} --save-adf --keyconfig {3}",
                    outputPath,
                    file,
                    dataDir,
                    keyConfigFile
                    ));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                string errorMsg = ExecuteProgram(string.Format("list {0} --keyconfig {1}", outputPath, keyConfigFile));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);
                CheckGetNspProperty(outputPath, keyConfigFile);
                CheckDataContents(outputPath, false, config.GetKeyConfiguration());

                TestContentMeta testMeta = new TestContentMeta();
                testMeta.SetFromMetaFile(file, "/NintendoSdkMeta/SystemData", 2, NintendoContentMeta.GetDefaultVersion()); // SystemData = 2
                CheckContentMeta(outputPath, "SystemData", testMeta, keyConfigFile);
            }

            // public
            foreach (var file in Directory.EnumerateFiles(metaDir))
            {
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type SystemData --public-system-data --meta {1} --data {2} --save-adf --keyconfig {3}",
                    outputPath,
                    file,
                    dataDir,
                    keyConfigFile
                    ));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                string errorMsg = ExecuteProgram(string.Format("list {0} --keyconfig {1}", outputPath, keyConfigFile));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);
                CheckGetNspProperty(outputPath, keyConfigFile);
                CheckDataContents(outputPath, true, config.GetKeyConfiguration());

                TestContentMeta testMeta = new TestContentMeta();
                testMeta.SetFromMetaFile(file, "/NintendoSdkMeta/SystemData", 2, NintendoContentMeta.GetDefaultVersion()); // SystemData = 2
                CheckContentMeta(outputPath, "SystemData", testMeta, keyConfigFile);
            }
        }

        [TestMethod]
        public void ErrorExecutionCreateSystemDataNspWithVersionAndReleaseVersion()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaFile = testSourceDir + "\\ErrorVersion\\systemdata.nmeta";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\data";
            var dataFile = dataDir + "\\data.dat";
            var keyConfigFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";
            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";

            var error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type SystemData --meta {1} --data {2} --save-adf --keyconfig {3}",
                outputPath,
                metaFile,
                dataDir,
                keyConfigFile
                ));
            Assert.IsTrue(error.IndexOf("<Version> cannot be specified with <ReleaseVersion>") >= 0);

            // public
            error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type SystemData --public-system-data --meta {1} --data {2} --save-adf --keyconfig {3}",
                outputPath,
                metaFile,
                dataDir,
                keyConfigFile
                ));
            Assert.IsTrue(error.IndexOf("<Version> cannot be specified with <ReleaseVersion>") >= 0);
        }

        [TestMethod]
        public void TestExecutionCreateSystemDataExFatNsp()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\SystemDataMetaExFat";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\data";
            var dataFile = dataDir + "\\data.dat";
            var keyConfigFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";
            var config = new AuthoringConfiguration();
            config.KeyConfigFilePath = keyConfigFile;
            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            foreach (var file in Directory.EnumerateFiles(metaDir))
            {
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type SystemData --meta {1} --data {2} --save-adf --keyconfig {3}",
                    outputPath,
                    file,
                    dataDir,
                    keyConfigFile
                    ));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                string errorMsg = ExecuteProgram(string.Format("list {0} --keyconfig {1}", outputPath, keyConfigFile));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);
                CheckDataContents(outputPath, false, config.GetKeyConfiguration());
            }

            // public
            foreach (var file in Directory.EnumerateFiles(metaDir))
            {
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type SystemData --public-system-data --meta {1} --data {2} --save-adf --keyconfig {3}",
                    outputPath,
                    file,
                    dataDir,
                    keyConfigFile
                    ));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                string errorMsg = ExecuteProgram(string.Format("list {0} --keyconfig {1}", outputPath, keyConfigFile));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);
                CheckDataContents(outputPath, true, config.GetKeyConfiguration());
            }
        }

        [TestMethod]
        public void TextExecutionCreateSystemUpdateMetaNsp()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var metaDirectoryPath = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources\\SystemUpdateMeta";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\data";
            var dataFile = dataDir + "\\data.dat";
            var keyConfigFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";
            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            foreach (var file in Directory.EnumerateFiles(metaDirectoryPath))
            {
                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                System.Console.WriteLine("Testing {0}...", file);
                string errorMsg = ExecuteProgram(string.Format("creatensp -o {0} --type SystemUpdate --meta {1} --data {2} --keyconfig {3}",
                    outputPath,
                    file,
                    dataDir,
                    keyConfigFile));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                errorMsg = ExecuteProgram(string.Format("list {0} --keyconfig {1}", outputPath, keyConfigFile));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);
                CheckGetNspProperty(outputPath, keyConfigFile);

                TestContentMeta testMeta = new TestContentMeta();
                testMeta.SetFromMetaFile(file, "/NintendoSdkMeta/SystemUpdate/ContentMeta", 3); // SystemUpdate = 3
                CheckContentMeta(outputPath, "SystemUpdate", testMeta, keyConfigFile);
            }
        }

        [TestMethod]
        public void ErrorExecutionCreateSystemUpdateMetaNspWithVersionAndReleaseVersion()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var metaFilePath = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources\\ErrorVersion\\systemupdate.nmeta";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\data";
            var dataFile = dataDir + "\\data.dat";
            var keyConfigFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";
            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            var outputPath = outputDir + "\\" + Path.GetFileName(metaFilePath) + ".nsp";
            System.Console.WriteLine("Testing {0}...", metaFilePath);
            string errorMsg = ExecuteProgram(string.Format("creatensp -o {0} --type SystemUpdate --meta {1} --data {2} --keyconfig {3}",
                outputPath,
                metaFilePath,
                dataDir,
                keyConfigFile));
            Assert.IsTrue(errorMsg.IndexOf("<Version> cannot be specified with <ReleaseVersion>") >= 0);
        }

        public void MakeTestMetaFile(string metaFile, string[] lines, bool isOldRoot = false)
        {
            using (FileStream stream = File.Create(metaFile))
            {
                StreamWriter sw = new StreamWriter(stream);
                WriteTmpMetaFileHeader(sw, isOldRoot);

                foreach (var line in lines)
                {
                    sw.WriteLine(line);
                }

                WriteTmpMetaFileFooter(sw, isOldRoot);

                sw.Flush();
            }
        }

        public void ExecuteMetaTest(out string outputPath, out string standardError, out string standardOut, string metaFile, string outputDir, string codeDir)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

            string testNpdm = codeDir + "\\main.npdm";
            MakeNpdm(testNpdm, metaFile, descFile);

            outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";

            ExecuteProgram(
                out standardError,
                out standardOut,
                string.Format(
                "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                outputPath,
                metaFile,
                descFile,
                codeDir));

            return;
        }
        public string ExecuteMetaTest(out string outputPath, string metaFile, string outputDir, string codeDir, bool isStandardOut)
        {
            string standardError;
            string standardOut;
            ExecuteMetaTest(out outputPath, out standardError, out standardOut, metaFile, outputDir, codeDir);
            return (isStandardOut == true)? standardOut : standardError;
        }

        public string ExecuteMetaTest(out string outputPath, string metaFile, string outputDir, string codeDir)
        {
            return ExecuteMetaTest(out outputPath, metaFile, outputDir, codeDir, false);
        }

        [TestMethod]
        public void TestMetaFile()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var testDir = Path.Combine(outputDir, "test");
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var iconPath = testSourceDir + "\\Icon\\describe_all.bmp";

            SafeDeleteDirectory(outputDir);
            SafeDeleteDirectory(testDir);
            SafeDeleteDirectory(dataDir);

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            Directory.CreateDirectory(testDir);

            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Test0Test</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<Icon>",
                    "<Language>AmericanEnglish</Language>",
                    string.Format("<IconPath>{0}</IconPath>",iconPath),
                    "</Icon>",
                };
                string metaFile = testDir + "utf8_success1.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Test</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "</Title>",
                    "<Icon>",
                    "<Language>AmericanEnglish</Language>",
                    string.Format("<IconPath>{0}</IconPath>",iconPath),
                    "</Icon>",
                };

                string metaFile = testDir + "title_language_fail.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                string standardError;
                string standardOut;
                ExecuteMetaTest(out outputPath, out standardError, out standardOut, metaFile, outputDir, dataDir);

                Assert.IsTrue(standardError != string.Empty);
                Assert.IsTrue(standardError.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0);
                Assert.IsTrue(standardOut != string.Empty);
                Assert.IsTrue(standardOut.IndexOf("<Title>") >= 0 && standardOut.IndexOf("<Language>") >= 0);
                Assert.IsTrue(standardOut.IndexOf("[Error] (Issue 10-806)") >= 0);
                Assert.IsFalse(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Test</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>InvalidLanguage</Language>",
                    "</Title>",
                    "<Icon>",
                    "<Language>AmericanEnglish</Language>",
                    string.Format("<IconPath>{0}</IconPath>",iconPath),
                    "</Icon>",
                };

                string metaFile = testDir + "title_language_fail.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                string standardError;
                string standardOut;
                ExecuteMetaTest(out outputPath, out standardError, out standardOut, metaFile, outputDir, dataDir);

                Assert.IsTrue(standardError != string.Empty);
                Assert.IsTrue(standardError.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0);
                Assert.IsTrue(standardOut != string.Empty);
                Assert.IsTrue(standardOut.IndexOf("<Language>") >= 0);
                Assert.IsTrue(standardOut.IndexOf("[Error] (Issue 10-807)") >= 0);
                Assert.IsFalse(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Test\u0180Test</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<Icon>",
                    "<Language>AmericanEnglish</Language>",
                    string.Format("<IconPath>{0}</IconPath>",iconPath),
                    "</Icon>",
                };

                string metaFile = testDir + "utf8_fail1.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                string standardError;
                string standardOut;
                ExecuteMetaTest(out outputPath, out standardError, out standardOut,metaFile, outputDir, dataDir);

                Assert.IsTrue(standardError != string.Empty);
                Assert.IsTrue(standardError.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0);
                Assert.IsTrue(standardOut != string.Empty);
                Assert.IsTrue(standardOut.IndexOf("<Name>") >= 0);
                Assert.IsTrue(standardOut.IndexOf("[Error] (Issue 10-808)") >= 0);
                Assert.IsFalse(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Test</Name>",
                    "<Publisher>Publi\nsher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<Icon>",
                    "<Language>AmericanEnglish</Language>",
                    string.Format("<IconPath>{0}</IconPath>",iconPath),
                    "</Icon>",
                };

                string metaFile = testDir + "utf8_fail2.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                string standardError;
                string standardOut;
                ExecuteMetaTest(out outputPath, out standardError, out standardOut,metaFile, outputDir, dataDir);

                Assert.IsTrue(standardError != string.Empty);
                Assert.IsTrue(standardError.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0);
                Assert.IsTrue(standardOut != string.Empty);
                Assert.IsTrue(standardOut.IndexOf("<Publisher>") >= 0);
                Assert.IsTrue(standardOut.IndexOf("[Error] (Issue 10-808)") >= 0);
                Assert.IsFalse(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Name</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<Icon>",
                    "<Language>InvalidLanguage</Language>",
                    string.Format("<IconPath>{0}</IconPath>",iconPath),
                    "</Icon>",
                };

                string metaFile = testDir + "icon_language_fail.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error != string.Empty);
                Assert.IsTrue(error.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0, error);
                Assert.IsFalse(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Name</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<Icon>",
                    "<Language>AmericanEnglish</Language>",
                    "<IconPath>InvalidPath</IconPath>",
                    "</Icon>",
                };

                string metaFile = testDir + "icon_path_fail.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error != string.Empty);
                Assert.IsTrue(error.IndexOf("[Error] 使用されたパラメーターが有効ではありません。") >= 0, error);
                Assert.IsFalse(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<ApplicationErrorCodeCategory> ~aZ09</ApplicationErrorCodeCategory>",
                };
                string metaFile = testDir + "ascii_success1.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<ApplicationErrorCodeCategory>日本語</ApplicationErrorCodeCategory>",
                };
                string metaFile = testDir + "ascii_fail1.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                string standardError;
                string standardOut;
                ExecuteMetaTest(out outputPath, out standardError, out standardOut, metaFile, outputDir, dataDir);

                Assert.IsTrue(standardError != string.Empty);
                Assert.IsTrue(standardError.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0);
                Assert.IsTrue(standardOut != string.Empty);
                Assert.IsTrue(standardOut.IndexOf("<ApplicationErrorCodeCategory>") >= 0 && standardOut.IndexOf("ASCII") >= 0);
                Assert.IsTrue(standardOut.IndexOf("[Error] (Issue 10-809)") >= 0);
                Assert.IsFalse(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<Unknown>Test</Unknown>",
                };
                string metaFile = testDir + "unknowntag_fail.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error != string.Empty, error);
                Assert.IsTrue(error.IndexOf("<Unknown> is not supported in <Application>") >= 0, error);
                Assert.IsFalse(File.Exists(outputPath));
            }

            {
                // ApplicationControlPropertyModelに存在しないTouchScreenUsageタグがnmetaに含まれていても無視される
                string[] metaLines = new string[]
                {
                    "<TouchScreenUsage>Supported</TouchScreenUsage>",
                };
                string metaFile = testDir + "ignore_tags_success.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<Title>",
                    "<Name>Test0Test</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<Icon>",
                    "<Language>AmericanEnglish</Language>",
                    string.Format("<IconPath>{0}</IconPath>",iconPath),
                    "</Icon>",
                };
                string metaFile = testDir + "old_metafile_success1.nmeta";
                MakeTestMetaFile(metaFile, metaLines, true);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<BcatSaveDataSize >0x100000</BcatSaveDataSize>",
                };
                string metaFile = testDir + "BcatSaveDataSize.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error != string.Empty, error);
                Assert.IsTrue(error.IndexOf("<BcatSaveDataSize> is not supported in <Application>") >= 0);
                Assert.IsFalse(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<BcatDeliveryCacheStorageSize>0x500000</BcatDeliveryCacheStorageSize>",
                    "<BcatPassphrase>0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef</BcatPassphrase>",
                };
                string metaFile = testDir + "Bcat_success1.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(string.IsNullOrEmpty(error), error);
                Assert.IsTrue(File.Exists(outputPath));
            }
            {
                string[] metaLines = new string[]
                {
                    "<BcatDeliveryCacheStorageSize>0x4000000</BcatDeliveryCacheStorageSize>",
                    "<BcatPassphrase>0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF</BcatPassphrase>",
                };
                string metaFile = testDir + "Bcat_success2.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(string.IsNullOrEmpty(error), error);
                Assert.IsTrue(File.Exists(outputPath));
            }

            {
                string[] metaLines = new string[]
                {
                    "<BcatDeliveryCacheStorageSize>0x50000</BcatDeliveryCacheStorageSize>",
                    "<BcatPassphrase>0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef</BcatPassphrase>",
                };
                string metaFile = testDir + "Bcat_invalidSize1.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error != string.Empty, error);
                Assert.IsTrue(error.IndexOf("<BcatDeliveryCacheStorageSize> が 0x100000 (1 MiB) 単位になっていません: '0x50000'") >= 0);
                Assert.IsFalse(File.Exists(outputPath));
            }
            {
                string[] metaLines = new string[]
                {
                    "<BcatDeliveryCacheStorageSize>0x400000</BcatDeliveryCacheStorageSize>",
                    "<BcatPassphrase>0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef</BcatPassphrase>",
                };
                string metaFile = testDir + "Bcat_invalidSize2.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error != string.Empty, error);
                Assert.IsTrue(error.IndexOf("<BcatDeliveryCacheStorageSize> は 0x500000 (5 MiB) 以上で指定してください: '0x400000'") >= 0);
                Assert.IsFalse(File.Exists(outputPath));
            }
            {
                // 64 MiB 以上でもエラーにはならない
                string[] metaLines = new string[]
                {
                    "<BcatDeliveryCacheStorageSize>0x4100000</BcatDeliveryCacheStorageSize>",
                    "<BcatPassphrase>0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef</BcatPassphrase>",
                };
                string metaFile = testDir + "Bcat_success3.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(string.IsNullOrEmpty(error), error);
                Assert.IsTrue(File.Exists(outputPath));
            }
            {
                string[] metaLines = new string[]
                {
                    "<BcatDeliveryCacheStorageSize>0x500000</BcatDeliveryCacheStorageSize>",
                    "<BcatPassphrase>0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!@</BcatPassphrase>",
                };
                string metaFile = testDir + "Bcat_invalidPhrase1.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error != string.Empty, error);
                Assert.IsTrue(error.IndexOf("<BcatPassphrase> に 16 進数 (a-fA-F0-9) ではない文字が含まれています。: 0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!") >= 0);
                Assert.IsFalse(File.Exists(outputPath));
            }
            {
                string[] metaLines = new string[]
                {
                    "<BcatDeliveryCacheStorageSize>0x500000</BcatDeliveryCacheStorageSize>",
                    "<BcatPassphrase>0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcde</BcatPassphrase>",
                };
                string metaFile = testDir + "Bcat_invalidPhrase2.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                var error = ExecuteMetaTest(out outputPath, metaFile, outputDir, dataDir);

                Assert.IsTrue(error != string.Empty, error);
                Assert.IsTrue(error.IndexOf("<BcatPassphrase> は 64 文字で指定してください: '0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcde' (63 文字)") >= 0);
                Assert.IsFalse(File.Exists(outputPath));
            }
            {
                string[] metaLines = new string[]
                {
                    "<BcatDeliveryCacheStorageSize>0x500000</BcatDeliveryCacheStorageSize>",
                    "<BcatPassphrase>0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef0</BcatPassphrase>",
                };
                string metaFile = testDir + "Bcat_invalidPhrase3.nmeta";
                MakeTestMetaFile(metaFile, metaLines);

                string outputPath;
                string standardError;
                string standardOut;
                ExecuteMetaTest(out outputPath, out standardError, out standardOut, metaFile, outputDir, dataDir);

                Assert.IsTrue(standardError != string.Empty);
                Assert.IsTrue(standardError.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0);
                Assert.IsTrue(standardOut != string.Empty);
                Assert.IsTrue(standardOut.IndexOf("<BcatPassphrase>") >= 0 && standardOut.IndexOf("64") >= 0);
                Assert.IsTrue(standardOut.IndexOf("[Error] (Issue 10-810)") >= 0);
                Assert.IsFalse(File.Exists(outputPath));
            }
        }

        private void DeleteApplicationNodeChildMetaFile(string metaFile, string nodeName)
        {
            var document = new XmlDocument();
            document.Load(metaFile);

            var applicationNode = document.SelectSingleNode("//Application");
            var nodeList = applicationNode.SelectNodes(nodeName);

            foreach (XmlNode node in nodeList)
            {
                applicationNode.RemoveChild(node);
            }

            document.Save(metaFile);
        }

        private void AppendApplicationNodeChildMetaFile(string metaFile, string nodeName, string value)
        {
            var document = new XmlDocument();
            document.Load(metaFile);

            var applicationNode = document.SelectSingleNode("//Application");
            var node = document.CreateNode(XmlNodeType.Element, nodeName, string.Empty);

            node.InnerText = value;

            applicationNode.AppendChild(node);
            document.Save(metaFile);
        }

        private void AppendIconNodeMetaFile(string metaFile, string language, string iconPath, string nxIconPath)
        {
            var document = new XmlDocument();
            document.Load(metaFile);

            var applicationNode = document.SelectSingleNode("//Application");
            var iconNode = document.CreateNode(XmlNodeType.Element, "Icon", string.Empty);
            if (iconPath != null)
            {
                var node = document.CreateNode(XmlNodeType.Element, "IconPath", string.Empty);
                node.InnerText = iconPath;
                iconNode.AppendChild(node);
            }
            if (nxIconPath != null)
            {
                var node = document.CreateNode(XmlNodeType.Element, "NxIconPath", string.Empty);
                node.InnerText = nxIconPath;
                iconNode.AppendChild(node);
            }
            if (language != null)
            {
                var node = document.CreateNode(XmlNodeType.Element, "Language", string.Empty);
                node.InnerText = language;
                iconNode.AppendChild(node);
            }

            applicationNode.AppendChild(iconNode);
            document.Save(metaFile);
        }

        [TestMethod]
        public void TestExecutionErrorUnpublishableLegalinfoHashCheck()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var metaFile = GetDefaultApplicationMetaFile(metaDir);
            var iconDir = testSourceDir + "\\Icon";
            var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".bmp";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var codeDir = outputDir + "\\program";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var legalInfoZipDir = testSourceDir + "\\ErrorUnpublishable";
            var dataFile = codeDir + "\\data.dat";

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(codeDir);
            string testNpdm = codeDir + "\\main.npdm";
            MakeNpdm(testNpdm, metaFile, descFile);

            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";

            // nmetaファイルを書き換えるのでワーク領域にコピー
            var tmpMetaFilePath = outputDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + "_test.nmeta";
            File.Copy(metaFile, tmpMetaFilePath, true);

            // nmetaにIconエントリを追加
            AppendIconNodeMetaFile(tmpMetaFilePath, "AmericanEnglish", iconPath, null);
            AppendIconNodeMetaFile(tmpMetaFilePath, "Japanese", iconPath, null);

            var issueString = "[Error] (Issue 10-812)";
            var exceptionString = "[Error] Found one or more unpublishable error in nsp. The nsp will be deleted automatically.";

            Func<string, string, string, Tuple<string, string>> testCreateNsp = delegate (string outputNsp, string legalInfoZipPath, string addOption)
            {
                legalInfoZipPath = legalInfoZipDir + "//" + legalInfoZipPath;
                if (File.Exists(outputNsp))
                {
                    File.Delete(outputNsp);
                }
                // nmetaのLegalInformationFilePathを更新
                DeleteApplicationNodeChildMetaFile(tmpMetaFilePath, "LegalInformationFilePath");
                AppendApplicationNodeChildMetaFile(tmpMetaFilePath, "LegalInformationFilePath", legalInfoZipPath);

                string standardError;
                string standardOut;
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf {4}",
                    outputNsp,
                    tmpMetaFilePath,
                    descFile,
                    codeDir,
                    addOption));

                return Tuple.Create(standardOut, standardError);
            };

            // --error-unpublishableオプション無しでcreatensp実行
            {
                // ハッシュが正しいソフトリーガル情報
                // →Issue 10-812のエラーが表示されない + 例外が発生しない + nspが作成される ことを確認
                var result = testCreateNsp(outputPath, "legal_info.zip", string.Empty);
                Assert.IsTrue(result.Item1.Contains(issueString) == false);
                Assert.IsTrue(result.Item2 == string.Empty, result.Item2);
                Assert.IsTrue(File.Exists(outputPath));
            }
            {
                // ハッシュが不正なソフトリーガル情報
                // →Issue 10-812のエラーが表示される + 例外が発生する + nspが作成されない ことを確認
                var result = testCreateNsp(outputPath, "legal_info_invalid_hash.zip", string.Empty);
                Assert.IsTrue(result.Item1.Contains(issueString) == true);
                Assert.IsTrue(result.Item2.Contains(exceptionString) == true);
                Assert.IsFalse(File.Exists(outputPath));
            }
            {
                // ハッシュが不正だがフォーマットバージョンが2.x.xより下なソフトリーガル情報
                // →Issue 10-812のエラーが表示されない + 例外が発生しない + nspが作成される ことを確認
                var result = testCreateNsp(outputPath, "legal_info_invalid_hash_v1.zip", string.Empty);
                Assert.IsTrue(result.Item1.Contains(issueString) == false);
                Assert.IsTrue(result.Item2 == string.Empty, result.Item2);
                Assert.IsTrue(File.Exists(outputPath));
            }
            {
                // ipnotices.htdocs/index.htmlが存在しないソフトリーガル情報
                // →Issue 10-812のエラーが表示される + 例外が発生する + nspが作成されない ことを確認
                var result = testCreateNsp(outputPath, "legal_info_without_index_html.zip", string.Empty);
                Assert.IsTrue(result.Item1.Contains(issueString) == true);
                Assert.IsTrue(result.Item2.Contains(exceptionString) == true);
                Assert.IsFalse(File.Exists(outputPath));
            }

            // --error-unpublishableオプション有りでcreatensp実行
            {
                // ハッシュが正しいソフトリーガル情報
                // →Issue 10-812のエラーが表示されないことを確認
                // ダミープログラムの為、他のunpublishableエラーが発生する可能性があるので例外が発生しないことの確認は実施しない
                var result = testCreateNsp(outputPath, "legal_info.zip", "--error-unpublishable");
                Assert.IsTrue(result.Item1.Contains(issueString) == false);
            }
            {
                // ハッシュが不正なソフトリーガル情報
                // →Issue 10-812のエラーが表示される + 例外が発生する + nspが作成されない ことを確認
                var result = testCreateNsp(outputPath, "legal_info_invalid_hash.zip", "--error-unpublishable");
                Assert.IsTrue(result.Item1.Contains(issueString) == true);
                Assert.IsTrue(result.Item2.Contains(exceptionString) == true);
                Assert.IsFalse(File.Exists(outputPath));
            }

            // get-unpublishable-error実行
            {
                // ハッシュが正しいソフトリーガル情報
                // →Issue 10-812のエラーが表示されないことを確認
                testCreateNsp(outputPath, "legal_info.zip", "--ignore-unpublishable-error");
                var standardOut = ExecuteProgram("get-unpublishable-error " + outputPath, true);
                Assert.IsTrue(standardOut.Contains(issueString) == false);
            }
            {
                // ハッシュが不正なソフトリーガル情報
                // →Issue 10-812のエラーが表示されることを確認
                testCreateNsp(outputPath, "legal_info_invalid_hash.zip", "--ignore-unpublishable-error");
                var standardOut = ExecuteProgram("get-unpublishable-error " + outputPath, true);
                Assert.IsTrue(standardOut.Contains(issueString) == true);
            }

            var replaceNsp = outputDir + "\\" + Path.GetFileName(metaFile) + "_replaced.nsp";

            Func<string, string, Tuple<string, string>> testReplaceNsp = delegate (string originalNsp, string addOption)
            {
                // 展開して置換用ファイルのパスを取得
                var testExtractDir = Path.Combine(outputDir, "extract");
                SafeDeleteDirectory(testExtractDir);
                var errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", originalNsp, testExtractDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                var entryList = GetEntryFilePathList(testExtractDir);
                var testData = entryList.Find(entry => entry.EndsWith("/data.dat"));
                var testDataFilePath = Path.Combine(testExtractDir, testData);
                // データを加工
                byte[] data = File.ReadAllBytes(testDataFilePath);
                data[0] += 1;
                File.WriteAllBytes(testDataFilePath, data);

                if (File.Exists(replaceNsp))
                {
                    File.Delete(replaceNsp);
                }

                string standardError;
                string standardOut;
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "replace {0} {1} {2} --desc {3} -o {4} {5}",
                    originalNsp,
                    testData,
                    testDataFilePath,
                    descFile,
                    outputDir,
                    addOption));

                return Tuple.Create(standardOut, standardError);
            };

            // --error-unpublishableオプション無しでreplace実行
            {
                // ハッシュが正しいソフトリーガル情報
                // →Issue 10-812のエラーが表示されない + 例外が発生しない + nspが作成される ことを確認
                testCreateNsp(outputPath, "legal_info.zip", "--ignore-unpublishable-error");
                var result = testReplaceNsp(outputPath, string.Empty);
                Assert.IsTrue(result.Item1.Contains(issueString) == false);
                Assert.IsTrue(result.Item2 == string.Empty, result.Item2);
                Assert.IsTrue(File.Exists(replaceNsp));
            }
            {
                // ハッシュが不正なソフトリーガル情報
                // →Issue 10-812のエラーが表示される + 例外が発生する + nspが作成されない ことを確認
                testCreateNsp(outputPath, "legal_info_invalid_hash.zip", "--ignore-unpublishable-error");
                var result = testReplaceNsp(outputPath, string.Empty);
                Assert.IsTrue(result.Item1.Contains(issueString) == true);
                Assert.IsTrue(result.Item2.Contains(exceptionString) == true);
                Assert.IsFalse(File.Exists(replaceNsp));
            }

            // --error-unpublishableオプション有りでreplace実行
            {
                // ハッシュが正しいソフトリーガル情報
                // →Issue 10-812のエラーが表示されないことを確認
                // ダミープログラムの為、他のunpublishableエラーが発生する可能性があるので例外が発生しないことの確認は実施しない
                testCreateNsp(outputPath, "legal_info.zip", "--ignore-unpublishable-error");
                var result = testReplaceNsp(outputPath, "--error-unpublishable");
                Assert.IsTrue(result.Item1.Contains(issueString) == false);
            }
            {
                // ハッシュが不正なソフトリーガル情報
                // →Issue 10-812のエラーが表示される + 例外が発生する + nspが作成されない ことを確認
                testCreateNsp(outputPath, "legal_info_invalid_hash.zip", " --ignore-unpublishable-error");
                var result = testReplaceNsp(outputPath, "--error-unpublishable");
                Assert.IsTrue(result.Item1.Contains(issueString) == true);
                Assert.IsTrue(result.Item2.Contains(exceptionString) == true);
                Assert.IsFalse(File.Exists(replaceNsp));
            }

            // ncaが暗号化されているときにリーガル情報を取得してチェックできることを確認
            {
                // ハッシュが正常なソフトリーガル情報
                // →全てのコマンドで Issue 10-812 のエラーが表示されない ことを確認
                var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
                var result = testCreateNsp(outputPath, "legal_info.zip", string.Format(" --keyconfig {0} --keygeneration 3 --ticket", keyConfigFile));
                Assert.IsTrue(result.Item1.Contains(issueString) == false);
                Assert.IsTrue(result.Item2 == string.Empty, result.Item2);
                Assert.IsTrue(File.Exists(outputPath));

                var standardOut = ExecuteProgram("get-unpublishable-error " + outputPath, true);
                Assert.IsTrue(standardOut.Contains(issueString) == false);

                result = testReplaceNsp(outputPath, string.Empty);
                Assert.IsTrue(result.Item1.Contains(issueString) == false);
                Assert.IsTrue(result.Item2 == string.Empty, result.Item2);
                Assert.IsTrue(File.Exists(replaceNsp));
            }
            {
                // ハッシュが不正なソフトリーガル情報
                // →全てのコマンドで Issue 10-812 のエラーが表示される ことを確認
                var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
                var result = testCreateNsp(outputPath, "legal_info_invalid_hash.zip", string.Format(" --keyconfig {0} --keygeneration 3 --ticket", keyConfigFile) + " --error-unpublishable --ignore-unpublishable-error");
                Assert.IsTrue(result.Item1.Contains(issueString) == true);
                Assert.IsTrue(File.Exists(outputPath));

                var standardOut = ExecuteProgram("get-unpublishable-error " + outputPath, true);
                Assert.IsTrue(standardOut.Contains(issueString) == true);

                result = testReplaceNsp(outputPath, "--error-unpublishable");
                Assert.IsTrue(result.Item1.Contains(issueString) == true);
                Assert.IsTrue(result.Item2.Contains(exceptionString) == true);
                Assert.IsFalse(File.Exists(replaceNsp));
            }
        }

        [TestMethod]
        public void TestExecutionErrorUnpublishableIconCheck()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var baseMetaFile = GetDefaultApplicationMetaFile(metaDir);
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var codeDir = outputDir + "\\program";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var outputPath = outputDir + "\\" + Path.GetFileName(baseMetaFile) + ".nsp";

            var normalRawIconPath = testSourceDir + "\\Icon\\describe_all.bmp";
            var okNxIconPath = testSourceDir + "\\Icon\\describe_all.jpg";
            var ngNxIconPath = testSourceDir + "\\Icon\\Invalid\\nx_progressive.jpg";

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(codeDir);

            Action<bool, string> testExecuteFunction10815 = delegate (bool expectSuccsess, string metaFile)
            {
                string testNpdm = codeDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile);

                var issueString = "[Error] (Issue 10-815)";

                // --error-unpublishableを指定して実行 (他のエラーが発生してもnspを作成させるまで進めるために--ignore-unpublishable-errorオプションも指定する)
                string standardError;
                string standardOut;
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --error-unpublishable --ignore-unpublishable-error",
                    outputPath,
                    metaFile,
                    descFile,
                    codeDir));

                if (expectSuccsess == false)
                {
                    // nmeta内のチェック時と、nsp内のiconのチェック時で2回警告が検出される
                    int index = 0;
                    Assert.IsTrue((index = standardOut.IndexOf(issueString, index)) >= 0);
                    Assert.IsTrue((index = standardOut.IndexOf(issueString, index)) >= 0);
                    Assert.IsTrue((index = standardOut.IndexOf(issueString, index + 1)) >= 0);
                    Assert.IsTrue((index = standardOut.IndexOf(issueString, index)) >= 0);
                }
                else
                {
                    // エラーが発生していないことを確認
                    Assert.IsFalse(standardOut.Contains(issueString));
                }
            };

            Action<bool, string, string, string, string, string> createNmeta = delegate (bool useEnvironment, string filePath, string rawIcon1, string rawIcon2, string nxIcon1, string nxIcon2)
            {
                var nmetaNxIconTag1 = (nxIcon1 != null) ? string.Format("<NxIconPath>{0}</NxIconPath>", nxIcon1) : string.Empty;
                var nmetaNxIconTag2 = (nxIcon2 != null) ? string.Format("<NxIconPath>{0}</NxIconPath>", nxIcon2) : string.Empty;

                if (useEnvironment == true)
                {
                    Environment.SetEnvironmentVariable("TEST_ICON_DIR_PATH1", Path.GetDirectoryName(nxIcon1), EnvironmentVariableTarget.Process);
                    Environment.SetEnvironmentVariable("TEST_ICON_DIR_PATH2", Path.GetDirectoryName(nxIcon2), EnvironmentVariableTarget.Process);
                    nmetaNxIconTag1 = string.Format("<NxIconPath UseEnvironmentVariable = \"true\">%TEST_ICON_DIR_PATH1%\\{0}</NxIconPath>", Path.GetFileName(nxIcon1));
                    nmetaNxIconTag2 = string.Format("<NxIconPath UseEnvironmentVariable = \"true\">%TEST_ICON_DIR_PATH2%\\{0}</NxIconPath>", Path.GetFileName(nxIcon2));
                }

                string[] titleLines = new string[]
                {
                    "<Title>",
                    "<Language>AmericanEnglish</Language>",
                    "<Name>Name</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "</Title>",
                    "<Title>",
                    "<Language>Japanese</Language>",
                    "<Name>Name</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "</Title>",
                };

                string[] iconLines = new string[]
                {
                    "<Icon>",
                    "<Language>AmericanEnglish</Language>",
                    string.Format("<IconPath>{0}</IconPath>", rawIcon1),
                    nmetaNxIconTag1,
                    "</Icon>",
                    "<Icon>",
                    "<Language>Japanese</Language>",
                    string.Format("<IconPath>{0}</IconPath>", rawIcon2),
                    nmetaNxIconTag2,
                    "</Icon>",
                };

                List<string> metaLines = titleLines.ToList();
                if (nxIcon1 != null || nxIcon2 != null)
                {
                    metaLines.AddRange(iconLines.ToList());
                }
                MakeTestMetaFile(filePath, metaLines.ToArray());
            };

            // nmeta内にプログレッシブjpegのアイコンが指定されている
            {
                var metaFile = outputDir + "\\" + Path.GetFileNameWithoutExtension(ngNxIconPath) + ".nmeta";
                createNmeta(false, metaFile, normalRawIconPath, normalRawIconPath, okNxIconPath, ngNxIconPath);
                testExecuteFunction10815(false, metaFile);
            }

            // nmeta内にベースラインjpegのアイコンのみが指定されている
            {
                var metaFile = outputDir + "\\" + Path.GetFileNameWithoutExtension(okNxIconPath) + ".nmeta";
                createNmeta(false, metaFile, normalRawIconPath, normalRawIconPath, okNxIconPath, okNxIconPath);
                testExecuteFunction10815(true, metaFile);
            }

            // nmeta内にプログレッシブjpegのアイコンが指定されている(環境変数使用)
            {
                var metaFile = outputDir + "\\" + Path.GetFileNameWithoutExtension(ngNxIconPath) + "_env.nmeta";
                createNmeta(true, metaFile, normalRawIconPath, normalRawIconPath, ngNxIconPath, okNxIconPath);
                testExecuteFunction10815(false, metaFile);
            }

            // nmeta内にSDKに含まれるNintendoSDK_Application.bmpのアイコンが指定されている
            {
                var metaFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.aarch64.lp64.nmeta";

                string testNpdm = codeDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile);

                var issueString = "(Issue 10-047)";

                // --error-unpublishableを指定せずに実行
                File.Delete(outputPath);
                string standardError;
                string standardOut;
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                    outputPath,
                    metaFile,
                    descFile,
                    codeDir));

                // デフォルトアイコン指定のエラー・警告が発生せずにファイルの作成に成功することを確認
                Assert.IsFalse(standardOut.Contains(issueString));
                Assert.IsTrue(File.Exists(outputPath));
                File.Delete(outputPath);

                // --error-unpublishableを指定して実行
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --error-unpublishable",
                    outputPath,
                    metaFile,
                    descFile,
                    codeDir));

                // デフォルトアイコン指定のエラーが発生してファイルの作成に失敗することを確認
                issueString = "[Error] (Issue 10-047)";
                Assert.IsTrue(standardOut.Contains(issueString));
                Assert.IsFalse(File.Exists(outputPath));

                // デフォルトアイコン指定のエラーが発生しない場合は、Jpegデコーダの変更などでハッシュ値が変わっている可能性がある
                // ハッシュ値が変わっている場合はUnpublishableErrorCheck.csのCheckSbmChk9cで定義しているDefaultRawIconHashとDefaultNxIconHashの変更が必要
            }

            Action<bool, string, string> testExecuteFunction10614 = delegate (bool expectSuccsess, string errorLanguage, string metaFile)
            {
                string testNpdm = codeDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile);

                var issueString = "[Error] (Issue 10-614)";

                // --error-unpublishableを指定して実行 (他のエラーが発生してもnspを作成させるまで進めるために--ignore-unpublishable-errorオプションも指定する)
                string standardError;
                string standardOut;
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --error-unpublishable --ignore-unpublishable-error",
                    outputPath,
                    metaFile,
                    descFile,
                    codeDir));

                if (expectSuccsess == false)
                {
                    // エラーが発生することを確認
                    Assert.IsTrue(standardOut.Contains(issueString));
                    // エラーが発生している言語の確認
                    Assert.IsTrue(standardOut.Contains(": " + errorLanguage + Environment.NewLine));
                }
                else
                {
                    // エラーが発生していないことを確認
                    Assert.IsFalse(standardOut.Contains(issueString));
                }
            };

            // RawアイコンとNXアイコンの類似性のチェック(NxIconを指定しない＋RawIconに似たIconをNxIconに指定)
            {
                var baseIconPath = testSourceDir + "\\ErrorUnpublishable\\GradationIcon1.bmp";
                var editIconPath = outputDir + "\\" + Path.GetFileNameWithoutExtension(baseIconPath) + ".jpg";

                // rawIconに指定するIconをNxIcon用に256x256にリサイズして少し改変(対角線を描画)
                var bitmap = new System.Drawing.Bitmap(baseIconPath);
                var newBitmap = new System.Drawing.Bitmap(256, 256);
                System.Drawing.Graphics graphic = System.Drawing.Graphics.FromImage(newBitmap);
                graphic.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.HighQualityBilinear;
                graphic.DrawImage(bitmap, 0, 0, newBitmap.Width, newBitmap.Height);
                graphic.DrawLine(System.Drawing.Pens.Black, 0, 0, newBitmap.Width, newBitmap.Height);
                graphic.DrawLine(System.Drawing.Pens.White, 0, newBitmap.Height, newBitmap.Width, 0);
                newBitmap.Save(editIconPath, System.Drawing.Imaging.ImageFormat.Jpeg);

                var metaFile = outputDir + "\\" + Path.GetFileNameWithoutExtension(baseIconPath) + ".nmeta";
                createNmeta(false, metaFile, baseIconPath, baseIconPath, editIconPath, null);

                testExecuteFunction10614(true, null, metaFile);
            }

            // RawアイコンとNXアイコンの類似性のチェック(NxIconに別の画像とRawIconを大きく改変した画像を指定)
            {
                var baseIconPath = testSourceDir + "\\ErrorUnpublishable\\GradationIcon1.bmp";
                var editIconPath = outputDir + "\\" + Path.GetFileNameWithoutExtension(baseIconPath) + ".jpg";
                var DifferentIconPath = testSourceDir + "\\ErrorUnpublishable\\GradationIcon2.jpg";

                // rawIconに指定するIconをNxIcon用に256x256にリサイズして大きく改変(画像の真ん中に矩形を描画)
                var bitmap = new System.Drawing.Bitmap(baseIconPath);
                var newBitmap = new System.Drawing.Bitmap(256, 256);
                System.Drawing.Graphics graphic = System.Drawing.Graphics.FromImage(newBitmap);
                graphic.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.HighQualityBilinear;
                graphic.DrawImage(bitmap, 0, 0, newBitmap.Width, newBitmap.Height);
                graphic.FillRectangle(System.Drawing.Brushes.Black, newBitmap.Width / 3, newBitmap.Height / 3, (newBitmap.Width * 2) / 3, (newBitmap.Height * 2) / 3);
                newBitmap.Save(editIconPath, System.Drawing.Imaging.ImageFormat.Jpeg);

                var metaFile = outputDir + "\\" + Path.GetFileNameWithoutExtension(baseIconPath) + ".nmeta";
                createNmeta(false, metaFile, baseIconPath, baseIconPath, editIconPath, DifferentIconPath);

                testExecuteFunction10614(false, "AmericanEnglish, Japanese", metaFile);
            }

            // RawアイコンとNXアイコンの類似性のチェック(JapaneseにのみNxIconに別の画像を指定)
            {
                var baseIconPath = testSourceDir + "\\ErrorUnpublishable\\GradationIcon1.bmp";
                var DifferentIconPath = testSourceDir + "\\ErrorUnpublishable\\GradationIcon2.jpg";

                var metaFile = outputDir + "\\" + Path.GetFileNameWithoutExtension(baseIconPath) + ".nmeta";
                createNmeta(false, metaFile, baseIconPath, baseIconPath, null, DifferentIconPath);

                testExecuteFunction10614(false, "Japanese", metaFile);
            }
        }

        [TestMethod]
        public void TestExecutionErrorUnpublishableCoreCheck()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var baseMetaFile = GetDefaultApplicationMetaFile(metaDir);
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var codeDir = outputDir + "\\program";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var outputPath = outputDir + "\\" + Path.GetFileName(baseMetaFile) + ".nsp";

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(codeDir);

            var issueString = "[Error] (Issue 10-811)";

            // Core/FsAccessControlData/SaveDataOwnerIds/Id と Core/ApplicationId が同じnmetaファイルを作成
            var metaFile = outputDir + "\\" + "core_test.nmeta";
            var meta = string.Format(@"<?xml version=""1.0""?>
                                        <NintendoSdkMeta>
                                            <Core>
                                                <ApplicationId>0x0005000C10000001</ApplicationId>
                                                <FsAccessControlData>
                                                    <SaveDataOwnerIds>
                                                        <Accessibility>ReadWrite</Accessibility>
                                                        <Id>0x0005000C10000001</Id>
                                                    </SaveDataOwnerIds>
                                                </FsAccessControlData>
                                            </Core>
                                            <Application>
                                            </Application>
                                        </NintendoSdkMeta>", 1);
            File.WriteAllText(metaFile, meta);
            string testNpdm = codeDir + "\\main.npdm";
            MakeNpdm(testNpdm, metaFile, descFile);

            // --error-unpublishableを未指定で実行してエラーが発生することを確認
            string standardError;
            string standardOut;
            ExecuteProgram(
                out standardError,
                out standardOut,
                string.Format(
                "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                outputPath,
                metaFile,
                descFile,
                codeDir));
            Assert.IsTrue(standardOut.Contains(issueString) == true);
            Assert.IsTrue(standardError.IndexOf("[Error] Found one or more unpublishable error in nmeta.") >= 0, standardError);
            Assert.IsFalse(File.Exists(outputPath));

            // --ignore-unpublishable-errorオプションを指定した時にエラーは出力されるが例外が発生しないことを確認
            ExecuteProgram(
                out standardError,
                out standardOut,
                string.Format(
                "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --ignore-unpublishable-error",
                outputPath,
                metaFile,
                descFile,
                codeDir));
            // nmeta内のCoreチェック時と、nsp内のprograminfo.xmlのチェック時で2回警告が検出される
            int index = 0;
            Assert.IsTrue((index = standardOut.IndexOf(issueString, index)) >= 0);
            Assert.IsTrue((index = standardOut.IndexOf(issueString, index + 1)) >= 0);
            Assert.IsTrue(standardError == string.Empty);
            Assert.IsTrue(File.Exists(outputPath));
        }

        [TestMethod]
        public void TestExecutionErrorUnpublishableCardSpecCheck()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var baseMetaFile = GetDefaultApplicationMetaFile(metaDir);
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var codeDir = outputDir + "\\program";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var outputPath = outputDir + "\\" + Path.GetFileName(baseMetaFile) + ".nsp";

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(codeDir);

            string testNpdm = codeDir + "\\main.npdm";
            MakeNpdm(testNpdm, baseMetaFile, descFile);

            var issueString11014 = "[Warning] (Issue 11-014)";
            var issueString13801 = "[Error] (Issue 13-801)";

            // cardspecを持つnmetaファイルを作成
            var metaFile = outputDir + "\\" + "cardspec_test.nmeta";
            var meta = string.Format(@"<?xml version=""1.0""?>
                                        <NintendoSdkMeta>
                                            <Core>
                                                <ApplicationId>0x0005000C10000001</ApplicationId>
                                            </Core>
                                            <Application>
                                            </Application>
                                            <CardSpec>
                                                <LaunchFlags>{0}</LaunchFlags>
                                                <Size>{1}</Size>
                                            </CardSpec>
                                        </NintendoSdkMeta>", 1, 1);
            File.WriteAllText(metaFile, meta);

            // --error-unpublishableを指定して実行 (他のエラーが発生してもnspを作成させるまで進めるために--ignore-unpublishable-errorオプションも指定する)
            string standardError;
            string standardOut;
            ExecuteProgram(
                out standardError,
                out standardOut,
                string.Format(
                "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --error-unpublishable --ignore-unpublishable-error",
                outputPath,
                metaFile,
                descFile,
                codeDir));

            // nmeta内のCardSpecチェック時と、nsp内のcardspec.xmlのチェック時で2回警告が検出されることを確認する
            int index = 0;
            Assert.IsTrue((index = standardOut.IndexOf(issueString11014, index)) >= 0);
            Assert.IsTrue((index = standardOut.IndexOf(issueString13801, index + 1)) >= 0);
            Assert.IsTrue((index = standardOut.IndexOf(issueString11014, index + 1)) >= 0);
            Assert.IsTrue((index = standardOut.IndexOf(issueString13801, index + 1)) >= 0);
        }

        [TestMethod]
        public void TestExecutionErrorUnpublishableHtmlDocumentCheck()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var baseMetaFile = GetDefaultApplicationMetaFile(metaDir);
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var codeDir = outputDir + "\\program";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var outputPath = outputDir + "\\" + Path.GetFileName(baseMetaFile) + ".nsp";

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(codeDir);

            string testNpdm = codeDir + "\\main.npdm";
            MakeNpdm(testNpdm, baseMetaFile, descFile);

            Action<string, string> testExecuteFunction = delegate (string metaFile, string accessibleUrlsFile)
            {
                var issueString10609 = "[Error] (Issue 10-609)";
                var issueString10048 = "[Error] (Issue 10-048)";

                var okUrlList = @"^https://localhost
^https://www.nintendo.co.jp
---- ^https://localhost
";

                var ngUrlList = @"^https://localhost
https://www.nintendo.co.jp
^https://www.nintendo.co.jp
^http://www.nintendo.co.jp
---- ^http://www.nintendo.co.jp
----^http://www.nintendo.co.jp
";

                // 10-609のエラー説明では全てのURLが表示される
                var ngUrlString10609 = "URL: " + ngUrlList;
                // 10-048のエラー説明では行頭が "^https://" や "---- " ではないURLが表示される
                var ngUrlString10048 = "URL: " + @"https://www.nintendo.co.jp
^http://www.nintendo.co.jp
----^http://www.nintendo.co.jp
";

                // NGが発生するURLリストを指定
                {
                    File.WriteAllText(accessibleUrlsFile, ngUrlList);

                    // --error-unpublishableを指定して実行 (他のエラーが発生してもnspを作成させるまで進めるために--ignore-unpublishable-errorオプションも指定する)
                    string standardError;
                    string standardOut;
                    ExecuteProgram(
                        out standardError,
                        out standardOut,
                        string.Format(
                        "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --error-unpublishable --ignore-unpublishable-error",
                        outputPath,
                        metaFile,
                        descFile,
                        codeDir));
                    // nmeta内のAccessibleUrlsチェック時と、nsp内のhtmldocument.xmlのチェック時で2回警告が検出されることを確認する
                    int index = 0;
                    Assert.IsTrue((index = standardOut.IndexOf(issueString10609, index)) >= 0);
                    Assert.IsTrue((index = standardOut.IndexOf(issueString10609, index + 1)) >= 0);

                    index = 0;
                    Assert.IsTrue((index = standardOut.IndexOf(issueString10048, index)) >= 0);
                    Assert.IsTrue((index = standardOut.IndexOf(issueString10048, index + 1)) >= 0);

                    Action<string, string> chechUrlString = delegate (string issue, string expect)
                    {
                        string[] delimiter = { "[Error] " };
                        var errorDescription = Regex.Replace(standardOut.Split(delimiter, StringSplitOptions.None).Where(entry => entry.StartsWith(issue)).First(), Environment.NewLine + " +", Environment.NewLine);
                        Assert.IsTrue(errorDescription.Contains(expect));
                    };

                    chechUrlString("(Issue 10-609)", ngUrlString10609);
                    chechUrlString("(Issue 10-048)", ngUrlString10048);
                }

                // NGが発生しないURLリストを指定
                {
                    File.WriteAllText(accessibleUrlsFile, okUrlList);

                    // --error-unpublishableを指定して実行 (他のエラーが発生してもnspを作成させるまで進めるために--ignore-unpublishable-errorオプションも指定する)
                    string standardError;
                    string standardOut;
                    ExecuteProgram(
                        out standardError,
                        out standardOut,
                        string.Format(
                        "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --error-unpublishable --ignore-unpublishable-error",
                        outputPath,
                        metaFile,
                        descFile,
                        codeDir));
                    // エラーが発生していないことを確認
                    Assert.IsFalse(standardOut.Contains(issueString10609));
                    Assert.IsFalse(standardOut.Contains(issueString10048));
                }
            };

            // AccessibleUrlsFilePathが存在するnmetaファイルを作成
            {
                var metaFile = outputDir + "\\" + "AccessibleUrlsFilePath_test.nmeta";
                var meta = string.Format(@"<?xml version=""1.0""?>
                                        <NintendoSdkMeta>
                                            <Core>
                                                <ApplicationId>0x0005000C10000001</ApplicationId>
                                            </Core>
                                            <Application>
                                                <AccessibleUrlsFilePath>accessible-urls.txt</AccessibleUrlsFilePath>
                                            </Application>
                                        </NintendoSdkMeta>", 1);
                File.WriteAllText(metaFile, meta);

                var accessibleUrlsFile = outputDir + "\\" + "accessible-urls.txt";

                testExecuteFunction(metaFile, accessibleUrlsFile);
            }

            // AccessibleUrlsFilePathが存在するnmetaファイルを作成(環境変数使用)
            {
                var urlListDir = outputDir + "\\UrlListDir";
                Directory.CreateDirectory(urlListDir);

                var metaFile = outputDir + "\\" + "AccessibleUrlsFilePath_Env_test.nmeta";
                Environment.SetEnvironmentVariable("TEST_ACCESSIBLE_URLS_DIR_PATH", urlListDir, EnvironmentVariableTarget.Process);
                var meta = string.Format(@"<?xml version=""1.0""?>
                                            <NintendoSdkMeta>
                                                <Core>
                                                    <ApplicationId>0x0005000C10000001</ApplicationId>
                                                </Core>
                                                <Application>
                                                    <AccessibleUrlsFilePath UseEnvironmentVariable=""true"">%TEST_ACCESSIBLE_URLS_DIR_PATH%\accessible-urls.txt</AccessibleUrlsFilePath>
                                                </Application>
                                            </NintendoSdkMeta>", 1);
                File.WriteAllText(metaFile, meta);

                var accessibleUrlsFile = urlListDir + "\\" + "accessible-urls.txt";

                testExecuteFunction(metaFile, accessibleUrlsFile);
            }
        }

        [TestMethod]
        public void TestExecutionIgnoreErrorUnpublishableOption()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var testDir = Path.Combine(outputDir, "test");
            var codeDir = outputDir + "\\program";
            var dataFile = codeDir + "\\data.dat";
            var iconPath = GetIconContainApplicationMetaFile(metaDir);

            SafeDeleteDirectory(outputDir);
            SafeDeleteDirectory(testDir);
            SafeDeleteDirectory(codeDir);

            Directory.CreateDirectory(codeDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            Directory.CreateDirectory(testDir);

            // 下記のnmetaファイルを作成する
            // Iconエントリがない
            // Nameに简体字が設定されている
            // Isbn,DisplayVersion,ApplicationErrorCodeCategoryにUTF-8を設定
            string[] metaLines = new string[]
            {
                    "<Title>",
                    "<Name>Name\u7b80\u4f53\u5b57</Name>",
                    "<Publisher>Publisher</Publisher>",
                    "<Language>AmericanEnglish</Language>",
                    "</Title>",
                    "<Isbn>\uFF11</Isbn>",
                    "<DisplayVersion>\uFF11</DisplayVersion>",
                    "<ApplicationErrorCodeCategory>\uFF11</ApplicationErrorCodeCategory>",
            };
            string metaFile = testDir + "TitleEntryOnly_InvalidCharacter.nmeta";
            var issueString10608 = "[Error] (Issue 10-608)"; // Iconエントリがない
            var issueString10808 = "[Error] (Issue 10-808)"; // 日米欧フォント以外の文字が指定されている
            var issueString10809 = "[Error] (Issue 10-809)"; // Isbn,DisplayVersion,ApplicationErrorCodeCategoryにAscii以外を設定
            MakeTestMetaFile(metaFile, metaLines);

            {
                string standardError;
                string standardOut;
                var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

                string testNpdm = codeDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --ignore-unpublishable-error",
                    outputPath,
                    metaFile,
                    descFile,
                    codeDir));

                // エラーの警告が表示されているが例外は発生せず、nspの作成ができていることを確認
                Assert.IsTrue(standardError == string.Empty, standardError);
                Assert.IsTrue(standardOut != string.Empty);
                Assert.IsTrue(standardOut.IndexOf(issueString10608) >= 0);
                Assert.IsTrue(standardOut.IndexOf(issueString10808) >= 0);
                var index = 0;
                Assert.IsTrue((index = standardOut.IndexOf(issueString10809, index)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf(issueString10809, index + 1)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf(issueString10809, index + 1)) >= 0);
                Assert.IsTrue(File.Exists(outputPath));
            }

            {
                string outputPath;
                string standardError;
                string standardOut;
                outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nspd";
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2} --ignore-unpublishable-error",
                    outputPath,
                    metaFile,
                    codeDir));

                // エラーの警告が表示されているが例外は発生せず、nspの作成ができていることを確認
                Assert.IsTrue(standardError == string.Empty, standardError);
                Assert.IsTrue(standardOut != string.Empty);
                Assert.IsTrue(standardOut.IndexOf(issueString10608) >= 0);
                Assert.IsTrue(standardOut.IndexOf(issueString10808) >= 0);
                var index = 0;
                Assert.IsTrue((index = standardOut.IndexOf(issueString10809, index)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf(issueString10809, index + 1)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf(issueString10809, index + 1)) >= 0);
                Assert.IsTrue(Directory.Exists(outputPath));

                // createnspdで--ignore-unpublishable-errorのファイル指定によるチェック無効化を確認
                // "Issue 10-809"を無効化するリストを作成
                string[] ignoreList = new string[]
                {
                    "10-809",
                };
                string ignoreListFile = testDir + "\\" + "ignoreList.txt";
                File.WriteAllLines(ignoreListFile, ignoreList);

                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "createnspd -o {0} --type Application --meta {1} --program {2} --ignore-unpublishable-error {3}",
                    outputPath,
                    metaFile,
                    codeDir,
                    ignoreListFile));

                // "Issue 10-809"が表示されていないことを確認
                Assert.IsTrue(standardError == string.Empty, standardError);
                Assert.IsTrue(standardOut != string.Empty);
                Assert.IsTrue(standardOut.Contains(issueString10608));
                Assert.IsTrue(standardOut.Contains(issueString10808));
                Assert.IsFalse(standardOut.Contains(issueString10809));
                Assert.IsTrue(Directory.Exists(outputPath));
            }

            {
                var inputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
                var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + "_replaced.nsp";
                var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

                // 展開して置換用ファイルのパスを取得
                var testExtractDir = Path.Combine(outputDir, "extract");
                SafeDeleteDirectory(testExtractDir);
                var errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", inputPath, testExtractDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                var entryList = GetEntryFilePathList(testExtractDir);
                var testData = entryList.Find(entry => entry.EndsWith("/data.dat"));
                var testDataFilePath = Path.Combine(testExtractDir, testData);
                // データを加工
                byte[] data = File.ReadAllBytes(testDataFilePath);
                data[0] += 1;
                File.WriteAllBytes(testDataFilePath, data);

                string standardError;
                string standardOut;
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "replace {0} {1} {2} --desc {3} -o {4} --error-unpublishable --ignore-unpublishable-error",
                    inputPath,
                    testData,
                    testDataFilePath,
                    descFile,
                    outputDir));

                // エラーの警告が表示されているが例外は発生せず、nspの作成ができていることを確認
                Assert.IsTrue(standardError == string.Empty, standardError);
                Assert.IsTrue(standardOut != string.Empty);
                Assert.IsTrue(standardOut.IndexOf(issueString10608) >= 0);
                Assert.IsTrue(standardOut.IndexOf(issueString10808) >= 0);
                var index = 0;
                Assert.IsTrue((index = standardOut.IndexOf(issueString10809, index)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf(issueString10809, index + 1)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf(issueString10809, index + 1)) >= 0);
                Assert.IsTrue(File.Exists(outputPath));

                // replaceで--ignore-unpublishable-errorのファイル指定によるチェック無効化を確認
                // "Issue 10-809"を無効化するリストを作成
                string[] ignoreList = new string[]
                {
                    "10-809",
                };
                string ignoreListFile = testDir + "\\" + "ignoreList.txt";
                File.WriteAllLines(ignoreListFile, ignoreList);

                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "replace {0} {1} {2} --desc {3} -o {4} --error-unpublishable --ignore-unpublishable-error {5}",
                    inputPath,
                    testData,
                    testDataFilePath,
                    descFile,
                    outputDir,
                    ignoreListFile));

                // "Issue 10-809"が表示されていないことを確認
                Assert.IsTrue(standardError == string.Empty, standardError);
                Assert.IsTrue(standardOut != string.Empty);
                Assert.IsTrue(standardOut.Contains(issueString10608));
                Assert.IsTrue(standardOut.Contains(issueString10808));
                Assert.IsFalse(standardOut.Contains(issueString10809));
                Assert.IsTrue(File.Exists(outputPath));
            }

            {
                // 発生する全てのエラーを無視リストに追加
                string[] ignoreList = new string[]
                {
                    "10-014",
                    "10-607",
                    "10-608",
                    "10-808",
                    "10-809",
                    "11-002",
                    "11-009",
                    "12-003",
                };
                string ignoreListFile = testDir + "\\" + "ignoreList.txt";
                File.WriteAllLines(ignoreListFile, ignoreList);

                var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

                string testNpdm = codeDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
                string standardError;
                string standardOut;
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --ignore-unpublishable-error {4} --error-unpublishable",
                    outputPath,
                    metaFile,
                    descFile,
                    codeDir,
                    ignoreListFile));

                // UnpublishableErrorの警告が表示されずに、nspの作成ができていることを確認
                Assert.IsTrue(standardError == string.Empty, standardError);
                // 「このコンテンツにはエラーはありません。」が表示されるが、環境によって英語で表示されるので"(Issue "が無いことを確認する
                Assert.IsFalse(standardOut.Contains("(Issue "));
                Assert.IsTrue(File.Exists(outputPath));
            }

            {
                // 10-608以外の全てのエラーを無視リストに追加＋フォーマットが不正なパターンを追加
                string[] ignoreList = new string[]
                {
                        "10-014",
                        "10-607",
                        "10-808",
                        "10-809",
                        "11-002",
                        "11-009",
                        "12-003",
                        "10-6080",
                        "10-608-",
                        " 10-608",
                        "10_608",
                        "10 608",
                        "99_999",
                };
                string ignoreListFile = testDir + "\\" + "ignoreList.txt";
                File.WriteAllLines(ignoreListFile, ignoreList);

                var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

                string testNpdm = codeDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
                string standardError;
                string standardOut;
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --ignore-unpublishable-error {4} --error-unpublishable",
                    outputPath,
                    metaFile,
                    descFile,
                    codeDir,
                    ignoreListFile));

                // 10-608の警告のみ表示され、nspの作成ができていることを確認
                Assert.IsTrue(standardError == string.Empty, standardError);
                Assert.IsTrue(standardOut != string.Empty);
                // nmetaのチェックとnspのチェックで 10-608 の警告が2回表示されることを確認
                int index = 0;
                Assert.IsTrue((index = standardOut.IndexOf(issueString10608, index)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf(issueString10608, index + 1)) >= 0);
                // "Issue 10-608"以外に"Issue"が無いことを確認
                Assert.IsTrue(!standardOut.Replace(issueString10608, string.Empty).Contains("(Issue "));
                Assert.IsTrue(File.Exists(outputPath));
            }

            SafeDeleteDirectory(outputDir);
            SafeDeleteDirectory(testDir);
            SafeDeleteDirectory(codeDir);
        }

        [StructLayout(LayoutKind.Sequential)]
        private class TestContentMeta
        {
            public UInt64 Id { get; set; }
            public UInt32 Version { get; set; }
            public Byte Type { get; set; }

            public void SetFromMetaFile(string metaFilePath, string baseXmlPath, byte typeNumber, UInt32 defaultVersion = 0)
            {
                XmlDocument xml = new XmlDocument();
                xml.Load(metaFilePath);

                var idNode = xml.SelectSingleNode(baseXmlPath + "/Id");
                var versionNode = xml.SelectSingleNode(baseXmlPath + "/Version");
                var releaseVersionNode = xml.SelectSingleNode(baseXmlPath + "/ReleaseVersion");
                var privateVersionNode = xml.SelectSingleNode(baseXmlPath + "/PrivateVersion");

                SetFromMetaFileImpl(idNode, versionNode, releaseVersionNode, privateVersionNode, typeNumber, defaultVersion);
            }

            public void SetFromApplicationMetaFile(string metaFilePath, string baseXmlPath, byte typeNumber)
            {
                XmlDocument xml = new XmlDocument();
                xml.Load(metaFilePath);

                var idNode = xml.SelectSingleNode(baseXmlPath + "/Core/ApplicationId");
                var versionNode = xml.SelectSingleNode(baseXmlPath + "/Application/Version");
                var releaseVersionNode = xml.SelectSingleNode(baseXmlPath + "/Application/ReleaseVersion");
                var privateVersionNode = xml.SelectSingleNode(baseXmlPath + "/Application/PrivateVersion");

                SetFromMetaFileImpl(idNode, versionNode, releaseVersionNode, privateVersionNode, typeNumber);
            }

            private void SetFromMetaFileImpl(XmlNode idNode, XmlNode versionNode, XmlNode releaseVersionNode, XmlNode privateVersionNode, byte typeNumber, UInt32 defaultVersion = 0)
            {
                Assert.IsTrue(idNode != null);
                Id = Convert.ToUInt64(idNode.InnerText, 16);

                Version = defaultVersion;
                if (versionNode != null)
                {
                    Version = Convert.ToUInt32(versionNode.InnerText);
                }
                if (releaseVersionNode != null)
                {
                    Assert.IsTrue(versionNode == null);
                    Version = (UInt32)(Convert.ToUInt16(releaseVersionNode.InnerText) << 16);
                }
                if (privateVersionNode != null)
                {
                    Assert.IsTrue(versionNode == null);
                    Version |= Convert.ToUInt32(privateVersionNode.InnerText);
                }
                Type = typeNumber;
            }
        }

        private void CheckContentMeta(string nspPath, string contentMetaType, TestContentMeta meta, string keyFilePath)
        {
            var contentMeta = new Regex(@".*/" + contentMetaType + @"_.*\.cnmt");
            var contentMetaXml = new Regex(@".*\.cnmt\.xml");

            var extractedDirectoryPath = nspPath + ".extract";
            SafeDeleteDirectory(extractedDirectoryPath);
            Directory.CreateDirectory(extractedDirectoryPath);
            var errorMsg = ExecuteProgram(string.Format("extract -o {0} {1} --keyconfig {2}", extractedDirectoryPath, nspPath, keyFilePath));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            bool isChecked = false;
            foreach (var entry in Directory.EnumerateFiles(extractedDirectoryPath, "*", SearchOption.AllDirectories))
            {
                var entryPath = entry.Substring(extractedDirectoryPath.Length + 1).Replace("\\", "/");

                if (contentMeta.IsMatch(entryPath))
                {
                    TestContentMeta nspMeta = null;
                    using (FileStream fs = File.Open(entry, FileMode.Open))
                    {
                        int size = 13;
                        byte[] buffer = new byte[size];
                        fs.Read(buffer, 0, size);
                        GCHandle gch = GCHandle.Alloc(buffer, GCHandleType.Pinned);
                        nspMeta = (TestContentMeta)Marshal.PtrToStructure(gch.AddrOfPinnedObject(), typeof(TestContentMeta));
                        gch.Free();
                    }

                    Assert.AreEqual(meta.Id, nspMeta.Id);
                    Assert.AreEqual(meta.Version, nspMeta.Version);
                    Assert.AreEqual(meta.Type, nspMeta.Type);
                    isChecked = true;
                }

                if (contentMetaXml.IsMatch(entryPath))
                {
                    TestContentMeta nspContentMetaXml = new TestContentMeta();
                    nspContentMetaXml.SetFromMetaFile(entry, "/ContentMeta", 0);
                    Assert.AreEqual(meta.Id, nspContentMetaXml.Id);
                    Assert.AreEqual(meta.Version, nspContentMetaXml.Version);
                }
            }

            Assert.IsTrue(isChecked);
        }

        private void TestNacpFile(string testExtractDir)
        {
            var nacpXml = Directory.EnumerateFiles(testExtractDir, "*.nacp.xml", SearchOption.TopDirectoryOnly).Single();
            using (var fs = File.OpenRead(nacpXml))
            {
                StreamReader sr = new StreamReader(fs);
                var contents = sr.ReadToEnd();
                Assert.IsTrue(contents.IndexOf("<IconPath>") < 0);
                Assert.IsTrue(contents.IndexOf("<NxIconPath>") < 0);
                Assert.IsTrue(contents.IndexOf("<HtmlDocumentPath>") < 0);
                Assert.IsTrue(contents.IndexOf("<LegalInformationFilePath>") < 0);
                Assert.IsTrue(contents.IndexOf("<AccessibleUrlsFilePath>") < 0);
            }
        }

        private void CheckEntriesInDirectory(string targetDirA, string targetDirB)
        {
            var srcEntryFileListA = Directory.EnumerateFiles(targetDirA, "*", SearchOption.AllDirectories).ToList();
            var srcEntryFileListB = Directory.EnumerateFiles(targetDirB, "*", SearchOption.AllDirectories).ToList();
            Assert.IsTrue(srcEntryFileListA.Count() == srcEntryFileListB.Count());

            var hashDictA = new Dictionary<string, string>();
            var hashDictB = new Dictionary<string, string>();

            for (int i = 0; i < srcEntryFileListA.Count; i++)
            {
                var crc = Checksum(srcEntryFileListA[i]);
                var text = BitConverter.ToString(crc);
                hashDictA[text] = srcEntryFileListA[i];
            }

            for (int i = 0; i < srcEntryFileListB.Count; i++)
            {
                var crc = Checksum(srcEntryFileListB[i]);
                var text = BitConverter.ToString(crc);
                Assert.IsTrue(hashDictA.ContainsKey(text));
            }
        }

        private void TestCreateNspMeta(string nspPath, string testExtractDir, string metaFilePath, string contentMetaType, string iconPath)
        {
            // createnspmeta
            var nspdPath = nspPath.Replace(".nsp", ".nspd");
            string errorMsg;
            if (iconPath != null)
            {
                errorMsg = ExecuteProgram(string.Format("createnspmeta -o {0} --type {1} --meta {2} --icon AmericanEnglish {3} Japanese {3}", nspdPath, contentMetaType, metaFilePath, iconPath));
            }
            else
            {
                errorMsg = ExecuteProgram(string.Format("createnspmeta -o {0} --type {1} --meta {2}", nspdPath, contentMetaType, metaFilePath));
            }
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            // extract
            errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", nspPath, testExtractDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            var controlNacp = new Regex(@".*/control\.nacp");
            var contentMeta = new Regex(@".*/" + contentMetaType + @"_.*\.cnmt");
            var iconAmerican = new Regex(@".*/icon_AmericanEnglish\.dat");
            var iconJapanese = new Regex(@".*/icon_Japanese\.dat"); var iconAmericanNx = new Regex(@".*/icon_nx_AmericanEnglish\.jpg");

            bool nspControlNacpFound = false;
            bool nspContentMetaFound = false;
            bool nspIconAmericanFound = false;
            bool nspIconJapaneseFound = false;

            bool nspdControlNacpFound = false;
            bool nspdContentMetaFound = false;
            bool nspdIconAmericanFound = false;
            bool nspdIconJapaneseFound = false;

            foreach (var entry in Directory.EnumerateFiles(testExtractDir, "*", SearchOption.AllDirectories))
            {
                var entryPath = entry.Substring(testExtractDir.Length + 1).Replace("\\", "/");

                // 管理データを比較
                if (controlNacp.IsMatch(entryPath))
                {
                    CheckFileDiff(Path.GetFullPath(entry), nspdPath + "/control0.ncd/data/control.nacp", 0);
                    nspControlNacpFound = true;
                }

                // コンテンツメタの ID, Version, Type までを比較
                if (contentMeta.IsMatch(entryPath))
                {
                    CheckFileDiff(Path.GetFullPath(entry), nspdPath + "/meta0.ncd/data/" + Path.GetFileName(entry), 13);
                    nspContentMetaFound = true;
                }



                // アイコンの比較
                if (iconAmerican.IsMatch(entryPath) || iconJapanese.IsMatch(entryPath))
                {
                    CheckFileDiff(Path.GetFullPath(entry), nspdPath + "/control0.ncd/data/" + Path.GetFileName(entry), 0);
                }

                if (iconAmerican.IsMatch(entryPath))
                {
                    nspIconAmericanFound = true;
                }

                if (iconJapanese.IsMatch(entryPath))
                {
                    nspIconJapaneseFound = true;
                }
            }

            foreach (var entry in Directory.EnumerateFiles(nspdPath, "*", SearchOption.AllDirectories))
            {
                var entryPath = entry.Substring(nspdPath.Length + 1).Replace("\\", "/");

                if (controlNacp.IsMatch(entryPath))
                {
                    nspdControlNacpFound = true;
                }

                if (contentMeta.IsMatch(entryPath))
                {
                    nspdContentMetaFound = true;
                }

                if (iconAmerican.IsMatch(entryPath))
                {
                    nspdIconAmericanFound = true;
                }

                if (iconJapanese.IsMatch(entryPath))
                {
                    nspdIconJapaneseFound = true;
                }
            }

            if (contentMetaType == "Application" || contentMetaType == "Patch")
            {
                Assert.IsTrue(nspControlNacpFound);
                Assert.IsTrue(nspdControlNacpFound);
            }
            else
            {
                Assert.IsFalse(nspControlNacpFound);
                Assert.IsFalse(nspdControlNacpFound);
            }

            Assert.IsTrue(nspContentMetaFound);
            Assert.IsTrue(nspdContentMetaFound);

            if (iconPath != null)
            {
                Assert.IsTrue(nspIconAmericanFound);
                Assert.IsTrue(nspIconJapaneseFound);
                Assert.IsTrue(nspdIconAmericanFound);
                Assert.IsTrue(nspdIconJapaneseFound);
            }
            else
            {
                Assert.IsFalse(nspIconAmericanFound);
                Assert.IsFalse(nspIconJapaneseFound);
                Assert.IsFalse(nspdIconAmericanFound);
                Assert.IsFalse(nspdIconJapaneseFound);
            }

            SafeDeleteDirectory(testExtractDir);
        }

        [Flags]
        private enum TestLegalInformationFilesFlag
        {
            None                = 0,
            NormalFileInZip     = 1 << 0,
            LegalInfoXmlInZip   = 1 << 1,
            LegalInfoXmlInRoot  = 1 << 2,
            All                 = 1 << 3 - 1,
        }

        private void TestLegalInformation(string nspPath, string testExtractDir)
        {
            SafeDeleteDirectory(testExtractDir);

            var errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", nspPath, testExtractDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            TestNacpFile(testExtractDir);

            var flags = TestLegalInformationFilesFlag.None;

            foreach (var ncaDir in Directory.EnumerateDirectories(testExtractDir))
            {
                if (File.Exists(Path.Combine(new string[] { ncaDir, "fs0", TestZipFileName })))
                {
                    Assert.IsTrue((flags & TestLegalInformationFilesFlag.NormalFileInZip) == 0);
                    flags |= TestLegalInformationFilesFlag.NormalFileInZip;
                }
                if (File.Exists(Path.Combine(new string[] { ncaDir, "fs0", LegalInfoXmlFileName})))
                {
                    Assert.IsTrue((flags & TestLegalInformationFilesFlag.LegalInfoXmlInZip) == 0);
                    flags |= TestLegalInformationFilesFlag.LegalInfoXmlInZip;
                }
            }
            foreach (var rootFile in Directory.EnumerateFiles(testExtractDir))
            {
                if (rootFile.EndsWith(LegalInfoXmlFileName))
                {
                    Assert.IsTrue((flags & TestLegalInformationFilesFlag.LegalInfoXmlInRoot) == 0);
                    flags |= TestLegalInformationFilesFlag.LegalInfoXmlInRoot;
                }
            }
            Assert.IsTrue((flags & TestLegalInformationFilesFlag.All) == TestLegalInformationFilesFlag.All);

            SafeDeleteDirectory(testExtractDir);
        }

        private void TestHtmlDocumentDir(string nspPath, string testExtractDir, string baseDir, string htmlDocXmlPath, bool isHtmlDocument)
        {
            SafeDeleteDirectory(testExtractDir);

            var errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", nspPath, testExtractDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            TestNacpFile(testExtractDir);

            bool found = false;
            foreach (var ncaDir in Directory.EnumerateDirectories(testExtractDir))
            {
                string baseDirName = isHtmlDocument ? "html-document" : "accessible-urls";
                string dirPath = Path.Combine(new string[] { ncaDir, "fs0", baseDirName }).Replace("\\","/");
                if (Directory.Exists(dirPath))
                {
                    var iNca = Directory.EnumerateFiles(dirPath, "*", SearchOption.AllDirectories);
                    var iDir = Directory.EnumerateFiles(baseDir, "*", SearchOption.AllDirectories);
                    Assert.IsTrue(iNca.Count() == iDir.Count());
                    foreach (var filePath in iNca)
                    {
                        var slashPath = filePath.Replace("\\", "/");
                        var trimedPath = slashPath.Replace(dirPath, "").TrimStart('/');
                        Assert.IsTrue(File.Exists(Path.Combine(baseDir, trimedPath)));
                    }
                    found = true;
                    break;
                }
            }
            Assert.IsTrue(found);

            found = false;
            var iFile = Directory.EnumerateFiles(testExtractDir, "*", SearchOption.AllDirectories);
            foreach (var filePath in iFile)
            {
                if (filePath.EndsWith("htmldocument.xml"))
                {
                    found = true;
                    CompareTextFiles(htmlDocXmlPath, filePath);
                }
            }
            Assert.IsTrue(found);
            SafeDeleteDirectory(testExtractDir);
        }

        public class CreateNspdTestUtil
        {
            public static readonly string DefaultProgramNcdName = "program0.ncd";
            public static readonly string DefaultCodeDirName = "code";
            public static readonly string DefaultDataDirName = "data";
            public static readonly string DefaultLogoDirName = "logo";
            public static readonly string DefaultControlNcdName = "control0.ncd";
            public static readonly string DefaultControlNacpFile = "control.nacp";
            public static readonly string DefaultHtmlDocumentNcdName = "htmlDocument0.ncd";
            public static readonly string DefaultHtmlDocumentDirName = "html-document";
            public static readonly string DefaultAccessibleUrlsDirName = "accessible-urls";
            public static readonly string DefaultLegalInformationNcdName = "legalInformation0.ncd";
            public static readonly string DefaultMetaNcdName = "meta0.ncd";
            public static readonly string DefaultContentMetaFile = "data/Application_0100000000003802.cnmt";

            public string MetaFile { get; private set;  }
            public string CodeDir { get; private set; }
            public string DataDir { get; private set; }
            public string LogoDir { get; private set; }
            public string ControlDir { get; private set; }
            public string HtmlDocumentDir { get; private set; }
            public string AccessibleUrlsDir { get; private set; }
            public string LegalInformationDir { get; private set; }
            public string LegalInformationZipFile { get; private set; }

            public CreateNspdTestUtil(string resourceDir)
            {
                MetaFile = Path.Combine(resourceDir, "test.nmeta");
                CodeDir = Path.Combine(resourceDir, "code");
                DataDir = Path.Combine(resourceDir, "data");
                LogoDir = Path.Combine(resourceDir, "logo");
                ControlDir = Path.Combine(resourceDir, "control");
                HtmlDocumentDir = Path.Combine(resourceDir, "html");
                AccessibleUrlsDir = Path.Combine(resourceDir, "urls");
                LegalInformationDir = Path.Combine(resourceDir, "legal");
                LegalInformationZipFile = Path.Combine(resourceDir, "legal.zip");

                MakeMetaFile(MetaFile, null);

                PrepareDirectory(CodeDir);
                PrepareDirectory(DataDir);
                PrepareDirectory(LogoDir);
                PrepareDirectory(ControlDir);
                PrepareDirectory(HtmlDocumentDir);
                PrepareDirectory(AccessibleUrlsDir);
                PrepareDirectory(LegalInformationDir);

                ZipFile.CreateFromDirectory(LegalInformationDir, LegalInformationZipFile);
            }

            public static string GetTestFileName(string testDir)
            {
                return testDir.Split(Path.DirectorySeparatorChar).Last() + ".txt";
            }

            public static void DeleteNspd(string nspdPath)
            {
                Utils.DeleteIncludingJunctionDirectoryIfExisted(nspdPath);
            }

            private void MakeMetaFile(string path, string applicationControlPropertyXml)
            {
                using (var fs = File.Create(path))
                {
                    var sw = new StreamWriter(fs);
                    sw.WriteLine("<?xml version=\"1.0\"?>");
                    sw.WriteLine("<NintendoSdkMeta>");
                    sw.WriteLine("<Core>");
                    sw.WriteLine("<ApplicationId>0x0100000000003802</ApplicationId>");
                    sw.WriteLine("</Core>");
                    sw.WriteLine("<Application>");
                    if (applicationControlPropertyXml != null && applicationControlPropertyXml.Length > 0)
                    {
                        sw.WriteLine(applicationControlPropertyXml);
                    }
                    sw.WriteLine("</Application>");
                    sw.WriteLine("</NintendoSdkMeta>");

                    sw.Flush();
                }
            }

            private void PrepareDirectory(string path)
            {
                Utils.DeleteDirectoryIfExisted(path);
                Directory.CreateDirectory(path);
                var fileName = GetTestFileName(path);
                using (var fs = File.Create(Path.Combine(path, fileName)))
                {
                }
            }
        }

        [TestMethod]
        public void TestExecutionCreateApplicationNspdDefault()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testDir = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());
            var resourceDir = Path.Combine(testDir, "resource");
            var nspdPath = Path.Combine(".", "output.nspd");
            var nspdRootPath = Path.Combine(".", "output.nspd_root");

            Directory.CreateDirectory(resourceDir);
            var testParams = new CreateNspdTestUtil(resourceDir);

            var error = ExecuteProgram(string.Format(
                "createnspd --type Application --meta {0} --program {1}",
                testParams.MetaFile,
                testParams.CodeDir));
            Assert.IsTrue(error == null || error.Length == 0);

            Assert.IsTrue(Directory.Exists(nspdPath));
            Assert.IsTrue(File.Exists(nspdRootPath));

            var programNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultProgramNcdName);
            Assert.IsTrue(Directory.Exists(programNcd));

            var codeDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultCodeDirName);
            Assert.IsTrue(Directory.Exists(codeDir));
            Assert.IsTrue((File.GetAttributes(codeDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

            var dataDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultDataDirName);
            Assert.IsFalse(Directory.Exists(dataDir));

            var logoDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultLogoDirName);
            Assert.IsTrue(Directory.Exists(logoDir));

            var controlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultControlNcdName);
            Assert.IsTrue(Directory.Exists(controlNcd));

            var controlDir = Path.Combine(controlNcd, CreateNspdTestUtil.DefaultDataDirName);
            Assert.IsTrue(Directory.Exists(controlDir));

            var controlNacp = Path.Combine(controlDir, CreateNspdTestUtil.DefaultControlNacpFile);
            Assert.IsTrue(File.Exists(controlNacp));

            var htmlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultHtmlDocumentNcdName);
            Assert.IsFalse(Directory.Exists(htmlNcd));

            var legalInformationNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultLegalInformationNcdName);
            Assert.IsFalse(Directory.Exists(legalInformationNcd));

            var metaNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultMetaNcdName);
            Assert.IsTrue(Directory.Exists(metaNcd));

            var contentMetaFile = Path.Combine(metaNcd, CreateNspdTestUtil.DefaultContentMetaFile);
            Assert.IsTrue(File.Exists(contentMetaFile));

            CreateNspdTestUtil.DeleteNspd(nspdPath);
            SafeDeleteDirectory(testDir);
            File.Delete(nspdRootPath);
        }

        [TestMethod]
        public void TestExecutionCreateApplicationNspdWithOption()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var nxIconPath = iconDir + "\\describe_all.jpg";

            var testDir = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());
            var resourceDir = Path.Combine(testDir, "resource");
            var nspdPath = Path.Combine(testDir, "output.nspd");
            var nspdRootPath = Path.Combine(testDir, "output.nspd_root");

            foreach (var file in Directory.EnumerateFiles(metaDir))
            {
                Directory.CreateDirectory(resourceDir);
                var testParams = new CreateNspdTestUtil(resourceDir);

                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";

                var format = "createnspd -o {0} --type Application --meta {1} --code {2} --data {3} --logo {4} --control {5} --html-document {6} --accessible-urls {7} --legal-information-dir {8}";
                if (string.Compare(Path.GetFileName(file), "no_application.nmeta") == 0)
                {
                    // Applicationエントリがないnmetaを使用した場合、アイコンを設定するとエラーとなるので--Iconオプションを付けない
                    iconPath = null;
                }
                else
                {
                    format += " --icon AmericanEnglish {9} Japanese {9} --nx-icon AmericanEnglish {10} Japanese {10} --nx-icon-max-size {11}";
                }
                var error = ExecuteProgram(string.Format(
                    format,
                    nspdPath,
                    file,
                    testParams.CodeDir,
                    testParams.DataDir,
                    testParams.LogoDir,
                    testParams.ControlDir,
                    testParams.HtmlDocumentDir,
                    testParams.AccessibleUrlsDir,
                    testParams.LegalInformationDir,
                    iconPath,
                    nxIconPath,
                    1000 * 1024
                    ));

                Assert.IsTrue(error == null || error.Length == 0);

                Assert.IsTrue(Directory.Exists(nspdPath));
                Assert.IsTrue(File.Exists(nspdRootPath));

                var programNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultProgramNcdName);
                Assert.IsTrue(Directory.Exists(programNcd));

                var codeDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultCodeDirName);
                Assert.IsTrue(Directory.Exists(codeDir));
                Assert.IsTrue((File.GetAttributes(codeDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var codeFile = Path.Combine(codeDir, CreateNspdTestUtil.GetTestFileName(testParams.CodeDir));
                Assert.IsTrue(File.Exists(codeFile));

                var dataDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(dataDir));
                Assert.IsTrue((File.GetAttributes(dataDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var dataFile = Path.Combine(dataDir, CreateNspdTestUtil.GetTestFileName(testParams.DataDir));
                Assert.IsTrue(File.Exists(dataFile));

                var logoDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultLogoDirName);
                Assert.IsTrue(Directory.Exists(logoDir));
                Assert.IsTrue((File.GetAttributes(logoDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var logoFile = Path.Combine(logoDir, CreateNspdTestUtil.GetTestFileName(testParams.LogoDir));
                Assert.IsTrue(File.Exists(logoFile));

                var controlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultControlNcdName);
                Assert.IsTrue(Directory.Exists(controlNcd));

                var controlDir = Path.Combine(controlNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(controlDir));
                Assert.IsTrue((File.GetAttributes(controlDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var controlNacp = Path.Combine(controlDir, CreateNspdTestUtil.DefaultControlNacpFile);
                Assert.IsTrue(File.Exists(controlNacp));

                var americanIcon = Path.Combine(controlDir, "icon_AmericanEnglish.dat");
                var japaneseIcon = Path.Combine(controlDir, "icon_Japanese.dat");
                if (iconPath != null)
                {
                    Assert.IsTrue(File.Exists(americanIcon));
                    Assert.IsTrue(File.Exists(japaneseIcon));
                }
                else
                {
                    Assert.IsFalse(File.Exists(americanIcon));
                    Assert.IsFalse(File.Exists(japaneseIcon));
                }
                var controlFile = Path.Combine(controlDir, CreateNspdTestUtil.GetTestFileName(testParams.ControlDir));
                Assert.IsTrue(File.Exists(controlFile));

                var htmlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultHtmlDocumentNcdName);
                Assert.IsTrue(Directory.Exists(htmlNcd));

                var htmlDataDir = Path.Combine(htmlNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(htmlDataDir));

                var htmlDir = Path.Combine(htmlDataDir, CreateNspdTestUtil.DefaultHtmlDocumentDirName);
                Assert.IsTrue(Directory.Exists(htmlDir));
                Assert.IsTrue((File.GetAttributes(htmlDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var htmlFile = Path.Combine(htmlDir, CreateNspdTestUtil.GetTestFileName(testParams.HtmlDocumentDir));
                Assert.IsTrue(File.Exists(htmlFile));

                var accessibleUrlsDir = Path.Combine(htmlDataDir, CreateNspdTestUtil.DefaultAccessibleUrlsDirName);
                Assert.IsTrue(Directory.Exists(accessibleUrlsDir));
                Assert.IsTrue((File.GetAttributes(accessibleUrlsDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var accessibleUrlsFile = Path.Combine(accessibleUrlsDir, CreateNspdTestUtil.GetTestFileName(testParams.AccessibleUrlsDir));
                Assert.IsTrue(File.Exists(accessibleUrlsFile));

                var legalInformationNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultLegalInformationNcdName);
                Assert.IsTrue(Directory.Exists(legalInformationNcd));
                Assert.IsTrue((File.GetAttributes(legalInformationNcd) & ~FileAttributes.NotContentIndexed) == FileAttributes.Directory);

                var legalInformationDataDir = Path.Combine(legalInformationNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(legalInformationDataDir));
                Assert.IsTrue((File.GetAttributes(legalInformationDataDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var legalInforamtionFile = Path.Combine(legalInformationDataDir, CreateNspdTestUtil.GetTestFileName(testParams.LegalInformationDir));
                Assert.IsTrue(File.Exists(legalInforamtionFile));

                var metaNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultMetaNcdName);
                Assert.IsTrue(Directory.Exists(metaNcd));

                CreateNspdTestUtil.DeleteNspd(nspdPath);
                CreateNspdTestUtil.DeleteNspd(resourceDir);
            }

            SafeDeleteDirectory(testDir);
        }

        [TestMethod]
        public void TestExecutionCreateApplicationNspdWithSameDirectory()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var nxIconPath = iconDir + "\\describe_all.jpg";

            var testDir = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());
            var resourceDir = Path.Combine(testDir, "resource");
            var nspdPath = Path.Combine(testDir, "output.nspd");
            var nspdRootPath = Path.Combine(testDir, "output.nspd_root");

            Directory.CreateDirectory(resourceDir);
            var testParams = new CreateNspdTestUtil(resourceDir);

            var file = GetIconContainApplicationMetaFile(metaDir);
            {
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";
                var error = ExecuteProgram(string.Format(
                    "createnspd --type Application -o {0} --meta {1} --code {2} --accessible-urls {3}",
                    nspdPath,
                    file,
                    testParams.CodeDir,
                    testParams.AccessibleUrlsDir
                    ));
                Assert.IsTrue(error == null || error.Length == 0);
                Assert.IsTrue(Directory.Exists(nspdPath));

                var programNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultProgramNcdName);
                Assert.IsTrue(Directory.Exists(programNcd));

                var codeDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultCodeDirName);
                Assert.IsTrue(Directory.Exists(codeDir));
                Assert.IsTrue((File.GetAttributes(codeDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var codeFile = Path.Combine(codeDir, CreateNspdTestUtil.GetTestFileName(testParams.CodeDir));
                Assert.IsTrue(File.Exists(codeFile));

                var htmlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultHtmlDocumentNcdName);
                Assert.IsTrue(Directory.Exists(htmlNcd));

                var htmlDataDir = Path.Combine(htmlNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(htmlDataDir));

                var accessibleUrlsDir = Path.Combine(htmlDataDir, CreateNspdTestUtil.DefaultAccessibleUrlsDirName);
                Assert.IsTrue(Directory.Exists(accessibleUrlsDir));
                Assert.IsTrue((File.GetAttributes(accessibleUrlsDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var accessibleUrlsFile = Path.Combine(accessibleUrlsDir, CreateNspdTestUtil.GetTestFileName(testParams.AccessibleUrlsDir));
                Assert.IsTrue(File.Exists(accessibleUrlsFile));

                error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --code {2} --data {3} --logo {4} --control {5} --html-document {6} --legal-information-dir {7} --icon AmericanEnglish {8} Japanese {8} --nx-icon AmericanEnglish {9} Japanese {9} --nx-icon-max-size {10}",
                    nspdPath,
                    file,
                    codeDir,
                    testParams.DataDir,
                    testParams.LogoDir,
                    testParams.ControlDir,
                    testParams.HtmlDocumentDir,
                    testParams.LegalInformationDir,
                    iconPath,
                    nxIconPath,
                    1000 * 1024
                    ));

                Assert.IsTrue(error == null || error.Length == 0);

                Assert.IsTrue(Directory.Exists(nspdPath));
                Assert.IsTrue(File.Exists(nspdRootPath));

                Assert.IsTrue(Directory.Exists(programNcd));

                Assert.IsTrue(Directory.Exists(codeDir));
                Assert.IsTrue((File.GetAttributes(codeDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                Assert.IsTrue(File.Exists(codeFile));

                var dataDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(dataDir));
                Assert.IsTrue((File.GetAttributes(dataDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var dataFile = Path.Combine(dataDir, CreateNspdTestUtil.GetTestFileName(testParams.DataDir));
                Assert.IsTrue(File.Exists(dataFile));

                var logoDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultLogoDirName);
                Assert.IsTrue(Directory.Exists(logoDir));
                Assert.IsTrue((File.GetAttributes(logoDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var logoFile = Path.Combine(logoDir, CreateNspdTestUtil.GetTestFileName(testParams.LogoDir));
                Assert.IsTrue(File.Exists(logoFile));

                var controlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultControlNcdName);
                Assert.IsTrue(Directory.Exists(controlNcd));

                var controlDir = Path.Combine(controlNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(controlDir));
                Assert.IsTrue((File.GetAttributes(logoDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var controlNacp = Path.Combine(controlDir, CreateNspdTestUtil.DefaultControlNacpFile);
                Assert.IsTrue(File.Exists(controlNacp));

                var americanIcon = Path.Combine(controlDir, "icon_AmericanEnglish.dat");
                Assert.IsTrue(File.Exists(americanIcon));

                var japaneseIcon = Path.Combine(controlDir, "icon_Japanese.dat");
                Assert.IsTrue(File.Exists(japaneseIcon));

                var controlFile = Path.Combine(controlDir, CreateNspdTestUtil.GetTestFileName(testParams.ControlDir));
                Assert.IsTrue(File.Exists(controlFile));

                htmlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultHtmlDocumentNcdName);
                Assert.IsTrue(Directory.Exists(htmlNcd));

                htmlDataDir = Path.Combine(htmlNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(htmlDataDir));
                Assert.IsTrue((File.GetAttributes(htmlDataDir) & ~FileAttributes.NotContentIndexed) == FileAttributes.Directory);

                var htmlDir = Path.Combine(htmlDataDir, CreateNspdTestUtil.DefaultHtmlDocumentDirName);
                Assert.IsTrue(Directory.Exists(htmlDir));
                Assert.IsTrue((File.GetAttributes(htmlDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var htmlFile = Path.Combine(htmlDir, CreateNspdTestUtil.GetTestFileName(testParams.HtmlDocumentDir));
                Assert.IsTrue(File.Exists(htmlFile));

                accessibleUrlsDir = Path.Combine(htmlDataDir, CreateNspdTestUtil.DefaultAccessibleUrlsDirName);
                Assert.IsTrue(Directory.Exists(accessibleUrlsDir));
                Assert.IsTrue((File.GetAttributes(accessibleUrlsDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                accessibleUrlsFile = Path.Combine(accessibleUrlsDir, CreateNspdTestUtil.GetTestFileName(testParams.AccessibleUrlsDir));
                Assert.IsTrue(File.Exists(accessibleUrlsFile));

                var legalInformationNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultLegalInformationNcdName);
                Assert.IsTrue(Directory.Exists(legalInformationNcd));
                Assert.IsTrue((File.GetAttributes(legalInformationNcd) & ~FileAttributes.NotContentIndexed) == FileAttributes.Directory);

                var legalInformationDataDir = Path.Combine(legalInformationNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(legalInformationDataDir));
                Assert.IsTrue((File.GetAttributes(legalInformationDataDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var legalInforamtionFile = Path.Combine(legalInformationDataDir, CreateNspdTestUtil.GetTestFileName(testParams.LegalInformationDir));
                Assert.IsTrue(File.Exists(legalInforamtionFile));

                var metaNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultMetaNcdName);
                Assert.IsTrue(Directory.Exists(metaNcd));

                CreateNspdTestUtil.DeleteNspd(nspdPath);
            }

            SafeDeleteDirectory(testDir);
        }

        [TestMethod]
        public void TestExecutionCreateApplicationNspdWithLegalInformationZip()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testDir = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());
            var resourceDir = Path.Combine(testDir, "resource");
            var nspdPath = Path.Combine(".", "output.nspd");
            var nspdRootPath = Path.Combine(".", "output.nspd_root");

            Directory.CreateDirectory(resourceDir);
            var testParams = new CreateNspdTestUtil(resourceDir);

            var error = ExecuteProgram(string.Format(
                "createnspd --type Application --meta {0} --code {1} --legal-information {2}",
                testParams.MetaFile,
                testParams.CodeDir,
                testParams.LegalInformationZipFile));
            Assert.IsTrue(error == null || error.Length == 0);

            Assert.IsTrue(Directory.Exists(nspdPath));
            Assert.IsTrue(File.Exists(nspdRootPath));

            var programNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultProgramNcdName);
            Assert.IsTrue(Directory.Exists(programNcd));

            var codeDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultCodeDirName);
            Assert.IsTrue(Directory.Exists(codeDir));
            Assert.IsTrue((File.GetAttributes(codeDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

            var codeFile = Path.Combine(codeDir, CreateNspdTestUtil.GetTestFileName(testParams.CodeDir));
            Assert.IsTrue(File.Exists(codeFile));

            var dataDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultDataDirName);
            Assert.IsFalse(Directory.Exists(dataDir));

            var logoDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultLogoDirName);
            Assert.IsTrue(Directory.Exists(logoDir));

            var controlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultControlNcdName);
            Assert.IsTrue(Directory.Exists(controlNcd));

            var controlDir = Path.Combine(controlNcd, CreateNspdTestUtil.DefaultDataDirName);
            Assert.IsTrue(Directory.Exists(controlDir));

            var controlNacp = Path.Combine(controlDir, CreateNspdTestUtil.DefaultControlNacpFile);
            Assert.IsTrue(File.Exists(controlNacp));

            var htmlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultHtmlDocumentNcdName);
            Assert.IsFalse(Directory.Exists(htmlNcd));

            var legalInformationNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultLegalInformationNcdName);
            Assert.IsTrue(Directory.Exists(legalInformationNcd));
            Assert.IsTrue((File.GetAttributes(legalInformationNcd) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory));

            var legalInformationDataDir = Path.Combine(legalInformationNcd, CreateNspdTestUtil.DefaultDataDirName);
            Assert.IsTrue(Directory.Exists(legalInformationDataDir));
            Assert.IsTrue((File.GetAttributes(legalInformationDataDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory));

            var legalInforamtionFile = Path.Combine(legalInformationDataDir, CreateNspdTestUtil.GetTestFileName(testParams.LegalInformationDir));
            Assert.IsTrue(File.Exists(legalInforamtionFile));

            var metaNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultMetaNcdName);
            Assert.IsTrue(Directory.Exists(metaNcd));

            var contentMetaFile = Path.Combine(metaNcd, CreateNspdTestUtil.DefaultContentMetaFile);
            Assert.IsTrue(File.Exists(contentMetaFile));

            CreateNspdTestUtil.DeleteNspd(nspdPath);
            SafeDeleteDirectory(testDir);
            File.Delete(nspdRootPath);
        }

        [TestMethod]
        public void TestExecutionCreateApplicationNspdWithNotExistPath()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testDir = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());
            var resourceDir = Path.Combine(testDir, "resource");
            var nspdPath = Path.Combine(".", "output.nspd");
            var nspdRootPath = Path.Combine(".", "output.nspd_root");

            Directory.CreateDirectory(resourceDir);
            var testParams = new CreateNspdTestUtil(resourceDir);

            var error = ExecuteProgram(string.Format(
                "createnspd --type Application --meta {0} --code {1} --html-document {2}",
                testParams.MetaFile,
                testParams.CodeDir,
                "NotFoundPath"));
            Assert.IsTrue(error.IndexOf("Directory not found: ./NotFoundPath") >= 0);

            CreateNspdTestUtil.DeleteNspd(nspdPath);
            SafeDeleteDirectory(testDir);
            if (File.Exists(nspdRootPath))
            {
                File.Delete(nspdRootPath);
            }
        }

        [TestMethod]
        public void TestExecutionCreateApplicationNspdWithPathIncludingBlank()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testDir = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());
            var resourceDir = Path.Combine(testDir, "resource");
            var nspdPath = Path.Combine(testDir, "output blank.nspd");
            var nspdRootPath = Path.Combine(testDir, "output blank.nspd_root");

            Directory.CreateDirectory(resourceDir);
            var testParams = new CreateNspdTestUtil(resourceDir);

            Assert.IsTrue(!Directory.Exists(nspdPath));
            Assert.IsTrue(!File.Exists(nspdRootPath));

            var error = ExecuteProgram(string.Format(
                "createnspd --type Application --meta {0} --code {1} -o \"{2}\"",
                testParams.MetaFile,
                testParams.CodeDir,
                nspdPath));
            Assert.IsTrue(string.IsNullOrEmpty(error), error);
            Assert.IsTrue(Directory.Exists(nspdPath));
            Assert.IsTrue(File.Exists(nspdRootPath));

            SafeDeleteDirectory(testDir);
        }

        [TestMethod]
        public void TestExecutionCreateApplicationNspdWithFilter()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var fdfSpecifiedMetaFile = env.SourceDir + "\\FilterTest\\filter_specified.nmeta";
            var fdfFileName = "application.fdf";
            var fdfNotSpecifiedMetaFile = env.SourceDir + "\\FilterTest\\filter_not_specified.nmeta";
            var dataDir = Path.Combine(env.OutputDir, "data");

            SafeDeleteDirectory(dataDir);
            Directory.CreateDirectory(dataDir);
            const int DataCount = 3;
            for(int i = 0; i < DataCount; ++i)
            {
                var filePath = Path.Combine(dataDir, string.Format("Data_{0}.bin", i));
                using (FileStream stream = File.Create(filePath))
                {
                    int fileSize = 1024;
                    byte[] data = new byte[fileSize];
                    for (int j = 0; j < fileSize; j++)
                    {
                        data[j] = (byte)j;
                   }
                    stream.Write(data, 0, fileSize);
                }
            }

            var fdfSrcPath = Path.Combine(env.SourceDir, "FilterTest\\" + fdfFileName);
            var fdfDestPath = Path.Combine(Directory.GetCurrentDirectory(), fdfFileName);
            File.Copy(fdfSrcPath, fdfDestPath, true);

            Action<string, string, string> createNspd = (nspd, metaFile, addCmd) =>
            {
                var cmd = string.Format("createnspd -o {0} --type Application --meta {1} --program {2} --data {3}", nspd, metaFile, env.TestCodeDir, dataDir) + addCmd;
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);

                var nspdDataDir = Path.Combine(nspd, "program0.ncd\\data");
                string testFile;
                testFile = Path.Combine(nspdDataDir, "Data_0.bin");
                Assert.IsTrue(File.Exists(testFile));
                testFile = Path.Combine(nspdDataDir, "Data_1.bin");
                Assert.IsFalse(File.Exists(testFile));
                testFile = Path.Combine(nspdDataDir, "Data_2.bin");
                Assert.IsTrue(File.Exists(testFile));
            };

            Action<string, string, string> createNspdExpectError = (nspd, metaFile, addCmd) =>
            {
                var cmd = string.Format("createnspd -o {0} --type Application --meta {1} --program {2} --data {3}", nspd, metaFile, env.TestCodeDir, dataDir) + addCmd;
                var errorMsg = ExecuteProgram(cmd);
                Assert.IsTrue(errorMsg.Contains("It is impossible to specify fdf file from both nmeta and command line option."));
            };

            // FDF のパスを を nmeta の中で指定
            var testNspd = env.OutputDir + "\\01.nspd";
            createNspd(testNspd, fdfSpecifiedMetaFile, "");

            // FDF のパスを を nmeta の中で指定 + "--forcibly-copy-with-filtering" オプション
            testNspd = env.OutputDir + "\\02.nspd";
            createNspd(testNspd, fdfSpecifiedMetaFile, " --forcibly-copy-with-filtering");

            // FDF のパスをコマンドラインで指定
            testNspd = env.OutputDir + "\\03.nspd";
            createNspd(testNspd, fdfNotSpecifiedMetaFile, string.Format(" --filter {0}", fdfDestPath));

            // FDF のパスをコマンドラインで指定 + "--forcibly-copy-with-filtering" オプション
            testNspd = env.OutputDir + "\\04.nspd";
            createNspd(testNspd, fdfNotSpecifiedMetaFile, string.Format(" --filter {0} --forcibly-copy-with-filtering", fdfDestPath));

            // FDF のパスを を nmeta の中で指定 + "--forcibly-copy-data" オプション
            testNspd = env.OutputDir + "\\05.nspd";
            createNspd(testNspd, fdfSpecifiedMetaFile, " --forcibly-copy-data");

            // FDF のパスを を nmeta の中で指定 + "--forcibly-copy-with-filtering" オプション + "--forcibly-copy-data" オプション
            testNspd = env.OutputDir + "\\06.nspd";
            createNspd(testNspd, fdfSpecifiedMetaFile, " --forcibly-copy-with-filtering --forcibly-copy-data");

            // FDF のパスをコマンドラインで指定 + "--forcibly-copy-data" オプション
            testNspd = env.OutputDir + "\\07.nspd";
            createNspd(testNspd, fdfNotSpecifiedMetaFile, string.Format(" --filter {0} --forcibly-copy-data", fdfDestPath));

            // FDF のパスをコマンドラインで指定 + "--forcibly-copy-with-filtering" オプション + "--forcibly-copy-data" オプション
            testNspd = env.OutputDir + "\\08.nspd";
            createNspd(testNspd, fdfNotSpecifiedMetaFile, string.Format(" --filter {0} --forcibly-copy-with-filtering --forcibly-copy-data", fdfDestPath));

            // FDF を nmeta, コマンドライン両方で指定 ==> エラー
            testNspd = env.OutputDir + "\\09.nspd";
            createNspdExpectError(testNspd, fdfSpecifiedMetaFile,
                string.Format(" --filter {0}", fdfDestPath));
        }

        [TestMethod]
        public void TestExecutionCreateApplicationNspdCopyOptions()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);
            var metaFile = env.SourceDir + "\\ApplicationMeta\\describe_all_with_icon.nmeta";

            var resourceDir = Path.Combine(env.OutputDir, "resource");
            Directory.CreateDirectory(resourceDir);
            var testParams = new CreateNspdTestUtil(resourceDir);

            Action<string, bool> checkDirectory = (path, actual) =>
            {
                Assert.IsTrue(Directory.Exists(path));
                Assert.IsTrue((File.GetAttributes(path) & ~FileAttributes.NotContentIndexed) == (actual ? FileAttributes.Directory : (FileAttributes.Directory | FileAttributes.ReparsePoint)));
            };

            Action<string, bool, bool, bool, bool, bool, bool> createNspd = (nspd, copyCode, copyData, copyControl, copyHtmlDoc, copyAccessibleUrls, copyLegalInfo) =>
            {
                var cmd = string.Format("createnspd -o {0} --type Application --meta {1} --code {2} --data {3} --control {4} --html-document {5} --accessible-urls {6} --legal-information-dir {7}",
                    nspd, metaFile,
                    testParams.CodeDir,
                    testParams.DataDir,
                    testParams.ControlDir,
                    testParams.HtmlDocumentDir,
                    testParams.AccessibleUrlsDir,
                    testParams.LegalInformationDir);
                if (copyCode)
                {
                    cmd += " --forcibly-copy-code";
                }
                if (copyData)
                {
                    cmd += " --forcibly-copy-data";
                }
                if (copyControl)
                {
                    cmd += " --forcibly-copy-control";
                }
                if (copyHtmlDoc)
                {
                    cmd += " --forcibly-copy-html-document";
                }
                if (copyAccessibleUrls)
                {
                    cmd += " --forcibly-copy-accessible-urls";
                }
                if (copyLegalInfo)
                {
                    cmd += " --forcibly-copy-legal-information";
                }

                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);

                var programNcd = Path.Combine(nspd, CreateNspdTestUtil.DefaultProgramNcdName);
                checkDirectory(programNcd, true);
                var codeDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultCodeDirName);
                checkDirectory(codeDir, copyCode);
                var codeFile = Path.Combine(codeDir, CreateNspdTestUtil.GetTestFileName(testParams.CodeDir));
                Assert.IsTrue(File.Exists(codeFile));
                var dataDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultDataDirName);
                checkDirectory(dataDir, copyData);
                var dataFile = Path.Combine(dataDir, CreateNspdTestUtil.GetTestFileName(testParams.DataDir));
                Assert.IsTrue(File.Exists(dataFile));

                var controlNcd = Path.Combine(nspd, CreateNspdTestUtil.DefaultControlNcdName);
                Assert.IsTrue(Directory.Exists(controlNcd));
                checkDirectory(controlNcd, true);
                var controlDir = Path.Combine(controlNcd, CreateNspdTestUtil.DefaultDataDirName);
                checkDirectory(controlDir, copyControl);
                var controlFile = Path.Combine(controlDir, CreateNspdTestUtil.GetTestFileName(testParams.ControlDir));
                Assert.IsTrue(File.Exists(controlFile));

                var htmlNcd = Path.Combine(nspd, CreateNspdTestUtil.DefaultHtmlDocumentNcdName);
                checkDirectory(htmlNcd, true);
                var htmlDataDir = Path.Combine(htmlNcd, CreateNspdTestUtil.DefaultDataDirName);
                checkDirectory(htmlDataDir, true);
                var htmlDir = Path.Combine(htmlDataDir, CreateNspdTestUtil.DefaultHtmlDocumentDirName);
                checkDirectory(htmlDir, copyHtmlDoc);
                var htmlFile = Path.Combine(htmlDir, CreateNspdTestUtil.GetTestFileName(testParams.HtmlDocumentDir));
                Assert.IsTrue(File.Exists(htmlFile));
                var accessibleUrlsDir = Path.Combine(htmlDataDir, CreateNspdTestUtil.DefaultAccessibleUrlsDirName);
                checkDirectory(accessibleUrlsDir, copyAccessibleUrls);
                var accessibleUrlsFile = Path.Combine(accessibleUrlsDir, CreateNspdTestUtil.GetTestFileName(testParams.AccessibleUrlsDir));
                Assert.IsTrue(File.Exists(accessibleUrlsFile));

                var legalInformationNcd = Path.Combine(nspd, CreateNspdTestUtil.DefaultLegalInformationNcdName);
                checkDirectory(legalInformationNcd, true);
                var legalInformationDataDir = Path.Combine(legalInformationNcd, CreateNspdTestUtil.DefaultDataDirName);
                checkDirectory(legalInformationDataDir, copyLegalInfo);
                var legalInforamtionFile = Path.Combine(legalInformationDataDir, CreateNspdTestUtil.GetTestFileName(testParams.LegalInformationDir));
                Assert.IsTrue(File.Exists(legalInforamtionFile));
            };

            // 6つの --forcibly-copy-* オプションの有り無し、それぞれのパターンで nspd を作成
            for (uint i = 0; i < (1 << 6) ; i++)
            {
                var testNspd = env.OutputDir + "\\ForciblyCopy" + i.ToString("x2") + ".nspd";
                createNspd(testNspd, (i & 0x20) != 0, (i & 0x10) != 0, (i & 0x08) != 0, (i & 0x04) != 0, (i & 0x02) != 0,  (i & 0x01) != 0);
            }
        }

        [TestMethod]
        public void TestCleanupNspdContentMeta()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";
            var iconDir = testSourceDir + "\\Icon";
            var nxIconPath = iconDir + "\\describe_all.jpg";

            var testDir = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());
            var resourceDir = Path.Combine(testDir, "resource");
            var nspdPath = Path.Combine(testDir, "output.nspd");
            var nspdRootPath = Path.Combine(testDir, "output.nspd_root");

            Directory.CreateDirectory(resourceDir);
            var testParams = new CreateNspdTestUtil(resourceDir);

            var file = GetDefaultApplicationMetaFile(metaDir);
            {
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";

                var error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --code {2} --data {3} --logo {4} --control {5} --html-document {6} --accessible-urls {7} --legal-information-dir {8} --icon AmericanEnglish {9} Japanese {9} --nx-icon AmericanEnglish {10} Japanese {10} --nx-icon-max-size {11}",
                    nspdPath,
                    file,
                    testParams.CodeDir,
                    testParams.DataDir,
                    testParams.LogoDir,
                    testParams.ControlDir,
                    testParams.HtmlDocumentDir,
                    testParams.AccessibleUrlsDir,
                    testParams.LegalInformationDir,
                    iconPath,
                    nxIconPath,
                    1000 * 1024
                    ));

                Assert.IsTrue(error == null || error.Length == 0);

                Assert.IsTrue(Directory.Exists(nspdPath));
                Assert.IsTrue(File.Exists(nspdRootPath));

                var programNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultProgramNcdName);
                Assert.IsTrue(Directory.Exists(programNcd));

                var codeDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultCodeDirName);
                Assert.IsTrue(Directory.Exists(codeDir));
                Assert.IsTrue((File.GetAttributes(codeDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var codeFile = Path.Combine(codeDir, CreateNspdTestUtil.GetTestFileName(testParams.CodeDir));
                Assert.IsTrue(File.Exists(codeFile));

                var dataDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(dataDir));
                Assert.IsTrue((File.GetAttributes(dataDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var dataFile = Path.Combine(dataDir, CreateNspdTestUtil.GetTestFileName(testParams.DataDir));
                Assert.IsTrue(File.Exists(dataFile));

                var logoDir = Path.Combine(programNcd, CreateNspdTestUtil.DefaultLogoDirName);
                Assert.IsTrue(Directory.Exists(logoDir));
                Assert.IsTrue((File.GetAttributes(logoDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var logoFile = Path.Combine(logoDir, CreateNspdTestUtil.GetTestFileName(testParams.LogoDir));
                Assert.IsTrue(File.Exists(logoFile));

                var controlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultControlNcdName);
                Assert.IsTrue(Directory.Exists(controlNcd));

                var controlDir = Path.Combine(controlNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(controlDir));
                Assert.IsTrue((File.GetAttributes(controlDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var controlNacp = Path.Combine(controlDir, CreateNspdTestUtil.DefaultControlNacpFile);
                Assert.IsTrue(File.Exists(controlNacp));

                var americanIcon = Path.Combine(controlDir, "icon_AmericanEnglish.dat");
                Assert.IsTrue(File.Exists(americanIcon));

                var japaneseIcon = Path.Combine(controlDir, "icon_Japanese.dat");
                Assert.IsTrue(File.Exists(japaneseIcon));

                var controlFile = Path.Combine(controlDir, CreateNspdTestUtil.GetTestFileName(testParams.ControlDir));
                Assert.IsTrue(File.Exists(controlFile));

                var htmlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultHtmlDocumentNcdName);
                Assert.IsTrue(Directory.Exists(htmlNcd));

                var htmlDataDir = Path.Combine(htmlNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(htmlDataDir));

                var htmlDir = Path.Combine(htmlDataDir, CreateNspdTestUtil.DefaultHtmlDocumentDirName);
                Assert.IsTrue(Directory.Exists(htmlDir));
                Assert.IsTrue((File.GetAttributes(htmlDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var htmlFile = Path.Combine(htmlDir, CreateNspdTestUtil.GetTestFileName(testParams.HtmlDocumentDir));
                Assert.IsTrue(File.Exists(htmlFile));

                var accessibleUrlsDir = Path.Combine(htmlDataDir, CreateNspdTestUtil.DefaultAccessibleUrlsDirName);
                Assert.IsTrue(Directory.Exists(accessibleUrlsDir));
                Assert.IsTrue((File.GetAttributes(accessibleUrlsDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var accessibleUrlsFile = Path.Combine(accessibleUrlsDir, CreateNspdTestUtil.GetTestFileName(testParams.AccessibleUrlsDir));
                Assert.IsTrue(File.Exists(accessibleUrlsFile));

                var legalInformationNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultLegalInformationNcdName);
                Assert.IsTrue(Directory.Exists(legalInformationNcd));
                Assert.IsTrue((File.GetAttributes(legalInformationNcd) & ~FileAttributes.NotContentIndexed) == FileAttributes.Directory);

                var legalInformationDataDir = Path.Combine(legalInformationNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(legalInformationDataDir));
                Assert.IsTrue((File.GetAttributes(legalInformationDataDir) & ~FileAttributes.NotContentIndexed) == (FileAttributes.Directory | FileAttributes.ReparsePoint));

                var legalInforamtionFile = Path.Combine(legalInformationDataDir, CreateNspdTestUtil.GetTestFileName(testParams.LegalInformationDir));
                Assert.IsTrue(File.Exists(legalInforamtionFile));

                var metaNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultMetaNcdName);
                Assert.IsTrue(Directory.Exists(metaNcd));

                var findResult = Directory.EnumerateFiles(Path.Combine(metaNcd, "data"), "*.cnmt");
                Assert.IsTrue(findResult.Count() == 1);

                var cnmtFile = findResult.First();

                // TODO: 出力先のパスを残すオプションが追加されたら置き換える
                error = ExecuteProgram(string.Format(
                    "createnspd -o {0} --type Application --meta {1} --code {2}",
                    nspdPath,
                    testParams.MetaFile,
                    Path.Combine(nspdPath, CreateNspdTestUtil.DefaultProgramNcdName, CreateNspdTestUtil.DefaultCodeDirName)
                    ));
                Assert.IsTrue(error == null || error.Length == 0);

                metaNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultMetaNcdName);
                Assert.IsTrue(Directory.Exists(metaNcd));

                findResult = Directory.EnumerateFiles(Path.Combine(metaNcd, "data"), "*.cnmt");
                Assert.AreEqual(1, findResult.Count());
                Assert.AreNotEqual(cnmtFile, findResult.First());

                CreateNspdTestUtil.DeleteNspd(nspdPath);
            }

            SafeDeleteDirectory(testDir);
        }

        [TestMethod]
        public void TestCleanupCreateNspMeta()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var metaDir = testSourceDir + "\\ApplicationMeta";

            var testDir = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());
            var resourceDir = Path.Combine(testDir, "resource");
            var nspdPath = Path.Combine(testDir, "output.nspd");

            Directory.CreateDirectory(resourceDir);
            var testParams = new CreateNspdTestUtil(resourceDir);

            var file = GetIconContainApplicationMetaFile(metaDir);
            {
                var error = ExecuteProgram(string.Format(
                    "createnspmeta -o {0} --type Application --meta {1}",
                    nspdPath,
                    file
                    ));
                Assert.IsTrue(error == null || error.Length == 0);

                Assert.IsTrue(Directory.Exists(nspdPath));

                var controlNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultControlNcdName);
                Assert.IsTrue(Directory.Exists(controlNcd));

                var controlDir = Path.Combine(controlNcd, CreateNspdTestUtil.DefaultDataDirName);
                Assert.IsTrue(Directory.Exists(controlDir));
                Assert.IsTrue((File.GetAttributes(controlDir) & ~FileAttributes.NotContentIndexed) == FileAttributes.Directory);

                var controlNacp = Path.Combine(controlDir, CreateNspdTestUtil.DefaultControlNacpFile);
                Assert.IsTrue(File.Exists(controlNacp));

                var metaNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultMetaNcdName);
                Assert.IsTrue(Directory.Exists(metaNcd));

                var findResult = Directory.EnumerateFiles(Path.Combine(metaNcd, "data"), "*.cnmt");
                Assert.IsTrue(findResult.Count() == 1);

                var cnmtFile = findResult.First();

                error = ExecuteProgram(string.Format(
                    "createnspmeta -o {0} --type Application --meta {1}",
                    nspdPath,
                    testParams.MetaFile
                    ));
                Assert.IsTrue(error == null || error.Length == 0);

                metaNcd = Path.Combine(nspdPath, CreateNspdTestUtil.DefaultMetaNcdName);
                Assert.IsTrue(Directory.Exists(metaNcd));

                findResult = Directory.EnumerateFiles(Path.Combine(metaNcd, "data"), "*.cnmt");
                Assert.AreEqual(1, findResult.Count());
                Assert.AreNotEqual(cnmtFile, findResult.First());

                CreateNspdTestUtil.DeleteNspd(nspdPath);
            }

            SafeDeleteDirectory(testDir);
        }

        [TestMethod]
        public void TestExecutionCreateNspWithTicket()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";

            var appMetaDir = testSourceDir + "\\ApplicationMeta";
            var targetAppMeta = new List<string>()
            {
                appMetaDir + "\\describe_all_with_icon.nmeta",
                appMetaDir + "\\no_application.nmeta"
            };
            var aocMetaDir = testSourceDir + "\\AddOnContentMeta";
            var iconDir = testSourceDir + "\\Icon";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var testExtractDir = Path.Combine(outputDir, "extract");
            var testExtractEntryDir = Path.Combine(outputDir, "extractEntry");
            var testReplaceDir = Path.Combine(outputDir, "replace");

            var aocDataDir = outputDir  + "\\data";
            var aocDataFile = aocDataDir + "\\data.dat";

            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var config = new AuthoringConfiguration();

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(testExtractDir);
            Directory.CreateDirectory(dataDir);
            Directory.CreateDirectory(aocDataDir);
            CreateTestData(dataFile, 1024);
            CreateTestData(aocDataFile, 1024);

            // Application
            foreach (var file in targetAppMeta)
            {
                ForceGC();

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, file, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(file) + ".bmp";
                var legalInformationZip = MakeLegalInfoZipfile(outputDir);

                var format = "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --html-document {3} --legal-information {5} --save-adf --ticket";
                if (string.Compare(Path.GetFileName(file), "no_application.nmeta") == 0)
                {
                    // Applicationエントリがないnmetaを使用した場合、アイコンを設定するとエラーとなるので--Iconオプションを付けない
                    iconPath = null;
                }
                else
                {
                    format += " --icon AmericanEnglish {4} Japanese {4}";
                }
                var error = ExecuteProgram(string.Format(
                    format,
                    outputPath,
                    file,
                    descFile,
                    dataDir,
                    iconPath,
                    legalInformationZip));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);

                string[] entryRuleStringArray = new string[] {
                    @".*\.nca/fs0/data\.dat",
                    @".*\.nca/fs0/main\.npdm",
                    @".*\.nca/fs2/NintendoLogo\.png",
                    @".*\.nca/fs2/StartupMovie\.gif",
                    @".*\.nca/fs0/control\.nacp",
                    @".*\.nca/fs0/icon_AmericanEnglish\.dat",
                    @".*\.nca/fs0/icon_Japanese\.dat",
                    @".*\.nca/fs0/html-document/data\.dat",
                    @".*\.nca/fs0/html-document/main\.npdm",
                    @".*\.nca/fs0/legal-information\.txt",
                    @".*\.nca/fs0/legalinfo\.xml",
                    @".*\.cnmt\.nca/fs0/Application_0005000c10000001.cnmt",
                    @".*\.cnmt.xml",
                    @".*\.nacp.xml",
                    @".*\.programinfo.xml",
                    @".*\.legalinfo.xml",
                    @".*\.htmldocument.xml",
                    @"cardspec.xml",
                    @"[0-9a-fA-F]*\.tik",
                    @"[0-9a-fA-F]*\.cert",
                    @".*\.nx\.AmericanEnglish\.jpg",
                    @".*\.nx\.Japanese\.jpg",
                    @".*\.raw\.AmericanEnglish\.jpg",
                    @".*\.raw\.Japanese\.jpg",
                    @"authoringtoolinfo.xml",
                };

                if (iconPath == null)
                {
                    // icon未設定の場合は期待値のリストからiconファイルを削除する
                    entryRuleStringArray = entryRuleStringArray.Where(entry => !(entry.Contains("AmericanEnglish") || entry.Contains("Japanese"))).ToArray();
                }

                CheckTicketInNsp(outputPath, config.GetKeyConfiguration());
                TestExtractWithNsp(outputPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);

                // replace
                {
                    Tuple<Regex, bool>[] ReplaceEntryInNspTestList = new Tuple<Regex, bool>[] {
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/data.dat"), true),
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/control.nacp"), false),
                    };
                    var testCombination = new List<int[]>() { new int[] { 0, 1 } };
                    TestReplaceWithNspOrNca(outputPath, testReplaceDir, testExtractDir, testExtractEntryDir, ReplaceEntryInNspTestList, testCombination);

                    // ticket 有り版は nca 差し替えテストを省略
                }

                SafeDeleteDirectory(testExtractDir);
                SafeDeleteDirectory(testExtractEntryDir);

                errorMsg = ExecuteProgram(string.Format("prodencryption --no-check --keyconfig {0} -o {1} {2}", keyConfigFile, outputDir, outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            }

            // AddOnContent
            foreach (var file in Directory.EnumerateFiles(aocMetaDir))
            {
                ForceGC();

                // meta ファイル内のパスが meta ファイルからの間接参照なので、outputDir に meta ファイルをコピーする
                var metaFile = Path.Combine(outputDir, Path.GetFileName(file));
                File.Copy(file, metaFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(file) + ".nsp";

                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type AddOnContent --meta {1} --data {2} --ticket --ignore-unpublishable-error",
                    outputPath,
                    metaFile,
                    aocDataDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                string errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);

                string[] entryRuleStringArray;
                if (file.Contains("multiple") || file.Contains("env"))
                {
                    entryRuleStringArray = new string[] {
                        @".*\.nca/fs0/data\.dat",
                        @".*\.cnmt.xml",
                        @".*\.nca/fs0/data\.dat",
                        @".*\.cnmt.xml",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c[13]000[01]003.cnmt",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c[13]000[01]004.cnmt",
                        @"0005000c[13]000[01]003[0]*\d\.tik",
                        @"0005000c[13]000[01]003[0]*\d\.cert",
                        @"0005000c[13]000[01]004[0]*\d\.tik",
                        @"0005000c[13]000[01]004[0]*\d\.cert",
                        @"authoringtoolinfo.xml",
                    };
                }
                else
                {
                    entryRuleStringArray = new string[] {
                        @".*\.nca/fs0/data\.dat",
                        @".*\.cnmt.xml",
                        @".*\.cnmt\.nca/fs0/AddOnContent_0005000c[13]000[01]00[34].cnmt",
                        @"0005000c[13]000[01]00[34][0]*\d\.tik",
                        @"0005000c[13]000[01]00[34][0]*\d\.cert",
                        @"authoringtoolinfo.xml",
                    };
                }

                CheckTicketInNsp(outputPath, config.GetKeyConfiguration());
                TestExtractWithNsp(outputPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);

                // replace
                {
                    Tuple<Regex, bool>[] ReplaceEntryInNspTestList = new Tuple<Regex, bool>[] {
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/data.dat"), false),
                        new Tuple<Regex, bool>(new Regex(@".*\.cnmt\.nca/fs0/AddOnContent_0005000c[13]000[01]004.cnmt"), false),
                    };
                    var testCombination = new List<int[]>() { new int[] { 0, 1 } };
                    TestReplaceWithNspOrNca(outputPath, testReplaceDir, testExtractDir, testExtractEntryDir, ReplaceEntryInNspTestList, testCombination);

                    // ticket 有り版は nca 差し替えテストを省略
                }

                SafeDeleteDirectory(testExtractDir);
                SafeDeleteDirectory(testExtractEntryDir);

                errorMsg = ExecuteProgram(string.Format("prodencryption --no-check --keyconfig {0} -o {1} {2}", keyConfigFile, outputDir, outputPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            }
        }

        private struct SaveDataParams
        {
            public UInt64 owner;
            public int userSave;
            public int userSaveJ;
            public int deviceSave;
            public int deviceSaveJ;
            public int bcat;
            public int temporaryStorage;
            public int cacheStorage;
            public int cacheStorageJ;
            public int userSaveMax;
            public int userSaveJMax;
            public int cacheStorageDJ;
            public int cacheStorageIdxMax;

            public static SaveDataParams MakeDefaultZero()
            {
                return new SaveDataParams()
                {
                    owner = 0x0005000C10000000
                };
            }

            public static SaveDataParams MakeDefault()
            {
                return new SaveDataParams()
                {
                    owner = 0x0005000C10000000,
                    userSave = 0x10000,
                    userSaveJ = 0x10000,
                    deviceSave = 0x10000,
                    deviceSaveJ = 0x10000,
                    bcat = 0x600000,
                    temporaryStorage = 0x600000,
                    userSaveMax = 0x100000,
                    userSaveJMax = 0x100000,
                    cacheStorage = 0x10000,
                    cacheStorageJ = 0x10000,
                    cacheStorageDJ = 0x0,
                    cacheStorageIdxMax = 0x0
                };
            }

            public static SaveDataParams Make(Func<SaveDataParams, SaveDataParams> modify)
            {
                var param = MakeDefault();
                return modify(param);
            }
        };

        [TestMethod]
        public void TestSaveDataModificationCheck()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

            Func<string, UInt32, SaveDataParams, string> makeMeta = (string path, UInt32 version, SaveDataParams param) =>
            {
                var bcatStr = param.bcat == 0 ? "" : string.Format(@"<BcatDeliveryCacheStorageSize>0x{0}</BcatDeliveryCacheStorageSize>
                                                                     <BcatPassphrase>0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef</BcatPassphrase>",  param.bcat.ToString("x16"));
                var meta = string.Format(@"<?xml version=""1.0""?>
                                           <NintendoSdkMeta>
                                             <Core>
                                               <ApplicationId>0x0005000C10000001</ApplicationId>
                                             </Core>
                                             <Application>
                                               <Version>{0}</Version>
                                               <SaveDataOwnerId>0x{1}</SaveDataOwnerId>
                                               <UserAccountSaveDataSize>0x{2}</UserAccountSaveDataSize>
                                               <UserAccountSaveDataJournalSize>0x{3}</UserAccountSaveDataJournalSize>
                                               <DeviceSaveDataSize>0x{4}</DeviceSaveDataSize>
                                               <DeviceSaveDataJournalSize>0x{5}</DeviceSaveDataJournalSize>
                                               {6}
                                               <TemporaryStorageSize>0x{7}</TemporaryStorageSize>
                                               <CacheStorageSize>0x{8}</CacheStorageSize>
                                               <CacheStorageJournalSize>0x{9}</CacheStorageJournalSize>
                                               <UserAccountSaveDataSizeMax>0x{10}</UserAccountSaveDataSizeMax>
                                               <UserAccountSaveDataJournalSizeMax>0x{11}</UserAccountSaveDataJournalSizeMax>
                                               <CacheStorageDataAndJournalSizeMax>0x{12}</CacheStorageDataAndJournalSizeMax>
                                               <CacheStorageIndexMax>0x{13}</CacheStorageIndexMax>
                                             </Application>
                                           </NintendoSdkMeta>", version, param.owner.ToString("x16"), param.userSave.ToString("x16"), param.userSaveJ.ToString("x16"), param.deviceSave.ToString("x16"), param.deviceSaveJ.ToString("x16"), bcatStr, param.temporaryStorage.ToString("x16"), param.cacheStorage.ToString("x16"), param.cacheStorageJ.ToString("x16"), param.userSaveMax.ToString("x16"), param.userSaveJMax.ToString("x16"), param.cacheStorageDJ.ToString("x16"), param.cacheStorageIdxMax.ToString("x16"));
                File.WriteAllText(path, meta);
                var metaPrefix = Path.GetFileName(path).Replace(".nmeta", string.Empty);
                return metaPrefix;
            };

            Action<string, string, string> makeApp = (string output, string meta, string dataDirectory) =>
            {
                string testNpdm = dataDirectory + "\\main.npdm";
                MakeNpdm(testNpdm, meta, descFile);

                var error = ExecuteProgram(string.Format(
                     "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3}",
                     output,
                     meta,
                     descFile,
                     dataDirectory));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(output));
                VerifyNsp(output, null);
            };

            Action<string, string> makePatch = (string original, string current) =>
            {
                var output = current.Replace("save", "patch-save");
                var error = ExecuteProgram(string.Format(
                    "makepatch -o {0} --desc {1} --original {2}  --current {3}",
                    output,
                    descFile,
                    original,
                    current));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(output));
                VerifyNsp(output, null);
            };

            Action<string, string, string, string> makePatchError = (string original, string current, string previous, string expectedMessage) =>
            {
                var output = current.Replace("save", "patch-save");
                var error = ExecuteProgram(string.Format(
                    "makepatch -o {0} --desc {1} --original {2}  --current {3} {4}",
                    output,
                    descFile,
                    original,
                    current,
                    previous != null ? string.Format($"--previous {previous}") :  string.Empty));
                Assert.IsTrue(error.Contains(expectedMessage), error);
            };

            Action<string, string, string> checkSaveSize = (string originalMeta, string currentMeta, string expectedMessage) =>
            {
                Console.WriteLine($"{originalMeta}:{currentMeta}");
                Func<string, ApplicationControlPropertyModel> readXml = (metaPath) =>
                {
                    var document = new XmlDocument();
                    document.Load(metaPath);
                    var node = document.SelectSingleNode("//Application");
                    var nodeString = node == null ? "<Application></Application>" : node.OuterXml;

                    var reader = XmlReader.Create(new StringReader(nodeString));

                    XmlSerializer serializer = new System.Xml.Serialization.XmlSerializer(typeof(ApplicationControlPropertyModel));
                    serializer.UnknownElement += XmlUnknownElementHandler.DeserializationHandler;
                    return (ApplicationControlPropertyModel)serializer.Deserialize(reader);
                };
                var orgModel = readXml(originalMeta);
                var curModel = readXml(currentMeta);
                try
                {
                    ArchiveReconstructionUtils.CheckSaveDataSize(orgModel, curModel);
                    Assert.IsTrue(false);
                }
                catch (Exception e)
                {
                    Assert.IsTrue(e.Message.IndexOf(expectedMessage) >= 0);
                }
            };

            var testRoot = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());
            SafeDeleteDirectory(testRoot);
            Directory.CreateDirectory(testRoot);

            var codeDir = Path.Combine(testRoot, "code");
            SafeDeleteDirectory(codeDir);
            Directory.CreateDirectory(codeDir);

            // 基準の v0
            makeMeta(Path.Combine(testRoot, "save-v0.nmeta"),    0, SaveDataParams.MakeDefault());
            makeMeta(Path.Combine(testRoot, "save-v0-ui.nmeta"), 0, SaveDataParams.Make(x => { x.cacheStorage = 0x0; x.cacheStorageJ = 0x0; x.cacheStorageDJ = 0x100000; x.cacheStorageIdxMax = 0x10; return x; }));

            // 新規セーブ利用
            makeMeta(Path.Combine(testRoot, "nosave-v0.nmeta"), 0, SaveDataParams.MakeDefaultZero());

            // 変更なし
            makeMeta(Path.Combine(testRoot, "save-v1-ok-nc.nmeta"), 65536, SaveDataParams.MakeDefault());
            makeMeta(Path.Combine(testRoot, "save-v1-ui-ok-nc.nmeta"), 65536, SaveDataParams.Make(x => { x.cacheStorage = 0x0; x.cacheStorageJ = 0x0; x.cacheStorageDJ = 0x100000; x.cacheStorageIdxMax = 0x10; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v2-ng-nc.nmeta"), 131072, SaveDataParams.MakeDefault());

            // owner Id 変更
            makeMeta(Path.Combine(testRoot, "save-v1-ng-o1.nmeta"), 65536, SaveDataParams.Make(x => { x.owner = 0x0005000C10000001L; return x; }));

            // サイズ縮小
            makeMeta(Path.Combine(testRoot, "save-v1-ng-s1.nmeta"),    65536, SaveDataParams.Make(x => { x.userSave = 0xC000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-s2.nmeta"),    65536, SaveDataParams.Make(x => { x.userSaveJ = 0xC000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-s3.nmeta"),    65536, SaveDataParams.Make(x => { x.deviceSave = 0xC000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-s4.nmeta"),    65536, SaveDataParams.Make(x => { x.deviceSaveJ = 0xC000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-s5.nmeta"),    65536, SaveDataParams.Make(x => { x.bcat = 0x500000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-s6.nmeta"),    65536, SaveDataParams.Make(x => { x.temporaryStorage = 0x500000; return x; })); // TORIAEZU
            makeMeta(Path.Combine(testRoot, "save-v1-ng-s7.nmeta"),    65536, SaveDataParams.Make(x => { x.cacheStorage = 0xC000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-s8.nmeta"),    65536, SaveDataParams.Make(x => { x.cacheStorageJ = 0xC000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-s9.nmeta"),    65536, SaveDataParams.Make(x => { x.userSaveMax = 0xC0000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-s10.nmeta"),   65536, SaveDataParams.Make(x => { x.userSaveJMax = 0xC0000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ui-ng-s1.nmeta"), 65536, SaveDataParams.Make(x => { x.cacheStorage = 0x0; x.cacheStorageJ = 0x0; x.cacheStorageDJ = 0x1C000; x.cacheStorageIdxMax = 0x10; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ui-ng-s2.nmeta"), 65536, SaveDataParams.Make(x => { x.cacheStorage = 0x0; x.cacheStorageJ = 0x0; x.cacheStorageDJ = 0x100000; x.cacheStorageIdxMax = 0xC; return x; }));

            // サイズ拡張 1 MiB 違反
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e1.nmeta"),  65536, SaveDataParams.Make(x => { x.userSave = 0x20000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e2.nmeta"),  65536, SaveDataParams.Make(x => { x.userSave = 0x100000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e3.nmeta"),  65536, SaveDataParams.Make(x => { x.userSaveJ = 0x20000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e4.nmeta"),  65536, SaveDataParams.Make(x => { x.userSaveJ = 0x100000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e5.nmeta"),  65536, SaveDataParams.Make(x => { x.deviceSave = 0x20000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e6.nmeta"),  65536, SaveDataParams.Make(x => { x.deviceSave = 0x100000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e7.nmeta"),  65536, SaveDataParams.Make(x => { x.deviceSaveJ = 0x20000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e8.nmeta"),  65536, SaveDataParams.Make(x => { x.deviceSaveJ = 0x100000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e9.nmeta"),  65536, SaveDataParams.Make(x => { x.cacheStorage = 0x20000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e10.nmeta"), 65536, SaveDataParams.Make(x => { x.cacheStorage = 0x100000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e11.nmeta"), 65536, SaveDataParams.Make(x => { x.cacheStorageJ = 0x20000; return x; }));
            makeMeta(Path.Combine(testRoot, "save-v1-ng-e12.nmeta"), 65536, SaveDataParams.Make(x => { x.cacheStorageJ = 0x100000; return x; }));

            // サイズ拡張 OK
            makeMeta(Path.Combine(testRoot, "save-v1-ok-e1.nmeta"), 65536, SaveDataParams.Make(x =>
            {
                x.userSave = 0x1000000;
                x.userSaveJ = 0x1000000;
                x.deviceSave = 0x1000000;
                x.deviceSaveJ = 0x1000000;
                x.bcat = 0x700000;
                x.temporaryStorage = 0x610000; // アライン制限無し
                x.cacheStorage = 0x1000000;
                x.cacheStorageJ = 0x1000000;
                x.userSaveMax = 0x200000;
                x.userSaveJMax = 0x200000;
                return x;
            }));

            makeMeta(Path.Combine(testRoot, "save-v1-ui-ok-e1.nmeta"), 65536, SaveDataParams.Make(x =>
            {
                x.cacheStorage = 0x0;
                x.cacheStorageJ = 0x0;
                x.cacheStorageDJ = 0x110000; // アライン制限無し
                x.cacheStorageIdxMax = 0x11;
                return x;
            }));

            // make patch は代表ケースのみ確認。残りは CheckSaveDataSize を直接実行
            makeApp(Path.Combine(testRoot, "nosave-v0.nsp"), Path.Combine(testRoot, "nosave-v0.nmeta"), codeDir);
            makeApp(Path.Combine(testRoot, "save-v0.nsp"), Path.Combine(testRoot, "save-v0.nmeta"), codeDir);
            makeApp(Path.Combine(testRoot, "save-v0-ui.nsp"), Path.Combine(testRoot, "save-v0-ui.nmeta"), codeDir);

            makeApp(Path.Combine(testRoot, "save-v1-ok-nc.nsp"), Path.Combine(testRoot, "save-v1-ok-nc.nmeta"), codeDir);
            makePatch(Path.Combine(testRoot, "nosave-v0.nsp"), Path.Combine(testRoot, "save-v1-ok-nc.nsp"));
            makePatch(Path.Combine(testRoot, "save-v0.nsp"), Path.Combine(testRoot, "save-v1-ok-nc.nsp"));

            makeApp(Path.Combine(testRoot, "save-v1-ui-ok-nc.nsp"), Path.Combine(testRoot, "save-v1-ui-ok-nc.nmeta"), codeDir);
            makePatch(Path.Combine(testRoot, "save-v0-ui.nsp"), Path.Combine(testRoot, "save-v1-ui-ok-nc.nsp"));
            makePatch(Path.Combine(testRoot, "save-v0.nsp"), Path.Combine(testRoot, "save-v1-ui-ok-nc.nsp"));

            makeApp(Path.Combine(testRoot, "save-v1-ng-o1.nsp"), Path.Combine(testRoot, "save-v1-ng-o1.nmeta"), codeDir);
            makePatchError(Path.Combine(testRoot, "save-v0.nsp"), Path.Combine(testRoot, "save-v1-ng-o1.nsp"), null, "SaveDataOwnerId can't be changed by patch.");

            makeApp(Path.Combine(testRoot, "save-v1-ng-s1.nsp"), Path.Combine(testRoot, "save-v1-ng-s1.nmeta"), codeDir);
            makePatchError(Path.Combine(testRoot, "save-v0.nsp"), Path.Combine(testRoot, "save-v1-ng-s1.nsp"), null, "Save data size shrink from original application is detected. Currently this feature is not supported.");
            foreach (var i in Enumerable.Range(2, 7))
            {
                checkSaveSize(Path.Combine(testRoot, "save-v0.nmeta"), Path.Combine(testRoot, string.Format($"save-v1-ng-s{i}.nmeta")), i == 6 ? "TemporaryStorageSize can't be smaller than original." : "Save data size shrink from original application is detected. Currently this feature is not supported.");
            }
            foreach (var i in Enumerable.Range(9, 2))
            {
                checkSaveSize(Path.Combine(testRoot, "save-v0.nmeta"), Path.Combine(testRoot, string.Format($"save-v1-ng-s{i}.nmeta")), "can't be smaller than original.");
            }

            makeApp(Path.Combine(testRoot, "save-v1-ui-ng-s1.nsp"), Path.Combine(testRoot, "save-v1-ui-ng-s1.nmeta"), codeDir);
            makePatchError(Path.Combine(testRoot, "save-v0-ui.nsp"), Path.Combine(testRoot, "save-v1-ui-ng-s1.nsp"), null, "can't be smaller than original.");
            makePatchError(Path.Combine(testRoot, "save-v0.nsp"), Path.Combine(testRoot, "save-v1-ui-ng-s1.nsp"), null, "Switing to IndexedCacheStorage from CachcStorage requires that CacheStorageDataAndJournalSizeMax be larger than sum of CacheStorageSize and CacheStorageJournalSize for original application.");
            {
                checkSaveSize(Path.Combine(testRoot, "save-v0-ui.nmeta"), Path.Combine(testRoot, string.Format($"save-v1-ui-ng-s2.nmeta")), "can't be smaller than original.");
            }

            makeApp(Path.Combine(testRoot, "save-v1-ng-e1.nsp"), Path.Combine(testRoot, "save-v1-ng-e1.nmeta"), codeDir);
            makePatchError(Path.Combine(testRoot, "save-v0-ui.nsp"), Path.Combine(testRoot, "save-v1-ng-e1.nsp"), null, "Save data extension requires that the data and journal size is multiple of 1 MiB.");
            foreach (var i in Enumerable.Range(2, 11))
            {
                checkSaveSize(Path.Combine(testRoot, "save-v0.nmeta"), Path.Combine(testRoot, string.Format($"save-v1-ng-e{i}.nmeta")), "Save data extension requires that the data and journal size is multiple of 1 MiB.");
            }

            makeApp(Path.Combine(testRoot, "save-v1-ok-e1.nsp"), Path.Combine(testRoot, "save-v1-ok-e1.nmeta"), codeDir);
            makePatch(Path.Combine(testRoot, "save-v0.nsp"), Path.Combine(testRoot, "save-v1-ok-e1.nsp"));
            {
                // 拡張 ⇒ 縮小で前回パッチとのチェックが走るか
                makeApp(Path.Combine(testRoot, "save-v2-ng-nc.nsp"), Path.Combine(testRoot, "save-v2-ng-nc.nmeta"), codeDir);
                makePatchError(Path.Combine(testRoot, "save-v0.nsp"), Path.Combine(testRoot, "save-v2-ng-nc.nsp"), Path.Combine(testRoot, "patch-save-v1-ok-e1.nsp"), "Save data size shrink from original application is detected. Currently this feature is not supported.");
            }

            makeApp(Path.Combine(testRoot, "save-v1-ui-ok-e1.nsp"), Path.Combine(testRoot, "save-v1-ui-ok-e1.nmeta"), codeDir);
            makePatch(Path.Combine(testRoot, "save-v0-ui.nsp"), Path.Combine(testRoot, "save-v1-ui-ok-e1.nsp"));

            SafeDeleteDirectory(testRoot);
        }

        [TestMethod]
        public void CardSizeLogic()
        {
            Action<bool, int, int, long> e1 = (bool autoSetSize, int cardSize, int requiredCardSize, long appAreaUsedSize) =>
            {
                Utils.CheckReturnException(new ArgumentException(string.Format("This nsp requires CardSpec/Size = {0}, though it is set as {1}.", requiredCardSize, cardSize)),
                    () =>
                    {
                        CardSpecXmlSource.SetupCardSize(ref autoSetSize, ref cardSize, requiredCardSize, appAreaUsedSize);
                        return true;
                    }
                );
            };
            Action<bool, int, int, long> e2 = (bool autoSetSize, int cardSize, int requiredCardSize, long appAreaUsedSize) =>
            {
                Utils.CheckReturnException(new ArgumentException(string.Format("CardSpec/Size in nmeta is invalid value. Please remove CardSpec/Size to set the value automatically.")),
                    () =>
                    {
                        CardSpecXmlSource.SetupCardSize(ref autoSetSize, ref cardSize, requiredCardSize, appAreaUsedSize);
                        return true;
                    }
                );
            };

            Func<bool, int, int, long, bool, int, bool> s = (bool autoSetSize, int cardSize, int requiredCardSize, long appAreaUsedSize, bool expAutoSetSize, int expCardSize) =>
            {
                CardSpecXmlSource.SetupCardSize(ref autoSetSize, ref cardSize, requiredCardSize, appAreaUsedSize);
                return autoSetSize == expAutoSetSize && cardSize == expCardSize;
            };

            Func<double, long> ToGb = (double gb) => (long)(gb * 1024 * 1024 * 1024);
            // warning メッセージまでは見ない
            Assert.IsTrue(s(false, 0, 4, ToGb(3.2), true, 4));
            Assert.IsTrue(s(false, 0, 32, ToGb(25), true, 32));
            Assert.IsTrue(s(false, 0, -1, ToGb(40), true, -1));
            Assert.IsTrue(s(false, 4, 4, ToGb(3.2), false, 4));
            Assert.IsTrue(s(false, 8, 4, ToGb(3.2), false, 8));

            e1(false, 4, 8, ToGb(7));
            e2(false, 4, -1, ToGb(50));

            // ここからが on card patch
            Assert.IsTrue(s(true, 4, 4, ToGb(3.2), true, 4));
            Assert.IsTrue(s(true, 4, 8, ToGb(7), true, 8));
            Assert.IsTrue(s(false, 8, 8, ToGb(7), false, 8));
            Assert.IsTrue(s(false, 4, 4, ToGb(2.5), false, 4));

            e1(false, 4, 8, ToGb(7));

            Assert.IsTrue(s(true, -1, -1, ToGb(35), true, -1));
            Assert.IsTrue(s(true, -1, -1, ToGb(35), true, -1));
            Assert.IsTrue(s(true, 32, -1, ToGb(35), true, -1));

            e2(false, 32, -1, ToGb(35));
        }

        private void TestAccessibleUrlsNewLineCharactersNspImpl(TestEnvironment env, string fileName, string filePath, string compareFilePath)
        {
            TestMetaFile metaFile = new TestMetaFile();
            metaFile.AddApplicationTags(string.Format("<AccessibleUrlsFilePath>{0}</AccessibleUrlsFilePath>", filePath));
            metaFile.Write();

            env.MakeNpdm(metaFile.GetFilePath(), env.DefaultDescFile);
            var outputPath = Path.Combine(env.OutputDir, string.Format("{0}.nsp", fileName));

            var error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                outputPath,
                metaFile.GetFilePath(),
                env.DefaultDescFile,
                env.TestCodeDir));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(File.Exists(outputPath));
            VerifyNsp(outputPath, null);

            var outputFileName = "accessible-urls.txt";
            var outputFilePath = Path.Combine(env.OutputDir, outputFileName);
            ExtractFileFromNsp(env.OutputDir, outputPath, outputFileName);

            CompareTextFiles(outputFilePath, compareFilePath);
        }

        private void TestAccessibleUrlsNewLineCharactersNspdImpl(TestEnvironment env, string fileName, string filePath, string compareFilePath)
        {
            TestMetaFile metaFile = new TestMetaFile();
            metaFile.AddApplicationTags(string.Format("<AccessibleUrlsFilePath>{0}</AccessibleUrlsFilePath>", filePath));
            metaFile.Write();

            env.MakeNpdm(metaFile.GetFilePath(), env.DefaultDescFile);
            var outputPath = Path.Combine(env.OutputDir, string.Format("{0}.nspd", fileName));

            var error = ExecuteProgram(string.Format(
                "createnspd -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                outputPath,
                metaFile.GetFilePath(),
                env.DefaultDescFile,
                env.TestCodeDir));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(Directory.Exists(outputPath));

            var outputFilePath = Path.Combine(outputPath, "htmlDocument0.ncd/data/accessible-urls/accessible-urls.txt");
            CompareTextFiles(outputFilePath, compareFilePath);
        }

        [TestMethod]
        public void TestAccessibleUrlsNewLineCharacters()
        {
            TestEnvironment env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var testExtractDir = Path.Combine(env.OutputDir, "extract");
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var compareFileName = "accessible-urls.txt";
            var crlfFileName = "accessible-urlsCrLf.txt";
            var crFileName = "accessible-urlsCr.txt";
            var lfFileName = "accessible-urlsLf.txt";
            var compareFilePath = Path.Combine(accessibleUrlsDir, compareFileName);
            var crlfFilePath = Path.Combine(accessibleUrlsDir, crlfFileName);
            var crFilePath = Path.Combine(accessibleUrlsDir, crFileName);
            var lfFilePath = Path.Combine(accessibleUrlsDir, lfFileName);

            MakeFileImpl(crlfFilePath, "http://test0.com\r\nhttp://test1.com\r\n");
            MakeFileImpl(crFilePath, "http://test0.com\rhttp://test1.com\r");
            MakeFileImpl(lfFilePath, "http://test0.com\nhttp://test1.com\n");

            // lf は lf のまま
            TestAccessibleUrlsNewLineCharactersNspImpl(env, lfFileName, lfFilePath, lfFilePath);
            TestAccessibleUrlsNewLineCharactersNspdImpl(env, lfFileName, lfFilePath, lfFilePath);

            // cr は lf に置換
            TestAccessibleUrlsNewLineCharactersNspImpl(env, crFileName, crFilePath, lfFilePath);
            TestAccessibleUrlsNewLineCharactersNspdImpl(env, crFileName, crFilePath, lfFilePath);

            // crlf は lf に置換
            TestAccessibleUrlsNewLineCharactersNspImpl(env, crlfFileName, crlfFilePath, lfFilePath);
            TestAccessibleUrlsNewLineCharactersNspdImpl(env, crlfFileName, crlfFilePath, lfFilePath);
        }

        [TestMethod]
        public void TestReplaceAccessibleUrls()
        {
            TestEnvironment env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var testExtractDir = Path.Combine(env.OutputDir, "extract");
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var test1File = "test1.txt";
            var test2File = "test2.txt";
            var test3File = "accessible-urls.txt";
            var test1FilePath = Path.Combine(accessibleUrlsDir, test1File);
            var test2FilePath = Path.Combine(accessibleUrlsDir, test2File);
            var test3FilePath = Path.Combine(accessibleUrlsDir, test3File);

            MakeFileImpl(test1FilePath, "http://test0.com\n");
            MakeFileImpl(test2FilePath, "http://test1.com\n");
            MakeFileImpl(test3FilePath, "http://test2.com\n");

            TestMetaFile metaFile = new TestMetaFile();
            metaFile.AddApplicationTags(string.Format("<AccessibleUrlsFilePath>{0}</AccessibleUrlsFilePath>", test1FilePath));
            metaFile.Write();

            env.MakeNpdm(metaFile.GetFilePath(), env.DefaultDescFile);
            var outputPath = Path.Combine(env.OutputDir, "test1.nsp");

            var error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                outputPath,
                metaFile.GetFilePath(),
                env.DefaultDescFile,
                env.TestCodeDir));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(File.Exists(outputPath));
            VerifyNsp(outputPath, null);

            var outputFileName = "accessible-urls.txt";
            var outputFilePath = Path.Combine(env.OutputDir, outputFileName);
            ExtractFileFromNsp(env.OutputDir, outputPath, outputFileName);

            CompareTextFiles(outputFilePath, test1FilePath);

            var origFilePath = ExtractFilePathFromNsp(outputPath, outputFileName);
            var replacedNspPath = Path.Combine(env.OutputDir, "test1_replaced.nsp");
            error = ExecuteProgram(string.Format(
                "replace -o {0} {1} {2} {3}",
                env.OutputDir,
                outputPath,
                origFilePath,
                test2FilePath));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(File.Exists(replacedNspPath));
            VerifyNsp(replacedNspPath, null);

            ExtractFileFromNsp(env.OutputDir, replacedNspPath, outputFileName);
            CompareTextFiles(outputFilePath, test2FilePath);

            // nca を追加した場合
            TestMetaFile noHtmlMetaFile = new TestMetaFile();
            noHtmlMetaFile.Write();

            var noHtmlNsp = Path.Combine(env.OutputDir, "noHtml.nsp");
            error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                noHtmlNsp,
                noHtmlMetaFile.GetFilePath(),
                env.DefaultDescFile,
                env.TestCodeDir));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(File.Exists(noHtmlNsp));
            VerifyNsp(noHtmlNsp, null);

            var htmlNca = Path.Combine(env.OutputDir, "html.nca");
            error = ExecuteProgram(string.Format(
                "createnca -o {0} --html-document {1} --meta {2} --meta-type Application",
                htmlNca,
                accessibleUrlsDir,
                noHtmlMetaFile.GetFilePath()));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(File.Exists(htmlNca));

            replacedNspPath = Path.Combine(env.OutputDir, "noHtml_replaced.nsp");
            error = ExecuteProgram(string.Format(
                "replace -o {0} {1} null.nca@HtmlDocument {2}",
                env.OutputDir,
                noHtmlNsp,
                htmlNca));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(File.Exists(replacedNspPath));
            VerifyNsp(replacedNspPath, null);

            ExtractFileFromNsp(env.OutputDir, replacedNspPath, outputFileName);
            CompareTextFiles(outputFilePath, test3FilePath);
        }

        [TestMethod]
        public void TestLocalCommunicationId()
        {
            TestEnvironment env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);
            string metaFile = Path.Combine(env.OutputDir, "localCommunicationMeta0.nmeta");

            // LocalCommunicationId == 0 を明示的に指定することが出来る
            ulong id = 0;
            MakeTestMetaFile(metaFile, new string[]
                {
                    string.Format("<LocalCommunicationId>{0:X16}</LocalCommunicationId>", id),
                });
            env.MakeNpdm(metaFile, env.DefaultDescFile);

            var nspFilePath = Path.Combine(env.OutputDir, "localCommunicationMeta0.nsp");
            var error = ExecuteProgram(string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3}", nspFilePath, metaFile, env.DefaultDescFile, env.TestCodeDir));
            Assert.IsTrue(error == string.Empty, error);

            using (var nspReader = new NintendoSubmissionPackageReader(nspFilePath))
            {
                var file = nspReader.ListFileInfo().Single(x => x.Item1.EndsWith(".nacp.xml"));
                using (var ms = new MemoryStream(nspReader.ReadFile(file.Item1, 0, nspReader.GetFileSize(file.Item1))))
                {
                    var document = new XmlDocument();
                    document.Load(ms);
                    var node = document.SelectSingleNode("//Application");
                    Assert.IsTrue(node != null);
                    var nodeString = node.OuterXml;

                    var reader = XmlReader.Create(new StringReader(nodeString));
                    XmlSerializer serializer = new System.Xml.Serialization.XmlSerializer(typeof(ApplicationControlPropertyModel));
                    var model = (ApplicationControlPropertyModel)serializer.Deserialize(reader);
                    Assert.AreEqual(model.LocalCommunicationId.Count, 2);
                    Assert.IsTrue(model.LocalCommunicationId.Contains(string.Format("0x{0:X16}", id)));
                    Assert.IsTrue(model.LocalCommunicationId.Contains("0x0100000000002802"));
                }
            }
        }

        [TestMethod]
        public void TestMergeAndSplitNsp()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);
            var iconPath = env.SourceDir + "\\Icon\\describe_all.bmp";

            var baseMetaFile = env.SourceDir + "\\ApplicationMeta\\version.nmeta";
            var meta1FileV1 = env.OutputDir + "\\1_v1.nmeta";
            var meta1FileV2 = env.OutputDir + "\\1_v2.nmeta";
            var meta2FileV1 = env.OutputDir + "\\2_v1.nmeta";
            var meta3FileV3 = env.OutputDir + "\\3_v3.nmeta";

            MakeSpecifiedVersionMetaFile(baseMetaFile, meta1FileV1, 1, 0x0005000C10000001);
            MakeSpecifiedVersionMetaFile(baseMetaFile, meta1FileV2, 2, 0x0005000C10000001);
            MakeSpecifiedVersionMetaFile(baseMetaFile, meta2FileV1, 1, 0x0005000C10000002);
            MakeSpecifiedVersionMetaFile(baseMetaFile, meta3FileV3, 3, 0x0005000C10000003);

            Action<string, string, bool> createNsp = (nsp, metaFile, hasTicket) =>
            {
                var cmd = string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {3} --icon AmericanEnglish {4} Japanese {4}", nsp, metaFile, env.DefaultDescFile, env.TestCodeDir, iconPath);
                if (hasTicket)
                {
                    cmd += " --ticket";
                }
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(nsp, null);
            };

            var app1v1 = env.OutputDir + "\\app1v1.nsp";
            var app1v2 = env.OutputDir + "\\app1v2.nsp";
            var app2v1 = env.OutputDir + "\\app2v1.nsp";
            var app3v3 = env.OutputDir + "\\app3v3.nsp";

            var merged = env.OutputDir + "\\merged.nsp";
            var splitDir = env.OutputDir + "\\split";

            createNsp(app1v1, meta1FileV1, false);
            createNsp(app1v2, meta1FileV2, false);
            createNsp(app2v1, meta2FileV1, false);
            createNsp(app3v3, meta3FileV3, true);

            {
                var error = ExecuteProgram(string.Format("mergensp -o {0} {1} {2} {3} {4}", merged, app1v1, app1v2, app2v1, app3v3));
                Assert.IsTrue(error == string.Empty, error);
            }

            Action<string, string> split = (option, expectedError) =>
            {
                SafeDeleteDirectory(splitDir);
                var cmd = string.Format("splitnsp -o {0} {1} ", splitDir, merged) + option;
                var error = ExecuteProgram(cmd);
                if (expectedError == null)
                {
                    Assert.IsTrue(error == string.Empty, error);
                }
                else
                {
                    Assert.IsTrue(error.IndexOf(expectedError) >= 0);
                }
            };

            Action<string, string> compare = (baseNsp, nsp) =>
            {
                using (var baseNspReader = new NintendoSubmissionPackageReader(baseNsp))
                using (var nspReader = new NintendoSubmissionPackageReader(nsp))
                {
                    var fileInfos = baseNspReader.ListFileInfo();
                    Assert.AreEqual(fileInfos.Count, nspReader.ListFileInfo().Count);
                    foreach (var fileInfo in fileInfos)
                    {
                        var baseData = baseNspReader.ReadFile(fileInfo.Item1, 0, fileInfo.Item2);
                        var data = nspReader.ReadFile(fileInfo.Item1, 0, fileInfo.Item2);
                        Assert.IsTrue(baseData.SequenceEqual(data));
                    }
                }
            };

            var s_app1v1 = Path.Combine(splitDir, "0x0005000C10000001_v1.nsp");
            var s_app1v2 = Path.Combine(splitDir, "0x0005000C10000001_v2.nsp");
            var s_app2v1 = Path.Combine(splitDir, "0x0005000C10000002_v1.nsp");
            var s_app3v3 = Path.Combine(splitDir, "0x0005000C10000003_v3.nsp");

            split("", null);
            compare(app1v1, s_app1v1);
            compare(app1v2, s_app1v2);
            compare(app2v1, s_app2v1);
            compare(app3v3, s_app3v3);

            split("--id 0x0005000C10000002", null);
            compare(app2v1, s_app2v1);
            Assert.IsFalse(File.Exists(s_app1v1));
            Assert.IsFalse(File.Exists(s_app1v2));
            Assert.IsFalse(File.Exists(s_app3v3));

            split("--version 1", null);
            compare(app1v1, s_app1v1);
            compare(app2v1, s_app2v1);
            Assert.IsFalse(File.Exists(s_app1v2));
            Assert.IsFalse(File.Exists(s_app3v3));

            split("--id 0x0005000C10000003 --version 3", null);
            compare(app3v3, s_app3v3);
            Assert.IsFalse(File.Exists(s_app1v1));
            Assert.IsFalse(File.Exists(s_app1v2));
            Assert.IsFalse(File.Exists(s_app2v1));

            split("--id 0x0005000C10000004", "There is no content that matches --id or --version");
            Assert.IsFalse(File.Exists(s_app1v1));
            Assert.IsFalse(File.Exists(s_app1v2));
            Assert.IsFalse(File.Exists(s_app2v1));
            Assert.IsFalse(File.Exists(s_app3v3));

            split("--id 0x0005000C10000001 --version 3", "There is no content that matches --id or --version");
            Assert.IsFalse(File.Exists(s_app1v1));
            Assert.IsFalse(File.Exists(s_app1v2));
            Assert.IsFalse(File.Exists(s_app2v1));
            Assert.IsFalse(File.Exists(s_app3v3));
        }

        [TestMethod]
        public void TestExecutionDiffPatch()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var metaFileV0 = env.SourceDir + "\\UppTest\\describe_all_required_system_version.nmeta";
            var metaFileV1 = env.OutputDir + "\\" + Path.GetFileName(metaFileV0).Replace(".nmeta", "_v1.nmeta");
            var metaFileV2 = env.OutputDir + "\\" + Path.GetFileName(metaFileV0).Replace(".nmeta", "_v2.nmeta");
            var metaFileV3 = env.OutputDir + "\\" + Path.GetFileName(metaFileV0).Replace(".nmeta", "_v3.nmeta");
            MakeSpecifiedVersionMetaFile(metaFileV0, metaFileV1, 1);
            MakeSpecifiedVersionMetaFile(metaFileV0, metaFileV2, 2);
            MakeSpecifiedVersionMetaFile(metaFileV0, metaFileV3, 3);

            var iconPath = env.SourceDir + "\\Icon\\describe_all.bmp";

            var dataDirV0 = env.OutputDir + "\\data_v0";
            var dataDirV1 = env.OutputDir + "\\data_v1";
            var dataDirV2 = env.OutputDir + "\\data_v2";
            var dataDirV3 = env.OutputDir + "\\data_v3";

            Action<string, Tuple<string, int, char>[]> createDataDir = (dataDir, createInfoList) =>
            {
                Directory.CreateDirectory(dataDir);

                foreach (var createInfo in createInfoList)
                {
                    var file = Path.Combine(dataDir, createInfo.Item1);
                    var size = createInfo.Item2;
                    var paddingPattern = createInfo.Item3;

                    using (FileStream stream = File.Create(file))
                    {
                        byte[] data = new byte[size];
                        for (int i = 0; i < size; i++)
                        {
                            data[i] = (byte)paddingPattern;
                        }
                        stream.Write(data, 0, size);
                    }
                }
            };

            Action<string, string, string> createNsp = (nsp, metaFile, data) =>
            {
                var cmd = string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {4} --icon AmericanEnglish {5} Japanese {5}", nsp, metaFile, env.DefaultDescFile, env.TestCodeDir, data, iconPath);
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(nsp, null);
            };

            Action<string, string, string, string> createPatch = (nsp, orig, prev, cur) =>
            {
                var cmd = string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", nsp, env.DefaultDescFile, orig, cur);
                if (!string.IsNullOrEmpty(prev))
                {
                    cmd += string.Format(" --previous {0}", prev);
                }
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(nsp, null);
            };

            Func<string, string, string, string> diffPatch = (orig, prev, cur) =>
            {
                var cmd = string.Format("diffpatch {0} {1} {2} --keyconfig {3}", orig, prev, cur, env.DefaultKeyConfigFile);
                return ExecuteProgram(cmd, true);
            };

            Action<string, Tuple<char, string>[]> checkDiffPatch = (input, expectedInfoList) =>
            {
                var inputLines = input.Split('\n');

                foreach (var expectedInfo in expectedInfoList)
                {
                    char diffMark = expectedInfo.Item1;
                    string fileName = expectedInfo.Item2;
                    bool isMatch = false;

                    if (diffMark == '=')
                    {
                        foreach (var line in inputLines)
                        {
                            if (line.IndexOf(fileName) >= 0)
                            {
                                isMatch = true;
                                break;
                            }
                        }
                        Assert.IsFalse(isMatch);
                    }
                    else
                    {
                        foreach (var line in inputLines.Where(x => (x.Length > 0) && (x[0] == diffMark)))
                        {
                            if (line.IndexOf(fileName) >= 0)
                            {
                                isMatch = true;
                                break;
                            }
                        }
                        Assert.IsTrue(isMatch);
                    }
                }
            };

            var dataInfoListV0 = new Tuple<string, int, char>[] {
                new Tuple<string, int, char>("data0.bin", 512 * 1024, '1'),
                new Tuple<string, int, char>("data1.bin", 64 * 1024, '1'),
                new Tuple<string, int, char>("data2.bin", 64 * 1024, '0'),
            };
            var dataInfoListV1 = new Tuple<string, int, char>[] {
                new Tuple<string, int, char>("data0.bin", 512 * 1024, '1'),
                new Tuple<string, int, char>("data1.bin", 64 * 1024, '1'),
                new Tuple<string, int, char>("data2.bin", 64 * 1024, '0'),
                new Tuple<string, int, char>("data3.bin", 64 * 1024, '0'),
                new Tuple<string, int, char>("data4.bin", 64 * 1024, '1'), // 追加ファイル
                new Tuple<string, int, char>("data5.bin", 64 * 1024, '1'), // 追加ファイル
            };
            var dataInfoListV2 = new Tuple<string, int, char>[] {
                new Tuple<string, int, char>("data0.bin", 512 * 1024, '1'),
                new Tuple<string, int, char>("data1.bin", 64 * 1024, '1'),
                new Tuple<string, int, char>("data4.bin", 64 * 1024, '0'), // サイズ不変、内容変更
                new Tuple<string, int, char>("data5.bin", 128 * 1024, '1'), // サイズ増加
                new Tuple<string, int, char>("data7.bin", 64 * 1024, '0'), // 追加ファイル
                new Tuple<string, int, char>("data8.bin", 64 * 1024, '2'), // 追加ファイル
            };
            var dataInfoListV3 = new Tuple<string, int, char>[] {
                new Tuple<string, int, char>("data0.bin", 512 * 1024, '1'),
                new Tuple<string, int, char>("data1.bin", 32 * 1024, '1'), // サイズ縮小
                new Tuple<string, int, char>("data2.bin", 64 * 1024, '0'), // 削除ファイルのrevert
                new Tuple<string, int, char>("data4.bin", 64 * 1024, '1'), // v1内容にrevert
                new Tuple<string, int, char>("data5.bin", 128 * 1024, '1'),
                new Tuple<string, int, char>("data6.bin", 64 * 1024, '0'), // 追加ファイル
                new Tuple<string, int, char>("data7.bin", 64 * 1024, '3'), // サイズ不変、内容変更
            };
            var expectedInfoList_V1_V2 = new Tuple<char, string>[] {
                new Tuple<char, string>('=', "data0.bin"),
                new Tuple<char, string>('=', "data1.bin"),
                new Tuple<char, string>('-', "data2.bin"),
                new Tuple<char, string>('-', "data3.bin"),
                new Tuple<char, string>('!', "data4.bin"),
                new Tuple<char, string>('!', "data5.bin"),
                new Tuple<char, string>('=', "data6.bin"),
                new Tuple<char, string>('+', "data7.bin"),
                new Tuple<char, string>('+', "data8.bin"),
            };
            var expectedInfoList_V2_V3 = new Tuple<char, string>[] {
                new Tuple<char, string>('=', "data0.bin"),
                new Tuple<char, string>('!', "data1.bin"),
                new Tuple<char, string>('+', "data2.bin"),
                new Tuple<char, string>('=', "data3.bin"),
                new Tuple<char, string>('!', "data4.bin"),
                new Tuple<char, string>('=', "data5.bin"),
                new Tuple<char, string>('+', "data6.bin"),
                new Tuple<char, string>('!', "data7.bin"),
                new Tuple<char, string>('-', "data8.bin"),
            };
            var expectedInfoList_V1_V3 = new Tuple<char, string>[] {
                new Tuple<char, string>('=', "data0.bin"),
                new Tuple<char, string>('!', "data1.bin"),
                new Tuple<char, string>('=', "data2.bin"),
                new Tuple<char, string>('-', "data3.bin"),
                new Tuple<char, string>('=', "data4.bin"),
                new Tuple<char, string>('!', "data5.bin"),
                new Tuple<char, string>('+', "data6.bin"),
                new Tuple<char, string>('+', "data7.bin"),
                new Tuple<char, string>('=', "data8.bin"),
            };

            createDataDir(dataDirV0, dataInfoListV0);
            createDataDir(dataDirV1, dataInfoListV1);
            createDataDir(dataDirV2, dataInfoListV2);
            createDataDir(dataDirV3, dataInfoListV3);

            // app v0
            var v0 = env.OutputDir + "\\v0.nsp";
            createNsp(v0, metaFileV0, dataDirV0);

            // app v1
            var v1 = env.OutputDir + "\\v1.nsp";
            createNsp(v1, metaFileV1, dataDirV1);

            // app v2
            var v2 = env.OutputDir + "\\v2.nsp";
            createNsp(v2, metaFileV2, dataDirV2);

            // app v3
            var v3 = env.OutputDir + "\\v3.nsp";
            createNsp(v3, metaFileV3, dataDirV3);

            // patch v1
            var v1patch = env.OutputDir + "\\v1patch.nsp";
            createPatch(v1patch, v0, null, v1);

            // patch v2
            var v2patch = env.OutputDir + "\\v2patch.nsp";
            createPatch(v2patch, v0, v1patch, v2);

            // patch v3
            var v3patch = env.OutputDir + "\\v3patch.nsp";
            createPatch(v3patch, v0, v2patch, v3);

            string stdout;
            // diffpatch v1-v2
            stdout = diffPatch(v0, v1patch, v2patch);
            checkDiffPatch(stdout, expectedInfoList_V1_V2);

            // diffpatch v2-v3
            stdout = diffPatch(v0, v2patch, v3patch);
            checkDiffPatch(stdout, expectedInfoList_V2_V3);

            // diffpatch v1-v3
            stdout = diffPatch(v0, v1patch, v3patch);
            checkDiffPatch(stdout, expectedInfoList_V1_V3);
        }

        [TestMethod]
        public void TestExecutionPrograminfoInherit()
        {
            string[] buildTargetList = { "32", "64" };
            foreach (var buildTarget in buildTargetList)
            {
                var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);
                var testPath = new TestUtility.TestPath(this.TestContext);
                var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
                var metaDir = env.SourceDir + "\\ApplicationMeta";
                var iconPath = env.SourceDir + "\\Icon\\" + Path.GetFileNameWithoutExtension(GetDefaultApplicationMetaFile(metaDir)) + ".bmp";
                var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
                var originalNspFile = Path.Combine(outputDir, MethodBase.GetCurrentMethod().Name + ".nsp");

                // programinfo.xml のDescFileとBuildTarget情報をチェック
                Action<string, string> checkProgramInfo = (nsp, descFilePath) =>
                {
                    using (var nspReader = new NintendoSubmissionPackageReader(nsp))
                    {
                        var file = nspReader.ListFileInfo().Single(x => x.Item1.EndsWith(".programinfo.xml"));
                        var model = ArchiveReconstructionUtils.ReadXml<ProgramInfoModel>(nspReader, file.Item1, file.Item2);
                        Assert.AreEqual(Path.GetFileName(descFilePath), model.DescFileName);
                        Assert.AreEqual(buildTarget, model.BuildTarget);
                    }
                };

                TestMetaFile metaFile = MakeIconPathMetaFile(iconPath, outputDir);
                metaFile.Write();
                // パッチ作成用にバージョンを設定する
                MakeSpecifiedVersionMetaFile(metaFile.GetFilePath(), metaFile.GetFilePath(), 0);

                // ビルドターゲットを設定
                {
                    var document = new XmlDocument();
                    document.Load(metaFile.GetFilePath());

                    var coreNode = document.SelectSingleNode("//Core");

                    var instructionNode = document.CreateNode(XmlNodeType.Element, "Is64BitInstruction", string.Empty);
                    var addressSpaceNode = document.CreateNode(XmlNodeType.Element, "ProcessAddressSpace", string.Empty);

                    switch (buildTarget)
                    {
                        case "32":
                            instructionNode.InnerText = "False";
                            addressSpaceNode.InnerText = "AddressSpace32Bit";
                            break;
                        case "64":
                            instructionNode.InnerText = "True";
                            addressSpaceNode.InnerText = "AddressSpace64Bit";
                            break;
                    }

                    coreNode.AppendChild(instructionNode);
                    coreNode.AppendChild(addressSpaceNode);
                    document.Save(metaFile.GetFilePath());
                }

                // descファイル名が正しく設定されることを確認する為にファイル名を変更
                var originalDescFile = Path.Combine(outputDir, "Application1.desc");
                var newDescFile = Path.Combine(outputDir, "Application2.desc");
                File.Copy(descFile, originalDescFile);
                File.Copy(descFile, newDescFile);

                // creatensp
                {
                    env.MakeNpdm(metaFile.GetFilePath(), originalDescFile);
                    var error = ExecuteProgram(string.Format(
                        "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                        originalNspFile,
                        metaFile.GetFilePath(),
                        originalDescFile,
                        env.TestCodeDir));
                    Assert.IsTrue(error == string.Empty, error);
                    Assert.IsTrue(File.Exists(originalNspFile));
                    checkProgramInfo(originalNspFile, originalDescFile);
                }

                // replace
                {
                    var extractDir = Path.Combine(outputDir, "extract");
                    ExecuteProgram(string.Format("extract -o {0} {1}", extractDir, originalNspFile));
                    var replaceFile = ExtractFilePathFromNsp(originalNspFile, Path.GetFileName(env.TestCodeFile));
                    var replaceFileFullPath = extractDir + "\\" + replaceFile;
                    Assert.IsTrue(File.Exists(replaceFileFullPath));

                    // 名前が違うdescを指定してnspを編集
                    var replacedNspFile = originalNspFile.Replace(".nsp", "_replaced.nsp");
                    var error = ExecuteProgram(string.Format(
                        "replace -o {0} {1} {2} {3} --desc {4}",
                        outputDir,
                        originalNspFile,
                        replaceFile,
                        replaceFileFullPath,
                        newDescFile,
                        env.TestCodeDir));
                    Assert.IsTrue(error == string.Empty, error);
                    Assert.IsTrue(File.Exists(replacedNspFile));
                    checkProgramInfo(replacedNspFile, newDescFile);
                }

                // makepatch
                {
                    var newNspFile = originalNspFile.Replace(".nsp", "_new.nsp");
                    {
                        // パッチ作成用に新バージョンのnsp作成
                        var metaFielv1 = metaFile.GetFilePath().Replace(".nmeta", "_v1.nmeta");
                        MakeSpecifiedVersionMetaFile(metaFile.GetFilePath(), metaFielv1, 1);
                        env.MakeNpdm(metaFielv1, originalDescFile);
                        var error = ExecuteProgram(string.Format(
                            "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf",
                            newNspFile,
                            metaFielv1,
                            originalDescFile,
                            env.TestCodeDir));
                        Assert.IsTrue(error == string.Empty, error);
                        Assert.IsTrue(File.Exists(newNspFile));
                    }

                    // 名前が違うdescを指定してパッチ作成
                    {
                        var patchNspFile = originalNspFile.Replace(".nsp", "_patch.nsp");
                        var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}",
                            patchNspFile,
                            newDescFile,
                            originalNspFile,
                            newNspFile));
                        Assert.IsTrue(error == string.Empty, error);
                        Assert.IsTrue(File.Exists(patchNspFile));
                        checkProgramInfo(patchNspFile, newDescFile);
                    }
                }

                // prodencryption
                {
                    var prodNspFile = originalNspFile.Replace(".nsp", "_prod.nsp");
                    var error = ExecuteProgram(string.Format("prodencryption --no-check --keyconfig {0} --requiredSystemVersion {3} -o {1} {2}",
                        env.DefaultKeyConfigFile,
                        outputDir,
                        originalNspFile,
                        NintendoContentMeta.GetRequiredSystemVersion() + 1));
                    Assert.IsTrue(error == string.Empty, error);
                    Assert.IsTrue(File.Exists(prodNspFile));
                    checkProgramInfo(prodNspFile, originalDescFile);
                }
            }
        }

        [TestMethod]
        public void TestExecutionNacpIconHash()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);
            var testPath = new TestUtility.TestPath(this.TestContext);
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var metaDir = env.SourceDir + "\\ApplicationMeta";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var originalNspFile = Path.Combine(outputDir, MethodBase.GetCurrentMethod().Name + ".nsp");

            var rawIconEnglishPath = env.SourceDir + "\\Icon\\describe_all.bmp";
            var rawIconJapanesePath = env.SourceDir + "\\Icon\\no_application.bmp";
            var nxIconEnglishPath = env.SourceDir + "\\Icon\\describe_all.jpg";
            var nxIconJapanesePath = env.SourceDir + "\\Icon\\no_application.jpg";

            string[] metaLines = new string[]
            {
                "<Icon>",
                "<Language>AmericanEnglish</Language>",
                string.Format("<IconPath>{0}</IconPath>", rawIconEnglishPath),
                string.Format("<NxIconPath>{0}</NxIconPath>", nxIconEnglishPath),
                "</Icon>",
                "<Icon>",
                "<Language>Japanese</Language>",
                string.Format("<IconPath>{0}</IconPath>", rawIconJapanesePath),
                string.Format("<NxIconPath>{0}</NxIconPath>", nxIconJapanesePath),
                "</Icon>",
            };
            string metaFile = Path.Combine(outputDir, MethodBase.GetCurrentMethod().Name + ".nmeta");
            MakeTestMetaFile(metaFile, metaLines);

            // nacp.xmlからIconエントリを取得する
            Func<string, List<Icon>> getIconFromNsp = (nsp) =>
            {
                var iconInfoList = new List<Icon>();
                using (var nspReader = new NintendoSubmissionPackageReader(nsp))
                {
                    var file = nspReader.ListFileInfo().Single(x => x.Item1.EndsWith(".nacp.xml"));
                    var model = ArchiveReconstructionUtils.ReadXml<ApplicationControlPropertyModel>(nspReader, file.Item1, file.Item2);
                    return model.Icon;
                }
            };

            // Iconエントリから指定したタイプのアイコンのハッシュを取得する
            Func<List<Icon>, string, string, string> getIconHash = (iconList, language, type) =>
            {
                var icon = iconList.Where(entry => entry.Language == language).First();
                if (type == "NxIcon")
                {
                    return icon.NxIconHash;
                }
                else
                {
                    return icon.RawIconHash;
                }
            };

            // データからハッシュ値を計算する
            Func<byte[], string> calculateHash = (data) =>
            {
                SHA256CryptoServiceProvider hashCalculator = new SHA256CryptoServiceProvider();
                byte[] hash = hashCalculator.ComputeHash(data, 0, data.Length);
                return BitConverter.ToString(hash, 0, hash.Length).Replace("-", string.Empty).ToLower().Substring(0, hash.Length);
            };

            List<Icon> originalIconList = null;
            // creatensp
            {
                env.MakeNpdm(metaFile, descFile);
                var error = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} --save-adf --ignore-unpublishable-error",
                    originalNspFile,
                    metaFile,
                    descFile,
                    env.TestCodeDir));
                Assert.IsTrue(error == string.Empty, error);
                originalIconList = getIconFromNsp(originalNspFile);

                // no_application.jpg(Exif無しJPEG)を設定しているJapaneseのNXアイコンのハッシュが一致することを確認する
                Assert.IsTrue(getIconHash(originalIconList, "Japanese", "NxIcon") == calculateHash(File.ReadAllBytes(nxIconJapanesePath)));
                // no_application.jpg以外のアイコンはBMPからJPGへの変換やExif削除などをAuthoringTool内で実施している
                // nspに含まれるファイルと元のファイルは別のバイナリなので、ハッシュ値が設定されていることのみ確認する
                Assert.IsTrue(getIconHash(originalIconList, "AmericanEnglish", "RawIcon").Length == 32);
                Assert.IsTrue(getIconHash(originalIconList, "AmericanEnglish", "NxIcon").Length == 32);
                Assert.IsTrue(getIconHash(originalIconList, "Japanese", "RawIcon").Length == 32);

                // 全てのハッシュ値が違う事を確認する
                Assert.IsTrue((new string[]
                {
                    getIconHash(originalIconList, "AmericanEnglish", "RawIcon"),
                    getIconHash(originalIconList, "AmericanEnglish", "NxIcon"),
                    getIconHash(originalIconList, "Japanese", "RawIcon"),
                    getIconHash(originalIconList, "Japanese", "NxIcon")
                }).Distinct().Count() == 4);
            }

            // replace (1言語のNXアイコンのみ)
            {
                var extractDir = Path.Combine(outputDir, "extract");
                SafeDeleteDirectory(extractDir);
                ExecuteProgram(string.Format("extract -o {0} {1}", extractDir, originalNspFile));
                string stdout = ExecuteProgram(string.Format("list {0}", originalNspFile), true);
                Assert.IsTrue(!string.IsNullOrEmpty(stdout));
                var replaceFile = stdout.Split().ToList().Find(entry => entry.EndsWith(".nx.Japanese.jpg"));

                // JapaneseのNXアイコンを置き換える
                var replacedNspFile = originalNspFile.Replace(".nsp", "_replaced.nsp");
                var error = ExecuteProgram(string.Format(
                    "replace -o {0} {1} {2} {3} --desc {4} --ignore-unpublishable-error",
                    outputDir,
                    originalNspFile,
                    replaceFile,
                    nxIconEnglishPath,
                    descFile,
                    env.TestCodeDir));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(replacedNspFile));
                var replaceIconList = getIconFromNsp(replacedNspFile);
                // JapaneseのNXアイコンのハッシュのみ変更されていることを確認
                Assert.IsTrue(getIconHash(originalIconList, "AmericanEnglish", "RawIcon") == getIconHash(replaceIconList, "AmericanEnglish", "RawIcon"));
                Assert.IsTrue(getIconHash(originalIconList, "AmericanEnglish", "NxIcon") == getIconHash(replaceIconList, "AmericanEnglish", "NxIcon"));
                Assert.IsTrue(getIconHash(originalIconList, "Japanese", "RawIcon") == getIconHash(replaceIconList, "Japanese", "RawIcon"));
                Assert.IsTrue(getIconHash(originalIconList, "Japanese", "NxIcon") != getIconHash(replaceIconList, "Japanese", "NxIcon"));
            }

            // replace (複数言語の全アイコン)
            {
                var extractDir = Path.Combine(outputDir, "extract");
                SafeDeleteDirectory(extractDir);
                ExecuteProgram(string.Format("extract -o {0} {1}", extractDir, originalNspFile));
                string stdout = ExecuteProgram(string.Format("list {0}", originalNspFile), true);
                Assert.IsTrue(!string.IsNullOrEmpty(stdout));
                var fileList = stdout.Split().ToList();
                // replaceリストを作成（JapaneseアイコンとAmericanEnglishアイコンを交換）
                List<Tuple<string, string>> replaceRules = new List<Tuple<string, string>>
                {
                    Tuple.Create(fileList.Find(entry => entry.EndsWith(".nx.Japanese.jpg")), nxIconEnglishPath),
                    Tuple.Create(fileList.Find(entry => entry.EndsWith(".raw.Japanese.jpg")), rawIconEnglishPath),
                    Tuple.Create(fileList.Find(entry => entry.EndsWith(".nx.AmericanEnglish.jpg")), nxIconJapanesePath),
                    Tuple.Create(fileList.Find(entry => entry.EndsWith(".raw.AmericanEnglish.jpg")), rawIconJapanesePath),
                };
                var replaceRuleFile = outputDir + "\\replaceRule.txt";
                OutputReplaceList(replaceRuleFile, replaceRules);

                // 全てのアイコンを置き換える
                var replacedNspFile = originalNspFile.Replace(".nsp", "_replaced.nsp");
                var error = ExecuteProgram(string.Format(
                    "replace {0} -o {1} {2} --desc {3} --ignore-unpublishable-error",
                    originalNspFile,
                    outputDir,
                    replaceRuleFile,
                    descFile));
                Assert.IsTrue(error == string.Empty, error);
                Assert.IsTrue(File.Exists(replacedNspFile));
                var replaceIconList = getIconFromNsp(replacedNspFile);

                // 置き換えたファイルのハッシュが設定されていることを確認(replaceではアイコンはエンコードされないのでreplace時に指定したアイコンが設定される)
                Assert.IsTrue(getIconHash(replaceIconList, "Japanese", "RawIcon") == calculateHash(File.ReadAllBytes(rawIconEnglishPath)));
                Assert.IsTrue(getIconHash(replaceIconList, "Japanese", "NxIcon") == calculateHash(File.ReadAllBytes(nxIconEnglishPath)));
                Assert.IsTrue(getIconHash(replaceIconList, "AmericanEnglish", "RawIcon") == calculateHash(File.ReadAllBytes(rawIconJapanesePath)));
                Assert.IsTrue(getIconHash(replaceIconList, "AmericanEnglish", "NxIcon") == calculateHash(File.ReadAllBytes(nxIconJapanesePath)));
            }
        }

        [TestMethod]
        public void TestExecutionGetAllXmlModels()
        {
            string standardOut;
            standardOut = ExecuteProgram("get-all-xml-models", true);

            Assert.IsTrue(standardOut != string.Empty);
            // 子の階層が出力されていることを確認
            {
                int index = 0;
                Assert.IsTrue((index = standardOut.IndexOf("<Application>", index)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf("<Icon>", index + 1)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf("<Language></Language>", index + 1)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf("</Icon>", index + 1)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf("</Application>", index + 1)) >= 0);
            }

            string[] rootTags =
            {
                "Application",
                "AddOnContent",
                "AuthoringToolInfo",
                "CardSpec",
                "ContentMeta",
                "HtmlDocument",
                "SoftwareLegalInformation",
                "MultiApplicationCard",
                "OnCardAocInfo",
                "ProgramInfo",
            };

            // 各々のModelが出力されていることを確認
            foreach (var rootTag in rootTags)
            {
                int index = 0;
                Assert.IsTrue((index = standardOut.IndexOf("<" + rootTag + ">", index)) >= 0);
                Assert.IsTrue((index = standardOut.IndexOf("</" + rootTag + ">", index + 1)) >= 0);
            }
        }

        public class NcaKeyGeneratorWithoutEncryption : IKeyGenerator
        {
            public virtual byte[] Generate(byte[] encryptedKey, int keyType)
            {
                return new byte[encryptedKey.Length];
            }

            public virtual bool GetUseDevHsm()
            {
                return true;
            }
        };

        [TestMethod]
        public void TestExecutionCreateNspAndNcaWithoutEncryption()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var metaFile = env.SourceDir + "\\ApplicationMeta\\describe_all.nmeta";
            var metaFileV0 = env.OutputDir + "\\" + Path.GetFileName(metaFile).Replace(".nmeta", "_v0.nmeta");
            var metaFileV1 = env.OutputDir + "\\" + Path.GetFileName(metaFile).Replace(".nmeta", "_v1.nmeta");
            MakeSpecifiedVersionMetaFile(metaFile, metaFileV0, 0);
            MakeSpecifiedVersionMetaFile(metaFile, metaFileV1, 65536);

            var iconPath = env.SourceDir + "\\Icon\\describe_all.bmp";
            var legalInformationZip = MakeLegalInfoZipfile(env.OutputDir);
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");
            MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

            var config = new AuthoringConfiguration();
            config.KeyConfigFilePath = env.DefaultKeyConfigFile;

            Action<string, byte, bool> checkNcaIsNotEncrypted = (ncaFilePath, headerEncryptionType, checkEntries) =>
            {
                using (var fs = new FileStream(ncaFilePath, FileMode.Open, FileAccess.Read))
                {
                    if (headerEncryptionType ==  (byte)NintendoContentArchiveHeaderEncryptionType.Auto)
                    {
                        using (var ncaReader = new NintendoContentArchiveReader(fs, new NcaKeyGenerator(config.GetKeyConfiguration())))
                        {
                            Assert.AreEqual(ncaReader.GetHeaderEncryptionType(), headerEncryptionType);
                        }
                        return;
                    }

                    // 正しい鍵を使えない IKeyGenerator で読み出す
                    using (var ncaReader = new NintendoContentArchiveReader(fs, new NcaKeyGeneratorWithoutEncryption()))
                    {
                        Assert.AreEqual(ncaReader.GetHeaderEncryptionType(), headerEncryptionType);

                        if (!checkEntries)
                        {
                            return;
                        }

                        foreach (var fsInfo in ncaReader.ListFsInfo())
                        {
                            var fsReader = ncaReader.OpenFileSystemArchiveReader(fsInfo.Item1);
                            foreach (var fileInfo in fsReader.ListFileInfo())
                            {
                                var data = fsReader.ReadFile(fileInfo.Item1, 0, fileInfo.Item2);
                                using (var cor = new FileStream(Directory.GetFiles(env.TestCodeDir, fileInfo.Item1).Single(), FileMode.Open, FileAccess.Read))
                                {
                                    Assert.AreEqual(data.Length, cor.Length);
                                    var corData = new byte[cor.Length];
                                    cor.Read(corData, 0, corData.Length);
                                    Assert.IsTrue(data.SequenceEqual(corData));
                                }
                            }
                        }
                    }
                }
            };

            Action<string, byte> checkNspIsNotEncrypted = (nspFilePath, headerEncryptionType) =>
            {
                using (var nspReader = new NintendoSubmissionPackageReader(nspFilePath))
                {
                    foreach (var ncaFile in nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".nca")))
                    {
                        using (var ncaReader = nspReader.OpenNintendoContentArchiveReader(ncaFile.Item1, new NcaKeyGenerator(config.GetKeyConfiguration())))
                        {
                            Assert.AreEqual(ncaReader.GetHeaderEncryptionType(), headerEncryptionType);
                        }
                    }
                }
            };

            Action<string, Tuple<Regex, bool>[], List<int[]>> testExtractAndReplace = (target, testList, testCombination) =>
            {
                // extract
                var extractDir = env.OutputDir + "\\extract";
                var error = ExecuteProgram(string.Format("extract {0} -o {1}", target, extractDir));
                Assert.IsTrue(error == string.Empty, error);

                // replace
                var replaceDir = env.OutputDir + "\\replace";
                var extractDir2 = env.OutputDir + "\\extract2";
                TestReplaceWithNspOrNca(target, replaceDir, extractDir, extractDir2, testList, testCombination);

                SafeDeleteDirectory(extractDir);
                SafeDeleteDirectory(extractDir2);
            };

            // nca
            var nca = env.OutputDir + "\\test.nca";
            {
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                var error = ExecuteProgram(string.Format("createnca -o {0} --meta-type Application --meta {1} --desc {2} --program {3} {3}", nca, metaFile, env.DefaultDescFile, env.TestCodeDir));
                Assert.IsTrue(error == string.Empty, error);
                checkNcaIsNotEncrypted(nca, (byte)NintendoContentArchiveHeaderEncryptionType.Auto, false);
                error = ExecuteProgram(string.Format("createnca -o {0} --meta-type Application --meta {1} --desc {2} --program {3} {3} --no-encryption", nca, metaFile, env.DefaultDescFile, env.TestCodeDir));
                Assert.IsTrue(error == string.Empty, error);
                checkNcaIsNotEncrypted(nca, (byte)NintendoContentArchiveHeaderEncryptionType.None, true);

                Tuple<Regex, bool>[] replaceEntryInNcaTestList = new Tuple<Regex, bool>[] {
                    new Tuple<Regex, bool>(new Regex(@"fs0/code.dat"), true),
                };
                var testCombination = new List<int[]>() { new int[] { 0 } };
                testExtractAndReplace(nca, replaceEntryInNcaTestList, testCombination);
                checkNcaIsNotEncrypted(nca.Replace("test.nca", "\\replace\\test_replaced.nca"), (byte)NintendoContentArchiveHeaderEncryptionType.None, false);
            }

            // nsp
            var nsp = env.OutputDir + "\\test_v0.nsp";
            var nsp2 = env.OutputDir + "\\test_v1.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFileV0, env.DefaultDescFile);
                CreateNsp(nsp, env, metaFileV0, string.Format(" --accessible-urls {0} --legal-information {1} --icon AmericanEnglish {2} Japanese {2}", accessibleUrlsDir, legalInformationZip, iconPath));
                checkNspIsNotEncrypted(nsp, (byte)NintendoContentArchiveHeaderEncryptionType.Auto);
                CreateNsp(nsp, env, metaFileV0, string.Format(" --accessible-urls {0} --legal-information {1} --icon AmericanEnglish {2} Japanese {2} --no-encryption", accessibleUrlsDir, legalInformationZip, iconPath));
                checkNspIsNotEncrypted(nsp, (byte)NintendoContentArchiveHeaderEncryptionType.None);

                Tuple<Regex, bool>[] replaceEntryInNcaTestList = new Tuple<Regex, bool>[] {
                    new Tuple<Regex, bool>(new Regex(@"fs0/code.dat"), true),
                    new Tuple<Regex, bool>(new Regex(@"fs0/control.nacp"), false),
                };
                var testCombination = new List<int[]>() { new int[] { 0, 1 } };
                testExtractAndReplace(nsp, replaceEntryInNcaTestList, testCombination);
                checkNspIsNotEncrypted(nsp.Replace("test_v0.nsp", "\\replace\\test_v0_replaced.nsp"), (byte)NintendoContentArchiveHeaderEncryptionType.None);

                // patch
                var patch = nsp2.Replace(".nsp", "_patch.nsp");
                var error = string.Empty;
                {
                    MakeNpdm(env.NpdmFile, metaFileV1, env.DefaultDescFile);
                    CreateNsp(nsp2, env, metaFileV1, string.Format(" --accessible-urls {0} --legal-information {1} --icon AmericanEnglish {2} Japanese {2} --no-encryption", accessibleUrlsDir, legalInformationZip, iconPath));

                    error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", env.OutputDir, env.DefaultDescFile, nsp, nsp2));
                    Assert.IsTrue(error.IndexOf("The nsp created without encryption is not supported for making patch.") >= 0, error);
                }

                // prodencryption
                {
                    error = ExecuteProgram(string.Format("prodencryption -o {0} --keyconfig {1} --no-check --no-nspu {2}", env.OutputDir, env.DefaultKeyConfigFile, nsp));
                    Assert.IsTrue(error.IndexOf("This nsp cannot be prod-encrypted because it is created without encryption.") >= 0, error);
                    error = ExecuteProgram(string.Format("prodencryption -o {0} --keyconfig {1} --gamecard --no-check --no-padding --no-xcie {2}", env.OutputDir, env.DefaultKeyConfigFile, nsp));
                    Assert.IsTrue(error.IndexOf("This nsp cannot be prod-encrypted because it is created without encryption.") >= 0, error);
                }
            }

            SafeDeleteDirectory(env.OutputDir);
        }

        [TestMethod]
        public void TestExcecutionCreateNacp()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var metaDirectoryPath = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\NacpTest\\Meta";
            var outputPath = Directory.GetCurrentDirectory() + "/test.nacp";

            foreach (var file in Directory.EnumerateFiles(metaDirectoryPath))
            {
                System.Console.WriteLine("Testing {0}...", file);
                var msg = ExecuteProgram(string.Format("createnacp --meta {0} -o {1}", file, outputPath));
                Assert.IsTrue(msg == string.Empty, msg);
                Assert.IsTrue(File.Exists(outputPath));

                File.Delete(outputPath);
            }
        }

        [TestMethod]
        public void TestExecutionReplaceNspNmeta()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var iconPath1 = Path.Combine(env.SourceDir, "Icon\\describe_all.bmp");
            var iconPath2 = Path.Combine(env.SourceDir, "Icon\\no_application.bmp");
            var legalInformationZip1 = Path.Combine(env.SourceDir, "ErrorUnpublishable\\legal_info.zip");
            var legalInformationZip2 = Path.Combine(env.SourceDir, "ErrorUnpublishable\\legal_info_invalid_hash.zip");
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath1 = Path.Combine(accessibleUrlsDir, "accessible-urls1.txt");
            var accessibleUrlsFilePath2 = Path.Combine(accessibleUrlsDir, "accessible-urls2.txt");
            MakeFileImpl(accessibleUrlsFilePath1, "http://test1.com\n");
            MakeFileImpl(accessibleUrlsFilePath2, "^https://test2.com\n");

            Action<string, string> createHtmlDir = (path, data) =>
            {
                SafeDeleteDirectory(path);
                Directory.CreateDirectory(path);
                using (var fs = File.OpenWrite(Path.Combine(path, "nothing.html")))
                {
                    var sw = new StreamWriter(fs);
                    sw.WriteLine(data);
                    sw.Flush();
                }
            };
            string htmlDirPath1 = Path.Combine(env.OutputDir, "HtmlDocumentTestDir1");
            string htmlDirPath2 = Path.Combine(env.OutputDir, "HtmlDocumentTestDir2");
            createHtmlDir(htmlDirPath1, "test1");
            createHtmlDir(htmlDirPath2, "test2");

            var config = new AuthoringConfiguration();
            config.KeyConfigFilePath = env.DefaultKeyConfigFile;

            // 各種情報が空の nmeta
            var metaFileEmpty = env.OutputDir + "\\replacenmeta_empty.nmeta";
            {
                var meta = string.Format(@"<?xml version=""1.0""?>
                        <NintendoSdkMeta>
                          <Core>
                            <ApplicationId>0x0005000C10000001</ApplicationId>
                          </Core>
                          <Application>
                          </Application>
                        </NintendoSdkMeta>", iconPath1, legalInformationZip1, accessibleUrlsFilePath1, htmlDirPath1);
                File.WriteAllText(metaFileEmpty, meta);
            }
            // replacenmeta_empty.nmeta に title などを追加した nmeta
            var metaFileAdd = env.OutputDir + "\\replacenmeta_add.nmeta";
            {
                var meta = string.Format(@"<?xml version=""1.0""?>
                        <NintendoSdkMeta>
                          <Core>
                            <ApplicationId>0x0005000C10000001</ApplicationId>
                          </Core>
                          <Application>
                            <Title>
                              <Language>AmericanEnglish</Language>
                              <Name>AmericanEnglishTitle1</Name>
                              <Publisher>AmericanEnglishPublisher1</Publisher>
                            </Title>
                            <Icon>
                              <Language>AmericanEnglish</Language>
                              <IconPath>{0}</IconPath>
                            </Icon>
                            <DisplayVersion>1.0.0</DisplayVersion>
                            <SupportedLanguage>AmericanEnglish</SupportedLanguage>
                            <Rating>
                                <Organization>ESRB</Organization>
                                <Age>0</Age>
                            </Rating>
                            <UserAccountSaveDataSize>0x0000000000400000</UserAccountSaveDataSize>
                            <UserAccountSaveDataJournalSize>0x0000000000100000</UserAccountSaveDataJournalSize>
                            <LogoType>LicensedByNintendo</LogoType>
                            <ReleaseVersion>0</ReleaseVersion>
                            <LegalInformationFilePath>{1}</LegalInformationFilePath>
                            <AccessibleUrlsFilePath>{2}</AccessibleUrlsFilePath>
                            <HtmlDocumentPath>{3}</HtmlDocumentPath>
                          </Application>
                        </NintendoSdkMeta>", iconPath1, legalInformationZip1, accessibleUrlsFilePath1, htmlDirPath1);
                File.WriteAllText(metaFileAdd, meta);
            }
            // replacenmeta_add.nmeta から一部情報・コンテンツが削除されている nmeta
            var metaFileDelete = env.OutputDir + "\\replacenmeta_delete.nmeta";
            {
                var meta = string.Format(@"<?xml version=""1.0""?>
                        <NintendoSdkMeta>
                          <Core>
                            <ApplicationId>0x0005000C10000001</ApplicationId>
                          </Core>
                          <Application>
                            <Title>
                              <Language>AmericanEnglish</Language>
                              <Name>AmericanEnglishTitle1</Name>
                              <Publisher>AmericanEnglishPublisher1</Publisher>
                            </Title>
                            <DisplayVersion>1.0.0</DisplayVersion>
                            <SupportedLanguage>AmericanEnglish</SupportedLanguage>
                            <LogoType>LicensedByNintendo</LogoType>
                            <ReleaseVersion>0</ReleaseVersion>
                            <LegalInformationFilePath>{1}</LegalInformationFilePath>
                            <HtmlDocumentPath>{2}</HtmlDocumentPath>
                          </Application>
                        </NintendoSdkMeta>", iconPath1, legalInformationZip1, htmlDirPath1);
                File.WriteAllText(metaFileDelete, meta);
            }
            // replacenmeta_add.nmeta から各種情報が変更されている nmeta
            var metaFileModify = env.OutputDir + "\\replacenmeta_modify.nmeta";
            {
                var meta = string.Format(@"<?xml version=""1.0""?>
                        <NintendoSdkMeta>
                          <Core>
                            <ApplicationId>0x0005000C10000001</ApplicationId>
                          </Core>
                          <Application>
                            <Title>
                              <Language>AmericanEnglish</Language>
                              <Name>AmericanEnglishTitle2</Name>
                              <Publisher>AmericanEnglishPublisher2</Publisher>
                            </Title>
                            <Title>
                              <Language>Japanese</Language>
                              <Name>JapaneseTitle</Name>
                              <Publisher>JapanesePublisher</Publisher>
                            </Title>
                            <Icon>
                              <Language>AmericanEnglish</Language>
                              <IconPath>{0}</IconPath>
                            </Icon>
                            <Icon>
                              <Language>Japanese</Language>
                              <IconPath>{0}</IconPath>
                            </Icon>
                            <DisplayVersion>1.0.1</DisplayVersion>
                            <SupportedLanguage>AmericanEnglish</SupportedLanguage>
                            <SupportedLanguage>Japanese</SupportedLanguage>
                            <Rating>
                                <Organization>CERO</Organization>
                                <Age>1</Age>
                            </Rating>
                            <UserAccountSaveDataSize>0x0000000000800000</UserAccountSaveDataSize>
                            <UserAccountSaveDataJournalSize>0x0000000000200000</UserAccountSaveDataJournalSize>
                            <LogoType>Nintendo</LogoType>
                            <ReleaseVersion>1</ReleaseVersion>
                            <LegalInformationFilePath>{1}</LegalInformationFilePath>
                            <AccessibleUrlsFilePath>{2}</AccessibleUrlsFilePath>
                            <HtmlDocumentPath>{3}</HtmlDocumentPath>
                          </Application>
                        </NintendoSdkMeta>", iconPath2, legalInformationZip2, accessibleUrlsFilePath2, htmlDirPath2);
                File.WriteAllText(metaFileModify, meta);
            }

            // creatensp 実行時に nmeta 毎に main.npdm を作成すると code 領域のデータが変わってファイルが一致しなくなるので同じ nmeta から作成した main.npdm を使用する
            MakeNpdm(env.NpdmFile, metaFileEmpty, env.DefaultDescFile);

            // それぞれの nmeta を指定した nsp を作成
            Action<string, string, string, string> createNsp = ( outputPath,  metaFile, keyconfig, addOption) =>
            {
                var keyconfigOption = string.Empty;
                if (keyconfig != null)
                {
                    keyconfigOption = "--keyconfig " + keyconfig;
                }
                var cmd = string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {3} ", outputPath, metaFile, env.DefaultDescFile, env.TestCodeDir) + keyconfigOption + addOption;
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(outputPath, keyconfig);
            };

            // それぞれの nmeta を使用した nsp を作成
            createNsp(metaFileEmpty + ".nsp",  metaFileEmpty, null, "--ignore-unpublishable-error");
            createNsp(metaFileAdd + ".nsp", metaFileAdd, null, "--ignore-unpublishable-error");
            createNsp(metaFileDelete + ".nsp", metaFileDelete, null, "--ignore-unpublishable-error");
            createNsp(metaFileModify + ".nsp",  metaFileModify, null, "--ignore-unpublishable-error");

            Action<string, string, string, string, string> replaceNspNmeta = (nspFile, outdir, metaFile, keyconfig, addOption) =>
            {
                var keyconfigOption = string.Empty;
                if (keyconfig != null)
                {
                    keyconfigOption = "--keyconfig " + keyconfig;
                }
                var cmd = string.Format("replacenspmeta {0} -o {1} --meta {2} --desc {3} ", nspFile, outdir, metaFile, env.DefaultDescFile) + keyconfigOption + addOption;
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
            };

            var outputDir = env.OutputDir;
            // replacenspmeta で各種情報・コンテンツを追加
            // 下記の方法で作成した nsp が一致することを確認
            //  1) replacenmeta_empty.nmeta で作成した nsp を replacenmeta_add.nmeta で replacenspmeta した nsp
            //  2) replacenmeta_add.nmeta で作成した nsp
            replaceNspNmeta(metaFileEmpty + ".nsp", outputDir, metaFileAdd, null, "--ignore-unpublishable-error");
            CheckFileDiff(metaFileEmpty + "_replaced.nsp", metaFileAdd + ".nsp", 0);

            // replacenspmeta で全情報・コンテンツを削除
            // 下記の方法で作成した nsp が一致することを確認
            //  1) replacenmeta_add.nmeta で作成した nsp を replacenmeta_empty.nmeta で replacenspmeta した nsp
            //  2) replacenmeta_empty.nmeta で作成した nsp
            replaceNspNmeta(metaFileAdd + ".nsp", outputDir, metaFileEmpty, null, "--ignore-unpublishable-error");
            CheckFileDiff(metaFileAdd + "_replaced.nsp", metaFileEmpty + ".nsp", 0);

            // replacenspmeta で一部情報・コンテンツを削除
            // 下記の方法で作成した nsp が一致することを確認
            //  1) replacenmeta_add.nmeta で作成した nsp を replacenmeta_delete.nmeta で replacenspmeta した nsp
            //  2) replacenmeta_delete.nmeta で作成した nsp
            replaceNspNmeta(metaFileAdd + ".nsp", outputDir, metaFileDelete, null, "--ignore-unpublishable-error");
            CheckFileDiff(metaFileAdd + "_replaced.nsp", metaFileDelete + ".nsp", 0);

            // replacenspmeta で各種情報・コンテンツを変更
            // 下記の方法で作成した nsp が一致することを確認
            //  1) replacenmeta_add.nmeta で作成した nsp を replacenmeta_modify.nmeta で replacenspmeta した nsp
            //  2) replacenmeta_modify.nmeta で作成した nsp
            replaceNspNmeta(metaFileAdd + ".nsp", outputDir, metaFileModify, null, "--ignore-unpublishable-error");
            CheckFileDiff(metaFileAdd + "_replaced.nsp", metaFileModify + ".nsp", 0);

            // cardspec が変更されないことを確認
            Action<string, string> addCardSpecElement = (metaFile, element) =>
            {
                var text = File.ReadAllText(metaFile);
                text = text.Replace(@"</NintendoSdkMeta>", element + @"</NintendoSdkMeta>");
                File.WriteAllText(metaFile, text);
            };
            // cardspec 要素を追加
            {
                var cardspec = @"<CardSpec>
                                    <Size>2</Size>
                                    <ClockRate>25</ClockRate>
                                    <AutoSetSize>false</AutoSetSize>
                                    <AutoSetClockRate>false</AutoSetClockRate>
                                </CardSpec>";
                addCardSpecElement(metaFileEmpty, cardspec);
            }
            {
                var cardspec = @"<CardSpec>
                                    <Size>8</Size>
                                    <ClockRate>50</ClockRate>
                                    <AutoSetSize>true</AutoSetSize>
                                    <AutoSetClockRate>true</AutoSetClockRate>
                                </CardSpec>";
                addCardSpecElement(metaFileAdd, cardspec);
            }

            createNsp(metaFileEmpty + ".nsp", metaFileEmpty, null, "--ignore-unpublishable-error");
            replaceNspNmeta(metaFileEmpty + ".nsp", outputDir, metaFileAdd, null, "--ignore-unpublishable-error");

            Func<string, byte[]> getCardSpecData = delegate (string nspFile)
            {
                using (var nspReader = new NintendoSubmissionPackageReader(nspFile))
                {
                    var cardSpecXml = nspReader.ListFileInfo().Where(x => x.Item1 == "cardspec.xml").Single();
                    return nspReader.ReadFile(cardSpecXml.Item1, 0, cardSpecXml.Item2);
                }
            };

            // replacenspmeta 実行前と実行後の cardspec.xml が一致することを確認
            var data1 = getCardSpecData(metaFileEmpty + ".nsp");
            var data2 = getCardSpecData(metaFileEmpty + "_replaced.nsp");
            Assert.IsTrue(data1.SequenceEqual(data2));

            // keyconfig が指定できることを確認
            outputDir = Path.Combine(env.OutputDir, "tmp");
            Directory.CreateDirectory(outputDir);
            var keyconfigPath = (new TestUtility.TestPath(this.TestContext)).GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";
            createNsp(metaFileEmpty + ".nsp", metaFileEmpty, keyconfigPath, " --keyindex 2 --ignore-unpublishable-error");
            replaceNspNmeta(metaFileEmpty + ".nsp", outputDir, metaFileAdd, keyconfigPath, " --keyindex 2 --ignore-unpublishable-error");
            // 出力ディレクトリ以下の中間ファイルが削除されて置換後の nsp のみ作成されていることを確認
            Assert.IsTrue(File.Exists(Path.Combine(outputDir, metaFileEmpty + "_replaced.nsp")));
            Assert.IsTrue(Directory.GetFiles(outputDir).Count() == 1);

            SafeDeleteDirectory(env.OutputDir);
        }
    }
}
