﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;

namespace AuthoringToolsTest
{
    [TestClass]
    public class SourceTest
    {
        [TestMethod]
        public void FileSourceTest()
        {
        }

        [TestMethod]
        public void MemorySourceTest()
        {
        }

        [TestMethod]
        public void StreamSourceTest()
        {
            int bufferSize = 1024;
            byte[] buffer = Utils.GetTestBuffer(bufferSize);
            StreamSource source = new StreamSource(new MemoryStream(buffer), 0, bufferSize);
            ByteData data;
            data = source.PullData(0, bufferSize);
            Assert.IsTrue(data.Buffer.Count == bufferSize);
            Utils.CheckBufferEquality(data.Buffer.Array, 0, buffer, 0, bufferSize);
            data = source.PullData(512, bufferSize);
            Assert.IsTrue(data.Buffer.Count == 512);
            Utils.CheckBufferEquality(data.Buffer.Array, 0, buffer, 512, 512);
            data = source.PullData(bufferSize, 1);
            Assert.IsTrue(data.Buffer.Count == 0);
        }

        [TestMethod]
        public void PaddingSourceTest()
        {
        }

        [TestMethod]
        public void ConcatenatedSourceTest()
        {
            List<ConcatenatedSource.Element> elements = new List<ConcatenatedSource.Element>();
            int bufferSize1 = 1024;
            int bufferSize2 = 16;
            int paddingSize = 128;
            int bufferSize3 = 256;
            {
                ConcatenatedSource.Element element = new ConcatenatedSource.Element(
                    new MemorySource(Utils.GetTestBuffer(bufferSize1), 0, bufferSize1), "mem1", 0);
                elements.Add(element);
            }
            {
                ConcatenatedSource.Element element = new ConcatenatedSource.Element(
                    new MemorySource(Utils.GetTestBuffer(bufferSize2), 0, bufferSize2), "mem2", bufferSize1);
                elements.Add(element);
            }
            {
                // 0 バイト挿入
                ConcatenatedSource.Element element = new ConcatenatedSource.Element(
                    new PaddingSource(0), "zero1", bufferSize1 + bufferSize2 + paddingSize);
                elements.Add(element);
            }
            {
                // 0 バイト挿入
                ConcatenatedSource.Element element = new ConcatenatedSource.Element(
                    new PaddingSource(0), "zero2", bufferSize1 + bufferSize2 + paddingSize);
                elements.Add(element);
            }
            {
                ConcatenatedSource.Element element = new ConcatenatedSource.Element(
                    new MemorySource(Utils.GetTestBuffer(bufferSize3), 0, bufferSize3), "mem3", bufferSize1 + bufferSize2 + paddingSize);
                elements.Add(element);
            }
            ConcatenatedSource source = new ConcatenatedSource(elements);
            // 順番に読み出し
            {
                ByteData data = source.PullData(0, bufferSize1);
                Assert.IsTrue(data.Buffer.Count == bufferSize1);
                Utils.CheckBufferEquality(Utils.GetTestBuffer(bufferSize1), 0, data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count);
            }
            {
                ByteData data = source.PullData(bufferSize1, bufferSize2);
                Assert.IsTrue(data.Buffer.Count == bufferSize2);
                Utils.CheckBufferEquality(Utils.GetTestBuffer(bufferSize2), 0, data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count);
            }
            {
                ByteData data = source.PullData(bufferSize1 + bufferSize2, paddingSize);
                Assert.IsTrue(data.Buffer.Count == paddingSize);
                Utils.CheckBufferEquality(new byte[paddingSize], 0, data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count);
            }
            {
                ByteData data = source.PullData(bufferSize1 + bufferSize2 + paddingSize, bufferSize3);
                Assert.IsTrue(data.Buffer.Count == bufferSize3);
                Utils.CheckBufferEquality(Utils.GetTestBuffer(bufferSize3), 0, data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count);
            }
            // 順番に途中から読み出し
            {
                ByteData data = source.PullData(0 + 8, bufferSize1);
                Assert.IsTrue(data.Buffer.Count == bufferSize1);
                Utils.CheckBufferEquality(Utils.GetTestBuffer(bufferSize1), 8, data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count - 8);
                Utils.CheckBufferEquality(Utils.GetTestBuffer(bufferSize2), 0, data.Buffer.Array, data.Buffer.Offset + bufferSize1 - 8, 8);
            }
            {
                ByteData data = source.PullData(bufferSize1 + 8, bufferSize2);
                Assert.IsTrue(data.Buffer.Count == bufferSize2);
                Utils.CheckBufferEquality(Utils.GetTestBuffer(bufferSize2), 8, data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count - 8);
                Utils.CheckBufferEquality(new byte[paddingSize], 0, data.Buffer.Array, data.Buffer.Offset + bufferSize2 - 8, 8);
            }
            {
                ByteData data = source.PullData(bufferSize1 + bufferSize2 + 8, paddingSize);
                Assert.IsTrue(data.Buffer.Count == paddingSize);
                Utils.CheckBufferEquality(new byte[paddingSize], 8, data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count - 8);
                Utils.CheckBufferEquality(Utils.GetTestBuffer(bufferSize3), 0, data.Buffer.Array, data.Buffer.Offset + paddingSize - 8, 8);
            }
            {
                ByteData data = source.PullData(bufferSize1 + bufferSize2 + paddingSize + 8, bufferSize3);
                Assert.IsTrue(data.Buffer.Count == bufferSize3 - 8);
                Utils.CheckBufferEquality(Utils.GetTestBuffer(bufferSize3), 8, data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count);
            }
        }

        [TestMethod]
        public void SinkLinkedSourceTest()
        {
        }

        public class PartiallyAvailableSource : ISource
        {
            public long Size { get; private set; }
            private SourceStatus m_status;

            public PartiallyAvailableSource(long size, Range availableRange)
            {
                Size = size;
                m_status = new SourceStatus();
                m_status.AvailableRangeList.MergingAdd(availableRange);
            }
            public ByteData PullData(long offset, int size)
            {
                return new ByteData(new ArraySegment<byte>(new byte[size], 0, size));
            }
            public SourceStatus QueryStatus()
            {
                return m_status;
            }
        }

        [TestMethod]
        public void AlignedSourceTest()
        {
            const int AlignmentSize = 512;
            // offset: aligned, size: aligned
            {
                Range range = new Range(0, 1024);
                PartiallyAvailableSource ps = new PartiallyAvailableSource(1024, range);
                AlignedSource source = new AlignedSource(ps, AlignmentSize);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList.Count == 1);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Offset == 0);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Size == 1024);
            }
            // offset: aligned, size: not aligned ( < AlignmentSize)
            {
                Range range = new Range(0, 511);
                PartiallyAvailableSource ps = new PartiallyAvailableSource(1024, range);
                AlignedSource source = new AlignedSource(ps, AlignmentSize);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList.Count == 0);
            }
            // offset: aligned, size: not aligned (size >= AlignmentSize)
            {
                Range range = new Range(0, 513);
                PartiallyAvailableSource ps = new PartiallyAvailableSource(1024, range);
                AlignedSource source = new AlignedSource(ps, AlignmentSize);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList.Count == 1);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Offset == 0);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Size == 512);
                ByteData data = source.PullData(0, 513);
                Assert.AreEqual(data.Buffer.Count, 512);
            }
            // offset: aligned, size: not aligned ( tail )
            {
                Range range = new Range(0, 1023);
                PartiallyAvailableSource ps = new PartiallyAvailableSource(1023, range);
                AlignedSource source = new AlignedSource(ps, AlignmentSize);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList.Count == 1);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Offset == 0);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Size == 1023);
                ByteData data;
                data = source.PullData(0, 512);
                data = source.PullData(0, 1023);
            }
            // offset: not aligned, size: aligned (size < AlignmentSize)
            {
                Range range = new Range(1, 511);
                PartiallyAvailableSource ps = new PartiallyAvailableSource(1024, range);
                AlignedSource source = new AlignedSource(ps, AlignmentSize);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList.Count == 0);
            }
            // offset: not aligned, size: aligned (size >= AlignmentSize)
            {
                Range range = new Range(1, 1024);
                PartiallyAvailableSource ps = new PartiallyAvailableSource(1024, range);
                AlignedSource source = new AlignedSource(ps, AlignmentSize);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList.Count == 1);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Offset == 512);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Size == 512);
                ByteData data = source.PullData(512, 512);
            }
            // offset: not aligned, size: aligned ( tail )
            {
                Range range = new Range(1, 1024);
                PartiallyAvailableSource ps = new PartiallyAvailableSource(1024, range);
                AlignedSource source = new AlignedSource(ps, AlignmentSize);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList.Count == 1);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Offset == 512);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Size == 512);
                ByteData data = source.PullData(512, 512);
            }
            // offset: not aligned, size: not aligned (size < AlignmentSize)
            {
                Range range = new Range(1, 512);
                PartiallyAvailableSource ps = new PartiallyAvailableSource(1024, range);
                AlignedSource source = new AlignedSource(ps, AlignmentSize);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList.Count == 0);
            }
            // offset: not aligned, size: not aligned (size >= AlignmentSize)
            {
                Range range = new Range(1, 1025);
                PartiallyAvailableSource ps = new PartiallyAvailableSource(1536, range);
                AlignedSource source = new AlignedSource(ps, AlignmentSize);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList.Count == 1);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Offset == 512);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Size == 512);
            }
            // offset: not aligned, size: not aligned ( tail )
            {
                Range range = new Range(1, 1025);
                PartiallyAvailableSource ps = new PartiallyAvailableSource(1025, range);
                AlignedSource source = new AlignedSource(ps, AlignmentSize);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList.Count == 1);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Offset == 512);
                Assert.IsTrue(source.QueryStatus().AvailableRangeList[0].Size == 512);
                ByteData data = source.PullData(512, 512);
                data = source.PullData(512, 513);
            }
            // call exception 1
            {
                Range range = new Range(0, 1024);
                PartiallyAvailableSource ps = new PartiallyAvailableSource(1024, range);
                AlignedSource source = new AlignedSource(ps, AlignmentSize);
                Utils.CheckReturnException(new ArgumentException(), () =>
                        {
                            ByteData data = source.PullData(1, 512);
                            return true;
                        });
                Utils.CheckReturnException(new ArgumentException(), () =>
                        {
                            ByteData data = source.PullData(1, 1024);
                            return true;
                        });
            }
        }

        private void InitializeCounter(byte[] ctr, ulong value)
        {
            // big endian
            int i, j;
            if (ctr.Length < 8)
            {
                throw new ArgumentException();
            }
            for (i = ctr.Length - 1, j = 0; j < 8; --i, j++)
            {
                ctr[i] = (byte)((value >> j * 8) & 0xFF);
            }
        }
        private void UpdateCounter(byte[] ctr)
        {
            for (int i = ctr.Length - 1; i >= 0; --i)
            {
                if (++ctr[i] != 0)
                {
                    break;
                }
            }
        }

//        [TestMethod]
        public void Aes128CtrEncryptedSourceTest()
        {
        }

//        [TestMethod]
        public void Aes128XtsEncryptedSourceTest()
        {
        }

        [TestMethod]
        public void Sha256HierarchicalHashCalculatedSourceTest()
        {
#if false
            // layer: 3, BlockSize: 4 KB
            {
                List<int> dataSizeList = new List<int> {
                     512 * 1024,  512 * 1024 - 1,  512 * 1024 + 1,
                    1024 * 1024, 1024 * 1024 - 1, 1024 * 1024 + 1,
                    1536 * 1024, 1536 * 1024 - 1, 1536 * 1024 + 1,
                };
                foreach (var dataSize in dataSizeList)
                {
                    byte[] src = new byte[dataSize];
                    SHA256CryptoServiceProvider hashCalculator = new SHA256CryptoServiceProvider();
                    const int BlockSize = 4 * 1024;

                    // L:2
                    byte[] dstL2 = new byte[((src.Length + BlockSize - 1) / BlockSize) * 32];
                    {
                        int offset = 0;
                        for (int i = 0; i < src.Length; i += BlockSize)
                        {
                            int hashTargetSize = Math.Min(src.Length - i, BlockSize);
                            byte[] hash = hashCalculator.ComputeHash(src, i, hashTargetSize);
                            Buffer.BlockCopy(hash, 0, dstL2, offset, 32);
                            offset += 32;
                        }
                    }

                    // L:1
                    byte[] dstL1 = new byte[((dstL2.Length + BlockSize - 1) / BlockSize) * 32];
                    {
                        int offset = 0;
                        for (int i = 0; i < dstL2.Length; i += BlockSize)
                        {
                            int hashTargetSize = Math.Min(dstL2.Length - i, BlockSize);
                            byte[] hash = hashCalculator.ComputeHash(dstL2, i, hashTargetSize);
                            Buffer.BlockCopy(hash, 0, dstL1, offset, 32);
                            offset += 32;
                        }
                    }

                    // L:0
                    byte[] masterHash = hashCalculator.ComputeHash(dstL1, 0, dstL1.Length);

                    Sha256HierarchicalHashCalculatedSource source = new Sha256HierarchicalHashCalculatedSource(new MemorySource(src, 0, src.Length), BlockSize, 3);
                    ISource hashSource = source.GetHashValueSource();
                    ISource masterHashSource = source.GetMasterHashValueSource();
                    ByteData data = source.PullData(0, (int)source.Size);

                    Assert.AreEqual(hashSource.Size, dstL1.Length + dstL2.Length);
                    Assert.AreEqual(masterHashSource.Size, 32);

                    ByteData hashData = hashSource.PullData(0, (int)hashSource.Size);
                    ByteData masterHashData = masterHashSource.PullData(0, (int)masterHashSource.Size);

                    for (int i = 0; i < hashData.Buffer.Count; i++)
                    {
                        if (i < dstL1.Length)
                        {
                            Assert.AreEqual(dstL1[i], hashData.Buffer.Array[i]);
                        }
                        else
                        {
                            Assert.AreEqual(dstL2[i - dstL1.Length], hashData.Buffer.Array[i]);
                        }
                    }

                    for (int i = 0; i < masterHash.Length; i++)
                    {
                        Assert.AreEqual(masterHash[i], masterHashData.Buffer.Array[i]);
                    }
                }
            }
#endif
            // layer: 2, BlockSize: 4 KB
            {
                int hashBlockSizeMin = 4 * 1024;
                int hashBlockSizeMax = 16 * 1024;
                for (int hashBlockSize = hashBlockSizeMin; hashBlockSize <= hashBlockSizeMax; hashBlockSize *= 2)
                {
                    int BlockSize = hashBlockSize;
                    List<int> dataSizeList = new List<int> {
                        1 * BlockSize,  1 * BlockSize + 1, 1 * BlockSize - 1,
                        2 * BlockSize,  2 * BlockSize + 1, 2 * BlockSize - 1,
                        ((BlockSize - 1) / 32) * BlockSize, ((BlockSize - 1) / 32) * BlockSize - 1,
                        (BlockSize / 32) * BlockSize, (BlockSize / 32) * BlockSize - 1
                    };
                    foreach (var dataSize in dataSizeList)
                    {
                        byte[] src = new byte[dataSize];
                        SHA256CryptoServiceProvider hashCalculator = new SHA256CryptoServiceProvider();

                        byte[] dst = new byte[((src.Length + BlockSize - 1) / BlockSize) * 32];
                        int offset = 0;

                        for (int i = 0; i < src.Length; i += BlockSize)
                        {
                            int hashTargetSize = Math.Min(src.Length - i, BlockSize);
                            byte[] hash = hashCalculator.ComputeHash(src, i, hashTargetSize);
                            Buffer.BlockCopy(hash, 0, dst, offset, 32);
                            offset += 32;
                        }

                        byte[] masterHash = hashCalculator.ComputeHash(dst, 0, dst.Length);

                        Sha256HierarchicalHashCalculatedSource source = new Sha256HierarchicalHashCalculatedSource(new MemorySource(src, 0, src.Length), BlockSize, 2);
                        ISource hashSource = source.GetLayerHashSource();
                        ISource masterHashSource = source.GetMasterHashSource();
                        ByteData data = source.PullData(0, (int)source.Size);

                        Assert.AreEqual(hashSource.Size, dst.Length);
                        Assert.AreEqual(masterHashSource.Size, 32);

                        ByteData hashData = hashSource.PullData(0, (int)hashSource.Size);
                        ByteData masterHashData = masterHashSource.PullData(0, (int)masterHashSource.Size);

                        for (int i = 0; i < hashData.Buffer.Count; i++)
                        {
                            Assert.AreEqual(dst[i], hashData.Buffer.Array[i]);
                        }

                        for (int i = 0; i < masterHash.Length; i++)
                        {
                            Assert.AreEqual(masterHash[i], masterHashData.Buffer.Array[i]);
                        }
                    }
                }
            }
            // layer: 1, BlockSize: source size
            {
                byte[] src = new byte[4 * 1024 * 1024 + 1024]; // 4 MB + 1 KB
                SHA256CryptoServiceProvider hashCalculator = new SHA256CryptoServiceProvider();
                byte[] hash = hashCalculator.ComputeHash(src, 0, src.Length);

                Sha256HierarchicalHashCalculatedSource source = new Sha256HierarchicalHashCalculatedSource(new MemorySource(src, 0, src.Length), src.Length, 1);
                ISource hashSource = source.GetLayerHashSource();
                ByteData data = source.PullData(0, (int)source.Size);

                Assert.AreEqual(hashSource.Size, 32);

                ByteData hashData = hashSource.PullData(0, (int)hashSource.Size);

                for (int i = 0; i < hashData.Buffer.Count; i++)
                {
                    Assert.AreEqual(hash[i], hashData.Buffer.Array[i]);
                }
            }
#if false // ハッシュ結果確認用
            {
                SHA256CryptoServiceProvider hashCalculator = new SHA256CryptoServiceProvider();

                string hexValues = "50 46 53 30 01 00 00 00 18 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 04 00 00 00 00 00 00 00 00 00 00 00 00 00 00 64 61 74 61 2E 64 61 74 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00";
                string[] hexValuesSplit = hexValues.Split(' ');
                byte[] src = new byte[hexValuesSplit.Length + 1024];

                int index = 0;
                foreach (String hex in hexValuesSplit)
                {
                    // Convert the number expressed in base-16 to an integer.
                    int value = Convert.ToInt32(hex, 16);
                    src[index++] = (byte)value;
                }
                for (int i = 0; i < 1024; i++)
                {
                    src[i + index] = (byte)i;
                }

                byte[] hash = hashCalculator.ComputeHash(src, 0, src.Length);
                byte[] masterHash = hashCalculator.ComputeHash(hash, 0, hash.Length);
            }
            {
                SHA256CryptoServiceProvider hashCalculator = new SHA256CryptoServiceProvider();
                byte [] hash;
                using (System.IO.FileStream fs = new System.IO.FileStream("C:/Windows/Temp/cygwin.c0.nca", System.IO.FileMode.Open,
                       System.IO.FileAccess.Read, System.IO.FileShare.None, 4096, System.IO.FileOptions.SequentialScan))
                {
                    hash = hashCalculator.ComputeHash(fs);
                }
            }
#endif
        }

        [TestMethod]
        public void Rsa2048SignedSourceTest()
        {
        }

        [TestMethod]
        public void AdaptedSourceTest()
        {
            // PullData1
            {
                byte[] src = new byte[1024];
                MemorySource source = new MemorySource(src, 0, 1024);
                byte[] buffer = new byte[256];
                for (int i = 0; i < 256; i++)
                {
                    buffer[i] = (byte)i;
                }
                MemorySource adapt = new MemorySource(buffer, 0, 256);
                AdaptedSource adapted = new AdaptedSource(source, adapt, 128, 256);

                {
                    ByteData data = adapted.PullData(0, 1024);
                    for (int i = 0; i < 1024; i++)
                    {
                        if (i >= 128 && i < 384)
                        {
                            Assert.AreEqual(data.Buffer.Array[i + data.Buffer.Offset], i - 128);
                        }
                        else
                        {
                            Assert.AreEqual(data.Buffer.Array[i + data.Buffer.Offset], 0);
                        }
                    }
                }
                {
                    ByteData data = adapted.PullData(0, 256);
                    for (int i = 0; i < 256; i++)
                    {
                        if (i >= 128 && i < 256)
                        {
                            Assert.AreEqual(data.Buffer.Array[i + data.Buffer.Offset], i - 128);
                        }
                        else
                        {
                            Assert.AreEqual(data.Buffer.Array[i + data.Buffer.Offset], 0);
                        }
                    }
                }
                {
                    ByteData data = adapted.PullData(256, 768);
                    for (int i = 0; i < 768; i++)
                    {
                        if (i < 128)
                        {
                            Assert.AreEqual(data.Buffer.Array[i + data.Buffer.Offset], i + 128);
                        }
                        else
                        {
                            Assert.AreEqual(data.Buffer.Array[i + data.Buffer.Offset], 0);
                        }
                    }
                }
                {
                    ByteData data = adapted.PullData(256, 128);
                    for (int i = 0; i < 128; i++)
                    {
                        Assert.AreEqual(data.Buffer.Array[i + data.Buffer.Offset], i + 128);
                    }
                }
                {
                    ByteData data = adapted.PullData(0, 128);
                    for (int i = 0; i < 128; i++)
                    {
                        Assert.AreEqual(data.Buffer.Array[i + data.Buffer.Offset], 0);
                    }
                }
                {
                    ByteData data = adapted.PullData(384, 640);
                    for (int i = 0; i < 256; i++)
                    {
                        Assert.AreEqual(data.Buffer.Array[i + data.Buffer.Offset], 0);
                    }
                }
                // 元のソースは変わらない
                for (int i = 0; i < src.Length; i++)
                {
                    Assert.AreEqual(src[i], 0);
                }
                for (int i = 0; i < buffer.Length; i++)
                {
                    Assert.AreEqual(buffer[i], i);
                }
            }
            // QueryStatus
            {
                MemorySource source = new MemorySource(new byte[1024], 0, 1024);
                PartiallyAvailableSource adapt = new PartiallyAvailableSource(512, new Range(0, 256));
                AdaptedSource adapted = new AdaptedSource(source, adapt, 256, 512);
                Assert.AreEqual(adapted.QueryStatus().AvailableRangeList.Count, 0);
            }
            // call exception
            {
                MemorySource source = new MemorySource(new byte[1024], 0, 1024);
                MemorySource adapt = new MemorySource(new byte[256], 0, 256);
                Utils.CheckReturnException(new ArgumentException(), () =>
                        {
                            AdaptedSource adapted = new AdaptedSource(source, adapt, 1024, 256);
                            return true;
                        });
                Utils.CheckReturnException(new ArgumentException(), () =>
                        {
                            AdaptedSource adapted = new AdaptedSource(source, adapt, 512, 512);
                            return true;
                        });
                Utils.CheckReturnException(new ArgumentException(), () =>
                        {
                            AdaptedSource adapted = new AdaptedSource(source, adapt, 768 + 1, 256);
                            return true;
                        });
            }
        }

        [TestMethod]
        public void SubSourceTest()
        {
            byte[] buffer = Utils.GetTestBuffer(1024);
            MemorySource source = new MemorySource(buffer, 0, 1024);
            {
                SubSource ss = new SubSource(source, 128, 512);
                ByteData data = ss.PullData(0, 1024);
                Assert.IsTrue(data.Buffer.Count == 512);
                for (int i = 0; i < data.Buffer.Count; i++)
                {
                    Assert.IsTrue(data.Buffer.Array[i + data.Buffer.Offset] == buffer[i + 128]);
                }
            }
            {
                SubSource ss = new SubSource(source, 512, 1024);
                ByteData data = ss.PullData(0, 1024);
                Assert.IsTrue(data.Buffer.Count == 512);
                for (int i = 0; i < data.Buffer.Count; i++)
                {
                    Assert.IsTrue(data.Buffer.Array[i + data.Buffer.Offset] == buffer[i + 512]);
                }
            }
        }

        [TestMethod]
        public void ReadOnlySourceReaderTest()
        {
            var seed = Utils.GenerateRandomSeed();
            Utils.WriteLine("random seed is " + seed + ".");
            var random = new Random(seed);

            // ブロックサイズ 1 で基本的な読み出し
            {
                // オフセット付きのバイト列を使うために MemorySource とする
                var data = new byte[1024];
                random.NextBytes(data);
                var source = new MemorySource(data, 100, 903);

                // テスト用のインスタンスを生成
                var reader = new IndirectStorageReadOnlySource(source, 1);
                var memory = Marshal.AllocHGlobal(1024);

                // 読み出し
                {
                    var size = reader.Read(memory, 103, 94);
                    Assert.AreEqual(94, size);

                    Marshal.WriteByte(memory, 94, 0xFE);

                    var check = new int[] { 0, 1, 93 };
                    foreach (var i in check)
                    {
                        Assert.AreEqual(data[100 + 103 + i], Marshal.ReadByte(memory, i));
                    }

                    Assert.AreEqual(0xFE, Marshal.ReadByte(memory, 94));
                }

                // 末尾からの読み出し
                {
                    var size = reader.Read(memory, 902, 4);
                    Assert.AreEqual(1, size);
                }

                // 後始末
                Marshal.FreeHGlobal(memory);
            }
        }

#if false
        [TestMethod]
        public void PatchSourceTest()
        {
            string oldFilePath = @"E:\projects\SDK\IndirectStorageBuilder\zero.bin";
            string newFilePath = @"E:\projects\SDK\IndirectStorageBuilder\zero2.bin";
            long oldFileSize = (new FileInfo( oldFilePath )).Length;
            long newFileSize = (new FileInfo( newFilePath )).Length;

            FileSource oldSource = new FileSource( oldFilePath, 0, oldFileSize );
            FileSource newSource = new FileSource( newFilePath, 0, newFileSize );

            PatchSource source = new PatchSource();
            source.Build( oldSource, newSource );
        }
#endif
    }
}
