﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;

namespace AuthoringToolsTest.ContentArchiveLibraryTest
{
    [TestClass]
    public class DeltaTest
    {
        static readonly int CommandSizeMin = Nintendo.Authoring.FileSystemMetaLibrary.DeltaMeta.GetWriteCommandSize(0, 0);

        [TestMethod]
        public void TestSameSize()
        {
            var sourceData = new byte[]
            {
                0x55, 0xAA, 0x55, 0xAA,
                0xCD, 0xCD, 0xCD, 0xCD,
                0x55, 0xAA, 0x55, 0xAA,
            };
            var destinationData = new byte[]
            {
                0x55, 0xAA, 0x55, 0xAA,
                0xFE, 0xFE, 0xFE, 0xFE,
                0x55, 0xAA, 0x55, 0xAA,
            };

            var source = new MemorySource(sourceData, 0, sourceData.Length);
            var destination = new MemorySource(destinationData, 0, destinationData.Length);
            var info = DeltaInfo.ExtractDeltaCommands(source, destination, 1024, 4, null);
            Assert.AreEqual(1, info.Count);
            Assert.AreEqual(4, info[0].Offset);
            Assert.AreEqual(4, info[0].Size);
        }

        [TestMethod]
        public void TestNewData()
        {
            {
                var sourceData = new byte[]
                {
                    0x55, 0xAA, 0x55, 0xAA,
                };
                var destinationData = new byte[]
                {
                    0x55, 0xAA, 0x55, 0xAA,
                    0xFE, 0xFE, 0xFE, 0xFE,
                };

                var source = new MemorySource(sourceData, 0, sourceData.Length);
                var destination = new MemorySource(destinationData, 0, destinationData.Length);
                var info = DeltaInfo.ExtractDeltaCommands(source, destination, 1024, 4, null);
                Assert.AreEqual(1, info.Count);
                Assert.AreEqual(4, info[0].Offset);
                Assert.AreEqual(4, info[0].Size);
            }

            {
                var sourceData = new byte[]
                {
                    0x55, 0xAA, 0x55, 0xAA,
                };
                var destinationData = new byte[]
                {
                    0xFE, 0xFE, 0xFE, 0xFE,
                    0x55, 0xAA, 0x55, 0xAA,
                };

                var source = new MemorySource(sourceData, 0, sourceData.Length);
                var destination = new MemorySource(destinationData, 0, destinationData.Length);
                var info = DeltaInfo.ExtractDeltaCommands(source, destination, 1024, 4, null);
                Assert.AreEqual(1, info.Count);
                Assert.AreEqual(0, info[0].Offset);
                Assert.AreEqual(8, info[0].Size);
            }
        }

        [TestMethod]
        public void TestDeleteData()
        {
            {
                var sourceData = new byte[]
                {
                    0x55, 0xAA, 0x55, 0xAA,
                    0xFE, 0xFE, 0xFE, 0xFE,
                };
                var destinationData = new byte[]
                {
                    0x55, 0xAA, 0x55, 0xAA,
                };

                var source = new MemorySource(sourceData, 0, sourceData.Length);
                var destination = new MemorySource(destinationData, 0, destinationData.Length);
                var info = DeltaInfo.ExtractDeltaCommands(source, destination, 1024, 4, null);
                Assert.AreEqual(0, info.Count);
            }

            {
                var sourceData = new byte[]
                {
                    0xFE, 0xFE, 0xFE, 0xFE,
                    0xFE, 0xFE, 0xFE, 0xFE,
                };
                var destinationData = new byte[]
                {
                    0x55, 0xAA, 0x55, 0xAA,
                };

                var source = new MemorySource(sourceData, 0, sourceData.Length);
                var destination = new MemorySource(destinationData, 0, destinationData.Length);
                var info = DeltaInfo.ExtractDeltaCommands(source, destination, 1024, 4, null);
                Assert.AreEqual(1, info.Count);
                Assert.AreEqual(0, info[0].Offset);
                Assert.AreEqual(4, info[0].Size);
            }
        }

        [TestMethod]
        public void TestSplitBlockSize()
        {
            var previosData = new byte[]
            {
                0x55, 0xAA, 0x55, 0xAA,
                0xCD, 0xCD, 0xCD, 0xCD,
                0xCD, 0xCD, 0xCD, 0xCD,
                0x55, 0xAA, 0x55, 0xAA,
            };
            var currentData = new byte[]
            {
                0x55, 0xAA, 0x55, 0xAA,
                0xFE, 0xFE, 0xFE, 0xFE,
                0xFE, 0xFE, 0xFE, 0xFE,
                0x55, 0xAA, 0x55, 0xAA,
                0xFE, 0xFE, 0xFE, 0xFE,
                0xFE, 0xFE, 0xFE, 0xFE,
            };

            var source = new MemorySource(previosData, 0, previosData.Length);
            var destination = new MemorySource(currentData, 0, currentData.Length);

            {
                var info = DeltaInfo.ExtractDeltaCommands(source, destination, 1024, 4, null);
                Assert.AreEqual(2, info.Count);
                Assert.AreEqual(4, info[0].Offset);
                Assert.AreEqual(8, info[0].Size);
                Assert.AreEqual(16, info[1].Offset);
                Assert.AreEqual(8, info[1].Size);
            }

            {
                var info = DeltaInfo.ExtractDeltaCommands(source, destination, CommandSizeMin + 4, 4, null);
                Assert.AreEqual(4, info.Count);
                Assert.AreEqual(4, info[0].Offset);
                Assert.AreEqual(4, info[0].Size);
                Assert.AreEqual(8, info[1].Offset);
                Assert.AreEqual(4, info[1].Size);
                Assert.AreEqual(16, info[2].Offset);
                Assert.AreEqual(4, info[2].Size);
                Assert.AreEqual(20, info[3].Offset);
                Assert.AreEqual(4, info[3].Size);
            }
        }

        [TestMethod]
        public void TestCompareBlockSize()
        {
            var sourceData = new byte[]
            {
                0x55, 0xAA, 0x55, 0xAA,
                0xCD, 0xCD, 0xCD, 0xCD,
                0x55, 0xAA, 0x55, 0xAA,
                0xCD, 0xCD, 0xCD, 0xCD,
                0x55, 0xAA, 0x55, 0xAA,
                0x55, 0xAA, 0x55, 0xAA,
            };
            var destinationData = new byte[]
            {
                0x55, 0xAA, 0x55, 0xAA,
                0xCD, 0xCD, 0xCD, 0xCD,
                0x55, 0xAA, 0x55, 0xAA,
                0xFE, 0xFE, 0xFE, 0xFE,
                0x55, 0xAA, 0x55, 0xAA,
                0x55, 0xAA, 0x55, 0xAA,
                0xFE, 0xFE, 0xFE, 0xFE,
            };

            var source = new MemorySource(sourceData, 0, sourceData.Length);
            var destination = new MemorySource(destinationData, 0, destinationData.Length);

            {
                {
                    var info = DeltaInfo.ExtractDeltaCommands(source, destination, 1024, 4, null);
                    Assert.AreEqual(2, info.Count);
                    Assert.AreEqual(12, info[0].Offset);
                    Assert.AreEqual(4, info[0].Size);
                    Assert.AreEqual(24, info[1].Offset);
                    Assert.AreEqual(4, info[1].Size);
                }

                {
                    var info = DeltaInfo.ExtractDeltaCommands(source, destination, 1024, 8, null);
                    Assert.AreEqual(2, info.Count);
                    Assert.AreEqual(8, info[0].Offset);
                    Assert.AreEqual(8, info[0].Size);
                    Assert.AreEqual(24, info[1].Offset);
                    Assert.AreEqual(4, info[1].Size);
                }
            }
        }

        private static void CreateSampleDeltaSource(out ConcatenatedSource source, out ConcatenatedSource destination)
        {
            var sourceElements = new List<ConcatenatedSource.Element>();
            var destinationElements = new List<ConcatenatedSource.Element>();

            {
                long offset = 0;

                sourceElements.Add(new ConcatenatedSource.Element(
                    new PaddingSource((long)2 * 1024 * 1024 * 1024),
                    "zero_fill",
                    0));
                offset += (long)2 * 1024 * 1024 * 1024;

                {
                    var data = new byte[512];
                    for (int i = 0; i < data.Length; i++)
                    {
                        data[i] = 0xFE;
                    }
                    sourceElements.Add(new ConcatenatedSource.Element(
                        new MemorySource(data, 0, data.Length),
                        "data",
                        offset));
                    offset += data.Length;
                }

                sourceElements.Add(new ConcatenatedSource.Element(
                    new PaddingSource((long)4 * 1024 * 1024 * 1024 - offset),
                    "zero_fill",
                    offset));
            }

            {
                long offset = 0;

                destinationElements.Add(new ConcatenatedSource.Element(
                    new PaddingSource((long)2 * 1024 * 1024 * 1024),
                    "zero_fill",
                    0));
                offset += (long)2 * 1024 * 1024 * 1024;

                {
                    var data = new byte[512];
                    for (int i = 0; i < data.Length; i++)
                    {
                        data[i] = 0xCD;
                    }
                    destinationElements.Add(new ConcatenatedSource.Element(
                        new MemorySource(data, 0, data.Length),
                        "data",
                        offset));
                    offset += data.Length;
                }

                destinationElements.Add(new ConcatenatedSource.Element(
                    new PaddingSource((long)4 * 1024 * 1024 * 1024 - offset),
                    "zero_fill",
                    offset));
            }

            source = new ConcatenatedSource(sourceElements);
            destination = new ConcatenatedSource(destinationElements);
        }

        [TestMethod]
        public void TestLargeFile()
        {
            ConcatenatedSource source, destination;
            CreateSampleDeltaSource(out source, out destination);
            var info = DeltaInfo.ExtractDeltaCommands(source, destination, 1024, 512, null);
            Assert.AreEqual(1, info.Count);
            Assert.AreEqual((long)2 * 1024 * 1024 * 1024, info[0].Offset);
            Assert.AreEqual((long)512, info[0].Size);
        }

        [TestMethod]
        public void TestNearestAddress()
        {
            ConcatenatedSource source, destination;
            CreateSampleDeltaSource(out source, out destination);
            var sourceInfo = new DeltaSourceInfo(source.Size);
            var deltaCommands = DeltaInfo.ExtractDeltaCommands(source, destination, 256, 256, null);
            var deltaSource = new DeltaArchiveSource(sourceInfo, destination, deltaCommands);

            const int HeaderSize = 64;

            // ヘッダー部分は削れない
            Assert.AreEqual(HeaderSize, deltaSource.FindNearestEndOffset(0));
            Assert.AreEqual(HeaderSize, deltaSource.FindNearestEndOffset(HeaderSize / 2));
            Assert.AreEqual(HeaderSize, deltaSource.FindNearestEndOffset(HeaderSize));

            // ボディー部分
            Assert.AreEqual(HeaderSize + 256 +   0, deltaSource.FindNearestEndOffset(HeaderSize + 508));
            Assert.AreEqual(HeaderSize + 256 + 253, deltaSource.FindNearestEndOffset(HeaderSize + 509));
            Assert.AreEqual(HeaderSize + 256 + 253, deltaSource.FindNearestEndOffset(HeaderSize + 510));
        }

        [TestMethod]
        public void TestCombineNeighbor()
        {
            var listCommands = new List<DeltaCommand>(new DeltaCommand[]
                {
                    // サイズ制限の確認
                    new DeltaCommand(0, 1, false),
                    new DeltaCommand(1, 12, false), // 連結される
                    new DeltaCommand(16, 1, false),
                    new DeltaCommand(17, 13, false), // 連結されない
                    // シーク制限の確認
                    new DeltaCommand(32, 1, false),
                    new DeltaCommand(37, 1, false), // 連結される
                    new DeltaCommand(48, 1, false),
                    new DeltaCommand(54, 1, false), // 連結されない
                    // 3つ以上の連結
                    new DeltaCommand(64, 1, false),
                    new DeltaCommand(65, 1, false), // 連結される
                    new DeltaCommand(66, 1, false),
                    new DeltaCommand(67, 1, false), // 連結されない
                    // 単体で超える
                    new DeltaCommand(80, 32, false),
                    // 末尾の連結
                    new DeltaCommand(128, 1, false),
                    new DeltaCommand(129, 1, false),
                });
            DeltaInfo.CombineNeighbor(listCommands, 4, 16);

            Assert.AreEqual(9, listCommands.Count);

            Assert.AreEqual(0, listCommands[0].Offset);
            Assert.AreEqual(13, listCommands[0].Size);

            Assert.AreEqual(16, listCommands[1].Offset);
            Assert.AreEqual(1, listCommands[1].Size);

            Assert.AreEqual(17, listCommands[2].Offset);
            Assert.AreEqual(13, listCommands[2].Size);

            Assert.AreEqual(32, listCommands[3].Offset);
            Assert.AreEqual(6, listCommands[3].Size);

            Assert.AreEqual(48, listCommands[4].Offset);
            Assert.AreEqual(1, listCommands[4].Size);

            Assert.AreEqual(54, listCommands[5].Offset);
            Assert.AreEqual(1, listCommands[5].Size);

            Assert.AreEqual(64, listCommands[6].Offset);
            Assert.AreEqual(4, listCommands[6].Size);

            Assert.AreEqual(80, listCommands[7].Offset);
            Assert.AreEqual(32, listCommands[7].Size);

            Assert.AreEqual(128, listCommands[8].Offset);
            Assert.AreEqual(2, listCommands[8].Size);
        }
    }
}
