﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using TestUtility;

namespace AuthoringToolsTest
{
    [TestClass]
    public class ContentArchiveTest
    {
        public TestContext TestContext { get; set; }

        private string GetOutputPath(string testName)
        {
            return Path.Combine(Directory.GetCurrentDirectory().Replace("\\" + Assembly.GetExecutingAssembly().GetName().Name, string.Empty), testName);
        }

        private AuthoringConfiguration GetDefaultConfiguration(TestPath testPath)
        {
            var config = new AuthoringConfiguration();
            config.KeyConfigFilePath = testPath.GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";
            return config;
        }

        private void CreateNca(TestPath testPath, string outputPath, string baseDir, Action<string> generateData, Action<string> hookAdf, AuthoringConfiguration config)
        {
            string outputAdfPath = outputPath + ".adf";
            string metaFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.aarch64.lp64.nmeta";
            string descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

            string codePath = baseDir + @"\code";
            if (!Directory.Exists(codePath))
            {
                Directory.CreateDirectory(codePath);
            }
            MakeMetaUtil.MakeNpdm(testPath, codePath + @"\main.npdm", metaFile, descFile);

            string dataPath = baseDir + @"\data";
            if (!Directory.Exists(dataPath))
            {
                Directory.CreateDirectory(dataPath);
            }
            generateData(dataPath);

            var list = new List<Pair<string, string>>();
            list.Add(new Pair<string, string>(codePath.Replace(@"\", "/"), null));
            list.Add(new Pair<string, string>(dataPath.Replace(@"\", "/"), null));

            var reader = new MetaFileReader(metaFile, NintendoContentMetaConstant.ContentMetaTypeApplication);
            Func<Tuple<ulong, byte>> getNcaProgramId = () =>
            {
                var meta = reader.GetContentMetaList().First() as ApplicationMeta;
                if (meta != null)
                {
                    return Tuple.Create(meta.ProgramId, meta.ProgramIndex ?? (byte)0);
                }
                return Tuple.Create(reader.GetContentMetaList().First().Id, (byte)0);
            };
            var ncaProgramId = getNcaProgramId();

            var writer = new NintendoContentAdfWriter(outputAdfPath, NintendoContentMetaConstant.ContentTypeProgram, metaFile, descFile, -1, 0, ncaProgramId.Item1, ncaProgramId.Item2, false, false);
            writer.Write(list, null);
            hookAdf(outputAdfPath);

            using (FileStream fs = new FileStream(outputPath, FileMode.Create,
                       FileAccess.ReadWrite, FileShare.None, 4096, FileOptions.RandomAccess))
            {
                ContentArchiveLibraryInterface.CreateArchiveFromAdf(fs, NintendoContentMetaConstant.ContentMetaTypeApplication, outputAdfPath, config);
            }
        }

        [SuppressMessage("Microsoft.Usage", "CA2202", Justification = "NintendoContentArchiveReader は Stream を Dispose しない")]
        private void OpenNcaFile(string path, AuthoringConfiguration config, Action<NintendoContentArchiveReader> onRead)
        {
            using (var stream = new FileStream(path, FileMode.Open, FileAccess.Read))
            using (var nca = new NintendoContentArchiveReader(stream, new NcaKeyGenerator(config.GetKeyConfiguration())))
            {
                onRead(nca);
            }
        }

        [TestMethod]
        [SuppressMessage("Microsoft.Usage", "CA2202", Justification = "NintendoContentArchiveReader は Stream を Dispose しない")]
        public void TestNintendoContentArchiveReaderDispose()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var config = GetDefaultConfiguration(testPath);
            var outputDir = GetOutputPath(MethodBase.GetCurrentMethod().Name);
            string outputNcaPath = outputDir + @"\test.nca";

            Utils.DeleteDirectoryIfExisted(outputDir);
            Directory.CreateDirectory(outputDir);

            Action<string> generateData = (dataPath) =>
            {
                var testFile = dataPath + @"\test.bin";
                using (var file = new FileStream(testFile, FileMode.Create, FileAccess.Write))
                {
                }
            };

            CreateNca(testPath, outputNcaPath, outputDir, generateData, (adf) => { }, config);

            var disposed = new DisposeCheckStream.DisposedValue();
            using (var stream = new DisposeCheckStream(outputNcaPath, disposed))
            {
                using (var nca = new NintendoContentArchiveReader(stream, new NcaKeyGenerator(config.GetKeyConfiguration())))
                {
                }
                Assert.IsFalse(disposed.Value);
            }
        }

        [TestMethod]
        public void TestNintendoContentArchiveReaderGetFileFragmentList()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var config = GetDefaultConfiguration(testPath);
            var outputDir = GetOutputPath(MethodBase.GetCurrentMethod().Name);
            string outputNcaPath = outputDir + @"\test.nca";

            Utils.DeleteDirectoryIfExisted(outputDir);
            Directory.CreateDirectory(outputDir);

            Action<string> generateData = (dataPath) =>
            {
                for (int i = 0; i < 2; ++i)
                {
                    var testFile = dataPath + @"\test" + i + ".bin";
                    using (var file = new FileStream(testFile, FileMode.Create, FileAccess.Write))
                    {
                        file.WriteByte(0xFE);
                        file.Seek(16 * 1024, SeekOrigin.Begin);
                        file.WriteByte(0xEF);
                    }
                }
            };

            Action<string> hookAdf = (adfPath) =>
            {
                var data = File.ReadAllText(adfPath).Replace("encryptionType : 3", "encryptionType : 1");
                File.WriteAllText(adfPath, data);
            };

            CreateNca(testPath, outputNcaPath, outputDir, generateData, hookAdf, config);

            OpenNcaFile(outputNcaPath, config, (nca) =>
            {
                using (var fs = nca.OpenFileSystemArchiveReader((int)NintendoContentArchivePartitionType.Data))
                {
                    var offset = nca.GetFsStartOffset((int)NintendoContentArchivePartitionType.Data);
                    Utils.WriteLine("base offset: " + offset);

                    long hashTargetOffset;
                    using (var header = nca.GetFsHeaderInfo((int)NintendoContentArchivePartitionType.Data))
                    {
                        hashTargetOffset = (long)header.GetHashTargetOffset();
                        Utils.WriteLine("hash target offset: " + hashTargetOffset);
                    }

                    foreach (var file in fs.ListFileInfo())
                    {
                        Utils.WriteLine(file.Item1);

                        foreach (var fragment in fs.GetFileFragmentList(file.Item1))
                        {
                            Utils.WriteLine("  " + fragment.Item1 + ", " + fragment.Item2);

                            using (var raw = File.OpenRead(outputNcaPath))
                            {
                                var dataOffset = offset + hashTargetOffset + fragment.Item1;
                                raw.Seek(dataOffset, SeekOrigin.Begin);
                                Assert.AreEqual(0xFE, raw.ReadByte());

                                dataOffset += fragment.Item2 - 1;
                                raw.Seek(dataOffset, SeekOrigin.Begin);
                                Assert.AreEqual(0xEF, raw.ReadByte());
                            }
                        }
                    }
                }
            });
        }

        [TestMethod]
        public void TestPatchConstructionUtilityGetHintsFromNcaFiles()
        {
            const int FsIndexExists = (int)NintendoContentArchivePartitionType.Data;
            const int FsIndexNotExists = (int)NintendoContentArchivePartitionType.Logo;

            var testPath = new TestUtility.TestPath(this.TestContext);
            var config = GetDefaultConfiguration(testPath);
            var outputDir = GetOutputPath(MethodBase.GetCurrentMethod().Name);

            {
                var hints = PatchConstructionUtility.GetHintsFromNcaFiles(null, null, 0);
                Assert.AreEqual(0, hints.Item1.Length);
                Assert.AreEqual(0, hints.Item2.Length);
            }

            {
                for (int fileIndex = 0; fileIndex < 2; ++fileIndex)
                {
                    string workingDirectory = outputDir + "/test" + fileIndex;
                    string outputNcaPath = outputDir + @"\test" + fileIndex + ".nca";

                    Utils.DeleteDirectoryIfExisted(workingDirectory);
                    Directory.CreateDirectory(workingDirectory);

                    Action<string> generateData = (dataPath) =>
                    {
                        Func<int, string> generateFilePath = (index) => string.Format("{0}\\test{1:D2}.bin", dataPath, index);

                        // マージされるケース
                        for (int i = 0; i < 3; ++i)
                        {
                            using (var file = new FileStream(generateFilePath(i), FileMode.Create, FileAccess.Write))
                            {
                                file.SetLength(16 * 1024);
                            }
                        }

                        // オフセットがずれるケース
                        if (fileIndex == 1)
                        {
                            using (var file = new FileStream(generateFilePath(3), FileMode.Create, FileAccess.Write))
                            {
                                file.SetLength(1);
                            }
                        }

                        {
                            using (var file = new FileStream(generateFilePath(4), FileMode.Create, FileAccess.Write))
                            {
                                file.SetLength(16 * 1024);
                            }
                        }

                        // オフセット調整 兼 仕切り板
                        if (fileIndex == 0)
                        {
                            using (var file = new FileStream(generateFilePath(5), FileMode.Create, FileAccess.Write))
                            {
                                file.SetLength(1);
                            }
                        }

                        // サイズが異なるケース
                        {
                            using (var file = new FileStream(generateFilePath(6), FileMode.Create, FileAccess.Write))
                            {
                                if (fileIndex == 0)
                                {
                                    file.SetLength(16 * 1024);
                                }
                                else
                                {
                                    file.SetLength(16 * 1024 + 16);
                                }
                            }
                        }

                        // オフセット調整 兼 仕切り板
                        if (fileIndex == 0)
                        {
                            using (var file = new FileStream(generateFilePath(7), FileMode.Create, FileAccess.Write))
                            {
                                file.SetLength(1);
                            }
                        }

                        // 合計して 16KiB のデータ
                        for (int i = 8; i < 8 + 16; ++i)
                        {
                            using (var file = new FileStream(generateFilePath(i), FileMode.Create, FileAccess.Write))
                            {
                                file.SetLength(1024);
                            }
                        }

                        // 仕切り板
                        {
                            using (var file = new FileStream(generateFilePath(24 + fileIndex), FileMode.Create, FileAccess.Write))
                            {
                                file.SetLength(1);
                            }
                        }

                        // 16KiB 未満のデータ
                        {
                            using (var file = new FileStream(generateFilePath(26), FileMode.Create, FileAccess.Write))
                            {
                                file.SetLength(8 * 1024);
                            }
                        }
                    };

                    Action<string> hookAdf = (adfPath) =>
                    {
                        var data = File.ReadAllText(adfPath).Replace("encryptionType : 3", "encryptionType : 1");
                        File.WriteAllText(adfPath, data);
                    };

                    CreateNca(testPath, outputNcaPath, workingDirectory, generateData, hookAdf, config);
                }

                string outputNcaPath1 = outputDir + @"\test0.nca";
                string outputNcaPath2 = outputDir + @"\test1.nca";
                OpenNcaFile(outputNcaPath1, config, (nca1) =>
                {
                    OpenNcaFile(outputNcaPath2, config, (nca2) =>
                    {
                        const long Offset = 80 * 1024;

                        var hints = PatchConstructionUtility.GetHintsFromNcaFiles(nca1, nca2, FsIndexExists);
                        Assert.AreEqual(5, hints.Item1.Length);
                        Assert.AreEqual(5, hints.Item2.Length);

                        // 連続した領域はマージされる
                        Assert.AreEqual(Offset + 512, hints.Item2[0].oldOffset);
                        Assert.AreEqual(3 * 16 * 1024, hints.Item2[0].oldSize);
                        Assert.AreEqual(Offset + 512, hints.Item2[0].newOffset);
                        Assert.AreEqual(3 * 16 * 1024, hints.Item2[0].newSize);

                        // オフセットがずれるケース
                        Assert.AreEqual(Offset + 49664, hints.Item2[1].oldOffset);
                        Assert.AreEqual(16 * 1024, hints.Item2[1].oldSize);
                        Assert.AreEqual(Offset + 49680, hints.Item2[1].newOffset);
                        Assert.AreEqual(16 * 1024, hints.Item2[1].newSize);

                        // サイズが異なるケース
                        Assert.AreEqual(Offset + 66064, hints.Item2[2].oldOffset);
                        Assert.AreEqual(16 * 1024, hints.Item2[2].oldSize);
                        Assert.AreEqual(Offset + 66064, hints.Item2[2].newOffset);
                        Assert.AreEqual(16 * 1024 + 16, hints.Item2[2].newSize);

                        // 合計してサイズが 16KiB になるケース
                        Assert.AreEqual(Offset + 82464, hints.Item2[3].oldOffset);
                        Assert.AreEqual(16 * 1024, hints.Item2[3].oldSize);
                        Assert.AreEqual(Offset + 82464, hints.Item2[3].newOffset);
                        Assert.AreEqual(16 * 1024, hints.Item2[3].newSize);

                        // 16 KiB 未満のデータ
                        Assert.AreEqual(Offset + 98864, hints.Item2[4].oldOffset);
                        Assert.AreEqual(8 * 1024, hints.Item2[4].oldSize);
                        Assert.AreEqual(Offset + 98864, hints.Item2[4].newOffset);
                        Assert.AreEqual(8 * 1024, hints.Item2[4].newSize);

                        // 存在しないパーティション
                        hints = PatchConstructionUtility.GetHintsFromNcaFiles(nca1, nca2, FsIndexNotExists);
                        Assert.AreEqual(0, hints.Item1.Length);
                        Assert.AreEqual(0, hints.Item2.Length);
                    });
                });
            }

            // サイズ 0 の領域は含まない
            {
                for (int fileIndex = 0; fileIndex < 2; ++fileIndex)
                {
                    string workingDirectory = outputDir + "/test" + fileIndex;
                    string outputNcaPath = outputDir + @"\test" + fileIndex + ".nca";

                    Utils.DeleteDirectoryIfExisted(workingDirectory);
                    Directory.CreateDirectory(workingDirectory);

                    Action<string> generateData = (dataPath) =>
                    {
                        for (int i = 0; i < 2; ++i)
                        {
                            var testFile = dataPath + @"\test" + i + ".bin";
                            using (var file = new FileStream(testFile, FileMode.Create, FileAccess.Write))
                            {
                            }
                        }
                    };

                    Action<string> hookAdf = (adfPath) =>
                    {
                        var data = File.ReadAllText(adfPath).Replace("encryptionType : 3", "encryptionType : 1");
                        File.WriteAllText(adfPath, data);
                    };

                    CreateNca(testPath, outputNcaPath, workingDirectory, generateData, hookAdf, config);
                }

                string outputNcaPath1 = outputDir + @"\test0.nca";
                string outputNcaPath2 = outputDir + @"\test1.nca";
                OpenNcaFile(outputNcaPath1, config, (nca1) =>
                {
                    OpenNcaFile(outputNcaPath2, config, (nca2) =>
                    {
                        var hints = PatchConstructionUtility.GetHintsFromNcaFiles(nca1, nca2, FsIndexExists);
                        Assert.AreEqual(0, hints.Item1.Length);
                        Assert.AreEqual(0, hints.Item2.Length);
                    });
                });
            }

            Utils.DeleteIncludingJunctionDirectoryIfExisted(outputDir);
       }
    }
}
