﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test.ReplaceNsp
{
    public class NspReplacerTestBase : IDisposable, IClassFixture<BuildTestAppNspFixture>
    {
        protected readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        protected readonly BuildTestAppNspFixture _fixture;

        public string NspFilePath => _fixture.NspFilePath;

        public NspReplacerTestBase(BuildTestAppNspFixture fixture)
        {
            _fixture = fixture.Initialize();
        }

        protected static void AssertTitleIconExists(bool expected, IEnumerable<Title> titles, LanguageType l)
        {
            var t = titles.FirstOrDefault(x => x.Language == l);
            Assert.Equal(expected, File.Exists(t.OriginalIconFilePath));
            Assert.Equal(expected, File.Exists(t.OriginalNxIconFilePath));
        }

        protected async Task AssertNspFileEqual(INspFile a, INspFile b, string pathInNsp)
        {
            var r = await GetNspFileTupleInternal(a, b, pathInNsp).ConfigureAwait(false);
            Assert.Equal(r.Item1, r.Item2);
        }

        protected async Task AssertNspFileNotEqual(INspFile a, INspFile b, string pathInNsp)
        {
            var r = await GetNspFileTupleInternal(a, b, pathInNsp).ConfigureAwait(false);
            Assert.NotEqual(r.Item1, r.Item2);
        }

        protected enum NspIconType
        {
            Nx,
            Raw
        }

        protected async Task AssertNspIconEqualAsync(INspFile a, INspFile b, LanguageType language, NspIconType suffix)
        {
            var r = await GetNspIconTupleInternal(a, b, language, suffix).ConfigureAwait(false);
            Assert.Equal(r.Item1, r.Item2);
        }

        protected async Task AssertNspIconNotEqualAsync(INspFile a, INspFile b, LanguageType language, NspIconType suffix)
        {
            var r = await GetNspIconTupleInternal(a, b, language, suffix).ConfigureAwait(false);
            Assert.NotEqual(r.Item1, r.Item2);
        }

        private async Task<Tuple<byte[], byte[]>> GetNspIconTupleInternal(INspFile a, INspFile b,
            LanguageType language, NspIconType suffix)
        {
            var suffixName = suffix.ToString().ToLowerInvariant();
            var left = a.Files(new NspFileEnumerateParameter { EnumerationType = NspFileEnumeration.RootOnly })
                .FirstOrDefault(x => x.FilePath.EndsWith($".{suffixName}.{language}.jpg"));
            var right = b.Files(new NspFileEnumerateParameter { EnumerationType = NspFileEnumeration.RootOnly })
                .FirstOrDefault(x => x.FilePath.EndsWith($".{suffixName}.{language}.jpg"));

            return await GetNspFileTupleInternal(a, b, left.FilePath, right.FilePath).ConfigureAwait(false);
        }

        private async Task<Tuple<byte[], byte[]>> GetNspFileTupleInternal(INspFile a, INspFile b, string aPathInNsp, string bPathInNsp = null)
        {
            Assert.NotNull(aPathInNsp);

            var leftFile = _context.CreateFile($"a-{Path.GetRandomFileName()}");
            var rightFile = _context.CreateFile($"b-{Path.GetRandomFileName()}");

            var left = a.Files(new NspFileEnumerateParameter { EnumerationType = NspFileEnumeration.All }).FirstOrDefault(x => x.FilePath.Contains(aPathInNsp));
            Assert.NotNull(left);
            await a.ExtractAsync(left.FilePath, leftFile).ConfigureAwait(false);

            var right = b.Files(new NspFileEnumerateParameter { EnumerationType = NspFileEnumeration.All }).FirstOrDefault(x => x.FilePath.Contains(bPathInNsp ?? aPathInNsp));
            Assert.NotNull(right);
            await b.ExtractAsync(right.FilePath, rightFile).ConfigureAwait(false);

            return Tuple.Create(File.ReadAllBytes(leftFile), File.ReadAllBytes(rightFile));
        }

        protected static void AreEqualDirectoryFileSystemEntries(string source, string target)
        {
            var sourceFiles =
                Directory.EnumerateFileSystemEntries(source, "*", SearchOption.AllDirectories)
                    .Select(x => x.Substring(source.Length + 1));
            var targetFiles =
                Directory.EnumerateFileSystemEntries(target, "*", SearchOption.AllDirectories)
                    .Select(x => x.Substring(target.Length + 1));
            Assert.Equal(sourceFiles, targetFiles);
        }

        protected static Title GetTitle(ApplicationMeta appMeta, LanguageType languageType)
        {
            return appMeta.Application.Titles.First(x => x.Language == languageType);
        }

        protected string GetTemporaryIcon()
        {
            var modifiedIconPath = Path.Combine(_context.TempDirPath, "changed.bmp");
            {
                var iconPath = Path.Combine(TestContext.TestDataDirPath, @"icon_testdata\1024x1024x24.bmp");
                File.Copy(iconPath, modifiedIconPath, true);

                var black = new byte[] { 0x00, 0x00, 0x00, 0x00 };
                using (var writer = new FileStream(modifiedIconPath, FileMode.Open, FileAccess.ReadWrite))
                {
                    writer.Seek(0x00200000, SeekOrigin.Begin);
                    for (var i = 0; i < 1024; ++i)
                    {
                        writer.Write(black, 0, black.Length);
                    }
                }
            }
            return modifiedIconPath;
        }

        protected async Task<NspImporter.ResultType> OpenNspAsync(string nspPath, DisposableDirectory tempDir)
        {
            var output = await new NspImporter().ToMetaAsync(new NspImporter.Option
            {
                DiContainer = _context.DiContainer,
                InputNspFile = new NspFile(nspPath),
                TempNspExtractFilePath = tempDir.RootPath
            }).ConfigureAwait(false);

            await output.ExtraResourceImporter.ReadHtmlDocumentAsync().ConfigureAwait(false);
            await output.ExtraResourceImporter.ReadLegalInformationAsync().ConfigureAwait(false);

            Assert.Equal(NspHandleResult.Ok, output.Result);

            return output;
        }

    }

    public class NspReplacerTestDefaultCtor : NspReplacerTestBase
    {
        public NspReplacerTestDefaultCtor(BuildTestAppNspFixture fixture) : base(fixture)
        {
        }

        [Fact]
        public void DefaultCtor()
        {
            _context.DiContainer.GetInstance<NspReplacer>();
        }
    }

    public class NspReplacerTestReplaceTitle : NspReplacerTestBase
    {
        public NspReplacerTestReplaceTitle(BuildTestAppNspFixture fixture) : base(fixture)
        {
        }

        [Fact]
        public async Task ReplaceTitle()
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                var outputPath = Path.Combine(_context.TempDirPath, "change.nsp");
                var replacer = _context.DiContainer.GetInstance<NspReplacer>();
                var option = new NspReplacer.ReplaceOption
                {
                    InputNspPath = NspFilePath,
                    CancellationToken = default(CancellationToken),
                    Project = project,
                    OutputNspPath = outputPath,
                };

                var originalRawIconPath = Path.Combine(_context.TempDirPath, "original.jpg");
                using (var d = new DisposableDirectory())
                {
                    var nsp = await OpenNspAsync(NspFilePath, d).ConfigureAwait(false);
                    var title = GetTitle(nsp.ApplicationMeta, LanguageType.AmericanEnglish);
                    Assert.True(File.Exists(title.OriginalIconFilePath));
                    File.Move(title.OriginalIconFilePath, originalRawIconPath);
                }


                project.Meta.Application.Titles.Add(new Title
                {
                    Name = "ほげほげ",
                    Publisher = "ぱぶぱぶりっしゃー",
                    Language = LanguageType.Japanese,
                });

                project.Meta.Application.Titles.Add(new Title
                {
                    Name = "Foobar",
                    Publisher = "ReplacedIconPublisher",
                    IsReplaceIcon = true,
                    IconFilePath = GetTemporaryIcon(),
                    Language = LanguageType.AmericanEnglish,
                });

                {
                    var r = await replacer.ReplaceNspAsync(option).ConfigureAwait(false);
                    Assert.Equal(NspHandleResult.Ok, r.Result);
                }

                using (var d = new DisposableDirectory())
                {
                    var nsp = await OpenNspAsync(outputPath, d).ConfigureAwait(false);
                    {
                        var title = GetTitle(nsp.ApplicationMeta, LanguageType.AmericanEnglish);
                        Assert.Equal("Foobar", title.Name);
                        Assert.Equal("ReplacedIconPublisher", title.Publisher);
                        Assert.True(File.Exists(title.OriginalIconFilePath));
                        Assert.NotEqual(
                            File.ReadAllBytes(originalRawIconPath),
                            File.ReadAllBytes(title.OriginalIconFilePath));
                        Assert.DoesNotContain("<Original", _context.MakeAppMetaXml(nsp.ApplicationMeta));

                        await AssertNspFileNotEqual(
                            new NspFile(outputPath),
                            new NspFile(NspFilePath),
                            $"icon_{title.Language}.dat").ConfigureAwait(false);
                    }

                    {
                        var title = GetTitle(nsp.ApplicationMeta, LanguageType.Japanese);
                        Assert.Equal("ほげほげ", title.Name);
                        Assert.Equal("ぱぶぱぶりっしゃー", title.Publisher);
                    }
                }
            }
        }
    }

    public class NspReplacerTestReplaceLegalInformation : NspReplacerTestBase
    {
        public NspReplacerTestReplaceLegalInformation(BuildTestAppNspFixture fixture) : base(fixture)
        {

        }

        [Fact]
        public async Task ReplaceLegalInformation()
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                var outputPath = Path.Combine(_context.TempDirPath, "change.nsp");
                var replacer = _context.DiContainer.GetInstance<NspReplacer>();
                var option = new NspReplacer.ReplaceOption
                {
                    InputNspPath = NspFilePath,
                    CancellationToken = default(CancellationToken),
                    Project = project,
                    OutputNspPath = outputPath,
                };

                using (var tempDir = new DisposableDirectory())
                {
                    var nsp = await OpenNspAsync(NspFilePath, tempDir).ConfigureAwait(false);
                    Assert.False(Directory.Exists(nsp.ApplicationMeta.Application.OriginalLegalInformationPath));
                }

                var legalInfoZips = new[]
                {
                    "legal_info_0100000000001000_75e7.zip"
                };

                foreach (var zip in legalInfoZips)
                {
                    using (var tempDir = new DisposableDirectory())
                    {
                        project.Meta.Application.IsReplaceLegalInformationFilePath = true;
                        project.Meta.Application.LegalInformationFilePath =
                            Path.Combine(TestContext.TestDataDirPath, "project_testdata", zip);

                        var r = await replacer.ReplaceNspAsync(option).ConfigureAwait(false);
                        Assert.Equal(NspHandleResult.Ok, r.Result);

                        var nsp = await OpenNspAsync(outputPath, tempDir).ConfigureAwait(false);
                        Assert.True(Directory.Exists(nsp.ApplicationMeta.Application.OriginalLegalInformationPath));
                        Assert.DoesNotContain("<Original", _context.MakeAppMetaXml(nsp.ApplicationMeta));
                    }
                }
            }
        }
    }

    public class NspReplacerTestReplaceAccessibleUrls : NspReplacerTestBase
    {
        public NspReplacerTestReplaceAccessibleUrls(BuildTestAppNspFixture fixture) : base(fixture)
        {

        }

        [Fact]
        public async Task ReplaceAccessibleUrls()
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                var outputPath = Path.Combine(_context.TempDirPath, "change.nsp");
                var replacer = _context.DiContainer.GetInstance<NspReplacer>();
                var option = new NspReplacer.ReplaceOption
                {
                    InputNspPath = NspFilePath,
                    CancellationToken = default(CancellationToken),
                    Project = project,
                    OutputNspPath = outputPath,
                };

                using (var tempDir = new DisposableDirectory())
                {
                    var nsp = await OpenNspAsync(NspFilePath, tempDir).ConfigureAwait(false);
                    Assert.False(File.Exists(nsp.ApplicationMeta.Application.OriginalAccessibleUrlsFilePath));
                }

                var urlPatternsTextSet = new[]
                {
                    "^http://example.com/$\n^http://www.nintendo.co.jp/.+\n",
                    "foobar\nhoge\n"
                };

                foreach (var urlsText in urlPatternsTextSet)
                {
                    using (var tempDir = new DisposableDirectory())
                    {
                        var accessibleUrlsPath = Path.Combine(tempDir.RootPath, "foo.txt");
                        File.WriteAllText(accessibleUrlsPath, urlsText);
                        project.Meta.Application.IsReplaceAccessibleUrlsFilePath = true;
                        project.Meta.Application.AccessibleUrlsFilePath = accessibleUrlsPath;

                        var r = await replacer.ReplaceNspAsync(option).ConfigureAwait(false);
                        Assert.Equal(NspHandleResult.Ok, r.Result);

                        var nsp = await OpenNspAsync(outputPath, tempDir).ConfigureAwait(false);
                        Assert.Equal(
                            File.ReadAllText(accessibleUrlsPath),
                            File.ReadAllText(nsp.ApplicationMeta.Application.OriginalAccessibleUrlsFilePath));
                        Assert.DoesNotContain("<Original", _context.MakeAppMetaXml(nsp.ApplicationMeta));
                    }
                }
            }
        }
    }

    public class NspReplacerTestReplaceOfflineHtml : NspReplacerTestBase
    {
        public NspReplacerTestReplaceOfflineHtml(BuildTestAppNspFixture fixture) : base(fixture)
        {

        }

        [Fact]
        public async Task ReplaceOfflineHtml()
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                var outputPath = Path.Combine(_context.TempDirPath, "change.nsp");
                var replacer = _context.DiContainer.GetInstance<NspReplacer>();
                var option = new NspReplacer.ReplaceOption
                {
                    InputNspPath = NspFilePath,
                    CancellationToken = default(CancellationToken),
                    Project = project,
                    OutputNspPath = outputPath,
                };

                using (var tempDir = new DisposableDirectory())
                {
                    var nsp = await OpenNspAsync(NspFilePath, tempDir).ConfigureAwait(false);
                    Assert.False(Directory.Exists(nsp.ApplicationMeta.Application.OriginalHtmlDocumentPath));
                }

                var testSet = new[]
                {
                    "project_testdata",
                    "icon_testdata"
                };

                foreach (var testFolderName in testSet)
                {
                    using (var tempDir = new DisposableDirectory())
                    {
                        var htmlDocumentPath = Path.Combine(TestContext.TestDataDirPath, testFolderName);
                        project.Meta.Application.IsReplaceHtmlDocumentPath = true;
                        project.Meta.Application.HtmlDocumentPath = htmlDocumentPath;

                        var r = await replacer.ReplaceNspAsync(option).ConfigureAwait(false);
                        Assert.Equal(NspHandleResult.Ok, r.Result);

                        var nsp = await OpenNspAsync(outputPath, tempDir).ConfigureAwait(false);
                        Assert.True(Directory.Exists(nsp.ApplicationMeta.Application.OriginalHtmlDocumentPath));
                        AreEqualDirectoryFileSystemEntries(htmlDocumentPath,
                            nsp.ApplicationMeta.Application.OriginalHtmlDocumentPath);
                        Assert.DoesNotContain("<Original", _context.MakeAppMetaXml(nsp.ApplicationMeta));
                    }
                }
            }
        }
    }

    public class NspReplacerTestReplacePropertyAndExternalResources : NspReplacerTestBase
    {
        public NspReplacerTestReplacePropertyAndExternalResources(BuildTestAppNspFixture fixture) : base(fixture)
        {

        }

        [Fact]
        public async Task ReplaceApplicationPropertyAndExternalResources()
        {
            using (var project = Project.Import(_context.DiContainer, ImportableFileType.Nsp, NspFilePath))
            {
                var outputPath = Path.Combine(_context.TempDirPath, "change.nsp");
                var replacer = _context.DiContainer.GetInstance<NspReplacer>();
                var option = new NspReplacer.ReplaceOption
                {
                    InputNspPath = NspFilePath,
                    CancellationToken = default(CancellationToken),
                    Project = project,
                    OutputNspPath = outputPath,
                };

                var displayVersion = "9.9.9";
                project.Meta.Application.DisplayVersion = displayVersion;

                ushort releaseVersion = 10;
                project.Meta.Application.ReleaseVersion = releaseVersion;

                var htmlDocumentPath = Path.Combine(TestContext.TestDataDirPath, "icon_testdata");
                project.Meta.Application.IsReplaceHtmlDocumentPath = true;
                project.Meta.Application.HtmlDocumentPath = htmlDocumentPath;

                var urlsText = "http://example.com\nhoge\nfuga\n";
                var accessibleUrlsPath = Path.Combine(_context.TempDirPath, "foo.txt");
                File.WriteAllText(accessibleUrlsPath, urlsText);
                project.Meta.Application.IsReplaceAccessibleUrlsFilePath = true;
                project.Meta.Application.AccessibleUrlsFilePath = accessibleUrlsPath;

                var legalInformationFilePath = Path.Combine(TestContext.TestDataDirPath,
                    @"project_testdata\legal_info_0100000000001000_75e7.zip");
                project.Meta.Application.IsReplaceLegalInformationFilePath = true;
                project.Meta.Application.LegalInformationFilePath = legalInformationFilePath;

                project.Meta.Application.IsUseSeedForPseudoDeviceAppId = false;
                project.Meta.Application.SeedForPseudoDeviceId = 0x1234ul;

                var originalRawIconPath = Path.Combine(_context.TempDirPath, "original.jpg");
                {
                    var title = project.Meta.Application.Titles.First(x => File.Exists(x.OriginalIconFilePath));
                    File.Copy(title.OriginalIconFilePath, originalRawIconPath, true);
                    title.IsReplaceIcon = true;
                    title.IconFilePath = GetTemporaryIcon();
                }

                var r = await replacer.ReplaceNspAsync(option).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.Ok, r.Result);

                using (var d = new DisposableDirectory())
                {
                    var nsp = await OpenNspAsync(outputPath, d).ConfigureAwait(false);

                    Assert.Equal(displayVersion, nsp.ApplicationMeta.Application.DisplayVersion);
                    Assert.Equal(releaseVersion, nsp.ApplicationMeta.Application.ReleaseVersion);
                    Assert.Equal(urlsText,
                        File.ReadAllText(nsp.ApplicationMeta.Application.OriginalAccessibleUrlsFilePath));
                    Assert.True(Directory.Exists(nsp.ApplicationMeta.Application.OriginalLegalInformationPath));

                    if (project.AppCapability.IsSupportSeedForPseudoDeviceId)
                    {
                        Assert.Equal(0x1234ul, nsp.ApplicationMeta.Application.SeedForPseudoDeviceId);
                    }

                    AreEqualDirectoryFileSystemEntries(htmlDocumentPath,
                        nsp.ApplicationMeta.Application.OriginalHtmlDocumentPath);

                    var title = nsp.ApplicationMeta.Application.Titles.First(x => File.Exists(x.OriginalIconFilePath));

                    // OriginalIcon の置き換えが行えているか
                    {
                        Assert.True(File.Exists(title.OriginalIconFilePath));
                        Assert.NotEqual(
                            File.ReadAllBytes(originalRawIconPath),
                            File.ReadAllBytes(title.OriginalIconFilePath));
                    }

                    // icon_*.dat の置き換えが行えているか
                    await AssertNspFileNotEqual(
                        new NspFile(outputPath),
                        new NspFile(NspFilePath),
                        $"icon_{title.Language}.dat").ConfigureAwait(false);
                }
            }
        }
    }

    public class NspReplacerTestAddOrRemoveTitleIcons : NspReplacerTestBase
    {
        public NspReplacerTestAddOrRemoveTitleIcons(BuildTestAppNspFixture fixture) : base(fixture)
        {

        }

        [Theory]
        [InlineData(true)]
        [InlineData(false)]
        public async Task AddOrRemoveTitleIcons(bool isExpandEnvVar)
        {
            using (var project = Project.Import(_context.DiContainer, ImportableFileType.Nsp, NspFilePath))
            {
                if (isExpandEnvVar &&
                    project.AppCapability.IsSupportEnvironmentVariableToPath == false)
                {
                    return;
                }

                var outputPath = Path.Combine(_context.TempDirPath, "change.nsp");
                var replacer = _context.DiContainer.GetInstance<NspReplacer>();
                var option = new NspReplacer.ReplaceOption
                {
                    InputNspPath = NspFilePath,
                    CancellationToken = default(CancellationToken),
                    Project = project,
                    OutputNspPath = outputPath,
                };

                {
                    var titles = project.Meta.Application.Titles;

                    // 画像の削除
                    titles.Remove(titles.FirstOrDefault(x => x.Language == LanguageType.AmericanEnglish));

                    // 既存タイトルへのアイコン追加
                    var tempIconPath = GetTemporaryIcon();
                    project.ProjectDirectory = Path.GetDirectoryName(tempIconPath);
                    {
                        var t = titles.FirstOrDefault(x => x.Language == LanguageType.Japanese);
                        Assert.NotNull(t);
                        t.IsReplaceIcon = true;
                        t.IconFilePath = Path.GetFileName(tempIconPath);
                        t.IconFilePath.IsExpandEnvironmentVariable = isExpandEnvVar;
                    }
                }

                var r = await replacer.ReplaceNspAsync(option).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.Ok, r.Result);

                using (var d = new DisposableDirectory())
                {
                    var nsp = await OpenNspAsync(outputPath, d).ConfigureAwait(false);
                    var titles = nsp.ApplicationMeta.Application.Titles;
                    Assert.Equal(1, titles.Count);

                    if (project.AppCapability.IsSupportNspTitleAddOrRemove == false)
                        return;

                    AssertTitleIconExists(true, titles, LanguageType.Japanese);

                    // ファイルリストに削除したアイコンが存在しないことを確認
                    var files = new NspFile(outputPath).Files(new NspFileEnumerateParameter { EnumerationType = NspFileEnumeration.All }).ToArray();
                    foreach (var i in new[] {"raw", "nx"})
                    {
                        Assert.Empty(files.Where(x => x.FilePath.EndsWith($".{i}.{nameof(LanguageType.AmericanEnglish)}.jpg")));
                    }
                }
            }
        }

        [Theory]
        [InlineData(true)]
        [InlineData(false)]
        public async Task AddTitleIcons(bool isExpandEnvVar)
        {
            using (var project = Project.Import(_context.DiContainer, ImportableFileType.Nsp, NspFilePath))
            {
                if (isExpandEnvVar &&
                    project.AppCapability.IsSupportEnvironmentVariableToPath == false)
                {
                    return;
                }

                var outputPath = Path.Combine(_context.TempDirPath, "change.nsp");
                var replacer = _context.DiContainer.GetInstance<NspReplacer>();
                var option = new NspReplacer.ReplaceOption
                {
                    InputNspPath = NspFilePath,
                    CancellationToken = default(CancellationToken),
                    Project = project,
                    OutputNspPath = outputPath,
                };

                {
                    var titles = project.Meta.Application.Titles;

                    // 新規タイトルの追加
                    var newTitle = new Title
                    {
                        Language = LanguageType.German,
                        Name = "German",
                        Publisher = "Test German Publisher",
                        IsReplaceIcon = true,
                        IconFilePath = NintendoSdkHelper.ApplicationIconPath.ToString()
                    };
                    newTitle.IconFilePath.IsExpandEnvironmentVariable = isExpandEnvVar;
                    titles.Add(newTitle);

                    // 既存タイトルのアイコン置き換え
                    var tempIconPath = GetTemporaryIcon();
                    project.ProjectDirectory = Path.GetDirectoryName(tempIconPath);
                    {
                        var t = titles.FirstOrDefault(x => x.Language == LanguageType.Japanese);
                        Assert.NotNull(t);
                        t.IsReplaceIcon = true;
                        t.IconFilePath = Path.GetFileName(tempIconPath);
                        t.IconFilePath.IsExpandEnvironmentVariable = isExpandEnvVar;
                    }
                }

                var r = await replacer.ReplaceNspAsync(option).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.Ok, r.Result);

                using (var d = new DisposableDirectory())
                {
                    var nsp = await OpenNspAsync(outputPath, d).ConfigureAwait(false);
                    var titles = nsp.ApplicationMeta.Application.Titles;
                    Assert.Equal(3, titles.Count);

                    if (project.AppCapability.IsSupportNspTitleAddOrRemove == false)
                        return;

                    // 英語のアイコンは変更されていないことを確認
                    await AssertNspIconEqualAsync(new NspFile(NspFilePath), new NspFile(outputPath),
                        LanguageType.AmericanEnglish, NspIconType.Nx).ConfigureAwait(false);
                    // 日本語アイコンは差し替えたので変更されたことを確認
                    await AssertNspIconNotEqualAsync(new NspFile(NspFilePath), new NspFile(outputPath),
                        LanguageType.Japanese, NspIconType.Nx).ConfigureAwait(false);

                    AssertTitleIconExists(true, titles, LanguageType.German);
                    AssertTitleIconExists(true, titles, LanguageType.AmericanEnglish);
                    AssertTitleIconExists(true, titles, LanguageType.Japanese);
                }
            }
        }
    }

    public class NspReplacerTestRelativePathItems : NspReplacerTestBase
    {
        public NspReplacerTestRelativePathItems(BuildTestAppNspFixture fixture) : base(fixture)
        {

        }

        [Theory]
        [InlineData(true)]
        [InlineData(false)]
        public async Task RelativePathItems(bool isExpandEnvVar)
        {
            using (var project = Project.Import(_context.DiContainer, ImportableFileType.Nsp, NspFilePath))
            {
                if (isExpandEnvVar && project.AppCapability.IsSupportEnvironmentVariableToPath == false)
                {
                    return;
                }

                var outputPath = Path.Combine(_context.TempDirPath, "change.nsp");
                var replacer = _context.DiContainer.GetInstance<NspReplacer>();
                var option = new NspReplacer.ReplaceOption
                {
                    InputNspPath = NspFilePath,
                    CancellationToken = default(CancellationToken),
                    Project = project,
                    OutputNspPath = outputPath,
                };

                var htmlDocumentPath = Path.Combine(TestContext.TestDataDirPath, "icon_testdata");
                project.Meta.Application.IsReplaceHtmlDocumentPath = true;
                project.Meta.Application.HtmlDocumentPath = project.ToRealtivePath(htmlDocumentPath);
                project.Meta.Application.HtmlDocumentPath.IsExpandEnvironmentVariable = isExpandEnvVar;

                var urlsText = "http://example.com\r\nhoge\r\nfuga";
                var accessibleUrlsPath = Path.Combine(_context.TempDirPath, "foo.txt");
                File.WriteAllText(accessibleUrlsPath, urlsText);
                project.Meta.Application.IsReplaceAccessibleUrlsFilePath = true;
                project.Meta.Application.AccessibleUrlsFilePath = project.ToRealtivePath(accessibleUrlsPath);
                project.Meta.Application.AccessibleUrlsFilePath.IsExpandEnvironmentVariable = isExpandEnvVar;

                var legalInformationFilePath = Path.Combine(TestContext.TestDataDirPath,
                    @"project_testdata\legal_info_0100000000001000_75e7.zip");
                project.Meta.Application.IsReplaceLegalInformationFilePath = true;
                project.Meta.Application.LegalInformationFilePath = project.ToRealtivePath(legalInformationFilePath);
                project.Meta.Application.LegalInformationFilePath.IsExpandEnvironmentVariable = isExpandEnvVar;

                var r = await replacer.ReplaceNspAsync(option).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.Ok, r.Result);
            }
        }
    }

    public class NspReplaerTestRatings : NspReplacerTestBase
    {
        public NspReplaerTestRatings(BuildTestAppNspFixture fixture) : base(fixture)
        {

        }

        [Fact]
        public async Task PartialImportNmeta()
        {
            var ratingNmeta = Path.Combine(_context.TempDirPath, "rating.nmeta");

            using (var project = new Project())
            {
                var i = 0;
                foreach (var r in project.Meta.Application.Ratings.OrderBy(x => x.Organization))
                {
                    r.IsUse = true;
                    r.Age = i;
                    i++;
                }
                project.OutputAppMetaXmlFileForAuthoringTool(ratingNmeta);
            }

            using (var project = Project.Import(_context.DiContainer, ImportableFileType.Nsp, NspFilePath))
            {
                project.ImportPartial(ratingNmeta);

                {
                    var i = 0;
                    foreach (var r in project.Meta.Application.Ratings.OrderBy(x => x.Organization))
                    {
                        Assert.True(r.IsUse);
                        Assert.Equal(i, r.Age);
                        i++;
                    }
                }

                {
                    var replacer = _context.DiContainer.GetInstance<NspReplacer>();
                    var r = await replacer.ReplaceNspAsync(new NspReplacer.ReplaceOption
                    {
                        InputNspPath = NspFilePath,
                        CancellationToken = default(CancellationToken),
                        OutputNspPath = $"{NspFilePath}.changed.nsp",
                        Project = project
                    });
                    Assert.Equal(NspHandleResult.Ok, r.Result);
                }
            }

            using (var project =
                Project.Import(_context.DiContainer, ImportableFileType.Nsp, $"{NspFilePath}.changed.nsp"))
            {
                var i = 0;
                foreach (var r in project.Meta.Application.Ratings.OrderBy(x => x.Organization))
                {
                    Assert.True(r.IsUse);
                    Assert.Equal(i, r.Age);
                    i++;
                }
            }
        }
    }
}
