﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Threading.Tasks;
using Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class ImporterTest : IDisposable, IClassFixture<BuildTestAppNspFixture>
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        private readonly BuildTestAppNspFixture _fixture;

        public ImporterTest(BuildTestAppNspFixture fixture)
        {
            _fixture = fixture.Initialize();
        }

        [Fact]
        // ReSharper disable once FunctionComplexityOverflow
        public void FromMetaFile()
        {
            _context.DiContainer.GetInstance<AppProfile>().AppMode = AppModeType.ApplicationMeta;

            var projectFilePath = Path.Combine(TestContext.TestDataDirPath, @"project_testdata\test.narp");
            var metaFilePath = Path.Combine(TestContext.TestDataDirPath, @"project_testdata\exported.nmeta");

            var expected = Project.Import(_context.DiContainer, ImportableFileType.Project, projectFilePath);
            var actual = Project.Import(_context.DiContainer, ImportableFileType.Meta, metaFilePath);

            Assert.NotNull(actual);
            Assert.NotNull(actual.Meta);
            Assert.Equal(actual.ProgramCodeDirectoryPath, string.Empty);
            Assert.Equal(actual.ProgramDataDirectoryPath, string.Empty);
            Assert.Null(actual.Meta.Application.HtmlDocumentPath.Path);
            Assert.Null(actual.Meta.Application.AccessibleUrlsFilePath.Path);
            Assert.Null(actual.Meta.Application.LegalInformationFilePath.Path);

            Assert.NotNull(actual.Meta.Core);
            Assert.Equal(actual.Meta.Core.Name, expected.Meta.Core.Name);
            Assert.Equal(actual.Meta.Core.Is64BitInstruction, expected.Meta.Core.Is64BitInstruction);
            Assert.Equal(actual.Meta.Core.ApplicationId, expected.Meta.Core.ApplicationId);
            Assert.Equal(actual.Meta.Core.ApplicationIdHex, expected.Meta.Core.ApplicationIdHex);
            Assert.Equal(actual.Meta.Core.MainThreadStackSize, expected.Meta.Core.MainThreadStackSize);
            Assert.Equal(actual.Meta.Core.MainThreadStackSizeHex, expected.Meta.Core.MainThreadStackSizeHex);
            Assert.Equal(actual.Meta.Core.ProcessAddressSpace, expected.Meta.Core.ProcessAddressSpace);
            Assert.Equal(actual.Meta.Core.IsProcessAddressSpace64Bit, expected.Meta.Core.IsProcessAddressSpace64Bit);

            Assert.NotNull(actual.Meta.Application);
            Assert.Equal(actual.Meta.Application.Titles.Count, expected.Meta.Application.Titles.Count);

            for (var i = 0; i != actual.Meta.Application.Titles.Count; ++i)
            {
                Assert.Equal(actual.Meta.Application.Titles[i].Language, expected.Meta.Application.Titles[i].Language);
                Assert.Equal(actual.Meta.Application.Titles[i].Name, expected.Meta.Application.Titles[i].Name);
                Assert.Equal(actual.Meta.Application.Titles[i].Publisher, expected.Meta.Application.Titles[i].Publisher);

                Assert.False(actual.Meta.Application.Titles[i].IsReplaceIcon);
                Assert.False(actual.Meta.Application.Titles[i].IsReplaceNxIcon);

                Assert.Null(actual.Meta.Application.Titles[i].IconFilePath.Path);
                Assert.Null(actual.Meta.Application.Titles[i].NxIconFilePath.Path);
            }

            Assert.Equal(actual.Meta.Application.ReleaseVersion, expected.Meta.Application.ReleaseVersion);
            Assert.Equal(actual.Meta.Application.DisplayVersion, expected.Meta.Application.DisplayVersion);
            Assert.Equal(actual.Meta.Application.StartupUserAccount, expected.Meta.Application.StartupUserAccount);
            Assert.Equal(actual.Meta.Application.ParentalControl, expected.Meta.Application.ParentalControl);
            Assert.Equal(actual.Meta.Application.SupportedLanguages.Count, expected.Meta.Application.SupportedLanguages.Count);

            for (var i = 0; i != actual.Meta.Application.SupportedLanguages.Count; ++i)
            {
                Assert.Equal(actual.Meta.Application.SupportedLanguages[i].Language, expected.Meta.Application.SupportedLanguages[i].Language);
                Assert.Equal(actual.Meta.Application.SupportedLanguages[i].IsSupported, expected.Meta.Application.SupportedLanguages[i].IsSupported);
            }

            Assert.Equal(actual.Meta.Application.PresenceGroupId, expected.Meta.Application.PresenceGroupId);
            Assert.Equal(actual.Meta.Application.PresenceGroupIdHex, expected.Meta.Application.PresenceGroupIdHex);
            Assert.Equal(actual.Meta.Application.IsAllowScreenshot, expected.Meta.Application.IsAllowScreenshot);
            Assert.Equal(actual.Meta.Application.DataLossConfirmation, expected.Meta.Application.DataLossConfirmation);
            Assert.Equal(actual.Meta.Application.SaveDataSize, expected.Meta.Application.SaveDataSize);
            Assert.Equal(actual.Meta.Application.SaveDataSizeHex, expected.Meta.Application.SaveDataSizeHex);
            Assert.Equal(actual.Meta.Application.IsUseSaveData, expected.Meta.Application.IsUseSaveData);
            Assert.Equal(actual.Meta.Application.SaveDataJournalSize, expected.Meta.Application.SaveDataJournalSize);
            Assert.Equal(actual.Meta.Application.SaveDataJournalSizeHex, expected.Meta.Application.SaveDataJournalSizeHex);
            Assert.Equal(actual.Meta.Application.IsSpecifiedSaveDataJournal, expected.Meta.Application.IsSpecifiedSaveDataJournal);
            Assert.Equal(actual.Meta.Application.ApplicationErrorCodeCategory, expected.Meta.Application.ApplicationErrorCodeCategory);
            Assert.Equal(actual.Meta.Application.IsUseApplicationErrorCode, expected.Meta.Application.IsUseApplicationErrorCode);
            Assert.Equal(actual.Meta.Application.LogoType, expected.Meta.Application.LogoType);
            Assert.Equal(actual.Meta.Application.IsDemo, expected.Meta.Application.IsDemo);

            Assert.Equal(actual.Meta.Application.LocalCommunicationIds.Count, expected.Meta.Application.LocalCommunicationIds.Count);

            for (var i = 0; i != actual.Meta.Application.LocalCommunicationIds.Count; ++i)
                Assert.Equal(actual.Meta.Application.LocalCommunicationIds[i].Id, expected.Meta.Application.LocalCommunicationIds[i].Id);

            Assert.Equal(actual.Meta.Application.Ratings.Count, expected.Meta.Application.Ratings.Count);

            for (var i = 0; i != actual.Meta.Application.Ratings.Count; ++i)
            {
                Assert.Equal(actual.Meta.Application.Ratings[i].Age, expected.Meta.Application.Ratings[i].Age);
                Assert.Equal(actual.Meta.Application.Ratings[i].Organization, expected.Meta.Application.Ratings[i].Organization);
            }

            Assert.Equal(actual.Meta.CardSpec.Size, expected.Meta.CardSpec.Size);
            Assert.Equal(actual.Meta.CardSpec.ClockRate, expected.Meta.CardSpec.ClockRate);
        }

        [Fact]
        public void FromMetaFile_FileNotFound()
        {
            Assert.Throws<FileNotFoundException>(() => Project.Import(_context.DiContainer, ImportableFileType.Meta, "AAAAA"));
        }

        [Fact]
        public void FromProjectFile()
        {
            var projectFilePath = Path.Combine(TestContext.TestDataDirPath, @"project_testdata\test.narp");

            var actual = Project.Import(_context.DiContainer, ImportableFileType.Project, projectFilePath);
            Assert.NotNull(actual);
        }

        [Fact]
        public void FromProjectFile_FileNotFound()
        {
            Assert.Throws<FileNotFoundException>(() => Project.Import(_context.DiContainer, ImportableFileType.Project, "AAAAA"));
        }

        [Fact]
        public void FromProjectFile_FileFormatException()
        {
            var projectFilePath = Path.Combine(TestContext.TestDataDirPath, @"project_testdata\broken.narp");

            Assert.Throws<FileFormatException>(() => Project.Import(_context.DiContainer, ImportableFileType.Project, projectFilePath));
        }

        [Fact]
        public async Task FromNspFile()
        {
            var input = new NspBuilder.ExportOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, "test.nsp"),
                OutputType = OutputTypes.Nsp,
                InputMetaFilePath = TestContext.DefaultMetaFilePath,
                InputDescFilePath = TestContext.DefaultDescFilePath,
                InputProgramCodePath = _fixture.CodeDirPath,
                InputProgramDataPath = _fixture.DataDirPath
            };

            var builder = _context.DiContainer.GetInstance<NspBuilder>();
            var i = await builder.ExportNspAsync(input).ConfigureAwait(false);
            Assert.Equal(NspHandleResult.Ok, i.Result);

            var actual = Project.Import(_context.DiContainer, ImportableFileType.Nsp, input.OutputPath);
            Assert.NotNull(actual);
        }

        [Fact]
        public void FromNspFile_FileNotFound()
        {
            Assert.Throws<FileNotFoundException>(() => Project.Import(_context.DiContainer, ImportableFileType.Nsp, "AAAAA"));
        }

        [Fact]
        public void FromNspFile_FileFormatException()
        {
            var dummyNspFilePath = Path.Combine(TestContext.TestDataDirPath, @"project_testdata\broken.narp");

            Assert.Throws<NspHandleErrorException>(() => Project.Import(_context.DiContainer, ImportableFileType.Nsp, dummyNspFilePath));
        }

        // インポート時に自動で調整が行われるプロパティのテスト
        [Theory]
        [InlineData(true)]
        [InlineData(false)]
        public void IsUseParam(bool isUse)
        {
            _context.DiContainer.GetInstance<AppProfile>().AppMode = AppModeType.ApplicationMeta;

            var metaFilePath = Path.Combine(TestContext.TestDataDirPath, $@"project_testdata\isuse_{isUse}.nmeta");
            var project = Project.Import(_context.DiContainer, ImportableFileType.Meta, metaFilePath);

            Assert.Equal(isUse, project.Meta.Application.IsUseApplicationErrorCode);
            Assert.Equal(isUse, project.Meta.Application.IsUseSaveData);
            Assert.Equal(isUse, project.Meta.Application.IsSpecifiedSaveDataJournal);
            Assert.Equal(isUse, project.Meta.Application.IsUsePresenceGroupIdAppId);
            Assert.Equal(isUse, project.Meta.CardSpec.IsAutomaticSettingSize);
            Assert.Equal(isUse, project.Meta.CardSpec.IsAutomaticSettingClockRate);

        }

        [Theory]
        [InlineData(nameof(Application.IsReplaceHtmlDocumentPath))]
        [InlineData(nameof(Application.IsReplaceAccessibleUrlsFilePath))]
        [InlineData(nameof(Application.IsReplaceLegalInformationFilePath))]
        public void ReplaceFlags(string propertyName)
        {
            Func<Application, bool> getter = x => (bool)x.GetType().GetProperty(propertyName).GetValue(x);
            Action<Application, bool> setter = (x, value) => x.GetType().GetProperty(propertyName).SetValue(x, value);

            using (var project = _context.DiContainer.GetInstance<Project>())
            {
                Assert.False(getter(project.Meta.Application));
            }

            _context.CreateFile("flags.nmeta");
            var tempMetaFile = Path.Combine(_context.TempDirPath, "flags.nmeta");

            using (var app = _context.DiContainer.GetInstance<App>())
            {
                setter(app.Project.Meta.Application, true);
                app.Save(ExportableFileType.Project, tempMetaFile);

                var project = Project.Import(_context.DiContainer, ImportableFileType.Project, tempMetaFile);
                Assert.True(getter(project.Meta.Application));
            }
        }
    }
}
