﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Globalization;
using System.IO;
using System.Linq;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class ConfigTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var c = new Config())
            {
                Assert.NotNull(c.MruProjects);
                Assert.Equal(0, c.MruProjects.Count);

                Assert.NotNull(c.MruMetas);
                Assert.Equal(0, c.MruMetas.Count);

                Assert.NotNull(c.MruNsps);
                Assert.Equal(0, c.MruNsps.Count);
            }
        }

        [Fact]
        public void Destroy()
        {
            using (var c = new Config())
            {
                var path = Path.Combine(_context.TempDirPath, "config.yaml");

                c.Destory(path);

                Assert.True(File.Exists(path));
            }
        }


        [Fact]
        public void Culture()
        {
            using (var c = new Config())
            {
                // ReSharper disable once ConvertIfStatementToConditionalTernaryExpression
                if (CultureInfo.CurrentUICulture.TwoLetterISOLanguageName == "ja")
                    Assert.Equal("ja", c.Culture);
                else
                    Assert.Equal("en", c.Culture);

                c.Culture = "XX";

                Assert.Equal("XX", c.Culture);
            }
        }

        [Fact]
        public void Initialize()
        {
            using (var c = new Config())
                c.Initialize();
        }

        [Fact]
        public void Lists()
        {
            using (var c = new Config())
            {
                c.MruProjects = new ObservableCollection<MruItem>();
                c.MruMetas = new ObservableCollection<MruItem>();
                c.MruNsps = new ObservableCollection<MruItem>();
            }
        }

        public static IEnumerable<object> ImportableFileTypes =>
            Enum.GetValues(typeof(ImportableFileType)).Cast<object>().Select(x => new[] {x});

        [Theory]
        [MemberData(nameof(ImportableFileTypes))]
        public void FirstOrDefaut(ImportableFileType type)
        {
            using (var c = new Config())
            {
                Assert.Null(c.FirstOrDefault(type, "XXX"));
            }
        }

        [Theory]
        [MemberData(nameof(ImportableFileTypes))]
        public void AddToMruList(ImportableFileType type)
        {
            using (var c = new Config())
            {
                c.AddToMruList(type, $"c:\\{type}");
            }
        }

        [Theory]
        [MemberData(nameof(ImportableFileTypes))]
        public void AddToMruList_TopOrder(ImportableFileType type)
        {
            using (var c = new Config())
            {
                c.AddToMruList(type, $"c:\\{type}_A");
                c.AddToMruList(type, $"c:\\{type}_B");
                c.AddToMruList(type, $"c:\\{type}_C");

                var l = c.TargetMruList(type);

                Assert.Equal(3, l.Count);
                Assert.Equal($"c:\\{type}_C", l[0].FilePath);
                Assert.Equal($"c:\\{type}_B", l[1].FilePath);
                Assert.Equal($"c:\\{type}_A", l[2].FilePath);
            }
        }

        [Theory]
        [MemberData(nameof(ImportableFileTypes))]
        public void AddToMruList_Pin(ImportableFileType type)
        {
            using (var c = new Config())
            {
                var l = c.TargetMruList(type);

                c.AddToMruList(type, $"c:\\{type}_A");
                l[0].IsPinned = true;
                c.AddToMruList(type, $"c:\\{type}_B");
                c.AddToMruList(type, $"c:\\{type}_C");

                Assert.Equal(3, l.Count);
                Assert.Equal($"c:\\{type}_A", l[0].FilePath);
                Assert.Equal($"c:\\{type}_C", l[1].FilePath);
                Assert.Equal($"c:\\{type}_B", l[2].FilePath);
            }
        }

        [Theory]
        [MemberData(nameof(ImportableFileTypes))]
        public void AddToMruList_Same(ImportableFileType type)
        {
            using (var c = new Config())
            {
                c.AddToMruList(type, $"c:\\{type}_A");
                c.AddToMruList(type, $"c:\\{type}_A");
                c.AddToMruList(type, $"c:\\{type}_A");

                var l = c.TargetMruList(type);

                Assert.Equal(1, l.Count);
                Assert.Equal($"c:\\{type}_A", l[0].FilePath);
            }
        }

        [Theory]
        [MemberData(nameof(ImportableFileTypes))]
        public void RemoveFromMruList(ImportableFileType type)
        {
            using (var c = new Config())
            {
                c.AddToMruList(type, $"c:\\{type}");
                Assert.NotNull(c.FirstOrDefault(type, $"c:\\{type}"));

                c.RemoveFromMruList(type, $"c:\\{type}");
                Assert.Null(c.FirstOrDefault(type, $"c:\\{type}"));
            }
        }

        [Theory]
        [MemberData(nameof(ImportableFileTypes))]
        public void RemoveFromMruList_null(ImportableFileType type)
        {
            using (var c = new Config())
            {
                c.RemoveFromMruList(type, "XXX");
            }
        }

        [Theory]
        [MemberData(nameof(ImportableFileTypes))]
        public void LaterPinned(ImportableFileType type)
        {
            using (var c = new Config())
            {
                var l = c.TargetMruList(type);

                c.AddToMruList(type, $"c:\\{type}_A");
                c.AddToMruList(type, $"c:\\{type}_B");
                c.AddToMruList(type, $"c:\\{type}_C");

                Assert.Equal(3, l.Count);

                Assert.Equal($"c:\\{type}_C", l[0].FilePath);
                Assert.Equal($"c:\\{type}_B", l[1].FilePath);
                Assert.Equal($"c:\\{type}_A", l[2].FilePath);

                l[1].IsPinned = true;

                Assert.Equal($"c:\\{type}_B", l[0].FilePath);
                Assert.Equal($"c:\\{type}_C", l[1].FilePath);
                Assert.Equal($"c:\\{type}_A", l[2].FilePath);
            }
        }
    }
}
