﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;
using System.Reflection;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using Nintendo.G3dTool.Entities;
using _3dToolsTestUtility;

namespace G3dLibraryTest
{
    [TestClass]
    public class CreateEmptyEntityTest
    {
        public delegate void PreWriteProcess(Nintendo.G3dTool.Entities.IntermediateFile file);

        [TestMethod]
        public void TestWriteReadEmptyEntityWithoutSchemaCheck()
        {
            WriteReadEmptyEntityWithoutSchemaCheck<materialType>("test.fmtb");
            WriteReadEmptyEntityWithoutSchemaCheck<modelType>("test.fmdb", file =>
            {
                var rootEntity = file.GetRootEntity<Model>();
                var dummy = new Bone();
                rootEntity.Skeleton.RootBone = dummy;
            });
            WriteReadEmptyEntityWithoutSchemaCheck<textureType>("test.ftxb", file =>
            {
                var rootEntity = file.GetRootEntity<Texture>();
                var texDataStream = new StreamInt();
                rootEntity.Streams.Add(texDataStream);
                rootEntity.TextureInfo.Stream = texDataStream;
            });
            WriteReadEmptyEntityWithoutSchemaCheck<skeletal_animType>("test.fskb", file =>
            {
                var rootEntity = file.GetRootEntity<SkeletalAnim>();
                var dummy = new BoneAnim();
                rootEntity.BoneAnims.Add(dummy);
            });
            WriteReadEmptyEntityWithoutSchemaCheck<material_animType>("test.fmab");
            WriteReadEmptyEntityWithoutSchemaCheck<bone_visibility_animType>("test.fvbb");
            WriteReadEmptyEntityWithoutSchemaCheck<shape_animType>("test.fshb");
            WriteReadEmptyEntityWithoutSchemaCheck<shape_animType>("test.fsnb");
            WriteReadEmptyEntityWithoutSchemaCheck<shader_configType>("test.fsca", file =>
            {
                var rootEntity = file.GetRootEntity<ShaderConfig>();
                var dummy = new Shader();
                dummy.VertexShader.Path = "dummy.glsl";
                rootEntity.Shaders.Add(dummy);
            });
            WriteReadEmptyEntityWithoutSchemaCheck<shader_definitionType>("test.fsdb", file =>
            {
                var rootEntity = file.GetRootEntity<ShaderDefinition>();
                var dummyShadingModel = new ShadingModel();
                rootEntity.ShadingModels.Add(dummyShadingModel);
                var dummySource = new ShaderSrc();
                rootEntity.ShaderSrcs.Add(dummySource);
                var stream = new StreamWstring();
                dummySource.Stream = stream;
                rootEntity.Streams.Add(stream);
            });
            WriteReadEmptyEntityWithoutSchemaCheck<shader_variationType>("test.fsva", file =>
            {
                var rootEntity = file.GetRootEntity<ShaderVariation>();
                var dummy = new TargetShader();
                dummy.ShaderPrograms.Add(new ShaderProgram());
                rootEntity.TargetShaders.Add(dummy);
            });
            WriteReadEmptyEntityWithoutSchemaCheck<combiner_shaderType>("test.fcmb", file =>
            {
            });
        }

        [TestMethod]
        public void TestWriteReadEmptyEntity()
        {
            WriteReadEmptyEntity<materialType>("test.fmtb", file =>
            {
                var rootEntity = file.GetRootEntity<Material>();
                rootEntity.Name = "dummy";
            });

            WriteReadEmptyEntity<modelType>("test.fmdb", file =>
            {
                var rootEntity = file.GetRootEntity<Model>();
                var dummy = new Bone()
                {
                    Name = "dummy",
                };
                rootEntity.Skeleton.RootBone = dummy;
            });
            WriteReadEmptyEntity<textureType>("test.ftxb");
            WriteReadEmptyEntity<skeletal_animType>("test.fskb", file =>
            {
                var rootEntity = file.GetRootEntity<SkeletalAnim>();
                var dummy = new BoneAnim()
                {
                    Name = "dummy"
                };
                rootEntity.BoneAnims.Add(dummy);
            });
            WriteReadEmptyEntity<material_animType>("test.fmab");
            WriteReadEmptyEntity<bone_visibility_animType>("test.fvbb");
            WriteReadEmptyEntity<shape_animType>("test.fshb");
            WriteReadEmptyEntity<shape_animType>("test.fsnb");
            WriteReadEmptyEntity<shader_configType>("test.fsca", file =>
            {
                var rootEntity = file.GetRootEntity<ShaderConfig>();
                var dummy = new Shader()
                {
                    Name = "dummy"
                };
                dummy.VertexShader.Path = "dummy.glsl";
                rootEntity.Shaders.Add(dummy);
            });
            WriteReadEmptyEntity<shader_definitionType>("test.fsdb", file =>
            {
                var rootEntity = file.GetRootEntity<ShaderDefinition>();
                rootEntity.ShaderDefinitionInfo.ConfigPath = "dummy";
                var dummyShadingModel = new ShadingModel()
                {
                    Name = "dummy"
                };
                rootEntity.ShadingModels.Add(dummyShadingModel);
                var dummySource = new ShaderSrc()
                {
                    IncludePath = "dummy",
                    Path = "dummy",
                };
                rootEntity.ShaderSrcs.Add(dummySource);
            });
            WriteReadEmptyEntity<shader_variationType>("test.fsva", file =>
            {
                var rootEntity = file.GetRootEntity<ShaderVariation>();
                rootEntity.ShaderVariationInfo.ShaderArchive = "dummy";
                var dummy = new TargetShader()
                {
                    ShadingModelName = "dummy"
                };
                dummy.ShaderPrograms.Add(new ShaderProgram());
                rootEntity.TargetShaders.Add(dummy);
            });
            WriteReadEmptyEntity<combiner_shaderType>("test.fcmb", file =>
            {
            });
        }

        private void WriteReadEmptyEntityWithoutSchemaCheck<T>(string filePath, PreWriteProcess preWriteProcess = null)
            where T : class
        {
            G3dParallel.Job = 1;

            {
                Nintendo.G3dTool.Entities.IntermediateFile file = new Nintendo.G3dTool.Entities.IntermediateFile(GetKind<T>());
                if (preWriteProcess != null)
                {
                    preWriteProcess(file);
                }
                IfWriteUtility.WriteIntermediateFile(file, filePath, null);
            }

            // 書き出したデータが読めるかテスト
            {
                var fileData = IfReadUtility.Read(filePath, null);
                Assert.IsNotNull(fileData.Item as T);
                var file = new IntermediateFile(fileData);
            }
        }

        private void WriteReadEmptyEntity<T>(string filePath, PreWriteProcess preWriteProcess = null)
            where T : class
        {
            G3dParallel.Job = 1;

            {
                Nintendo.G3dTool.Entities.IntermediateFile file = new Nintendo.G3dTool.Entities.IntermediateFile(GetKind<T>());
                if (preWriteProcess != null)
                {
                    preWriteProcess(file);
                }
                IfWriteUtility.WriteIntermediateFile(file, filePath, IoUtility.GetXsdBasePath());
            }

            // 書き出したデータが読めるかテスト
            {
                var file = IfReadUtility.ReadIntermediateFile(filePath, IoUtility.GetXsdBasePath());
            }
        }

        private IntermediateFileKind GetKind<T>()
            where T : class
        {
            KeyValuePair<Type, IntermediateFileKind>[] typeKindList = new KeyValuePair<Type, IntermediateFileKind>[]
                {
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(materialType), IntermediateFileKind.Material),
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(modelType), IntermediateFileKind.Model),
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(textureType), IntermediateFileKind.Texture),
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(skeletal_animType), IntermediateFileKind.SkeletalAnim),
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(material_animType), IntermediateFileKind.MaterialAnim),
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(bone_visibility_animType), IntermediateFileKind.BoneVisibilityAnim),
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(shape_animType), IntermediateFileKind.ShapeAnim),
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(scene_animType), IntermediateFileKind.SceneAnim),
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(shader_configType), IntermediateFileKind.ShaderConfig),
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(shader_definitionType), IntermediateFileKind.ShaderDefinition),
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(shader_variationType), IntermediateFileKind.ShaderVariation),
                    new KeyValuePair<Type, IntermediateFileKind>(typeof(combiner_shaderType), IntermediateFileKind.CombinerShader),
                };

            foreach (var typeKind in typeKindList)
            {
                if (typeof(T) == typeKind.Key)
                {
                    return typeKind.Value;
                }
            }

            throw new NotImplementedException();
        }
    }
}
