﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nnt/gtest/gtest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnd/tmp451/tmp451.h>

namespace nnt { namespace tmp451 {

struct PorValues
{
    nnd::tmp451::Temperature        pin1RemoteLimit;
    nnd::tmp451::Temperature        pin1LocalLimit;
    nnd::tmp451::Temperature        pin2RemoteLowLimit;
    nnd::tmp451::Temperature        pin2RemoteHighLimit;
    nnd::tmp451::Temperature        pin2LocalLowLimit;
    nnd::tmp451::Temperature        pin2LocalHighLimit;
    nnd::tmp451::DetectionMode      pin2Mode;
    uint8_t                         thermHysterisis;
    nnd::tmp451::AlertConsecutive   alertConsecutive;
};

const PorValues g_PorValues = {
        { 0x6C, 0x00 }, { 0x55, 0x00 },
        { 0x00, 0x00 }, { 0x55, 0x00 }, { 0x00, 0x00 }, { 0x55, 0x00 },
        nnd::tmp451::DetectionMode::Alert,
        0x0a,
        nnd::tmp451::AlertConsecutive::Consecutive1
};

const PorValues g_ChangeValues = {
        { 0x60, 0x00 }, { 0x55, 0x00 },
        { 0x05, nnd::tmp451::DecimalUnit * 1 }, { 0x50, nnd::tmp451::DecimalUnit * 1 }, { 0x05, 0x00 }, { 0x50, 0x00 },
        nnd::tmp451::DetectionMode::Alert,
        0x0a,
        nnd::tmp451::AlertConsecutive::Consecutive1
};


// 計測中に設定の取得が可能なことを確認するテスト
TEST(InterruptSettingTest, Get)
{
    using namespace nnd::tmp451;
    Initialize();

    // 計測前が初期値と同じかを確認
    auto result = SetMeasurementMode(MeasurementMode::Continuous);
    EXPECT_TRUE(result.IsSuccess());

    // 計測中
    PorValues values;
    result = GetPin1DetectionSettings(&values.pin1RemoteLimit, &values.pin1LocalLimit);
    EXPECT_TRUE(result.IsSuccess());
    result = GetPin2DetectionSettings(&values.pin2RemoteLowLimit, &values.pin2RemoteHighLimit,
            &values.pin2LocalLowLimit, &values.pin2LocalHighLimit, &values.pin2Mode);
    EXPECT_TRUE(result.IsSuccess());
    result = GetThermHysterisis(&values.thermHysterisis);
    EXPECT_TRUE(result.IsSuccess());
    result = GetConsecutiveAlert(&values.alertConsecutive);
    EXPECT_TRUE(result.IsSuccess());

    // 初期値の確認
    EXPECT_EQ(g_PorValues.pin1RemoteLimit.integer, values.pin1RemoteLimit.integer);
    EXPECT_EQ(g_PorValues.pin1RemoteLimit.decimal, values.pin1RemoteLimit.decimal);
    EXPECT_EQ(g_PorValues.pin1LocalLimit.integer, values.pin1LocalLimit.integer);
    EXPECT_EQ(g_PorValues.pin1LocalLimit.decimal, values.pin1LocalLimit.decimal);
    EXPECT_EQ(g_PorValues.pin2RemoteLowLimit.integer, values.pin2RemoteLowLimit.integer);
    EXPECT_EQ(g_PorValues.pin2RemoteLowLimit.decimal, values.pin2RemoteLowLimit.decimal);
    EXPECT_EQ(g_PorValues.pin2RemoteHighLimit.integer, values.pin2RemoteHighLimit.integer);
    EXPECT_EQ(g_PorValues.pin2RemoteHighLimit.decimal, values.pin2RemoteHighLimit.decimal);
    EXPECT_EQ(g_PorValues.pin2LocalLowLimit.integer, values.pin2LocalLowLimit.integer);
    EXPECT_EQ(g_PorValues.pin2LocalLowLimit.decimal, values.pin2LocalLowLimit.decimal);
    EXPECT_EQ(g_PorValues.pin2LocalHighLimit.integer, values.pin2LocalHighLimit.integer);
    EXPECT_EQ(g_PorValues.pin2LocalHighLimit.decimal, values.pin2LocalHighLimit.decimal);
    EXPECT_EQ(g_PorValues.pin2Mode, values.pin2Mode);
    EXPECT_EQ(g_PorValues.thermHysterisis, values.thermHysterisis);
    EXPECT_EQ(g_PorValues.alertConsecutive, values.alertConsecutive);

    Finalize();
}

// 設定の変更ができることを確認するテスト
TEST(InterruptSettingTest, Set)
{
    using namespace nnd::tmp451;
    Initialize();

    // 設定を変更
    auto result = SetPin1DetectionSettings(g_ChangeValues.pin1RemoteLimit, g_ChangeValues.pin1LocalLimit);
    EXPECT_TRUE(result.IsSuccess());
    result = SetPin2DetectionSettings(g_ChangeValues.pin2RemoteLowLimit, g_ChangeValues.pin2RemoteHighLimit,
            g_ChangeValues.pin2LocalLowLimit, g_ChangeValues.pin2LocalHighLimit, g_ChangeValues.pin2Mode);
    EXPECT_TRUE(result.IsSuccess());
    result = SetThermHysterisis(g_ChangeValues.thermHysterisis);
    EXPECT_TRUE(result.IsSuccess());
    result = SetConsecutiveAlert(g_ChangeValues.alertConsecutive);
    EXPECT_TRUE(result.IsSuccess());

    // 設定が変更されているかの確認
    PorValues values;
    result = GetPin1DetectionSettings(&values.pin1RemoteLimit, &values.pin1LocalLimit);
    EXPECT_TRUE(result.IsSuccess());
    result = GetPin2DetectionSettings(&values.pin2RemoteLowLimit, &values.pin2RemoteHighLimit,
            &values.pin2LocalLowLimit, &values.pin2LocalHighLimit, &values.pin2Mode);
    EXPECT_TRUE(result.IsSuccess());
    result = GetThermHysterisis(&values.thermHysterisis);
    EXPECT_TRUE(result.IsSuccess());
    result = GetConsecutiveAlert(&values.alertConsecutive);
    EXPECT_TRUE(result.IsSuccess());

    // 比較
    EXPECT_EQ(g_ChangeValues.pin1RemoteLimit.integer, values.pin1RemoteLimit.integer);
    EXPECT_EQ(g_ChangeValues.pin1RemoteLimit.decimal, values.pin1RemoteLimit.decimal);
    EXPECT_EQ(g_ChangeValues.pin1LocalLimit.integer, values.pin1LocalLimit.integer);
    EXPECT_EQ(g_ChangeValues.pin1LocalLimit.decimal, values.pin1LocalLimit.decimal);
    EXPECT_EQ(g_ChangeValues.pin2RemoteLowLimit.integer, values.pin2RemoteLowLimit.integer);
    EXPECT_EQ(g_ChangeValues.pin2RemoteLowLimit.decimal, values.pin2RemoteLowLimit.decimal);
    EXPECT_EQ(g_ChangeValues.pin2RemoteHighLimit.integer, values.pin2RemoteHighLimit.integer);
    EXPECT_EQ(g_ChangeValues.pin2RemoteHighLimit.decimal, values.pin2RemoteHighLimit.decimal);
    EXPECT_EQ(g_ChangeValues.pin2LocalLowLimit.integer, values.pin2LocalLowLimit.integer);
    EXPECT_EQ(g_ChangeValues.pin2LocalLowLimit.decimal, values.pin2LocalLowLimit.decimal);
    EXPECT_EQ(g_ChangeValues.pin2LocalHighLimit.integer, values.pin2LocalHighLimit.integer);
    EXPECT_EQ(g_ChangeValues.pin2LocalHighLimit.decimal, values.pin2LocalHighLimit.decimal);
    EXPECT_EQ(g_ChangeValues.pin2Mode, values.pin2Mode);
    EXPECT_EQ(g_ChangeValues.thermHysterisis, values.thermHysterisis);
    EXPECT_EQ(g_ChangeValues.alertConsecutive, values.alertConsecutive);

    Finalize();
}

// 計測中に設定の変更ができないことを確認するテスト
TEST(InterruptSettingTest, SetError)
{
    using namespace nnd::tmp451;
    Initialize();

    // 計測開始
    auto result = SetMeasurementMode(MeasurementMode::Continuous);
    EXPECT_TRUE(result.IsSuccess());

    // 計測中はエラーになるはず
    // 設定を変更
    result = SetPin1DetectionSettings(g_ChangeValues.pin1RemoteLimit, g_ChangeValues.pin1LocalLimit);
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidState, result);
    result = SetPin2DetectionSettings(g_ChangeValues.pin2RemoteLowLimit, g_ChangeValues.pin2RemoteHighLimit,
            g_ChangeValues.pin2LocalLowLimit, g_ChangeValues.pin2LocalHighLimit, g_ChangeValues.pin2Mode);
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidState, result);
    result = SetThermHysterisis(g_ChangeValues.thermHysterisis);
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidState, result);
    result = SetConsecutiveAlert(g_ChangeValues.alertConsecutive);
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidState, result);

    Finalize();
}

}}
