﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/fs.h>
#include <nn/nifm.h>
#include <nn/nn_SdkLog.h>
#include <nn/oe.h>
#include <nn/time.h>
#include <nn/time/time_ApiForMenu.h>

#include <nn/time/time_ResultPrivate.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_SaveDataForDebug.h>

using namespace nn::time;

namespace nnt { namespace time { namespace savedata {
    const char MountName[] = "timetest";
    const char SteadyClockTimePoint[] = "timetest:/steadyclock";
}}};

namespace
{
    void ReadSteadyClockTimePointFromSaveData(nn::Result* pOutLastResult, SteadyClockTimePoint* pOut)
    {
        *pOutLastResult = nn::fs::MountSaveDataForDebug(nnt::time::savedata::MountName);
        NNT_ASSERT_RESULT_SUCCESS( *pOutLastResult );
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(nnt::time::savedata::MountName); };

        nn::fs::FileHandle handle;
        *pOutLastResult = nn::fs::OpenFile(&handle, nnt::time::savedata::SteadyClockTimePoint, nn::fs::OpenMode_Read);
        NNT_ASSERT_RESULT_SUCCESS( *pOutLastResult );
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };

        *pOutLastResult = nn::fs::ReadFile(handle, 0, pOut, sizeof(SteadyClockTimePoint));
        NNT_ASSERT_RESULT_SUCCESS( *pOutLastResult );
    }
}

TEST(TimeDebugFeature, SaveCurrentSteadyClockTimePoint)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug(nnt::time::savedata::MountName));
    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(nnt::time::savedata::MountName); };

    auto result = nn::fs::CreateFile( nnt::time::savedata::SteadyClockTimePoint, static_cast<int64_t>(sizeof(SteadyClockTimePoint)) );
    if(result.IsFailure() && !nn::fs::ResultPathAlreadyExists::Includes(result))
    {
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    {
        nn::fs::FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, nnt::time::savedata::SteadyClockTimePoint, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };

        SteadyClockTimePoint tp;
        NNT_ASSERT_RESULT_SUCCESS( StandardSteadyClock::GetCurrentTimePoint(&tp) );
        NNT_ASSERT_RESULT_SUCCESS( nn::fs::WriteFile(handle, 0, &tp, sizeof(tp), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)) );
    }
    NNT_ASSERT_RESULT_SUCCESS( nn::fs::CommitSaveData(nnt::time::savedata::MountName) );
}

// 直前に以下を行っている前提
//   - RunOnTarget testTime_DebugFeature.nsp -- --gtest_filter=TimeDebugFeature.SaveCurrentSteadyClockTimePoint
//   - RunOnTarget DevMenuCommand time reset-steady-clock
//   - ControlTarget --reset
TEST(TimeDebugFeature, ExpectSteadyClockWasReset)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    nn::Result result;
    SteadyClockTimePoint lastTimePoint;
    ReadSteadyClockTimePointFromSaveData(&result, &lastTimePoint);
    NNT_ASSERT_RESULT_SUCCESS(result);

    SteadyClockTimePoint currentTimePoint;
    NNT_ASSERT_RESULT_SUCCESS(StandardSteadyClock::GetCurrentTimePoint(&currentTimePoint));

    int64_t seconds;
    NNT_EXPECT_RESULT_FAILURE( nn::time::ResultNotComparable, nn::time::GetSpanBetween(&seconds, lastTimePoint, currentTimePoint) );
}

template <int64_t ElapsedMinutes>
void TestSteadyClockElapsed()
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    nn::Result result;
    SteadyClockTimePoint lastTimePoint;
    ReadSteadyClockTimePointFromSaveData(&result, &lastTimePoint);
    NNT_ASSERT_RESULT_SUCCESS(result);

    SteadyClockTimePoint currentTimePoint;
    NNT_ASSERT_RESULT_SUCCESS(StandardSteadyClock::GetCurrentTimePoint(&currentTimePoint));

    int64_t seconds;
    NNT_ASSERT_RESULT_SUCCESS( nn::time::GetSpanBetween(&seconds, lastTimePoint, currentTimePoint) );

    EXPECT_GE(seconds, ElapsedMinutes * 60);
}

// 直前に以下を行っている前提
//   - RunOnTarget testTime_DebugFeature.nsp -- --gtest_filter=TimeDebugFeature.SaveCurrentSteadyClockTimePoint
//   - RunOnTarget DevMenuCommand time add-steady-clock-test-offset 60 # 累積でOK
//   - ControlTarget --reset
TEST(TimeDebugFeature, ExpectElapsed60Minutes)
{
    TestSteadyClockElapsed<60>();
}

// 直前に以下を行っている前提
//   - RunOnTarget testTime_DebugFeature.nsp -- --gtest_filter=TimeDebugFeature.SaveCurrentSteadyClockTimePoint
//   - RunOnTarget DevMenuCommand time add-steady-clock-test-offset 120 # 累積でOK
//   - ControlTarget --reset
TEST(TimeDebugFeature, ExpectElapsed120Minutes)
{
    TestSteadyClockElapsed<120>();
}

// 直前に以下を行っている前提
//   - RunOnTarget DevMenuCommand set-network-clock --year 2016 --month 9 --day 6 --hour 11 --minute 32 # 適当な日時でOK
//   - RunOnTarget DevMenuCommand time set-auto-network-clock-ensure-policy disable
//   - ControlTarget --reset
//   - RunOnTarget DevMenuCommand time invalidate-network-clock
//   - ControlTarget --reset
TEST(TimeDebugFeature, ExpectNetworkClockIsInvalid)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    PosixTime current;
    NNT_EXPECT_RESULT_FAILURE( nn::time::ResultOffsetInvalid, StandardNetworkSystemClock::GetCurrentTime(&current) );
}

template <typename ClockType, int64_t TestDelayMinutes = 3>
void TestDateTime(int16_t year, int8_t month, int8_t day, int8_t hour, int8_t minute)
{
    const CalendarTime expectCalendar = { year, month, day, hour, minute, 0 };

    PosixTime expectPosix;
    int outCount;
    NNT_ASSERT_RESULT_SUCCESS( ToPosixTime(&outCount, &expectPosix, 1, expectCalendar) );
    ASSERT_NE(0, outCount);

    PosixTime currentPosix;
    NNT_ASSERT_RESULT_SUCCESS( ClockType::GetCurrentTime(&currentPosix) );

    LocationName location;
    GetDeviceLocationName(&location);
    NN_SDK_LOG("Current:%lld %s\n", currentPosix.value, location._value);

    ASSERT_GE(currentPosix.value, expectPosix.value);

    const nn::TimeSpan diff = currentPosix - expectPosix;

    // DevMenuCommand で時刻設定してから TestDelayMinutes 分以内にはテスト実行されることを期待
    EXPECT_LE( diff.GetSeconds(), nn::TimeSpan::FromMinutes(TestDelayMinutes).GetSeconds() );
}

// 直前に以下を行っている前提. ネットワーク時計の自動補正を切っておくこと.
//   - RunOnTarget DevMenuCommand time set-auto-network-clock-ensure-policy disable # 事前に自動補正OFFにしておかないとネットワーク時計が書き換わる可能性がある
//   - ControlTarget --reset
//   - RunOnTarget DevMenuCommand set-network-clock --year 2016 --month 9 --day 6 --hour 11 --minute 32
//   - ControlTarget --reset
TEST(TimeDebugFeature, ExpectNetworkClockIs2016_09_06_11_32)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    // 事前に決まった日時にセットしていることが前提
    TestDateTime<StandardNetworkSystemClock>(2016, 9, 6, 11, 32);
}

TEST(TimeDebugFeature, ExpectStandardUserClockAutomaticCorrectionIsEnabled)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    EXPECT_TRUE(nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled());
}

TEST(TimeDebugFeature, ExpectStandardUserClockAutomaticCorrectionIsDisabled)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    EXPECT_FALSE(nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled());
}

TEST(TimeDebugFeature, DeleteSaveData)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug(nnt::time::savedata::MountName));
    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(nnt::time::savedata::MountName); };

    NNT_EXPECT_RESULT_SUCCESS( nn::fs::DeleteFile(nnt::time::savedata::SteadyClockTimePoint) );
    NNT_ASSERT_RESULT_SUCCESS( nn::fs::CommitSaveData(nnt::time::savedata::MountName) );
}

TEST(TimeDebugFeature, ExpectLocationNameIsAsiaTokyo)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    LocationName location;
    GetDeviceLocationName(&location);

    EXPECT_STREQ("Asia/Tokyo", location._value);
}
