﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/os.h>
#include <nn/nifm.h>
#include <nn/time.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/time/time_ApiForRepair.h>

#include <nn/time/time_StandardNetworkSystemClockPrivilegeApi.h>
#include <nn/time/time_StandardUserSystemClockPrivilegeApi.h>
#include <nn/time/time_StandardLocalSystemClockPrivilegeApi.h>

#include <nn/time/detail/time_ClockSnapshotPrivateApi.h>


#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>

#if NN_BUILD_CONFIG_OS_WIN

// Windows.h 向けAPIと衝突しないかの確認

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif

#ifndef NOMINMAX
#define NOMINMAX
#endif

#include <nn/nn_Windows.h>

#endif // NN_BUILD_CONFIG_OS_WIN


// IStaticService が ISystemClock, ISteadyClock, ITimeZoneService の shared pointer を返すので、
// それらを先に include する必要があります
#include <nn/timesrv/detail/service/timesrv_ISystemClock.sfdl.h>
#include <nn/timesrv/detail/service/timesrv_ISteadyClock.sfdl.h>
#include <nn/timesrv/detail/service/timesrv_ITimeZoneService.sfdl.h>
#include <nn/timesrv/detail/service/timesrv_IStaticService.sfdl.h> // for nn::time::sf::ClockSnapshot

class ClockSnapshotTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize()); // win版だと SuspendAutonomicTimeCorrection() に必要

        NNT_ASSERT_RESULT_SUCCESS(nn::time::Initialize());
        nn::time::SuspendAutonomicTimeCorrection();
        NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
    }

    static void TearDownTestCase()
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::time::Initialize());
        nn::time::ResumeAutonomicTimeCorrection();
        NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
    }
};

namespace
{
    // ユーザー時計の自動補正ON/OFF
    // time:a 必須
    void SetUserClockAutomaticCorrectionEnabled(bool value) NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
        nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(value);
        EXPECT_EQ(value, nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled());
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
    }

    // ユーザ時計の補正
    // time:a 必須
    void SetUserClockCurrentTime(int64_t posixTimeValue) NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
        nn::time::PosixTime posixTime = {posixTimeValue};
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardLocalSystemClockCurrentTime(posixTime) );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
    }

    // ネットワーク時計の補正
    // time:s 必須
    void SetNetworkClockCurrentTime(int64_t posixTimeValue) NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForSystem() );
        nn::time::PosixTime posixTime = {posixTimeValue};
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardNetworkSystemClockCurrentTime(posixTime) );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
    }

    // ネットワーク時計の補正を解除
    // time:s 必須
    void UnadjustedNetworkClock() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForSystem() );
        nn::time::SystemClockContext context = {};
        context.timeStamp.sourceId = nn::util::InvalidUuid;
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardNetworkSystemClockContext(context) );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
    }

    // ユーザー時計の時刻を加算
    void AddUserSystemClockContextOffset(int64_t value) NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
        NN_UTIL_SCOPE_EXIT{ NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() ); };

        nn::time::SystemClockContext c;
        NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetSystemClockContext(&c) );
        c.offset += value;
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardLocalSystemClockContext(c) );
    }

    // ネットワーク時計の時刻を加算
    void AddNetworkSystemClockContextOffset(int64_t value) NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForSystem() );
        NN_UTIL_SCOPE_EXIT{ NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() ); };

        nn::time::SystemClockContext c;
        NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardNetworkSystemClock::GetSystemClockContext(&c) );
        c.offset += value;
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardNetworkSystemClockContext(c) );
    }
}

TEST_F(ClockSnapshotTest, Operator)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    nn::time::ClockSnapshot lhs, rhs;
    EXPECT_EQ(lhs, rhs);

    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&lhs);
    EXPECT_NE(lhs, rhs);

    rhs = lhs;
    EXPECT_EQ(lhs, rhs);

    auto p = nn::time::detail::GetSfClockSnapshotPtr(&lhs);

    p->userSystemClockContext.offset++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->netSystemClockContext.offset++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->userSystemClockPosixTime.value++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->netSystemClockPosixTime.value++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->userCalendarTime.second++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->netCalendarTime.second++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->userCalendarAdditionalInfo.dayOfWeek ++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->userCalendarAdditionalInfo.yearDay++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->userCalendarAdditionalInfo.timeZone.utcOffsetSeconds++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->netCalendarAdditionalInfo.dayOfWeek++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->netCalendarAdditionalInfo.yearDay++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->netCalendarAdditionalInfo.timeZone.utcOffsetSeconds++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->steadyClockTimePoint.value++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->locationName._value[0] = ' ';
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->isAutomaticCorrectionEnabled = !p->isAutomaticCorrectionEnabled;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->initialType++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->version++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;

    p->padding[0]++;
    EXPECT_NE(lhs, rhs);
    lhs = rhs;
}

TEST_F(ClockSnapshotTest, ClockSnapshotInitialType)
{
    nn::time::ClockSnapshot s;
    EXPECT_EQ(nn::time::detail::ClockSnapshotInitialType::Constructor, nn::time::detail::GetClockSnapshotInitialType(s));

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&s);
    EXPECT_EQ(nn::time::detail::ClockSnapshotInitialType::WithStandardSystemClock, nn::time::detail::GetClockSnapshotInitialType(s));

    nn::time::ClockSnapshot::CreateWithAdjustableSystemClock(&s);
    EXPECT_EQ(nn::time::detail::ClockSnapshotInitialType::WithAdjustableSystemClock, nn::time::detail::GetClockSnapshotInitialType(s));
}

TEST_F(ClockSnapshotTest, Copyable)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    nn::time::ClockSnapshot s;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&s);

    nn::time::ClockSnapshot s2;
    std::memcpy(&s2, &s, sizeof(nn::time::ClockSnapshot));
    EXPECT_TRUE(std::memcmp(&s, &s2, sizeof(nn::time::ClockSnapshot)) == 0);

    uint8_t buffer[sizeof(nn::time::ClockSnapshot)];
    std::memcpy(buffer, &s, sizeof(nn::time::ClockSnapshot));
    std::memcpy(&s2, buffer, sizeof(nn::time::ClockSnapshot));
    EXPECT_TRUE(std::memcmp(&s, &s2, sizeof(nn::time::ClockSnapshot)) == 0);
}

TEST_F(ClockSnapshotTest, InitialClockSnapshot)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    nn::time::ClockSnapshot s;

    EXPECT_STREQ(s.GetLocationName()._value, "UTC");

    // SteadyClock
    EXPECT_EQ(0, s.GetStandardSteadyClockTimePoint().value);
    EXPECT_EQ(nn::util::InvalidUuid, s.GetStandardSteadyClockTimePoint().sourceId);

    // UserClock
    EXPECT_EQ(nn::util::InvalidUuid, s.GetStandardUserSystemClockContext().timeStamp.sourceId);
    EXPECT_EQ(0, s.GetStandardUserSystemClockContext().offset);

    EXPECT_EQ(0, s.GetStandardUserSystemClockPosixTime().value);
    EXPECT_EQ(1970, s.GetStandardUserSystemClockCalendarTime().year);
    EXPECT_EQ(   1, s.GetStandardUserSystemClockCalendarTime().month);
    EXPECT_EQ(   1, s.GetStandardUserSystemClockCalendarTime().day);
    EXPECT_EQ(   0, s.GetStandardUserSystemClockCalendarTime().hour);
    EXPECT_EQ(   0, s.GetStandardUserSystemClockCalendarTime().minute);
    EXPECT_EQ(   0, s.GetStandardUserSystemClockCalendarTime().second);

    EXPECT_EQ(nn::time::DayOfWeek_Thursday, s.GetStandardUserSystemClockCalendarAdditionalInfo().dayOfWeek);
    EXPECT_EQ(0, s.GetStandardUserSystemClockCalendarAdditionalInfo().yearDay);
    EXPECT_STREQ("UTC", s.GetStandardUserSystemClockCalendarAdditionalInfo().timeZone.standardTimeName);
    EXPECT_FALSE(s.GetStandardUserSystemClockCalendarAdditionalInfo().timeZone.isDaylightSavingTime);
    EXPECT_EQ(0, s.GetStandardUserSystemClockCalendarAdditionalInfo().timeZone.utcOffsetSeconds);

    // NetworkClock
    EXPECT_EQ(nn::util::InvalidUuid, s.GetStandardNetworkSystemClockContext().timeStamp.sourceId);
    EXPECT_EQ(0, s.GetStandardNetworkSystemClockContext().offset);

    EXPECT_EQ(0, s.GetStandardNetworkSystemClockPosixTime().value);
    EXPECT_EQ(1970, s.GetStandardNetworkSystemClockCalendarTime().year);
    EXPECT_EQ(   1, s.GetStandardNetworkSystemClockCalendarTime().month);
    EXPECT_EQ(   1, s.GetStandardNetworkSystemClockCalendarTime().day);
    EXPECT_EQ(   0, s.GetStandardNetworkSystemClockCalendarTime().hour);
    EXPECT_EQ(   0, s.GetStandardNetworkSystemClockCalendarTime().minute);
    EXPECT_EQ(   0, s.GetStandardNetworkSystemClockCalendarTime().second);

    EXPECT_EQ(nn::time::DayOfWeek_Thursday, s.GetStandardNetworkSystemClockCalendarAdditionalInfo().dayOfWeek);
    EXPECT_EQ(0, s.GetStandardNetworkSystemClockCalendarAdditionalInfo().yearDay);
    EXPECT_STREQ("UTC", s.GetStandardNetworkSystemClockCalendarAdditionalInfo().timeZone.standardTimeName);
    EXPECT_FALSE(s.GetStandardNetworkSystemClockCalendarAdditionalInfo().timeZone.isDaylightSavingTime);
    EXPECT_EQ(0, s.GetStandardNetworkSystemClockCalendarAdditionalInfo().timeZone.utcOffsetSeconds);
}

template <class UserClockClass, class NetworkClockClass>
void TestClockSnapshotValue(
    const nn::time::ClockSnapshot& snapshot,
    int64_t userPosixValue, int64_t netPosixValue, bool isAutomaticCorrectionEnabled) NN_NOEXCEPT
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    // system settings
    {
        if(isAutomaticCorrectionEnabled)
        {
            EXPECT_TRUE(nn::time::detail::IsClockSnapshotAutomaticCorrectionEnabled(snapshot));
        }
        else
        {
            EXPECT_FALSE(nn::time::detail::IsClockSnapshotAutomaticCorrectionEnabled(snapshot));
        }

        nn::time::LocationName locationName;
        nn::time::GetDeviceLocationName(&locationName);
        EXPECT_STREQ(locationName._value, snapshot.GetLocationName()._value);
    }

    // UserSystemClock
    {
        EXPECT_TRUE((snapshot.GetStandardUserSystemClockPosixTime().value - userPosixValue) <= 1LL)
            << "diff:" << (snapshot.GetStandardUserSystemClockPosixTime().value - userPosixValue); // タイミングによって1秒経過はあり得る

        nn::time::SystemClockContext systemClockContext;
        UserClockClass::GetSystemClockContext(&systemClockContext);
        EXPECT_EQ(systemClockContext, snapshot.GetStandardUserSystemClockContext());

        nn::time::CalendarTime c;
        nn::time::CalendarAdditionalInfo info;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::ToCalendarTime(&c, &info, snapshot.GetStandardUserSystemClockPosixTime()));
        EXPECT_EQ(c, snapshot.GetStandardUserSystemClockCalendarTime());
        EXPECT_EQ(info, snapshot.GetStandardUserSystemClockCalendarAdditionalInfo());
    }

    // NetworkSystemClock
    {
        if(NN_STATIC_CONDITION(netPosixValue == 0))
        {
            EXPECT_EQ(0, snapshot.GetStandardNetworkSystemClockPosixTime().value);
        }
        else
        {
            EXPECT_TRUE((snapshot.GetStandardNetworkSystemClockPosixTime().value - netPosixValue) <= 1LL)
                << "diff:" << (snapshot.GetStandardNetworkSystemClockPosixTime().value - netPosixValue); // タイミングによって1秒経過はあり得る
        }

        nn::time::SystemClockContext systemClockContext;
        NetworkClockClass::GetSystemClockContext(&systemClockContext);
        EXPECT_EQ(systemClockContext, snapshot.GetStandardNetworkSystemClockContext());

        nn::time::CalendarTime c;
        nn::time::CalendarAdditionalInfo info;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::ToCalendarTime(&c, &info, snapshot.GetStandardNetworkSystemClockPosixTime()));
        EXPECT_EQ(c, snapshot.GetStandardNetworkSystemClockCalendarTime());
        EXPECT_EQ(info, snapshot.GetStandardNetworkSystemClockCalendarAdditionalInfo());
    }

    // StandardSteadyClock
    {
        nn::time::SteadyClockTimePoint timePoint;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&timePoint));
        int64_t elapsedSeconds;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::GetSpanBetween(&elapsedSeconds, snapshot.GetStandardSteadyClockTimePoint(), timePoint));
        EXPECT_TRUE(elapsedSeconds <= 1LL) << "elapsed:" << elapsedSeconds; // タイミングによって1秒経過はあり得る
    }
}
TEST_F(ClockSnapshotTest, ClockSnapshotWithStandardSystemClock)
{
    {
        SCOPED_TRACE("");

        SetUserClockAutomaticCorrectionEnabled(false);
        SetUserClockCurrentTime(1497934096LL);
        UnadjustedNetworkClock();

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT { NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        nn::time::ClockSnapshot snapshot;
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&snapshot);
        TestClockSnapshotValue<nn::time::StandardUserSystemClock, nn::time::StandardNetworkSystemClock>(snapshot, 1497934096LL, 0LL, false);
    }

    {
        SCOPED_TRACE("");
        SetUserClockAutomaticCorrectionEnabled(false);
        SetUserClockCurrentTime(1497934096LL);
        SetNetworkClockCurrentTime(1495257519LL);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT { NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        nn::time::ClockSnapshot snapshot;
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&snapshot);
        TestClockSnapshotValue<nn::time::StandardUserSystemClock, nn::time::StandardNetworkSystemClock>(snapshot, 1497934096LL, 1495257519LL, false);
    }

    {
        SCOPED_TRACE("");
        SetUserClockAutomaticCorrectionEnabled(true);
        SetUserClockCurrentTime(1497934096LL);
        SetNetworkClockCurrentTime(1495257519LL);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT { NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        nn::time::ClockSnapshot snapshot;
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&snapshot);
        TestClockSnapshotValue<nn::time::StandardUserSystemClock, nn::time::StandardNetworkSystemClock>(snapshot, 1495257519LL, 1495257519LL, true);
    }

    {
        // ClockSnapshot 生成、更新タイミングが異なるテスト

        SetUserClockCurrentTime(1497934096LL);
        SetNetworkClockCurrentTime(1495257519LL);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT { NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        nn::time::ClockSnapshot s1, s2;
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&s1);

        // s1 と s2 で2秒差
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&s2);

        EXPECT_TRUE(s1.GetStandardUserSystemClockPosixTime() < s2.GetStandardUserSystemClockPosixTime());
        EXPECT_TRUE(s1.GetStandardNetworkSystemClockPosixTime() < s2.GetStandardNetworkSystemClockPosixTime());

        EXPECT_EQ(s1.GetStandardUserSystemClockContext(), s2.GetStandardUserSystemClockContext());
        EXPECT_EQ(s1.GetStandardNetworkSystemClockContext(), s2.GetStandardNetworkSystemClockContext());

        // さらにs1だけ更新
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&s1);

        EXPECT_TRUE(s1.GetStandardUserSystemClockPosixTime() > s2.GetStandardUserSystemClockPosixTime());
        EXPECT_TRUE(s1.GetStandardNetworkSystemClockPosixTime() > s2.GetStandardNetworkSystemClockPosixTime());

        EXPECT_EQ(s1.GetStandardUserSystemClockContext(), s2.GetStandardUserSystemClockContext());
        EXPECT_EQ(s1.GetStandardNetworkSystemClockContext(), s2.GetStandardNetworkSystemClockContext());
    }
}

TEST_F(ClockSnapshotTest, ClockSnapshotWithAdjustableSystemClock)
{
    {
        SCOPED_TRACE("");
        SetUserClockAutomaticCorrectionEnabled(false);
        SetUserClockCurrentTime(1497934096LL);
        SetNetworkClockCurrentTime(1495257519LL);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT { NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };
        nn::time::AdjustableUserSystemClock::Adjust();
        nn::time::AdjustableNetworkSystemClock::Adjust();
    }

    {
        // 適当に時刻を補正する
        SCOPED_TRACE("");
        AddUserSystemClockContextOffset(321LL);
        AddNetworkSystemClockContextOffset(123LL);
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT { NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    {
        SCOPED_TRACE("");
        nn::time::ClockSnapshot snapshot;
        nn::time::ClockSnapshot::CreateWithAdjustableSystemClock(&snapshot);
        TestClockSnapshotValue<nn::time::AdjustableUserSystemClock, nn::time::AdjustableNetworkSystemClock>(
                snapshot, 1497934096LL, 1495257519LL, false);
    }

    {
        // 念のため Standard な方が変わっているか確認
        SCOPED_TRACE("");
        nn::time::ClockSnapshot snapshot;
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&snapshot);
        TestClockSnapshotValue<nn::time::StandardUserSystemClock, nn::time::StandardNetworkSystemClock>(
            snapshot, 1497934096LL + 321LL, 1495257519LL + 123LL, false);
    }

    // 追従
    nn::time::AdjustableUserSystemClock::Adjust();
    nn::time::AdjustableNetworkSystemClock::Adjust();
    {
        SCOPED_TRACE("");
        nn::time::ClockSnapshot snapshot;
        nn::time::ClockSnapshot::CreateWithAdjustableSystemClock(&snapshot);
        TestClockSnapshotValue<nn::time::AdjustableUserSystemClock, nn::time::AdjustableNetworkSystemClock>(
                snapshot, 1497934096LL + 321LL, 1495257519LL + 123LL, false);
    }
}

nn::time::ClockSnapshot GetClockSnapshot(int64_t userPosixAddValue, int64_t netPosixAddValue, bool isAutomaticCorrectionEnabled) NN_NOEXCEPT
{
    SetUserClockAutomaticCorrectionEnabled(isAutomaticCorrectionEnabled);
    if(userPosixAddValue != 0)
    {
        AddUserSystemClockContextOffset(userPosixAddValue);
    }
    if(netPosixAddValue != 0)
    {
        AddNetworkSystemClockContextOffset(netPosixAddValue);
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::ClockSnapshot context;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&context);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());

    return context;
}
TEST_F(ClockSnapshotTest, CalculateStandardUserSystemClockDifferenceByUser)
{
    SetUserClockCurrentTime(1497934096LL);
    SetNetworkClockCurrentTime(1495257519LL);

    {
        // 常に自動補正OFF、ユーザー時計だけ操作
        SCOPED_TRACE("");

        auto s1 = GetClockSnapshot( 0LL, 0LL, false);
        auto s2 = GetClockSnapshot(10LL, 0LL, false);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        auto diff = nn::time::CalculateStandardUserSystemClockDifferenceByUser(s1, s2);
        EXPECT_EQ(nn::TimeSpan::FromSeconds(10), diff) << "diff:" << diff.GetSeconds();

        diff = nn::time::CalculateStandardUserSystemClockDifferenceByUser(s2, s1);
        EXPECT_EQ(nn::TimeSpan::FromSeconds(-10), diff) << "diff:" << diff.GetSeconds();
    }
    {
        // 常に自動補正OFF、ネットワーク時計だけ操作
        SCOPED_TRACE("");

        auto s1 = GetClockSnapshot(0LL,  0LL, false);
        auto s2 = GetClockSnapshot(0LL, 10LL, false);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        EXPECT_EQ(nn::TimeSpan(0), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s1, s2));
        EXPECT_EQ(nn::TimeSpan(0), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s2, s1));
    }
    {
        // 常に自動補正ON、ネットワーク時計が無効な状態から補正される
        SCOPED_TRACE("");

        UnadjustedNetworkClock();
        auto s1 = GetClockSnapshot(0LL, 0LL, true);

        SetNetworkClockCurrentTime(1495257519LL);
        auto s2 = GetClockSnapshot(0LL, 0LL, true);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        // コンテキストは異なるが、ユーザーが操作したことにはならない
        EXPECT_NE(s1.GetStandardUserSystemClockContext(), s2.GetStandardUserSystemClockContext());
        EXPECT_EQ(nn::TimeSpan(0), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s1, s2));
        EXPECT_EQ(nn::TimeSpan(0), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s2, s1));
    }
    {
        // 常に自動補正ON、ネットワーク時計だけ操作
        SCOPED_TRACE("");

        auto s1 = GetClockSnapshot(0LL,  0LL, true);
        auto s2 = GetClockSnapshot(0LL, 10LL, true);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        EXPECT_EQ(nn::TimeSpan(0), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s1, s2));
        EXPECT_EQ(nn::TimeSpan(0), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s2, s1));
    }
    {
        // 自動補正OFF->ON
        // ONにしたときにユーザー時計がネットワーク時計の時刻で上書きされ、変更される
        SCOPED_TRACE("");

        SetUserClockAutomaticCorrectionEnabled(false);
        SetUserClockCurrentTime(1497934096LL);
        SetNetworkClockCurrentTime(1495257519LL);

        auto s1 = GetClockSnapshot(0LL, 0LL, false);
        auto s2 = GetClockSnapshot(0LL, 0LL, true);

        auto diff = s2.GetStandardUserSystemClockContext().offset - s1.GetStandardUserSystemClockContext().offset;
        EXPECT_TRUE((1495257519LL - 1497934096LL) - 1LL <= diff && diff <= (1495257519LL - 1497934096LL)) << "diff:" << diff; // 1秒ずれを許容

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        EXPECT_EQ(nn::TimeSpan::FromSeconds(diff), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s1, s2));
        EXPECT_EQ(nn::TimeSpan::FromSeconds(-diff), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s2, s1));
    }
    {
        // 2つのClockSnapshotは自動補正ONだが、ネットワーク時計がずっと未補正
        // 自動補正OFFのときに時計操作する
        SCOPED_TRACE("");

        UnadjustedNetworkClock(); // ネットワーク時計未補正
        SetUserClockCurrentTime(1497934096LL);

        auto s1 = GetClockSnapshot(0LL, 0LL, true); // 最初は自動補正ON

        // 自動補正OFFのときに時計操作、その後自動補正ONにして ClockSnapshot を取得
        SetUserClockAutomaticCorrectionEnabled(false);
        AddUserSystemClockContextOffset(100LL);
        auto s2 = GetClockSnapshot(0LL, 0LL, true); // 自動補正ONにしてもネットワーク時計が未補正なのでユーザー時計は上書きされない

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        auto diff = s2.GetStandardUserSystemClockContext().offset - s1.GetStandardUserSystemClockContext().offset;
        EXPECT_EQ(100LL, diff);

        EXPECT_NE(s1.GetStandardUserSystemClockContext(), s2.GetStandardUserSystemClockContext());
        EXPECT_EQ(nn::TimeSpan::FromSeconds(diff), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s1, s2));
        EXPECT_EQ(nn::TimeSpan::FromSeconds(-diff), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s2, s1));
    }
    {
        // 自動補正ON->OFF
        SCOPED_TRACE("");

        auto s1 = GetClockSnapshot(0LL, 0LL, true);
        auto s2 = GetClockSnapshot(0LL, 0LL, false);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        EXPECT_EQ(nn::TimeSpan(0), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s1, s2));
        EXPECT_EQ(nn::TimeSpan(0), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s2, s1));
    }
    {
        // 自動補正ON->OFF. その間にネットワーク時計の補正が行われる
        SCOPED_TRACE("");

        auto s1 = GetClockSnapshot(0LL,  0LL, true);
        auto s2 = GetClockSnapshot(0LL, 10LL, false);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        EXPECT_EQ(nn::TimeSpan(0), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s1, s2));
        EXPECT_EQ(nn::TimeSpan(0), nn::time::CalculateStandardUserSystemClockDifferenceByUser(s2, s1));
    }
}

// 通常ケース. sourceId 一致、ネットワーク時計保持.
TEST_F(ClockSnapshotTest, CalculateSpanBetween1)
{
    SetNetworkClockCurrentTime(1495257519LL);
    auto from = GetClockSnapshot(0LL, 0LL, false);
    auto to = from;

    // to の steadyClockTimePoint を 100 秒、ネットワーク時計を 110 秒進める
    auto toPtr = nn::time::detail::GetSfClockSnapshotPtr(&to);
    toPtr->steadyClockTimePoint.value += 100;
    toPtr->netSystemClockPosixTime += nn::TimeSpan::FromSeconds(110);

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    nn::TimeSpan span;
    EXPECT_TRUE(nn::time::CalculateSpanBetween(&span, from, to));
    EXPECT_EQ(100, span.GetSeconds());

    EXPECT_TRUE(nn::time::CalculateSpanBetween(&span, to, from));
    EXPECT_EQ(-100, span.GetSeconds());
}

// soruceId 食い違い, かつto,fromともにネットワーク時計なし
TEST_F(ClockSnapshotTest, CalculateSpanBetween2)
{
    UnadjustedNetworkClock();
    auto from = GetClockSnapshot(0LL, 0LL, false);
    auto to = from;

    // to の sourceId を改変 -> SteadyClockTimePoint は比較できない
    auto toPtr = nn::time::detail::GetSfClockSnapshotPtr(&to);
    toPtr->steadyClockTimePoint.sourceId.data[0]++;

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    nn::TimeSpan span;
    EXPECT_FALSE(nn::time::CalculateSpanBetween(&span, from, to));
    EXPECT_FALSE(nn::time::CalculateSpanBetween(&span, to, from));
}

// soruceId 食い違い, かつtoネットワーク時計なし、fromネットワーク時計あり
TEST_F(ClockSnapshotTest, CalculateSpanBetween3)
{
    UnadjustedNetworkClock();
    auto from = GetClockSnapshot(0LL, 0LL, false);

    SetNetworkClockCurrentTime(1495257519LL);
    auto to = GetClockSnapshot(0LL, 0LL, false);

    EXPECT_EQ(0, from.GetStandardNetworkSystemClockPosixTime().value);
    EXPECT_NE(0, to.GetStandardNetworkSystemClockPosixTime().value);

    // to の sourceId を改変 -> SteadyClockTimePoint は比較できない
    auto toPtr = nn::time::detail::GetSfClockSnapshotPtr(&to);
    toPtr->steadyClockTimePoint.sourceId.data[0]++;

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    nn::TimeSpan span;
    EXPECT_FALSE(nn::time::CalculateSpanBetween(&span, from, to));
    EXPECT_FALSE(nn::time::CalculateSpanBetween(&span, to, from));
}

// soruceId 食い違い, かつto,fromともにネットワーク時計あり
TEST_F(ClockSnapshotTest, CalculateSpanBetween4)
{
    SetNetworkClockCurrentTime(1495257519LL);
    auto from = GetClockSnapshot(0LL, 0LL, false);
    auto to = from;

    // to の steadyClockTimePoint を 100 秒、ネットワーク時計を 110 秒進める
    auto toPtr = nn::time::detail::GetSfClockSnapshotPtr(&to);
    toPtr->steadyClockTimePoint.value += 100;
    toPtr->netSystemClockPosixTime += nn::TimeSpan::FromSeconds(110);

    // to の sourceId を改変 -> SteadyClockTimePoint は比較できない
    toPtr->steadyClockTimePoint.sourceId.data[0]++;

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    nn::TimeSpan span;
    EXPECT_TRUE(nn::time::CalculateSpanBetween(&span, from, to));
    EXPECT_EQ(110, span.GetSeconds());

    EXPECT_TRUE(nn::time::CalculateSpanBetween(&span, to, from));
    EXPECT_EQ(-110, span.GetSeconds());
}
