﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>

#include <nn/fan/fan.h>
#include <nn/fan/fan_ControlDev.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/tc/impl/tc.h>
#include <nn/tc/impl/tc_PowerMode.h>
#include <nn/tc/tc.h>
#include <nn/tc/tc_IManager.sfdl.h>
#include <nn/tc/tc_ManagerImpl.h>
#include <nn/tc/tc_ShimInternal.h>

#include <nnt/nnt_Argument.h>

#include <nnt/base/testBase_Exit.h>
#include <nnt/gtest/gtest.h>

#include "Stubs/PcvStub.h"
#include "Stubs/TsStub.h"

// TODO: SIGLO-66561 の修正により温度センサ値取得異常時の ABORT 機能を確認するテストが存在しません。
// 異常時のシャットダウンを保証するためテストを復旧させる必要があります。

namespace nnt { namespace tc {

//#define USE_TABLE_HANDHELD

const nn::pcv::TemperatureThreshold TemperatureThresholds[] =
{
    { 15000, 16000 },
    { 30000, 31000 },
    { 50000, 51000 },
    { 70000, 71000 },
    { 90000, 91000 },
};

namespace {

struct ThermalCoordinatorTestSettingEntry
{
    nn::tc::impl::PowerMode powerMode;
    nn::tc::OperatingMode operatingMode;

    int pcbTempStartMilliC;
    int pcbTempEndMilliC;
    int socTempStartMilliC;
    int socTempEndMilliC;
    int steps;

    nn::fan::RotationSpeedLevel speedLevelMin;
    nn::fan::RotationSpeedLevel speedLevelMax;
    nn::fan::RotationSpeedLevel speedLevelGradientMax;
};

struct ThermalCoordinatorTestSetting
{
    const ThermalCoordinatorTestSettingEntry* pEntry;
    int size;
};

nn::sf::UnmanagedServiceObject<nn::tc::IManager, nn::tc::ManagerImpl> g_Manager;

const int SocTempMaxMilliC =  82000;
const int SocTempMinMilliC = -18000;
const int PcbTempMaxMilliC =  76000;
const int PcbTempMinMilliC = -18000;

#define PERCENTAGE_TO_SPEED_LEVEL(percentage) ((percentage) * 255 / 100)

const nn::fan::RotationSpeedLevel MinSpeedLevelConsoleToHandheld = PERCENTAGE_TO_SPEED_LEVEL(60);
const nn::fan::RotationSpeedLevel MinSpeedLevelHandheld = PERCENTAGE_TO_SPEED_LEVEL(0);
const nn::fan::RotationSpeedLevel MaxSpeedLevelHandheld = PERCENTAGE_TO_SPEED_LEVEL(100);
const nn::fan::RotationSpeedLevel MinSpeedLevelConsole  = PERCENTAGE_TO_SPEED_LEVEL(0);
const nn::fan::RotationSpeedLevel MaxSpeedLevelConsole  = PERCENTAGE_TO_SPEED_LEVEL(100);

#if defined(USE_TABLE_HANDHELD)

// table-handheld 用の設定
// 表面温度計算式も携帯モードと据置モードで一致させることを前提としている。
const ThermalCoordinatorTestSettingEntry TestEntriesHandheldTableHandheld[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
};

const ThermalCoordinatorTestSettingEntry TestEntriesConsoleTableHandheld[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console, PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
};

const ThermalCoordinatorTestSettingEntry TestEntriesHandheldToConsoleTableHandheld[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console, PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
};

const ThermalCoordinatorTestSettingEntry TestEntriesConsoleToHandheldTableHandheld[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
};

const ThermalCoordinatorTestSettingEntry TestEntriesMinimumAwakeHandheldTableHandheld[] =
{
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
};

const ThermalCoordinatorTestSettingEntry TestEntriesMinimumAwakeConsoleTableHandheld[] =
{
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console, PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
};

const ThermalCoordinatorTestSettingEntry TestEntriesMinimumAwakeHandheldToConsoleTableHandheld[] =
{
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console, PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
};

const ThermalCoordinatorTestSettingEntry TestEntriesMinimumAwakeConsoleToHandheldTableHandheld[] =
{
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
};

#endif // USE_TABLE_HANDHELD

// 携帯モードで、テスト対象の温度の下限→上限→下限→上限へと遷移する。
const ThermalCoordinatorTestSettingEntry TestEntriesHandheld[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
};

// 据置モードで、テスト対象の温度の下限→上限へと遷移する。
// 携帯モードで、テスト対象の温度の上限→下限へと遷移する。
// 据置モードで、テスト対象の温度の下限→上限へと遷移する。
// 据置モードから携帯モードへの遷移時に ThermalCoordinator の指定する回転数は大幅に低下する為遷移時には 102 までの duty rate 変動を許可している。
const ThermalCoordinatorTestSettingEntry TestEntriesConsoleToHandheld[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console,  PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelConsole,           MaxSpeedLevelConsole,   14},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMaxMilliC, PcbTempMaxMilliC, SocTempMaxMilliC, SocTempMaxMilliC,   1, MinSpeedLevelConsoleToHandheld, MaxSpeedLevelHandheld, 102},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMaxMilliC,            54054, SocTempMaxMilliC,            54054,  28, MinSpeedLevelConsoleToHandheld, MaxSpeedLevelHandheld,   6},

// Advanced(prev) ポリシーが無い場合は 102 まで回転数が低下する。
//    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld,            54053,            54053,            54053,            54053,   1,   0, 102,  58},
//    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld,            54053, PcbTempMinMilliC,            54053, SocTempMinMilliC,  73,   0, 102,   6},

    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld,            54053,            54053,            54053,            54053,   1, MinSpeedLevelHandheld, MaxSpeedLevelHandheld, 58},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld,            54053, PcbTempMinMilliC,            54053, SocTempMinMilliC,  73, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,  6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console,  PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelConsole, MaxSpeedLevelConsole,  14},
};

// 据置モードで、テスト対象の温度の下限→上限→下限→上限へと遷移する。
const ThermalCoordinatorTestSettingEntry TestEntriesConsole[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console,  PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelConsole, MaxSpeedLevelConsole,  14},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console,  PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100, MinSpeedLevelConsole, MaxSpeedLevelConsole,  14},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console,  PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelConsole, MaxSpeedLevelConsole,  14},
};

// 携帯モードで、テスト対象の温度の下限→上限へと遷移する。
// 据置モードで、テスト対象の温度の上限→下限へと遷移する。
// 携帯モードで、テスト対象の温度の下限→上限へと遷移する。
// 携帯モードから据置モードへの遷移時に ThermalCoordinator の指定する回転数は大幅に上昇する為遷移時には 102 までの duty rate 変動を許可している。
const ThermalCoordinatorTestSettingEntry TestEntriesHandheldToConsole[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,  6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console,  PcbTempMaxMilliC, PcbTempMaxMilliC, SocTempMaxMilliC, SocTempMaxMilliC,   1, MinSpeedLevelConsole, MaxSpeedLevelConsole, 102},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console,  PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100, MinSpeedLevelConsole, MaxSpeedLevelConsole,  14},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,  6},
};

const ThermalCoordinatorTestSettingEntry TestEntriesMinimumAwakeHandheld[] =
{
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100,   0,   0,   0},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100,   0,   0,   0},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100,   0,   0,   0},
};

const ThermalCoordinatorTestSettingEntry TestEntriesMinimumAwakeConsoleToHandheld[] =
{
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console,  PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100,   0,   0,   0},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100,   0,   0,   0},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console,  PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100,   0,   0,   0},
};

const ThermalCoordinatorTestSettingEntry TestEntriesMinimumAwakeConsole[] =
{
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console,  PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100,   0,   0,   0},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console,  PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100,   0,   0,   0},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console,  PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100,   0,   0,   0},
};

const ThermalCoordinatorTestSettingEntry TestEntriesMinimumAwakeHandheldToConsole[] =
{
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100,   0,   0,   0},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Console,  PcbTempMaxMilliC, PcbTempMinMilliC, SocTempMaxMilliC, SocTempMinMilliC, 100,   0,   0,   0},
    { nn::tc::impl::PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, PcbTempMinMilliC, PcbTempMaxMilliC, SocTempMinMilliC, SocTempMaxMilliC, 100,   0,   0,   0},
};

// 据置モードで、室温→表面温度 48C 以上へと移動する。
// 据置モードから携帯モードへの遷移させる。
// 携帯モードで、表面温度 48C 以上→室温へと移動する。
// 携帯モードへの遷移時 duty rate が急峻に上昇することを期待する。
const ThermalCoordinatorTestSettingEntry TestEntriesConsoleToHandheldOverThreshold[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console,  25000, 54054, 25000, 54054,  30, MinSpeedLevelConsole, MaxSpeedLevelConsole, 14},

    // 確実に特定の duty rate になっていることを保証する。
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 54054, 54054, 54054, 54054,   1, MinSpeedLevelConsoleToHandheld, MaxSpeedLevelHandheld, 102},

// Advanced(prev) ポリシーが無い場合は 102 まで回転数が低下する。
//    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 54053, 54053, 54053, 54053,   1,   0, 102,  58},
//    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 54053, 25000, 54053, 25000,  30,   0, 102,   6},

    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 54053, 54053, 54053, 54053,   1, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,  58},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 54053, 25000, 54053, 25000,  30, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
};

// 据置モードで、室温→表面温度 48C 未満へと移動する。
// 据置モードから携帯モードへの遷移させる。
// 携帯モードで、表面温度 48C 未満→室温へと移動する。
// 携帯モードへの遷移時 duty rate が急峻に上昇しないことを期待する。
// 具体的には上限 duty rate を 102 にする。102 の値は今後変わりうるが、153 よりもはるかに小さい値であることを保証し急峻な上昇が無いことを保証する。
const ThermalCoordinatorTestSettingEntry TestEntriesConsoleToHandheldUnderThreshold[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console,  25000, 54053, 25000, 54053,  30, MinSpeedLevelConsole, MaxSpeedLevelConsole,  14},
    //{ nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 54053, 54053, 54053, 54053,   1,   0, 102, 255}, // ここで 102 以下になっていることを保証したい。
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 54053, 54053, 54053, 54053,   1, MinSpeedLevelHandheld, MaxSpeedLevelHandheld, 255},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 54053, 25000, 54053, 25000,  30, MinSpeedLevelHandheld, MaxSpeedLevelHandheld,   6},
};

#if defined(USE_TABLE_HANDHELD)

// table-handheld 用の設定
const ThermalCoordinatorTestSetting ThermalCoordinatorTestSettingsTableHandheld[] =
{
    // FullAwake の広い温度域の duty rate の勾配確認テスト
    { &TestEntriesHandheldTableHandheld[0],          sizeof(TestEntriesHandheldTableHandheld)          / sizeof(TestEntriesHandheldTableHandheld[0])          },
    { &TestEntriesConsoleToHandheldTableHandheld[0], sizeof(TestEntriesConsoleToHandheldTableHandheld) / sizeof(TestEntriesConsoleToHandheldTableHandheld[0]) },
    { &TestEntriesConsoleTableHandheld[0],           sizeof(TestEntriesConsoleTableHandheld)           / sizeof(TestEntriesConsoleTableHandheld[0])           },
    { &TestEntriesHandheldToConsoleTableHandheld[0], sizeof(TestEntriesHandheldToConsoleTableHandheld) / sizeof(TestEntriesHandheldToConsoleTableHandheld[0]) },

    // MinimumAwake の広い温度域の duty rate の勾配確認テスト
    { &TestEntriesMinimumAwakeHandheldTableHandheld[0],          sizeof(TestEntriesMinimumAwakeHandheldTableHandheld)          / sizeof(TestEntriesMinimumAwakeHandheldTableHandheld[0])          },
    { &TestEntriesMinimumAwakeConsoleToHandheldTableHandheld[0], sizeof(TestEntriesMinimumAwakeConsoleToHandheldTableHandheld) / sizeof(TestEntriesMinimumAwakeConsoleToHandheldTableHandheld[0]) },
    { &TestEntriesMinimumAwakeConsoleTableHandheld[0],           sizeof(TestEntriesMinimumAwakeConsoleTableHandheld)           / sizeof(TestEntriesMinimumAwakeConsoleTableHandheld[0])           },
    { &TestEntriesMinimumAwakeHandheldToConsoleTableHandheld[0], sizeof(TestEntriesMinimumAwakeConsoleTableHandheld)           / sizeof(TestEntriesMinimumAwakeHandheldToConsoleTableHandheld[0]) },
};

class ThermalCoordinatorTestTableHandheld : public ::testing::TestWithParam<ThermalCoordinatorTestSetting>{};

INSTANTIATE_TEST_CASE_P(ThermalCoordinatorTestNameTableHandheld, ThermalCoordinatorTestTableHandheld, ::testing::ValuesIn(ThermalCoordinatorTestSettingsTableHandheld));

#endif // USE_TABLE_HANDHELD

const ThermalCoordinatorTestSetting ThermalCoordinatorTestSettings[] =
{
    // FullAwake の広い温度域の duty rate の勾配確認テスト
    { &TestEntriesHandheld[0],          sizeof(TestEntriesHandheld)          / sizeof(TestEntriesHandheld[0])          },
    { &TestEntriesConsoleToHandheld[0], sizeof(TestEntriesConsoleToHandheld) / sizeof(TestEntriesConsoleToHandheld[0]) },
    { &TestEntriesConsole[0],           sizeof(TestEntriesConsole)           / sizeof(TestEntriesConsole[0])           },
    { &TestEntriesHandheldToConsole[0], sizeof(TestEntriesHandheldToConsole) / sizeof(TestEntriesHandheldToConsole[0]) },

    // MinimumAwake の広い温度域の duty rate の勾配確認テスト
    { &TestEntriesMinimumAwakeHandheld[0],          sizeof(TestEntriesMinimumAwakeHandheld)          / sizeof(TestEntriesMinimumAwakeHandheld[0])          },
    { &TestEntriesMinimumAwakeConsoleToHandheld[0], sizeof(TestEntriesMinimumAwakeConsoleToHandheld) / sizeof(TestEntriesMinimumAwakeConsoleToHandheld[0]) },
    { &TestEntriesMinimumAwakeConsole[0],           sizeof(TestEntriesMinimumAwakeConsole)           / sizeof(TestEntriesMinimumAwakeConsole[0])           },
    { &TestEntriesMinimumAwakeHandheldToConsole[0], sizeof(TestEntriesMinimumAwakeConsole)           / sizeof(TestEntriesMinimumAwakeHandheldToConsole[0]) },

    // 表面温度 48C 以上で据置モードから携帯モードへ遷移した時に回転が急峻に変化する。
    { &TestEntriesConsoleToHandheldOverThreshold[0], sizeof(TestEntriesConsoleToHandheldOverThreshold) / sizeof(TestEntriesConsoleToHandheldOverThreshold[0]) },

    // 表面温度 48C 未満で据置モードから携帯モードへ遷移した時に回転が急峻に変化しない。
    { &TestEntriesConsoleToHandheldUnderThreshold[0], sizeof(TestEntriesConsoleToHandheldUnderThreshold) / sizeof(TestEntriesConsoleToHandheldUnderThreshold[0]) },
};

class ThermalCoordinatorTest : public ::testing::TestWithParam<ThermalCoordinatorTestSetting>{};

INSTANTIATE_TEST_CASE_P(ThermalCoordinatorTestName, ThermalCoordinatorTest, ::testing::ValuesIn(ThermalCoordinatorTestSettings));

nn::os::SystemEventType g_SystemEventSleep;

// TestEachTransition にて g_SystemEventSleep が1度シグナルされたことを表すフラグ。
bool g_SleepSignaled = false;

bool IsInPcvTemperatureArea(int index) NN_NOEXCEPT
{
    nn::pcv::MilliC pcvTemperature = nnt::pcv::GetTemperature();

    int size = sizeof(TemperatureThresholds) / sizeof(TemperatureThresholds[0]);

    if ( index < size )
    {
        if ( !(pcvTemperature <= TemperatureThresholds[index].maxMilliC) )
        {
            //NN_LOG("PCV temp:%d, HigherThreshold[%d]:%d\n", pcvTemperature, index, TemperatureThresholds[index].maxMilliC);
            return false;
        }
    }

    if ( index > 0 )
    {
        if ( !(pcvTemperature >= TemperatureThresholds[index - 1].minMilliC) )
        {
            //NN_LOG("PCV temp:%d, LowerThreshold[%d - 1]:%d\n", pcvTemperature, index, TemperatureThresholds[index - 1].minMilliC);
            return false;
        }
    }

    return true;
}

void CheckPcvTemperature(nn::pcv::MilliC temperatureMilliC) NN_NOEXCEPT
{
    nn::pcv::MilliC lowerThreshold = std::numeric_limits<int>::min();

    int size = sizeof(TemperatureThresholds) / sizeof(TemperatureThresholds[0]);

    int index;

    for ( index = 0; index < size; index++ )
    {
        // 重複域を優先して判定する
        if ( (TemperatureThresholds[index].minMilliC <= temperatureMilliC) && (temperatureMilliC <= TemperatureThresholds[index].maxMilliC) )
        {
            EXPECT_TRUE(IsInPcvTemperatureArea(index) || IsInPcvTemperatureArea(index + 1));
            break;
        }
        else if ( (lowerThreshold <= temperatureMilliC) && (temperatureMilliC <= TemperatureThresholds[index].minMilliC) )
        {
            EXPECT_TRUE(IsInPcvTemperatureArea(index));
            break;
        }

        lowerThreshold = TemperatureThresholds[index].maxMilliC;
    }

    if ( index == size )
    {
        EXPECT_TRUE(IsInPcvTemperatureArea(index));
    }
}

void CheckSleep(int temperatureMilliC, nn::tc::OperatingMode operatingMode) NN_NOEXCEPT
{
    const int SleepThresholdConsole = 82 * 1000;
    const int SleepThresholdHandheld = 82 * 1000;

    int threshold = (operatingMode == nn::tc::OperatingMode_Console) ? SleepThresholdConsole : SleepThresholdHandheld;

    if ( temperatureMilliC >= threshold && !g_SleepSignaled )
    {
        EXPECT_TRUE(nn::os::TryWaitSystemEvent(&g_SystemEventSleep));
        g_SleepSignaled = true;
    }
    else
    {
        EXPECT_FALSE(nn::os::TryWaitSystemEvent(&g_SystemEventSleep));
    }

    nn::os::ClearSystemEvent(&g_SystemEventSleep);
}

// 確認する項目：
//
// - ファン回転数は各モードで定義されている上限・下限回転数を超過しない
// - ファン回転数の急激な増加、減少が特定条件を除いて存在しない
// - 与えられたリスト通りに PCV の呼び出しができている
// - スリープのイベントが適切に取得できる
//
// ポリシー、表面温度計算式は今後更新される可能性がある。
// 厳密なファン回転数の裏付けは対象としない。
//
void DispatchEntry(const ThermalCoordinatorTestSettingEntry* pEntry) NN_NOEXCEPT
{
    const int LowestSpeedLevelThreshold = 51;

    nn::tc::impl::SetPowerMode(pEntry->powerMode);
    nn::tc::SetOperatingMode(pEntry->operatingMode);

    nn::fan::Controller controller;
    nn::fan::OpenController(&controller, nn::fan::FanName_Cpu);

    nn::fan::RotationSpeedLevel preSpeedLevel = 0;

    for ( int step = 0; step < pEntry->steps; step++ )
    {
        int internalTemperature;
        if ( pEntry->pcbTempEndMilliC == pEntry->pcbTempStartMilliC )
        {
            internalTemperature = pEntry->pcbTempStartMilliC;
        }
        else
        {
            internalTemperature = pEntry->pcbTempStartMilliC + (step * (pEntry->pcbTempEndMilliC - pEntry->pcbTempStartMilliC)) / pEntry->steps;
        }

        int externalTemperature;
        if ( pEntry->socTempEndMilliC == pEntry->socTempStartMilliC )
        {
            externalTemperature = pEntry->socTempStartMilliC;
        }
        else
        {
            externalTemperature = pEntry->socTempStartMilliC + (step * (pEntry->socTempEndMilliC - pEntry->socTempStartMilliC)) / pEntry->steps;
        }

        nnt::ts::SetTemperatureMilliC(nn::ts::Location_ThermalSensorInternal, internalTemperature);
        nnt::ts::SetTemperatureMilliC(nn::ts::Location_ThermalSensorExternal, externalTemperature);
        nn::tc::impl::Update();

        nn::fan::RotationSpeedLevel postSpeedLevel = nn::fan::GetRotationSpeedLevel(&controller);

        //NN_LOG("[TMP451: %d MilliC, SOC: %d MilliC] RotationSpeedLevel:%d\n", internalTemperature, externalTemperature, postSpeedLevel);

        EXPECT_GE(postSpeedLevel, pEntry->speedLevelMin);
        EXPECT_LE(postSpeedLevel, pEntry->speedLevelMax);

        postSpeedLevel = std::max(postSpeedLevel, LowestSpeedLevelThreshold);

        if ( step > 0 )
        {
            EXPECT_LE(0, std::abs(postSpeedLevel - preSpeedLevel));
            EXPECT_GE(pEntry->speedLevelGradientMax, std::abs(postSpeedLevel - preSpeedLevel));
        }

        CheckPcvTemperature(externalTemperature);

        CheckSleep(std::max(externalTemperature, internalTemperature), pEntry->operatingMode);

        preSpeedLevel = postSpeedLevel;
    }

    nn::fan::CloseController(&controller);
}

} // namespace

#if defined(USE_TABLE_HANDHELD)

// table-handheld 用のテスト
TEST_P(ThermalCoordinatorTestTableHandheld, TestEachTransition)
{
    // 初期状態を設定する。
    nn::tc::impl::SetPowerMode(GetParam().pEntry->powerMode);
    nn::tc::SetOperatingMode(GetParam().pEntry->operatingMode);
    nnt::ts::SetTemperatureMilliC(nn::ts::Location_ThermalSensorInternal, GetParam().pEntry->pcbTempStartMilliC);
    nnt::ts::SetTemperatureMilliC(nn::ts::Location_ThermalSensorExternal, GetParam().pEntry->socTempStartMilliC);

    g_SleepSignaled = false;

    // 60 回実行して初期状態を安定させる。
    // Advanced(prev) ポリシー用の処置。
    for ( int count = 0; count < 60; count++ )
    {
        nn::tc::impl::Update();
    }

    // 遷移を開始する。
    for ( int index = 0; index < GetParam().size; index++ )
    {
        DispatchEntry(GetParam().pEntry + index);
    }

    // TC 内部の signaled に相当するフラグの消去。
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_MinimumAwake);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_SleepReady);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_MinimumAwake);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_FullAwake);
}

#endif // USE_TABLE_HANDHELD

TEST_P(ThermalCoordinatorTest, TestEachTransition)
{
    // 初期状態を設定する。
    nn::tc::impl::SetPowerMode(GetParam().pEntry->powerMode);
    nn::tc::SetOperatingMode(GetParam().pEntry->operatingMode);
    nnt::ts::SetTemperatureMilliC(nn::ts::Location_ThermalSensorInternal, GetParam().pEntry->pcbTempStartMilliC);
    nnt::ts::SetTemperatureMilliC(nn::ts::Location_ThermalSensorExternal, GetParam().pEntry->socTempStartMilliC);

    g_SleepSignaled = false;

    // 60 回実行して初期状態を安定させる。
    // Advanced(prev) ポリシー用の処置。
    for ( int count = 0; count < 60; count++ )
    {
        nn::tc::impl::Update();
    }

    // 遷移を開始する。
    for ( int index = 0; index < GetParam().size; index++ )
    {
        DispatchEntry(GetParam().pEntry + index);
    }

    // TC 内部の signaled に相当するフラグの消去。
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_MinimumAwake);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_SleepReady);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_MinimumAwake);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_FullAwake);
}

// 表面温度 61C が 10.0 秒以上継続した場合のスリープイベントの検知テスト。
TEST(ThermalCounterTest, TestCounterSleep10)
{
    // 表面温度 61C に対応する External(SoC 側)のセンサの値。
    const int ThresholdTemperature = 78;
    const int CoolTemperature = 50;

    // 10.0 秒経過時の nn::tc::impl::Update() で結果が不定になるのを防ぐため 0.5 秒だけ Update のタイミングをずらす。
    const int64_t BaseDelayMs = 500;
    const int64_t LoopDelayMs = 1000;
    const int64_t SignalDelayMs = 10000;
    const int LoopCountMax = 15;

    bool signaled = false;

    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_FullAwake);
    nn::tc::SetOperatingMode(nn::tc::OperatingMode_Handheld);

    // 片方のセンサの値のみを使用する場合は Firmware Debug Settings を変更すること。
    nnt::ts::SetTemperature(nn::ts::Location_ThermalSensorInternal, ThresholdTemperature);
    nnt::ts::SetTemperature(nn::ts::Location_ThermalSensorExternal, ThresholdTemperature);

    nn::TimeSpan begin = nn::os::GetSystemTick().ToTimeSpan();
    nn::tc::impl::Update();

    for ( int loopCount = 0; loopCount < LoopCountMax; loopCount++ )
    {
        nn::TimeSpan target = nn::TimeSpan::FromMilliSeconds(loopCount * LoopDelayMs + BaseDelayMs) + begin;
        nn::TimeSpan delay = target - nn::os::GetSystemTick().ToTimeSpan();

        if ( delay > 0 )
        {
            nn::os::SleepThread(delay);
        }

        nn::tc::impl::Update();

        if ( (loopCount * LoopDelayMs + BaseDelayMs) >= SignalDelayMs && !signaled )
        {
            EXPECT_TRUE(nn::os::TryWaitSystemEvent(&g_SystemEventSleep));
            signaled = true;
        }
        else
        {
            EXPECT_FALSE(nn::os::TryWaitSystemEvent(&g_SystemEventSleep));
        }

        nn::os::ClearSystemEvent(&g_SystemEventSleep);

        // 10 周ごとに秒数を表示する。
        if ( (loopCount % 10) == 0 )
        {
            NN_LOG("%lld milli seconds.\n", loopCount * LoopDelayMs + BaseDelayMs);
        }
    }

    // クールダウン。
    nnt::ts::SetTemperature(nn::ts::Location_ThermalSensorInternal, CoolTemperature);
    nnt::ts::SetTemperature(nn::ts::Location_ThermalSensorExternal, CoolTemperature);
    nn::tc::impl::Update();

    // TC 内部の signaled に相当するフラグの消去。
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_MinimumAwake);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_SleepReady);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_MinimumAwake);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_FullAwake);
}

// 表面温度 58C が 60.0 秒以上継続した場合のスリープイベントの検知テスト。
TEST(ThermalCounterTest, TestCounterSleep60)
{
    // 表面温度 58C に対応する External(SoC 側)のセンサの値。
    const int ThresholdTemperature = 73;
    const int CoolTemperature = 50;

    // 60.0 秒経過時の nn::tc::impl::Update() で結果が不定になるのを防ぐため 0.5 秒だけ Update のタイミングをずらす。
    const int64_t BaseDelayMs = 500;
    const int64_t LoopDelayMs = 1000;
    const int64_t SignalDelayMs = 60000;
    const int LoopCountMax = 65;

    bool signaled = false;

    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_FullAwake);
    nn::tc::SetOperatingMode(nn::tc::OperatingMode_Handheld);

    // 片方のセンサの値のみを使用する場合は Firmware Debug Settings を変更すること。
    nnt::ts::SetTemperature(nn::ts::Location_ThermalSensorInternal, ThresholdTemperature);
    nnt::ts::SetTemperature(nn::ts::Location_ThermalSensorExternal, ThresholdTemperature);

    nn::TimeSpan begin = nn::os::GetSystemTick().ToTimeSpan();
    nn::tc::impl::Update();

    for ( int loopCount = 0; loopCount < LoopCountMax; loopCount++ )
    {
        nn::TimeSpan target = nn::TimeSpan::FromMilliSeconds(loopCount * LoopDelayMs + BaseDelayMs) + begin;
        nn::TimeSpan delay = target - nn::os::GetSystemTick().ToTimeSpan();

        if ( delay > 0 )
        {
            nn::os::SleepThread(delay);
        }

        nn::tc::impl::Update();

        if ( (loopCount * LoopDelayMs + BaseDelayMs) >= SignalDelayMs && !signaled )
        {
            EXPECT_TRUE(nn::os::TryWaitSystemEvent(&g_SystemEventSleep));
            signaled = true;
        }
        else
        {
            EXPECT_FALSE(nn::os::TryWaitSystemEvent(&g_SystemEventSleep));
        }

        nn::os::ClearSystemEvent(&g_SystemEventSleep);

        // 10 周ごとに秒数を表示する。
        if ( (loopCount % 10) == 0 )
        {
            NN_LOG("%lld milli seconds.\n", loopCount * LoopDelayMs + BaseDelayMs);
        }
    }

    // クールダウン。
    nnt::ts::SetTemperature(nn::ts::Location_ThermalSensorInternal, CoolTemperature);
    nnt::ts::SetTemperature(nn::ts::Location_ThermalSensorExternal, CoolTemperature);
    nn::tc::impl::Update();

    // TC 内部の signaled に相当するフラグの消去。
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_MinimumAwake);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_SleepReady);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_MinimumAwake);
    nn::tc::impl::SetPowerMode(nn::tc::impl::PowerMode_FullAwake);
}

void InitializeTcServer() NN_NOEXCEPT
{
    nn::tc::impl::Initialize();
    nn::tc::InitializeWith(g_Manager.GetShared());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::GetThermalEvent(&g_SystemEventSleep, nn::tc::EventTarget_AbnormalTemperature, nn::os::EventClearMode_ManualClear));
}

void FinalizeTcServer() NN_NOEXCEPT
{
    nn::os::DestroySystemEvent(&g_SystemEventSleep);

    nn::tc::Finalize();
    nn::tc::impl::Finalize();
}

}} // namespace nnt::tc

extern "C" void nnMain()
{
    int argc = ::nnt::GetHostArgc();
    char** argv = ::nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::ts::Initialize();
    nn::fan::Initialize();
    nn::pcv::Initialize();
    nnt::pcv::SetTemperatureThresholds(nnt::tc::TemperatureThresholds,
        sizeof(nnt::tc::TemperatureThresholds) / sizeof(nnt::tc::TemperatureThresholds[0]));
    nnt::tc::InitializeTcServer();

    int result = RUN_ALL_TESTS();

    nnt::tc::FinalizeTcServer();
    nn::pcv::Finalize();
    nn::fan::Finalize();
    nn::ts::Finalize();

    ::nnt::Exit(result);
}
