﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/os.h>

#include <nn/fan/fan.h>
#include <nn/tc/tc.h>
#include <nn/tc/tc_FanControl.h>

#include <nnt/gtest/gtest.h>

namespace nnt { namespace fan {

namespace {

bool g_IsFanControlEnabled = false;

class FanTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        g_IsFanControlEnabled = false;
        nn::tc::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::IsFanControlEnabled(&g_IsFanControlEnabled));
        if ( g_IsFanControlEnabled )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::DisableFanControl());
        }
    }

    static void TearDownTestCase()
    {
        if ( g_IsFanControlEnabled )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::EnableFanControl());
        }
        nn::tc::Finalize();
    }
};

void SetAndWaitRotationSpeedLevel(nn::fan::Controller* pController, int level, int64_t delayMilliSeconds) NN_NOEXCEPT
{
    NN_LOG("Set fan level %d.\n", level);
    nn::fan::SetRotationSpeedLevel(pController, level);
    EXPECT_EQ(level, nn::fan::GetRotationSpeedLevel(pController));

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(delayMilliSeconds));
    EXPECT_EQ(level, nn::fan::GetActualRotationSpeedLevel(pController));
}

void SetAndWaitFanEnabled(nn::fan::Controller* pController, bool enabled, int64_t delayMilliSeconds) NN_NOEXCEPT
{
    if ( enabled )
    {
        NN_LOG("Enable fan.\n");
        nn::fan::EnableFan(pController);
    }
    else
    {
        NN_LOG("Disable fan.\n");
        nn::fan::DisableFan(pController);
    }
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(delayMilliSeconds));
}

} // namespace

// ファンの RotationSpeedRate を 1 段階ずつ上げ下げして期待通りの動作になっているか回転音と排気で確認する。
TEST_F(FanTest, IncreaseDecrease)
{
    nn::fan::Initialize();

    nn::fan::Controller controller;
    nn::fan::OpenController(&controller, nn::fan::FanName_Cpu);

    // 初期値退避。
    auto initialLevel = nn::fan::GetRotationSpeedLevel(&controller);

    // 0 にする（15秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, 0, 15000);

    // 段階的な変化（100ミリ秒ずつ変化時間を割り当てる）。
    for ( nn::fan::RotationSpeedLevel level = 0; level <= nn::fan::RotationSpeedLevelMax; level++ )
    {
        SetAndWaitRotationSpeedLevel(&controller, level, 100);
    }

    // 段階的な変化（250ミリ秒ずつ変化時間を割り当てる）。
    for ( nn::fan::RotationSpeedLevel level = nn::fan::RotationSpeedLevelMax; level >= 0; level-- )
    {
        SetAndWaitRotationSpeedLevel(&controller, level, 250);
    }

    // 元の回転数に戻す（15秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, initialLevel, 15000);

    nn::fan::CloseController(&controller);

    nn::fan::Finalize();
}

// ファンの RotationSpeedRate を最大値と最低値に上げ下げして期待通りの動作になっているか回転音と排気で確認する。
TEST_F(FanTest, IncreaseDecreaseFast)
{
    nn::fan::Controller controller;

    nn::fan::Initialize();

    nn::fan::OpenController(&controller, nn::fan::FanName_Cpu);

    // 初期値退避。
    auto initialLevel = nn::fan::GetRotationSpeedLevel(&controller);

    // 0 にする（15秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, 0, 15000);

    // 急激な変化（15秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, nn::fan::RotationSpeedLevelMax, 15000);

    // 急激な変化（15秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, 0, 15000);

    // 元の回転数に戻す（15秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, initialLevel, 15000);

    nn::fan::CloseController(&controller);

    nn::fan::Finalize();
}

// ファンの RotationSpeedRate を最大値と最低値に上げ下げして期待通りの動作になっているか回転音と排気で確認する。
TEST_F(FanTest, IncreaseDecreaseDrastically)
{
    nn::fan::Controller controller;

    nn::fan::Initialize();

    nn::fan::OpenController(&controller, nn::fan::FanName_Cpu);

    // 初期値退避。
    auto initialLevel = nn::fan::GetRotationSpeedLevel(&controller);

    // Max にする（15秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, nn::fan::RotationSpeedLevelMax, 15000);

    // 急激な変化（1秒変化時間を割り当てる）。
    SetAndWaitFanEnabled(&controller, false, 1000);

    // 急激な変化（15秒変化時間を割り当てる）。
    SetAndWaitFanEnabled(&controller, true, 15000);
    EXPECT_EQ(nn::fan::RotationSpeedLevelMax, nn::fan::GetActualRotationSpeedLevel(&controller));

    // 元の回転数に戻す（15秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, initialLevel, 15000);

    nn::fan::CloseController(&controller);

    nn::fan::Finalize();
}

// ファンの RotationSpeedRate を GPIO 制御 で有効な範囲の値にして期待通りの動作になっているか回転音と排気で確認する。
TEST_F(FanTest, Gpio)
{
    nn::fan::Controller controller;

    nn::fan::Initialize();

    nn::fan::OpenController(&controller, nn::fan::FanName_Cpu);

    // 初期値退避。
    auto initialLevel = nn::fan::GetRotationSpeedLevel(&controller);

    // Off にする（5秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, 0, 5000);

    // Lo にする（5秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, 1, 5000);

    // Hi にする（5秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, nn::fan::RotationSpeedLevelMax, 5000);

    // Lo にする（5秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, nn::fan::RotationSpeedLevelMax - 1, 5000);

    // Off にする（5秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, 0, 5000);

    // 元の回転数に戻す（1秒変化時間を割り当てる）。
    SetAndWaitRotationSpeedLevel(&controller, initialLevel, 1000);

    nn::fan::CloseController(&controller);

    nn::fan::Finalize();
}

}} // namespace nnt::fan
