﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <Tasks/OsSemaphoreTask.h>
#include <FileUtility.h>
namespace nnt { namespace abuse {

    OsSemaphoreTask::OsSemaphoreTask(const String& typeName, const String& instanceName) :
        BaseTask(typeName, instanceName),
        m_threadArgs(nullptr),
        m_threads(nullptr),
        m_stacks(nullptr),
        m_semaphore(nullptr),
        m_groupCounter(nullptr),
        m_maxThreads(64),
        m_maxGroups(64),
        m_numGroups(1),
        m_numThreads(1),
        m_numActualThreads(0),
        m_priority(nn::os::DefaultThreadPriority),
        m_threadIterations(10)
    {}

    OsSemaphoreTask::~OsSemaphoreTask()
    {}

    InitStatus OsSemaphoreTask::Initialize(const String& params)
    {
        ArgVector args;
        FileUtility::ParamsToArgs(params, args);

        for(ScriptArg& arg : args)
        {
            if(arg.argName == "MaxThreads")
                FileUtility::TryParseInt(arg, 1, 64, &m_maxThreads);
            else if(arg.argName == "MaxGroups")
                FileUtility::TryParseInt(arg, 1, 64, &m_maxGroups);
            else if(arg.argName == "Priority")
                FileUtility::TryParseInt(arg, nn::os::HighestThreadPriority, nn::os::LowestThreadPriority, &m_priority);
            else if(arg.argName == "NumThreadIterations")
                FileUtility::TryParseInt(arg, 1, std::numeric_limits<int>::max(), &m_threadIterations);
        }

        m_threads = (nn::os::ThreadType*) Platform::Allocate(sizeof(nn::os::ThreadType) * m_maxThreads);

        if(!m_threads)
        {
            LogError("Init Error - could not allocate threads\n");
            return INIT_ERROR;
        }

        for(int i = 0; i < m_maxThreads; ++i)
            ::new (&m_threads[i]) nn::os::ThreadType;

        LogInfo("Created thread array at %p\n", m_threads);
        m_stacks = (void**)Platform::Allocate(sizeof(void*) * m_maxThreads);

        if(!m_stacks)
        {
            LogError("Init Error - could not allocate stack array\n");
            Platform::Free(m_threads);
            m_threads = nullptr;
            return INIT_ERROR;
        }
        int i = 0;
        for(; i < m_maxThreads; ++i)
        {
            m_stacks[i] = Platform::AllocateAligned(4096, nn::os::StackRegionAlignment);

            if(!m_stacks[i])
                break;
        }

        if(i != m_maxThreads)
        {
            LogError("Init Error - could not allocate thread stacks\n");
            for(int j = 0; j <i; ++j)
            {
                Platform::Free(m_stacks[j]);
            }
            Platform::Free(m_stacks);
            Platform::Free(m_threads);
            m_threads = nullptr;
            m_stacks = nullptr;
            return INIT_ERROR;
        }

        m_threadArgs = (SemaphoreThreadArg*)Platform::Allocate( m_maxThreads * sizeof(SemaphoreThreadArg));

        if(!m_threadArgs)
            return INIT_ERROR;

        m_semaphore = (nn::os::SemaphoreType*) Platform::Allocate(m_maxGroups * sizeof(nn::os::SemaphoreType));

        if(!m_semaphore)
            return INIT_ERROR;

        m_groupCounter = (int*) Platform::Allocate(m_maxGroups * sizeof(int));

        return INIT_OK;
    }

    StartStatus OsSemaphoreTask::Start()
    {
        m_numGroups *= 2;
        if(m_numGroups > m_maxGroups || m_numGroups > m_numThreads)
        {
            m_numGroups = 1;

            m_numThreads *= 2;
            if(m_numThreads > m_maxThreads)
                m_numThreads = 1;
        }

        LogInfo("Starting %d threads in %d groups\n", m_numThreads, m_numGroups);

        return START_OK;
    }

    RunStatus OsSemaphoreTask::Run()
    {
        LogVerbose("++Run\n");
        for(int i = 0; i < m_numGroups; ++i)
        {
            nn::os::InitializeSemaphore(&m_semaphore[i], 1, 1);
            m_groupCounter[i] = 0;
        }

        m_numActualThreads = 0;
        for(int i = 0; i < m_numThreads; ++i)
        {
            m_threadArgs[i].task = this;
            m_threadArgs[i].group = i % m_numGroups;
            if(! CreateThread(&m_threads[i], workerFunc, &m_threadArgs[i], m_stacks[i], 4096, m_priority) )
                break;
            ++m_numActualThreads;
        }

        if(m_numActualThreads != m_numThreads)
        {
            LogWarning("Could not create %d threads. Continuing this run with %d threads instead.\n", m_numThreads, m_numActualThreads);
        }

        for(int i = 0; i < m_numActualThreads; ++i)
            nn::os::StartThread(&m_threads[i]);

        for(int i = 0; i < m_numActualThreads; ++i)
            nn::os::WaitThread(&m_threads[i]);

        for(int i = 0; i < m_numActualThreads; ++i)
        {
            DestroyThread(&m_threads[i]);
        }

        for(int i = 0; i < m_numGroups; ++i)
        {
            int groupSize = m_numActualThreads / m_numGroups;

            if(i < m_numActualThreads % m_numGroups)
                ++groupSize;

            int expectedCount = m_threadIterations * groupSize;
            if(expectedCount != m_groupCounter[i])
                LogError("Error: expected count(%d) != actual count(%d)\n", expectedCount, m_groupCounter[i]);
        }

        for(int i = 0; i < m_numGroups; ++i)
            nn::os::FinalizeSemaphore(&m_semaphore[i]);
        LogVerbose("--Run\n");
        return RUN_OK;
    }

    StopStatus OsSemaphoreTask::Stop()
    {
        return STOP_OK;
    }

    ShutdownStatus OsSemaphoreTask::Shutdown()
    {
        if(m_threads)
            Platform::Free(m_threads);

        if(m_stacks)
        {
            for(int i = 0; i < m_maxThreads; ++i)
                Platform::Free(m_stacks[i]);
            Platform::Free(m_stacks);
        }

        if(m_threadArgs)
            Platform::Free(m_threadArgs);

        if(m_semaphore)
            Platform::Free(m_semaphore);

        if(m_groupCounter)
            Platform::Free(m_groupCounter);
        return SHUTDOWN_OK;
    }

    const char* OsSemaphoreTask::GetParamOptions()
    {
        return "MaxThreads=[1 - 64, default =6 4] - maximum number of threads to run\n\
\tMaxGroups=[1 - 64, default = 64] - maximum number of groups\n\
\tPriority=[0 - 31, default = 16] - priority of threads\n\
\tNumThreadIterations=[1+, default = 10] - number of iterations each thread runs before returning";
    }

    void OsSemaphoreTask::workerFunc(void* vSemaphoreThreadArg)
    {
        SemaphoreThreadArg* arg = (SemaphoreThreadArg*)vSemaphoreThreadArg;

        int numItr = arg->task->m_threadIterations;

        for(int i = 0; i < numItr; ++i)
        {
            lockedAdd(arg, i % 3);
        }
    }

    void OsSemaphoreTask::lockedAdd(SemaphoreThreadArg* arg, int lockType)
    {
        nn::os::SemaphoreType* semaphore = &arg->task->m_semaphore[arg->group];
        arg->task->LogVerbose("Adding from core %d\n", nn::os::GetCurrentCoreNumber());
        switch(lockType)
        {
        case 0:
            nn::os::AcquireSemaphore(semaphore);
            break;
        case 1:
            while(!nn::os::TryAcquireSemaphore(semaphore))
                nn::os::YieldThread();
            break;
        case 2:
            while(!nn::os::TimedAcquireSemaphore(semaphore, nn::TimeSpan::FromMicroSeconds(100)))
                nn::os::YieldThread();
            break;
        default:
            arg->task->LogWarning("Unrecognized lockType - received %d but must be between 0 and 2\n", lockType);
        }
        ++arg->task->m_groupCounter[arg->group];
        nn::os::ReleaseSemaphore(semaphore, 1);
    }

}    }
