﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <Tasks/NetworkStressTask.h>
#include <Platform.h>
#include <new>
#include <nn/os.h>
#include <Abuse.h>
#include <FileUtility.h>
#include <algorithm>
#include <cstdlib>
#define CALL_MEMBER_FN(object,ptrToMember)  ((object).*(ptrToMember))
const int ABUSE_MAX_BUFFER_SIZE = 4096;
const int ABUSE_THREAD_STACK_SIZE = 16384;


namespace nnt
{
    namespace abuse
    {

        int NetworkStressTask::s_socketLibraryInitialized = 0;
        int NetworkStressTask::s_currentNetworkTaskId = 0;
        NetworkStressTask::NetworkStressTask(const String& typeName, const String& instanceName)
             : BaseTask(typeName, instanceName),
             m_connected(false),
             m_idleTime(0),
             m_threadNumber(10),
             m_threadArray(nullptr),
             m_commandSocketDescriptor(0),
             m_testType(TestType_Fixed),
             m_protocol(nn::socket::Protocol::IpProto_Tcp),
             m_flood(true),
             m_fixed(true),
             m_direction(Direction_Both),
             m_packetSize(1024),
             m_ipAddress("192.168.0.1"),
             m_commandServerAddress(),
             m_serverAddress(),
             m_networkTaskId(s_currentNetworkTaskId++),
             m_sendCommandList(),
             m_commandListMutex(),
             m_packetMonitor(this)
        {
        }

        NetworkStressTask::~NetworkStressTask()
        {
        }

        bool NetworkStressTask::ProccessParams(const String& params)
        {
            char * buffer = reinterpret_cast<char *>(Platform::Allocate(params.length() + 1));
            if(!buffer)
            {
                LogError("Memory Allocation Failed for Param Proccessing!");
                return false;
            }
            memcpy(buffer,params.c_str(),params.length() + 1);
            int newSize = FileUtility::TrimBuffer(buffer,params.length() + 1,true,true);

            ArgVector argVector;
            FileUtility::ParseArgs(buffer,newSize,argVector);

            auto iter = argVector.begin();
            auto endIter = argVector.end();
            for(;iter != endIter; ++iter)
            {
                if(iter->argName == "Protocol")
                {
                    if(iter->argValue == "tcp")
                    {
                        m_protocol = nn::socket::Protocol::IpProto_Tcp;
                    }
                    else if(iter->argValue == "udpmodest")
                    {
                        m_protocol = nn::socket::Protocol::IpProto_Udp;
                        m_flood = false;
                    }
                    else if(iter->argValue == "udpflood")
                    {
                        m_protocol = nn::socket::Protocol::IpProto_Udp;
                        m_flood = true;
                    }
                    else
                    {
                        LogError("Invalid Argument %s after %s= in Params String!\n",iter->argValue.c_str(), iter->argName.c_str());
                        Platform::Free(buffer);
                        return false;
                    }
                }
                else if(iter->argName == "Type")
                {
                    if(iter->argValue == "fixed")
                    {
                        m_fixed = true;
                        m_testType = TestType_Fixed;
                    }
                    else if(iter->argValue == "varying")
                    {
                        m_fixed = false;
                        m_testType = TestType_Varying;
                    }
                    else if(iter->argValue == "disconnects")
                    {
                        m_testType = TestType_Disconnects;
                    }
                    else if(iter->argValue == "multi")
                    {
                        m_testType = TestType_Multiple;
                    }
                    else
                    {
                        LogError("Invalid Argument %s after %s= in Params String!\n",iter->argValue.c_str(), iter->argName.c_str());
                        Platform::Free(buffer);
                        return false;
                    }
                }
                else if(iter->argName == "Ip")
                {
                    m_ipAddress = iter->argValue;
                }
                else if(iter->argName == "PacketSize")
                {
                    m_packetSize = strtol(iter->argValue.c_str(),0,0);
                }
                else if(iter->argName == "TransferDirection")
                {
                    if(iter->argValue == "upload")
                    {
                        m_direction = Direction_Upload;
                    }
                    else if(iter->argValue == "download")
                    {
                        m_direction = Direction_Download;
                    }
                    else if(iter->argValue == "both")
                    {
                         m_direction = Direction_Both;
                    }
                    else
                    {
                        LogError("Invalid Argument %s after %s= in Params String!\n",iter->argValue.c_str(), iter->argName.c_str());
                        Platform::Free(buffer);
                        return false;
                    }
                }
                else if(iter->argName == "IdleTime")
                {
                    m_idleTime = strtol(iter->argValue.c_str(),0,0) / (double)1000;
                }
                else if(iter->argName == "ThreadCount")
                {
                    m_threadNumber = strtol(iter->argValue.c_str(),0,0);
                }
                else
                {
                    LogError("Invalid Argument %s in Params String!\n",iter->argName.c_str());
                    Platform::Free(buffer);
                    return false;
                }
            }

            Platform::Free(buffer);
            return true;
        }  // NOLINT(impl/function_size)

        NetworkStressTask::NetworkThreadArg::NetworkThreadArg(
             NetworkStressTask * parent,
             NetworkThread* networkThread,
             RunStatus (*runningTest)(int socketDescriptor, NetworkThreadArg * arg),
             TestType type,
             nn::socket::Protocol protocol,
             Direction direction,
             bool flood,
             nn::socket::SockAddrIn serverAddress,
             int index,
             int packetSize,
             double idleTime)
             :
             parent(parent),
             networkThread(networkThread),
             runningTest(runningTest),
             type(type),
             protocol(protocol),
             direction(direction),
             flood(flood),
             firstRun(true),
             serverAddress(serverAddress),
             index(index),
             packetSize(packetSize),
             packetId(0),
             varyingPacketSize(1),
             waitStart(0),
             idleTime(idleTime)
             {}

        NetworkStressTask::NetworkThread::NetworkThread(
             NetworkStressTask * parent,
             RunStatus (*runningTest)(int socketDescriptor, NetworkThreadArg * arg),
             TestType type,
             nn::socket::Protocol protocol,
             Direction direction,
             bool flood,
             nn::socket::SockAddrIn serverAddress,
             int index,
             int packetSize,
             double idleTime)
             :
             packetMonitor(parent),
             m_thread(),
             m_arg(parent,this,runningTest,type,protocol,direction,flood,serverAddress,index,packetSize,idleTime),
             m_stopping(false),
             m_exitedWithError(false)
         {
             m_stack = Platform::AllocateAligned(ABUSE_THREAD_STACK_SIZE, nn::os::StackRegionAlignment);
             if(!m_stack)
             {
                 parent->LogError("Failed To Allocate Stack For Network Thread!\nThread Index: Stack Size: %d\n",index,ABUSE_THREAD_STACK_SIZE);
             }

         }

         int NetworkStressTask::NetworkThread::Initialize()
         {
            if(!m_stack)
            {
                m_arg.parent->LogError("Failed To Initialize Thread ID %d Due To Unallocated Stack.\n",m_arg.index);
                return -1;
            }

            if(!m_arg.parent->CreateThread(&m_thread, &ThreadFunc,(void*) &m_arg, m_stack, ABUSE_THREAD_STACK_SIZE, 0))
            {
                m_arg.parent->LogError("Failed To Initialize Thread ID %d: CreateThread Failed.\n",m_arg.index);
                return -1;
            }
            nn::os::InitializeMutex(&m_threadMutex, false, 0);
            return 0;
         }

         void NetworkStressTask::NetworkThread::Start()
         {
             nn::os::StartThread(&m_thread);
         }

         void NetworkStressTask::NetworkThread::ThreadFunc(void * args)
         {
             NetworkThreadArg* threadArgs = (NetworkThreadArg* )args;
             threadArgs->parent->LogInfo("[Thread %d] Running... \n",threadArgs->index);
             char sendBuffer[1];
             int socketDescriptor;

             if(threadArgs->protocol == nn::socket::Protocol::IpProto_Tcp)
             {
                threadArgs->parent->LogInfo("[Thread %d] Creating TCP Socket... \n",threadArgs->index);
                socketDescriptor = threadArgs->parent->CreateTcpStreamSocket();
                if(socketDescriptor == -1)
                {
                    threadArgs->networkThread->m_exitedWithError = true;
                    return;
                }
                threadArgs->parent->LogInfo("[Thread %d] Initialize Socket Success!\n",threadArgs->index);

                Platform::connect(socketDescriptor,&threadArgs->serverAddress,sizeof(threadArgs->serverAddress));

                Header header(threadArgs->type,threadArgs->direction,threadArgs->parent->m_networkTaskId,threadArgs->index,0);
                SendDataTcp(socketDescriptor,header,sendBuffer,threadArgs->parent);
             }
             else
             {
                threadArgs->parent->LogInfo("[Thread %d] Creating UDP Socket... \n",threadArgs->index);
                socketDescriptor = threadArgs->parent->CreateUdpDgramSocket();
                if(socketDescriptor == -1)
                {
                    threadArgs->networkThread->m_exitedWithError = true;
                    return;
                }
                threadArgs->parent->LogInfo("[Thread %d] Initialize Socket Success!\n",threadArgs->index);
             }
             bool t = true;
             do
             {
                 //threadArgs->parent->LogInfo("[Thread %d] Run Thread!\n",threadArgs->index);
                 threadArgs->runningTest(socketDescriptor,threadArgs);

                 nn::os::LockMutex(&threadArgs->networkThread->m_threadMutex);
                 if(threadArgs->networkThread->m_stopping)
                 {
                     nn::os::UnlockMutex(&threadArgs->networkThread->m_threadMutex);
                     break;
                 }
                 nn::os::UnlockMutex(&threadArgs->networkThread->m_threadMutex);
                 nn::os::YieldThread();

             } while(t);
         }

         void NetworkStressTask::NetworkThread::WaitOnThread()
         {
             nn::os::WaitThread(&m_thread);
         }

         void NetworkStressTask::NetworkThread::Stop()
         {
            nn::os::LockMutex(&m_threadMutex);
            m_stopping = true;
            nn::os::UnlockMutex(&m_threadMutex);

         }

         void NetworkStressTask::NetworkThread::Shutdown()
         {
             nn::os::FinalizeMutex(&m_threadMutex);
             nn::os::DestroyThread(&m_thread);
         }

         NetworkStressTask::NetworkThread::~NetworkThread()
         {
             if(m_stack)
                 Platform::Free(m_stack);
         }

        //Pass Params for network stress test types
        // Protocol=<protocol> (tcp, udpmodest, udpflood)
        // Type=<type> (fixed, varying, disconnects, multi)
            // fixed - fixed size packets
            // varying - varying sized packets
            // disconnects - fixed size packets with random disconnects
            // multi - varying sized packets with random disconnects
        // Ip=<Ip> (Ip Address of server)
        // PacketSize=<size> (size of packets in bytes) in fixed this is the size sent, in varying this is the max size sent, with 0 being the min
        // TransferDirection=<direction> (upload,download,both)
        // IdleTime=<time in millisecond> time between transfers in milliseconds
        InitStatus NetworkStressTask::Initialize(const String& params)
        {


            LogInfo("Network Stress Test Initialize\n");
            LogInfo("Passed Parameters: %s\n", params.c_str());


            if(!ProccessParams( params))
            {
                return INIT_ERROR;
            }
            SetCurrentTest();

            nn::os::InitializeMutex(&m_commandListMutex, false, 0);

            LogInfo("Initializing Command Socket...\n");
            m_commandSocketDescriptor = CreateTcpStreamSocket();
            if(m_commandSocketDescriptor == -1)
            {
                return INIT_ERROR;
            }
            LogInfo("Initialize Command Socket Success!\n");


            memset(&m_commandServerAddress,0,sizeof(m_commandServerAddress));
            m_commandServerAddress.sin_family = nn::socket::Family::Af_Inet;
            m_commandServerAddress.sin_port = Platform::htons(8050);
            nn::socket::InetPton(m_commandServerAddress.sin_family,m_ipAddress.c_str(),&m_commandServerAddress.sin_addr.S_addr);

            LogInfo("Command Socket Connecting...\n");
            if(Connect(m_commandServerAddress,m_commandSocketDescriptor) == -1)
            {
                return INIT_ERROR;
            }
            LogInfo("Command Connection Initialized!\n");

            //Send initial command to tell server my abuse task id
            CommandHeader commandHeader;
            commandHeader.networkStressId = m_networkTaskId;
            CommandContainer command(commandHeader,0,0);
            SendCommand(command);

            if(Platform::SetNonBlocking(m_commandSocketDescriptor,true) == -1)
            {
               LogError("Error Setting Command Socket to Non-Blocking!\nError Num: %d",Platform::GetLastError());
               return INIT_ERROR;
            }


            memset(&m_serverAddress,0,sizeof(m_serverAddress));
            m_serverAddress.sin_family = nn::socket::Family::Af_Inet;

            if(m_protocol == nn::socket::Protocol::IpProto_Tcp)
                m_serverAddress.sin_port = Platform::htons(8052);
            else
                m_serverAddress.sin_port = Platform::htons(8054);

            nn::socket::InetPton(m_serverAddress.sin_family,m_ipAddress.c_str(),&m_serverAddress.sin_addr.S_addr);


            LogInfo("Initializing Network Threads...\n");
            m_threadArray = (NetworkThread *)Platform::Allocate(sizeof(NetworkThread) * m_threadNumber);
            for(int i = 0; i < m_threadNumber; ++i)
            {
                new (m_threadArray + i) NetworkThread(this,RunningTest,m_testType, m_protocol,m_direction,m_flood,m_serverAddress,i,m_packetSize,m_idleTime);
                if(m_threadArray[i].Initialize() == -1)
                {
                    return INIT_ERROR;
                }
            }
            LogInfo("Initialize Network Thread Success!\n");

            LogInfo("Starting Network Threads...\n");
            for(int i = 0; i < m_threadNumber; ++i)
            {
                m_threadArray[i].Start();
            }
            LogInfo("Network Threads Started!\n");

            return INIT_OK;
        }

        StartStatus NetworkStressTask::Start()
        {
            LogVerbose("Network Stress Test Start\n");

            return START_OK;
        }

        RunStatus NetworkStressTask::Run()
        {
            nn::os::LockMutex(&m_commandListMutex);
            auto iter = m_sendCommandList.begin();
            auto iterEnd = m_sendCommandList.end();
            while(iter != iterEnd)
            {
                SendCommand(*iter);
                auto tempIter = iter;
                ++iter;
                m_sendCommandList.erase(tempIter);

            }
            nn::os::UnlockMutex(&m_commandListMutex);
            bool t = true;
            while (t)
            {

                CommandContainer command(CommandHeader(), 0, ABUSE_MAX_BUFFER_SIZE);
                int result = ReceiveCommand(command);
                if (result == -2)
                {
                    break;
                }
                else if (result == 0)
                {
                    LogInfo("Abuse Server Connection Shutdown.\n");
                    return RUN_KILL;
                }
                else if(result == -1)
                {
                    LogError("Error! ReceiveCommand Failed.\n");
                    return RUN_ERROR;
                }
                HandleReceivedCommand(command);

            }

            return RUN_YIELD;
        }

        RunStatus NetworkStressTask::RunTcpUploadDownloadFixedTest(int socketDescriptor, NetworkThreadArg * arg)
        {
            //Yield if we are in the idle period between sends
            double currentTime = Platform::GetPlatformTimeSinceStart();
            if(currentTime - arg->waitStart < arg->idleTime)
            {
                return RUN_YIELD;
            }
            arg->waitStart = currentTime;
            // LogWarning("Network Download Run...\n");

            char receiveBuffer[ABUSE_MAX_BUFFER_SIZE];
            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];

            //
            memset(sendBuffer,0xAAAAAAAA,arg->packetSize);

            Header sendHeader(arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetSize,arg->idleTime);

          //  LogInfo("Sending Data...\n");
            int sendResult = SendDataTcp(socketDescriptor,sendHeader, sendBuffer,arg->parent);
            if(sendResult == -1)
            {
                return RUN_ERROR;
            }

          //  LogInfo("Receiving Data...\n");
            Header receiveHeader;
            memset(receiveBuffer,0,sizeof(receiveBuffer));
            int receiveResult = ReceiveDataTcp(socketDescriptor,receiveHeader,receiveBuffer,ABUSE_MAX_BUFFER_SIZE,arg->parent);
            if( receiveResult == -1)
            {
                 return RUN_ERROR;
            }
            else if(receiveResult == 0)
            {
                return RUN_KILL;
            }
            CheckDataValidity(&sendHeader,sizeof(Header),&receiveHeader,sizeof(Header),arg->parent);

            CheckDataValidity(sendBuffer,sendHeader.dataSize,receiveBuffer,receiveHeader.dataSize,arg->parent);

            return RUN_YIELD;
        }
        RunStatus NetworkStressTask::RunTcpUploadDownloadVaryingTest(int socketDescriptor, NetworkThreadArg * arg)
        {
            double currentTime = Platform::GetPlatformTimeSinceStart();
            if(currentTime - arg->waitStart < arg->idleTime)
            {
                return RUN_YIELD;
            }
            arg->waitStart = currentTime;

            char receiveBuffer[ABUSE_MAX_BUFFER_SIZE];
            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];

            //
            memset(sendBuffer,0xAAAAAAAA,arg->varyingPacketSize);

            Header sendHeader(arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetSize,arg->idleTime);
          //  LogInfo("Sending Data...\n");
            int sendResult = SendDataTcp(socketDescriptor,sendHeader, sendBuffer,arg->parent);
            if(sendResult == -1)
            {
                return RUN_ERROR;
            }

          //  LogInfo("Receiving Data...\n");
            Header receiveHeader;
            memset(receiveBuffer,0,sizeof(receiveBuffer));
            int receiveResult = ReceiveDataTcp(socketDescriptor,receiveHeader,receiveBuffer,ABUSE_MAX_BUFFER_SIZE,arg->parent);
            if( receiveResult == -1)
            {
                 return RUN_ERROR;
            }
            else if(receiveResult == 0)
            {
                return RUN_KILL;
            }
            CheckDataValidity(&sendHeader,sizeof(Header),&receiveHeader,sizeof(Header),arg->parent);

            CheckDataValidity(sendBuffer,sendHeader.dataSize,receiveBuffer,receiveHeader.dataSize,arg->parent);

            //increase packet size until we hit the set packet size, then wrap back around.
            ++arg->varyingPacketSize;
            if(arg->varyingPacketSize > arg->packetSize)
                arg->varyingPacketSize = 1;

            return RUN_YIELD;
        }
        RunStatus NetworkStressTask::RunTcpUploadFixedTest(int socketDescriptor, NetworkThreadArg * arg)
        {
            double currentTime = Platform::GetPlatformTimeSinceStart();
            if(currentTime - arg->waitStart < arg->idleTime)
            {
                return RUN_YIELD;
            }
            arg->waitStart = currentTime;

            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];

            //
            memset(sendBuffer,0xAAAAAAAA,arg->packetSize);

            Header sendHeader(arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetSize,arg->idleTime);
          //  LogInfo("Sending Data...\n");
            int sendResult = SendDataTcp(socketDescriptor,sendHeader, sendBuffer,arg->parent);
            if(sendResult == -1)
            {
                return RUN_ERROR;
            }

            return RUN_YIELD;
        }
        RunStatus NetworkStressTask::RunTcpUploadVaryingTest(int socketDescriptor, NetworkThreadArg * arg)
        {
            double currentTime = Platform::GetPlatformTimeSinceStart();
            if(currentTime - arg->waitStart < arg->idleTime)
            {
                return RUN_YIELD;
            }
            arg->waitStart = currentTime;

            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];

            //
            memset(sendBuffer,0xAAAAAAAA,arg->varyingPacketSize);

            Header sendHeader(arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetSize,arg->idleTime);
          //  LogInfo("Sending Data...\n");
            int sendResult = SendDataTcp(socketDescriptor,sendHeader, sendBuffer,arg->parent);
            if(sendResult == -1)
            {
                return RUN_ERROR;
            }

            ++arg->varyingPacketSize;
            if(arg->varyingPacketSize > arg->packetSize)
                arg->varyingPacketSize = 1;

            return RUN_YIELD;
        }
        RunStatus NetworkStressTask::RunTcpDownloadFixedTest(int socketDescriptor, NetworkThreadArg * arg)
        {

            double currentTime = Platform::GetPlatformTimeSinceStart();
            if(currentTime - arg->waitStart < arg->idleTime)
            {
                return RUN_YIELD;
            }
            arg->waitStart = currentTime;


            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];
            memset(sendBuffer,0xAAAAAAAA,arg->packetSize);

            Header sendHeader( arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetSize,arg->idleTime);

            if(arg->firstRun)
            {
                arg->firstRun = false;

              //  LogInfo("Sending Data...\n");
                int sendResult = SendDataTcp(socketDescriptor,sendHeader, sendBuffer,arg->parent);
                if(sendResult == -1)
                {
                    return RUN_ERROR;
                }

            }

            char receiveBuffer[ABUSE_MAX_BUFFER_SIZE];


            Header receiveHeader;
            memset(receiveBuffer,0,sizeof(receiveBuffer));
            int receiveResult = ReceiveDataTcp(socketDescriptor,receiveHeader,receiveBuffer,ABUSE_MAX_BUFFER_SIZE,arg->parent);
            if( receiveResult == -1)
            {
                 return RUN_ERROR;
            }
            else if(receiveResult == 0)
            {
                return RUN_KILL;
            }

            CheckDataValidity(&sendHeader,sizeof(Header),&receiveHeader,sizeof(Header),arg->parent);
            CheckDataValidity(sendBuffer,sendHeader.dataSize,receiveBuffer,receiveHeader.dataSize,arg->parent);

            return RUN_YIELD;
        }
        RunStatus NetworkStressTask::RunTcpDownloadVaryingTest(int socketDescriptor, NetworkThreadArg * arg)
        {
            static bool firstRun = true;
            double currentTime = Platform::GetPlatformTimeSinceStart();
            if(currentTime - arg->waitStart < arg->idleTime)
            {
                return RUN_YIELD;
            }
            arg->waitStart = currentTime;

            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];
            memset(sendBuffer,0xAAAAAAAA,arg->packetSize);

            if(firstRun)
            {
                firstRun = false;
                Header sendHeader(arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetSize,arg->idleTime);

              //  LogInfo("Sending Data...\n");
                int sendResult = SendDataTcp(socketDescriptor,sendHeader, sendBuffer,arg->parent);
                if(sendResult == -1)
                {
                    return RUN_ERROR;
                }

            }

            char receiveBuffer[ABUSE_MAX_BUFFER_SIZE];


            Header receiveHeader;
            memset(receiveBuffer,0,sizeof(receiveBuffer));
            int receiveResult = ReceiveDataTcp(socketDescriptor,receiveHeader,receiveBuffer,ABUSE_MAX_BUFFER_SIZE,arg->parent);
            if( receiveResult == -1)
            {
                 return RUN_ERROR;
            }
            else if(receiveResult == 0)
            {
                return RUN_KILL;
            }
           // CheckDataValidity(&sendHeader,sizeof(Header),&receiveHeader,sizeof(Header));

            CheckDataValidity(sendBuffer,receiveHeader.dataSize,receiveBuffer,receiveHeader.dataSize,arg->parent);

            ++arg->varyingPacketSize;
            if(arg->varyingPacketSize > arg->packetSize)
                arg->varyingPacketSize = 1;

            return RUN_YIELD;
        }

        RunStatus NetworkStressTask::RunUdpUploadDownloadFixedTest(int socketDescriptor, NetworkThreadArg * arg)
        {
            // LogWarning("Network Download Run...\n");

            char receiveBuffer[ABUSE_MAX_BUFFER_SIZE];
            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];
            if(arg->firstRun)
            {
                arg->firstRun = false;
                if(Platform::SetNonBlocking(socketDescriptor,true) == -1)
                {
                    arg->parent->LogError("[Thread %d] Error Setting Socket to Non-Blocking!\nError Num: %d",arg->index,Platform::GetLastError());
                    return RUN_ERROR;
                }
            }
            //
            memset(sendBuffer,0xAAAAAAAA,arg->packetSize);

            double currentTime = Platform::GetPlatformTimeSinceStart();
            if(currentTime - arg->waitStart >= arg->idleTime)
            {
              //  LogInfo("Sending Data...\n");

                UdpHeader sendHeader(arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetId++,arg->packetSize,Info_None,arg->idleTime);
                int sendResult = SendDataUdp(socketDescriptor,sendHeader, sendBuffer,arg->parent,&arg->serverAddress);
                if(sendResult == -1)
                {
                    return RUN_ERROR;
                }
                arg->waitStart = currentTime;
             }

            for(int i = 0; i < 20; ++i)
            {
              //  LogInfo("Receiving Data...\n");
                UdpHeader receiveHeader;
                memset(receiveBuffer,0,sizeof(receiveBuffer));
                int receiveResult = ReceiveDataUdp(socketDescriptor,receiveHeader,receiveBuffer,ABUSE_MAX_BUFFER_SIZE,arg->parent);
                if( receiveResult == -1)
                {
                    if(Platform::GetLastError() == nn::socket::Errno::EWouldBlock)
                    {
                        return RUN_YIELD;
                    }
                     return RUN_ERROR;
                }
                else if(receiveResult == 0)
                {
                    return RUN_KILL;
                }

                //CheckDataValidity(&sendHeader,sizeof(UdpHeader),&receiveHeader,sizeof(UdpHeader));
                CheckDataValidity(sendBuffer,receiveHeader.dataSize,receiveBuffer,receiveHeader.dataSize,arg->parent);
                arg->networkThread->packetMonitor.VerifyPacket(receiveHeader.packetID,arg->index);
            }
            return RUN_YIELD;
        }
        RunStatus NetworkStressTask::RunUdpUploadDownloadVaryingTest(int socketDescriptor, NetworkThreadArg * arg)
        {
            // LogWarning("Network Download Run...\n");

            char receiveBuffer[ABUSE_MAX_BUFFER_SIZE];
            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];
            if(arg->firstRun)
            {
                arg->firstRun = false;
                if(Platform::SetNonBlocking(socketDescriptor,true) == -1)
                {
                    arg->parent->LogError("[Thread %d] Error Setting Socket to Non-Blocking!\nError Num: %d",arg->index,Platform::GetLastError());
                    return RUN_ERROR;
                }
            }
            //
            memset(sendBuffer,0xAAAAAAAA,arg->packetSize);

            double currentTime = Platform::GetPlatformTimeSinceStart();
            if(currentTime - arg->waitStart >= arg->idleTime)
            {
              //  LogInfo("Sending Data...\n");

                UdpHeader sendHeader(arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetId++,arg->packetSize,Info_None,arg->idleTime);
                int sendResult = SendDataUdp(socketDescriptor,sendHeader, sendBuffer,arg->parent,&arg->serverAddress);
                if(sendResult == -1)
                {
                    return RUN_ERROR;
                }
                ++arg->varyingPacketSize;
                arg->varyingPacketSize %= arg->packetSize;

                arg->waitStart = currentTime;
             }

            for(int i = 0; i < 10; ++i)
            {
              //  LogInfo("Receiving Data...\n");
                UdpHeader receiveHeader;
                memset(receiveBuffer,0,sizeof(receiveBuffer));
                int receiveResult = ReceiveDataUdp(socketDescriptor,receiveHeader,receiveBuffer,ABUSE_MAX_BUFFER_SIZE,arg->parent);
                if( receiveResult == -1)
                {
                    if(Platform::GetLastError() == nn::socket::Errno::EWouldBlock)
                    {
                        return RUN_YIELD;
                    }
                     return RUN_ERROR;
                }
                else if(receiveResult == 0)
                {
                    return RUN_KILL;
                }

                //CheckDataValidity(&sendHeader,sizeof(UdpHeader),&receiveHeader,sizeof(UdpHeader));
                CheckDataValidity(sendBuffer,receiveHeader.dataSize,receiveBuffer,receiveHeader.dataSize,arg->parent);
                arg->networkThread->packetMonitor.VerifyPacket(receiveHeader.packetID,arg->index);
            }
            return RUN_YIELD;
        }
        RunStatus NetworkStressTask::RunUdpUploadFixedTest(int socketDescriptor, NetworkThreadArg * arg)
        {
           // (int)socketDescriptor;
            double currentTime = Platform::GetPlatformTimeSinceStart();
            if(currentTime - arg->waitStart < arg->idleTime)
            {
                return RUN_YIELD;
            }
            arg->waitStart = currentTime;


            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];
            memset(sendBuffer,0xAAAAAAAA,arg->packetSize);
            UdpHeader sendHeader;
            sendHeader.Fill(arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetId++,arg->packetSize,Info_None,arg->idleTime);

            int sendResult = SendDataUdp(socketDescriptor,sendHeader, sendBuffer,arg->parent,&arg->serverAddress);
            if(sendResult == -1)
            {
                arg->parent->LogError("[Thread %d] Error Sending UDP Data! Error Num:%d",arg->index, Platform::GetLastError());
                return RUN_ERROR;
            }

            return RUN_YIELD;
        }
        RunStatus NetworkStressTask::RunUdpUploadVaryingTest(int socketDescriptor, NetworkThreadArg * arg)
        {
        //    (int)socketDescriptor;
           double currentTime = Platform::GetPlatformTimeSinceStart();
            if(currentTime - arg->waitStart < arg->idleTime)
            {
                return RUN_YIELD;
            }
            arg->waitStart = currentTime;

            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];
            memset(sendBuffer,0xAAAAAAAA,arg->packetSize);
            arg->varyingPacketSize %= arg->packetSize;
            //arg->parent->LogInfo("[Thread %d][Packet:%d]Sending Upload...\n",arg->index,arg->packetId);
            UdpHeader sendHeader(arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetId++,arg->varyingPacketSize++,Info_None,arg->idleTime);



            int sendResult = SendDataUdp(socketDescriptor,sendHeader, sendBuffer,arg->parent,&arg->serverAddress);
            if(sendResult == -1)
            {
                arg->parent->LogError("[Thread %d] Error Sending UDP Data! Error Num:%d",arg->index, Platform::GetLastError());
                return RUN_ERROR;
            }


            return RUN_YIELD;
        }
        RunStatus NetworkStressTask::RunUdpDownloadFixedTest(int socketDescriptor, NetworkThreadArg * arg)
        {


            char receiveBuffer[ABUSE_MAX_BUFFER_SIZE];
            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];

            //
            UdpHeader sendHeader;
            memset(sendBuffer,0xAAAAAAAA,arg->packetSize);
            if(arg->firstRun)
            {
                arg->firstRun = false;
                sendHeader.Fill(arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetId++,arg->packetSize,Info_None,arg->idleTime);
                if(Platform::SetNonBlocking(socketDescriptor,true) == -1)
                {
                    arg->parent->LogError("[Thread %d] Error Setting Socket to Non-Blocking!\nError Num: %d",arg->index,Platform::GetLastError());
                    return RUN_ERROR;
                }
                arg->parent->LogInfo("[Thread %d] Sending Starting Data for UDP Download...\n",arg->index);
                int sendResult = SendDataUdp(socketDescriptor,sendHeader, sendBuffer,arg->parent,&arg->serverAddress);
                if(sendResult == -1)
                {
                    return RUN_ERROR;
                }
            }

          //  LogInfo("Receiving Data...\n");
            //Receive a few times to keep up with server send flood.
            for(int i = 0; i < 150; ++i)
            {
                UdpHeader receiveHeader;
                memset(receiveBuffer,0,sizeof(receiveBuffer));
                int receiveResult = ReceiveDataUdp(socketDescriptor,receiveHeader,receiveBuffer,ABUSE_MAX_BUFFER_SIZE,arg->parent);
                if( receiveResult == -1)
                {
                    nn::socket::Errno error = Platform::GetLastError();
                    if(error == nn::socket::Errno::EWouldBlock)
                    {
                        return RUN_YIELD;
                    }
                    return RUN_KILL;
                }
                else if(receiveResult == 0)
                {
                    return RUN_KILL;
                }

                int result = HandleInfo(receiveHeader,receiveBuffer,arg);
                if(result == 1)
                {
                    return RUN_YIELD;
                }
                if(result == -1)
                {
                    return RUN_ERROR;
                }


               // CheckDataValidity(&sendHeader,sizeof(UdpHeader),&receiveHeader,sizeof(UdpHeader));
                //Make sure data has not changed.
                CheckDataValidity(sendBuffer,receiveHeader.dataSize,receiveBuffer,receiveHeader.dataSize,arg->parent);
                //Make sure we aren't dropping packets
                arg->networkThread->packetMonitor.VerifyPacket(receiveHeader.packetID,arg->index);


                //UdpHeader shutdownHeader(TestType_Fixed,Direction_Download,0,sizeof(m_serverThreadId),Info_SocketShutdown,0);
               // SendDataUdp(m_udpSocketDescriptor,shutdownHeader, reinterpret_cast<char *>(&m_serverThreadId));
            }
            arg->parent->LogWarning("[Thread %d] Warning! Client Receive Loop Not Keeping Up With Sent Packets! Will Most Likely Result In Packet Loss.\n",arg->index);
            return RUN_YIELD;
        }
        RunStatus NetworkStressTask::RunUdpDownloadVaryingTest(int socketDescriptor, NetworkThreadArg * arg)
        {

            char receiveBuffer[ABUSE_MAX_BUFFER_SIZE];
            char sendBuffer[ABUSE_MAX_BUFFER_SIZE];
            arg->parent->LogInfo("Test!@...\n");
            //
            UdpHeader sendHeader;
            memset(sendBuffer,0xAAAAAAAA,arg->packetSize);
            if(arg->firstRun)
            {
                arg->firstRun = false;
                sendHeader.Fill(arg->type,arg->direction,arg->parent->m_networkTaskId,arg->index,arg->packetId++,arg->packetSize,Info_None,arg->idleTime);
                if(Platform::SetNonBlocking(socketDescriptor,true) == -1)
                {
                    arg->parent->LogError("[Thread %d] Error Setting Socket to Non-Blocking!\nError Num: %d",arg->index,Platform::GetLastError());
                    return RUN_ERROR;
                }
                arg->parent->LogInfo("Sending Startup Data...\n");
                int sendResult = SendDataUdp(socketDescriptor,sendHeader, sendBuffer,arg->parent,&arg->serverAddress);
                if(sendResult == -1)
                {
                    return RUN_ERROR;
                }
            }

          //  LogInfo("Receiving Data...\n");
            //Receive a few times to keep up with server send flood.
            for(int i = 0; i < 150; ++i)
            {
                UdpHeader receiveHeader;
                memset(receiveBuffer,0,sizeof(receiveBuffer));
                int receiveResult = ReceiveDataUdp(socketDescriptor,receiveHeader,receiveBuffer,ABUSE_MAX_BUFFER_SIZE,arg->parent);
                if( receiveResult == -1)
                {
                    nn::socket::Errno error = Platform::GetLastError();
                    if(error == nn::socket::Errno::EWouldBlock)
                    {
                        return RUN_YIELD;
                    }
                    return RUN_KILL;
                }
                else if(receiveResult == 0)
                {
                    return RUN_KILL;
                }

                int result = HandleInfo(receiveHeader,receiveBuffer,arg);
                if(result == 1)
                {
                    return RUN_YIELD;
                }
                if(result == -1)
                {
                    return RUN_ERROR;
                }


               // CheckDataValidity(&sendHeader,sizeof(UdpHeader),&receiveHeader,sizeof(UdpHeader));
                //Make sure data has not changed.
                CheckDataValidity(sendBuffer,receiveHeader.dataSize,receiveBuffer,receiveHeader.dataSize,arg->parent);
                //Make sure we aren't dropping packets
                arg->networkThread->packetMonitor.VerifyPacket(receiveHeader.packetID,arg->index);

                //UdpHeader shutdownHeader(TestType_Fixed,Direction_Download,0,sizeof(m_serverThreadId),Info_SocketShutdown,0);
               // SendDataUdp(m_udpSocketDescriptor,shutdownHeader, reinterpret_cast<char *>(&m_serverThreadId));
            }
            arg->parent->LogWarning("[Thread %d] Warning! Client Receive Loop Not Keeping Up With Sent Packets! Will Most Likely Result In Packet Loss.\n",arg->index);
            return RUN_YIELD;
        }

        StopStatus NetworkStressTask::Stop()
        {

            LogVerbose("Network Stress Test Stop\n");
            return STOP_OK;
        }

        ShutdownStatus NetworkStressTask::Shutdown()
        {

            LogInfo("Shutting Down Network Threads..\n");
            for(int i = 0; i < m_threadNumber; ++i)
            {
                m_threadArray[i].Stop();
                LogInfo("Shut Down Command For Thread %d Sent.\n",i);
            }
            LogInfo("All Threads Shutting Down.\n");
            for(int i = 0; i < m_threadNumber; ++i)
            {
                LogInfo("Waiting For Thread %d To Shut Down.\n",i);
                m_threadArray[i].WaitOnThread();
                LogInfo("Thread %d Shut Down.\n",i);
                if(m_protocol == nn::socket::Protocol::IpProto_Udp)
                {
                    LogInfo("Sending Remote Shutdown Signal For Server Side Thread.\n");
                    CommandHeader commandHeader(Command_KillThread,m_networkTaskId,i,0);
                    CommandContainer commandContainer(commandHeader,nullptr,0);
                    SendCommand(commandContainer);
                }
            }
            LogInfo("All Threads Shut Down.\n");

            nn::os::FinalizeMutex(&m_commandListMutex);

            if(m_threadArray)
            {
                for(int i = 0; i < m_threadNumber; ++i)
                {
                    m_threadArray[i].~NetworkThread();
                }
                Platform::Free(m_threadArray);
            }

            if(ShutdownConnectedSocket(m_commandSocketDescriptor) == -1)
            {
                return SHUTDOWN_ERROR;
            }

            LogInfo("Network Stress Test Shutdown Complete\n");

            return SHUTDOWN_OK;
        }

        int NetworkStressTask::InitializeSocketLibrary()
        {
            ScopedMutex mutex = Abuse::GeneralMutexScopedLock();
            if(!s_socketLibraryInitialized)
            {
                LogInfo("Initializing Socket0 Library...\n");

                if(Platform::InitializeSocketLib() == -1)
                {
                    LogError("Socket0 Library Initialize Failed!\n Error Num: %i\n", Platform::GetLastError());
                    return -1;
                }
                LogInfo("Socket0 Library Initialize Success!\n");
                ++s_socketLibraryInitialized;
            }
            else
                ++s_socketLibraryInitialized;

            return 0;
        }
        int NetworkStressTask::FinalizeSocketLibrary()
        {
            ScopedMutex mutex = Abuse::GeneralMutexScopedLock();
            if(s_socketLibraryInitialized <= 1)
            {
                LogWarning("Socket0 Finalizing...\n");

                if(Platform::FinalizeSocketLib() == -1)
                {
                    LogError("Socket0 Finalize Failed!\n Error Num: %i\n", Platform::GetLastError());
                    return -1;
                }
                --s_socketLibraryInitialized;
            }
            else
                --s_socketLibraryInitialized;
            return 0;
        }

        int NetworkStressTask::CreateTcpStreamSocket()
        {
            int socket = Platform::socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
            if(socket == -1)
            {
                LogError("TCP Stream Socket Creation Failed!\n Error Num: %i\n", Platform::GetLastError());
                return -1;
            }
            return socket;
        }

        int NetworkStressTask::CreateUdpDgramSocket()
        {
            int socket = Platform::socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
            if(socket == -1)
            {
                LogError("UDP DGram Socket Creation Failed!\n Error Num: %i\n", Platform::GetLastError());
                return -1;
            }
            return socket;
        }

        int NetworkStressTask::Connect(nn::socket::SockAddrIn& address,int socketDesciptor)
        {
            bool t = true;
            do
            {
                int connectionResult = Platform::connect(socketDesciptor,&address,sizeof(address));
                if(connectionResult == -1)
                {
                    nn::socket::Errno error = Platform::GetLastError();
                    //retry connection if the hardware was busy
                    if(error == nn::socket::Errno::EBusy)
                    {
                        continue;
                    }
                    LogError("Failed To Connect To Echo Server!\n  Error Num: %i\n", error);
                    return -1;
                }
                break;
            } while(t);
            m_connected = true;
            return 0;
        }

        int NetworkStressTask::ShutdownConnectedSocket(int socketDescriptor)
        {
            if(!m_connected)
                return 0;

//            char recvBuffer[ABUSE_MAX_BUFFER_SIZE];
            LogInfo("Shutting Down Command Socket ...\n");
            int shutdownResult;
            bool t = true;
            do
            {
                shutdownResult = Platform::shutdown(socketDescriptor, nn::socket::ShutdownMethod::Shut_Wr);
                if(shutdownResult == -1)
                {
                    nn::socket::Errno error = Platform::GetLastError();
                    //retry recv if the hardware was busy
                    if(error == nn::socket::Errno::EBusy)
                    {
                        continue;
                    }
                    LogError("Socket Write Shutdown Failed!\n Error Num: %i\n", error);
                    return -1;
                }
                break;
            } while(t);

            /*
            LogInfo("Waiting For Server Shutdown Acknowledgement...\n");
            int receiveResult;
            nn::TimeSpan startTime = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick());
            nn::TimeSpan currentTime;
            do
            {
                currentTime = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick());
                receiveResult = Platform::recv(socketDescriptor,recvBuffer,sizeof(recvBuffer),0);
                if(receiveResult == -1)
                {
                    nn::socket::Errno error = Platform::GetLastError();
                    //retry recv if the hardware was busy
                    if(error == nn::socket::Errno::EBusy)
                    {
                        receiveResult = 1;
                        continue;
                    }
                    LogError("Socket Shutdown Receive Failed!\n Error Num: %i\n",error );
                    return -1;
                }

            } while(receiveResult > 0 || (currentTime - startTime).GetSeconds() > 5);



            LogInfo("Shutting Down Socket Read...\n");
            shutdownResult = Platform::shutdown(socketDescriptor, SO_SHUT_RD);
            if(receiveResult == -1)
            {
                LogError("Socket Read Shutdown Failed!\n Error Num: %i\n", Platform::GetLastError());
                return -1;
            }
            LogInfo("Socket Shutdown Success!\n");*/
            return 0;
        }

        int NetworkStressTask::ReceiveErrorCheck(int socketDescriptor,NetworkStressTask * task)
        {
            nn::socket::PollFd pollStruct = {};
            memset(&pollStruct, 0, sizeof(pollStruct));
            pollStruct.events = nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollErr;
            pollStruct.fd = socketDescriptor;
            bool busy = false;
            int pollResult;
            do
            {
                busy = false;
                pollResult = Platform::poll(&pollStruct, 1, 5000);
                if(pollResult <= -1)
                {
                    nn::socket::Errno error = Platform::GetLastError();
                    //retry poll if the hardware was busy
                    if(error == nn::socket::Errno::EBusy)
                    {
                        busy = true;
                        continue;
                    }
                    task->LogError("Socket Poll Failed!\n Error Num: %i\n", error);
                    return -1;
                }
            } while(busy);

            if(pollResult == 0)
            {
                task->LogError("Socket Poll Timed Out!\n");
                return -1;
            }
            else if(!(pollStruct.revents & nn::socket::PollEvent::PollRdNorm))
            {
                task->LogError("Socket Not Readable After Poll!\n");
                task->LogError("revents: %x\n", pollStruct.revents);
                return -1;
            }
            else if(!!(pollStruct.revents & nn::socket::PollEvent::PollErr))
            {
                task->LogError("Socket Error After Poll!\n");
                task->LogError("revents: %x\n", pollStruct.revents);
                return -1;
            }
            return 0;
        }


        int NetworkStressTask::HandleInfo(UdpHeader & header,const void * buffer, NetworkThreadArg * arg)
        {
            switch (header.info)
            {

            case Info_None:
                return 0;
                break;

            case Info_Error:
                {
                    char errorLog[1024];
                    if(header.dataSize <= 1024)
                    {
                        memcpy(errorLog,buffer,header.dataSize);
                        errorLog[header.dataSize - 1] = '\0';
                        arg->parent->LogError("Server Side Error! : %s\n",errorLog);
                    }
                    else
                    {
                        arg->parent->LogError("Server Side Error! Error Received But Error Log Was Too Large!\n");
                    }
                    return 1;
                }
                break;

            case Info_ThreadIndex:
              // arg->serverThreadId = *(const unsigned *)buffer;
                return 1;
                break;

            case Info_SocketShutdown:
                break;

            default:
                return -1;
                break;
            }
           return 0;
        }

        void NetworkStressTask::FillHeader(Header& header,TestType type,Direction direction, unsigned dataSize, double idleTime)
        {
            header.type = type;
            header.direction = direction;
            header.dataSize = dataSize;
            header.idleTime = idleTime;
        }

        void NetworkStressTask::ExtractHeader(char * buffer, int bufferLen, Header & header, char ** dataStart)
        {
            if (bufferLen < sizeof(Header))
                return;

            Header const * receivedHeader = reinterpret_cast<Header const *>(buffer);

            header.type = (TestType)Platform::ntohl(receivedHeader->type);
            header.direction = (Direction)Platform::ntohl(receivedHeader->direction);
            header.dataSize = Platform::ntohl(receivedHeader->dataSize);
            uint64_t convertedIdleTime = Platform::ntohll(*reinterpret_cast<uint64_t const *>(&(receivedHeader->idleTime)));
            header.idleTime = *reinterpret_cast<double *>(&convertedIdleTime);

            if (dataStart)
                *dataStart = buffer + sizeof(Header);


        }
        void NetworkStressTask::FillUdpHeader(UdpHeader& header,TestType type,Direction direction,unsigned packetID, unsigned dataSize,Info info,double idleTime )
        {
            header.type = type;
            header.direction = direction;
            header.packetID = packetID;
            header.dataSize = dataSize;
            header.info = info;
            header.idleTime = idleTime;
        }
        void NetworkStressTask::ExtractUdpHeader(char * buffer, int bufferLen, UdpHeader & header, char ** dataStart)
        {
            if (bufferLen < sizeof(UdpHeader))
                return;

            UdpHeader const * receivedHeader = reinterpret_cast<UdpHeader const *>(buffer);

            header.type = (TestType)Platform::ntohl(receivedHeader->type);
            header.direction = (Direction)Platform::ntohl(receivedHeader->direction);
            header.dataSize = Platform::ntohl(receivedHeader->dataSize);
            uint64_t convertedIdleTime = Platform::ntohll(*reinterpret_cast<uint64_t const *>(&(receivedHeader->idleTime)));
            header.idleTime = *reinterpret_cast<double *>(&convertedIdleTime);

            if (dataStart)
                *dataStart = buffer + sizeof(UdpHeader);
        }

        int NetworkStressTask::CheckDataValidity(void *buff1, unsigned size1, void * buff2, unsigned size2,NetworkStressTask * task)
        {
            int compareResult = memcmp(buff1,buff2,size1);
            if(compareResult < 0 || compareResult > 0)
            {
                task->LogError("Recieved Data Does Not Match Sent Data!\n  Sent Data: ");
                for(int i = 0;i < (int)size1; ++i)
                {
                    task->LogError("%hhx",reinterpret_cast<char*>(buff1)[i]);
                }
                task->LogError("\nReceived Data: ");
                for(int i = 0;i < (int)size2; ++i)
                {
                    task->LogError("%hhx",reinterpret_cast<char*>(buff2)[i]);
                }
                task->LogError("\n");
                return -1;
            }
            return 0;
        }

        int NetworkStressTask::SendDataTcp(int socketDescriptor, const Header& header, const char * buffer,NetworkStressTask * task)
        {
            char sendBuffer[ABUSE_MAX_BUFFER_SIZE + sizeof(header)];
            Header* sendHeader = reinterpret_cast<Header*>(sendBuffer);

            sendHeader->Fill(header);
            sendHeader->ToNetwork();

           /* sendHeader->type = (TestType)Platform::htonl(header.type);
            sendHeader->direction = (Direction)Platform::htonl(header.direction);
            sendHeader->dataSize = (int)Platform::htonl(header.dataSize);
            uint64_t convertedIdleTime = Platform::htonll(*reinterpret_cast<uint64_t const *>(&header.idleTime));
            sendHeader->idleTime = *reinterpret_cast<double *>(&convertedIdleTime);*/

            memcpy(sendBuffer + sizeof(header),buffer,header.dataSize);
            int sendTotal = 0;
            do
            {
                int sendResult = Platform::send(socketDescriptor, sendBuffer + sendTotal, (header.dataSize + sizeof(header)) - sendTotal, nn::socket::MsgFlag::Msg_None);
               if(sendResult == -1)
               {
                   nn::socket::Errno error = Platform::GetLastError();
                   if(error == nn::socket::Errno::EBusy)
                   {
                       continue;
                   }
                   task->LogError("Failed To Send Data!\n  Error Num: %i\n", error);
                    return -1;
               }
               sendTotal += sendResult;

            }while(sendTotal < (int)(header.dataSize + sizeof(header)));
            return 0;
        }

        int NetworkStressTask::ReceiveDataTcp(int socketDescriptor, Header& header, char * buffer,int size,NetworkStressTask * task)
        {
            char receiveBuffer[sizeof(header)];
            int receiveTotal = 0;
            //Receive Header
            do
            {
               if(ReceiveErrorCheck(socketDescriptor,task) == -1)
               {
                   task->LogError("Failed To Pass Polling Check!\n");
                   return -1;
               }
               int receiveResult = Platform::recv(socketDescriptor, receiveBuffer + receiveTotal, sizeof(header) - receiveTotal, nn::socket::MsgFlag::Msg_None);
               if(receiveResult == 0)
               {
                   task->LogInfo("Connection Closed By Server.\n");
                   return 0;
               }
               else if(receiveResult == -1)
               {
                   nn::socket::Errno error = Platform::GetLastError();
                   if(error == nn::socket::Errno::EBusy)
                   {
                  //     LogWarning("Hardware Busy Error... Retrying\n  Error Num: %i\n", error);
                       continue;
                   }
                   task->LogError("Failed To Receive Data!\n  Error Num: %i\n", error);
                    return -1;
               }
               receiveTotal += receiveResult;
            }while(receiveTotal < sizeof(header));

            Header* receivedHeader = reinterpret_cast<Header*>(receiveBuffer);
            header.Fill(*receivedHeader);
            header.ToLocal();

           /* header.type = (TestType)Platform::ntohl(receivedHeader->type);
            header.direction = (Direction)Platform::ntohl(receivedHeader->direction);
            header.dataSize = Platform::ntohl(receivedHeader->dataSize);
            uint64_t convertedIdleTime = Platform::ntohll(*reinterpret_cast<uint64_t const *>(&(receivedHeader->idleTime)));
            header.idleTime = *reinterpret_cast<double *>(&convertedIdleTime);*/

            if((int)(header.dataSize) > size)
            {
                task->LogError("Error! Received Packet With Data Larger Than Max Buffer Size!\n");
                return -1;
            }

            receiveTotal = 0;
            do
            {
               if(ReceiveErrorCheck(socketDescriptor,task) == -1)
               {
                   task->LogError("Failed To Pass Polling Check!\n");
                   return -1;
               }
               int receiveResult = Platform::recv(socketDescriptor, buffer + receiveTotal, header.dataSize - receiveTotal, nn::socket::MsgFlag::Msg_None);
               if(receiveResult == 0)
               {
                   task->LogInfo("Connection Closed By Server.\n");
                   return 0;
               }
               else if(receiveResult == -1)
               {
                   nn::socket::Errno error = Platform::GetLastError();
                   if(error == nn::socket::Errno::EBusy)
                   {
                     //  LogWarning("Hardware Busy Error... Retrying\n  Error Num: %i\n", error);
                       continue;
                   }
                   task->LogError("Failed To Receive Data!\n  Error Num: %i\n", error);
                    return -1;
               }
               receiveTotal += receiveResult;

            }while(receiveTotal < (int)(header.dataSize));


            return receiveTotal;
        }

        int NetworkStressTask::SendCommand(CommandContainer & command)
        {
            char sendBuffer[ABUSE_MAX_BUFFER_SIZE + sizeof(command.command)];
            CommandHeader* commandHeader = reinterpret_cast<CommandHeader*>(sendBuffer);

            commandHeader->Fill(command.command);
            int dataSize = commandHeader->dataSize;
            commandHeader->ToNetwork();

            memcpy(sendBuffer + sizeof(CommandHeader),command.data,dataSize);
            int sendTotal = 0;
            do
            {
                int sendResult = Platform::send(m_commandSocketDescriptor, sendBuffer + sendTotal, (dataSize + sizeof(CommandHeader)) - sendTotal, nn::socket::MsgFlag::Msg_None);
               if(sendResult == -1)
               {
                   nn::socket::Errno error = Platform::GetLastError();
                   if(error == nn::socket::Errno::EBusy)
                   {
                       continue;
                   }
                   LogError("Failed To Send Data!\n  Error Num: %i\n", error);
                    return -1;
               }
               sendTotal += sendResult;

            }while(sendTotal < (int)(dataSize + sizeof(CommandHeader)));
            return 0;
        }

        int NetworkStressTask::ReceiveCommand(CommandContainer & command)
        {
            char receiveBuffer[sizeof(CommandHeader)];
            int receiveTotal = 0;
            //Receive Header
            do
            {
               int receiveResult = Platform::recv(m_commandSocketDescriptor, receiveBuffer + receiveTotal, sizeof(CommandHeader) - receiveTotal, nn::socket::MsgFlag::Msg_None);
               if(receiveResult == 0)
               {
                   LogInfo("Connection Closed By Server.\n");
                   return 0;
               }
               else if(receiveResult == -1)
               {
                   nn::socket::Errno error = Platform::GetLastError();
                   if(error == nn::socket::Errno::EBusy)
                   {
                  //     LogWarning("Hardware Busy Error... Retrying\n  Error Num: %i\n", error);
                       continue;
                   }
                   if(error == nn::socket::Errno::EWouldBlock)
                   {
                       return -2;
                   }
                   LogError("Failed To Receive Data!\n  Error Num: %i\n", error);
                   return -1;
               }
               receiveTotal += receiveResult;
            }while(receiveTotal < sizeof(CommandHeader));

            CommandHeader* receivedHeader = reinterpret_cast<CommandHeader*>(receiveBuffer);
            command.command.Fill(*receivedHeader);
            command.command.ToLocal();

            if((int)(command.command.dataSize) > command.dataSize)
            {
                LogError("Error! Received Packet With Data Larger Than Max Buffer Size!\n");
                return -1;
            }

            receiveTotal = 0;
            while(receiveTotal < (int)(command.command.dataSize))
            {

               int receiveResult = Platform::recv(m_commandSocketDescriptor, (char  *)command.data + receiveTotal, command.command.dataSize - receiveTotal, nn::socket::MsgFlag::Msg_None);
               if(receiveResult == 0)
               {
                   LogInfo("Connection Closed By Server.\n");
                   return 0;
               }
               else if(receiveResult == -1)
               {
                   nn::socket::Errno error = Platform::GetLastError();
                   if(error == nn::socket::Errno::EBusy)
                   {
                     //  LogWarning("Hardware Busy Error... Retrying\n  Error Num: %i\n", error);
                       continue;
                   }
                   if(error == nn::socket::Errno::EWouldBlock)
                   {
                       LogError("nn::socket::Errno::EWouldBlock on Data Receive After Header Received When Expecting Data! This should not happen!\n");
                       return -1;
                   }
                   LogError("Failed To Receive Data!\n  Error Num: %i\n", error);
                    return -1;
               }
               receiveTotal += receiveResult;
            }
            return receiveTotal;
        }

        int NetworkStressTask::HandleReceivedCommand(CommandContainer & command)
        {
            switch (command.command.command)
            {
                case Command_None:
                    break;
                case Command_KillThread:
                    break;
                case Command_LogError:
                    ((char *)command.data)[ABUSE_MAX_BUFFER_SIZE - 1] = '\0';
                    LogError("[Server][Thread %d] %s",command.command.networkStressThreadId,command.data);
                    return -1;
                    break;
                case Command_LogWarning:
                    ((char *)command.data)[ABUSE_MAX_BUFFER_SIZE - 1] = '\0';
                    LogWarning("[Server][Thread %d] %s",command.command.networkStressThreadId,command.data);
                    return 0;
                    break;
                case Command_LogInfo:
                    ((char *)command.data)[ABUSE_MAX_BUFFER_SIZE - 1] = '\0';
                    LogInfo("[Server][Thread %d] %s",command.command.networkStressThreadId,command.data);
                    return 0;
                    break;
                case Command_LogVerbose:
                    ((char *)command.data)[ABUSE_MAX_BUFFER_SIZE - 1] = '\0';
                    LogVerbose("[Server][Thread %d] %s",command.command.networkStressThreadId,command.data);
                    return 0;
                    break;
                default:
                    break;
            }
            return 0;
        }

        int NetworkStressTask::SendDataUdp(int socketDescriptor, const UdpHeader& header, const char * buffer,NetworkStressTask * task,nn::socket::SockAddrIn * serverAddr)
        {

            char sendBuffer[ABUSE_MAX_BUFFER_SIZE + sizeof(header)];
            UdpHeader* sendHeader = reinterpret_cast<UdpHeader*>(sendBuffer);

            sendHeader->Fill(header);
            sendHeader->ToNetwork();

            memcpy(sendBuffer + sizeof(header),buffer,header.dataSize);
            int sendResult;
            bool t = true;
            do
            {
                sendResult = Platform::sendto(socketDescriptor, sendBuffer, (header.dataSize + sizeof(header)), nn::socket::MsgFlag::Msg_None, serverAddr,sizeof(nn::socket::SockAddrIn));
                if(sendResult == -1)
                {
                    nn::socket::Errno error = Platform::GetLastError();
                    if(error == nn::socket::Errno::EWouldBlock)
                    {
                        continue;
                    }
                    else if(error == nn::socket::Errno::EBusy)
                    {
                        continue;
                    }

                    task->LogError("Failed To Send Data!\n  Error Num: %i\n", error);
                    return -1;
                }
                break;
            } while(t);

            //Platform::sendto(socketDescriptor,buffer,
            return sendResult;
        }

        int NetworkStressTask::ReceiveDataUdp(int socketDescriptor, UdpHeader& header, char * buffer, int size,NetworkStressTask * task,bool isBlocking)
        {

            char recvBuff[ABUSE_MAX_BUFFER_SIZE + sizeof(header)];
            int iResult;
            if(ABUSE_MAX_BUFFER_SIZE < size)
            {
                task->LogError("Unexpectedly Large Buffer!\nKeep Buffer Under %d Bytes.\n", ABUSE_MAX_BUFFER_SIZE);
                return -1;
            }
            bool t = true;
            do
            {

                if(isBlocking && ReceiveErrorCheck(socketDescriptor,task) == -1)
                {
                    return -1;
                }
                iResult = Platform::recvfrom(socketDescriptor, recvBuff, size + sizeof(Header), nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddrIn*>(nullptr), 0);
                if (iResult == -1)
                {
                    nn::socket::Errno error = Platform::GetLastError();
                    if(error == nn::socket::Errno::EWouldBlock)
                    {
                        return -1;
                    }
                    else if(error == nn::socket::Errno::EBusy)
                    {
                        continue;
                    }
                    task->LogError("Failed to receive UDP data!\nError Num: %d\n", Platform::GetLastError());

                    return -1;
                }
                break;
            } while(t);

            if (iResult < sizeof(header))
            {
                task->LogError("UDP Receive Failed!\nDataGram Smaller Than Required Header Size!\n");
                task->LogError("Bytes Recieved: %i Minimum Bytes Expected: %i \n",iResult,sizeof(header));
                return -1;
            }

            char * dataBuffer;
            header.Extract(recvBuff, iResult, &dataBuffer);
            header.ToLocal();

            if (iResult != (int)(sizeof(header) + header.dataSize))
            {
                task->LogError("UDP Receive Failed!\nDataGram Differs From Expected Size!\n Size Received: %i Size Expected: %i\n", iResult, sizeof(header) + header.dataSize);
                return -1;
            }

            memcpy(buffer,dataBuffer,header.dataSize);

            return iResult - sizeof(header);
        }


        //Sets up which test to run bassed on parameters proccessed by ProccessParams
        void NetworkStressTask::SetCurrentTest()
        {
            switch (m_protocol)
            {
            case nn::socket::Protocol::IpProto_Tcp:
                switch (m_direction)
                {
                case nnt::abuse::NetworkStressTask::Direction_Upload:

                    if(m_fixed)
                        RunningTest = &NetworkStressTask::RunTcpUploadFixedTest;
                    else
                        RunningTest = &NetworkStressTask::RunTcpUploadVaryingTest;

                    break;
                case nnt::abuse::NetworkStressTask::Direction_Download:
                    if(m_fixed)
                        RunningTest = &NetworkStressTask::RunTcpDownloadFixedTest;
                    else
                        RunningTest = &NetworkStressTask::RunTcpDownloadVaryingTest;


                    break;
                case nnt::abuse::NetworkStressTask::Direction_Both:
                    if(m_fixed)
                        RunningTest = &NetworkStressTask::RunTcpUploadDownloadFixedTest;
                    else
                        RunningTest = &NetworkStressTask::RunTcpUploadDownloadVaryingTest;
                    break;
                default:
                    LogError("Invalid Direction Value! Value Given: %i", m_direction);
                    break;
                }
                break;

            case nn::socket::Protocol::IpProto_Udp:
                switch (m_direction)
                {
                case nnt::abuse::NetworkStressTask::Direction_Upload:
                    if(m_fixed)
                        RunningTest = &NetworkStressTask::RunUdpUploadFixedTest;
                    else
                        RunningTest = &NetworkStressTask::RunUdpUploadVaryingTest;

                    break;

                case nnt::abuse::NetworkStressTask::Direction_Download:
                    if(m_fixed)
                        RunningTest = &NetworkStressTask::RunUdpDownloadFixedTest;
                    else
                        RunningTest = &NetworkStressTask::RunUdpDownloadVaryingTest;
                    break;

                case nnt::abuse::NetworkStressTask::Direction_Both:
                    if(m_fixed)
                        RunningTest = &NetworkStressTask::RunUdpUploadDownloadFixedTest;
                    else
                        RunningTest = &NetworkStressTask::RunUdpUploadDownloadVaryingTest;
                    break;

                default:
                    LogError("Invalid Direction Value! Value Given: %i", m_direction);
                    break;
                }
                break;

            default:
                LogError("Invalid Protocol Value! Value Given: %i", m_protocol);
                break;
            }
        }


        NetworkStressTask::Header::Header() :
            type(TestType_Fixed),direction(Direction_Both),idleTime(0.016),networkStressId(0),networkStressThreadId(0),dataSize(0) {}
        NetworkStressTask::Header::Header(TestType type,Direction direction,int networkStressId,int networkStressThreadId, unsigned dataSize, double idleTime) :
            type(type),direction(direction),idleTime(idleTime),networkStressId(networkStressId),networkStressThreadId(networkStressThreadId),dataSize(dataSize) {}
        void  NetworkStressTask::Header::Fill(TestType intype,Direction indirection,int innetworkStressId,int innetworkStressThreadId, unsigned indataSize, double inidleTime)
        {
            this->type = intype;
            this->direction = indirection;
            this->networkStressId = innetworkStressId;
            this->networkStressThreadId = innetworkStressThreadId;
            this->dataSize = indataSize;
            this->idleTime = inidleTime;
        }
         void  NetworkStressTask::Header::Fill(const Header& other)
        {
            this->type = other.type;
            this->direction = other.direction;
            this->networkStressId = other.networkStressId;
            this->networkStressThreadId = other.networkStressThreadId;
            this->dataSize = other.dataSize;
            this->idleTime = other.idleTime;
        }
        void NetworkStressTask::Header::Extract(char * buffer, int bufferLen, char ** dataStart)
        {
            if (bufferLen < sizeof(Header))
                return;

            Header const * receivedHeader = reinterpret_cast<Header const *>(buffer);

            type = receivedHeader->type;
            direction = receivedHeader->direction;
            networkStressId = receivedHeader->networkStressId;
            networkStressThreadId = receivedHeader->networkStressThreadId;
            dataSize = receivedHeader->dataSize;
            idleTime =  receivedHeader->idleTime;

            if (dataStart)
                *dataStart = buffer + sizeof(Header);
        }
        void NetworkStressTask::Header::ToNetwork()
        {
            type = (TestType)Platform::htonl(type);
            direction = (Direction)Platform::htonl(direction);
            dataSize = Platform::htonl(dataSize);
            networkStressId = Platform::htonl(networkStressId);
            networkStressThreadId = Platform::htonl(networkStressThreadId);
            uint64_t convertedIdleTime = Platform::htonll(*reinterpret_cast<uint64_t const *>(&(idleTime)));
            idleTime = *reinterpret_cast<double *>(&convertedIdleTime);
        }
        void NetworkStressTask::Header::ToLocal()
        {
            type = (TestType)Platform::ntohl(type);
            direction = (Direction)Platform::ntohl(direction);
            dataSize = Platform::ntohl(dataSize);
            networkStressId = Platform::ntohl(networkStressId);
            networkStressThreadId = Platform::ntohl(networkStressThreadId);
            uint64_t convertedIdleTime = Platform::ntohll(*reinterpret_cast<uint64_t const *>(&(idleTime)));
            idleTime = *reinterpret_cast<double *>(&convertedIdleTime);
        }


        NetworkStressTask::UdpHeader::UdpHeader() :
            type(TestType_Fixed),direction(Direction_Both),info(Info_None),idleTime(0.016),networkStressId(0),networkStressThreadId(0),packetID(0),dataSize(0) {}
        NetworkStressTask::UdpHeader::UdpHeader(TestType type,Direction direction,int networkStressId,int networkStressThreadId,unsigned packetID, unsigned dataSize,Info error,double idleTime) :
          type(type),direction(direction),info(error),idleTime(idleTime),networkStressId(networkStressId),networkStressThreadId(networkStressThreadId),packetID(packetID),dataSize(dataSize) {}
        void NetworkStressTask::UdpHeader::Fill(TestType intype,Direction indirection,int innetworkStressId,int innetworkStressThreadId,unsigned inpacketID, unsigned indataSize,Info inerror,double inidleTime)
        {
            this->type = intype;
            this->direction = indirection;
            this->packetID = inpacketID;
            this->networkStressId = innetworkStressId;
            this->networkStressThreadId = innetworkStressThreadId;
            this->dataSize = indataSize;
            this->info = inerror;
            this->idleTime = inidleTime;
        }
        void NetworkStressTask::UdpHeader::Fill(const UdpHeader & other)
        {
            this->type = other.type;
            this->direction = other.direction;
            this->networkStressId = other.networkStressId;
            this->networkStressThreadId = other.networkStressThreadId;
            this->packetID = other.packetID;
            this->dataSize = other.dataSize;
            this->info = other.info;
            this->idleTime = other.idleTime;
        }
        void NetworkStressTask::UdpHeader::Extract(char * buffer, int bufferLen, char ** dataStart)
        {
            if (bufferLen < sizeof(UdpHeader))
                return;

            UdpHeader const * receivedHeader = reinterpret_cast<UdpHeader const *>(buffer);

            type = receivedHeader->type;
            direction = receivedHeader->direction;
            networkStressId = receivedHeader->networkStressId;
            networkStressThreadId = receivedHeader->networkStressThreadId;
            packetID = receivedHeader->packetID;
            dataSize = receivedHeader->dataSize;
            info = receivedHeader->info;
            idleTime =  receivedHeader->idleTime;

            if (dataStart)
                *dataStart = buffer + sizeof(UdpHeader);
        }
        void NetworkStressTask::UdpHeader::ToNetwork()
        {
            type = (TestType)Platform::htonl(type);
            direction = (Direction)Platform::htonl(direction);
            networkStressId = (int)Platform::htonl(networkStressId);
            networkStressThreadId = (int)Platform::htonl(networkStressThreadId);
            packetID = Platform::htonl(packetID);
            dataSize = Platform::htonl(dataSize);
            info = (Info)Platform::htonl(info);
            uint64_t convertedIdleTime = Platform::htonll(*reinterpret_cast<uint64_t const *>(&(idleTime)));
            idleTime = *reinterpret_cast<double *>(&convertedIdleTime);
        }
        void NetworkStressTask::UdpHeader::ToLocal()
        {
            type = (TestType)Platform::ntohl(type);
            direction = (Direction)Platform::ntohl(direction);
            networkStressId = (int)Platform::ntohl(networkStressId);
            networkStressThreadId = (int)Platform::ntohl(networkStressThreadId);
            packetID = Platform::ntohl(packetID);
            dataSize = Platform::ntohl(dataSize);
            info = (Info)Platform::ntohl(info);
            uint64_t convertedIdleTime = Platform::ntohll(*reinterpret_cast<uint64_t const *>(&(idleTime)));
            idleTime = *reinterpret_cast<double *>(&convertedIdleTime);
        }


        NetworkStressTask::CommandHeader::CommandHeader() :
          command(Command_None),dataSize(0) {}
        NetworkStressTask::CommandHeader::CommandHeader(Command command,int networkStressId,int networkStressThreadId, unsigned dataSize) :
          command(command),networkStressId(networkStressId),networkStressThreadId(networkStressThreadId),dataSize(dataSize) {}
        void NetworkStressTask::CommandHeader::Fill(Command incommand, int innetworkStressId, int innetworkStressThreadId, unsigned indataSize)
        {
            this->command = incommand;
            this->networkStressId = innetworkStressId;
            this->networkStressThreadId = innetworkStressThreadId;
            this->dataSize = indataSize;
        }
        void NetworkStressTask::CommandHeader::Fill(const CommandHeader & other)
        {
            this->command = other.command;
            this->networkStressId = other.networkStressId;
            this->networkStressThreadId = other.networkStressThreadId;
            this->dataSize = other.dataSize;
        }
        void NetworkStressTask::CommandHeader::Extract(char * buffer, int bufferLen, char ** dataStart)
        {
            if (bufferLen < sizeof(CommandHeader))
                return;

            CommandHeader const * receivedHeader = reinterpret_cast<CommandHeader const *>(buffer);

            command = receivedHeader->command;
            networkStressId = receivedHeader->networkStressId;
            networkStressThreadId = receivedHeader->networkStressThreadId;
            dataSize = receivedHeader->dataSize;

            if (dataStart)
                *dataStart = buffer + sizeof(CommandHeader);
        }
        void NetworkStressTask::CommandHeader::ToNetwork()
        {
            command = (Command)Platform::htonl(command);
            networkStressId = (int)Platform::htonl(networkStressId);
            networkStressThreadId = (int)Platform::htonl(networkStressThreadId);
            dataSize = Platform::htonl(dataSize);
        }
        void NetworkStressTask::CommandHeader::ToLocal()
        {
            command = (Command)Platform::ntohl(command);
            networkStressId = (int)Platform::ntohl(networkStressId);
            networkStressThreadId = (int)Platform::ntohl(networkStressThreadId);
            dataSize = Platform::ntohl(dataSize);
        }

        NetworkStressTask::CommandContainer::CommandContainer()
        : command(), dataSize(0), data(nullptr)
        {
        }
        NetworkStressTask::CommandContainer::CommandContainer(CommandHeader command,void * data,int dataSize)
            : command(command),dataSize(dataSize),data(nullptr)
        {
            this->data = Platform::Allocate(dataSize);
            if(data)
                memcpy(this->data,data,dataSize);
            else
                memset(this->data,0,dataSize);
        }
        NetworkStressTask::CommandContainer::CommandContainer(const CommandContainer & other)
            : command(other.command), dataSize(other.dataSize),data(nullptr)
        {
            if(other.data)
            {
                data = Platform::Allocate(dataSize);
                memcpy(data,other.data,dataSize);
            }
        }
        NetworkStressTask::CommandContainer::~CommandContainer()
        {
            Platform::Free(data);
        }

        NetworkStressTask::CommandContainer & NetworkStressTask::CommandContainer::operator=(const CommandContainer & rhs)
        {
            command = rhs.command;
            if(rhs.dataSize == 0 || !rhs.data)
            {
                if(data)
                {
                    Platform::Free(data);
                    data = nullptr;
                }
                dataSize = 0;
            }
            else
            {
                if(!data || dataSize != rhs.dataSize)
                {
                   if(data)
                   {
                       Platform::Free(data);
                   }
                   dataSize = rhs.dataSize;
                   data = Platform::Allocate(dataSize);
                }
                memcpy(data,rhs.data,dataSize);
            }
            return *this;
        }

        NetworkStressTask::PacketMonitor::PacketMonitor(NetworkStressTask * parent) :
            m_lastPacketCountIndex(0),
            m_consecutiveCounts(0),
            m_parent(parent)

        {
            m_packetCountArray[0] = 0;
            m_packetCountArray[1] = 0;
            m_packetCountStatusArray[0] = CountStatus_Empty;
            m_packetCountStatusArray[1] = CountStatus_Empty;
            m_packetRangeStart[0] = 0;
            m_packetRangeStart[1] = countMaxSize;
        }
        void NetworkStressTask::PacketMonitor::VerifyPacket(unsigned packetId,unsigned threadID)
        {
            if(m_parent->m_flood)
            {
                ++m_consecutiveCounts;
                if(m_consecutiveCounts % 1000 == 0)
                {
                     m_parent->LogInfo("Thread ID:%d Recieved %d packets...\n",threadID,m_consecutiveCounts);
                }
                return;
            }
            unsigned packetCountArrayIndex = (packetId / countMaxSize) % 2;
            //Check if we've come back to the previous count without filling it first (only possible with packet loss)
           /* if(m_packetCountStatusArray[packetCountArrayIndex] != CountStatus_Empty && packetId >= m_packetRangeStart[packetCountArrayIndex] + 2*countMaxSize)
            {
                m_parent->LogWarning("Packet Loss Detected At Count Swap\n%d expected packets, %d received packets\n"
                    ,countMaxSize,m_packetCountArray[packetCountArrayIndex]);
                m_packetCountArray[packetCountArrayIndex] = 0;
                m_packetRangeStart[packetCountArrayIndex] += 2*countMaxSize;
                m_packetCountStatusArray[packetCountArrayIndex] = CountStatus_Empty;
            }*/

            ++m_packetCountArray[packetCountArrayIndex];

            if(m_packetCountStatusArray[packetCountArrayIndex] == CountStatus_Empty)
            {
                m_packetCountStatusArray[packetCountArrayIndex] = CountStatus_Accumulating;
            }
            else if(m_packetCountArray[packetCountArrayIndex] == countMaxSize)
            {
                m_parent->LogInfo("Thread ID:%d Incrementing Range %d-%d to %d-%d\n",
                    threadID,
                    m_packetRangeStart[packetCountArrayIndex],
                    m_packetRangeStart[packetCountArrayIndex] + countMaxSize - 1,
                    m_packetRangeStart[packetCountArrayIndex] + 2 * countMaxSize,
                    m_packetRangeStart[packetCountArrayIndex] + 3 * countMaxSize - 1);

                m_packetCountArray[packetCountArrayIndex] = 0;
                m_packetRangeStart[packetCountArrayIndex] += 2 * countMaxSize;
                m_packetCountStatusArray[packetCountArrayIndex] = CountStatus_Empty;
            }

            if(m_packetCountStatusArray[0] == CountStatus_Empty || m_packetCountStatusArray[1] == CountStatus_Empty)
                return;

            if(packetCountArrayIndex == m_lastPacketCountIndex)
            {
                ++m_consecutiveCounts;
            }
            else
            {
                m_consecutiveCounts = 0;
                m_lastPacketCountIndex = (m_lastPacketCountIndex + 1) % 2;
            }

            if(m_consecutiveCounts >= 10)
            {
                m_parent->LogWarning("Thread ID:%u Packet Loss Detected.\n%d expected packets, %d received packets, id range %d-%d\n",
                    threadID, countMaxSize, m_packetCountArray[(packetCountArrayIndex + 1) % 2],
                    m_packetRangeStart[(packetCountArrayIndex + 1) % 2], m_packetRangeStart[(packetCountArrayIndex + 1) % 2] + countMaxSize - 1);
                m_packetCountArray[(packetCountArrayIndex + 1) % 2] = 0;
                m_packetRangeStart[(packetCountArrayIndex + 1) % 2] += 2 * countMaxSize;
                m_packetCountStatusArray[(packetCountArrayIndex + 1) % 2] = CountStatus_Empty;
                m_consecutiveCounts = 0;
            }
            return;
        }

    }
}
