﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Log.h>
#include <FileUtility.h>
#include <Tasks/CpuThrashTask.h>

namespace nnt { namespace abuse {
    CpuThrashTask::CpuThrashTask(const String& typeName, const String& instanceName) :
        BaseTask(typeName, instanceName), m_threadInfo(nullptr),m_allocatedThreads(0), m_stackSize(4096), m_numThreads(4),
        m_priority(nn::os::DefaultThreadPriority), m_minRuntime(1), m_maxRuntime(10), m_useAllCores(false), m_sleep(false) {}

    InitStatus CpuThrashTask::Initialize(const String& params)
    {
        LogInfo("Running on core %d\n", nn::os::GetCurrentCoreNumber());
        char* buffer = (char*)Platform::Allocate( params.length() + 1);
        strncpy(buffer, params.c_str(), params.length());
        buffer[params.length()]=  '\0';

        int trimmedLength = FileUtility::TrimBuffer(buffer, params.length() + 1, true);

        ArgVector args;
        FileUtility::ParseArgs(buffer + 1, trimmedLength - 2, args);

        for(ScriptArg arg : args)
        {
            if(arg.argName == "MaxThreads")
                FileUtility::TryParseInt(arg, 0, std::numeric_limits<int>::max(), &m_numThreads);
            else if(arg.argName == "StackSize")
                FileUtility::TryParseInt(arg, 0, std::numeric_limits<int>::max(), &m_stackSize);
            else if(arg.argName=="UseAllCores")
                FileUtility::TryParseBool(arg, &m_useAllCores);
            else if(arg.argName=="Priority")
                FileUtility::TryParseInt(arg, nn::os::HighestThreadPriority, nn::os::LowestThreadPriority, &m_priority);
            else if(arg.argName=="MinRunTime")
                FileUtility::TryParseInt(arg, 0, std::numeric_limits<int>::max(), &m_minRuntime);
            else if(arg.argName=="MaxRunTime")
                FileUtility::TryParseInt(arg, 0, std::numeric_limits<int>::max(), &m_maxRuntime);
            else if(arg.argName=="Sleep")
                FileUtility::TryParseBool(arg, &m_sleep);
        }

        Platform::Free(buffer);

        if(m_minRuntime > m_maxRuntime)
        {
            LogWarning("Swapping min runtime and max runtime becase max runtime needs to be larger\n");
            std::swap(m_minRuntime, m_maxRuntime);
        }

        m_threadInfo = (ThrashThreadInfo*) Platform::Allocate(sizeof(ThrashThreadInfo) * m_numThreads);
        if(!m_threadInfo)
        {
            LogError("Init: Could not allocate ThreadInfo\n");
            return INIT_ERROR;
        }

        memset(m_threadInfo, 0, sizeof(ThrashThreadInfo) * m_numThreads);

        m_allocatedThreads = 0;
        for(int i = 0; i < m_numThreads; ++i)
        {
            m_threadInfo[i].stack = Platform::AllocateAligned(m_stackSize, nn::os::StackRegionAlignment);

            m_threadInfo[i].arg.sleep = m_sleep;
            m_threadInfo[i].arg.runtimeMillis = Abuse::RandRange(m_minRuntime, m_maxRuntime);
            m_threadInfo[i].arg.index= i;
            m_threadInfo[i].arg.task = this;
            if(!m_threadInfo[i].stack)
                break;
            ++m_allocatedThreads;
        }


        if(m_allocatedThreads != m_numThreads)
            LogWarning("Requested %d, but could only allocated enough stacks for %d threads.\nContinuing with %d threads\n", m_numThreads, m_allocatedThreads, m_allocatedThreads);

        return INIT_OK;
    }

    StartStatus CpuThrashTask::Start()
    {
        LogInfo("Starting CpuThrash\n");

        return START_OK;
    }

    RunStatus CpuThrashTask::Run()
    {
        int m_runningThreads = 0;
        for(int i = 0; i < m_allocatedThreads; ++i)
        {
            if(m_useAllCores)
            {
                if(!CreateThread(&m_threadInfo[i].thread, &threadFunc,(void*) &m_threadInfo[i].arg, m_threadInfo[i].stack, m_stackSize, m_priority))
                {
                    break;
                }
            }
            else
            {
                if(!CreateThread(&m_threadInfo[i].thread, &threadFunc, (void*) &m_threadInfo[i].arg, m_threadInfo[i].stack, m_stackSize, m_priority, nn::os::GetCurrentCoreNumber()))
                {
                    break;
                }
            }
            ++m_runningThreads;
        }
        if(m_runningThreads != m_allocatedThreads)
            LogWarning("Could only create %d threads\n", m_runningThreads);

        for(int i = 0; i < m_runningThreads; ++i)
        {
            nn::os::StartThread(&m_threadInfo[i].thread);
        }

        for(int i = 0; i < m_runningThreads; ++i)
        {
            nn::os::WaitThread(&m_threadInfo[i].thread);
        }

        for(int i = 0; i < m_runningThreads; ++i)
            DestroyThread(&m_threadInfo[i].thread);
        return RUN_OK;
    }

    StopStatus CpuThrashTask::Stop()
    {
        LogInfo("Stopping CpuThrash\n");

        return STOP_OK;
    }

    ShutdownStatus CpuThrashTask::Shutdown()
    {
        for(int i = 0; i < m_allocatedThreads; ++i)
        {
            if(m_threadInfo[i].stack)
                Platform::Free(m_threadInfo[i].stack);
        }

        if(m_threadInfo)
            Platform::Free(m_threadInfo);

        return SHUTDOWN_OK;
    }

    void CpuThrashTask::threadFunc(void* threadArg)
    {
        ThrashThreadArg* arg = (ThrashThreadArg*)threadArg;

        if(arg->sleep)
        {
            arg->task->LogVerbose("%i start sleep on core %d\n", arg->index, nn::os::GetCurrentCoreNumber());
            for(int i = 0; i < arg->runtimeMillis; ++i)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
            }
            arg->task->LogVerbose("%i stop sleep on core %d\n", arg->index, nn::os::GetCurrentCoreNumber());
        }
        else
        {
            nn::os::Tick endTime = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(arg->runtimeMillis));
            while(endTime > nn::os::GetSystemTick())
            {
                nn::os::YieldThread();
            }
        }
    }

    const char* CpuThrashTask::GetParamOptions()
    {
        return "NumThreads=[1 - 128] - number of threads to run\n\
\tStackSize=[4096+] - multiple of 4096, stack size for created threads\n\
\tUseAllCores=[0 || 1] - 0: create threads on task's core,  1: create threads evenly on all cores\n\
\tPriority=[0 - 31] - priority of created threads\n\
\tMinRunTime=[0+] - min time in ms to run each thread\n\
\tMaxRunTime=[0+] - max time in ms to run each thread\n\
\tSleep=[0 || 1] - 0: is created threads yield until runtime expires,  1: is created threads sleep for entire runtime";
    }
} }
