﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstddef>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/svc/svc_Types.h>

namespace nnt { namespace svc { namespace fwk {

enum TestResult
{
    TestResult_Success,
    // TestResult_Failure, // A test failure aborts the process
    TestResult_NotRun
};

struct TestCase
{
    const char* name;
};

struct TestCaseDependency
{
    int testCaseIndex;
    int requiredTestCaseIndex;
};

struct Test
{
    int testCaseIndex;
    const char* name;
    void (*function)();

    TestResult result = TestResult_NotRun;
};

class TestAdder
{
public:
    TestAdder(const char* testCaseName, const char* testName, void (*testFunction)());
};

class TestCaseDependencyAdder
{
public:
    TestCaseDependencyAdder(const char* testCaseName, const char* requiredTestCaseName);
};

class ScopedSubTest
{
public:
    NN_IMPLICIT ScopedSubTest(const char* subTestName);
    ~ScopedSubTest();
};

template<typename T>
struct TypedPrintImpl;

template<> struct TypedPrintImpl<int>
{
    static void Print(int a)
    {
        Printf("%d", a);
    }
};

template<> struct TypedPrintImpl<long>
{
    static void Print(long a)
    {
        Printf("%ld", a);
    }
};

template<> struct TypedPrintImpl<long long>
{
    static void Print(long long a)
    {
        Printf("%ld", a);
    }
};

template<> struct TypedPrintImpl<unsigned int>
{
    static void Print(unsigned int a)
    {
        Printf("%u", a);
    }
};

template<> struct TypedPrintImpl<unsigned long>
{
    static void Print(unsigned int a)
    {
        Printf("%lu", a);
    }
};

template<> struct TypedPrintImpl<unsigned long long>
{
    static void Print(unsigned int a)
    {
        Printf("%llu", a);
    }
};

template<> struct TypedPrintImpl<nn::Result>
{
    static void Print(nn::Result a)
    {
        Printf("Result(%d/%d)", a.GetModule(), a.GetDescription());
    }
};

template<> struct TypedPrintImpl<char*>
{
    static void Print(char* a)
    {
        Printf("%s", a);
    }
};

template<> struct TypedPrintImpl<nn::svc::Handle>
{
    static void Print(nn::svc::Handle a)
    {
        Printf("svc::Handle(0x%x)", a.GetPrintableBits());
    }
};

template<typename T> struct TypedPrintImpl<T*>
{
    static void Print(T* a)
    {
        Printf("%p", a);
    }
};

template<typename T>
void TypedPrint(T t)
{
    TypedPrintImpl<T>::Print(t);
}

void PushTracker(const char* name, void (*Print) (void*), void* userdata);
void PopTracker();

template<typename T>
class ScopedVariableTracker
{
public:
    ScopedVariableTracker(const char* variableName, T& t):
        m_T(t)
    {
        PushTracker(variableName,
            [](void* data)
            {
                ScopedVariableTracker<T>* self = static_cast<ScopedVariableTracker<T>*> (data);
                TypedPrint(self->m_T);
            }, this);
    }

    ~ScopedVariableTracker()
    {
        PopTracker();
    }

private:
    T& m_T;
};

template<typename Lambda>
class ScopedLambdaTracker
{
public:
    ScopedLambdaTracker(const char* lambdaName, Lambda lambda):
        m_Lambda(lambda)
    {
        PushTracker(lambdaName,
            [](void* data)
            {
                ScopedLambdaTracker<Lambda>* self = static_cast<ScopedLambdaTracker*> (data);
                auto toPrint = self->m_Lambda();
                TypedPrint(toPrint);
            }, this);
    }

    ~ScopedLambdaTracker()
    {
        PopTracker();
    }

private:
    Lambda m_Lambda;
};

template<typename Lambda>
ScopedLambdaTracker<Lambda> MakeScopedLambdaTracker(const char* name, Lambda lambda)
{
    return ScopedLambdaTracker<Lambda> (name, lambda);
}

template<typename... Results>
struct ResultIsOneOfImpl;

template<typename Result, typename... Results>
struct ResultIsOneOfImpl<Result, Results...>
{
    static bool Do (nn::Result result)
    {
        if (Result::Includes(result))
        {
            return true;
        }
        else
        {
            return ResultIsOneOfImpl<Results...>::Do(result);
        }
    }
};

template<>
struct ResultIsOneOfImpl<>
{
    static bool Do(nn::Result)
    {
        return false;
    }
};

template<typename... Results>
bool ResultIsOneOf(nn::Result result)
{
    return ResultIsOneOfImpl<Results...>::Do(result);
}

template<typename... Results>
struct PrintResultsImpl;

template<typename Result, typename... Results>
struct PrintResultsImpl<Result, Results...>
{
    static void Do()
    {
        ::nnt::svc::fwk::Printf("%d/%d, ", Result().GetModule(), Result().GetDescription());
        PrintResultsImpl<Results...>::Do();
    }
};

template<typename Result>
struct PrintResultsImpl<Result>
{
    static void Do()
    {
        ::nnt::svc::fwk::Printf("%d/%d", Result().GetModule(), Result().GetDescription());
    }
};

template<typename... Results>
void PrintResults()
{
    PrintResultsImpl<Results...>::Do();
}

} } }
