﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_Api.h"
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/TargetConfigs/build_Base.h>
#include "sdmmc_ClockResetController.h"
#include "sdmmc_IHostController.h"
#if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1))
    #include "sdmmc_SdHostStandardController.h"
#endif
#include "sdmmc_IDevice.h"
#if (defined(NN_SDMMC_PORT_MMC_0_ENABLE))
    #include "sdmmc_MmcDevice.h"
#endif
#if (defined(NN_SDMMC_PORT_SD_CARD_0_ENABLE))
    #include "sdmmc_SdCardDevice.h"
#endif

namespace nn { namespace sdmmc1 {

namespace
{
    #if (defined(NN_SDMMC_PORT_MMC_0_ENABLE) && defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1))
        const detail::ClockResetController::Module Mmc0Modlue = detail::ClockResetController::Module_Sdmmc4;
        const uint32_t NN_SVC_ADDR_SDMMC4_BEGIN = (0x700b0000 + 0x600U);
        const bool IsSdmmc4SupportBusPower1_8V = true;
        const bool IsSdmmc4SupportBusPower3_3V = false;
        detail::SdHostStandardController s_Mmc0HostController(NN_SVC_ADDR_SDMMC4_BEGIN, IsSdmmc4SupportBusPower1_8V, IsSdmmc4SupportBusPower3_3V);
        detail::IHostController* s_pMmc0HostController = &s_Mmc0HostController;
        detail::MmcDevice s_Mmc0Device;
        detail::IDevice* s_pMmc0Device = &s_Mmc0Device;
    #else
        const detail::ClockResetController::Module Mmc0Modlue = detail::ClockResetController::Module_None;
        detail::IHostController* s_pMmc0HostController = nullptr;
        detail::IDevice* s_pMmc0Device = nullptr;
    #endif

    #if (defined(NN_SDMMC_PORT_SD_CARD_0_ENABLE) && defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1))
        const detail::ClockResetController::Module SdCard0Modlue = detail::ClockResetController::Module_Sdmmc3;
        const uint32_t NN_SVC_ADDR_SDMMC3_BEGIN = (0x700b0000 + 0x400U);
        const bool IsSdmmc3SupportBusPower1_8V = false;     // TODO: 回路上は true にできる制御がありそう
        const bool IsSdmmc3SupportBusPower3_3V = true;
        detail::SdHostStandardController s_SdCard0HostController(NN_SVC_ADDR_SDMMC3_BEGIN, IsSdmmc3SupportBusPower1_8V, IsSdmmc3SupportBusPower3_3V);
        detail::IHostController* s_pSdCard0HostController = &s_SdCard0HostController;
        detail::SdCardDevice s_SdCard0Device;
        detail::IDevice* s_pSdCard0Device = &s_SdCard0Device;
    #else
        const detail::ClockResetController::Module SdCard0Modlue = detail::ClockResetController::Module_None;
        detail::IHostController* s_pSdCard0HostController = nullptr;
        detail::IDevice* s_pSdCard0Device = nullptr;
    #endif

    detail::IDevice* GetDevice(Port port) NN_NOEXCEPT
    {
        detail::IDevice* pDevice = nullptr;
        switch (port)
        {
        case Port_Mmc0:
            pDevice = s_pMmc0Device;
            break;
        case Port_SdCard0:
            pDevice = s_pSdCard0Device;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        NN_ABORT_UNLESS_NOT_NULL(pDevice);
        return pDevice;
    }

    void Initialize(Port port) NN_NOEXCEPT
    {
        detail::ClockResetController::Module modlue = detail::ClockResetController::Module_None;
        switch (port)
        {
        case Port_Mmc0:
            modlue = Mmc0Modlue;
            break;
        case Port_SdCard0:
            modlue = SdCard0Modlue;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        NN_ABORT_UNLESS(modlue != detail::ClockResetController::Module_None);

        detail::IHostController* pHostController = nullptr;
        switch (port)
        {
        case Port_Mmc0:
            pHostController = s_pMmc0HostController;
            break;
        case Port_SdCard0:
            pHostController = s_pSdCard0HostController;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        NN_ABORT_UNLESS_NOT_NULL(pHostController);

        detail::IDevice* pDevice = GetDevice(port);
        NN_ABORT_UNLESS_NOT_NULL(pDevice);

        if (detail::ClockResetController::IsEnableSourceClock(modlue))
        {
            // クロックソース変更によるグリッジを防ぐためにクロック供給を止める
            pHostController->DisableDeviceClock();
        }
        detail::ClockResetController::SetSourceClock(modlue);
        detail::ClockResetController::EnableSourceClock(modlue);
        pHostController->Reset();

        pDevice->Initialize(pHostController);
    }

}

void Initialize() NN_NOEXCEPT
{
    #if (defined(NN_SDMMC_PORT_MMC_0_ENABLE))
        Initialize(Port_Mmc0);
    #endif

    #if (defined(NN_SDMMC_PORT_SD_CARD_0_ENABLE))
        Initialize(Port_SdCard0);
    #endif
}

void Finalize() NN_NOEXCEPT
{
}

Result Activate(Port port) NN_NOEXCEPT
{
    detail::IDevice* pDevice = GetDevice(port);
    return pDevice->Activate();
}

void Deactivate(Port port) NN_NOEXCEPT
{
    // TODO: 削除、実装
    NN_UNUSED(port);
}

Result Read(void* pOutBuffer, size_t bufferSize, Port port, uint32_t sectorIndex, uint32_t numSectors) NN_NOEXCEPT
{
    detail::IDevice* pDevice = GetDevice(port);
    return pDevice->ReadWrite(sectorIndex, numSectors, pOutBuffer, bufferSize, true);
}

Result Write(Port port, uint32_t sectorIndex, uint32_t numSectors, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    detail::IDevice* pDevice = GetDevice(port);
    return pDevice->ReadWrite(sectorIndex, numSectors, const_cast<void*>(pData), dataSize, false);
}

}} // namespace nn { namespace sdmmc1 {
