﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "util_TestIpcData.h"
#include "test_Common.h"
#include "util_TestIpc.h"
#include "util_TestMemory.h"
#include <nn/svc/ipc/svc_SessionMessage.h>

#include <cstring>

namespace {

char g_IpcMapBuffer[0x3000] __attribute__((aligned(0x1000)));
char g_IpcRecvBuffer[2][0x1000] __attribute__((aligned(0x1000)));

const char* PointData1 = "PointData1";
const char* PointData2 = "PointData2";
const char* RawData1 = "12345";

const int32_t SendPointDataNum = 2;
const char* SendPointData[2] = { PointData1, PointData2};
const size_t SendPointDataSize[2] = { sizeof(PointData1), sizeof(PointData2)};
const uint8_t SendMapSendData = 0x1;
const uint8_t SendMapRecvData = 0x2;
const uint8_t SendMapExchData = 0x3;
const size_t MapSendSize = 0x1001;
const size_t MapRecvSize = 0xfff;
const size_t MapExchSize = 0x1000;
const size_t MapDefaultSize = 0x1000;
const char* SendRawData = RawData1;
const size_t SendRawDataSize = sizeof(RawData1);

} // namespace

void MakeClientIpcAllData(
        nn::svc::Handle* pCopyHandle, nn::svc::Handle* pMoveHandle, nn::Bit32* pMsgBuffer)
{
    char* recvBuffers[2] = { g_IpcRecvBuffer[0], g_IpcRecvBuffer[1] };
    size_t recvBufferSizes[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };

    MakeAllData(
            pCopyHandle, pMoveHandle,
            pMsgBuffer,
            SendPointData, SendPointDataSize, SendPointDataNum,
            g_IpcMapBuffer, MapDefaultSize, SendMapSendData,
            g_IpcMapBuffer + MapDefaultSize, MapDefaultSize, SendMapRecvData,
            g_IpcMapBuffer + MapDefaultSize * 2, MapDefaultSize, SendMapExchData,
            SendRawData, SendRawDataSize,
            recvBuffers, recvBufferSizes, 2
            );
}

void MakeClientProcessIdData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyProcessIdData(pMsgBuffer);
}

void MakeClientCopyHandleData(nn::svc::Handle* pOut, nn::Bit32* pMsgBuffer)
{
    MakeOnlyCopyHandleData(pOut, pMsgBuffer);
}

void MakeClientMoveHandleData(nn::svc::Handle* pOut, nn::Bit32* pMsgBuffer)
{
    MakeOnlyMoveHandleData(pOut, pMsgBuffer);
}

void MakeClientPointerDataWithIpcBuffer(nn::Bit32* pMsgBuffer)
{
    MakeOnlyPointerWithIpcBufferData(
            pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum);
}

void MakeClientPointerDataWithOneUserBufferData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyPointerWithOneUserBuffer(
            pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum,
            g_IpcRecvBuffer[0], sizeof(g_IpcRecvBuffer[0]));
}

void MakeClientPointerDataWithMultiUserBufferData(nn::Bit32* pMsgBuffer)
{
    size_t bufferSize[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };
    char* buffers[2] = { g_IpcRecvBuffer[0], g_IpcRecvBuffer[1] };
    MakeOnlyPointerWithMultiUserBuffer(
            pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum,
            buffers, bufferSize);
}

void MakeClientMapSendData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyMapSendData(pMsgBuffer, g_IpcMapBuffer, MapSendSize, SendMapSendData);
}

void MakeClientMapRecvData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyMapReceiveData(pMsgBuffer, g_IpcMapBuffer, MapRecvSize, SendMapRecvData);
}

void MakeClientMapExchData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyMapExchangeData(pMsgBuffer, g_IpcMapBuffer, MapExchSize, SendMapExchData);
}

void MakeClientRawData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyRawData(pMsgBuffer, SendRawData, SendRawDataSize);
}

void ServerCheckProcessId(nn::Bit32* pMsgBuffer)
{
    nn::Bit64 pid;
    nn::Result result = nn::svc::GetProcessId(&pid, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    NN_ASSERT_RESULT_SUCCESS(result);
    CheckOnlyProcessIdData(pid, pMsgBuffer);
}

void ServerCheckCopyHandle(nn::Bit32* pMsgBuffer)
{
    CheckOnlyCopyHandleData(pMsgBuffer, true);
}

void ServerCheckPointerWithIpc(nn::Bit32* pMsgBuffer, size_t msgSize)
{
    CheckOnlyPointerWithIpcBufferData(
            SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer, msgSize);
}

void ServerCheckPointerWithOne(nn::Bit32* pMsgBuffer)
{
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_IpcRecvBuffer[0]);
    size_t bufSize = sizeof(g_IpcRecvBuffer[0]);
    CheckOnlyPointerWithOneUserBuffer(
            addr, bufSize,
            SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer);
}

void ServerCheckPointerWithMulti(nn::Bit32* pMsgBuffer)
{
    size_t bufferSize[] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };
    uintptr_t buffers[] = {
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[0]),
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[1])
    };
    CheckOnlyPointerWithMultiUserBuffer(
            buffers, bufferSize,
            SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer);
}

void ServerCheckMapSendData(uintptr_t* pOut, nn::Bit32* pMsgBuffer)
{
    CheckOnlyMapSendData(pOut, MapSendSize, SendMapSendData, pMsgBuffer);
}

void ServerCheckMapRecvData(uintptr_t* pOut, nn::Bit32* pMsgBuffer)
{
    CheckOnlyMapReceiveData(pOut, MapRecvSize, SendMapRecvData, pMsgBuffer);
}

void ServerCheckMapExchData(uintptr_t* pOut, nn::Bit32* pMsgBuffer)
{
    CheckOnlyMapExchangeData(pOut, MapExchSize, SendMapExchData, pMsgBuffer);
}

void ServerCheckRawData(nn::Bit32* pMsgBuffer)
{
    CheckOnlyRawData(SendRawData, SendRawDataSize, pMsgBuffer);
}

void ServerCheckAllData(
        uintptr_t* sendMapAddr, uintptr_t* receiveMapAddr, uintptr_t* exchangeMapAddr,
        nn::Bit32* pMsgBuffer)
{
    uintptr_t recvBufferAddresses[2] = {
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[0]),
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[1]),
    };
    size_t recvBufferSizes[2] = {
        sizeof(g_IpcRecvBuffer[0]),
        sizeof(g_IpcRecvBuffer[1]),
    };

    CheckAllData(
            sendMapAddr, receiveMapAddr, exchangeMapAddr,
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, true, false,
            SendPointData, SendPointDataSize, SendPointDataNum,
            recvBufferAddresses, recvBufferSizes,
            MapDefaultSize, SendMapSendData,
            MapDefaultSize, SendMapRecvData,
            MapDefaultSize, SendMapExchData,
            SendRawData, SendRawDataSize,
            pMsgBuffer
            );
}

void ServerCheckIpcData(
        TestIpcMapData* pOutMapData, nn::Bit32* pMsgBuffer, size_t msgSize, DataType dataType)
{
    switch(dataType)
    {
    case DataType_ProcessId:
        ServerCheckProcessId(pMsgBuffer);
        break;
    case DataType_CopyHandle:
        ServerCheckCopyHandle(pMsgBuffer);
        break;
    case DataType_MoveHandle:
        break;
    case DataType_PointerWithIpc:
        ServerCheckPointerWithIpc(pMsgBuffer, msgSize);
        break;
    case DataType_PointerWithOne:
        ServerCheckPointerWithOne(pMsgBuffer);
        break;
    case DataType_PointerWithMulti:
        ServerCheckPointerWithMulti(pMsgBuffer);
        break;
    case DataType_MapSend:
        ServerCheckMapSendData(&pOutMapData->sendAddr, pMsgBuffer);
        CheckClientMapArea(dataType);
        break;
    case DataType_MapRecv:
        ServerCheckMapRecvData(&pOutMapData->receiveAddr, pMsgBuffer);
        CheckClientMapArea(dataType);
        break;
    case DataType_MapExch:
        ServerCheckMapExchData(&pOutMapData->exchangeAddr, pMsgBuffer);
        CheckClientMapArea(dataType);
        break;
    case DataType_Raw:
        ServerCheckRawData(pMsgBuffer);
        break;
    case DataType_All:
        ServerCheckAllData(
                &pOutMapData->sendAddr, &pOutMapData->receiveAddr, &pOutMapData->exchangeAddr,
                pMsgBuffer);
        CheckClientMapArea(dataType);
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void PrepairServerIpc(nn::Bit32* pMsgBuffer, DataType dataType)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    nn::svc::ipc::MessageBuffer::ReceiveListEntry
        recvEntry1(g_IpcRecvBuffer[0], sizeof(g_IpcRecvBuffer[0]));

    nn::svc::ipc::MessageBuffer::ReceiveListEntry
        recvEntry2(g_IpcRecvBuffer[1], sizeof(g_IpcRecvBuffer[1]));

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 1);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcOneHeader(0, 0, 0, 0, 0, 0, 0, 2);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcMultiHeader(0, 0, 0, 0, 0, 0, 0, 4);
    int offset = 0;

    switch(dataType)
    {
    case DataType_ProcessId:
        ipcMsg.SetNull();
        break;
    case DataType_CopyHandle:
        ipcMsg.SetNull();
        break;
    case DataType_MoveHandle:
        ipcMsg.SetNull();
        break;
    case DataType_PointerWithIpc:
        offset = ipcMsg.Set(ipcHeader);
        break;
    case DataType_PointerWithOne:
        offset = ipcMsg.Set(ipcOneHeader);
        offset = ipcMsg.Set(offset, recvEntry1);
        break;
    case DataType_PointerWithMulti:
        offset = ipcMsg.Set(ipcMultiHeader);
        offset = ipcMsg.Set(offset, recvEntry1);
        offset = ipcMsg.Set(offset, recvEntry2);
        break;
    case DataType_MapSend:
        ipcMsg.SetNull();
        break;
    case DataType_MapRecv:
        ipcMsg.SetNull();
        break;
    case DataType_MapExch:
        ipcMsg.SetNull();
        break;
    case DataType_Raw:
        ipcMsg.SetNull();
        break;
    case DataType_All:
        offset = ipcMsg.Set(ipcMultiHeader);
        offset = ipcMsg.Set(offset, recvEntry1);
        offset = ipcMsg.Set(offset, recvEntry2);
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void MakeClientData(nn::svc::Handle* pOutCopyHandle, nn::Bit32* pMsgBuffer, DataType dataType)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    switch(dataType)
    {
    case DataType_ProcessId:
        MakeClientProcessIdData(pMsgBuffer);
        break;
    case DataType_CopyHandle:
        NN_ASSERT(pOutCopyHandle != nullptr);
        MakeClientCopyHandleData(pOutCopyHandle, pMsgBuffer);
        break;
    case DataType_MoveHandle:
        break;
    case DataType_PointerWithIpc:
        MakeClientPointerDataWithIpcBuffer(pMsgBuffer);
        break;
    case DataType_PointerWithOne:
        MakeClientPointerDataWithOneUserBufferData(pMsgBuffer);
        break;
    case DataType_PointerWithMulti:
        MakeClientPointerDataWithMultiUserBufferData(pMsgBuffer);
        break;
    case DataType_MapSend:
        MakeClientMapSendData(pMsgBuffer);
        break;
    case DataType_MapRecv:
        MakeClientMapRecvData(pMsgBuffer);
        break;
    case DataType_MapExch:
        MakeClientMapExchData(pMsgBuffer);
        break;
    case DataType_Raw:
        MakeClientRawData(pMsgBuffer);
        break;
    case DataType_All:
        NN_ASSERT(pOutCopyHandle != nullptr);
        MakeClientIpcAllData(pOutCopyHandle, nullptr, pMsgBuffer);
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void GetClientMapArea(uintptr_t* pOutAddress, size_t* pOutSize, DataType dataType)
{
    switch(dataType)
    {
    case DataType_MapSend:
        *pOutAddress = reinterpret_cast<uintptr_t>(g_IpcMapBuffer);
        *pOutSize = MapSendSize;
        break;
    case DataType_MapRecv:
        *pOutAddress = reinterpret_cast<uintptr_t>(g_IpcMapBuffer);
        *pOutSize = MapRecvSize;
        break;
    case DataType_MapExch:
        *pOutAddress = reinterpret_cast<uintptr_t>(g_IpcMapBuffer);
        *pOutSize = MapExchSize;
        break;
    case DataType_All:
        *pOutAddress = reinterpret_cast<uintptr_t>(g_IpcMapBuffer);
        *pOutSize = 3 * MapDefaultSize;
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void CheckClientBaseMapArea(DataType dataType)
{
    uintptr_t addr;
    size_t size;

    GetClientMapArea(&addr, &size, dataType);

    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, addr);

    NN_ASSERT(info.baseAddress <= addr);
    NN_ASSERT(info.size >= size);
    NN_ASSERT(info.baseAddress + info.size - 1 >= addr + size - 1);
    NN_ASSERT(info.state == nn::svc::MemoryState_CodeData);
    NN_ASSERT(info.permission == nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT(info.attribute == 0);
    NN_ASSERT(info.ipcCount == 0);
}

void CheckClientMapArea(uintptr_t addr, size_t size, nn::svc::MemoryPermission perm)
{
    // 端数は属性が付かない
    size_t mapSize = size & ~(0xFFF);

    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, addr);

    NN_ASSERT(info.baseAddress <= addr);
    NN_ASSERT(info.size >= mapSize);
    NN_ASSERT(info.baseAddress + info.size - 1 >= addr + mapSize - 1);
    NN_ASSERT(info.state == nn::svc::MemoryState_CodeData);
    if (mapSize < 0x1000)
    {
        NN_ASSERT_EQUAL(info.permission, nn::svc::MemoryPermission_ReadWrite);
        NN_ASSERT_EQUAL(info.attribute, 0);
        NN_ASSERT_EQUAL(info.ipcCount, 0);
    }
    else
    {
        NN_ASSERT_EQUAL(info.permission, perm);
        NN_ASSERT_EQUAL(info.attribute, nn::svc::MemoryAttribute_IpcLocked);
        NN_ASSERT_EQUAL(info.ipcCount, 1);
    }
}

void CheckClientMapArea(DataType dataType)
{
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_IpcMapBuffer);
    switch(dataType)
    {
    case DataType_MapSend:
        CheckClientMapArea(addr, MapSendSize, nn::svc::MemoryPermission_Read);
        break;
    case DataType_MapRecv:
        CheckClientMapArea(addr, MapRecvSize, nn::svc::MemoryPermission_None);
        break;
    case DataType_MapExch:
        CheckClientMapArea(addr, MapExchSize, nn::svc::MemoryPermission_None);
        break;
    case DataType_All:
        CheckClientMapArea(addr, MapDefaultSize, nn::svc::MemoryPermission_Read);
        CheckClientMapArea(addr + MapDefaultSize, MapDefaultSize * 2, nn::svc::MemoryPermission_None);
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

